/*
**************************************************************************
                                 description
                             --------------------
    copyright            : (C) 2000-2001 by Leonardo Skorianez
    email                : lsk@if.ufrj.br
**************************************************************************

**************************************************************************
*                                                                        *
*  This program is free software; you can redistribute it and/or modify  *
*  it under the terms of the GNU General Public License as published by  *
*  the Free Software Foundation; either version 2 of the License, or     *
*  (at your option) any later version.                                   *
*                                                                        *
**************************************************************************/


#include "pmcylinder.h"

#include "pmoutputdevice.h"
#include "pmxmlhelper.h"
#include "pmboxedit.h"
#include "pmmemento.h"
#include "pm3dcontrolpoint.h"
#include "pmdefaults.h"

#include <kdebug.h>
#include "pmglobals.h"

#include <klocale.h>

#include "pmdistancecontrolpoint.h"

#include "pmcylinderedit.h"

const double defaultCylRadius = 0.5;
const double defaultHalfCylSize = 0.5;
const PMVector defaultEnd1 = PMVector ( 0, defaultHalfCylSize, 0 );
const PMVector defaultEnd2 = PMVector ( 0, -defaultHalfCylSize, 0 );
const bool defaultOpen = false;

/** default cylinder structure */
PMViewStructure* PMCylinder::s_pDefaultViewStructure = 0;
int PMCylinder::s_numSteps = c_defaultCylinderSteps;
int PMCylinder::s_parameterKey = 0;


PMCylinder::PMCylinder( )
      :Base( )
{
   m_end1 = defaultEnd1;
   m_end2 = defaultEnd2;
   m_radius = defaultCylRadius;
   m_open = defaultOpen;
}

PMCylinder::~PMCylinder( )
{
}

QString PMCylinder::description( ) const
{
   return i18n( "cylinder" );
}

void PMCylinder::serialize( PMOutputDevice& dev ) const
{
   dev.objectBegin( "cylinder" );

   serializeName( dev );
   QString str1;
   str1.setNum( m_radius );
   dev.writeLine( m_end1.serialize( ) + ", " + m_end2.serialize( )
                  + ", " + str1 );
   if( m_open )
      dev.writeLine( QString( "open" ) );
   
   Base::serialize( dev );
   dev.objectEnd( );
}

void PMCylinder::serialize( QDomElement& e, QDomDocument& doc ) const
{
   e.setAttribute( "end_a", m_end1.serializeXML( ) );
   e.setAttribute( "end_b", m_end2.serializeXML( ) );
   e.setAttribute( "radius", m_radius );
   e.setAttribute( "open", m_open );
   Base::serialize( e, doc );
}

void PMCylinder::readAttributes( const PMXMLHelper& h )
{
   m_end1 = h.vectorAttribute( "end_a", defaultEnd1 );
   m_end2 = h.vectorAttribute( "end_b", defaultEnd2 );
   m_radius = h.doubleAttribute( "radius", defaultCylRadius );
   m_open = h.boolAttribute( "open", defaultOpen );
   Base::readAttributes( h );
}

bool PMCylinder::isA( PMObjectType t ) const
{
   if( t == PMTCylinder )
      return true;
   return Base::isA( t );
}

void PMCylinder::setEnd1( const PMVector& p )
{
   if( p != m_end1 )
   {
      if( m_pMemento )
         m_pMemento->addData( PMTCylinder, PMEnd1ID, m_end1 );
      m_end1 = p;
      m_end1.resize( 3 );
      setViewStructureChanged( );
   }
}

void PMCylinder::setEnd2( const PMVector& p )
{
   if( p != m_end2 )
   {
      if( m_pMemento )
         m_pMemento->addData( PMTCylinder, PMEnd2ID, m_end2 );
      m_end2 = p;
      m_end2.resize( 3 );
      setViewStructureChanged( );
   }
}

void PMCylinder::setRadius( double radius )
{
   if( m_radius != radius )
   {
      if( m_pMemento )
        m_pMemento->addData( PMTCylinder, PMRadiusID, m_radius );
      m_radius = radius;
      setViewStructureChanged( );
   }
}

void PMCylinder::setOpen( bool op )
{
   if( op != m_open )
   {
      if( m_pMemento )
         m_pMemento->addData( PMTCylinder, PMOpenID, m_open );
      m_open = op;
   }
}

PMDialogEditBase* PMCylinder::editWidget( QWidget* parent ) const
{
   return new PMCylinderEdit( parent );
}

void PMCylinder::restoreMemento( PMMemento* s )
{
   PMMementoDataIterator it( s );
   PMMementoData* data;

   for( ; it.current( ); ++it )
   {
      data = it.current( );
      if( data->objectType( ) == PMTCylinder )
      {
         switch( data->valueID( ) )
         {
            case PMEnd1ID:
               setEnd1( data->vectorData( ) );
            break;
            case PMEnd2ID:
               setEnd2( data->vectorData( ) );
            break;
            case PMRadiusID:
               setRadius( data->doubleData( ) );
            break;
            case PMOpenID:
               setOpen( data->boolData( ) );
            break;
            default:
               kdError( PMArea ) << "Wrong ID in PMCylinder::restoreMemento\n";
            break;
         }
      }
   }
   Base::restoreMemento( s );
}

bool PMCylinder::isDefault( )
{
   if( ( m_end1 == defaultEnd1 ) && ( m_end2 == defaultEnd2 )
       && ( m_radius == defaultCylRadius ) && ( m_open == defaultOpen ) )
      return true;
   return false;
}

void PMCylinder::createViewStructure( )
{
   if( !m_pViewStructure )
   {
      m_pViewStructure = new PMViewStructure( defaultViewStructure ( ) );
      m_pViewStructure->points( ).detach( );
   }

   createPoints( m_pViewStructure->points( ), m_end1, m_end2, m_radius );
}

PMViewStructure* PMCylinder::defaultViewStructure( ) const
{
   if( !s_pDefaultViewStructure )
   {
      s_pDefaultViewStructure = new PMViewStructure( s_numSteps * 2, s_numSteps * 3 );
      PMLineArray & lines = s_pDefaultViewStructure->lines( );

      createPoints( s_pDefaultViewStructure->points( ), defaultEnd1,
                    defaultEnd2, defaultCylRadius );
      
      int i;
      for( i = 0; i < ( s_numSteps - 1 ); i++ )
      {
          lines[i] = PMLine( i, i + 1 );
          lines[i + s_numSteps] = PMLine( i + s_numSteps, i + s_numSteps + 1 );
      }
      lines[s_numSteps - 1] = PMLine( s_numSteps - 1, 0 );
      lines[s_numSteps * 2 - 1] = PMLine( s_numSteps * 2 - 1, s_numSteps );

      for( i = 0; i < s_numSteps; i++ )
      {
          lines[i + 2 * s_numSteps] = PMLine( i, i + s_numSteps );
      }
      
   }
   return s_pDefaultViewStructure;
}

void PMCylinder::createPoints( PMPointArray& points, const PMVector& end1,
                               const PMVector& end2, double radius )
{
   double angle = ( 2.0 * M_PI ) / ( double ) s_numSteps;

   PMVector pointAt = end2 - end1;
   double pl = pointAt.abs( );
   if( approxZero( pl ) )
      pointAt = PMVector( 0.0, 0.0, 1.0 );
   else
      pointAt /= pl;

   PMMatrix rotation = PMMatrix::rotation( pointAt, angle );
   PMVector endPoint = pointAt.orthogonal( );
   endPoint *= radius;

   int i;
   for( i = 0; i < s_numSteps; i++ )
   {
      points[i] = PMPoint( endPoint + end1 );
      points[i + s_numSteps] = PMPoint( endPoint + end2 );
      endPoint = rotation * endPoint;
   }
}

void PMCylinder::controlPoints( PMControlPointList & list )
{
   PMVector center, angle1, angle2;
   center = m_end1 - m_end2;
   double pl = center.abs( );
   if( approxZero( pl ) )
      center = PMVector( 0.0, 1.0, 0.0 );
   else
      center /= pl;

   angle1 = center.orthogonal( );
   angle2 = PMVector::cross( center, angle1 );

   PM3DControlPoint* pb = new PM3DControlPoint( m_end1, PMEnd1ID, i18n( "End 1" ) );
   list.append( pb );
   list.append( new PM3DControlPoint( m_end2, PMEnd2ID, i18n( "End 2" ) ) );
   list.append( new PMDistanceControlPoint( pb, angle1, m_radius, PMRadiusID, i18n( "Radius (1)" ) ) );
   list.append( new PMDistanceControlPoint( pb, angle2, m_radius, PMRadiusID, i18n( "Radius (2)" ) ) );
}


void PMCylinder::controlPointsChanged( PMControlPointList & list )
{
   PMControlPoint* p;
   bool pointChanged = false;
   bool radiusChanged = false;

   for( p = list.first( ); p; p = list.next( ) )
   {
      if( p->changed( ) )
      {
         switch( p->id( ) )
         {
            case PMEnd1ID:
               setEnd1( ( ( PM3DControlPoint *) p)->point( ) );
               pointChanged = true;
            break;
            case PMEnd2ID:
               setEnd2( ( ( PM3DControlPoint *) p)->point( ) );
               pointChanged = true;
            break;
            case PMRadiusID:
               setRadius( ( ( PMDistanceControlPoint *) p)->distance( ) );
               radiusChanged = true;
            break;
            default:
               kdError( PMArea ) << "Wrong ID in PMCylinder::controlPointsChanged\n";
               break;
         }
      }
   }

   if( pointChanged )
   {
      PMVector center, angle1, angle2;
      bool firstPoint = true;

      center = m_end1 - m_end2;
      double pl = center.abs( );
      if( approxZero( pl ) )
         center = PMVector( 0.0, 1.0, 0.0 );
      else
        center /= pl;

      angle1 = center.orthogonal( );
      angle2 = PMVector::cross( center, angle1 );

      for( p = list.first( ); p; p = list.next( ) )
         if( p->id( ) == PMRadiusID )
         {
            if( firstPoint )
            {
               ( ( PMDistanceControlPoint *) p)->setDirection( angle1 );
               firstPoint = false;
            }
            else
               ( ( PMDistanceControlPoint *) p)->setDirection( angle2 );
         }
    }

   if( radiusChanged )
     for( p = list.first( ); p; p = list.next( ) )
        if( p->id( ) == PMRadiusID )
           ( ( PMDistanceControlPoint *) p)->setDistance( m_radius );
}

void PMCylinder::setSteps( int s )
{
   if( s >= 4 )
   {
      s_numSteps = s;
      if( s_pDefaultViewStructure )
      {
         delete s_pDefaultViewStructure;
         s_pDefaultViewStructure = 0;
      }
   }
   else
      kdDebug( PMArea ) << "PMCylinder::setSteps: S must be greater than 3\n";
   s_parameterKey++;
}

void PMCylinder::cleanUp( ) const
{
   if( s_pDefaultViewStructure )
      delete s_pDefaultViewStructure;
   s_pDefaultViewStructure = 0;
   Base::cleanUp( );
}
