#***************************************************************************
#                            kalyptusCxxToRuby.pm -  Generates ruby SWIG style wrappers for C++ headers
#                             -------------------
#    begin                : Fri Oct 20 12:00:00 2000
#    copyright            : (C) 2000-2001 Lost Highway Ltd. All Rights Reserved.
#    email                : Richard_Dale@tipitina.demon.co.uk
#    author               : Richard Dale.
#***************************************************************************/

#/***************************************************************************
# *                                                                         *
# *   This program is free software; you can redistribute it and/or modify  *
# *   it under the terms of the GNU General Public License as published by  *
# *   the Free Software Foundation; either version 2 of the License, or     *
# *   (at your option) any later version.                                   *
# *                                                                         *
#***************************************************************************/

package kalyptusCxxToRuby;

use File::Path;
use File::Basename;

use Carp;
use Ast;
use kdocAstUtil;
use kdocUtil; 
use Iter;
use kalyptusDataDict;

use strict;
no strict "subs";

use vars qw/ @clist $host $who $now $gentext %functionId $parentClass $qtdocTop $kdedocTop
	$lib $rootnode $outputdir $opt $debug $typeprefix $signalCount $eventHandlerCount $constructorCount *RUBYSOURCE  /;




BEGIN
{
	@clist = ();


	# Page footer

	$who = kdocUtil::userName();
	$host = kdocUtil::hostName();
	$now = localtime;
	$gentext = "$who\@$host on $now, using kalyptus.";

	$qtdocTop = <<EOF;
                             -------------------
    begin                : $now
    copyright            : (C) 2000-2001 Lost Highway Ltd. All rights reserved.
    email                : Lost_Highway\@tipitina.demon.co.uk
    generated by         : $gentext
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
***************************************************************************/

EOF

	$kdedocTop = <<EOF;
                             -------------------
    begin                : $now
    copyright            : (C) 2000-2001 Lost Highway Ltd.
    email                : Richard_Dale\@tipitina.demon.co.uk
    generated by         : $gentext
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
***************************************************************************/

EOF

}

sub cplusplusToRuby
{
	my ( $cplusplusType )  = @_;

	if ( $cplusplusType =~ /bool/ && kalyptusDataDict::ctypemap($cplusplusType) eq "int" ) {
		return "boolean";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /^void\s*\*/ ) {
		return "int";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /^\s*int\s*\*/ ) {
		return "int[]";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /^\s*double\s*\*/ ) {
		return "double[]";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /^\s*short\s*\*/ ) {
		return "short[]";
	} elsif ( 	kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QStringList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QStrList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_QCStringList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QObjectList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QDomNodeList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QWidgetList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_KURL_List\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_KMainWindow\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_KFileItemList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_KFileViewItemList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_NodeList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_StyleSheetList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_MediaList\s*\*/
				|| kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_OfferList\s*\*/ ) {
		return "ArrayList"
	} elsif ( $cplusplusType =~ /(DOM::)?DOMString/ || $cplusplusType =~ /QString/ || $cplusplusType =~ /QCString/ || kalyptusDataDict::ctypemap($cplusplusType) =~ /^char\s*\*/ ) {
		return "String"
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QChar\s*\*/ ) {
		return "char"
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QTime\s*\*/ ) {
		return "Date"
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QDateTime\s*\*/ || kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_QDate\s*\*/ ) {
		return "Calendar"
	} elsif ( $cplusplusType =~ /QPaintDevice/ ) {
		return "QPaintDeviceInterface"
	} elsif ( $cplusplusType =~ /QByteArray/ ) {
		return "byte[]"
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /qt_([^\*\s]*)(.*)$/ ) {
		if ( kalyptusDataDict::interfacemap($1) ne () ) {
			return $1."Interface";
        } else {
			return $1;
		}
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /kde_([^\*\s]*)(.*)$/ ) {
		if ( kalyptusDataDict::interfacemap($1) ne () ) {
			return $1."Interface";
        } else {
			return $1;
		}
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /(const)? char *\s*/ ) {
		return "String";
	} elsif ( kalyptusDataDict::ctypemap($cplusplusType) =~ /(unsigned )?(.*)/ ) {
		return $2;
	} else {
		return kalyptusDataDict::ctypemap($cplusplusType);
	}

}

sub writeDoc
{
	( $lib, $rootnode, $outputdir, $opt ) = @_;

	$debug = $main::debuggen;

	mkpath( $outputdir ) unless -f $outputdir;


	# Document all compound nodes
	Iter::LocalCompounds( $rootnode, sub { writeClassDoc( shift ); } );

}




=head2 writeClassDoc

	Write documentation for one compound node.

=cut

sub writeClassDoc
{
	my( $node ) = @_;

	print "Enter: $node->{astNodeName}\n" if $debug;

	my $typeName = $node->{astNodeName}."*";

	if ( kalyptusDataDict::ctypemap($typeName) eq () ) {
		$typeprefix = ($typeName =~ /^Q/ ? "qt_" : "kde_");
		kalyptusDataDict::setctypemap($typeName, $typeprefix.$node->{astNodeName}."*");
		print "'$typeName' => '$typeprefix$typeName',\n";
	} elsif ( kalyptusDataDict::ctypemap($typeName) =~ /^qt_/ ) {
		$typeprefix = "qt_";
	} elsif ( kalyptusDataDict::ctypemap($typeName) =~ /^kde_/ ) {
		$typeprefix = "kde_";
	}

	my $docnode = $node->{DocNode};
	my @list = ();
	my $version = undef;
	my $author = undef;


	if( $#{$node->{Kids}} < 0 || $node->{Access} eq "private" || exists $node->{Tmpl} ) {
		return;
	}

	my $file = "$outputdir/".join("__", kdocAstUtil::heritage($node)).".cpp";
	open( RUBYSOURCE, ">$file" ) || die "Couldn't create $file\n";


	my $short = "";
	my $extra = "";


	print RUBYSOURCE "/***************************************************************************\n";
	print RUBYSOURCE "                            ", $node->{astNodeName}, ".cpp -  description\n";
	if ( $typeprefix eq "qt_" ) {
		print RUBYSOURCE $qtdocTop;
    } else {
		print RUBYSOURCE $kdedocTop;
	}

	my $sourcename = $node->{Source}->{astNodeName};
	$sourcename =~ s!.*/(.*)!$1!m;

	print RUBYSOURCE "#include <",$sourcename , ">\n";

		# ancestors
	my @ancestors = ();
	Iter::Ancestors( $node, $rootnode, undef, undef,
		sub { # print
			my ( $ances, $name, $type, $template ) = @_;
			push @ancestors, $name;
			},
			undef
		);

	%functionId = ();
	$signalCount = 0;
	$eventHandlerCount = 0;
	$constructorCount = 0;

	# Add any enums found to the %ctypemap
	Iter::MembersByType ( $node,
		sub {  $_[0], ""; print RUBYSOURCE "", $_[0], "";  },
		sub {	my ($node, $kid ) = @_;
                 preParseMember( $node, $kid );
               },
		sub {  print RUBYSOURCE ""; }
	);


	if ( kalyptusDataDict::interfacemap($node->{astNodeName}) ne () ) {
		print RUBYSOURCE "static VALUE m", $node->{astNodeName}, "Interface;\n";
	}

	print RUBYSOURCE "static VALUE c", $node->{astNodeName}, ";\n";
	print RUBYSOURCE "static void free_", $node->{astNodeName}, "(", $node->{astNodeName}, " *);\n";
	print RUBYSOURCE "#define Wrap_", $node->{astNodeName}, "(klass, ptr) (\\\n";
	print RUBYSOURCE "\t(ptr) ? Data_Wrap_Struct(klass, 0, VOIDFUNC(free_", $node->{astNodeName}, "), ptr) : Qnil )\n";
	print RUBYSOURCE "#define Get_", $node->{astNodeName}, "(val, ptr) {\\\n";
	print RUBYSOURCE "\tif (NIL_P(val)) ptr = NULL;\\\n";
	print RUBYSOURCE "\telse {\\\n";
	print RUBYSOURCE "\t\tif (!rb_obj_is_kind_of(val, c", $node->{astNodeName}, "))\\\n";
	print RUBYSOURCE "\t\t\trb_raise(rb_eTypeError, \"wrong argument type (expected ", $node->{astNodeName}, ")\");\\\n";
	print RUBYSOURCE "\t\tData_Get_Struct(val, ", $node->{astNodeName}, ", ptr);\\\n";
	print RUBYSOURCE "\t\tif (!ptr) rb_raise(rb_eRuntimeError, \"This ", $node->{astNodeName}, " already released\");\\\n";
	print RUBYSOURCE "\t}\\\n}\n";

	Iter::MembersByType ( $node,
		sub { print RUBYSOURCE "", $_[0], ""; print RUBYSOURCE "", $_[0], "";  },
		sub {	my ($node, $kid ) = @_;
                 parseMember( $node, $kid, $#ancestors );
               },
		sub { print RUBYSOURCE ""; print RUBYSOURCE ""; }
	);


	if ( $#ancestors < 0 ) {
		print RUBYSOURCE "\tc", $node->{astNodeName}, " = rb_define_class_under(", ( $typeprefix eq "qt_" ) ? "mQt" : "mKde", ", \"$node->{astNodeName}\", rb_cObject", ");\n";
	} else {
		print CLASS "public class ", $node->{astNodeName}, " extends ";
		my $ancestor;
		foreach $ancestor ( @ancestors ) {
			if ( kalyptusDataDict::interfacemap($ancestor) eq () ) {
				print RUBYSOURCE "\tc", $node->{astNodeName}, " = rb_define_class_under(", ( $typeprefix eq "qt_" ) ? "mQt" : "mKde", ", \"$node->{astNodeName}\", c$ancestor", ");\n";
           		} else {
				print RUBYSOURCE "\trb_include_module(c", $node->{astNodeName}, ", m$ancestor", ");\n";
			}
		}
        }

	print RUBYSOURCE "\n";

	if ( kalyptusDataDict::interfacemap($node->{astNodeName}) ne () ) {
		print RUBYSOURCE "\tm", $node->{astNodeName}, "Interface = rb_define_module_under(", ( $typeprefix eq "qt_" ) ? "mQt" : "mKde", ", \"", $node->{astNodeName}, "Interface\");\n";
		print RUBYSOURCE "\trb_include_module(c", $node->{astNodeName}, ", m", $node->{astNodeName}, "Interface);\n";
	}

	Iter::MembersByType ( $node,
		sub { print RUBYSOURCE "", $_[0], ""; print RUBYSOURCE "", $_[0], "";  },
		sub {	my ($node, $kid ) = @_;
                 generateClassMethodForEnum( $node, $kid );
               },
		sub { print RUBYSOURCE ""; print RUBYSOURCE ""; }
	);

	print RUBYSOURCE "\n";
	close RUBYSOURCE;


}


sub preParseMember
{
	my( $class, $m ) = @_;
	my $name = $m->{astNodeName};

	if( $m->{NodeType} eq "method" ) {
		if ( $functionId{$name} eq "" ) {
			$functionId{$name} = 0;
		} else {
			$functionId{$name}++;
		}

		# A JBridge class will only be generated if there is at least one
		# public or protected constructor
		if ( $name eq $class->{astNodeName} && $m->{Access} ne "private" ) {
			$constructorCount++;
		}
    }

	if( $m->{NodeType} eq "enum" ) {
		# Add a C++ to C type mapping for this enum - an int in C
		$name =~ s/\s//g;
		kalyptusDataDict::setctypemap($name, 'int');
		$name = $class->{astNodeName}."::".$name;
		kalyptusDataDict::setctypemap($name, 'int');
	}
}


sub parseMember
{
	my( $class, $m, $ancestorCount ) = @_;
	my $name;
	my $rubyFunction;
	my $stringargs;
	my $cplusplusparams = $m->{Params};
	my $rubyparams = "VALUE self";
	my $jnireturntype;
	my $javaaccess;
	my $javaargs;

	my $defaultstringargs;
	my $defaultcplusplusparams;
	my $defaultrubyparams;
	my $defaultjniFunction;
	my $defaultjniparams;
	my $defaultjavaargs;


	$name = $m->{astNodeName} ;
	my $type = $m->{NodeType};

	my $docnode = $m->{DocNode};

	if ( $m->{ReturnType} =~ /~/ ) {
		$name = "~".$name;
	}

	if( $type eq "method" && $m->{Access} ne "private" && $m->{Access} ne "protected" && $m->{Access} ne "private_slots" && $m->{Access} ne "signals" ) {
		if ( $m->{ReturnType} =~ /[<>]/ || $m->{Params} =~ /[<>]/  || $m->{Params} =~ /\.\.\./  ) {
			return;
		}

		my $returnType = $m->{ReturnType};

		$returnType =~ s/const\s*//;
		$returnType =~ s/inline\s*//;
		$returnType =~ s/\s*([,\*\&])\s*/$1/;
		$returnType =~ s/^\s*//;
		$returnType =~ s/\s*$//;

		if ( $returnType ne "" && cplusplusToRuby($returnType) eq () ) {
			$returnType =~ s/^.*::.*$/int/;
		} else {
			$returnType = cplusplusToRuby($returnType);
		}

		$rubyFunction = $name;
		$rubyFunction =~ s/_/_1/g;

		# TODO port to $m->{ParamList}
		$cplusplusparams =~ s/\s+/ /g;
		$cplusplusparams =~ s/\s*([,\*\&])\s*/$1 /g;
		$cplusplusparams =~ s/^\s*$//;
		$cplusplusparams =~ s/^\s*void\s*$//;
		my $argId = 0;
		my @cargs = kdocUtil::splitUnnested(",", $cplusplusparams);
		$cplusplusparams = "";
		foreach my $arg ( @cargs ) {
			my $argType;
			my $cargType;
			my $rubyargType;
			$arg =~ s/\s*([^\s].*[^\s])\s*/$1/;
			$arg =~ s/(\w+)\[\]/\* $1/;

			# A '<arg> = <value>' default parameter
			if ( $arg =~ s/=\s*(("[^\"]*")|(\'.\')|(([-\w:]*)\s*(\|\s*[-\w]*)*(\(\w*\))?))// ) {
				my $param = $1;
				if ( $defaultcplusplusparams eq "" ) {
					$defaultjniparams = "int argc, VALUE *argv, VALUE self";
					$defaultcplusplusparams = $cplusplusparams;
					$defaultstringargs = $stringargs;
					$defaultjniFunction = $rubyFunction;
					$defaultjavaargs = $javaargs;
				}

				if ( $param =~ /(.*)::(.*)/ && kalyptusDataDict::ctypemap($param) eq ()) {
					$defaultcplusplusparams .= "$1::$2, ";
				} else {
					if ( kalyptusDataDict::ctypemap($param) eq () ) {
						if ( $param =~ /^[A-Z].*[^\)]$/ ) {
							$defaultcplusplusparams .= $class->{astNodeName}."::".$param.", ";
						} else {
							$defaultcplusplusparams .= $param.", ";
                    	}
					} else {
						$defaultcplusplusparams .= $param.", ";
					}
				}
			}

			if ( $arg =~ /^(.*)\s+(\w+)\s*$/ ) {
				$argType = $1;
			} else {
				$argType = $arg;
			}
			$argId++;
			$arg = "arg".$argId;

			$arg =~ s/^id$/identifier/;
			$argType =~ s/\s*([^\s].*[^\s])\s*/$1/;
			$argType =~ s/\s*const//g;
			$argType =~ s/^\s*//;
			$argType =~ s/([\*\&])\s*([\*\&])/$1$2/;
			$cargType = kalyptusDataDict::ctypemap($argType);
			$rubyargType = cplusplusToRuby( $argType );
			if ( $rubyargType eq "" ) {
				$rubyargType = $argType;
				$rubyargType =~ s/\&/\*/;
				$rubyargType =~ s/^.*::.*$/int/;
			}

			if ( $functionId{$name} > 0 && $argId == 1 ) {
				$rubyFunction = $rubyFunction.$rubyargType;
			}

			$arg =~ s/^super$/superClass/;
			$javaargs .= " $arg,";
			$rubyparams .= ", VALUE v".$arg;

			if ( $cargType eq "" ) {
				print "'$argType' => '$typeprefix$argType',\n";
				if ( kalyptusDataDict::ctypemap($argType) =~ /^qt_/ || kalyptusDataDict::ctypemap($argType) =~ /^kde_/ ) {
					$argType =~ s/\&//;
					$stringargs = $stringargs."\n\tGet_$rubyargType(v$arg, $arg);";
					$cplusplusparams .= "$arg,";
				} else {
					$argType =~ s/\&//;
					$cplusplusparams .= "(".$argType.") ".$arg.",";
				}
			} else {
				if ( $rubyargType =~ /String/ ) {
					if ( kalyptusDataDict::ctypemap($argType) =~ /qt_QString\s*\*/ ) {
						$stringargs = "\n\tQString *$arg;".$stringargs;
						$stringargs = $stringargs."\n\tGet_QString(v$arg, $arg);";

						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "($argType) $arg,";
						} else {
							$cplusplusparams .= "($argType) * (QString *) $arg,";
						}
					} elsif ( kalyptusDataDict::ctypemap($argType) =~ /qt_QCString\s*\*/ ) {
						$stringargs = "\nstatic QCString * _qcstring_$arg = 0;".$stringargs;
						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "($argType) QtSupport::toQCString(env, $arg, &_qcstring_$arg),";
						} else {
							$cplusplusparams .= "($argType) * (QCString *) QtSupport::toQCString(env, $arg, &_qcstring_$arg), ";
						}
					} elsif ( kalyptusDataDict::ctypemap($argType) =~ /kde_DOMString\s*(\*)/ ) {
						$stringargs = "\nstatic DOM::DOMString * _domstring_$arg = 0;".$stringargs;
						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "(DOM::DOMString *) KDESupport::toDOMString(env, $arg, &_domstring_$arg), ";
						} else {
							$cplusplusparams .= "(DOM::DOMString) * (DOM::DOMString *) KDESupport::toDOMString(env, $arg, &_domstring_$arg), ";
						}
					} elsif ( $argType =~ /[\*].*[\*]/ ) {
						$stringargs = "\n\tchar **$arg = 0 ;".$stringargs;
						$stringargs = $stringargs."\n\tif(TYPE(v$arg) == T_ARRAY) {";
						$stringargs = $stringargs."\n\t\tint a_size = RARRAY(v$arg)->len;";
						$stringargs = $stringargs."\n\t\tchar** ar_str = ALLOC_N(char *,a_size);";
						$stringargs = $stringargs."\n\t\tfor(int i = 0; i<a_size;i++) {";
						$stringargs = $stringargs."\n\t\t\tar_str[i] = STR2CSTR(RARRAY(v$arg)->ptr[i]);";
						$stringargs = $stringargs."\n\t\t}";
						$stringargs = $stringargs."\n\t\t$arg = ar_str;";
						$stringargs = $stringargs."\n\t}else {";
						$stringargs = $stringargs."\n\t\trb_raise(rb_eTypeError, \"Expected a String Array\");";
						$stringargs = $stringargs."\n\t}";
						$cplusplusparams .= "$arg,";
					} else {
						$stringargs = "\n\t$argType $arg  = STR2CSTR(v$arg);".$stringargs;
						$cplusplusparams .= "($argType) $arg,";
					}
				} elsif ( $rubyargType =~ /byte\[\]/ ) {
						$stringargs = "\nstatic QByteArray * _qbyteArray_$arg = 0;".$stringargs;
						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "($argType) QtSupport::toQByteArray(env, $arg, &_qbyteArray_$arg), ";
						} else {
							$cplusplusparams .= "($argType) * (QByteArray *) QtSupport::toQByteArray(env, $arg, &_qbyteArray_$arg), ";
						}
				} elsif ( kalyptusDataDict::ctypemap($argType) =~ /qt_QChar\s*\*/ ) {
					$stringargs = "\nstatic QChar * _qchar_$arg = 0;".$stringargs;
					if ( $argType =~ /[\*]/ ) {
						$cplusplusparams .= "($argType) QtSupport::toQChar(env, $arg, &_qchar_$arg), ";
      				} else {
						$cplusplusparams .= "($argType) * (QChar *) QtSupport::toQChar(env, $arg, &_qchar_$arg), ";
					}
				} elsif ( $rubyargType =~ /Date/ ) {
					if ( kalyptusDataDict::ctypemap($argType) =~ /qt_QTime\s*\*/ ) {
						$stringargs = "\nstatic QTime * _qtime_$arg = 0;".$stringargs;
						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "($argType) QtSupport::toQTime(env, $arg, &_qtime_$arg), ";
                        } else {
							$cplusplusparams .= "($argType) * (QTime *) QtSupport::toQTime(env, $arg, &_qtime_$arg), ";
						}
      				}
				} elsif ( $rubyargType =~ /Calendar/ ) {
					if ( kalyptusDataDict::ctypemap($argType) =~ /qt_QDateTime\s*\*/ ) {
						$stringargs = "\nstatic QDateTime * _qdate_$arg = 0;".$stringargs;
						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "($argType) QtSupport::toQDateTime(env, $arg, &_qdate_$arg), ";
                        } else {
							$cplusplusparams .= "($argType) * (QDateTime *) QtSupport::toQDateTime(env, $arg, &_qdate_$arg), ";
						}
					} elsif ( kalyptusDataDict::ctypemap($argType) =~ /qt_QDate\s*\*/ ) {
						$stringargs = "\nstatic QDate * _qdate_$arg = 0;".$stringargs;
						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "($argType) QtSupport::toQDate(env, $arg, &_qdate_$arg), ";
                        } else {
							$cplusplusparams .= "($argType) * (QDate *) QtSupport::toQDate(env, $arg, &_qdate_$arg), ";
						}
					}
				} elsif ( $rubyargType =~ /ArrayList/ && kalyptusDataDict::ctypemap($argType) =~ /qt_(QStrList)\s*\*/ ) {
						$stringargs = "\nstatic $1 * _qlist_$arg = 0;".$stringargs;
						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "($argType) QtSupport::toQStrList(env, $arg, &_qlist_$arg), ";
						} else {
							$cplusplusparams .= "($argType) * (QStrList *) QtSupport::toQStrList(env, $arg, &_qlist_$arg), ";
						}
				} elsif ( $rubyargType =~ /ArrayList/ && kalyptusDataDict::ctypemap($argType) =~ /qt_(QStringList)\s*\*/ ) {
						$stringargs = "\nstatic $1 * _qlist_$arg = 0;".$stringargs;
						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "($argType) QtSupport::toQStringList(env, $arg, &_qlist_$arg), ";
						} else {
							$cplusplusparams .= "($argType) * (QStringList *) QtSupport::toQStringList(env, $arg, &_qlist_$arg), ";
						}
				} elsif ( $rubyargType =~ /ArrayList/ && kalyptusDataDict::ctypemap($argType) =~ /kde_(KFileItemList)\s*\*/ ) {
						$stringargs = "\nstatic $1 * _qlist_$arg = 0;".$stringargs;
						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "($argType) KDESupport::toKFileItemList(env, $arg, &_qlist_$arg), ";
						} else {
							$cplusplusparams .= "($argType) * (KFileItemList *) KDESupport::toKFileItemList(env, $arg, &_qlist_$arg), ";
						}
				} elsif ( $rubyargType =~ /ArrayList/ && kalyptusDataDict::ctypemap($argType) =~ /kde_(QCStringList)\s*\*/ ) {
						$stringargs = "\nstatic $1 * _qlist_$arg = 0;".$stringargs;
						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "($argType) KDESupport::toQCStringList(env, $arg, &_qlist_$arg), ";
						} else {
							$cplusplusparams .= "($argType) * (QCStringList *) KDESupport::toQCStringList(env, $arg, &_qlist_$arg), ";
						}
				} elsif ( $rubyargType =~ /ArrayList/ && kalyptusDataDict::ctypemap($argType) =~ /kde_(KURL_List)\s*\*/ ) {
						$stringargs = "\nstatic KURL::List * _qlist_$arg = 0;".$stringargs;
						if ( $argType =~ /[\*]/ ) {
							$cplusplusparams .= "($argType) KDESupport::toKURLList(env, $arg, &_qlist_$arg), ";
						} else {
							$cplusplusparams .= "($argType) * (KURL::List *) KDESupport::toKURLList(env, $arg, &_qlist_$arg), ";
						}
				} elsif ( $rubyargType =~ /^int\[\]/  ) {
					$cplusplusparams .= " ($argType) QtSupport::toIntPtr(env, $arg), ";
				} elsif ( $rubyargType =~ /double\[\]/  ) {
					$cplusplusparams .= " ($argType) QtSupport::toDoublePtr(env, $arg), ";
				} elsif ( $rubyargType =~ /short\[\]/  ) {
					$cplusplusparams .= " ($argType) QtSupport::toShortPtr(env, $arg), ";
				} elsif ( $argType =~ /QPaintDevice/  ) {
					$cplusplusparams .= "($argType) QtSupport::paintDevice(env, $arg), ";
				} elsif ( kalyptusDataDict::ctypemap($argType) =~ /^qt_/ || kalyptusDataDict::ctypemap($argType) =~ /^kde_/ ) {
					$stringargs = "\n\t$rubyargType *$arg;".$stringargs;
					if ( $argType =~ /^([^\*\&]*)&?$/ ) {
						$stringargs = $stringargs."\n\tGet_$rubyargType(v$arg, $arg);";
						$cplusplusparams .= "($argType)*$arg,";
					} else {
						$stringargs = $stringargs."\n\tGet_$rubyargType(v$arg, $arg);";
						$cplusplusparams .= "$arg,";
					}
				} elsif ( $argType =~ /^[A-Z][^:]*$/ && $cargType eq "int" && kalyptusDataDict::ctypemap($class->{astNodeName}."::".$argType) ne "" ) {
					$stringargs = $stringargs."\n\tint $arg = NUM2INT(v$arg);";
					$cplusplusparams .= "(".$class->{astNodeName}."::$argType)$arg,";
				} elsif ( $argType =~ /^\s*WFlags\s*$/ ) {
					$stringargs = $stringargs."\n\tQWidget::WFlags $arg = NUM2UINT(v$arg);";
					$cplusplusparams .= "$arg,";
				} elsif ( $argType =~ /^\s*ArrowType\s*$/ ) {
					$stringargs = $stringargs."\n\tQt::ArrowType $arg = NUM2INT(v$arg);";
					$cplusplusparams .= "$arg,";
				} elsif ( $argType =~ /^\s*Orientation\s*$/ ) {
					$stringargs = $stringargs."\n\tQt::Orientation $arg = NUM2INT(v$arg);";
					$cplusplusparams .= "$arg,";
				} elsif ( $argType =~ /^\s*BrushStyle\s*$/ ) {
					$stringargs = $stringargs."\n\tQt::BrushStyle $arg = NUM2INT(v$arg);";
					$cplusplusparams .= "$arg,";
				} elsif ( $argType =~ /^\s*BGMode\s*$/ ) {
					$stringargs = $stringargs."\n\tQt::BGMode $arg = NUM2INT(v$arg);";
					$cplusplusparams .= "$arg,";
				} elsif ( $argType =~ /^\s*PenCapStyle\s*$/ ) {
					$stringargs = $stringargs."\n\tQt::PenCapStyle $arg = NUM2INT(v$arg);";
					$cplusplusparams .= "$arg,";
				} elsif ( $argType =~ /^\s*PenStyle\s*$/ ) {
					$stringargs = $stringargs."\n\tQt::PenStyle $arg = NUM2INT(v$arg);";
					$cplusplusparams .= "$arg,";
				} elsif ( $argType =~ /^\s*PenJoinStyle\s*$/ ) {
					$stringargs = $stringargs."\n\tQt::PenJoinStyle $arg = NUM2INT(v$arg);";
					$cplusplusparams .= "$arg,";
				} elsif ( $argType =~ /^\s*RasterOp\s*$/ ) {
					$stringargs = $stringargs."\n\tQt::RasterOp $arg = NUM2INT(v$arg);";
					$cplusplusparams .= "$arg,";
				} elsif ( $argType =~ /^\s*TextFormat\s*$/ ) {
					$stringargs = $stringargs."\n\tQt::TextFormat $arg = NUM2INT(v$arg);";
					$cplusplusparams .= "$arg,";
				} elsif ( $argType =~ /^\s*QDragMode\s*$/ ) {
					$stringargs = $stringargs."\n\tQDragObject::DragMode $arg = NUM2INT(v$arg);";
					$cplusplusparams .= "$arg,";
				} elsif ( $argType =~ /^\s*GUIStyle\s*$/ ) {
					$stringargs = $stringargs."\n\tQt::GUIStyle $arg = NUM2INT(v$arg);";
					$cplusplusparams .= "$arg,";
				} elsif ( $argType =~ /^\s*Type\s*$/ ) {
					$stringargs = $stringargs."\n\tQEvent::Type $arg = NUM2INT(v$arg);";
					$cplusplusparams .= "$arg,";
				} elsif ( $argType =~ /^([^\*\&]*)&$/ ) {
					$cplusplusparams .= "($argType) * ($1 *) $arg, ";
				} elsif ( $argType =~ /^bool/ ) {
					$stringargs = "\n\t$argType $arg;".$stringargs;
					$stringargs = $stringargs."\n\t$arg = RTEST(v$arg);";
					$cplusplusparams .= "$arg,";
				} elsif ( kalyptusDataDict::ctypemap($argType) =~ /^unsigned long/ ) {
					$stringargs = "\n\t$argType $arg;".$stringargs;
					$stringargs = $stringargs."\n\t$arg = NUM2ULONG(v$arg);";
					$cplusplusparams .= "$arg,";
				} elsif ( kalyptusDataDict::ctypemap($argType) =~ /^long/ ) {
					$stringargs = "\n\t$argType $arg;".$stringargs;
					$stringargs = $stringargs."\n\t$arg = NUM2LONG(v$arg);";
					$cplusplusparams .= "$arg,";
				} elsif ( kalyptusDataDict::ctypemap($argType) =~ /^unsigned int/ ) {
					$stringargs = "\n\t$argType $arg;".$stringargs;
					$stringargs = $stringargs."\n\t$arg = NUM2UINT(v$arg);";
					$cplusplusparams .= "$arg,";
				} elsif ( kalyptusDataDict::ctypemap($argType) =~ /^int/ ) {
					$stringargs = "\n\t$argType $arg;".$stringargs;
					$stringargs = $stringargs."\n\t$arg = NUM2INT(v$arg);";
					$cplusplusparams .= "$arg,";
				} elsif ( kalyptusDataDict::ctypemap($argType) =~ /^unsigned short/  ) {
					$stringargs = "\n\t$argType $arg;".$stringargs;
					$stringargs = $stringargs."\n\t$arg = NUM2USHRT(v$arg);";
					$cplusplusparams .= "$arg,";
				} elsif ( kalyptusDataDict::ctypemap($argType) =~ /^short/ ) {
					$stringargs = "\n\t$argType $arg;".$stringargs;
					$stringargs = $stringargs."\n\t$arg = NUM2SHRT(v$arg);";
					$cplusplusparams .= "$arg,";
				} elsif ( kalyptusDataDict::ctypemap($argType) =~ /^unsigned char/ ) {
					$stringargs = "\n\t$argType $arg;".$stringargs;
					$stringargs = $stringargs."\n\t$arg = NUM2UINT(v$arg);";
					$cplusplusparams .= "$arg,";
				} elsif ( kalyptusDataDict::ctypemap($argType) =~ /^char/ ) {
					$stringargs = "\n\t$argType $arg;".$stringargs;
					$stringargs = $stringargs."\n\t$arg = NUM2INT(v$arg);";
					$cplusplusparams .= "$arg,";
				} elsif ( kalyptusDataDict::ctypemap($argType) =~ /^double/ ) {
					$stringargs = "\n\t$argType $arg;".$stringargs;
					$stringargs = $stringargs."\n\t$arg = NUM2DBL(v$arg);";
					$cplusplusparams .= "$arg,";
				} elsif ( kalyptusDataDict::ctypemap($argType) =~ /^float/ ) {
					$stringargs = "\n\t$argType $arg;".$stringargs;
					$stringargs = $stringargs."\n\t$arg = (float) NUM2DBL(v$arg);";
					$cplusplusparams .= "$arg,";
				} else {
					$stringargs = "\n\t$argType $arg;".$stringargs;
					$cplusplusparams .= "($argType) $arg,";
				}
			}

		}
		$cplusplusparams =~ s/\s*,\s*$//;
		$javaargs =~ s/\s*,\s*$//;

		$defaultcplusplusparams =~ s/\s*,\s*$//;
		$defaultrubyparams =~ s/\s*,\s*$//;
		$defaultjavaargs =~ s/\s*,\s*$//;

		my $flags = $m->{Flags};

		if ( !defined $flags ) {
			warn "Method ".$m->{astNodeName}.  " has no flags\n";
		}


		my $extra = "";
		$extra .= "static " if $flags =~ "s";
		if ( $name =~ /operator/ ) {
			return;
		}

		if ( $m->{Access} =~ /protected/ && $name ne $class->{astNodeName}  ) {
			if ( $name =~ /.*Event$/ ) {
				return;
			}
			
			if ( $class->{Pure} ) {
				return;
			}

			$name = "protected_".$name;
		}

		$m->{Access} =~ /([^_]*)(.*)?\s*/;
		$javaaccess = $1;
		if ( $extra =~ /static/ ) {
			$javaaccess .= " static";
		}
		$javaaccess .= " native";

		if ( $name eq $class->{astNodeName} ) {
			if ( exists $class->{Pure} ) {
				return;
            }
			# Constructor
			print RUBYSOURCE "\nstatic VALUE\n_wrap_new_", $class->{astNodeName}, "($rubyparams)", " {$stringargs\n";
			print RUBYSOURCE "\t", $class->{astNodeName}, " *result;\n" ;
			print RUBYSOURCE "\tVALUE vresult = Qnil;\n" ;
			print RUBYSOURCE "\tresult = (", $class->{astNodeName}, " *)new", $class->{astNodeName}, "($cplusplusparams);\n" ;
			print RUBYSOURCE "\tCHECK_PTR(result);\n";
			print RUBYSOURCE "\tvresult = Wrap_", $class->{astNodeName}, "(self, result);\n";
			print RUBYSOURCE "\treturn vresult\n";
			print RUBYSOURCE "}\n";

		}
		elsif ( $m->{ReturnType} =~ /~/ ) {
			# Destructor
			print RUBYSOURCE "\nstatic void\nfree_", $class->{astNodeName}, "(", $class->{astNodeName}, " *arg0) {\n";
			if ( $class->{astNodeName} =~ /^QWidget$/  || defined kdocAstUtil::findOverride( $rootnode, $class, "parentWidget" ) ) {
				print RUBYSOURCE "\tif (arg0->parentWidget(TRUE) == (QWidget *) 0) {\n";
				print RUBYSOURCE "\t\tdelete arg0;\n\t}\n}\n";
			} elsif ( $class->{astNodeName} =~ /^QObject$/  || defined kdocAstUtil::findOverride( $rootnode, $class, "parent" ) ) {
				print RUBYSOURCE "\tif (arg0->parent() == (QObject *) 0) {\n";
				print RUBYSOURCE "\t\tdelete arg0;\n\t}\n}\n";
			} else {
				print RUBYSOURCE "\tdelete arg0;\n}\n";
			}
		} else {
			# Class or instance method
			my $selfstring;

 			print RUBYSOURCE "\nstatic VALUE\n_wrap_", $class->{astNodeName}, "_$rubyFunction", "($rubyparams) {\n";
			
			if ( $javaaccess =~ /static/ ) {
 				$selfstring = $class->{astNodeName}."::";
			} else {
 				print RUBYSOURCE "\t", $class->{astNodeName}, " *arg0 ;$stringargs\n";
 				print RUBYSOURCE "\tGet_", $class->{astNodeName}, "(self, arg0);\n";
	 			$selfstring = "arg0->";
			}

			if ( $returnType eq "void" ) {
	 			print RUBYSOURCE "\t$selfstring$name($cplusplusparams);\n";
				print RUBYSOURCE "\treturn Qnil;\n}\n" ;
			} elsif ( $returnType =~ /int/ ) {
	 			print RUBYSOURCE "\tint result;\n\tVALUE vresult = Qnil;\n";
	 			print RUBYSOURCE "\tresult = (int )$selfstring$name($cplusplusparams);\n";
				print RUBYSOURCE "\tvresult = INT2NUM(result);\n" ;
				print RUBYSOURCE "\treturn vresult;\n}\n" ;
			} elsif ( $returnType =~ /boolean/ ) {
	 			print RUBYSOURCE "\tbool result;\n\tVALUE vresult = Qnil;\n";
	 			print RUBYSOURCE "\tresult = (bool )$selfstring$name($cplusplusparams);\n";
				print RUBYSOURCE "\tvresult = (result ? Qtrue : Qfalse);\n" ;
				print RUBYSOURCE "\treturn vresult;\n}\n" ;
			} elsif ( $returnType =~ /String/ ) {
				if ( $m->{ReturnType} =~ /QString\s*[\*\&]?\s*$/ ) {
					print RUBYSOURCE "\tQString *result = new QString ($selfstring$name\($cplusplusparams\));\n";
					print RUBYSOURCE "\tCHECK_PTR(result);\n";
					print RUBYSOURCE "\tvresult = Wrap_QString(cQString, result);\n";
					print RUBYSOURCE "\treturn vresult;\n}\n";
				} elsif ( $m->{ReturnType} =~ /QCString\s*[\*\&]?\s*$/ ) {
					print RUBYSOURCE "\t_qcstring = $selfstring$name($cplusplusparams);\n";
					if ( $m->{ReturnType} =~ /\*/ ) {
						print RUBYSOURCE "\treturn QtSupport::fromQCString(env, _qcstring);\n}\n";
                    			} else {
						print RUBYSOURCE "\treturn QtSupport::fromQCString(env, &_qcstring);\n}\n";
					}
				} elsif ( $m->{ReturnType} =~ /(DOM::)?DOMString\s*([\*\&]?)\s*$/ ) {
					print RUBYSOURCE "\t_domstring = $selfstring$name\($cplusplusparams\);\n";
					if ( $m->{ReturnType} =~ /\*/ ) {
						print RUBYSOURCE "\treturn KDESupport::fromDOMString(env, _domstring);\n}\n";
					} else {
						print RUBYSOURCE "\treturn KDESupport::fromDOMString(env, &_domstring);\n}\n";
					}
				} else {
	 				print RUBYSOURCE "\tchar *result;\n\tVALUE vresult = Qnil;\n";
	 				print RUBYSOURCE "\tresult = (char *)$selfstring$name($cplusplusparams);\n";
					print RUBYSOURCE "\tvresult = rb_str_new2(result);\n" ;
					print RUBYSOURCE "\treturn vresult;\n}\n" ;
				}
			} elsif ( $returnType =~ /byte\[\]/ ) {
					print RUBYSOURCE "\t_qbyteArray = $selfstring$name\($cplusplusparams\);\n";
					if ( $m->{ReturnType} =~ /\*/ ) {
						print RUBYSOURCE "\treturn QtSupport::fromQByteArray(env, _qbyteArray);\n}\n";
                    } else {
						print RUBYSOURCE "\treturn QtSupport::fromQByteArray(env, &_qbyteArray);\n}\n";
					}
			} elsif ( $returnType =~ /Date/ ) {
				if ( $m->{ReturnType} =~ /(QTime)\s*([\*\&])?\s*$/ ) {
					print RUBYSOURCE "\t_qtime = $selfstring$name($cplusplusparams);\n";
					if ( $2 eq "\*" ) {
						print RUBYSOURCE "\treturn (jobject) QtSupport::from$1(env, ($1 *) _qtime);\n}\n";
					} else {
						print RUBYSOURCE "\treturn (jobject) QtSupport::from$1(env, ($1 *) &_qtime);\n}\n";
					}
				}
			} elsif ( $m->{ReturnType} =~ /(QChar)\s*([\*\&])?\s*$/ ) {
				print RUBYSOURCE "\t_qchar = $selfstring$name($cplusplusparams);\n";
				if ( $2 eq "\*" ) {
					print RUBYSOURCE "\treturn (jchar) QtSupport::fromQChar(env, (QChar *) _qchar);\n}\n";
				} else {
					print RUBYSOURCE "\treturn (jchar) QtSupport::fromQChar(env, (QChar *) &_qchar);\n}\n";
				}
			} elsif ( $returnType =~ /Calendar/ ) {
				if ( $m->{ReturnType} =~ /(QDateTime|QDate)\s*([\*\&])?\s*$/ ) {
					print RUBYSOURCE "\t_qdate = $selfstring$name($cplusplusparams);\n";
					if ( $2 eq "\*" ) {
						print RUBYSOURCE "\treturn (jobject) QtSupport::from$1(env, ($1 *) _qdate);\n}\n";
					} else {
						print RUBYSOURCE "\treturn (jobject) QtSupport::from$1(env, ($1 *) &_qdate);\n}\n";
					}
				}
			} elsif ( $returnType =~ /ArrayList/ ) {
				if ( $m->{ReturnType} =~ /(QStrList|QStringList|QWidgetList|QDomNodeList|QObjectList)\s*([\*\&])?\s*$/ ) {
					print RUBYSOURCE "\t_qlist = $selfstring$name($cplusplusparams);\n";
					if ( $2 eq "\*" ) {
						print RUBYSOURCE "\treturn (jobject) QtSupport::arrayWith$1(env, ($1 *) _qlist);\n}\n";
					} else {
						print RUBYSOURCE "\treturn (jobject) QtSupport::arrayWith$1(env, ($1 *) &_qlist);\n}\n";
					}
				} elsif ( $m->{ReturnType} =~ /(QCStringList|DOM::NodeList|NodeList|DOM::StyleSheetList|StyleSheetList|DOM::MediaList|MediaList|KFileItemList|KFileViewItemList|OfferList|KURL::List)\s*([\*\&])?\s*$/ ) {
					print RUBYSOURCE "\t_qlist = $selfstring$name($cplusplusparams);\n";
					if ( $2 eq "\*" ) {
						print RUBYSOURCE "\treturn (jobject) KDESupport::arrayWith$1(env, ($1 *) _qlist);\n}\n";
					} else {
						print RUBYSOURCE "\treturn (jobject) KDESupport::arrayWith$1(env, ($1 *) &_qlist);\n}\n";
					}
			   }
			} elsif ( $m->{ReturnType} =~ /(QSqlIndex|QSqlRecord|QSqlQuery|QSqlError)\s*$/ ) {
				print RUBYSOURCE "\t_qsql = $selfstring$name($cplusplusparams);\n";
				print RUBYSOURCE "\treturn (jobject) QtSupport::objectForQtKey(env, (void *)new $1(_qsql), \"org.kde.qt.$1\", TRUE);\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QDateTime\s*$/ ) {
				print RUBYSOURCE "\tQDateTime _dt= $selfstring$name($cplusplusparams);\n" ;
				print RUBYSOURCE "\treturn (jobject) QtSupport::objectForQtKey(env, (void *)new QDateTime (_dt.date(),_dt.time()), \"org.kde.qt.QDateTime\", TRUE);\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QDate\s*$/ ) {
				print RUBYSOURCE "\tQDate _d= $selfstring$name($cplusplusparams);\n" ;
				print RUBYSOURCE "\treturn (jobject) QtSupport::objectForQtKey(env, (void *)new QDate(_d.year(),_d.month(),_d.day(), \"org.kde.qt.QDate\", TRUE));\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*(const)?\s*?QTime\s*$/ ) {
				print RUBYSOURCE "\tQTime _t= $selfstring$name($cplusplusparams);\n" ;
				print RUBYSOURCE "\treturn (jobject) QtSupport::objectForQtKey(env, (void *)new QTime(_t.hour(),_t.minute(),_t.second(),_t.msec()), \"java.util.Date\", TRUE);\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*int\s*\*\s*$/ ) {
				print RUBYSOURCE "\treturn ($jnireturntype) QtSupport::fromIntPtr(env, $selfstring$name($cplusplusparams));\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*double\s*\*\s*$/ ) {
				print RUBYSOURCE "\treturn ($jnireturntype) QtSupport::fromDoublePtr(env, $selfstring$name($cplusplusparams));\n}\n" ;
			} elsif ( $m->{ReturnType} =~ /^\s*(inline)?\s*short\s*\*\s*$/ ) {
				print RUBYSOURCE "\treturn ($jnireturntype) QtSupport::fromShortPtr(env, $selfstring$name($cplusplusparams));\n}\n" ;
			} else {
				print RUBYSOURCE "\t$returnType *result ;\n\tVALUE vresult = Qnil;\n";
				if ( $m->{ReturnType} =~ /\&/ ) {
				 	print RUBYSOURCE "\t".$m->{ReturnType}." result_ref = $selfstring$name($cplusplusparams);\n";
		 			print RUBYSOURCE "\tresult = ($returnType *) &result_ref;\n";
					print RUBYSOURCE "\tvresult = ((result) ? Data_Wrap_Struct( c", $returnType, ",0,0, result) : Qnil);\n" ;
				} elsif ( $m->{ReturnType} =~ /\*/ ) {
		 			print RUBYSOURCE "\tresult = ($returnType *)$selfstring$name($cplusplusparams);\n";
					print RUBYSOURCE "\tvresult = ((result) ? Data_Wrap_Struct( c", $returnType, ",0,0, result) : Qnil);\n" ;
				} else {
		 			print RUBYSOURCE "\tresult = new $returnType ($selfstring$name($cplusplusparams));\n\tCHECK_PTR(result);\n";
					print RUBYSOURCE "\tvresult = Wrap_$returnType(c$returnType, result);\n" ;
				}

				print RUBYSOURCE "\treturn vresult;\n}\n" ;
			}
		}
	}

}

sub generateClassMethodForEnum
{
	my( $class, $m ) = @_;

	if( $m->{NodeType} eq "enum" ) {
		my $enum = $m->{astNodeName};
		my @enums = split(",", $m->{Params});
		my $enumCount = 0;
		foreach my $enum ( @enums ) {
			$enum =~ s/\s//g;
			$enum =~ s/::/./g;
			if ( $enum =~ /(.*)=(.*)/ ) {
				print RUBYSOURCE "\trb_define_const(c", $class->{astNodeName}, ", \"$1\", INT2NUM($2));\n";
			} else {
				print RUBYSOURCE "\trb_define_const(c", $class->{astNodeName}, ", \"$enum\", INT2NUM($enumCount));\n";
				$enumCount++;
			}
		}
	} elsif( $m->{NodeType} eq "method" && $m->{Access} ne "private" && $m->{Access} ne "protected" && $m->{Access} ne "private_slots" && $m->{Access} ne "signals" ) {
		if ($class->{astNodeName} eq $m->{astNodeName}) {
			print RUBYSOURCE "\trb_define_singleton_method(c", $class->{astNodeName}, ", \"new\", VALUEFUNC(_wrap_new_", $m->{astNodeName}, "), ", $functionId{$m->{astNodeName}}, ");\n";
		} else {
			print RUBYSOURCE "\trb_define_method(c", $class->{astNodeName}, ", \"", $m->{astNodeName}, "\", VALUEFUNC(_wrap_", $class->{astNodeName}, "_", $m->{astNodeName}, "), ", $functionId{$m->{astNodeName}}, ");\n";
		}
	}

}


=head2 printIndexEntry

	Parameters: member node

	Prints an index entry for a single node.

	TODO: stub

=cut

sub printIndexEntry
{
	my ( @node ) = @_;
}



sub writeSrcHTML
{
	my ( $outfile, $infile ) = @_;

	open ( OUT, ">$outfile" ) || die "Couldn't open $outfile for".
			"writing.\n";


	close OUT;
}

1;




