/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/* Galeon includes */
#include "galeon.h"
#include "mozilla_i18n.h"
#include "misc.h"
#include "prefs.h"
#include "history.h"
#include "window.h"
#include "mime.h"
#include "session.h"
#include "bookmarks.h"
#include "mozilla.h"
#include "file-operations.h"
#include "filepicker.h"

/* system includes */
#include <errno.h>
#include <dirent.h>

#include <stdlib.h>
#include <string.h>
#include <gtk/gtkmain.h>
#include <gtk/gtkrc.h>
#include <gtk/gtkcheckmenuitem.h>
#include <gtk/gtkselection.h>
#include <libgnomeui/gnome-dialog-util.h>
#include <libgnomeui/gnome-uidefs.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-url.h>
#include <libgnome/gnome-util.h>
#include <libgnome/gnome-exec.h>
#include <libgnome/gnome-config.h>

/* GNOME includes */
#include <libgnomevfs/gnome-vfs.h>
#include <libgnomevfs/gnome-vfs-mime.h>
#include <gdk-pixbuf/gdk-pixbuf.h>
#include <gmodule.h>

/* external function prototypes imported from libxml1 */
extern int isolat1ToUTF8(unsigned char* out, int outlen,
			 const unsigned char* in, int inlen);
extern int UTF8Toisolat1(unsigned char* out, int outlen,
			 const unsigned char* in, int inlen);

/** Styles for coloured text */
GtkStyle *red_text_style;
GtkStyle *yellow_text_style;
GtkStyle *green_text_style;
GtkStyle *blue_text_style;

/* list of known protocols */
const char *known_protocols[] =
{
	"http",
	"https",
	"file",
	"about",
	"chrome",
	"resource",
	"javascript",
	"jar",
	"view-source",
       	NULL /* terminator: must be last! */
};

/* Local prototypes */
void save_url_with_dir (const gchar *url, const gchar *directory);
void save_url_without_dir (const gchar *url);
static void save_url_with_gtm (const gchar* url, const gchar *directory);
static void save_url_with_command_line (const gchar *url, const gchar *command,
					const gchar *directory);
static void unrecognised_protocol_cb(gint reply, gpointer url);
static gboolean is_punctuation (gchar c);
static void favicon_download_completed_cb (gpointer data);
static void glade_signal_connect_func (const gchar *cb_name, GtkObject *obj, 
				       const gchar *signal_name, 
				       const gchar *signal_data,
				       GtkObject *conn_obj, 
				       gboolean conn_after,
				       gpointer user_data);
static gchar* get_first_directory (const gchar *start_dir);

/**
 * glade_lookup_widget
 */
GtkWidget *
glade_lookup_widget (GtkWidget *widget, const gchar *widget_name)
{
	GtkWidget *found_widget, *parent;
	GladeXML *xml = NULL;

	while (xml == NULL)
	{
		/* the following line is to allow to override the
		 * GladeXML object or to set it to an object which was
		 * not created by libglade. Useful for popup menus */
		xml = gtk_object_get_data (GTK_OBJECT(widget), "widget_tree");
		if (!xml) xml = glade_get_widget_tree(widget);

		if (GTK_IS_MENU(widget))
			parent = gtk_menu_get_attach_widget (GTK_MENU(widget));
		else
			parent = widget->parent;

		if (parent == NULL) 
			break;

		widget = parent;
	}

	found_widget = glade_xml_get_widget(xml, widget_name);

	if (!found_widget)
		g_warning ("Widget not found: %s", widget_name);
 
	return found_widget;
}

/**
 * user_file: returns the pathname of galeon shared files (e.g., galeon.glade)
 *
 * fname: just the filename with no path information
 * critical: critical file? (halt if not found)
 */
gchar *
user_file (gchar *fname, gboolean critical)
{
	static GHashTable *already_found = NULL;
	gchar *alternative[6];
	gchar *file;
	gint i;
	
	/* create cache hash table if it doesnt already exist */
	if (already_found == NULL)
	{
		already_found = g_hash_table_new (g_str_hash, g_str_equal);
	}

        /* Have we already found this? */
	file = g_hash_table_lookup (already_found, fname);
	if (file != NULL) 
	{
		return g_strdup (file);
	}

	/* try the default */
	file = g_strconcat (g_get_home_dir (), "/.galeon/", fname, NULL);
	
	/* success? */
	if (g_file_exists (file))
	{
		return file;
	}
	g_free(file);

	/* specify alternate locations in order of precedence */
	alternative[0] = g_strdup (fname);
	alternative[1] = g_strconcat ("../", fname, NULL);
	alternative[2] = g_strconcat ("ui/", fname, NULL);
	alternative[3] = g_strconcat ("../ui/", fname, NULL);
	alternative[4] = g_strconcat (SHARE_DIR "/", fname, NULL);
	alternative[5] = NULL;  /* NULL terminator needed */
	
	/* select one of the alternatives */
	file = NULL;
	for (i = 0; alternative[i] != NULL; i++)
	{
		if (file == NULL && g_file_exists (alternative[i])) 
		{
			file = alternative[i];
                        /* don't warn for the install default */
			if (i != 4)
			{
				g_warning ("Using %s (usually OK)\n", file);
			}
		}
		else
		{
			/* free unused string */
			g_free (alternative[i]);
		}
	}

	/* check for success */
	if (file != NULL)
	{
		/* add it to the set of found files */
		g_hash_table_insert (already_found, g_strdup (fname), 
				     g_strdup (file));
	}
	/* if nothing then theres an error */
	else if (critical)
	{
		g_error(_("%s not found"), fname);
	}

	/* return result */
	return file;
}

/**
 * glade_widget_new: build a new widget of the provided name, with all
 * signals attached and data set to the provided parameter.
 */
GladeXML *
glade_widget_new (gchar *file, const gchar *widget_name, 
		  GtkWidget **root, gpointer data)
{
	GladeXML *gxml;
	gchar *glade_file;

	glade_file = user_file (file, TRUE);
	g_return_val_if_fail (glade_file != NULL, NULL);

	/* build the widget */
	/* note that libglade automatically caches the parsed file,
	 * so we don't need to worry about the efficiency of this */
	gxml = glade_xml_new (glade_file, widget_name);
	g_free (glade_file);
	g_return_val_if_fail (gxml != NULL, NULL);

	/* lookup the root widget if requested */
	if (root != NULL)
	{
		*root = glade_xml_get_widget (gxml, widget_name);
	}

	/* connect signals and data */
	glade_xml_signal_autoconnect_full
		(gxml, glade_signal_connect_func, data);

	/* return xml document for subsequent widget lookups */
	return gxml;
}

/*
 * glade_signal_connect_func: used by glade_xml_signal_autoconnect_full
 */
static void
glade_signal_connect_func (const gchar *cb_name, GtkObject *obj, 
			   const gchar *signal_name, const gchar *signal_data,
			   GtkObject *conn_obj, gboolean conn_after,
			   gpointer user_data)
{
	/** Module with all the symbols of the program */
	static GModule *mod_self = NULL;
	gpointer handler_func;

 	/* initialize gmodule */
	if (mod_self == NULL)
	{
		mod_self = g_module_open (NULL, 0);
		g_assert (mod_self != NULL);
	}

	/*g_print( "glade_signal_connect_func: cb_name = '%s', signal_name = '%s', signal_data = '%s'\n",
	  cb_name, signal_name, signal_data ); */
	
	if (g_module_symbol (mod_self, cb_name, &handler_func))
	{
		/* found callback */
		if (conn_obj)
		{
			if (conn_after)
			{
				gtk_signal_connect_object_after
					(obj, signal_name, 
					 handler_func, conn_obj);
			}
			else
			{
				gtk_signal_connect_object
					(obj, signal_name, 
					 handler_func, conn_obj);
			}
		}
		else
		{
			/* no conn_obj; use standard connect */
			gpointer data = NULL;
			
			data = user_data;
			
			if (conn_after)
			{
				gtk_signal_connect_after
					(obj, signal_name, 
					 handler_func, data);
			}
			else
			{
				gtk_signal_connect
					(obj, signal_name, 
					 handler_func, data);
			}
		}
	}
	else
	{
		g_warning("callback function not found: %s", cb_name);
	}
}

/**
 * handle_foreign_protocols: work out what protocol is specified
 * by a URL, return TRUE if we find a handler and execute it,
 * otherwise return FALSE to tell mozilla to try to load it
 */
gboolean
handle_foreign_protocols (const char *url)
{
	gint i, length;
	gchar *protocol;
	gchar *key, *result;
	gchar *url_copy;

	/* ignore any spaces preceding the url */
	while (*url == ' ' && *url != '\0')
		url++;

	/* find the colon */
	length = strlen(url);
	for (i = 0; i < length && url[i] != ':' && url[i] != '.'; i++)
		;
	
	/* did we run over? */
	if (i == length)
	{
		/* don't know what to do here so leave it to the lizard */
		return FALSE;
	}

	/* did we find a period? */
	if (url[i] == '.')
	{
		/* must be a site designation, not a protocol */
		return FALSE;
	}

	/* okay, duplicate this */
	protocol = g_strndup(url, i);

        /* see if it's definitely a known protocol */
        for (i = 0; known_protocols[i] != NULL; i++)
        {
                if (strcasecmp (protocol, known_protocols[i]) == 0)
		{
			/* fine, give it to mozilla */
			g_free(protocol);
                        return FALSE;
		}
        }

	/* if it's FTP, check if the user has setup to use mozilla */
	if (strcasecmp (protocol, "ftp") == 0 &&
	    gnome_config_get_int ("/galeon/Handlers/ftp=0") == 0)
	{
			/* give it to mozilla */
			g_free(protocol);
                        return FALSE;
	}

	/* we don't know the protocol, check to see what GNOME has */
	/* unfortunately there's not an easy function to do this... */

	/* build the config key */
	key = g_strconcat("/Gnome/URL Handlers/", protocol, "-show", NULL);
	g_free(protocol);

	/* find it */
	result = gnome_config_get_string(key);
	g_free(key);
	if (result != NULL && strlen(result) != 0)
	{
		/* we have a GNOME handler for this, pass it through
		 * and tell mozilla not to try showing it */
		g_free(result);
		gnome_url_show(url);
		return TRUE;
	}

	/* free */
	if (result)
		g_free(result);

	/* no luck, so offer the user the option of trying the
	 * default handler -- we don't do this automatically in
	 * case the default handler is erroneously set to galeon */
	result = gnome_config_get_string("/Gnome/URL Handlers/default-show");

	/* check there is a default */
	if (result == NULL || strlen(result) == 0)
	{
		/* throw the error */
		gnome_error_dialog(_("Galeon cannot handle this protocol,\n"
				     "and no GNOME default handler is set"));

		/* free */
		if (result) g_free(result);

		/* don't let mozilla try blindly */
		return TRUE;
	}
 
	/* free */
	g_free(result);

	/* offer the choice */
	url_copy = g_strdup(url);
	gnome_question_dialog(_("The protocol specified is not recognised.\n\n"
				"Would you like to try the GNOME default?"),
			      unrecognised_protocol_cb, (gpointer)url_copy);

	/* keep mozilla out of it */
	return TRUE;
}

/**
 * first_time_cb: called if the user decides to import existing
 * netscape bookmarks or not
 */
static void
unrecognised_protocol_cb(gint reply, gpointer url)
{
        /* use default GNOME URL handler if asked to */
        if (reply == GNOME_YES)
	{
		gnome_url_show((gchar *)url);
	}

	/* free the copied url */
	g_free(url);
}

/**
 * save_url
 */
void
save_url (const gchar *url)
{
	if (gnome_config_get_bool ("/galeon/Downloading/"
				   "ask_for_download_dir=true"))
	{
		gchar *dirName, *retPath = NULL;
		gboolean ret;
		dirName = g_strdup (gnome_config_get_string 
				   	("/galeon/Downloading/download_dir"));
		if (!dirName)
			dirName = g_strdup (g_get_home_dir());
		ret = show_file_picker (NULL, N_("Choose destination folder"),
				        dirName, NULL, modeGetFolder, &retPath);
		if (ret == TRUE)
		{
			save_url_with_dir (url, retPath);
		}
		g_free (dirName);
		g_free (retPath);
	}
	else
	{
		save_url_without_dir (url);
	}
}

void
save_url_with_dir (const gchar *url, const char *directory)
{
	gint use_program;
	gchar *command;

	use_program = gnome_config_get_int (CONF_DOWNLOADING_FTP_PROGRAM);
	command = gnome_config_get_string (CONF_DOWNLOADING_FTP_COMMAND);
 
	if (use_program && command != NULL)
	{
		if (strlen (command) > 0)
		{
			save_url_with_command_line (url, command, directory);
			return;
		}
		g_free (command);
	}

	/* default */
	save_url_with_gtm (url, directory);
}

void
save_url_without_dir (const gchar *url)
{
	gint use_program;
	gchar *command, *dir;

	use_program = gnome_config_get_int (CONF_DOWNLOADING_FTP_PROGRAM);
	command = gnome_config_get_string (CONF_DOWNLOADING_FTP_COMMAND);
	dir = gnome_config_get_string ("/galeon/Downloading/download_dir");
 
	if (use_program && command != NULL)
	{
		if (strlen (command) > 0)
		{
			save_url_with_command_line (url, command, dir);
			g_free (command);
			g_free (dir);
			return;
		}
		g_free (command);
	}

	/* default */
	save_url_with_gtm (url, dir);
	g_free (dir);
}

static void 
save_url_with_command_line (const gchar *url, const gchar *command,
			    const gchar *directory)
{
	char *full_command = g_strdup_printf(command, url);
	int i;

	gchar **argv = g_strsplit (full_command, " ", 100);

	for (i=0;argv[i]!=NULL;i++);

	gnome_execute_async (directory, i, argv);			 

	g_free (full_command);
	g_strfreev (argv);
}

/**
 * save_url_with_gtm
 */
static void
save_url_with_gtm (const gchar *url, const gchar *directory)
{  
	gboolean disable_auto_download;

	if (gnome_config_get_bool ("/galeon/Downloading/auto_download=true"))
		disable_auto_download = FALSE;
	else
		disable_auto_download = TRUE;

	gtm_add_url (strdup(url), (char *)directory, FALSE,
		     disable_auto_download);
		
}

/**
 * Frees an array of strings
 */ 
void 
free_string_array (char *array[], int size)
{
	int i;
	for (i = 0; i < size; i++)
		g_free (array[i]);
	g_free (array);
}

/**
 * Creates a string with a numbered/lettered accel (caller must free)
 * returns NULL if num is out of the range of acceleratable nums/letters
 */
gchar *
new_num_accel_str (gint num, gchar *text, gboolean lettersok)
{
	gchar *label = NULL;
	if (num < 9)
		label = g_strdup_printf	("_%i. %s", num+1, text);
	else if (num == 9)
		label = g_strdup_printf	("_%i. %s", 0, text);
	else if (num < 36 && lettersok)
		label = g_strdup_printf	("_%c. %s", 'a'+num-10, text);
	return label;
}

/**
 * Sets a labels text with shortcut key handling in the standard _<key> accel way
 */
void
label_set_accel_text (gchar *text, GtkWidget *label, GtkWidget *menu, 
		      GtkWidget *item)
{
	gint tmp_key = gtk_label_parse_uline (GTK_LABEL (label), text);
	GtkAccelGroup *menu_accels = 
		gtk_menu_ensure_uline_accel_group (GTK_MENU (menu));
	gtk_widget_add_accelerator (item, "activate_item", menu_accels,
			tmp_key, 0, 0);
}

/**
 * Strip the _ out of a string like gtk_label_parse_uline would do
 * caller is responsible for freeing the returned string
 */
gchar *
strip_uline_accel (const gchar *text)
{
	GString *out;
	const gchar *u, *cur = text;
	out = g_string_new (NULL);
	while ((u = strchr (cur, '_')))
	{
		if (*(u+1) == '_')
		{
			/* __ in the string is equal to _ in the output 
			 * so include the _ in the output, skip over the 
			 * second _ and continue scanning. */
			g_string_sprintfa (out, "%.*s", u - cur + 1, cur);
			cur = u + 2;
		} else {
			/* copy everything before the _ and skip over it */
			g_string_sprintfa (out, "%.*s", u - cur , cur);
			cur = u + 1;
			/* only one accel per string, so we are done now */
			break;
		}
	}
	if (cur && *cur)
	{
		/* attach rest of string */
		g_string_append (out, cur);
	}

	u = out->str;
	g_string_free (out, FALSE); /* don't free char data, caller must */
	return (gchar *)u;
}

/**
 * Escape _'s in string such that the gtk_label_parse_uline will display as normal
 * caller is responsible for freeing the returned string
 */
gchar *
escape_uline_accel (const gchar *text)
{
	GString *out;
	const gchar *u, *cur = text;
	out = g_string_new (NULL);
	while ((u = strchr (cur, '_')))
	{
		/* All we need to do is double every _, so include the _ in 
		 * the output, add another _, and continue scanning. */
		g_string_sprintfa (out, "%.*s_", u - cur + 1, cur);
		cur = u + 1;
	}
	if (cur && *cur)
	{
		/* attach rest of string */
		g_string_append (out, cur);
	}

	u = out->str;
	g_string_free (out, FALSE); /* don't free char data, caller must */
	return (gchar *)u;
}

/**
 * Creates a label with a numbered/lettered accel
 */
GtkWidget *
new_num_accel_label (gint num, gchar *origtext, gboolean lettersok, 
		     GtkWidget *menu, GtkWidget *item)
{
	gchar *text = new_num_accel_str(num, origtext, lettersok);
	if (text == NULL)
		return gtk_label_new (origtext);
	else
	{
		GtkWidget *label = gtk_label_new ("");
		label_set_accel_text (text, label, menu, item);
		g_free(text);
		return label;
	}
}

void
menu_position_under_widget (GtkMenu *menu, gint *x, gint *y, 
			    gpointer user_data)
{
	GtkWidget *w = GTK_WIDGET(user_data);
	gint width, height;
	gint screen_width, screen_height;
	GtkRequisition requisition;

	gdk_window_get_size (w->window, &width, &height);
	gdk_window_get_deskrelative_origin(w->window, x, y);
	*y = *y + height;

	gtk_widget_size_request (GTK_WIDGET (menu), &requisition);
      
	screen_width = gdk_screen_width ();
	screen_height = gdk_screen_height ();
	  
	*x = CLAMP (*x, 0, MAX (0, screen_width - requisition.width));
	*y = CLAMP (*y, 0, MAX (0, screen_height - requisition.height));
}

/**
 * copy_to_clipboard: Copies some text into the clipboard
 **/
void
copy_to_clipboard (gchar *text, GaleonEmbed *embed)
{
	gint have_selection;
	GtkWidget *window = embed->parent_window->WMain;

	/* FIXME: free previous data? */
	gtk_object_set_data (GTK_OBJECT (window),
			     "selection", g_strdup (text));
	have_selection = gtk_selection_owner_set (GTK_WIDGET (window),
					 gdk_atom_intern("CLIPBOARD",FALSE), 
					  GDK_CURRENT_TIME)&&
		         gtk_selection_owner_set (window,
					 GDK_SELECTION_PRIMARY,
					 GDK_CURRENT_TIME);
	if (!have_selection)
	{
		g_warning("Selection not found");
	}
} 

static void
galeon_close_all_windows (void)
{
	GList *wl, *next;

	/* force ourselves out of server mode -- this means that closing
	 * the last window really will quit out of galeon */
	galeon_server_mode = FALSE;

	/* close all windows: this in turn will close all embeds */
	/* at the closing of the last window galeon_exit will be called */
	for (wl = all_windows; wl != NULL;)
	{
		next = wl->next;
		window_close ((GaleonWindow *)(wl->data));
		wl = next;
	}
}

void galeon_quit (GaleonWindow *window)
{
	/* location bar history is saved automatically when the widget is
	   destroyed. we don't want to save the text currently in the entry */
	window_clear_url_entry (window);

	/* save window size if we are not in fullscreen and if 
	 * the window is not a popup */
	if (!GTK_CHECK_MENU_ITEM (window->view_fullscreen)->active)
	{	
		gnome_config_set_int("/galeon/Appearance/winwidth", 
				     window->WMain->allocation.width);
		gnome_config_set_int("/galeon/Appearance/winheight", 
				     window->WMain->allocation.height);
	}

	/* close all windows, this will also quit */
	galeon_close_all_windows ();
}

void
galeon_quit_with_session (void)
{
	gchar *filename;

	/* make the session filename string */
	filename = g_strconcat (g_get_home_dir (),
				"/.galeon/session_saved.xml", NULL);

	/* save it out */
	session_save(filename);

	/* set config marker */
	gnome_config_set_int ("/galeon/General/session_saved", 1);
	gnome_config_sync ();

	/* close all windows and exit */
	galeon_close_all_windows ();

}

void galeon_exit (void)
{
	gnome_config_set_int("/galeon/General/crashed", FALSE);

	/* shut down galeon subsystems */
	preferences_save ();
	bookmarks_exit ();
	history_exit ();
	mozilla_save_prefs ();

	/* check... */
	g_assert (g_list_length (all_embeds) == 0);
	g_assert (g_list_length (all_windows) == 0);

	/* shut down GNOME subsystmes */
	gnome_vfs_shutdown ();

	/* absolutely no more mozembeds */
	/* FIXME This cause galeon not exiting properly
	  gtk_moz_embed_pop_startup ();
	*/
	gtk_main_quit();
}

static gboolean
is_punctuation (gchar c)
{
	return (c == ' ' || c == '.' || c == '!' || c == '|' ||
		c == ',' || c == ':' || c == ';');
}

/**
 * shorten_name: try to shorten a page title (ideally to target_length or 
 * less). The heurstics here seems to give pretty good results even down
 * to quite small lengths, generally remaining comprehensible down to
 * around six to eight characters.
 */
gchar *
shorten_name(const gchar *input_name, gint target_length)
{
	gint i, j, length;
	gchar *name;
	char c;

	if (input_name == NULL)
		return (g_strdup (_("Untitled")));
	
	/* copy and clean name */
	name = g_strdup (input_name);
	g_strstrip (name);
	length = strlen (name);

	/* look for some common prefixes -- should these be translated? */
	if (gnome_config_get_bool (CONF_APPEARANCE_TABBED_PREFIX))
	{
		/* prefixes that only come by themselves */
		if (strncasecmp (name, "index of ", 9) == 0)
		{
			length -= 9;
			memmove(name, name + 9, length + 1);
		}
		else if (strncasecmp (name, "re: ", 4) == 0)
		{
			length -= 4;
			memmove(name, name + 4, length + 1);
		}
		else if (strncasecmp (name, "fwd: ", 5) == 0)
		{
			length -= 5;
			memmove(name, name + 5, length + 1);
		}
		else if (strncasecmp (name, "www.", 4) == 0)
		{
			length -= 4;
			memmove(name, name + 4, length + 1);
		}
		else 
		{
			/* prefixes that can be followed by other
			 * prefixes */
			if (strncasecmp (name, "welcome to ", 11) == 0)
			{
				length -= 11;
				memmove(name, name + 11, length + 1);
			}

			/* prefixes that follow the ones in the
			 * previous group */
			if (strncasecmp (name, "a ", 2) == 0)
			{
				length -= 2;
				memmove(name, name + 2, length + 1);
			}
			else if (strncasecmp (name, "my ", 3) == 0)
			{
				length -= 3;
				memmove(name, name + 3, length + 1);
			}
			else if (strncasecmp (name, "the ", 4) == 0)
			{
				length -= 4;
				memmove(name, name + 4, length + 1);
			}
		}

		/* remove any leading whitespace */
		g_strchug (name);
	}

	/* check length */
	length = strlen (name);
	if (length <= target_length)
	{
		return name;
	}

	/* find in name the first occurence of one of
	 * several common separators, and set it to '\0' */
	if (gnome_config_get_bool (CONF_APPEARANCE_TABBED_SEPARATOR))
	{
		gchar *first;
		gchar *str;

		/* set first to initially point to the terminating '\0'
		 * character */
		first = name + sizeof (gchar) * strlen (name);

		/* search for various separators... we can now search
		 * for ": ", becuase because we have stripped "re:" and
		 * "fwd: " in an earlier test */
		str = strstr (name, " - ");
		if (str != NULL && str < first) first = str;
		str = strstr (name, " -- ");
		if (str != NULL && str < first) first = str;
		str = strstr (name, " | ");
		if (str != NULL && str < first) first = str;
		str = strstr (name, " || ");
		if (str != NULL && str < first) first = str;
		str = strstr (name, ": ");
		if (str != NULL && str < first) first = str;
		str = strstr (name, " :: ");
		if (str != NULL && str < first) first = str;

		*first = '\0';

		/* check length */
		g_strchomp (name);
		length = strlen (name);
		if (length <= target_length)
		{
			return name;
		}
	}

	/* getting tricky, try losing a few vowels */
	/* yes, this is very anglocentric -- I don't know of any strategies
	 * for other languages (particularly non-European) -- MattA */
	if (gnome_config_get_bool (CONF_APPEARANCE_TABBED_VOWEL))
	{
		for (i = length - 1, j = length - 1; i >= 0; i--)
		{
			c = name[i];
			if (length <= target_length ||
			    (c != 'a' && c != 'e' && c != 'i' &&
			     c != 'o' && c != 'u'))
			{
				name[j--] = c;
			}
			else
			{
				length--;
			}
		}
		/* shift */
		memmove(name, name + j + 1, length + 1);

		/* short enough yet? */
		if (length <= target_length)
		{
			return name;
		}
	}

	/* argh -- try chopping out whole words */
	for (i = target_length; i > 0; i--)
	{
		if (is_punctuation (name[i]))
		{
			while (is_punctuation (name[i]))
			{
				i--;
			}
			/* this will do */
			name[i + 1] = '\0';
			g_strchomp (name);
			return name;
		}
	}

	/* just chop off and add ellipsis */
	for (i = 0; i < 3; i++)
	{
		/* don't overflow string length */
		if (name[target_length + i] == '\0')
			break;
		
		name[target_length + i] = '.';
	}
	
	/* terminate string */
	name[target_length + i] = '\0';
	
	/* return it */
	return name;
}

/**
 * initialise_colours: initialise global colour styles
 */
void
initialise_colours (void)
{
	GdkColor red, yellow, green, blue;
	GtkWidget *label;
	GtkStyle  *rcstyle;

	/* create a dummy label so we can fetch the associated rc style */
	label = gtk_label_new("");
	rcstyle = gtk_rc_get_style(label);
	gtk_widget_destroy(label);

	/* this is needed for some (broken?) themes */
	if (rcstyle == NULL)
	{
		rcstyle = gtk_style_new ();
	}
	
	/* build colour structures */
	gdk_color_parse ("#ff0000", &red);
	gdk_color_parse ("#ffff00", &yellow);
	gdk_color_parse ("#00ff00", &green);
	gdk_color_parse ("#0000ff", &blue);

	/* make styles */
	red_text_style = gtk_style_copy(rcstyle);
	yellow_text_style = gtk_style_copy(rcstyle);
	green_text_style = gtk_style_copy(rcstyle);
	blue_text_style = gtk_style_copy(rcstyle);

	/* fill in colours */
	red_text_style->fg[0] = red;
	yellow_text_style->fg[0] = yellow;
	green_text_style->fg[0] = green;
	blue_text_style->fg[0] = blue;
}

/**
 * read_line_from_file: reads a line from an opened file and returns it in a 
 * new allocated string
 */
gchar *
read_line_from_file (FILE *f)
{
	gchar *line = g_strdup ("");
	gchar *t;
	gchar *buf = g_new0 (gchar, 256);
	while ( ! ( strchr (buf, '\n') || feof (f) ) ) {
		fgets(buf, 256, f);
		t = line;
		line = g_strconcat (line, buf, NULL);
		g_free (t);
	}
	return line;
}

/**
 * g_strcasestr: test if a string b is a substring of string a, independent
 * of case.
 */
const gchar *
g_strcasestr (const gchar *a, const gchar *b)
{
	gchar *down_a;
	gchar *down_b;
	gchar *ptr;

	/* copy and lower case the strings */
	down_a = g_strdup (a);
	down_b = g_strdup (b);
	g_strdown (down_a);
	g_strdown (down_b);

	/* compare */
	ptr = strstr (down_a, down_b);

	/* free allocated strings */
	g_free (down_a);
	g_free (down_b);
	
	/* return result of comparison */
	return ptr == NULL ? NULL : (a + (ptr - down_a));
}

static gchar*
get_first_directory (const gchar *start_dir) 
{
	GnomeVFSResult rc;
	GList *list = NULL;
	GList *node;
	GnomeVFSFileInfo *info;
	gchar *prefs_dir = NULL;

	rc = gnome_vfs_directory_list_load(&list,
					   start_dir,
					   (GNOME_VFS_FILE_INFO_GET_MIME_TYPE
					    | GNOME_VFS_FILE_INFO_FORCE_FAST_MIME_TYPE
					    | GNOME_VFS_FILE_INFO_FOLLOW_LINKS),
					   NULL);

	for (node = list; node; node = node->next) {
		info = node->data;

		if (info->type == GNOME_VFS_FILE_TYPE_DIRECTORY && 
				strcmp (info->name, ".") &&
				strcmp (info->name, ".."))
		{
			prefs_dir = g_strdup_printf("%s/%s/",start_dir, info->name);
			break;
		}
	}

	gnome_vfs_file_info_list_free(list);

	return prefs_dir;
}

gchar* get_mozilla_prefs_directory () 
{
	gchar *prefs_dir = NULL;
	gchar *default_dir;

	default_dir = g_strconcat (g_get_home_dir(), MOZILLA_PREFS_DIR, NULL);

	/* descend first dir (_usually_ .mozilla/default) */
	prefs_dir = get_first_directory (default_dir);
	g_free (default_dir);
	if (!prefs_dir) return NULL;
	default_dir = prefs_dir;
	
	/* descend second dir something like (.mozilla/default/n8u3dbaou.slt) */
	prefs_dir = get_first_directory (default_dir);
	g_free (default_dir);

	return prefs_dir;
}

void
launch_external_viewer (const gchar *filename)
{
	gchar *argv[2] = { NULL, (gchar *)filename };
	gchar *prog, *msg;
	int pid;

	g_return_if_fail(filename != NULL);

	/* get the program to execute */
	prog = gnome_config_get_string (CONF_HANDLERS_EXTERNAL_SOURCE_VIEWER);
	argv[0] = prog;

	/* execute the viewer */
	pid = gnome_execute_async (NULL, 2, argv);

	/* check */
	if (pid == -1)
	{
		msg = g_strdup_printf (_("Error executing `%s': \n%s"),
				       prog, g_strerror (errno));
		gnome_error_dialog (msg);
		g_free (msg);
	}

	/* FIXME: need to detect child termination and remove the temporary
	 * file -- this used to be done with "sleep", but that's a horrible
	 * solution -- MattA */

	/* free allocated string */
	g_free (prog);
}

/**
 * xmlSetPropISOLatin1: set a property in an XML file which has value
 * encoded as ISOLatin1. This works around the strictness of libxml1.
 */
xmlAttrPtr
xmlSetPropISOLatin1 (xmlNodePtr node, 
		     const xmlChar *name, 
		     const xmlChar *value)
{
	xmlChar buffer[4096];
	gint count;

	/* check for nullification */
	if (value == NULL)
	{
		return xmlSetProp (node, name, NULL);
	}

	/* do the conversion */
	count = isolat1ToUTF8 (buffer, 4096, value, strlen (value));
	buffer[count] = '\0';
	
	/* store in the node */
	return xmlSetProp (node, name, buffer);
}

/**
 * xmlGetPropISOLatin1: get a property in an XML file which has value
 * encoded as UTF8 but was originally ISOLatin1. This works around the
 * strictness of libxml1.
 */
xmlChar *
xmlGetPropISOLatin1 (xmlNodePtr node, const xmlChar *name)
{
	xmlChar *value;
	xmlChar buffer[4096]; /* FIXME? */
	gint count;

	/* get the value */
	value = xmlGetProp (node, name);
	if (value == NULL)
	{
		return NULL;
	}

	/* do the conversion */
	count = UTF8Toisolat1 (buffer, 4096, value, strlen (value));

	/* check and do the best we can */
	if (count >= 0)
	{
		/* terminate string */
		buffer[count] = '\0';
		xmlFree (value);
		
		/* copy and return */
		return g_strdup (buffer);
	}
	else
	{
		/* return the undecoded result */
		return value;
	}
}

char *messages [5] = {NULL, N_("Looking for a bookmarks icon"), NULL};

/**
 * get_siteicon: Parse an http URL ans get its favicon if there is one
 */
void
get_siteicon (GaleonEmbed *embed, gchar *url)
{
	GnomeVFSURI *vfs_uri;
	gchar *favicon_url, *favicon_path;
	GList *source_uris = NULL, *target_uris = NULL;

	g_return_if_fail (url != NULL);
	
	vfs_uri = gnome_vfs_uri_new (url);
	g_return_if_fail (vfs_uri != NULL);
	
	if (embed == NULL || !mozilla_get_favicon_location (embed, &favicon_url))
	{
		GnomeVFSURI *vfs_favicon_uri;
		
		while (gnome_vfs_uri_has_parent (vfs_uri))
		{
			GnomeVFSURI *vfs_parent_uri;
			vfs_parent_uri = gnome_vfs_uri_get_parent (vfs_uri);
			gnome_vfs_uri_unref (vfs_uri);
			vfs_uri = vfs_parent_uri;
		} 
		vfs_favicon_uri = gnome_vfs_uri_append_file_name (vfs_uri, 
								 "favicon.ico");
		favicon_url = gnome_vfs_uri_to_string (vfs_favicon_uri, 0);
		gnome_vfs_uri_unref (vfs_favicon_uri);
	}
	else if (favicon_url == NULL)
	{
		gnome_vfs_uri_unref (vfs_uri);
		return;
	}

	gnome_vfs_uri_unref (vfs_uri);

	favicon_path = favicon_filename (favicon_url);

	source_uris = g_list_append(source_uris, favicon_url);
        target_uris = g_list_append(target_uris, favicon_path);

        file_operations_copy_move (source_uris, target_uris,
                                   FALSE,
                                   messages, 
				   favicon_download_completed_cb,
				   favicon_path);
	g_free (favicon_url);
}

static void 
favicon_download_completed_cb (gpointer data)
{
	GdkPixbuf *test;
	char *favicon_path = (char*)data;
	
	test = gdk_pixbuf_new_from_file (favicon_path);
	if (test == NULL)
	{
		gnome_vfs_unlink (favicon_path);
	}
	else
	{
		gdk_pixbuf_unref (test);
	}

	g_free(data);
}

PixmapData *
pixmap_data_new_from_file (const gchar *file)
{
	GdkPixbuf *pixbuf;
	PixmapData *pixmap_data;

	/* load pixbuf from disk */
	pixbuf = gdk_pixbuf_new_from_file (file);

	/* check... */
	if (pixbuf == NULL)
	{
		/* warn */
		g_warning ("`%s' does not exist or is not "
			   "a valid image file\n", file);

		/* callers should handle this */
		return NULL;
	}
	
	/* allocate space for a new pixmap */
	pixmap_data = g_new0 (PixmapData, 1);

	/* render the pixbuf into the pixmap data */
	gdk_pixbuf_render_pixmap_and_mask (pixbuf, &(pixmap_data->pixmap),
					   &(pixmap_data->mask), 100);

	/* lose the pixbuf */
	gdk_pixbuf_unref (pixbuf);
	
	/* return completed structure */
	return pixmap_data; 
}

gchar *
favicon_filename (gchar *favicon_url)
{
	GnomeVFSURI *vfs_uri = gnome_vfs_uri_new (favicon_url);
	gchar  *result;
	
	/* this is the RIGHT thing to do, but it breaks for the link rel icons
	gchar *slashpos, *filename;
	filename = gnome_vfs_uri_to_string (vfs_uri, 
					    GNOME_VFS_URI_HIDE_USER_NAME |
					    GNOME_VFS_URI_HIDE_PASSWORD |
					    GNOME_VFS_URI_HIDE_TOPLEVEL_METHOD);

	while ((slashpos = strstr (filename, "/")) != NULL) *slashpos = '_';

	result = g_strconcat (g_get_home_dir (), "/.galeon/favicons/",
			      filename, NULL);
	g_free (filename);
	*/
	
	/* some braindamaged handling until the situation is fixed. FIXME!! 
	 * this leaves the filename as tld_favicon.ico, which isnt right
	 * but will work until we decide to store the favicon remote location
	 * in the bookmarks file */
	result = g_strconcat (g_get_home_dir (), "/.galeon/favicons/",
			      gnome_vfs_uri_get_host_name (vfs_uri), "_",
			      "favicon.ico", NULL);
	gnome_vfs_uri_unref (vfs_uri);

	return result;
}

/* NOTE: str must be at least 256 chars long */
void
store_time_in_string (GTime t, gchar *str)
{
	struct tm stm;
	int length;

	if (t > 0)
	{
		/* convert to local time */
		localtime_r ((time_t *)&t, &stm);

		/* format into string */
		length = strftime (str, 255, _("%-m/%-d/%Y"), &stm);
		str[length] = '\0';
	}
	else
	{
		str[0] = '\0';
	}
}

gchar *
time_to_string (GTime t)
{
	gchar str[256];

	/* write into stack string */
	store_time_in_string (t, str);

	/* copy in heap and return */
	return g_strdup (str);
}

gint
xmlGetIntProp (xmlNodePtr node, const gchar *attribute)
{
	gchar *string_value;
	gint value;

	/* get the attribute as a string */
	string_value = xmlGetProp (node, attribute);

	/* convert to integer */
	/* FIXME: atoi is crap */
	value = atoi (string_value);

	/* free allocated string */
	xmlFree (string_value);

	/* return discovered value */
	return value;
}

void
lookup_widgets (GladeXML *gxml, WidgetLookup *lookup_table)
{
	gint i;
	GtkWidget *value;

	/* check */
	g_return_if_fail (gxml != NULL);
	g_return_if_fail (lookup_table != NULL);

        /* lookup all of the widgets */
        for (i = 0; lookup_table[i].name != NULL; i++)
        {
		g_return_if_fail (lookup_table[i].name != NULL);
		g_return_if_fail (lookup_table[i].ptr != NULL);

		value = glade_xml_get_widget (gxml, lookup_table[i].name);
		*(lookup_table[i].ptr) = value;
	}
}

gchar *
misc_parse_nautilus_uri (const gchar *uri)
{
	const gchar *filename;
	const gchar *mime_type;
	xmlDocPtr doc;
	gchar *url;

	/* check it's a file uri */
	if (strncmp (uri, "file:", 5) != 0)
	{
		return NULL;
	}

	/* check it exists */
	filename = uri + 5;
	if (!(g_file_exists (filename)))
	{
		return NULL;
	}

	/* get the mime type and check its a nautilus link */
	mime_type = gnome_vfs_get_file_mime_type (filename, NULL, FALSE);
	if (strcmp (mime_type, "application/x-nautilus-link") != 0)
	{
		return NULL;
	}

	/* attempt to parse it */
	doc = xmlParseFile (filename);

	/* check the document */
	if (doc == NULL)
	{
		return NULL;
	}

	/* check the structure */
	if (doc->root == NULL || doc->root->name == NULL ||
	    g_strcasecmp (doc->root->name, "nautilus_object") != 0)
	{
		xmlFreeDoc (doc);
		return NULL;
	}

	/* get the url */
	url = xmlGetProp (doc->root, "link");
	if (url == NULL || strlen (url) == 0)
	{
		xmlFreeDoc (doc);
		return NULL;
	}

	/* return the URL */
	xmlFreeDoc (doc);
	return url;
}

/*************************************************************************/
/* tree utility functions stolen from gtkctree code, GTK+ CREDITS        */
/* these are necessary because the original GtkCTree is a total f*ck-up, */
/* and has O(n^2) sorting complexity :-(                    -- MattA     */
/*************************************************************************/

static GtkCTreeNode *
gtk_ctree_last_visible (GtkCTree     *ctree,
			GtkCTreeNode *node)
{
  GtkCTreeNode *work;
  
  if (!node)
    return NULL;

  work = GTK_CTREE_ROW (node)->children;

  if (!work || !GTK_CTREE_ROW (node)->expanded)
    return node;

  while (GTK_CTREE_ROW (work)->sibling)
    work = GTK_CTREE_ROW (work)->sibling;

  return gtk_ctree_last_visible (ctree, work);
}

void
gtk_ctree_link (GtkCTree     *ctree,
		GtkCTreeNode *node,
		GtkCTreeNode *parent,
		GtkCTreeNode *sibling,
		gboolean      update_focus_row)
{
  GtkCList *clist;
  GList *list_end;
  GList *list;
  GList *work;
  gboolean visible = FALSE;
  gint rows = 0;
  
  clist = GTK_CLIST (ctree);

  for (rows = 1, list_end = (GList *)node; list_end->next;
       list_end = list_end->next)
    rows++;

  GTK_CTREE_ROW (node)->parent = parent;
  GTK_CTREE_ROW (node)->sibling = sibling;

  if (!parent || (parent && (gtk_ctree_is_viewable (ctree, parent) &&
			     GTK_CTREE_ROW (parent)->expanded)))
    {
      visible = TRUE;
      clist->rows += rows;
    }

  if (parent)
    work = (GList *)(GTK_CTREE_ROW (parent)->children);
  else
    work = clist->row_list;

  if (sibling)
    {
      if (work != (GList *)sibling)
	{
	  while (GTK_CTREE_ROW (work)->sibling != sibling)
	    work = (GList *)(GTK_CTREE_ROW (work)->sibling);
	  GTK_CTREE_ROW (work)->sibling = node;
	}

      if (sibling == GTK_CTREE_NODE (clist->row_list))
	clist->row_list = (GList *) node;
      if (GTK_CTREE_NODE_PREV (sibling) &&
	  GTK_CTREE_NODE_NEXT (GTK_CTREE_NODE_PREV (sibling)) == sibling)
	{
	  list = (GList *)GTK_CTREE_NODE_PREV (sibling);
	  list->next = (GList *)node;
	}
      
      list = (GList *)node;
      list->prev = (GList *)GTK_CTREE_NODE_PREV (sibling);
      list_end->next = (GList *)sibling;
      list = (GList *)sibling;
      list->prev = list_end;
      if (parent && GTK_CTREE_ROW (parent)->children == sibling)
	GTK_CTREE_ROW (parent)->children = node;
    }
  else
    {
      if (work)
	{
	  /* find sibling */
	  while (GTK_CTREE_ROW (work)->sibling)
	    work = (GList *)(GTK_CTREE_ROW (work)->sibling);
	  GTK_CTREE_ROW (work)->sibling = node;
	  
	  /* find last visible child of sibling */
	  work = (GList *) gtk_ctree_last_visible (ctree,
						   GTK_CTREE_NODE (work));
	  
	  list_end->next = work->next;
	  if (work->next)
	    list = work->next->prev = list_end;
	  work->next = (GList *)node;
	  list = (GList *)node;
	  list->prev = work;
	}
      else
	{
	  if (parent)
	    {
	      GTK_CTREE_ROW (parent)->children = node;
	      list = (GList *)node;
	      list->prev = (GList *)parent;
	      if (GTK_CTREE_ROW (parent)->expanded)
		{
		  list_end->next = (GList *)GTK_CTREE_NODE_NEXT (parent);
		  if (GTK_CTREE_NODE_NEXT(parent))
		    {
		      list = (GList *)GTK_CTREE_NODE_NEXT (parent);
		      list->prev = list_end;
		    }
		  list = (GList *)parent;
		  list->next = (GList *)node;
		}
	      else
		list_end->next = NULL;
	    }
	  else
	    {
	      clist->row_list = (GList *)node;
	      list = (GList *)node;
	      list->prev = NULL;
	      list_end->next = NULL;
	    }
	}
    }

  if (clist->row_list_end == NULL ||
      clist->row_list_end->next == (GList *)node)
    clist->row_list_end = list_end;

  if (visible && update_focus_row)
    {
      gint pos;
	  
      pos = g_list_position (clist->row_list, (GList *)node);
  
      if (pos <= clist->focus_row)
	{
	  clist->focus_row += rows;
	  clist->undo_anchor = clist->focus_row;
	}
    }
}

void
gtk_ctree_unlink (GtkCTree     *ctree, 
		  GtkCTreeNode *node,
                  gboolean      update_focus_row)
{
  GtkCList *clist;
  gint rows;
  gint level;
  gint visible;
  GtkCTreeNode *work;
  GtkCTreeNode *parent;
  GList *list;

  g_return_if_fail (ctree != NULL);
  g_return_if_fail (GTK_IS_CTREE (ctree));
  g_return_if_fail (node != NULL);

  clist = GTK_CLIST (ctree);
  
  if (update_focus_row && clist->selection_mode == GTK_SELECTION_EXTENDED)
    {
//    GTK_CLIST_GET_CLASS (clist)->resync_selection (clist, NULL);
      
      g_list_free (clist->undo_selection);
      g_list_free (clist->undo_unselection);
      clist->undo_selection = NULL;
      clist->undo_unselection = NULL;
    }

  visible = gtk_ctree_is_viewable (ctree, node);

  /* clist->row_list_end unlinked ? */
  if (visible &&
      (GTK_CTREE_NODE_NEXT (node) == NULL ||
       (GTK_CTREE_ROW (node)->children &&
	gtk_ctree_is_ancestor (ctree, node,
			       GTK_CTREE_NODE (clist->row_list_end)))))
    clist->row_list_end = (GList *) (GTK_CTREE_NODE_PREV (node));

  /* update list */
  rows = 0;
  level = GTK_CTREE_ROW (node)->level;
  work = GTK_CTREE_NODE_NEXT (node);
  while (work && GTK_CTREE_ROW (work)->level > level)
    {
      work = GTK_CTREE_NODE_NEXT (work);
      rows++;
    }

  if (visible)
    {
      clist->rows -= (rows + 1);

      if (update_focus_row)
	{
	  gint pos;
	  
	  pos = g_list_position (clist->row_list, (GList *)node);
	  if (pos + rows < clist->focus_row)
	    clist->focus_row -= (rows + 1);
	  else if (pos <= clist->focus_row)
	    {
	      if (!GTK_CTREE_ROW (node)->sibling)
		clist->focus_row = MAX (pos - 1, 0);
	      else
		clist->focus_row = pos;
	      
	      clist->focus_row = MIN (clist->focus_row, clist->rows - 1);
	    }
	  clist->undo_anchor = clist->focus_row;
	}
    }

  if (work)
    {
      list = (GList *)GTK_CTREE_NODE_PREV (work);
      list->next = NULL;
      list = (GList *)work;
      list->prev = (GList *)GTK_CTREE_NODE_PREV (node);
    }

  if (GTK_CTREE_NODE_PREV (node) &&
      GTK_CTREE_NODE_NEXT (GTK_CTREE_NODE_PREV (node)) == node)
    {
      list = (GList *)GTK_CTREE_NODE_PREV (node);
      list->next = (GList *)work;
    }

  /* update tree */
  parent = GTK_CTREE_ROW (node)->parent;
  if (parent)
    {
      if (GTK_CTREE_ROW (parent)->children == node)
	{
	  GTK_CTREE_ROW (parent)->children = GTK_CTREE_ROW (node)->sibling;
	  if (!GTK_CTREE_ROW (parent)->children)
	  {
//	    gtk_ctree_collapse (ctree, parent);
	  }
	}
      else
	{
	  GtkCTreeNode *sibling;

	  sibling = GTK_CTREE_ROW (parent)->children;
	  while (GTK_CTREE_ROW (sibling)->sibling != node)
	    sibling = GTK_CTREE_ROW (sibling)->sibling;
	  GTK_CTREE_ROW (sibling)->sibling = GTK_CTREE_ROW (node)->sibling;
	}
    }
  else
    {
      if (clist->row_list == (GList *)node)
	clist->row_list = (GList *) (GTK_CTREE_ROW (node)->sibling);
      else
	{
	  GtkCTreeNode *sibling;

	  sibling = GTK_CTREE_NODE (clist->row_list);
	  while (GTK_CTREE_ROW (sibling)->sibling != node)
	    sibling = GTK_CTREE_ROW (sibling)->sibling;
	  GTK_CTREE_ROW (sibling)->sibling = GTK_CTREE_ROW (node)->sibling;
	}
    }
}

/*************************************************************************/
