/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "history.h"
#include "window.h"
#include "state.h"

#include <libgnome/gnome-defs.h>
#include <libgnome/gnome-config.h>
#include <gtk/gtkcheckmenuitem.h>
#include <gtk/gtkctree.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkcheckbutton.h>
#include <gtk/gtkmenu.h>
#include <gtk/gtkoptionmenu.h>
#include <gtk/gtkmenushell.h>
#include <gtk/gtkentry.h>

/* local function prototypes */
gboolean history_clist_button_press_event_cb (GtkWidget *widget, 
					      GdkEventButton *event,
					      HistoryView *hv);
gboolean history_dialog_delete_event_cb (GtkWidget *widget, GdkEvent *event, 
					 HistoryView *hv);
void history_ok_button_clicked_cb(GtkButton *button, HistoryView *hv);
void history_attach_button_clicked_cb(GtkButton *button, HistoryView *hv);
void history_dock_close_button_clicked_cb (GtkButton *button, 
					   HistoryView *hv);
void history_dock_detach_button_clicked_cb (GtkButton *button, 
					    HistoryView *hv);
void history_entry_activate_cb (GtkEditable *editable,
				HistoryView *hv);
void history_time_optionmenu_leave_cb (GtkOptionMenu *optionmenu,
				       HistoryView *hv);
void history_host_checkbutton_toggled_cb (GtkCheckButton *check_button,
					  HistoryView *hv);
void history_dock_destroy_cb (GtkWidget *dock, HistoryView *hv);
void history_ctree_click_column_cb (GtkCTree *ctree, gint column,
				    HistoryView *hv);

/**
 * Loads the selected row in the browser
 */
gboolean
history_clist_button_press_event_cb (GtkWidget *widget, GdkEventButton *event,
				     HistoryView *hv)
{
        gint arow, acolumn;

	return_val_if_not_window (hv->window, FALSE);

	/* get the selection */
        gtk_clist_get_selection_info (GTK_CLIST (hv->ctree), 
				      event->x, event->y, &arow, &acolumn);

	/* get the item that's been selected */
	hv->selected_item = gtk_clist_get_row_data (GTK_CLIST (widget), arow);

	/* select it unless it's a left button selection */
        if (event->button != 1)
	{
                gtk_clist_select_row (GTK_CLIST (hv->ctree), arow, acolumn);
        }

	/* double click or middle click opens the window */
        if ((event->button == 2) || (event->type == GDK_2BUTTON_PRESS))
	{
		if (hv->selected_item != NULL && 
		    hv->selected_item->url != NULL &&
		    hv->selected_item->title != NULL)
		{
			window_load_url (hv->window, hv->selected_item->url);
		}
        }
        return FALSE;
}

/**
 * Hides the window
 */
gboolean
history_dialog_delete_event_cb (GtkWidget *widget, GdkEvent *event,
				HistoryView *hv)
{
	history_hide_dialog (hv);
        return TRUE;
}

/**
 * Hides the window
 */
void
history_ok_button_clicked_cb (GtkButton *button, HistoryView *hv)
{
	history_hide_dialog (hv);
}

/**
 * Hides the dock
 */
void
history_dock_close_button_clicked_cb (GtkButton *button, HistoryView *hv)
{
	return_if_not_window (hv->window);
	window_undock (hv->window);
}

/*
 * history_entry_activate_cb: search text is changed, update the tree
 */
void 
history_entry_activate_cb (GtkEditable *editable, HistoryView *hv)
{
	/* free existing text */
	if (hv->search != NULL)
	{
		g_free (hv->search);
	}

	/* get the text */
	hv->search = g_strdup (gtk_entry_get_text (GTK_ENTRY (editable)));
	
	/* store in config */
	gnome_config_set_string (CONF_HISTORY_SEARCH_TEXT, hv->search);
	gnome_config_sync ();

	/* refresh view */
	history_refresh_view (hv);
}

/*
 * history_time_optionmenu_deactivate_cb: time option menu is changed
 * update the tree
 */
void
history_time_optionmenu_deactivate_cb (GtkWidget *menushell, HistoryView *hv)
{
	GtkWidget *active_item;

	/* find which item has been selected */
	active_item = gtk_menu_get_active (GTK_MENU (menushell));
	hv->time = g_list_index (GTK_MENU_SHELL (menushell)->children,
				 active_item);

	/* store in config */
	gnome_config_set_int (CONF_HISTORY_SEARCH_TIME, hv->time);
	gnome_config_sync ();

	/* refresh view */
	history_refresh_view (hv);
}

/*
 * history_host_checkbutton_toggled_cb: the host group checkbox status is
 * changed, update the tree
 */
void
history_host_checkbutton_toggled_cb (GtkCheckButton *check_button,
				     HistoryView *hv)
{
	/* get new state */
	hv->group = gtk_toggle_button_get_active 
		(GTK_TOGGLE_BUTTON (check_button));

	/* store in config */
	gnome_config_set_bool (CONF_HISTORY_HOST_GROUP, hv->group);
	gnome_config_sync ();

	/* refresh view */
	history_refresh_view (hv);
}

void 
history_attach_button_clicked_cb (GtkButton *button, HistoryView *hv)
{
	GaleonWindow *window;

	window = hv->window;
	return_if_not_window (window);
	history_hide_dialog (hv);
	history_show_dock (window);
}


void 
history_dock_detach_button_clicked_cb (GtkButton *button, HistoryView *hv)
{
	GaleonWindow *window;

	window = hv->window;
	return_if_not_window (window);
	window_undock (window);
	history_show_dialog (window);
}

/**
 * history_hide_destroy_cb: called when the dock is destroyed
 */
void 
history_dock_destroy_cb (GtkWidget *dock, HistoryView *hv)
{
	/* hack because we don't want to emit toggled signal */
	(GTK_CHECK_MENU_ITEM (hv->window->view_history_dock))->active = FALSE;
	
	/* save dock column widths */
	state_save_column_widths (hv->ctree, "history_dock");	

	/* destroy the view structure */
	history_destroy_view (hv);
}

/**
 * history_ctree_click_column_cb: called when a column is clicked, change
 * the sort type accordingly.
 */
void
history_ctree_click_column_cb (GtkCTree *ctree, gint column, HistoryView *hv)
{
	gint sort_column;
	gint sort_order;

	/* fetch current setup */
	sort_column = gnome_config_get_int (CONF_HISTORY_SORT_COLUMN);
	sort_order  = gnome_config_get_int (CONF_HISTORY_SORT_ORDER);

	/* if we're already sorting on this column, reverse the sort */
	if (sort_column == column)
	{
		if (sort_order == GTK_SORT_ASCENDING)
		{
			sort_order = GTK_SORT_DESCENDING;
		}
		else
		{
			sort_order = GTK_SORT_ASCENDING;
		}
	}
	else
	{
		sort_order = GTK_SORT_ASCENDING;
	}

	/* store new setup */
	gnome_config_set_int (CONF_HISTORY_SORT_COLUMN, column);
	gnome_config_set_int (CONF_HISTORY_SORT_ORDER, sort_order);
	gnome_config_sync ();

	/* request a re-sort */
	history_resort_view (hv);
}

/**
 * history_compare_title_cb; compare function for "sort on last title"
 */
gint 
history_compare_title_cb (gconstpointer a, gconstpointer b)
{
	HistoryItem *hi1 = (HistoryItem *)((GTK_CTREE_ROW (a)->row).data);
	HistoryItem *hi2 = (HistoryItem *)((GTK_CTREE_ROW (b)->row).data);

	if (hi1->url == NULL && hi2->url == NULL)
	{
		HistoryHost *hh1 = (HistoryHost *)hi1;
		HistoryHost *hh2 = (HistoryHost *)hi2;

		return g_strcasecmp (hh1->dominant_item->title,
				     hh2->dominant_item->title);
	}
	else
	{
		return g_strcasecmp (hi1->title, hi2->title);
	}
}

/**
 * history_compare_url_cb; compare function for "sort on last url"
 */
gint 
history_compare_url_cb (gconstpointer a, gconstpointer b)
{
	HistoryItem *hi1 = (HistoryItem *)((GTK_CTREE_ROW (a)->row).data);
	HistoryItem *hi2 = (HistoryItem *)((GTK_CTREE_ROW (b)->row).data);

	if (hi1->url == NULL && hi2->url == NULL)
	{
		return g_strcasecmp (hi1->title, hi2->title);
	}
	else
	{
		return g_strcasecmp (hi1->url, hi2->url);
	}
}

/**
 * history_compare_last_cb: compare function for "sort on last visit"
 */
gint 
history_compare_last_cb (gconstpointer a, gconstpointer b)
{
	HistoryItem *hi1 = (HistoryItem *)((GTK_CTREE_ROW (a)->row).data);
	HistoryItem *hi2 = (HistoryItem *)((GTK_CTREE_ROW (b)->row).data);

	return (hi2->last - hi1->last);
}

/**
 * history_compare_first_cb: compare function for "sort on first visit"
 */
gint 
history_compare_first_cb (gconstpointer a, gconstpointer b)
{
	HistoryItem *hi1 = (HistoryItem *)((GTK_CTREE_ROW (a)->row).data);
	HistoryItem *hi2 = (HistoryItem *)((GTK_CTREE_ROW (b)->row).data);

	return (hi2->first - hi1->first);
}

/**
 * history_compare_visits_cb: compare function for "sort on number of visits"
 */
gint 
history_compare_visits_cb (gconstpointer a, gconstpointer b)
{
	HistoryItem *hi1 = (HistoryItem *)((GTK_CTREE_ROW (a)->row).data);
	HistoryItem *hi2 = (HistoryItem *)((GTK_CTREE_ROW (b)->row).data);

	return (hi2->visits - hi1->visits);
}

/**
 * history_periodic_save_cb: save the history (if dirty) every once in a while
 */
gboolean
history_periodic_save_cb (gpointer data)
{
	/* save it */
	history_save ();

	/* call again */
	return TRUE;
}
