/*
 *  Copyright (C) 2000 Marco Pesenti Gritti
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "galeon.h"
#include "window.h"
#include "embed.h"
#include "misc.h"

#include <libgnome/gnome-config.h>
#include <libgnome/gnome-mime.h>
#include <gtkmozembed.h>

/* for nautilus dnd desktop items */
#include <gnome-xml/tree.h>
#include <gnome-xml/parser.h>
#include <libgnomevfs/gnome-vfs-mime.h>

/** 
 * embed_drag_data_received:
 */
void
embed_drag_data_received (GtkWidget *widget, GdkDragContext *context,
			  gint x, gint y, GtkSelectionData *selection_data,
			  guint info, guint time, GaleonEmbed *embed)
{
	gchar *data = selection_data->data;
	gboolean tabbed;
	GList *uris, *l;	
	gchar *drop_uri;
	gchar *parse_uri;
	gchar *real_uri;

	/* get tab status */
	tabbed = gnome_config_get_bool (CONF_APPEARANCE_TABBED);

	switch (info)
	{
	case DND_TARGET_NETSCAPE_URL:
	case DND_TARGET_STRING:
	case DND_TARGET_GALEON_URL:
		/* standard types are just strings */
		embed_load_url (embed, data);
		break;

	case DND_TARGET_TEXT_URI_LIST:
		/* list as passed by Nautilus */
		uris = gnome_uri_list_extract_uris (data);
		for (l = uris; l != NULL; l = g_list_next (l))
		{
			/* find the uri to open */
			drop_uri = (gchar *)(l->data);
			parse_uri = misc_parse_nautilus_uri (drop_uri);
			real_uri = (parse_uri == NULL ? drop_uri : parse_uri);

			/* open in first or next embed */
			if (l == uris)
			{
				embed_load_url (embed, real_uri);
			}
			else
			{
				embed_create_from_url (embed, real_uri,
						       FALSE, !tabbed);
			}
			
			/* free allocated string */
			g_free (drop_uri);
			if (parse_uri != NULL)
			{
				g_free (parse_uri);
			}
		}
		/* free parsed list */
		g_list_free (uris);
		break;
       
	default:
		/* shouldn't happen */
		g_warning ("unexpected drop type %ud\n", info);
		break;
	}
}

/**
 * embed_notebook_switch_page: called in tabbed mode when the user
 * selects a different browser tab
 */
void
embed_notebook_switch_page_cb (GtkNotebook *notebook, 
			       GtkNotebookPage *page, guint page_num)
{
	GtkMozEmbed *mozEmbed;
	GaleonWindow *window;
	GaleonEmbed *old_embed;
	GaleonEmbed *embed;

	g_return_if_fail (notebook != NULL);
	g_return_if_fail (page != NULL);

	/* check we REALLY REALLY have switched page */
	if (GTK_NOTEBOOK (notebook)->cur_page != page)
	{
		return;
	}

	/* find the GtkMozEmbed from the page */
	mozEmbed = (GtkMozEmbed *)page->child;
	g_return_if_fail (GTK_IS_MOZ_EMBED (mozEmbed));

	/* find the GaleonEmbed from the GtkMozEmbed */
	embed = gtk_object_get_data (GTK_OBJECT (mozEmbed), "GaleonEmbed");
	return_if_not_embed (embed);

	/* find the GaleonWindow form the GaleonEmbed */
	window = embed->parent_window;
	return_if_not_window (window);

	/* set this as the active embed and save the content of the
	 * location bar of the old embed */
	old_embed = window->active_embed;
	if (old_embed != NULL)
	{
		/* deactivate button */
		if (old_embed->notebook_close_button != NULL)
		{
			gtk_widget_hide (old_embed->notebook_close_button);
			gtk_widget_show (old_embed->notebook_close_pixmap);
		}

		/* save edited site location information */
		if (old_embed->site_location) 
		{
			g_free (old_embed->site_location);
		}
		old_embed->site_location = gtk_editable_get_chars 
			(GTK_EDITABLE (window->toolbar_entry), 0, -1);

		/* no longer the active window */
		old_embed->is_active = FALSE;
	}

	/* this is now the active window */
	embed->is_active = TRUE;
	window->active_embed = embed;

	/* turn on the close button */
	if (embed->notebook_close_button != NULL)
	{
		gtk_widget_show (embed->notebook_close_button);
		gtk_widget_hide (embed->notebook_close_pixmap);
	}

	/* set the global title and location -- this is the lazy way! */
	embed_update_page_title (embed);
	embed_update_page_location (embed);

	/* update button status */
	window_update_nav_controls (window);

	/* update the status bar */
	window_update_status_bar (window);

	/* update the zoom control */
	window_update_zoom (window);

	/* set focus to the embed -- this seems to work although it
	 * is a bit hacky, anyone got any better ideas? -- MattA */
	gtk_widget_grab_focus (GTK_WIDGET (embed->mozEmbed)->parent);
}

/**
 * embed_notebook_close_clicked_cb: close a tab
 */
void 
embed_notebook_close_clicked_cb (GtkWidget *b, GdkEventButton *event,
				 GaleonEmbed *embed)
{
	return_if_not_embed (embed);

	if (event->button == 1)
	{
		embed_close (embed);
	}
}
