/*
 *  Copyright (C) 2001 Philip Langdale
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2, or (at your option)
 *  any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

/*
 * The functioning of the download architecture, as described by Philip
 * on 28 May 2001:
 * 
 * When mozilla runs into a file it cannot render internally or that it
 * does not have a plugin for, it calls the
 * nsIExternalHelperAppService. This service will then either attempt to
 * save the file or run it with a helper app depending on what the
 * mozilla mime database returns.
 * 
 * nsIExternalHelperAppService then calls out to the nsIHelperAppDialog
 * interface which handles the UI for the service. This is the interface
 * which we have reimplemented. Therefore, with a major caveat, we have
 * put a GNOME/GTK frontend on an unmodified mozilla backend.
 * 
 * Now for the caveat. With respect to saving files to disk, the mozilla
 * backend works exactly the same as it does in
 * mozilla-the-browser. However, for dealing with helper apps, we do not
 * use the mozilla backend at all. This is because we want to use the
 * gnome-vfs database to retrieve helper-app info, rather than the
 * mozilla helper app database.
 * 
 * How it works:
 * 
 * a) The user clicks on a link or follows a redirect to a file of a type
 * that mozilla cannot handle. Mozilla passes the link to the
 * ExternalHelperAppService which in turn calls the Show() method of
 * nsIHelperAppDialog.
 * 
 * b) In our implementation of Show() we first compare the passed mime
 * type to galeon's mime list. If the mime type is in the list, we then
 * lookup the Action associated with the mime type. Currently, the
 * possible mime-actions are:
 * 
 * Save to Disk
 * Save to Disk with External Download Handler
 * Run with Helper App
 * Ask User
 * ( Override mozilla built-in type ) doesn't really work yet. :-) We'll
 * ignore it for now.
 * 
 * The default action is Ask User, and if the mime-type is not in the
 * list, this is what will be assumed.
 * 
 * c) If Ask User is the chosen action, a dialog will be shown to the
 * user allowing the user to choose from the other three possible actions
 * as well as a checkbox to let the user set the default action to the
 * chosen action for the future.
 * 
 * d-1) The "Save to Disk" action.  This action is handled by the mozilla
 * backend. Our nsIHelperAppDialog does the same thing that the
 * mozilla-the-browser one does, which is to call the SaveToDisk() method
 * of nsIExternalHelperAppService. This in turn calls the
 * PromptForSaveToFile() method of nsIHelperAppDialog putting the ball
 * back in our court.
 * 
 * d-2) Now, if galeon is configured to always ask for a download
 * directory, it will pop up a file selector so that the user can select
 * the directory and filename to save the file to.  Otherwise, it will
 * use galeon's default download directory and proceed without
 * interaction.
 * 
 * d-3) When PromptForSaveToFile() returns, nsIExternalHelperAppService
 * will then call the ShowProgressDialog() method of
 * nsIHelperAppDialog. This progress dialog, obviously, tracks the
 * progress of the download. It is worth noting that mozilla starts the
 * actual download as soon as the user clicks on the link or follows the
 * redirect. While the user is deciding what action to take, the file is
 * downloading. Often, for small files, the file is already downloaded
 * when the user decides what directory to put it in. The progress dialog
 * does not appear in these cases. Also, we currently have a small
 * problem where our progress dialog times the download from the point
 * the dialog appears, not from the time the download starts. This is due
 * to the timestamp that is passed to us is just plain weird, and I
 * haven't worked out how to turn it into a useable time. The fact that
 * the download starts early means that the file is actually downloaded
 * to a temp file and only at the end is it moved to it's final location.
 * 
 * e-1) The "Save to Disk with External Handler" action.  This option is
 * handled completely by galeon. The first thing that we do is call the
 * Cancel() method of nsIExternalHelperAppService to cancel the mozilla
 * download. We then pass the url to our own LaunchExternalDownloader()
 * method. This method will ask for a download directory as appropriate
 * as with the "Save to disk" action.
 * 
 * e-2) Finally, depending on whether GTM or a command line handler was
 * selected in prefs, the external handler will be called with the url
 * passed and the directory selected.
 * 
 * f-1) The "Run with Helper App" action.  This action is currently only
 * working with a minimal implementation.  First, we explicitly call
 * ShowProgressDialog() so the user knows that the file is being
 * downloaded. We also need this so that we only run the helper after the
 * file is completely downloaded. The file will download to temp location
 * that it would be moved from if the action was "Save to Disk".  We have
 * to call ShowProgressDialog() ourselves because we are not using
 * mozilla's helper mechanism which would usually make the call for us.
 * 
 * f-2) We call out to gnome vfs with the mime-type of the document and
 * attempt to retrieve the default helper app associated with that
 * mime-type in gnome-vfs. If the mime type has a
 * GNOME_VFS_MIME_ACTION_TYPE_APPLICATION action type and it also has a
 * default helper app, then we execute that helper with the file. If the
 * action type is any other type or there is no default helper app, we
 * inform the user of this fact and then Cancel() the download.
 * 
 * f-3) Once the helper app is launched, galeon no longer has anything to
 * do with the helper or the temp file. We cannot delete the temp file as
 * we don't have a way of telling when the user is finished with
 * it. Mozilla acts in the same way.
 * 
 * g) In the future, I plan to expand helper support so that the user is
 * presented with a list of all the helper apps that gnome-vfs knows of
 * for that mime type. It is often the case that there will be more than
 * one possible helper. eg: pdf files have ggv, gv and xpdf as possible
 * helpers with ggv as default in a normal gnome-vfs setup. currently we
 * always use the default ggv.
 * 
 * It is also likely that we will provide an option to provide a path to
 * an arbirtrary program to use as a helper in case the user doesn't want
 * to use the gnome-vfs helpers or there are no helpers in gnome-vfs.
 * 
 * We will also need to expand the info stored in prefs to note which
 * helper app the user prefers for a given mime-type.
 * 
 * h) General notes.  We cannot use this infrastructure to override
 * native mozilla types. mozilla will attempt to render these types and
 * never call out to us. We are at the end of the chain as the handler of
 * last resort, so native and plugin types will never reach us. This also
 * means that a file with an incorrect mime-type ( eg: .tar.bz2 marked as
 * text/plain ) will be incorrectly rendered by mozilla. We cannot help
 * this.
 * 
 * We will be expanding the current context menu "download link" option
 * so that it can use the mozilla downloading mechanism as well as the
 * External Handler as is currently possible.  This will be the way to
 * download file with bad mime-types, as is the case in
 * mozilla-the-browser.  We will also implement shift-click to match this
 * behaviour.
 * 
 * Matt would like the modifiy the progress dialog so each file currently
 * being downloaded becomes a clist entry in a master dialog rather than
 * causing a separate progress dialog. a lot of progress dialogs gets
 * really messy.
 */

#include "galeon.h"
#include "mime.h"
#include "misc.h"
#include "window.h"
#include "mozilla.h"

extern "C" {
#include "libgnomevfs/gnome-vfs-mime-handlers.h"
}

#include <gtk/gtkentry.h>
#include <gtk/gtktogglebutton.h>
#include <gtk/gtkprogress.h>
#include <gtk/gtkoptionmenu.h>
#include <libgnome/gnome-exec.h>
#include <libgnome/gnome-i18n.h>
#include <libgnome/gnome-config.h>

#include "FilePicker.h"

#include "nsIHelperAppLauncherDialog.h"
#include "nsIWebProgressListener.h"
#include "nsIExternalHelperAppService.h"

#include "nsCRT.h"
#include "nsCOMPtr.h"
#include "nsIFactory.h"
#include "nsISupportsArray.h"
#include "nsIServiceManager.h"
#include "nsWeakReference.h"
#include "nsXPComFactory.h"

#include "nsString.h"
#include "nsIURI.h"
#include "nsIMIMEInfo.h"
#include "nsIChannel.h"
#include "nsILocalFile.h"
#include "nsIPref.h"
#include "nsIDOMWindow.h"
#include "nsIDOMWindowInternal.h"

class GContentHandler;
class GDownloadProgressListener;
struct MimeAskActionDialog;
struct HelperAppChooserDialog;

extern "C" {
void mime_ask_dialog_ok_clicked_cb (GtkButton *button,
				    MimeAskActionDialog *dialog);
gint mime_ask_dialog_cancel_clicked_cb (GtkButton *button,
					MimeAskActionDialog *dialog);
void helper_chooser_ok_clicked_cb (GtkButton *button,
				   HelperAppChooserDialog *dialog);
gint helper_chooser_cancel_clicked_cb (GtkButton *button,
				       HelperAppChooserDialog *dialog);
void helper_chooser_browse_clicked_cb (GtkButton *button,
				       HelperAppChooserDialog *dialog);
void helper_chooser_auto_default_toggled (GtkToggleButton *button,
					  HelperAppChooserDialog *dialog);
void helper_chooser_radio_list_toggled (GtkToggleButton *button,
					HelperAppChooserDialog *dialog);
void helper_chooser_radio_other_toggled (GtkToggleButton *button,
					 HelperAppChooserDialog *dialog);
gint progress_dialog_cancel_cb (GtkButton *button,
				GDownloadProgressListener *aListener);
}

/**
 * MimeAskActionDialog: the representation of dialogs used to ask
 * about actions on MIME types
 */
struct MimeAskActionDialog
{
	gchar *url;
	gchar *mime_type;
	GContentHandler *mContentHandler;
	GladeXML *gxml;
};

struct HelperAppChooserDialog
{
	HelperAppChooserDialog (void);
	~HelperAppChooserDialog () {};

	GList *mHelperApps;
	GContentHandler *mContentHandler;
	GladeXML *mGXML;
	gboolean mAutoDefaultChecked;
	gboolean mOtherHelperApp;

	GtkWidget *mDialogWidget;
	GtkWidget *mBrowseButton;
	GtkWidget *mHelperEntry;
	GtkWidget *mOptionMenu;
};

class GContentHandler : public nsIHelperAppLauncherDialog
{
  public:
 	NS_DECL_ISUPPORTS
	NS_DECL_NSIHELPERAPPLAUNCHERDIALOG

	GContentHandler();
	virtual ~GContentHandler();

	NS_IMETHODIMP LaunchExternalDownloader (void);
	NS_IMETHODIMP FindHelperApp (void);
	NS_IMETHODIMP LaunchHelperApp (void);
	NS_IMETHODIMP ShowHelperProgressDialog (void);

	NS_IMETHODIMP GetLauncher (nsIHelperAppLauncher * *_retval);
	NS_IMETHODIMP GetContext (nsISupports * *_retval);
	NS_IMETHODIMP SetHelperApp(GnomeVFSMimeApplication *mHelperApp);

  private:
	/* additional members */
	NS_IMETHODIMP Init (void);
	NS_IMETHODIMP ProcessMimeInfo (void);
	NS_IMETHODIMP MIMEAskAction (void);
	NS_IMETHODIMP ChooseHelperApp (void);
	
	nsCOMPtr<nsIHelperAppLauncher> mLauncher;
	nsCOMPtr<nsISupports> mContext;

	nsCOMPtr<nsIURI> mUri;
	PRInt64 mTimeDownloadStarted;
	nsCOMPtr<nsIFile> mTempFile;

	MimeAction mMimeAction;

	char *mUrl;
	char *mMimeType;

	PRBool mDownloadCanceled;
	PRBool mHelperProgress;

	nsCOMPtr<nsIWebProgressListener> mListener;
	
	GnomeVFSMimeApplication *mHelperApp;
};

class GDownloadProgressListener : public nsIWebProgressListener,
				  public nsSupportsWeakReference
{
 public:
 	NS_DECL_ISUPPORTS
	NS_DECL_NSIWEBPROGRESSLISTENER

	GDownloadProgressListener (nsIHelperAppLauncher *aLauncher,
				   nsISupports *aContext,
				   GContentHandler *aHandler,
				   PRBool aUsedByHelper);
	virtual ~GDownloadProgressListener ();

	NS_IMETHODIMP CancelHelperProgress (void);

 private:

	char * FormatTime (PRUint32 aTime);
	
	nsCOMPtr<nsIHelperAppLauncher> mLauncher;
	nsCOMPtr<nsISupports> mContext;
	nsCOMPtr<GContentHandler> mHandler;
	PRBool mUsedByHelper;
	
	nsCOMPtr<nsIURI> mUri;
	PRInt64 mTimeDownloadStarted;
	nsCOMPtr<nsIFile> mFile;
	
	PRInt64 mStartTime;
	PRInt64 mElapsed;
	
	PRInt64 mLastUpdate;
	PRInt32 mInterval;

	PRFloat64 mPriorKRate;
	PRInt32 mRateChanges;
	PRInt32 mRateChangeLimit;

	GtkWidget *mProgressDialog;
	GtkWidget *mProgressBar;
	GtkWidget *mLocation;
	GtkWidget *mFileName;
	GtkWidget *mStatus;
	GtkWidget *mTimeElapsed;
	GtkWidget *mTimeRemaining;
};


/* Implementation file */
NS_IMPL_ISUPPORTS1(GContentHandler, nsIHelperAppLauncherDialog)

NS_IMPL_ISUPPORTS2(GDownloadProgressListener, nsIWebProgressListener,
		   nsISupportsWeakReference)

GContentHandler::GContentHandler()
{
	NS_INIT_ISUPPORTS();
	/* member initializers and constructor code */
	mUri = nsnull;
	mMimeType = nsnull;
	mMimeAction = MIME_UNKNOWN;
	mDownloadCanceled = PR_FALSE;
	mHelperProgress = PR_FALSE;
	mHelperApp = nsnull;
}

GContentHandler::~GContentHandler()
{
	/* destructor code */
	g_free (mUri);
	g_free (mMimeType);
	if (mHelperApp)
	{
		if (nsCRT::strcmp(mHelperApp->id,"OtHeR") == 0)
		{
			g_free (mHelperApp->command);
			g_free (mHelperApp->id);
			delete mHelperApp;
		}
		else
		{
			gnome_vfs_mime_application_free (mHelperApp);
		};
	};
}

////////////////////////////////////////////////////////////////////////////////
// begin nsIHelperAppLauncher impl
////////////////////////////////////////////////////////////////////////////////

/* void show (in nsIHelperAppLauncher aLauncher, in nsISupports aContext); */
NS_IMETHODIMP GContentHandler::Show(nsIHelperAppLauncher *aLauncher,
					      nsISupports *aContext)
{
	nsresult rv;

	mLauncher = aLauncher;
	mContext = aContext;
	rv = Init ();
	
	switch (mMimeAction)
	{
	 case MIME_UNKNOWN:
		mime_set_action (mMimeType, MIME_ASK_ACTION);
	 case MIME_ASK_ACTION:
		MIMEAskAction ();
		break;
	 case MIME_IGNORE:
		mLauncher->Cancel ();
		break;
	 case MIME_SAVE_TO_DISK:
		mLauncher->SaveToDisk (nsnull,PR_FALSE);
		break;
	 case MIME_SAVE_WITH_GTM:
		mLauncher->Cancel ();
		LaunchExternalDownloader ();
	 	break;
	 case MIME_RUN_PROGRAM:
		FindHelperApp ();
	 	break;
	 case MIME_OVERRIDE_MOZILLA:
		/* shouldn't get here */
		g_warning ("Unexpected MIME action");
		break;
	};
	return NS_OK;
}

/* nsILocalFile promptForSaveToFile (in nsISupports aWindowContext, in wstring aDefaultFile, in wstring aSuggestedFileExtension); */
NS_IMETHODIMP GContentHandler::
		PromptForSaveToFile(nsISupports *aWindowContext,
				    const PRUnichar *aDefaultFile,
				    const PRUnichar *aSuggestedFileExtension,
				    nsILocalFile **_retval)
{
	nsresult rv;

	mContext = aWindowContext;

	nsCOMPtr<nsIDOMWindowInternal> aWindowInternal = 
					do_QueryInterface (aWindowContext);
	

	nsCOMPtr<nsIPref> prefs = do_GetService (NS_PREF_CONTRACTID);

	nsCOMPtr<nsILocalFile> saveDir;
	rv = prefs->GetFileXPref ("browser.download.dir",
				  getter_AddRefs(saveDir));

	if (NS_FAILED(rv))
	{
		char *dirName = g_strdup (gnome_config_get_string 
				("/galeon/Downloading/download_dir"));

		if (!dirName)
		{
			dirName = g_strdup (g_get_home_dir());
		};
		saveDir = do_CreateInstance (NS_LOCAL_FILE_CONTRACTID);
		saveDir->InitWithPath (dirName);
		g_free (dirName);
	};

	nsCOMPtr<nsIFilePicker> aFilePicker =
				do_CreateInstance (G_FILEPICKER_CONTRACTID);

	nsString title = NS_LITERAL_STRING (N_("Save As:"));

	aFilePicker->Init (aWindowInternal, title.get(), 
			   nsIFilePicker::modeSave);
	aFilePicker->SetDefaultString (aDefaultFile);
	aFilePicker->SetDisplayDirectory (saveDir);

	nsCOMPtr <nsILocalFile> aFile (do_CreateInstance(NS_LOCAL_FILE_CONTRACTID));

	PRInt16 okToSave;
	if (gnome_config_get_bool("/galeon/Downloading/ask_for_download_dir"))
	{
		aFilePicker->Show (&okToSave);

		if (okToSave == nsIFilePicker::returnOK)
		{
			aFilePicker->GetFile (getter_AddRefs(aFile));
		};
	}
	else
	{
		char *cDirName;
		saveDir->GetPath (&cDirName);
		char *cDefaultFile = mozilla_unicode_to_locale (aDefaultFile);
		char *cFileName = g_strconcat (cDirName, "/", cDefaultFile, NULL);
		aFile->InitWithPath (cFileName);
		g_free (cDefaultFile);
		g_free (cFileName);
		nsMemory::Free (cDirName);
		okToSave = nsIFilePicker::returnOK;
	};


	if (okToSave == nsIFilePicker::returnCancel)
		return NS_ERROR_FAILURE;
	else
	{
		nsCOMPtr<nsIFile> aDirectory;
		rv = aFile->GetParent (getter_AddRefs(aDirectory));
		nsCOMPtr<nsILocalFile> aLocalDirectory 
					(do_QueryInterface(aDirectory));
		rv = prefs->SetFileXPref ("browser.download.dir",
					  aLocalDirectory);
		NS_IF_ADDREF (*_retval = aFile);
		return NS_OK;
	};
}

/* void showProgressDialog (in nsIHelperAppLauncher aLauncher, in nsISupports aContext); */
NS_IMETHODIMP GContentHandler::
			ShowProgressDialog(nsIHelperAppLauncher *aLauncher,
					   nsISupports *aContext)
{
	GDownloadProgressListener *aListener = 
				new GDownloadProgressListener (aLauncher,
							       aContext,
							       this,
							       mHelperProgress);
	NS_ADDREF (aListener);
	mListener = NS_STATIC_CAST (nsIWebProgressListener *, aListener);
	NS_RELEASE (aListener);
	
	return NS_OK;
}

////////////////////////////////////////////////////////////////////////////////
// begin local public methods impl
////////////////////////////////////////////////////////////////////////////////

NS_IMETHODIMP GContentHandler::LaunchExternalDownloader (void)
{
	save_url (mUrl);
	return NS_OK;
}

NS_IMETHODIMP GContentHandler::FindHelperApp (void)
{
	nsresult rv;

/*
 	GnomeVFSMimeActionType actionType = 
 		gnome_vfs_mime_get_default_action_type (mMimeType);

	GnomeVFSMimeApplication *helper = 
		gnome_vfs_mime_get_default_application (mMimeType);
*/
// 	if ((actionType & GNOME_VFS_MIME_ACTION_TYPE_APPLICATION) &&
// 	    helper)
 	{
 		rv = ChooseHelperApp ();
 	}
/*	else if (!(actionType & GNOME_VFS_MIME_ACTION_TYPE_APPLICATION) ||
		 !helper)
	{
		char *msgStr = g_strdup_printf (_("There are no known helper "
						"applications for MIME type: "
						"%s."),mMimeType);

		GtkWidget *msgDialog = 
			gnome_message_box_new (msgStr,
					       GNOME_MESSAGE_BOX_INFO,
					       GNOME_STOCK_BUTTON_OK,
					       NULL);
					       
		nsCOMPtr<nsIDOMWindow> parent = do_QueryInterface (mContext);
		GtkWidget *aParentWidget = mozilla_find_gtk_parent (parent);

		if (aParentWidget)
		{
			gnome_dialog_set_parent (GNOME_DIALOG(msgDialog),
						 GTK_WINDOW(aParentWidget));
		}
		else
		{
			gtk_widget_show (msgDialog);
			window_set_layer (msgDialog);
		};
		
		gnome_dialog_run_and_close (GNOME_DIALOG (msgDialog));
		g_free (msgStr);
		mLauncher->Cancel ();

	};*/
	return NS_OK;
}

NS_IMETHODIMP GContentHandler::LaunchHelperApp (void)
{
	GnomeVFSMimeApplication *helper = 
		gnome_vfs_mime_get_default_application (mMimeType);
	
	if (helper)
	{
		char *aFileName, *document(NULL);
		mTempFile->GetPath(&aFileName);

		if (helper->expects_uris == 
			GNOME_VFS_MIME_APPLICATION_ARGUMENT_TYPE_PATHS)
		{
			document = aFileName;
		}
		else
		{
			document = mUrl;
		}
		if (helper->requires_terminal == FALSE)
		{
			char *argv[2] = { mHelperApp->command, document };
			gnome_execute_async ("/tmp", 2, argv);
		}
		else
		{
			char *argv[4] = { "gnome-terminal", "-e",
					  mHelperApp->command, document };
			gnome_execute_async ("/tmp", 4, argv);
		}
		g_free (aFileName);
	}
	else
	{
		mLauncher->Cancel ();
	}

	return NS_OK;
}

NS_IMETHODIMP GContentHandler::ShowHelperProgressDialog (void)
{
	mHelperProgress = PR_TRUE;
	return ShowProgressDialog (mLauncher,mContext);
}

NS_IMETHODIMP GContentHandler::GetLauncher (nsIHelperAppLauncher * *_retval)
{
	NS_IF_ADDREF (*_retval = mLauncher);
	return NS_OK;
}

NS_IMETHODIMP GContentHandler::GetContext (nsISupports * *_retval)
{
	NS_IF_ADDREF (*_retval = mContext);
	return NS_OK;
}

NS_IMETHODIMP GContentHandler::SetHelperApp(GnomeVFSMimeApplication *aHelperApp)
{
	if (nsCRT::strcmp(aHelperApp->id,"OtHeR") == 0)
		mHelperApp = aHelperApp;
	else
		mHelperApp = gnome_vfs_mime_application_copy (aHelperApp);
	return NS_OK;
}

////////////////////////////////////////////////////////////////////////////////
// begin local private methods impl
////////////////////////////////////////////////////////////////////////////////
NS_IMETHODIMP GContentHandler::Init (void)
{
	nsresult rv;

	nsCOMPtr<nsIMIMEInfo> aMIMEInfo;
	rv = mLauncher->GetMIMEInfo (getter_AddRefs(aMIMEInfo));
	rv = aMIMEInfo->GetMIMEType (&mMimeType);
	
	ProcessMimeInfo ();

	rv = mLauncher->GetDownloadInfo(getter_AddRefs(mUri),
					&mTimeDownloadStarted,
					getter_AddRefs(mTempFile));

	rv = mUri->GetSpec (&mUrl);
#if 0
	/* GetSource seems redundant and isn't in 0.9 This code is here while
	   it remains unclear what GetSource is for. --phil */
	nsCOMPtr<nsIURI> aUri;
	rv = mLauncher->GetSource(getter_AddRefs(aUri));
	rv = aUri->GetSpec (&mUrl);
#endif	
	return NS_OK;
}

NS_IMETHODIMP GContentHandler::ProcessMimeInfo (void)
{
	/* FIXME this it to work around a mozilla bug, when closing popups
	   sometimes mime types handled internally by mozilla are passed
	   to the content handler. Force the correct behavior */
	if
	(!strcasecmp(mMimeType, "text/html")   ||
	 !strcasecmp(mMimeType, "text/plain")  ||
	 !strcasecmp(mMimeType, "application/vnd.mozilla.xul+xml")    ||
	 !strcasecmp(mMimeType, "text/rdf")    ||
	 !strcasecmp(mMimeType, "text/xml")    ||
	 !strcasecmp(mMimeType, "application/xml")    ||
	 !strcasecmp(mMimeType, "application/xhtml+xml")    ||
	 !strcasecmp(mMimeType, "text/css")    ||
	 !strcasecmp(mMimeType, "image/gif")   ||
	 !strcasecmp(mMimeType, "image/jpeg")  ||
	 !strcasecmp(mMimeType, "image/png")   ||
	 !strcasecmp(mMimeType, "application/http-index-format"))
	{
		mMimeAction = MIME_IGNORE;
		g_warning ("wrong mime type detection !");
		return NS_OK;
	};

	for (GList *mime_type = g_list_first(mime_types) ;
	     mime_type!=NULL ;
	     mime_type = g_list_next(mime_type))
	{
		gchar *type = ((MimeItem *)mime_type->data)->name;
		if (nsCRT::strcasecmp(mMimeType, type) == 0)
		{
			mMimeAction = ((MimeItem *)mime_type->data)->action;
			break;
		};
	};
	return NS_OK;
}

NS_IMETHODIMP GContentHandler::MIMEAskAction (void)
{
	MimeAskActionDialog *dialog = new MimeAskActionDialog;
	GtkWidget *label;
	GtkWidget *dialogWidget;
	gchar *msg;
		
	dialog->url = g_strdup (mUrl);
	dialog->mime_type = g_strdup (mMimeType);
	dialog->mContentHandler = this;
	dialog->gxml = glade_widget_new ("galeon.glade", "mime_ask_action_dialog", 
					 &dialogWidget, dialog);

	nsCOMPtr<nsIDOMWindow> parent = do_QueryInterface (mContext);
	GtkWidget *aParentWidget = mozilla_find_gtk_parent (parent);

	if (aParentWidget)
		gnome_dialog_set_parent (GNOME_DIALOG(dialogWidget),
					 GTK_WINDOW(aParentWidget));
	window_set_layer (dialogWidget);

	label = glade_xml_get_widget (dialog->gxml, "mime_ask_action_message");
	msg = g_strdup_printf (_("The url %s has MIME type %s.\n"
				 "What do you want to do with it?"),
			       mUrl, mMimeType);
	gtk_label_set_text (GTK_LABEL (label), msg);
	g_free (msg);
	return NS_OK;
}

NS_IMETHODIMP GContentHandler::ChooseHelperApp (void)
{
	HelperAppChooserDialog *dialog = new HelperAppChooserDialog;

	dialog->mHelperApps = gnome_vfs_mime_get_all_applications (mMimeType);
	dialog->mContentHandler = this;
	dialog->mGXML = glade_widget_new ("galeon.glade",
					  "helper_chooser_dialog", 
					  &(dialog->mDialogWidget), dialog);

	dialog->mOptionMenu = glade_xml_get_widget (dialog->mGXML,
						    "option_menu");
	dialog->mBrowseButton = glade_xml_get_widget (dialog->mGXML,
						      "browse_button");
	dialog->mHelperEntry = glade_xml_get_widget (dialog->mGXML,
						     "other_helper_entry");

	nsCOMPtr<nsIDOMWindow> parent = do_QueryInterface (mContext);
	GtkWidget *aParentWidget = mozilla_find_gtk_parent (parent);

	if (aParentWidget)
		gnome_dialog_set_parent (GNOME_DIALOG(dialog->mDialogWidget),
					 GTK_WINDOW(aParentWidget));
	window_set_layer (dialog->mDialogWidget);

	GtkWidget *menu = gtk_menu_new ();
	for (GList *helperApp = g_list_first(dialog->mHelperApps) ;
	     helperApp!=NULL ;
	     helperApp = g_list_next(helperApp))
	{
		GtkWidget *menuItem = gtk_menu_item_new_with_label (
			((GnomeVFSMimeApplication *)(helperApp->data))->name);
		gtk_menu_append (GTK_MENU(menu),menuItem);
		gtk_widget_show (menuItem);
	};

	gtk_option_menu_remove_menu (GTK_OPTION_MENU(dialog->mOptionMenu));
	gtk_option_menu_set_menu (GTK_OPTION_MENU(dialog->mOptionMenu),
				  menu);

	GtkWidget *label = glade_xml_get_widget (dialog->mGXML, "mime_label");
	char *msg = g_strdup_printf (_("MIME type: %s"), mMimeType);
	gtk_label_set_text (GTK_LABEL(label), msg);
	g_free (msg);

	return NS_OK;
}

//------------------------------------------------------------------------------

NS_DEF_FACTORY (GContentHandler, GContentHandler);

/**
 * NS_NewContentHandlerFactory:
 */ 
nsresult NS_NewContentHandlerFactory(nsIFactory** aFactory)
{
	NS_ENSURE_ARG_POINTER(aFactory);
	*aFactory = nsnull;

	nsGContentHandlerFactory *result = new nsGContentHandlerFactory;
	if (result == NULL)
	{
		return NS_ERROR_OUT_OF_MEMORY;
	};
    
	NS_ADDREF(result);
	*aFactory = result;

	return NS_OK;
}

//------------------------------------------------------------------------------

GDownloadProgressListener::
	GDownloadProgressListener(nsIHelperAppLauncher *aLauncher,
				  nsISupports *aContext,
				  GContentHandler *aHandler,
				  PRBool aUsedByHelper)
{
	NS_INIT_ISUPPORTS();
	
	/* member initializers and constructor code */
	nsresult rv;
	
	mInterval = 500000; //in microsecs == 500ms == 0.5s
	mPriorKRate = 0;
	mRateChanges = 0;
	mRateChangeLimit = 2; //Only update rate every second
	
	mLauncher = aLauncher;
	mContext = aContext;
	mHandler = aHandler;
	mUsedByHelper = aUsedByHelper;

	rv = mLauncher->GetDownloadInfo(getter_AddRefs(mUri),
					&mTimeDownloadStarted,
					getter_AddRefs(mFile));

	nsCOMPtr<nsIDOMWindow> parent = do_QueryInterface (mContext);
	GtkWidget *aParentWidget = mozilla_find_gtk_parent (parent);

	GladeXML *gxml = glade_widget_new ("galeon.glade", 
					   "dl_progress_dialog",
					   &mProgressDialog, this);

	mProgressBar = glade_xml_get_widget (gxml, "progressbar");
	mLocation = glade_xml_get_widget (gxml, "location_entry");
	mFileName = glade_xml_get_widget (gxml, "filename_entry");
	mStatus = glade_xml_get_widget (gxml, "status_entry");
	mTimeElapsed = glade_xml_get_widget (gxml, "elapsed_entry");
	mTimeRemaining = glade_xml_get_widget (gxml, "remaining_entry");

	if (aParentWidget)
		gnome_dialog_set_parent (GNOME_DIALOG(mProgressDialog),
					 GTK_WINDOW(aParentWidget));

	PRInt64 now = PR_Now ();
	mLastUpdate = now;
#if MOZILLA_VERSION > VERSION3(0,9,1)
	mStartTime = mTimeDownloadStarted;
#else
	mStartTime = now;
#endif
	mElapsed = now - mStartTime;

	char *text;

	rv = mUri->GetSpec (&text);
	gtk_entry_set_text (GTK_ENTRY(mLocation),text);
	g_free (text);
	
	rv = mFile->GetPath (&text);
	gtk_entry_set_text (GTK_ENTRY(mFileName),text);
	g_free (text);
	
	gtk_label_set_text (GTK_LABEL(mTimeElapsed),
			    FormatTime(mElapsed/1000000));
	gtk_label_set_text (GTK_LABEL(mTimeRemaining),FormatTime(0));

	gtk_widget_show (mProgressDialog);
	window_set_layer (mProgressDialog);

	aLauncher->SetWebProgressListener (this);
}

GDownloadProgressListener::~GDownloadProgressListener()
{
  /* destructor code */
}

/* void onStateChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in long aStateFlags, in unsigned long aStatus); */
NS_IMETHODIMP GDownloadProgressListener::
			OnStateChange(nsIWebProgress *aWebProgress,
				      nsIRequest *aRequest,PRInt32 aStateFlags,
				      PRUint32 aStatus)
{
	if (aStateFlags & nsIWebProgressListener::STATE_STOP)
	{
		if (GTK_IS_WIDGET(mProgressDialog))
			gtk_widget_destroy (mProgressDialog);
		mLauncher->CloseProgressWindow ();
	}

	if (mUsedByHelper)
		mHandler->LaunchHelperApp ();

	return NS_OK;
}

/* void onProgressChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in long aCurSelfProgress, in long aMaxSelfProgress, in long aCurTotalProgress, in long aMaxTotalProgress); */
NS_IMETHODIMP GDownloadProgressListener::
			OnProgressChange(nsIWebProgress *aWebProgress,
					 nsIRequest *aRequest,
					 PRInt32 aCurSelfProgress,
					 PRInt32 aMaxSelfProgress,
					 PRInt32 aCurTotalProgress,
					 PRInt32 aMaxTotalProgress)
{
	char *text, *rateStr, *totalStr;
	
	PRInt64 now = PR_Now ();

	if ((now - mLastUpdate < mInterval) && 
	     (aMaxTotalProgress != -1) &&  
	     (aCurTotalProgress < aMaxTotalProgress))
		return NS_OK;

	mLastUpdate = now;
	mElapsed = now - mStartTime;

	gtk_label_set_text (GTK_LABEL(mTimeElapsed),
			    FormatTime(mElapsed/1000000));

	PRInt32 currentKBytes = (PRInt32)(aCurTotalProgress/1024 +.5);

	PRInt32 totalKBytes = (PRInt32)(aMaxTotalProgress/1024 +.5);
	if (aMaxTotalProgress != -1)
	{
		gfloat progress = (gfloat)aCurTotalProgress/
				  (gfloat)aMaxTotalProgress;
		gtk_progress_set_percentage (GTK_PROGRESS(mProgressBar),
					     progress);
		totalStr = g_strdup_printf ("%d",totalKBytes);
	}
	else
	{
		gtk_progress_set_format_string (GTK_PROGRESS(mProgressBar),
						"?? %%");
		totalStr = g_strdup ("??");
	}

	PRInt64 currentRate;
	if (mElapsed)
		currentRate = ((PRInt64)(aCurTotalProgress))*1000000 / mElapsed;
	else
		currentRate = 0;
		
	if (currentRate)
	{
		PRFloat64 currentKRate = ((PRFloat64)currentRate)/1024;
		if (currentKRate != mPriorKRate)
		{
			if (mRateChanges++ == mRateChangeLimit)
			{
				mPriorKRate = currentKRate;
				mRateChanges = 0;
			}
			else
			{
				currentKRate = mPriorKRate;
			}
		}
		else
		{
			mRateChanges = 0;
		}
		rateStr = g_strdup_printf("%.2f",currentKRate);
	}
	else
	{
		rateStr = g_strdup ("??.??");
	}

	text = g_strdup_printf (_("%dK of %sK bytes at %sK bytes/sec"),
				currentKBytes, totalStr, rateStr);
	gtk_label_set_text (GTK_LABEL(mStatus),text);
	g_free (text);
	g_free (totalStr);
	g_free (rateStr);

	if (currentRate && (aMaxTotalProgress != -1))
	{
		PRInt32 remaining = 
			(PRInt32)((aMaxTotalProgress - aCurTotalProgress)
				   /currentRate +.5);
		gtk_label_set_text (GTK_LABEL(mTimeRemaining),
				    FormatTime(remaining));
	}
	else
	{
		gtk_label_set_text (GTK_LABEL(mTimeRemaining), _("Unknown"));
	}

	return NS_OK;
}

/* void onLocationChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in nsIURI location); */
NS_IMETHODIMP GDownloadProgressListener::
			OnLocationChange(nsIWebProgress *aWebProgress,
					 nsIRequest *aRequest, nsIURI *location)
{
    return NS_OK;
}

/* void onStatusChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in nsresult aStatus, in wstring aMessage); */
NS_IMETHODIMP GDownloadProgressListener::
			OnStatusChange(nsIWebProgress *aWebProgress,
				       nsIRequest *aRequest, nsresult aStatus,
				       const PRUnichar *aMessage)
{
    return NS_OK;
}

/* void onSecurityChange (in nsIWebProgress aWebProgress, in nsIRequest aRequest, in long state); */
NS_IMETHODIMP GDownloadProgressListener::
			OnSecurityChange(nsIWebProgress *aWebProgress,
					 nsIRequest *aRequest, PRInt32 state)
{
    return NS_OK;
}

NS_IMETHODIMP GDownloadProgressListener::CancelHelperProgress (void)
{
	mUsedByHelper = PR_FALSE;
	return mLauncher->Cancel ();
}

char *GDownloadProgressListener::FormatTime (PRUint32 aTime)
{
	PRUint32 secs = (PRUint32)(aTime+.5);
	PRUint32 hours = secs/3600;
	secs -= hours*3600;
	PRUint32 mins = secs/60;
	secs -= mins*60;
	char *result;
	if (hours)
		result = g_strdup_printf ("%u:%02u.%02u",hours,mins,secs);
	else
		result = g_strdup_printf ("%02u.%02u",mins,secs);
	return result;
}

////////////////////////////////////////////////////////////////////////////////
// begin MIMEAskActionDialog callbacks.
////////////////////////////////////////////////////////////////////////////////

void
mime_ask_dialog_ok_clicked_cb (GtkButton *button, MimeAskActionDialog *dialog)
{
	MimeAction action;
	GtkWidget *save_check = glade_xml_get_widget 
		(dialog->gxml, "mime_action_save");
	GtkWidget *save_gtm_check = glade_xml_get_widget
		(dialog->gxml, "mime_action_save_gtm");
	GtkWidget *run_program_check = 
		glade_xml_get_widget (dialog->gxml, "mime_action_run_program");
	GtkWidget *remember_check =
		glade_xml_get_widget (dialog->gxml, "mime_remember_check");

	gtk_widget_hide (glade_xml_get_widget (dialog->gxml, 
					       "mime_ask_action_dialog"));

	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (save_check)))
		action = MIME_SAVE_TO_DISK;
	else if (gtk_toggle_button_get_active 
			(GTK_TOGGLE_BUTTON (save_gtm_check)))
		action = MIME_SAVE_WITH_GTM;
	else if (gtk_toggle_button_get_active 
			(GTK_TOGGLE_BUTTON (run_program_check)))
		action = MIME_RUN_PROGRAM;
	else
		action = MIME_UNKNOWN;

	nsresult rv;
	nsCOMPtr<nsIHelperAppLauncher> aLauncher;
	rv = dialog->mContentHandler->GetLauncher (getter_AddRefs(aLauncher));
	
	switch (action) {
	case MIME_SAVE_TO_DISK:
		aLauncher->SaveToDisk (nsnull,PR_FALSE);
		break;
	case MIME_SAVE_WITH_GTM:
		aLauncher->Cancel ();
		dialog->mContentHandler->LaunchExternalDownloader ();
		break;
	case MIME_RUN_PROGRAM:
		dialog->mContentHandler->FindHelperApp ();
		break;
	case MIME_ASK_ACTION:
	case MIME_UNKNOWN:
	case MIME_OVERRIDE_MOZILLA:
	case MIME_IGNORE:
		/* shouldn't get here */
		g_warning ("Unexpected MIME action");
		break;
	};
	
	/* check if this should be the new default action */
	if (gtk_toggle_button_get_active (GTK_TOGGLE_BUTTON (remember_check))) 
		mime_set_action (dialog->mime_type, action);

	g_free (dialog->mime_type);
	g_free (dialog->url);
	gtk_widget_destroy (glade_xml_get_widget (dialog->gxml, 
						  "mime_ask_action_dialog"));
	gtk_object_unref (GTK_OBJECT (dialog->gxml));
	delete dialog;
}

gint
mime_ask_dialog_cancel_clicked_cb (GtkButton *button,
				   MimeAskActionDialog *dialog)
{
	nsresult rv;
	nsCOMPtr<nsIHelperAppLauncher> aLauncher;
	rv = dialog->mContentHandler->GetLauncher (getter_AddRefs(aLauncher));

	aLauncher->Cancel ();
	g_free (dialog->mime_type);
	g_free (dialog->url);
	gtk_widget_destroy (glade_xml_get_widget (dialog->gxml, 
						  "mime_ask_action_dialog"));
	gtk_object_unref (GTK_OBJECT (dialog->gxml));
	delete dialog;
	return 0; /* FIXME: philipl, is this the right thing to return? */
}

//------------------------------------------------------------------------------

HelperAppChooserDialog::HelperAppChooserDialog (void)
{
	mAutoDefaultChecked = FALSE;
	mOtherHelperApp = FALSE;
}

////////////////////////////////////////////////////////////////////////////////
// begin HelperAppChooserDialog callbacks.
////////////////////////////////////////////////////////////////////////////////

void
helper_chooser_ok_clicked_cb (GtkButton *button,
			      HelperAppChooserDialog *dialog)
{
	GnomeVFSMimeApplication *helperApp;
	if (dialog->mOtherHelperApp == FALSE)
	{
		GtkWidget *menu = GTK_OPTION_MENU (dialog->mOptionMenu)->menu;
		GList *menuItems = GTK_MENU_SHELL (menu)->children;
		gpointer item = gtk_menu_get_active (GTK_MENU(menu));
		gint index = g_list_index (menuItems,item);
		GList *helperApps = g_list_nth(dialog->mHelperApps,index);
		helperApp = (GnomeVFSMimeApplication *)(helperApps->data);
	}
	else
	{
		helperApp = new GnomeVFSMimeApplication;
		helperApp->command = g_strdup (gtk_entry_get_text(
						GTK_ENTRY(dialog->mHelperEntry)));
		helperApp->id = g_strdup ("OtHeR");
		helperApp->requires_terminal = FALSE;
		helperApp->expects_uris =
				GNOME_VFS_MIME_APPLICATION_ARGUMENT_TYPE_PATHS;
	};

	dialog->mContentHandler->SetHelperApp (helperApp);
	dialog->mContentHandler->ShowHelperProgressDialog ();

	gnome_vfs_mime_application_list_free (dialog->mHelperApps);

	gtk_widget_destroy (dialog->mDialogWidget);
	gtk_object_unref (GTK_OBJECT (dialog->mGXML));
	delete dialog;
}

gint
helper_chooser_cancel_clicked_cb (GtkButton *button,
				  HelperAppChooserDialog *dialog)
{
	nsresult rv;
	nsCOMPtr<nsIHelperAppLauncher> aLauncher;
	rv = dialog->mContentHandler->GetLauncher (getter_AddRefs(aLauncher));

	aLauncher->Cancel ();

	gnome_vfs_mime_application_list_free (dialog->mHelperApps);

	gtk_widget_destroy (dialog->mDialogWidget);
	gtk_object_unref (GTK_OBJECT (dialog->mGXML));
	delete dialog;
	
	return 0; /* FIXME: philipl, is this the right thing to return? */
}

void helper_chooser_browse_clicked_cb (GtkButton *button,
				       HelperAppChooserDialog *dialog)
{
	char *fileName = g_strdup (g_get_home_dir());

	nsCOMPtr<nsISupports> aContext;
	dialog->mContentHandler->GetContext (getter_AddRefs(aContext));
	nsCOMPtr<nsIDOMWindowInternal> aWindowInternal = 
				do_QueryInterface (aContext);

	nsCOMPtr<nsILocalFile> aSaveDir = 
			do_CreateInstance (NS_LOCAL_FILE_CONTRACTID);
	aSaveDir->InitWithPath (fileName);

	nsCOMPtr<nsIFilePicker> aFilePicker =
			do_CreateInstance (G_FILEPICKER_CONTRACTID);

	nsString title = NS_LITERAL_STRING (N_("Choose Helper"
					       " Application"));

	aFilePicker->Init (aWindowInternal, title.get(), 
			   nsIFilePicker::modeOpen);
	aFilePicker->SetDefaultString (nsnull);
	aFilePicker->SetDisplayDirectory (aSaveDir);
		
	PRInt16 okToSave;
	aFilePicker->Show (&okToSave);

	g_free (fileName);
	if (okToSave == nsIFilePicker::returnOK)
	{
		nsCOMPtr<nsILocalFile> aFile;
		aFilePicker->GetFile (getter_AddRefs(aFile));
		aFile->GetPath (&fileName);
		gtk_entry_set_text (GTK_ENTRY(dialog->mHelperEntry),
				    fileName);
		nsMemory::Free (fileName);
	};
}

void helper_chooser_auto_default_toggled (GtkToggleButton *button,
					  HelperAppChooserDialog *dialog)
{
	dialog->mAutoDefaultChecked = gtk_toggle_button_get_active (button);
}

void helper_chooser_radio_list_toggled (GtkToggleButton *button,
					HelperAppChooserDialog *dialog)
{
	gboolean active = gtk_toggle_button_get_active (button);
	if (active == TRUE)
		dialog->mOtherHelperApp = FALSE;
	gtk_widget_set_sensitive (dialog->mOptionMenu, active);
}

void helper_chooser_radio_other_toggled (GtkToggleButton *button,
					 HelperAppChooserDialog *dialog)
{
	gboolean active = gtk_toggle_button_get_active (button);
	if (active == TRUE)
		dialog->mOtherHelperApp = TRUE;
	gtk_widget_set_sensitive (dialog->mBrowseButton, active);
	gtk_widget_set_sensitive (dialog->mHelperEntry, active);
}

//////////////////////////////////////////////////////////////////////////////
// begin ProgressDialog callbacks.
//////////////////////////////////////////////////////////////////////////////

gint
progress_dialog_cancel_cb (GtkButton *button,
			   GDownloadProgressListener *aListener)
{
	aListener->CancelHelperProgress ();
	return 0; /* FIXME: philipl, is this the right thing to return? */
}
