/*
    parted - a frontend to libparted
    Copyright (C) 1999, 2000 Free Software Foundation, Inc.

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include <parted/parted.h>

#include "../config.h"
#include "command.h"
#include "strlist.h"
#include "ui.h"

#include <libintl.h>
#include <locale.h>
#define N_(String) String
#if ENABLE_NLS
#  define _(String) dgettext (PACKAGE, String)
#else
#  define _(String) (String)
#endif /* ENABLE_NLS */

#ifdef HAVE_LIBREADLINE

#ifdef HAVE_TERMCAP_H
#include <termcap.h>
#else
extern int tgetnum (char* key);
#endif

#include <readline/readline.h>
#include <readline/history.h>

#ifndef HAVE_RL_COMPLETION_MATCHES
#define rl_completion_matches completion_matches
#endif

#ifndef rl_compentry_func_t
#define rl_compentry_func_t void
#endif

#endif /* HAVE_LIBREADLINE */

#include <ctype.h>
#include <signal.h>
#include <stdlib.h>
#include <string.h>
#include <unistd.h>

char* prog_name = "GNU Parted " VERSION "\n";

static char* banner_msg = N_(
"Copyright (C) 1998, 1999, 2000, 2001 Free Software Foundation, Inc.\n"
"This program is free software, covered by the GNU General Public License.\n"
"\n"
"This program is distributed in the hope that it will be useful, "
"but WITHOUT ANY WARRANTY; without even the implied warranty of "
"MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the "
"GNU General Public License for more details.\n");

static char* usage_msg = N_(
"Usage: parted [OPTION]... [DEVICE [COMMAND [PARAMETERS]...]...]\n"
"Apply COMMANDs with PARAMETERS to DEVICE.  If no COMMAND(s) are given, runs in\n"
"interactive mode.\n");

static char* bug_msg = N_(
"You found a bug in GNU Parted.  Please email a bug report to bug-parted@gnu.org "
"containing the version (%s), and the following message:\n");

static char*			words [256];
static int			word_num;
static Command**		commands;
static StrList*			ex_opt_str [64];
static PedExceptionOption	current_exception_opt = 0;

int
screen_width ()
{
	int	width = 0;

/* HACK: don't specify termcap separately - it'll annoy the users. */
#ifdef HAVE_LIBREADLINE
	width = tgetnum ("co");
#endif

	if (width <= 0)
		width = 80;

	return width;
}

char*
_readline (char* prompt)
{
	char*		line;
	char*		result;

#ifdef HAVE_LIBREADLINE
	line = readline (prompt);
	if (line)
		add_history (line);
	return line;
#else
	printf ("%s", prompt);
	fflush (stdout);
	line = (char*) malloc (256);
	result = fgets (line, 256, stdin);
	if (result) {
		result [strlen (result) - 1] = 0;	/* kill trailing CR */
		return result;
	} else {
		free (line);
		return NULL;
	}
#endif
}

static PedExceptionOption
option_get_next (PedExceptionOption options, PedExceptionOption current)
{
	PedExceptionOption	i;

	if (current == 0)
		i = PED_EXCEPTION_OPTION_FIRST;
	else
		i = current * 2;

	for (; i <= options; i *= 2) {
		if (options & i)
			return i;
	}

	return 0;
}

static int
log2 (int x)
{
	int	i;
	for (i=0; (1 << i) <= x; i++) ;
	return i - 1;
}

static StrList*
option_get_str_list (PedExceptionOption opt)
{
	return ex_opt_str [log2 (opt)];
}

static PedExceptionOption
match_exception_option (PedExceptionOption options, char* word)
{
	PedExceptionOption	opt;
	PedExceptionOption	partial_match = 0;
	int			ambiguous = 0;
	int			status;

	for (opt = option_get_next (options, 0); opt; 
	     opt = option_get_next (options, opt)) {
		status = str_list_match (option_get_str_list (opt), word);
		if (status == 2)
			return opt;
		if (status == 1 && !ambiguous) {
			if (partial_match)
				ambiguous = 1;
			else
				partial_match = opt;
		}
	}

	if (ambiguous)
		return 0;
	else
		return partial_match;
}

static void
_print_exception_text (PedException* ex)
{
	StrList*		text;

	if (ex->type == PED_EXCEPTION_BUG) {
		printf (bug_msg, VERSION);
		text = str_list_create ("\n", ex->message, "\n", NULL);
	} else {
		text = str_list_create (
			   _(ped_exception_get_type_string (ex->type)),
			   ": ", ex->message, "\n", NULL);
	}

	str_list_print_wrap (text, screen_width (), 0, 0);
	str_list_destroy (text);
}

static void
_build_options_str (PedException* ex, char* options_str)
{
	PedExceptionOption	opt;

	strcpy (options_str, "");
	for (opt = option_get_next (ex->options, 0); opt; 
	     opt = option_get_next (ex->options, opt)) {
		strcat (options_str, _(ped_exception_get_option_string (opt)));
		strcat (options_str, " ");
	}
	strcat (options_str, "? ");
}

static PedExceptionOption
exception_handler (PedException* ex)
{
	PedExceptionOption	opt;
	char			options_str [512];
	char*			line;
	char*			word;

	_print_exception_text (ex);

/* only one choice?  Take it ;-) */
	opt = option_get_next (ex->options, 0);
	if (!option_get_next (ex->options, opt))
		return opt;

/* script-mode: don't handle the exception */
	if (opt_script_mode)
		return PED_EXCEPTION_UNHANDLED;

	_build_options_str (ex, options_str);

	current_exception_opt = ex->options;	/* for readline stuff (yuck) */

	while (1) {
		line = _readline (options_str);
		if (!line)
			break;

		word = strtok (line, " \t");
		if (!word)
			goto next_line;

		opt = match_exception_option (ex->options, word);
		if (opt)
			break;

	next_line:
		free (line);
	}

	current_exception_opt = 0;

	free (line);
	return opt;
}

#ifdef HAVE_LIBREADLINE
/* returns matching commands for text */
static int last_match_cmd;
static StrList* last_match_str;
static char*
command_generator (char* text, int state)
{
	int		i = (state)? last_match_cmd : 0;
	int		skip = state;
	StrList*	walk;

	for (; commands [i]; i++) {
		if (skip) {
			walk = last_match_str->next;
			skip = 0;
		} else {
			walk = commands [i]->names;
		}
		for (; walk; walk = walk->next) {
			if (str_list_match_node (walk, text)) {
				last_match_cmd = i;
				last_match_str = walk;
				return str_list_convert_node (walk);
			}
		}
	}

	return NULL;
}

static PedExceptionOption ex_opt;
char*
exception_command_generator (char* text, int state)
{
	int	text_length = strlen (text);
	char*	opt_str;

	if (state)
		ex_opt = option_get_next (current_exception_opt, ex_opt);
	else
		ex_opt = option_get_next (current_exception_opt, 0);

	for (; ex_opt;
	     ex_opt = option_get_next (current_exception_opt, ex_opt)) {
		opt_str = _(ped_exception_get_option_string (ex_opt));
		if (strncasecmp (opt_str, text, text_length) == 0)
			return strdup (opt_str);
	}

	return NULL;
}

/* completion function for readline() */
char**
complete_function (char* text, int start, int end)
{
	if (current_exception_opt)
		return rl_completion_matches (
				text,
				(rl_compentry_func_t*)
					exception_command_generator);
	else
		return rl_completion_matches (
				text,
				(rl_compentry_func_t*) command_generator);
}
#endif /* HAVE_LIBREADLINE */

void
interrupt_handler (int signum)
{
	signal (signum, &interrupt_handler);
}

char*
get_word ()
{
	return words [word_num++];
}

int
get_integer ()
{
	return atoi (words [word_num++]);
}

/* input (from command) is in megabytes (possibly with decimals), output is in
 * sectors */
PedSector
get_sector ()
{
	return atof (words [word_num++]) * MEGABYTE_SECTORS;
}

int
get_state ()
{
	char*	word;

	word = get_word();
	if (strcasecmp (_("on"), word) == 0 || strcasecmp ("on", word) == 0)
		return 1;
	return 0;
}

char*
peek_word ()
{
	return words [word_num];
}

int
is_integer ()
{
	char*		word = peek_word ();

	if (!word)
		return 0;
	for (; *word; word++) {
		if (!isdigit (*word))
			return 0;
	}
	return 1;
}

int
is_sector ()
{
	char*		word = peek_word ();

	if (!word)
		return 0;
	for (; *word; word++) {
		if (!isdigit (*word) && *word != '.')
			return 0;
	}
	return 1;
}

int
count_words ()
{
	int	i;

	for (i=0; words [i]; i++);
	return i - word_num;
}

void
free_line ()
{
	int	i;

	for (i=0; words [i]; i++)
		free (words[i]);
}

void
parse_line (char* line)
{
	int	quoted = 0;
	char	quote_char = 0;	/* shut gcc up! */
	char	this_word [256];
	int	i;

	free_line ();

	word_num = 0;
	this_word [i=0] = 0;

	for (; 1; line++) {
		if (!*line || (!quoted && *line == ' ')) {
			this_word [i] = 0;
			words [word_num++] = strdup (this_word);
			while (*line == ' ')
			       	line++;
			if (!*line)
				break;
			i=0;
		}
		if (quoted && *line == quote_char) {
			quoted = 0;
			continue;
		}
		if (!quoted && (*line == '\'' || *line == '"')) {
			quoted = 1;
			quote_char = *line;
			continue;
		}
		this_word [i++] = *line;
	}

	words [word_num] = NULL;
	word_num = 0;
}

void
init_ex_opt_str ()
{
	int			i;
	PedExceptionOption	opt;

	for (i = 0; (1 << i) <= PED_EXCEPTION_OPTION_LAST; i++) {
		opt = (1 << i);
		ex_opt_str [i]
			= str_list_create (
				ped_exception_get_option_string (opt),
				_(ped_exception_get_option_string (opt)),
				NULL);
	}

	ex_opt_str [i] = NULL;
}

void
done_ex_opt_str ()
{
	int	i;

	for (i=0; ex_opt_str [i]; i++)
		str_list_destroy (ex_opt_str [i]);
}

int
init_ui ()
{
	opt_script_mode = !isatty (0);

	if (!init_str_list ())
		return 0;

	init_ex_opt_str ();
	ped_exception_set_handler (exception_handler);

	signal (SIGINT, &interrupt_handler);

#ifdef HAVE_LIBREADLINE
	rl_initialize ();
	rl_attempted_completion_function = (CPPFunction*) complete_function;
#endif

	return 1;
}

void
done_ui ()
{
	done_ex_opt_str ();
	done_str_list ();
}

void
help_msg ()
{
	printf (_(usage_msg));

	printf ("\n%s\n", _("OPTIONs:"));
	print_options_help ();

	printf ("\n%s\n", _("COMMANDs:"));
	print_commands_help ();
	exit (0);
}

void
print_using_dev (PedDevice* dev)
{
	printf (_("\nUsing %s\n"), dev->path);

#if defined (__i386__)
	if (dev->type != PED_DEVICE_UNKNOWN
	    && dev->type != PED_DEVICE_FILE) {
		if (dev->cylinders > 1024)
			ped_exception_throw (
				PED_EXCEPTION_WARNING,
				PED_EXCEPTION_OK,
			_("The operating system thinks the geometry on %s is "
			  "%d/%d/%d.  Therefore, cylinder 1024 ends at %.3fM.  "
			  "You should check that this matches the BIOS "
			  "geometry before using this program."),
				dev->path,
				dev->cylinders,
				dev->heads,
				dev->sectors,
				(1024 * dev->heads * dev->sectors - 1)
					/ 2 / 1024.0);
		else
			ped_exception_throw (
				PED_EXCEPTION_WARNING,
				PED_EXCEPTION_OK,
			_("The operating system thinks the geometry on %s is "
			  "%d/%d/%d.  You should check that this matches the "
			  "BIOS geometry before using this program."),
				dev->path,
				dev->cylinders,
				dev->heads,
				dev->sectors);
	}
#endif
}

int
interactive_mode (PedDevice* dev, Command* cmd_list[])
{
	char*		line;
	Command*	cmd;
	StrList*	list;

	commands = cmd_list;	/* FIXME yucky, nasty, evil hack */

	printf ("%s", prog_name);

	list = str_list_create (_(banner_msg), NULL);
	str_list_print_wrap (list, screen_width (), 0, 0);
	str_list_destroy (list);

	print_using_dev (dev);

	while (1) {
		line = _readline ("(parted) ");
		if (!line) break;
		parse_line (line);

		if (!peek_word ()) {
			print_commands_help ();
			continue;
		}

		cmd = command_get (commands, get_word ());
		if (cmd)
			command_run (cmd, &dev);
		else
			print_commands_help ();

		free (line);
	}

	free (line);
	return 1;
}

int
non_interactive_mode (PedDevice* dev, Command* cmd_list[],
		      int argc, char* argv[])
{
	int		i;
	Command*	cmd;

	commands = cmd_list;	/* FIXME yucky, nasty, evil hack */

	for (i = 0; i < argc; i++)
		words [i] = argv[i];
	words [argc] = NULL;
	word_num = 0;

	while (peek_word ()) {
		cmd = command_get (commands, get_word ());
		if (!cmd) {
			help_msg ();
			goto error;
		}
		if (!command_run (cmd, &dev))
			goto error;
	}

	return 1;

error:
	return 0;
}

