/*
    parted - a frontend to libparted
    Copyright (C) 1999  Andrew Clausen, Lennert Buytenhek and Red Hat, Inc.

    Andrew Clausen		<clausen@gnu.org>
    Lennert Buytenhek		<buytenh@gnu.org>
    Matt Wilson, Red Hat Inc.	<msw@redhat.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "config.h"

#include <parted/debug.h>

#include <ctype.h>
#include <errno.h>
#include <stdarg.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef ENABLE_NLS

#include <iconv.h>
#include <wchar.h>
#include <wctype.h>

#else /* ENABLE_NLS */

#ifdef wchar_t
#undef wchar_t
#endif

#define wchar_t char

#endif /* !ENABLE_NLS */

#include "strlist.h"

#define MIN(a,b)	( (a<b)?  a : b )

#ifdef ENABLE_NLS

char*			language;
static char*		gettext_charset;
static char*		term_charset;
static iconv_t		gettext_to_wchar_handle;
static iconv_t		wchar_to_term_handle;

#endif /* ENABLE_NLS */

int
wchar_strlen (const wchar_t* str)
{
#ifdef ENABLE_NLS
	return wcslen (str);
#else
	return strlen (str);
#endif
}

wchar_t*
wchar_strchr (const wchar_t* str, char ch)
{
#ifdef ENABLE_NLS
	return wcschr (str, ch << (3 * 8));
#else
	return strchr (str, ch);
#endif
}

int
wchar_strcasecmp (const wchar_t* a, const wchar_t* b)
{
#ifdef ENABLE_NLS
	return wcscasecmp (a, b);
#else
	return strcasecmp (a, b);
#endif
}

int
wchar_strncasecmp (const wchar_t* a, const wchar_t* b, int n)
{
#ifdef ENABLE_NLS
	return wcsncasecmp (a, b, n);
#else
	return strncasecmp (a, b, n);
#endif
}

static const char*
get_gettext_encoding ()
{
#ifdef ENABLE_NLS
	if (!language)
		return "ISO-8859-1";
	if (strncasecmp (language, "ja", 2) == 0)
		return "EUC-JP";
	if (strncasecmp (language, "ru", 2) == 0)
		return "KOI8-R";
#endif
	return "ISO-8859-1";
}

static const char*
get_terminal_encoding ()
{
#ifdef ENABLE_NLS
	if (!language)
		return "ISO-8859-1";
	if (strncasecmp (language, "ja", 2) == 0)
		return "EUC-JP";
	if (strncasecmp (language, "ru", 2) == 0)
		return "KOI8-R";
#endif
	return "ISO-8859-1";
}

int
init_str_list ()
{
#ifdef ENABLE_NLS
	gettext_to_wchar_handle = iconv_open ("UCS4", get_gettext_encoding ());
	if (gettext_to_wchar_handle == (iconv_t) -1)
		goto error;

	wchar_to_term_handle = iconv_open (get_terminal_encoding (), "UCS4");
	if (wchar_to_term_handle == (iconv_t) -1)
		goto error_close_handle;

	return 1;

error_close_handle:
	iconv_close (gettext_to_wchar_handle);
error:
	printf ("Error initializing character set conversion: %s\n",
		strerror (errno));
	return 0;

#else /* ENABLE_NLS */
	return 1;
#endif /* !ENABLE_NLS */
}

int
done_str_list ()
{
#ifdef ENABLE_NLS
	iconv_close (gettext_to_wchar_handle);
	iconv_close (wchar_to_term_handle);
#endif
	return 1;
}

/* converts a string from the encoding in the gettext catalogues to wide
 * character strings (of type wchar_t*).  Wide character strings are encoded
 * with UCS4.
 */
#ifdef ENABLE_NLS
static wchar_t*
gettext_to_wchar (const char* str)
{
	size_t		in_bytes_left = strlen (str);
	wchar_t*	result = malloc (64);
	char*		out_buf = (char*) result;
	size_t		out_buf_size = 64;
	size_t		out_bytes_left = 60;	/* leave room for terminator */
	size_t		status;

/* FIXME: add this code when glibc is fixed */
/*
	status = iconv (gettext_to_wchar_handle,
			NULL, NULL,
			&out_buf, &out_bytes_left);
	if (status == (iconv_t) -1)
		goto error;
*/

	while (1) {
		status = iconv (gettext_to_wchar_handle,
			   &str, &in_bytes_left,
			   &out_buf, &out_bytes_left);
		if (status == -1 && errno != E2BIG)
			goto error;

		if (!in_bytes_left)
			break;

		out_buf_size += 64;
		out_bytes_left += 64;
		result = realloc (result, out_buf_size);
		out_buf = (char*) result + (out_buf_size-4) - out_bytes_left;
	}

	* (wchar_t*) out_buf = 0;
	return (wchar_t*) result;

error:
	printf ("Error during translation: %s\n", strerror (errno));
	exit (1);
}

#else /* ENABLE_NLS */

static wchar_t*
gettext_to_wchar (const char* str)
{
	return strdup (str);
}

#endif /* !ENABLE_NLS */


#ifdef ENABLE_NLS
static char*
wchar_to_str (const wchar_t* str, int count)
{
	size_t		in_bytes_left = ((count) ? count : wcslen (str))
					* (sizeof (wchar_t));
	char*		result = malloc (64);
	char*		out_buf = result;
	size_t		out_buf_size = 64;
	size_t		out_bytes_left = 63;	/* leave room for terminator */
	size_t		status;

/* FIXME: add this code when glibc is fixed */
/*
	status = iconv (wchar_to_term_handle,
			NULL, NULL,
			&out_buf, &out_bytes_left);
	if (status == (iconv_t) -1)
		goto error;
*/

	while (1) {
		status = iconv (wchar_to_term_handle,
			   (const char**) &str, &in_bytes_left,
			   &out_buf, &out_bytes_left);
		if (status == -1 && errno != E2BIG)
			goto error;

		if (!in_bytes_left)
			break;

		out_buf_size += 64;
		out_bytes_left += 64;
		result = realloc (result, out_buf_size);
		out_buf = result + (out_buf_size-1) - out_bytes_left;
	}

	*out_buf = 0;
	return result;

error:
	printf ("Error during translation: %s\n", strerror (errno));
	exit (1);
}

#else /* ENABLE_NLS */

static char*
wchar_to_str (const wchar_t* str, int count)
{
	char*		result;

	result = strdup (str);
	if (count && count < strlen (result))
		result [count] = 0;
	return result;
}

#endif /* !ENABLE_NLS */

static void
print_wchar (const wchar_t* str, int count)
{
	char*	tmp = wchar_to_str (str, count);
	printf ("%s", tmp);
	free (tmp);
}

static StrList* 
str_list_alloc ()
{
	StrList*	list;

	list = (StrList*) malloc (sizeof (StrList));
	list->next = NULL;

	return list;
}

void
str_list_destroy (StrList* list)
{
	if (list->next)
		str_list_destroy (list->next);
	free ((wchar_t*) list->str);
	free (list);
}

static StrList*
_str_list_append (StrList* list, const wchar_t* str)
{
	StrList*	walk;

	if (list) {
		for (walk = list; walk->next; walk = walk->next);
		walk->next = str_list_alloc ();
		walk = walk->next;
	} else {
		walk = list = str_list_alloc ();
	}
	walk->str = str;

	return list;
}

StrList*
str_list_append (StrList* list, const char* str)
{
	return _str_list_append (list, gettext_to_wchar (str));
}

StrList*
str_list_append_unique (StrList* list, const char* str)
{
	StrList*	walk;
	wchar_t*	new_str = gettext_to_wchar (str);

	for (walk=list; walk; walk=walk->next) {
		if (walk->str) {
			if (wchar_strcasecmp (new_str, walk->str) == 0) {
				free (new_str);
				return list;
			}
		}
	}

	return _str_list_append (list, new_str);
}

StrList*
str_list_create (const char* first, ...)
{
	va_list		args;
	char*		str;
	StrList*	list;

	list = str_list_append (NULL, first);

	if (first) {
		va_start (args, first);
		while ( (str = va_arg (args, char*)) )
			str_list_append (list, str);
		va_end (args);
	}

	return list;
}

StrList*
str_list_create_unique (const char* first, ...)
{
	va_list		args;
	char*		str;
	StrList*	list;

	list = str_list_append (NULL, first);

	if (first) {
		va_start (args, first);
		while ( (str = va_arg (args, char*)) )
			str_list_append_unique (list, str);
		va_end (args);
	}

	return list;
}

char*
str_list_convert_node (const StrList* list)
{
	return wchar_to_str (list->str, 0);
}

char*
str_list_convert (const StrList* list)
{
	const StrList*	walk;
	int		pos = 0;
	int		length = 1;
	char*		str = strdup ("");

	for (walk = list; walk; walk = walk->next) {
		if (walk->str) {
			char*	tmp = wchar_to_str (walk->str, 0);

			length += strlen (tmp);

			str = realloc (str, length);
			strcpy (str + pos, tmp);

			pos = length - 1;
			free (tmp);
		}
	}

	return str;
}

void
str_list_print (const StrList* list)
{
	const StrList*	walk;

	for (walk=list; walk; walk=walk->next) {
		if (walk->str)
			print_wchar (walk->str, 0);
	}
}

static char*
get_spaces (int space_count)
{
	char*	str;
	int	i;

	str = malloc (space_count + 1);
	for (i = 0; i < space_count; i++)
		str [i] = ' ';
	str [i] = 0;

	return str;
}

static int
str_search (const wchar_t* str, int n, wchar_t c)
{
	int	i;

#ifdef ENABLE_NLS
	for (i=0; i<n; i++)
		if (str [i] == c << (3 * 8))
			return i;
#else
	for (i=0; i<n; i++)
		if (str [i] == c)
			return i;
#endif
	return -1;
}


/* Japanese don't leave spaces between words, so ALL Japanese characters
 * are treated as delimiters.  Note: since the translations should already
 * be properly formatted (eg: spaces after commas), there should be no
 * need to include them.  Best not to avoid side effects, like 3.
14159 :-)
 * FIXME: how do we exclude "." and "(" ?
 * FIXME: glibc doesn't like umlaute.  i.e. \"o (TeX equivalent)
 */

static int
is_break_point (wchar_t c)
{
#ifdef ENABLE_NLS
	return !iswalnum (c >> (3 * 8)) && !iswpunct (c >> (3 * 8));
#else
	return !isalnum (c) && !ispunct (c);
#endif
}

/* NOTE: this should not return '\n' as a space, because explicit '\n' may
 * be placed inside strings.
 */
static int
is_space (wchar_t c)
{
#ifdef ENABLE_NLS
	return (c >> (3 * 8)) == ' ';		/* FIXME: why big-endian? */
#else
	return c == ' ';
#endif
}

void
str_list_print_wrap (const StrList* list, int line_length, int offset,
		     int indent)
{
	const StrList*	walk;
	const wchar_t*	str;
	int		str_len;
	int		cut_right;
	int		cut_left;
	int		line_left;
	char*		spaces;
	int		search_result;
	int		line_break;

	PED_ASSERT (line_length - indent > 10, return);

	spaces = get_spaces (indent);
	line_left = line_length - offset;

	for (walk=list; walk; walk=walk->next) {
		if (!walk->str)
			continue;
		str = walk->str;
		str_len = wchar_strlen (str);

		while (line_left < str_len || wchar_strchr (str, '\n')) {
			line_break = 0;

			cut_left = MIN (line_left - 1, str_len - 1);

			/* we can have a space "over", but not a comma */
			if (cut_left < str_len
					&& is_space (str [cut_left + 1]))
				cut_left++;

			while (cut_left && !is_break_point (str [cut_left]))
				cut_left--;
			while (cut_left && is_space (str [cut_left]))
				cut_left--;

		/* str [cut_left] is either the end of a word, or a
		 * Japanese character, or the start of a blank line.
		 */

			search_result = str_search (str, cut_left + 1, '\n');
			if (search_result != -1) {
				cut_left = search_result - 1;
				line_break = 1;
			}

			for (cut_right = cut_left + 2;
			     cut_right < str_len && is_space (str [cut_right]);
			     cut_right++);

			if (cut_left > 0)
				print_wchar (str, cut_left + 1);

			str += cut_right;
			str_len -= cut_right;
			line_left = line_length - indent;

			if (walk->next || *str)
				printf ("\n%s", spaces);
			else if (line_break)
				printf ("\n");
		}

		print_wchar (str, 0);
		line_left -= wchar_strlen (str);
	}

	free (spaces);
}

static int
_str_list_match_node (StrList* list, wchar_t* str)
{
	if (wchar_strcasecmp (list->str, str) == 0)
		return 2;
	if (wchar_strncasecmp (list->str, str, wchar_strlen (str)) == 0)
		return 1;
	return 0;
}

int
str_list_match_node (StrList* list, char* str)
{
	wchar_t*	wc_str = gettext_to_wchar (str);	/* FIXME */
	int		status;

	status = _str_list_match_node (list, wc_str);
	free (wc_str);

	return status;
}

/* returns:  2 for full match
	     1 for partial match
	     0 for no match
 */
int
str_list_match (StrList* list, char* str)
{
	StrList*	walk;
	int		status;
	wchar_t*	wc_str = gettext_to_wchar (str);	/* FIXME */

	for (walk = list; walk; walk = walk->next) {
		status = _str_list_match_node (walk, wc_str);
		if (status) {
			free (wc_str);
			return status;
		}
	}

	free (wc_str);
	return 0;
}

