 /*
    libparted - a library for manipulating disk partitions
    Copyright (C) 1998-2000  Andrew Clausen, Lennert Buytenhek and Red Hat Inc.

	Andrew Clausen			<clausen@gnu.org>
	Lennert Buytenhek		<buytenh@gnu.org>
	Matt Wilson, Red Hat Inc.	<msw@redhat.com>

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*/

#include "config.h"

#include <parted/parted.h>

#include <libintl.h>
#if ENABLE_NLS
#  define _(String) gettext (String)
#else
#  define _(String) (String)
#endif /* ENABLE_NLS */

#include <errno.h>
#include <string.h>
#include <unistd.h>
#include <sys/ioctl.h>
#include <sys/stat.h>

#ifdef linux
#include <linux/fs.h>
#endif

/* UPDATE MODE functions */
static int _disk_raw_insert_before (PedDisk* disk, PedPartition* loc,
				    PedPartition* part);
static int _disk_raw_insert_after (PedDisk* disk, PedPartition* loc,
				   PedPartition* part);
static int _disk_raw_remove (PedDisk* disk, PedPartition* part);

static PedDiskType*	disk_types = NULL;

void
ped_register_disk_type (PedDiskType* type)
{
	PED_ASSERT (type != NULL, return);
	PED_ASSERT (type->ops != NULL, return);
	PED_ASSERT (type->name != NULL, return);
	
	((struct _PedDiskType*) type)->next = disk_types;
	disk_types = (struct _PedDiskType*) type;
}

void ped_unregister_disk_type (PedDiskType* type)
{
	PedDiskType*	walk;
	PedDiskType*	last = NULL;

	PED_ASSERT (type != NULL, return);

	for (walk = disk_types; walk != NULL; last = walk, walk = walk->next) {
		if (walk == type) break;
	}

	if (last)
		((struct _PedDiskType*) last)->next = type->next;
	else
		disk_types = type->next;
}

PedDiskType*
ped_disk_type_get_next (PedDiskType* type)
{
	if (type)
		return type->next;
	else
		return disk_types;
}

PedDiskType*
ped_disk_type_get (const char* name)
{
	PedDiskType*	walk = NULL;
	while (1) {
		walk = ped_disk_type_get_next (walk);
		if (!walk) break;
		if (strcasecmp (walk->name, name) == 0) break;
	}
	return walk;
}

PedDiskType*
ped_disk_probe (const PedDevice* dev)
{
	PedDiskType*	walk = NULL;

	PED_ASSERT (dev != NULL, return NULL);

	ped_exception_leave_all ();

	while (1) {
		walk = ped_disk_type_get_next (walk);
		if (!walk) break;
		if (walk->ops->probe (dev)) break;
	}

	if (ped_exception)
		ped_exception_catch ();
	ped_exception_fetch_all ();
	return walk;
}

PedDisk*
ped_disk_open (PedDevice* dev)
{
	PedDiskType*	type;

	PED_ASSERT (dev != NULL, return NULL);

	type = ped_disk_probe (dev);
	if (!type) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Unable to open %s - unrecognised disk label."),
			dev->path);
		return 0;
	}
	return type->ops->open (dev);
}

PedDisk*
ped_disk_create (PedDevice* dev, PedDiskType* type)
{
	PED_ASSERT (dev != NULL, return NULL);
	PED_ASSERT (type != NULL, return NULL);

	if (!type->ops->create) {
		ped_exception_throw (PED_EXCEPTION_NO_FEATURE,
			PED_EXCEPTION_CANCEL,
			_("Creating new %s disklabels is not implemented yet."),
			type->name);
		return 0;
	}
	return type->ops->create (dev);
}

PedDisk*
ped_disk_alloc (PedDevice* dev, PedDiskType* disk_type)
{
	PedDisk*	disk;

	disk = (PedDisk*) ped_malloc (sizeof (PedDisk));
	if (!disk)
		return NULL;

	disk->dev = dev;
	disk->type = disk_type;
	disk->part_list = NULL;
	disk->update_mode = 0;

	return disk;
}

int
ped_disk_close (PedDisk* disk)
{
	PED_ASSERT (disk != NULL, return 0);
	return disk->type->ops->close (disk);
}

int
ped_disk_read (PedDisk* disk)
{
	PED_ASSERT (disk != NULL, return 0);
	return disk->type->ops->read (disk);
}

int
ped_disk_write (PedDisk* disk)
{
	PED_ASSERT (disk != NULL, return 0);

	if (!disk->type->ops->write) {
		ped_exception_throw (
			PED_EXCEPTION_NO_FEATURE,
			PED_EXCEPTION_CANCEL,
			_("The code to write the partition table hasn't been "
			"written for %s yet"),
			disk->type->name);
		return 0;
	}
	if (!disk->type->ops->write (disk))
		return 0;

	return 1;
}

static int
ped_disk_align_partition (PedDisk* disk, PedPartition* part)
{
	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (part != NULL, return 0);

	return disk->type->ops->align_partition (disk, part);
}

static int
ped_disk_enumerate_partition (PedDisk* disk, PedPartition* part)
{
	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (part != NULL, return 0);

	return disk->type->ops->enumerate_partition (disk, part);
}

static char
ped_disk_get_extended_system (const PedDisk* disk)
{
	PED_ASSERT (disk != NULL, return 0);

	return disk->type->ops->get_extended_system ();
}

static int
ped_disk_last_partition_num (PedDisk* disk)
{
	PedPartition*	walk;
	int		highest = -1;

	PED_ASSERT (disk != NULL, return 0);

	for (walk = disk->part_list; walk;
	     walk = ped_disk_next_partition (disk, walk)) {
		if (walk->num > highest)
			highest = walk->num;
	}

	return highest;
}

/* gives all the (active) partitions a number.  It should preserve the numbers
 * and orders as much as possible.
 */
static int
ped_disk_enumerate_partitions (PedDisk* disk)
{
	PedPartition*	walk;
	int		i;
	int		end;

	PED_ASSERT (disk != NULL, return 0);

/* first "sort" already-numbered partitions.  (Eg: if a logical partition
 * is removed, then all logical partitions that were number higher MUST be
 * renumbered)
 */
	end = ped_disk_last_partition_num (disk);
	for (i=1; i<=end; i++) {
		walk = ped_disk_get_partition (disk, i);
		if (!walk)
			continue;
		ped_disk_enumerate_partition (disk, walk);
	}

/* now, number un-numbered partitions */
	for (walk = disk->part_list; walk;
	     walk = ped_disk_next_partition (disk, walk)) {
		if (ped_partition_is_active (walk)) {
			if (!ped_disk_enumerate_partition (disk, walk))
				return 0;
		}
	}

	return 1;
}

static int
_disk_remove_metadata (PedDisk* disk)
{
	PedPartition*	walk = NULL;
	PedPartition*	next;

	PED_ASSERT (disk != NULL, return 0);

	next = ped_disk_next_partition (disk, walk);

	while (next) {
		walk = next;
		while (1) {
			next = ped_disk_next_partition (disk, next);
			if (!next || next->type & PED_PARTITION_METADATA)
				break;
		}
		if (walk->type & PED_PARTITION_METADATA)
			ped_disk_delete_partition (disk, walk);
	}
	return 1;
}

static int
_disk_alloc_metadata (PedDisk* disk)
{
	PED_ASSERT (disk != NULL, return 0);

	if (!disk->update_mode)
		_disk_remove_metadata (disk);

	return disk->type->ops->alloc_metadata (disk);
}

static int
_disk_remove_freespace (PedDisk* disk)
{
	PedPartition*	walk;
	PedPartition*	next;

	walk = ped_disk_next_partition (disk, NULL);
	for (; walk; walk = next) {
		next = ped_disk_next_partition (disk, walk);

		if (walk->type & PED_PARTITION_FREESPACE)
			_disk_raw_remove (disk, walk);
	}

	return 1;
}

static int
_alloc_extended_freespace (PedDisk* disk)
{
	PedSector	last_end;
	PedPartition*	walk;
	PedPartition*	last;
	PedPartition*	free_space;
	PedPartition*	extended_part;

	extended_part = ped_disk_extended_partition (disk);
	if (!extended_part)
		return 1;

	last_end = extended_part->geom.start;
	last = NULL;
	
	for (walk = extended_part->part_list; walk; walk = walk->next) {
		if (walk->geom.start > last_end + 1) {
			free_space = ped_partition_new (
					disk,
					PED_PARTITION_FREESPACE
						| PED_PARTITION_LOGICAL,
					NULL,
					last_end + 1, walk->geom.start - 1);
			_disk_raw_insert_before (disk, walk, free_space);
		}

		last = walk;
		last_end = last->geom.end;
	}

	if (last_end < extended_part->geom.end) {
		free_space = ped_partition_new (
				disk,
				PED_PARTITION_FREESPACE | PED_PARTITION_LOGICAL,
				NULL,
				last_end + 1, extended_part->geom.end);

		if (last)
			return _disk_raw_insert_after (disk, last, free_space);
		else
			extended_part->part_list = free_space;
	}

	return 1;
}

static int
_disk_alloc_freespace (PedDisk* disk)
{
	PedSector	last_end;
	PedPartition*	walk;
	PedPartition*	last;
	PedPartition*	free_space;

	if (!_disk_remove_freespace (disk))
		return 0;
	if (!_alloc_extended_freespace (disk))
		return 0;

	last = NULL;
	last_end = 0;

	for (walk = disk->part_list; walk; walk = walk->next) {
		if (walk->geom.start > last_end + 1) {
			free_space = ped_partition_new (disk,
					PED_PARTITION_FREESPACE, NULL,
					last_end + 1, walk->geom.start - 1);
			_disk_raw_insert_before (disk, walk, free_space);
		}

		last = walk;
		last_end = last->geom.end;
	}

	if (last_end < disk->dev->length - 1) {
		free_space = ped_partition_new (disk,
					PED_PARTITION_FREESPACE, NULL,
					last_end + 1, disk->dev->length - 1);
		if (last)
			return _disk_raw_insert_after (disk, last, free_space);
		else
			disk->part_list = free_space;
	}

	return 1;
}

/* update mode: used when updating the internal representation of the partition
 * table.  In update mode, the metadata and freespace placeholder/virtual
 * partitions are removed, making it much easier for various manipulation
 * routines...
 */
static void
_disk_push_update_mode (PedDisk* disk)
{
	if (!disk->update_mode) {
		_disk_remove_freespace (disk);
		disk->update_mode++;
		_disk_remove_metadata (disk);
	} else {
		disk->update_mode++;
	}
}

static void
_disk_pop_update_mode (PedDisk* disk)
{
	if (disk->update_mode == 1) {
	/* re-allocate metadata BEFORE leaving update mode, to prevent infinite
	 * recursion (metadata allocation requires update mode)
	 */
		_disk_alloc_metadata (disk);
		disk->update_mode--;
		_disk_alloc_freespace (disk);
	} else {
		disk->update_mode--;
	}
}

PedPartition*
ped_partition_new (const PedDisk* disk, PedPartitionType type,
		   PedFileSystemType* fs_type, PedSector start, PedSector end)
{
	PedPartition*	part;

	PED_ASSERT (disk != NULL, return 0);

	if (end < start) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
		   _("Can't create a partition with the start after the end."));
		return NULL;
	}

	part = (PedPartition*) ped_malloc (sizeof (PedPartition));
	if (!part)
		return NULL;

	part->prev = NULL;
	part->next = NULL;

	part->geom.disk = (PedDisk*) disk;
	ped_geometry_set (&part->geom, start, end - start + 1);

	part->type = type;
	part->part_list = NULL;

	part->bootable = 0;
	part->hidden = 0;
	if (type & PED_PARTITION_EXTENDED) {
		part->system = ped_disk_get_extended_system (disk);
	} else if (fs_type) {
		part->system = fs_type->ops->get_system (NULL, part,
							 disk->type);
	} else {
		part->system = 0;
	}

	part->fs_type = NULL;
	return part;
}

PedPartition*
ped_partition_duplicate (const PedPartition* part)
{
	PedPartition*	result;

	PED_ASSERT (part != NULL, return 0);

	result = ped_partition_new (part->geom.disk, part->type, NULL,
				    part->geom.start, part->geom.end);
	if (!result)
		return NULL;
	result->system = part->system;
	return result;
}

void
ped_partition_destroy (PedPartition* part)
{
	PED_ASSERT (part != NULL, return);

	ped_free (part);
}

int
ped_partition_is_active (const PedPartition* part)
{
	PED_ASSERT (part != NULL, return 0);

	return !(part->type & PED_PARTITION_FREESPACE
		 || part->type & PED_PARTITION_METADATA);
}

int
ped_partition_get_hide_state (const PedPartition* part)
{
	PED_ASSERT (part != NULL, return 0);

	return part->hidden;
}

int
ped_partition_set_hide_state (PedPartition* part, int state)
{
	PedFileSystem*		fs;
	char			old_hide_state = part->hidden;
	char			new_system;

	PED_ASSERT (part != NULL, return 0);

	if (part->type & PED_PARTITION_EXTENDED)
		return 0;
	part->hidden = state;

	ped_exception_fetch_all();
	fs = ped_file_system_open (&part->geom);
	if (!fs) {
		ped_exception_catch();
		ped_exception_leave_all();

		if (part->fs_type) {
			/* hack! */
			new_system = part->fs_type->ops->get_system (NULL, part,
							part->geom.disk->type);
			if (new_system) {
				part->system = new_system;
				return 1;
			}
		}

		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
				_("No filesystem found on partition.  Can't "
				  "determine what partition system to use."));
		goto error;
	}
	ped_exception_leave_all();

	new_system = ped_file_system_get_system (fs, part,
						 part->geom.disk->type);
	if (!new_system)
		goto error_close_fs;
	part->system = new_system;

	ped_file_system_close (fs);
	return 1;

error_close_fs:
	ped_file_system_close (fs);
error:
	part->hidden = old_hide_state;
	return 0;
}

/* Sets the system (i.e. partition system type) on a partition.  It attempts
 * to open the file system, because this is the only way to absolutely ensure
 * the correct partition type is selected.  (Even if fs_type == fat_type,
 * it could be FAT16 or FAT32...)
 *	If it can't open the file system, it assumes there isn't a proper
 * one anyway, and guesses the best partition system.
 */
int
ped_partition_set_system (PedPartition* part, PedFileSystemType* fs_type)
{
	char			new_system;
	PedFileSystem*		fs;

	PED_ASSERT (part != NULL, return 0);
	if (!(part->type & PED_PARTITION_EXTENDED))
		PED_ASSERT (fs_type != NULL, return 0);

	if (part->type & PED_PARTITION_EXTENDED) {
		new_system = ped_disk_get_extended_system (part->geom.disk);
		if (!new_system)
			goto error;
		part->system = new_system;
		return 1;
	}

	ped_exception_fetch_all();
	fs = ped_file_system_open (&part->geom);
	if (fs && fs->type != fs_type) {
		ped_file_system_close (fs);
		fs = NULL;
	}
	if (!fs) {
		ped_exception_catch();
		ped_exception_leave_all();

		new_system = fs_type->ops->get_system (NULL, part,
							part->geom.disk->type);
		if (!new_system)
			goto error;
		return 1;
	}
	ped_exception_leave_all();

	new_system = ped_file_system_get_system (fs, part,
						 part->geom.disk->type);
	ped_file_system_close (fs);
	if (!new_system)
		goto error;
	part->system = new_system;
	return 1;

error:
	return 0;
}

#ifdef linux

/* FIXME: should this be part of the API?  Should it report a warning if the
 * stat fails? */
static int
_partition_is_root_device (const PedPartition* part)
{
	char		part_name [256];
	int		root_dev;
	struct stat	part_stat;
	FILE*		proc_real_root_dev;

	sprintf (part_name, "%s%d", part->geom.disk->dev->path, part->num);

	if (stat (part_name, &part_stat))
		goto error;

	proc_real_root_dev = fopen ("/proc/sys/kernel/real-root-dev", "r");
	if (!proc_real_root_dev)
		goto error;
	fscanf (proc_real_root_dev, "%d", &root_dev);
	fclose (proc_real_root_dev);

	return root_dev == part_stat.st_rdev;

error:
	return 0;
}

#endif

/* returns:
 * 	1  == busy
 * 	0  == not busy
 * 	-1 == if unknown
 *
 * NOTE: /proc/mounts doesn't list the real root device (in Linux), but
 * /dev/root instead.  Since the root device is always mounted (by
 * definition), this is checked separately in ped_partition_is_busy()
 */
static int
_check_mount_table_is_busy (const char* mtab_path, const char* part_name)
{
	char		line [256];
	FILE*		mtab;

	mtab = fopen (mtab_path, "r");
	if (!mtab)
		goto error;

	while (1) {
		if (!fgets (line, 256, mtab)) {
			if (feof (mtab)) {
				fclose (mtab);
				return 0;
			} else {
				goto error_close_mtab;
			}
		}
		if (strncmp (line, part_name, strlen (part_name)) == 0) {
			fclose (mtab);
			return 1;
		}
	}

error_close_mtab:
	fclose (mtab);
error:
	ped_exception_throw (
		PED_EXCEPTION_WARNING,
		PED_EXCEPTION_IGNORE,
		_("Error reading %s (%s) to determine if partition is "
		  "mounted."),
		mtab_path, strerror (errno));

	return -1;
}

int
ped_partition_is_busy (const PedPartition* part)
{
	char		part_name [256];
	int		is_root_dev;
	int		status;

	PED_ASSERT (part != NULL, return 0);

	if (part->num == -1)
		goto error;

	sprintf (part_name, "%s%d", part->geom.disk->dev->path, part->num);

#ifdef linux
	if (_partition_is_root_device (part))
		return 1;

	status = _check_mount_table_is_busy ("/proc/mounts", part_name);
	if (status == 1)
		return 1;

	if (status != -1) {
		status = _check_mount_table_is_busy ("/proc/swaps", part_name);
		if (status != -1)
			return status;
	}

#endif

	status = _check_mount_table_is_busy ("/etc/mtab", part_name);
	if (status != -1)
		return status;

error:
	return ped_exception_throw (
			PED_EXCEPTION_WARNING,
			PED_EXCEPTION_IGNORE_CANCEL,
			_("Unable to determine if partition is mounted."))
				!= PED_EXCEPTION_IGNORE;
}

int
ped_disk_is_busy (const PedDisk* disk)
{
	PedPartition*		walk;

	PED_ASSERT (disk != NULL, return 0);

	for (walk = disk->part_list; walk; walk = walk->next) {
		if (walk->num == -1)
			continue;
		if (ped_partition_is_busy (walk))
			return 1;
	}
	return 0;
}

PedPartition*
ped_disk_extended_partition (const PedDisk* disk)
{
	PedPartition*		walk;

	PED_ASSERT (disk != NULL, return 0);

	for (walk = disk->part_list; walk; walk = walk->next) {
		if (walk->type == PED_PARTITION_EXTENDED)
			break;
	}
	return walk;
}

PedPartition*
ped_disk_get_boot_partition (const PedDisk* disk)
{
	PedPartition*		walk;

	PED_ASSERT (disk != NULL, return 0);

	for (walk = disk->part_list; walk; walk = walk->next) {
		if (walk->bootable)
			break;
	}
	return walk;
}

int
ped_disk_set_boot_partition (PedDisk* disk, PedPartition* part)
{
	PedPartition*		old_boot_part;

	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (part != NULL, return 0);

	old_boot_part = ped_disk_get_boot_partition (disk);
	if (old_boot_part)
		old_boot_part->bootable = 0;
	part->bootable = 1;
	return 1;
}

/* returns the next partition.  If the current partition is an extended
 * partition, returns the first logical partition.
 */
PedPartition*
ped_disk_next_partition (const PedDisk* disk, const PedPartition* part)
{
	PED_ASSERT (disk != NULL, return 0);

	if (!part)
		return disk->part_list;
	if (part->type == PED_PARTITION_EXTENDED)
		return part->part_list;
	if (part->next)
		return part->next;
	if (part->type & PED_PARTITION_LOGICAL)
		return ped_disk_extended_partition (disk)->next;
	return NULL;
}

static int
_disk_check_sanity (PedDisk* disk)
{
	PedPartition*	walk;

	PED_ASSERT (disk != NULL, return 0);

	for (walk = disk->part_list; walk; walk = walk->next) {
		PED_ASSERT (!(walk->type & PED_PARTITION_LOGICAL), return 0);
		if (walk->prev)
			PED_ASSERT (walk->prev->next == walk, return 0);
	}

	if (!ped_disk_extended_partition (disk))
		return 1;

	for (walk = ped_disk_extended_partition (disk)->part_list; walk;
	     walk = walk->next) {
		PED_ASSERT (walk->type & PED_PARTITION_LOGICAL, return 0);
		if (walk->prev)
			PED_ASSERT (walk->prev->next == walk, return 0);
	}
	return 1;
}

PedPartition*
ped_disk_get_partition (const PedDisk* disk, int num)
{
	PedPartition*	walk;

	PED_ASSERT (disk != NULL, return 0);

	for (walk = disk->part_list; walk;
	     walk = ped_disk_next_partition (disk, walk)) {
		if (walk->num == num && !(walk->type & PED_PARTITION_FREESPACE))
			return walk;
	}

	return NULL;
}

/* Returns the partition that contains sect.  If sect lies within a logical
 * partition, then the logical partition is returned (not the extended
 * partition)
 */
PedPartition*
ped_disk_get_partition_by_sector (const PedDisk* disk, PedSector sect)
{
	PedPartition*	walk;

	PED_ASSERT (disk != NULL, return 0);

	for (walk = disk->part_list; walk;
	     walk = ped_disk_next_partition (disk, walk)) {
		if (sect < walk->geom.start || sect > walk->geom.end)
			continue;
		if (walk->type != PED_PARTITION_EXTENDED)
			return walk;
	}

	/* should never get here, unless sect is outside of disk's useable
	 * part, or we're in "update mode", and the free space place-holders
	 * have been removed with _disk_remove_freespace()
	 */
	return NULL;
}

/* I'm beginning to agree with Sedgewick :-/
 * UPDATE MODE ONLY
 */
static int
_disk_raw_insert_before (PedDisk* disk, PedPartition* loc, PedPartition* part)
{
	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (loc != NULL, return 0);
	PED_ASSERT (part != NULL, return 0);

	part->prev = loc->prev;
	part->next = loc;
	if (part->prev) {
		part->prev->next = part;
	} else {
		if (loc->type & PED_PARTITION_LOGICAL)
			ped_disk_extended_partition (disk)->part_list = part;
		else
			disk->part_list = part;
	}
	loc->prev = part;

	return 1;
}

/* inserts a partition *after* loc.
 * UPDATE MODE ONLY
 */
static int
_disk_raw_insert_after (PedDisk* disk, PedPartition* loc, PedPartition* part)
{
	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (loc != NULL, return 0);
	PED_ASSERT (part != NULL, return 0);

	part->prev = loc;
	part->next = loc->next;
	if (loc->next)
		loc->next->prev = part;
	loc->next = part;

	return 1;
}

/* UPDATE MODE ONLY
 */
static int
_disk_raw_remove (PedDisk* disk, PedPartition* part)
{
	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (part != NULL, return 0);

	if (part->prev) {
		part->prev->next = part->next;
		if (part->next)
			part->next->prev = part->prev;
	} else {
		if (part->type & PED_PARTITION_LOGICAL) {
			ped_disk_extended_partition (disk)->part_list
				= part->next;
		} else {
			disk->part_list = part->next;
		}
		if (part->next)
			part->next->prev = NULL;
	}

	return 1;
}

/* returns 0 if the partition, "part" overlaps with any partitions on the
 * "disk".  The geometry of "part" is taken to be "geom", NOT "part->geom"
 * (the idea here is to check if "geom" is valid, before changing "part").
 * 
 * This is useful for seeing if a resized partitions new geometry is going to
 * fit, without the existing geomtry getting in the way.
 *
 * Note: overlap with an extended partition is also allowed, provided that
 * "geom" lies completely inside the extended partition.
 */
int
ped_disk_check_overlap_new_geom (PedDisk* disk, PedPartition* part,
				 PedGeometry* geom)
{
	PedPartition*	walk;

	for (walk = ped_disk_next_partition (disk, NULL); walk;
	     walk = ped_disk_next_partition (disk, walk)) {
		if (walk->type & PED_PARTITION_FREESPACE)
			continue;
		if (walk == part)
			continue;
		if (part->type & PED_PARTITION_EXTENDED
		    && walk->type & PED_PARTITION_LOGICAL)
			continue;

		if (ped_geometry_test_overlap (&walk->geom, geom)) {
			if (walk->type & PED_PARTITION_EXTENDED
			    && part->type & PED_PARTITION_LOGICAL
			    && ped_geometry_test_inside (&walk->geom, geom))
				continue;
			return 0;
		}
	}

	return 1;
}

int
ped_disk_check_overlap (PedDisk* disk, PedPartition* part)
{
	return ped_disk_check_overlap_new_geom (disk, part, &part->geom);
}

static int
_check_new_partition_basic_sanity (PedDisk* disk, PedPartition* part)
{
	PedPartition*	ext_part = ped_disk_extended_partition (disk);

	if (part->geom.start < 0 || part->geom.end >= disk->dev->length) {
		ped_exception_throw (
			PED_EXCEPTION_ERROR,
			PED_EXCEPTION_CANCEL,
			_("Can't create a partition outside of the device."),
			disk->dev->path);
		return 0;
	}

	if ((part->type & PED_PARTITION_LOGICAL) && !ext_part) {
		ped_exception_throw (
			PED_EXCEPTION_ERROR,
			PED_EXCEPTION_CANCEL,
			_("Can't add a logical partition to %s, because "
			"there is no extended partition."),
			disk->dev->path);
		return 0;
	}

	return 1;
}

static int
_check_new_partition (PedDisk* disk, PedPartition* part)
{
	PedPartition*	ext_part = ped_disk_extended_partition (disk);

	if (part->type == PED_PARTITION_EXTENDED && ext_part) {
		ped_exception_throw (
			PED_EXCEPTION_ERROR,
			PED_EXCEPTION_CANCEL,
			_("Can't have more than one extended partition on %s"),
			disk->dev->path);
		return 0;
	}

	if (part->type & PED_PARTITION_LOGICAL
	    && !ped_geometry_test_inside (&ext_part->geom, &part->geom)) {
		ped_exception_throw (
			PED_EXCEPTION_ERROR,
			PED_EXCEPTION_CANCEL,
			_("Can't add a logical partition outside of the "
			  "extended partition on %s."),
			disk->dev->path);
		return 0;
	}

	if (!ped_disk_check_overlap (disk, part)) {
		ped_exception_throw (
			PED_EXCEPTION_ERROR,
			PED_EXCEPTION_CANCEL,
			_("The new partition overlaps with another "
			  "partition."));
		return 0;
	}

	if (part->type == PED_PARTITION_PRIMARY
	    && ext_part
	    && ped_geometry_test_inside (&ext_part->geom, &part->geom)) {
		ped_exception_throw (PED_EXCEPTION_ERROR, PED_EXCEPTION_CANCEL,
			_("Can't add a primary partition inside an extended "
			 "partition."));
		return 0;
	}

	return 1;
}

static int
_add_partition (PedDisk* disk, PedPartition* part)
{
	PedPartition*	walk;
	PedPartition*	last;
	PedPartition*	ext_part;

	ext_part = ped_disk_extended_partition (disk);

	last = NULL;
	walk = (part->type & PED_PARTITION_LOGICAL) ?
			ext_part->part_list : disk->part_list;

	for (; walk; last = walk, walk = walk->next) {
		if (walk->geom.start > part->geom.end)
			break;
	}

	if (walk) {
		return _disk_raw_insert_before (disk, walk, part);
	} else {
		if (last) {
			return _disk_raw_insert_after (disk, last, part);
		} else {
			if (part->type & PED_PARTITION_LOGICAL)
				ext_part->part_list = part;
			else
				disk->part_list = part;
		} 
	}

	return 1;
}

int
ped_disk_add_partition (PedDisk* disk, PedPartition* part)
{
	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (part != NULL, return 0);

#ifdef VERBOSE
	printf ("ped_disk_add_partition (dev=\"%s\", start=%d, end=%d,"
		" type=%x)\n",
		disk->dev->path, (int) part->geom.start, (int) part->geom.end,
		(int) part->system);
#endif

	if (!_check_new_partition_basic_sanity (disk, part))
		return 0;

	_disk_push_update_mode (disk);

	if (ped_partition_is_active (part)) {
		if (!ped_disk_enumerate_partition (disk, part))
			goto error;
		if (!ped_disk_align_partition (disk, part))
			goto error;
	}
	if (!_check_new_partition (disk, part))
		goto error;
	if (!_add_partition (disk, part))
		goto error;

	_disk_pop_update_mode (disk);
	if (!_disk_check_sanity (disk))
		return 0;
	return 1;

error:
	_disk_pop_update_mode (disk);
	return 0;
}

static int
ped_disk_delete_all_logical (PedDisk* disk);

int
ped_disk_delete_partition (PedDisk* disk, PedPartition* part)
{
	int	is_metadata_part;

	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (part != NULL, return 0);

	is_metadata_part = part->type & PED_PARTITION_METADATA;

	_disk_push_update_mode (disk);

	if (part->type & PED_PARTITION_EXTENDED)
		ped_disk_delete_all_logical (disk);
	_disk_raw_remove (disk, part);
	ped_partition_destroy (part);

	_disk_pop_update_mode (disk);

	ped_disk_enumerate_partitions (disk);

	return 1;
}

static int
ped_disk_delete_all_logical (PedDisk* disk)
{
	PedPartition*		walk;
	PedPartition*		next;
	PedPartition*		ext_part;

	PED_ASSERT (disk != NULL, return 0);
	ext_part = ped_disk_extended_partition (disk);
	PED_ASSERT (ext_part != NULL, return 0);

	for (walk = ext_part->part_list; walk; walk = next) {
		next = walk->next;

		if (!(walk->type & PED_PARTITION_FREESPACE)) {
			if (!ped_disk_delete_partition (disk, walk))
				return 0;
		}
	}
	return 1;
}

int
ped_disk_delete_all (PedDisk* disk)
{
	PedPartition*		walk;
	PedPartition*		next;

	PED_ASSERT (disk != NULL, return 0);

	_disk_push_update_mode (disk);

	for (walk = disk->part_list; walk; walk = next) {
		next = walk->next;

		if (!ped_disk_delete_partition (disk, walk))
			return 0;
	}

	_disk_pop_update_mode (disk);

	return 1;
}

static PedGeometry*
_calc_aligned_geom (PedDisk* disk, PedPartition* part,
		    PedSector start, PedSector end)
{
	PedGeometry	old_geom;
	PedGeometry*	aligned_geom;

	old_geom = part->geom;
	ped_geometry_set (&part->geom, start, end - start + 1);

	if (ped_partition_is_active (part)) {
		if (!ped_disk_align_partition (disk, part))
			goto error;
	}

	aligned_geom = ped_geometry_duplicate (&part->geom);
	part->geom = old_geom;
	return aligned_geom;

error:
	part->geom = old_geom;
	return NULL;
}

static int
_check_new_extended_geometry (PedDisk* disk, PedGeometry* new_geom)
{
	PedPartition*		walk;
	PedPartition*		ext_part;

	PED_ASSERT (disk != NULL, return 0);
	ext_part = ped_disk_extended_partition (disk);
	PED_ASSERT (ext_part != NULL, return 0);

	for (walk = ext_part->part_list; walk; walk = walk->next) {
		if (!ped_geometry_test_inside (new_geom, &walk->geom)) {
			ped_exception_throw (
				PED_EXCEPTION_ERROR,
				PED_EXCEPTION_CANCEL,
				_("Can't resize an extended partition so as to "
				  "exclude a logical partition."));
			return 0;
		}
	}
	return 1;
}

/* FIXME: do metadata collion handling properly! (we're depending on being
 * in update mode, at the moment, which is yucky)
 */
static int
_check_changed_geometry (PedDisk* disk, PedPartition* part,
			 PedGeometry* new_geom)
{
	PedPartition*	ext_part;

	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (part != NULL, return 0);
	PED_ASSERT (new_geom != NULL, return 0);

	if (new_geom->start > new_geom->end) {
		ped_exception_throw (
			PED_EXCEPTION_ERROR,
			PED_EXCEPTION_CANCEL,
			_("Can't move the end of the partition behind the "
			  "start."));
		goto error;
	}

	if (part->type & PED_PARTITION_LOGICAL) {
		ext_part = ped_disk_extended_partition (disk);
		if (!ped_geometry_test_inside (&ext_part->geom, new_geom)) {
			ped_exception_throw (
				PED_EXCEPTION_ERROR,
				PED_EXCEPTION_CANCEL,
				_("Can't grow a logical partition outside "
				  "the extended partition."));
			return 0;
		}
	}

	if (part->type & PED_PARTITION_EXTENDED) {
		if (!_check_new_extended_geometry (disk, new_geom))
			goto error;
	}

	if (!ped_disk_check_overlap_new_geom (disk, part, new_geom)) {
		ped_exception_throw (
			PED_EXCEPTION_ERROR,
			PED_EXCEPTION_CANCEL,
			_("Can't grow partition onto used space, or outside "
			  "the disk."));
		goto error;
	}

	return 1;

error:
	return 0;
}

int
ped_disk_set_partition_geom (PedDisk* disk, PedPartition* part,
			     PedSector start, PedSector end)
{
	PedPartition*		free_space;
	PedPartition*		ext_free_space;
	PedGeometry*		new_geom;

	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (part != NULL, return 0);

	new_geom = _calc_aligned_geom (disk, part, start, end);
	if (!new_geom)
		goto error;

	_disk_push_update_mode (disk);

	if (!_check_changed_geometry (disk, part, new_geom))
		goto error;

	part->geom = *new_geom;
	ped_geometry_destroy (new_geom);

	_disk_pop_update_mode (disk);

	return 1;

error:
	_disk_pop_update_mode (disk);
	return 0;
}

int
ped_disk_maximize_partition (PedDisk* disk, PedPartition* part)
{
	PedGeometry	old_geom;

	PED_ASSERT (disk != NULL, return 0);
	PED_ASSERT (part != NULL, return 0);

	old_geom = part->geom;

	if (part->prev && part->prev->type & PED_PARTITION_FREESPACE) {
		if (!ped_disk_set_partition_geom (
				disk, part, part->prev->geom.start,
				part->geom.end))
			goto error;
	}
	if (part->next && part->next->type & PED_PARTITION_FREESPACE) {
		if (!ped_disk_set_partition_geom (
				disk, part, part->geom.start,
				part->next->geom.end))
			goto error;
	}
	return 1;

error:
	ped_disk_set_partition_geom (disk, part, old_geom.start, old_geom.end);
	return 0;
}

/* Reduces the size of the extended partition to wrap the logical partitions.
 * If there are no logical partitions, it removes the extended partition.
 */
int
ped_disk_minimize_extended_partition (PedDisk* disk)
{
	PedPartition*		first_logical;
	PedPartition*		last_logical;
	PedPartition*		walk;
	PedPartition*		ext_part;

	PED_ASSERT (disk != NULL, return 0);

	ext_part = ped_disk_extended_partition (disk);
	if (!ext_part)
		return 1;

	first_logical = ext_part->part_list;
	if (!first_logical)
		return ped_disk_delete_partition (disk, ext_part);

	for (walk = first_logical; walk->next; walk = walk->next);
	last_logical = walk;

	return ped_disk_set_partition_geom (disk, ext_part,
					    first_logical->geom.start,
					    last_logical->geom.end);
}

char*
ped_partition_type_get_name (PedPartitionType type)
{
	if (type & PED_PARTITION_METADATA)
		return _("metadata");
	else if (type & PED_PARTITION_FREESPACE)
		return _("free");
	else if (type & PED_PARTITION_EXTENDED)
		return _("extended");
	else if (type & PED_PARTITION_LOGICAL)
		return _("logical");
	else
		return _("primary");
}

void
ped_partition_print (PedPartition* part)
{
	PED_ASSERT (part != NULL, return);

	printf ("  %-10s %02d  (%d->%d)\n",
		ped_partition_type_get_name (part->type),
		part->num,
		(int) part->geom.start, (int) part->geom.end);
}

void
ped_disk_print (PedDisk* disk)
{
	PedPartition*	part;

	PED_ASSERT (disk != NULL, return);

	for (part = disk->part_list; part;
	     part = ped_disk_next_partition (disk, part))
		ped_partition_print (part);
}

