/*
 *  desktop -- The 3dfx Desktop Demo 
 *  COPYRIGHT 3DFX INTERACTIVE, INC. 1999
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <stdlib.h>

#include "window_state.h"
#include "window.h"
#include "spider.h"
#include "tux.h"
#include "bird.h"

/* This function compares two window states.  It is passed to qsort */
static int
window_state_compare (const void *a, const void *b)
{
  Window_State *ws1 = *(Window_State**)a;
  Window_State *ws2 = *(Window_State**)b;

  if (ws1->window_id < ws2->window_id)
    return (-1);
  else if (ws1->window_id > ws2->window_id)
    return (1);

  return (0);
}

/* Add a new window to the list */
void
window_state_insert (Window_State **list, int *n, Window_State *new_ws)
{
  /* I assume that there are WINDOW_MAX elements allocated to the list */
  if (*n < WINDOW_MAX - 1)
  {
    /* There is free space */
    list[*n] = new_ws;

    /* Now resort the list by the window id */
    qsort (list, *n+1, sizeof(Window_State*), window_state_compare);

    (*n)++;
  }
}

Window_State *
window_state_remove_index (Window_State **list, int *n, int index)
{
  Window_State *ws = NULL;
  int my_n = *n;
  int i;

  if (index >= 0)
  {
    ws = list[index];
    /* Now slide all of the data back */
    //memcpy (&list[index], &list[index+1], 
    //	    (my_n - index - 1) * sizeof (Window_State));
    for (i = index + 1; i < my_n; i++)
    {
      list[i-1] = list[i];
    }
    my_n--;
  }
  *n = my_n;
  return (ws);
}

Window_State *
window_state_remove_key (Window_State **list, int *n, Window key)
{
  int index;

  /* Find where key is */
  index = window_state_search (list, n, key);
  
  return (window_state_remove_index (list, n, index));
}

int
window_state_search (Window_State **list, int *n, Window key)
{
  int hi, low, mid;

  if (*n > 0)
  {
    /* binary search for the window_id */
    hi = *n-1;
    low = 0;
    
    while (hi - low > 1)
    {
      mid = (hi + low) / 2;
      
      if (list[mid]->window_id == key)
      {
	return (mid);
      }
      else if (key > list[mid]->window_id)
      {
	low = mid;
      }
      else 
      {
	hi = mid;
      }
    }

    /* We haven't found it, check the endpoints */
    if (list[hi]->window_id == key)
      return (hi);
    if (list[low]->window_id == key)
      return (low);
  }
  /* Still nothing return -1 */
  return (-1);
}

static void
window_unproject (Demo_State *state, Window_State *ws)
{
  float ul[4] = {0, 0, 0, 0}, lr[4] = {0, 0, 0, 0};

  /* extract a vector in correct format from the GrVertex structure */
  ul[0] =  vtx[ws->index][0].x / vtx[ws->index][0].oow;
  ul[1] =  vtx[ws->index][0].y / vtx[ws->index][0].oow;
  /* I keep the positive z value */
  ul[2] = vtx[ws->index][0].z / vtx[ws->index][0].oow;
  ul[3] = 1.f / vtx[ws->index][0].oow;;

  /* Unproject the vertex */
  MatMultVec4x4_4 (ws->world_upper_left, state->unproject, ul);

  /* Now do the lower right vertex the same way */
  lr[0] =  vtx[ws->index][2].x / vtx[ws->index][2].oow;
  lr[1] =  vtx[ws->index][2].y / vtx[ws->index][2].oow;
  lr[2] =  vtx[ws->index][2].z / vtx[ws->index][2].oow;
  lr[3] =  1.f / vtx[ws->index][2].oow;

  MatMultVec4x4_4 (ws->world_lower_right, state->unproject, lr);

  //#define TRACE_VERTS
#ifdef TRACE_VERTS
  {
    float temp[4];
    MatMultVec4x4_4 (temp, state->project, ws->world_upper_left);
    temp[0] /= temp[3];
    temp[1] /= temp[3];
    temp[2] /= temp[3];
    
    printf ("ul: (%f %f %f %f)-->(%f %f %f %f)-->(%f %f %f %f)-->"
	    "(%f %f %f %f)\n", 
	    vtx[ws->index][0].x,
	    vtx[ws->index][0].y,
	    vtx[ws->index][0].z,
	    1.f / vtx[ws->index][0].oow,
	    ul[0], ul[1], ul[2], ul[3],
	    ws->world_upper_left[0], 
	    ws->world_upper_left[1],
	    ws->world_upper_left[2],
	    ws->world_upper_left[3],
	    temp[0], temp[1], temp[2], temp[3]
	    );
    printf ("lr: (%f %f %f %f)-->(%f %f %f %f)-->(%f %f %f %f)\n", 
	    vtx[ws->index][2].x,
	    vtx[ws->index][2].y,
	    vtx[ws->index][2].z,
	    1.f / vtx[ws->index][2].oow,
	    lr[0], lr[1], lr[2], lr[3],
	    ws->world_lower_right[0], 
	    ws->world_lower_right[1],
	    ws->world_lower_right[2],
	    ws->world_lower_right[3]);
  }
#endif
}


void
window_state_update (Demo_State *state, Window_State **list, int *n)
{
  int i;
  int ws_index;
  Window_State *new_ws;
  //int init = 0;

  /* Find all new windows */
  for (i = 0; i < win_count; i++)
  {
    /* Find this window in the state list */
    ws_index = window_state_search (list, n, windows[i]);    

    /* If there isn't one then build a new state */
    if (ws_index >= 0)
    {
      /* I've looked at this window so mark it as used */
      list[ws_index]->used = 1;    
      list[ws_index]->init = 0;
    }
    else
    {
      /* Create a new state */
      new_ws = (Window_State*) malloc (sizeof (Window_State));
      new_ws->window_id = windows[i];

      /* Add a random color */
      new_ws->color1 = (FxU32)rand () | 0xFF000000;
      new_ws->color2 = (FxU32)rand () | 0xFF000000;

      /* FIXME: add other relevant state data including animation stuff */
      
      /* Now insert the new state */
      window_state_insert (list, n, new_ws);
      ws_index = window_state_search (list, n, windows[i]);
      if (ws_index < 0)
	printf ("wtf n = %d\n\n", *n);

      /* Add attraction and repulsion to the window */
      list[ws_index]->attraction[0] = 0.7;
      list[ws_index]->attraction[1] = 0.7;
      list[ws_index]->attraction[2] = 0.7;
      list[ws_index]->attraction[3] = 0.7;
      list[ws_index]->repulsion = -1.8;

      /* flag this window as new */
      new_ws->init = 1;
      
      /* I've looked at this window so mark it as used */
      new_ws->used = 1;    
    }
  }

  /* Now remove all of the unused windows since they must have closed */
  i = 0;
  while (i < *n)
  {
    Window_State *temp;

    if (list[i]->used == 0)
    {
      /* Check to see if this is tux's window */
      if (state->tux_window == list[i])
      {
	state->tux_window = NULL;
      }
      /* Check if this was the bird's window */
      if (state->bird_window == list[i])
      {
	state->bird_window = NULL;
      }
      /* FIXME: free this window state */
      temp = window_state_remove_index (list, n, i);
      /* Clear temp's data out to help trace bugs */
      temp->index = -1;
      temp->decorated = 0;
      temp->world_upper_left[0] = 0;
      temp->world_upper_left[1] = 0;

      /* FIXME: free temp */
      free (temp);
    }
    else
    {
      /* Clear the used flag for the next frame */
      list[i]->used = 0;
      i++;
    }
  }

  /* Make sure that the list is still sorted */
  /* Now resort the list by the window id */
  qsort (list, *n, sizeof(Window_State*), window_state_compare);

  /* Now I am left with a sorted list of visible windows.  I need
   * to update them.  */
  for (i = 0; i < win_count; i++)
  //for (ws_index = 0; ws_index < *n; ws_index++)
  {
    /* Find this window in the state list */
    ws_index = window_state_search (list, n, windows[i]);        

    if (ws_index < 0)
      printf ("Could not find window %d\n", i);

    /* Check if the window is big enough to be decorated */
    if (vtx[i][1].x - vtx[i][0].x >= MIN_WIN_WIDTH &&
	vtx[i][2].y - vtx[i][0].y >= MIN_WIN_HEIGHT)
    {
      list[ws_index]->decorated = 1;
    }
    else
    {
      list[ws_index]->decorated = 0;
    }

    /* Update the index for this window */
    list[ws_index]->index = i;

    /* Unproject the window so that I have world space coordinates */
    window_unproject(state, list[ws_index]);

    /* Store the index of the state in the list of 
     * states that is parallel to the list of windows
     */
    ordered_states[i] = ws_index;

    /* Mark this window as used */
    //list[ws_index]->used = 1;

    /* Initialize the spider */
    if (list[ws_index]->init && list[ws_index]->decorated)
    {
      put_spider_in_window (state, list[ws_index]);

      /* Setup the spider attractor model */
      list[ws_index]->spider_center_attraction = 0.5;

    }    

    /* Update the spider */
    if (list[ws_index]->decorated)
    {
      /* Update the spider on the window */
      update_spider (state, list[ws_index]);
    }

    /* Update tux */
    if (list[ws_index]->decorated)
    {
      /* If tux is not currently assigned to a window then just
       * put him on this one.
       */
      if (state->tux_window == NULL)
      {
	state->tux_window = list[ws_index];

	/* Put tux on his window */
	put_tux_on_window_corner (state);
      }

      /* Same deal with the bird */
      if (state->bird_window == NULL)
      {
	state->bird_window = list[ws_index];
	put_bird_on_window (state);
      }
    }
  }
}
