/*
 * IRC - Internet Relay Chat, ircd/channel.c
 * Copyright (C) 1996 Carlo Wood (I wish this was C++ - this sucks :/)
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include "sys.h"

#include "h.h"
#include "s_serv.h"
#include "struct.h"
#include "common.h"
#include "numnicks.h"
#include "ircd.h"
#include "parse.h"
#include "s_misc.h"
#include "match.h"
#include "s_bsd.h"
#ifdef DEBUGMODE
#include "s_serv.h"
#endif

RCSTAG_CC("$Id: numnicks.c,v 1.31 1999/02/18 12:06:53 ircu2_run Exp $");

/*
 * Numeric nicks are new as of version ircu2.10.00beta1.
 *
 * The idea is as follows:
 * In most messages (for protocol 10+) the original nick will be
 * replaced by a 3 character string: YXX
 * Where 'Y' represents the server, and 'XX' the nick on that server.
 *
 * 'YXX' should not interfer with the input parser, and therefore is
 * not allowed to contain spaces or a ':'.
 * Also, 'Y' can't start with a '+' because of m_server().
 *
 * We keep the characters printable for debugging reasons too.
 *
 * The 'XX' value can be larger then the maximum number of clients
 * per server, we use a mask (aServer::nn_mask) to get the real
 * client numeric. The overhead is used to have some redundancy so
 * just-disconnected-client aren't confused with just-connected ones.
 */

/* The internal counter for the 'XX' of local clients */
static u_char last_x1, last_x2;

/* *INDENT-OFF* */

/*
 * convert2y[] converts a numeric to the corresponding character.
 * The following characters are currently known to be forbidden:
 *
 * '\0' : Because we use '\0' as end of line.
 *
 * ' '	: Because parse_*() uses this as parameter seperator.
 * ':'	: Because parse_server() uses this to detect if a prefix is a
 *	  numeric or a name.
 * '+'	: Because m_nick() uses this to determine if parv[6] is a
 *	  umode or not.
 * '&', '#', '+', '$', '@' and '%' :
 *	  Because m_message() matches these characters to detect special cases.
 */
char convert2y[NUMNICKBASE] = {
  'A','B','C','D','E','F','G','H','I','J','K','L','M',
  'N','O','P','Q','R','S','T','U','V','W','X','Y','Z',
  'a','b','c','d','e','f','g','h','i','j','k','l','m',
  'n','o','p','q','r','s','t','u','v','w','x','y','z',
  '0','1','2','3','4','5','6','7','8','9',
  '[',']'
};

u_char convert2n[NUMNICKMAXCHAR + 1] = {
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0, 0,
  0, 0, 0, 0, 0, 0, 0, 0,52,53,54,55,56,57,58,59,60,61, 0, 0,
  0, 0, 0, 0, 0, 0, 1, 2, 3, 4, 5, 6, 7, 8, 9,10,11,12,13,14,
 15,16,17,18,19,20,21,22,23,24,25,62, 0,63, 0, 0, 0,26,27,28,
 29,30,31,32,33,34,35,36,37,38,39,40,41,42,43,44,45,46,47,48,
 49,50,51
};

/* *INDENT-ON* */

aClient *server_list[NUMNICKBASE];

void SetServerYXX(aClient *cptr, aClient *server, char *yxx)
{
#ifndef NO_PROTOCOL9
  /* Use cptr, because we do protocol 9 -> 10 translation for numeric nicks ! */
  if (Protocol(cptr) > 9)
  {
#endif
    server->yxx[0] = yxx[0];
    server->yxx[1] = yxx[1];
    server->yxx[2] = yxx[2];
    server->serv->nn_mask = (convert2n[(unsigned char)yxx[1]] << NUMNICKLOG) +
	convert2n[(unsigned char)yxx[2]];
#ifndef NO_PROTOCOL9
  }
  else
  {
    static const char *server_name_table[50] = {
/*  1 */ "Amsterdam.NL.EU.undernet.org",
/*  2 */ "Baltimore.MD.US.Undernet.Org",
/*  3 */ "Caen.Fr.Eu.UnderNet.org",
/*  4 */ "Dallas-Test.TX.US.Undernet.org",
/*  5 */ "Chicago-R.IL.US.Undernet.org",
/*  6 */ "Chicago.IL.US.Undernet.org",
/*  7 */ "Dallas.Tx.US.Undernet.org",
/*  8 */ "Des-Moines.IA.US.Undernet.Org",
/*  9 */ "Diemen.NL.EU.undernet.org",
/* 10 */ "Dallas-R.TX.US.Undernet.org",
/* 11 */ "London.UK.eu.Undernet.org",
/* 12 */ "Lulea-R.se.eu.undernet.org",
/* 13 */ "Antwerpen.BE.Eu.Undernet.org",
/* 14 */ "Norman-r.ok.us.undernet.org",
/* 15 */ "Oslo-R.NO.EU.Undernet.org",
/* 16 */ "Regensburg.DE.EU.undernet.org",
/* 17 */ "RockHill.SC.US.Undernet.Org",
/* 18 */ "SaltLake.UT.US.UnderNET.Org",
/* 19 */ "SanDiego.CA.US.Undernet.org",
/* 20 */ "Almere.NL.Eu.Undernet.org",
/* 21 */ "StLouis.MO.US.UnderNet.org",
/* 22 */ "Uworld.undernet.org",
/* 23 */ "Uworld2.undernet.org",
/* 24 */ "Oslo1.NO.Eu.Undernet.org",
/* 25 */ "Vancouver.BC.CA.Undernet.Org",
/* 26 */ "ann-arbor.mi.us.undernet.org",
/* 27 */ "auckland.nz.undernet.org",
/* 28 */ "aunet.",
/* 29 */ "austin.tx.us.undernet.org",
/* 30 */ "channels.undernet.org",
/* 31 */ "channels2.undernet.org",
/* 32 */ "gateway1.undernet.org",
/* 33 */ "los-angeles.ca.us.undernet.org",
/* 34 */ "lowell.ma.us.undernet.org",
/* 35 */ "montreal.qu.ca.undernet.org",
/* 36 */ "okc.ok.us.undernet.org",
/* 37 */ "phoenix.az.us.undernet.org",
/* 38 */ "Goettingen.DE.Eu.Undernet.org",
/* 39 */ "springfield.mo.us.undernet.org",
/* 40 */ "toronto.on.ca.undernet.org",
/* 41 */ "washington-1.dc.us.undernet.org",
/* 42 */ "washington-2.dc.us.undernet.org",
/* 43 */ "newbrunswick.nj.us.undernet.org",
/* 44 */ "lasvegas.nv.us.undernet.org",
/* 45 */ "washington-r.dc.us.undernet.org",
/* 46 */ "washington.dc.us.undernet.org",
/* 47 */ "protocol.undernet.org",
/* 48 */ "StLouis-R.MO.US.Undernet.Org",
/* 49 */ "Charlotte-R.NC.US.Undernet.Org",
/* 50 */ "Los-Angeles-R.CA.US.Undernet.Org"
    };

    int i;
    for (i = 1; i <= 50; ++i)
      if (!strCasediff(server_name_table[i - 1], server->name))
	break;
    if (i == 51)
      i = NUMNICKBASE - 1;
    if (server_list[i])
    {
      i = NUMNICKBASE;
      while (server_list[--i]);
    }
    *server->yxx = convert2y[i];
    server->serv->nn_mask = MAX_MAXCLIENTS - 1;	/* Uses more memory then
						   necessary :( */
    server->yxx[1] = convert2y[(MAX_MAXCLIENTS - 1) >> NUMNICKLOG];
    server->yxx[2] = convert2y[(MAX_MAXCLIENTS - 1) & NUMNICKMASK];
  }
#endif
  server_list[convert2n[(unsigned char)*server->yxx]] = server;
  /* Note, exit_one_client uses the fact that `client_list' != NULL to
   * determine that SetServerYXX has been called - and then calls
   * ClearServerYXX. However, freeing the allocation happens in free_client() */
  server->serv->client_list =
      (aClient **)RunCalloc(server->serv->nn_mask + 1, sizeof(aClient *));
}

void ClearServerYXX(aClient *server)
{
  if (FindNServer(*server->yxx) == server)	/* Sanity check */
    server_list[convert2n[(unsigned char)*server->yxx]] = NULL;
}

/*
 * SetRemoteNumNick()
 *
 * Register numeric of new, remote, client.
 * Add it to the appropriate client_list.
 */

void SetRemoteNumNick(aClient *acptr, const char *yxx)
{
  register aClient **acptrp;
#ifdef DEBUGMODE
  if (*acptr->user->server->yxx != *yxx)
    MyCoreDump;
#endif
  acptr->yxx[0] = *++yxx;
  acptr->yxx[1] = *++yxx;
  acptr->yxx[2] = 0;

  acptrp = &acptr->user->server->serv->
      client_list[Index(acptr, acptr->user->server)];

  if (*acptrp)
    exit_client(acptr->from, *acptrp, acptr->user->server,
	"Numeric nick collision (Ghost)");

  *acptrp = acptr;
}


/*
 * SetLocalNumNick()
 *
 * Register numeric of new, local, client. Add it to our client_list.
 */

void SetLocalNumNick(aClient *cptr)
{
  u_char x1, x2;
  register int index;

#ifdef DEBUGMODE
  if (cptr->user->server != &me)
    MyCoreDump;
#endif

  do
  {
    x1 = last_x1;
    x2 = last_x2++;
    if (x2 == NUMNICKBASE - 1)
    {
      last_x2 = 0;
      last_x1++;
      if (x1 == NUMNICKBASE - 1)
	last_x1 = 0;
    }
    index = ((x1 << NUMNICKLOG) + x2) & me.serv->nn_mask;
  }
  while (me.serv->client_list[index]);

  me.serv->client_list[index] = cptr;	/* Reserve the numeric ! */
  cptr->yxx[0] = convert2y[x1];
  cptr->yxx[1] = convert2y[x2];
  cptr->yxx[2] = 0;
}

aClient *findNUser(const char *yxx)
{
  register aClient *server = FindNServer(*yxx);
  register aClient *client;
  if (!server)
    return NULL;
  client = server->serv->client_list[((convert2n[(unsigned char)yxx[1]] <<
      NUMNICKLOG) + convert2n[(unsigned char)yxx[2]]) & server->serv->nn_mask];
  if (!client)
    return NULL;
  /* Is it still the same client ? */
  return *client->yxx == yxx[1] ? client : NULL;
}

/* 
 * markMatchexServer()
 * Mark all servers whose name matches the given (compiled) mask
 * and return their count, abusing FLAGS_MAP for this :)
 */
int markMatchexServer(const char *cmask, int minlen)
{
  register int cnt = 0;
  register int i;
  register aClient *acptr;

  for (i = 0; i < NUMNICKBASE; i++)
    if ((acptr = server_list[i]))
    {
      if (matchexec(acptr->name, cmask, minlen))
	acptr->flags &= ~FLAGS_MAP;
      else
      {
	acptr->flags |= FLAGS_MAP;
	cnt++;
      }
    }
  return cnt;
}

aClient *find_match_server(char *mask)
{
  aClient *acptr;
  int i;

  if (!(BadPtr(mask)))
  {
    collapse(mask);
    for (i = 0; i < NUMNICKBASE; i++)
    {
      if ((acptr = server_list[i]) && (!match(mask, acptr->name)))
	return acptr;
    }
  }
  return NULL;
}


#ifndef NO_PROTOCOL9

/******************************************************************************
 *
 * The following functions can be removed as soon as all servers have upgraded
 * to ircu2.10.
 */

/*
 * CreateNNforProtocol9server
 *
 * We do not receive numeric nicks from servers behind a protocol 9 link
 * so we generate it ourselfs.
 */
char *CreateNNforProtocol9server(aClient *server)
{
  static char YXX[4];
  register aServer *serv = server->serv;
  register int index;
  u_char x1, x2;
  int cnt = 0;
#ifdef DEBUGMODE
  if (!IsServer(server) || Protocol(server) != 9)
    MyCoreDump;
#endif
  YXX[0] = *server->yxx;
  YXX[3] = 0;
  do
  {
    if (cnt++ == MAX_MAXCLIENTS)
      return NULL;
    x1 = serv->last_x1;
    x2 = serv->last_x2++;
    if (x2 == NUMNICKBASE - 1)
    {
      serv->last_x2 = 0;
      serv->last_x1++;
      if (x1 == NUMNICKBASE - 1)
	serv->last_x1 = 0;
    }
    index = ((x1 << NUMNICKLOG) + x2) & serv->nn_mask;
  }
  while (serv->client_list[index]);
  YXX[1] = convert2y[x1];
  YXX[2] = convert2y[x2];
  return YXX;
}

#endif /* !NO_PROTOCOL9 */
