/*
**	CAP afpfile interface - maps file extensions to TYPE/CREATOR
**
**	Adapted from mkhfs routines for mkhybrid
**
**	James Pearson 1/5/97
**	Bug fix JCP 4/12/97
*/

#ifdef APPLE_HYB

#include <errno.h>
#include "hfs.h"
#include "apple.h"

static afpmap	**map;				/* list of mappings */
static afpmap	*defmap;			/* the default mapping */
static int	last_ent;			/* previous mapped entry */
static int	num;				/* number of mappings */
static int	mlen;				/* min extension length */
static char	tmp[MAXPATHLEN];		/* tmp working buffer */

/*
**	read_cap_info:	open and read a finderinfo file for a CAP file
**			or directory
*/
int
read_cap_info(char *name, FileInfo *info)
/* char		*name;				finderinfo filename */
/* FileInfo 	*info;				CAP info struct */
{
	FILE	*fp;
	int	num;

	/* clear out any old finderinfo stuf */
	memset(info, 0, sizeof(FileInfo));

	if ((fp = fopen(name,"rb")) == NULL)
	    return(-1);

	/* read and ignore if the file is short - checked later */
	num = fread(info,1,sizeof(FileInfo),fp);

	fclose(fp);

	return(num);
}
/*
**	get_cap_dir_name: get the Mac name for a directory
*/
int
get_cap_dir_name(char *wname, int wlen, char *dname, hfsdirent *hfs_ent)
/* char		*wname				whole path */
/* int		wlen				length of parent pathname */
/* char		*dname				this dir name */
/* hfsdirent	*hfs_ent			HFS parameters */
{
	char		*info_name;		/* finderinfo filename */
	FileInfo	info;			/* finderinfo struct */
	char		*name;			/* Mac name */
	int		num = -1;		/* bytes read */

	strcpy(tmp, wname);

	sprintf(tmp+wlen, "%s/%s", FINDERINFO, dname);
	num = read_cap_info(tmp, &info);

	/* check finder info is OK */
	if (num > 0
		&& info.fi_magic1 == FI_MAGIC1
		&& info.fi_magic == FI_MAGIC
		&& info.fi_bitmap & FI_BM_MACINTOSHFILENAME) {
	    /* use the finderinfo name if it exists */
	    name = info.fi_macfilename;
	}
	else {
	    /* otherwise give it it's Unix name */
	    name = dname;
	}

	strncpy(hfs_ent->name, name, HFS_MAX_FLEN);
	/* truncate the Mac name to a maximum of 31 characters */
	hfs_ent->name[HFS_MAX_FLEN] = '\0';
}

/*
**	get_cap_info:	get CAP finderinfo for a file
*/
int
get_cap_info(char *wname, int wlen, char *dname, hfsdirent *hfs_ent)
/* char		*wname				whole path */
/* int		wlen				length of parent pathname */
/* char		*dname				this dir name */
/* hfsdirent	*hfs_ent			HFS parameters */
{
	FileInfo 	info;			/* finderinfo struct */
	int		num = -1;		/* bytes read */
	char		*name;
	char		*c;
	char		*t;

	/* finder info lives in the ".finderinfo" sub directory with
	   the same filename - insert .finderinfo/ in the whole_path */
	strcpy(tmp, wname);

	sprintf(tmp+wlen, "%s/%s", FINDERINFO, dname);
	num = read_cap_info(tmp, &info);

	/* check finder info is OK */
	if (num > 0 
		&& info.fi_magic1 == FI_MAGIC1
		&& info.fi_magic == FI_MAGIC) {

	    if (info.fi_bitmap & FI_BM_MACINTOSHFILENAME) {
		/* use the finderinfo name if it exists */
		name = info.fi_macfilename;
	    }
	    else {
		name = dname;
	    }

	    /* type and creator from finder info */
	    t = info.fdType;
	    c = info.fdCreator;

	    /* finder flags */
	    hfs_ent->fdflags = d_getw((unsigned char *)&info.fdFlags);
	    /* clear HFS_FNDR_HASBEENINITED to have tidy desktop ?? */
	    hfs_ent->fdflags &= 0xfeff;

#ifdef USE_MAC_DATES
	    /* set created/modified dates - these date should have already
	       been set from the Unix data fork dates. The finderinfo dates
	       are in Mac format - but we have to convert them back to Unix
	       for the time being  */
	    if ((info.fi_datemagic & FI_CDATE)) {
		/* use libhfs routines to get correct byte order */
		hfs_ent->crdate = d_toutime(d_getl(info.fi_ctime));
	    }
	    if (info.fi_datemagic & FI_MDATE) {
		hfs_ent->mddate = d_toutime(d_getl(info.fi_mtime));
	    }
#endif /* USE_MAC_DATES */
	}
	else {
	    /* failed to open/read finderinfo - so try afpfile mapping */
	    map_ext(dname, &t, &c, &hfs_ent->fdflags);
	    name = dname;
	}

	/* set name, creator, type */

	strncpy(hfs_ent->name, name, HFS_MAX_FLEN);
	/* truncate the Mac name to a maximum of 31 characters */
	hfs_ent->name[HFS_MAX_FLEN] = '\0';

	strncpy(hfs_ent->type, t, CT_SIZE);
	strncpy(hfs_ent->creator, c, CT_SIZE);
	hfs_ent->type[CT_SIZE] = '\0';
	hfs_ent->creator[CT_SIZE] = '\0';
}

/*
**	set_ext: sets up the mapping list from the afpfile as well
**		 the default mapping (with or without) an afpfile
*/
void
set_ext(char *name, char *deftype, char *defcreator, unsigned short fdflags)
/* char		*name;				/* afpfile name */
/* char		*deftype;			/* default type */
/* char		*defcreator;			/* default creator */
/* u_short	*fdflags;			/* default finder flags */
{
	FILE	*fp;				/* File pointer */
	int	count = NUMMAP;			/* max number of entries */
	char	buf[MAXPATHLEN];		/* working buffer */
	afpmap	*amap;				/* mapping entry */
	char	*c, *t, *e;

	/* min length set to max to start with */
	mlen = MAXPATHLEN;

	/* set defaults */
	num = last_ent = 0;

	/* allocate memory for the default entry */
	if((defmap = (afpmap *)malloc(sizeof(afpmap))) == NULL)
	    perr("not enough memory");

	/* set default values */
	defmap->extn = DEFMATCH;

	/* make sure creator and type are 4 chars long */
	strcpy(defmap->type, BLANK);
	strcpy(defmap->creator, BLANK);
	
	e = deftype;
	t = defmap->type;

	while(*e && (e - deftype) < CT_SIZE)
	    *t++ = *e++;

	e = defcreator;
	c = defmap->creator;

	while(*e && (e - defcreator) < CT_SIZE)
	    *c++ = *e++;

	/* length is not important here */
	defmap->elen = 0;

	/* no flags */
	defmap->fdflags = fdflags;

	/* no afpfile - no mappings */
	if (*name == '\0') {
	    map = NULL;
	    return;
	}

	if((fp = fopen(name,"r")) == NULL)
	    perr("unable to open mapping file: %s", name);

	if((map = (afpmap **)malloc(NUMMAP * sizeof(afpmap *))) == NULL)
	    perr("not enough memory");

	/* read afpfile line by line */
	while(fgets(buf, MAXPATHLEN, fp) != NULL) {
	    /* ignore any comment lines */
	    c = tmp;
	    *c = '\0';
	    if (sscanf(buf,"%1s", c) == EOF || *c == '#')
		continue;

	    /* increase list size if needed */
	    if (num == count) {
		count += NUMMAP;
		map = (afpmap **)realloc(map, count * sizeof(afpmap *));
		if (map == NULL)
		    perr("not enough memory");
	    }

	    /* allocate memory for this entry */
	    if((amap = (afpmap *)malloc(sizeof(afpmap))) == NULL)
		perr("not enough memory");

	    t = amap->type;
	    c = amap->creator;

	    /* extract the info */
	    if(sscanf(buf, "%s%*s%*1s%c%c%c%c%*1s%*1s%c%c%c%c%*1s",
		    tmp, c, c+1, c+2, c+3, t, t+1, t+2, t+3) != 9) {
  		fprintf(stderr,"error scanning afpfile %s - continuing", name);
		free(amap);
		continue;
	    }

	    /* copy the extension found */
	    if ((amap->extn = (char *)strdup(tmp)) == NULL)
		perr("not enough memory");

	    /* set end-of-string */
	    *(t+4) = *(c+4) = '\0';

	    /* find the length of the extension */
	    amap->elen = strlen(amap->extn);

	    /* set flags */
	    amap->fdflags = fdflags;

	    /* see if we have the default creator/type */
	    if(!strcmp(amap->extn, DEFMATCH)) {
		/* get rid of the old default */
		free(defmap);
		/* make this the default */
		defmap = amap;
		continue;
	    }

	    /* update the smallest extension length */
	    mlen = MIN(mlen, amap->elen);

	    /* add entry to the list */
	    map[num++] = amap;

	}

	/* free up some memory */
	if (num != count) {
	    map = (afpmap **)realloc(map, num * sizeof(afpmap *));
	    if (map == NULL)
		perr("not enough memory");
	}

}

/*
**	map_ext: map a files extension with the list to get type/creator
*/
map_ext(char *name, char **type, char **creator, unsigned short *fdflags)
/* char		*name;				/* filename */
/* char		**type;				/* set type */
/* char		**creator;			/* set creator */
/* u_short	*fdflags;			/* set finder flags */
{
	int	i;				/* loop counter */
	int	len;				/* filename length */
	afpmap	*amap;				/* mapping entry */

	len = strlen(name);

	/* have an afpfile and filename if long enough */
	if(map && len >= mlen) {
	    /* search through the list - we start where we left
	       off last time in case this file is of the same type
	       as the last one */
	    for(i=0;i<num;i++) {
		amap = map[last_ent];

		/* compare the end of the filename */
/*		if (!strcmp((name + len - amap->elen), amap->extn)) { */
		if (!strcasecmp((name + len - amap->elen), amap->extn)) {
		    /* set the required info */
		    *type = amap->type;
		    *creator = amap->creator;
		    *fdflags = amap->fdflags;
		    return;
		}
		/* move on to the next entry - wrapping round if neccessary */
		last_ent++;
		last_ent %= num;
	    }
	}

	/* if no matches are found, file name too short, or no 
	   afpfile, then take defaults */
	*type = defmap->type;
	*creator = defmap->creator;
	*fdflags = defmap->fdflags;
}

#endif /* APPLE_HYB */

perr(char *a)
{
	if (a)
		fprintf(stderr,"mkhybrid: %s\n",a);
	perror("mkhybrid");
	exit(1);
}
