#ifndef lint
static char sccsid[] = "@(#)mode.c	3.9 96/05/25 xlockmore";

#endif

/*-
 * mode.c - Modes for xlock. 
 *
 * See xlock.c for copying information.
 *
 * Revision History:
 * 23-Feb-96: Extensive revision to implement new mode hooks and stuff
 *		Ron Hitchens <ron@utw.com>
 * 04-Sep-95: Moved over from mode.h (previously resource.h) with new
 *            "&*_opts" by Heath A. Kehoe <hakehoe@icaen.uiowa.edu>.
 *
 */

#include <ctype.h>
#include "xlock.h"

/* -------------------------------------------------------------------- */

LockStruct  LockProcs[] =
{
	{"ball", init_ball, draw_ball, release_ball,
   init_ball, init_ball, NULL, &ball_opts, 10000, 10, 20, 1.0,
	 "Bouncing balls", 0, NULL},
#ifndef VMS
	{"bomb", init_bomb, draw_bomb, release_bomb,
   refresh_bomb, NULL, NULL, &bomb_opts, 100000, 6, 20, 1.0,
	 "Bomb", 0, NULL},
#endif
#ifdef HAS_XPM
	{"cartoon", init_cartoon, draw_cartoon, release_cartoon,
   NULL, init_cartoon, NULL, &cartoon_opts, 1000, 30, 50, 1.0,
	 "Cartoon", 0, NULL},
#endif
	{"dclock", init_dclock, draw_dclock, release_dclock,
   refresh_dclock, init_dclock, NULL, &dclock_opts, 10000, 20, 10000, 1.0,
	 "Digital Clock", 0, NULL},
#ifdef DRIFT
	{"drift", init_drift, draw_drift, release_drift,
   draw_drift, init_drift, NULL, &drift_opts, 100, 3, 10000, 1.0,
	 "New Cosmic Drift Fractals", 0, NULL},
#endif
	{"flamen", init_flamen, draw_flamen, release_flamen,
   draw_flamen, init_flamen, NULL, &flamen_opts, 100, 3, 10000, 1.0,
	 "New Cosmic Flame Fractals", 0, NULL},
#ifdef HAS_GL
  {"gear", init_gear, draw_gear, release_gear,
   draw_gear, init_gear, NULL, &gear_opts, 1, 1, 1, 1.0,
  "GL Gears"},
#endif
#ifdef HAS_XPM
	{"husker", init_husker, draw_husker, release_husker,
   draw_husker, init_husker, NULL, &husker_opts, 2000000, 1, 20, 1.0,
	 "Husker (American) Football", 0, NULL},
#endif
  {"julia", init_julia, draw_julia, release_julia,
   refresh_julia, init_julia, NULL, &julia_opts, 10000, 1000, 2500, 1.0,
   "Julia set", 0, NULL},
	{"pacman", init_pacman, draw_pacman, release_pacman,
   draw_pacman, init_pacman, NULL, &pacman_opts, 50000, 10, 200, 1.0,
	 "Pacman", 0, NULL},
	{"polygon", init_polygon, draw_polygon, release_polygon,
   draw_polygon, init_polygon, NULL, &polygon_opts, 750000, 40, 140, 1.0,
	 "Polygon", 0, NULL},
	{"sphere2", init_sphere2, draw_sphere2, release_sphere2,
   draw_sphere2, init_sphere2, NULL, &sphere2_opts, 100000, 6, 20, 1.0,
	 "3D ball", 0, NULL},
	{"star", init_star, draw_star, release_star,
   draw_star, init_star, NULL, &star_opts, 10000, 1, 100, 1.0,
	 "Stars", 0, NULL},
	{"turtle", init_turtle, draw_turtle, release_turtle,
   init_turtle, init_turtle, NULL, &turtle_opts, 500000, 6, 20, 1.0,
	 "Turtle", 0, NULL},
	{"random", init_random, draw_random, NULL,
   refresh_random, change_random, NULL, &random_opts, 1, 0, 0, 0.0,
	 "Random mode", 0, NULL},
};

int         numprocs = sizeof (LockProcs) / sizeof (LockProcs[0]);

/* -------------------------------------------------------------------- */

static LockStruct *last_initted_mode = NULL;
static LockStruct *default_mode = NULL;

/* -------------------------------------------------------------------- */

void
set_default_mode(LockStruct * ls)
{
	default_mode = ls;
}

/* -------------------------------------------------------------------- */

static
void
set_window_title(ModeInfo * mi)
{
	XTextProperty prop;
	char        buf[512];
	char       *ptr = buf;

	(void) sprintf(buf, "%s: %s", MI_NAME(mi), MI_DESC(mi));

	XStringListToTextProperty(&ptr, 1, &prop);
	XSetWMName(MI_DISPLAY(mi), MI_WINDOW(mi), &prop);
}

/* -------------------------------------------------------------------- */

/* 
 *    This hook is called prior to calling the init hook of a
 *      different mode.  It is to inform the mode that it is losing
 *      control, and should therefore release any dynamically created
 *      resources.
 */

static
void
call_release_hook(LockStruct * ls, ModeInfo * mi)
{
	if (ls == NULL) {
		return;
	}
	MI_LOCKSTRUCT(mi) = ls;

	if (ls->release_hook != NULL) {
		ls->release_hook(mi);
	}
	ls->flags &= ~(LS_FLAG_INITED);

	last_initted_mode = NULL;
}

void
release_last_mode(ModeInfo * mi)
{
	if (last_initted_mode == NULL) {
		return;
	}
	call_release_hook(last_initted_mode, mi);

	last_initted_mode = NULL;
}

/* -------------------------------------------------------------------- */

/* 
 *    Call the init hook for a mode.  If this mode is not the same
 *      as the last one, call the release proc for the previous mode
 *      so that it will surrender its dynamic resources.
 *      A mode's init hook may be called multiple times, without
 *      intervening release calls.
 */

void
call_init_hook(LockStruct * ls, ModeInfo * mi)
{
	if (ls == NULL) {
		if (default_mode == NULL) {
			return;
		} else {
			ls = default_mode;
		}
	}
	if (ls != last_initted_mode) {
		call_release_hook(last_initted_mode, mi);
	}
	MI_LOCKSTRUCT(mi) = ls;

	set_window_title(mi);

	ls->init_hook(mi);

	ls->flags |= LS_FLAG_INITED;
	MI_WIN_SET_FLAG_STATE(mi, WI_FLAG_JUST_INITTED, True);

	last_initted_mode = ls;
}

/* -------------------------------------------------------------------- */

/* 
 *    Call the callback hook for a mode.  This hook is called repeatedly,
 *      at (approximately) constant time intervals.  The time between calls
 *      is controlled by the -delay command line option, which is mapped
 *      to the variable named delay.
 */

void
call_callback_hook(LockStruct * ls, ModeInfo * mi)
{
	if (ls == NULL) {
		if (default_mode == NULL) {
			return;
		} else {
			ls = default_mode;
		}
	}
	MI_LOCKSTRUCT(mi) = ls;

	ls->callback_hook(mi);

	MI_WIN_SET_FLAG_STATE(mi, WI_FLAG_JUST_INITTED, False);
}

/* -------------------------------------------------------------------- */

/* 
 *    Window damage has occurred.  If the mode has been initted and
 *      supplied a refresh proc, call that.  Otherwise call its init
 *      hook again.
 */

#define JUST_INITTED(mi)	(MI_WIN_FLAG_IS_SET(mi, WI_FLAG_JUST_INITTED))

void
call_refresh_hook(LockStruct * ls, ModeInfo * mi)
{
	if (ls == NULL) {
		if (default_mode == NULL) {
			return;
		} else {
			ls = default_mode;
		}
	}
	MI_LOCKSTRUCT(mi) = ls;

	if (ls->refresh_hook == NULL) {
		/*
		 * No refresh hook supplied.  If the mode has been
		 * initialized, and the callback has been called at least
		 * once, then call the init hook to do the refresh.
		 * Note that two flags are examined here.  The first
		 * indicates if the mode has ever had its init hook called,
		 * the second is a per-screen flag which indicates
		 * if the draw (callback) hook has been called since the
		 * init hook was called for that screen.
		 * This second test is a hack.  A mode should gracefully
		 * deal with its init hook being called twice in a row.
		 * Once all the modes have been updated, this hack should
		 * be removed.
		 */
		if (MODE_NOT_INITED(ls) || JUST_INITTED(mi)) {
			return;
		}
		call_init_hook(ls, mi);
	} else {
		ls->refresh_hook(mi);
	}
}

/* -------------------------------------------------------------------- */

/* 
 *    The user has requested a change, by pressing the middle mouse
 *      button.  Let the mode know about it.
 */

void
call_change_hook(LockStruct * ls, ModeInfo * mi)
{
	if (ls == NULL) {
		if (default_mode == NULL) {
			return;
		} else {
			ls = default_mode;
		}
	}
	MI_LOCKSTRUCT(mi) = ls;

	if (ls->change_hook != NULL) {
		ls->change_hook(mi);
	}
}

/* -------------------------------------------------------------------- */
