/* Chown-advanced command -- for the Midnight Commander
   Copyright (C) 1994, 1995 Radek Doulik

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.  
 */

#include <string.h>
#include <stdio.h>
#include <stdlib.h>		/* For malloc() */
#include <ncurses.h>
#include "mad.h"
#include "win.h"
#include "input.h"
#include "color.h"
#include "dlg.h"
#include "widget.h"
#include "dialog.h"		/* For do_refresh() */
#include "wtools.h"		/* For init_box_colors() */

/* Needed for the extern declarations of integer parameters */
#include <sys/types.h>
#include <sys/param.h>
#include <sys/stat.h>
#include <grp.h>
#include <pwd.h>
#ifdef HAVE_UNISTD_H
#include <unistd.h>
#endif
#include "dir.h"
#include "panel.h"		/* Needed for the externs */
#include "util.h"		/* Needed for the externs */
#include "file.h"
#include "chmod.h"
#include "main.h"

#define BX		5
#define BY		6

#define TX              50
#define TY              2

#define BUTTONS		9

#define B_SETALL        B_USER
#define B_SETUSR        B_USER + 1
#define B_SETGRP        B_USER + 2
#define B_OWN           B_USER + 3
#define B_GRP           B_USER + 4
#define B_OTH           B_USER + 5
#define B_OOWN          B_USER + 6
#define B_OGRP          B_USER + 7

struct stat *sf_stat;
static int need_update, end_chown;

static WListbox *l_user, *l_group;

struct {
    int ret_cmd, y, x;
    char *text;
    int hkey, hpos;
} chown_advanced_but[BUTTONS] = {

    {
	B_CANCEL, 4, 55, "[ Cancel ]", 'c', 2,
    },
    {
	B_ENTER, 4, 46, "[ Set ]", 's', 2,
    },
    {
	B_SETUSR, 4, 36, "[ Skip ]", 'k', 3,
    },
    {
	B_SETGRP, 4, 23, "[ Set all ]", 'a', 6,
    },
    {
	B_OGRP, 0, 48, "[               ]", 0, -1,
    },
    {
	B_OOWN, 0, 30, "[               ]", 0, -1,
    },
    {
	B_OTH, 0, 19, "[   ]", 0, -1,
    },
    {
	B_GRP, 0, 11, "[   ]", 0, -1,
    },
    {
	B_OWN, 0, 3, "[   ]", 0, -1,
    },
};

WButton *b_own, *b_grp, *b_oth,	/* permission */
*b_user, *b_group;		/* owner */

static int files_on_begin, flag_pos;
static char ch_flags[11];
static char *ch_perm = "rwx";
static umode_t ch_cmode;

static int inc_flag_pos (int f_pos)
{
    if (flag_pos == 10) {
	flag_pos = 0;
	return 0;
    }
    flag_pos++;
    if (!(flag_pos % 3) || f_pos > 2)
	return 0;
    return 1;
}

static int dec_flag_pos (int f_pos)
{
    if (!flag_pos) {
	flag_pos = 10;
	return 0;
    }
    flag_pos--;
    if (!((flag_pos + 1) % 3) || f_pos > 2)
	return 0;
    return 1;
}

static void set_perm_by_flags (char *s, int f_p)
{
    int i;

    for (i = 0; i < 3; i++)
	if (ch_flags[f_p + i] == '+')
	    s[i] = ch_perm[i];
	else if (ch_flags[f_p + i] == '-')
	    s[i] = '-';
	else
	    s[i] = (ch_cmode & (1 << (8 - f_p - i))) ? ch_perm[i] : '-';
}

static void print_flags (void)
{
    int i;

    wattrset (ch_win, COLOR_NORMAL);

    for (i = 0; i < 3; i++)
	mvwaddch (ch_win, BY + 1, 9 + i, ch_flags[i]);
    for (i = 0; i < 3; i++)
	mvwaddch (ch_win, BY + 1, 17 + i, ch_flags[i + 3]);
    for (i = 0; i < 3; i++)
	mvwaddch (ch_win, BY + 1, 25 + i, ch_flags[i + 6]);

    set_perm_by_flags (b_own->text + 1, 0);
    set_perm_by_flags (b_grp->text + 1, 3);
    set_perm_by_flags (b_oth->text + 1, 6);

    for (i = 0; i < 15; i++)
	mvwaddch (ch_win, BY + 1, 36 + i, ch_flags[9]);
    for (i = 0; i < 15; i++)
	mvwaddch (ch_win, BY + 1, 54 + i, ch_flags[10]);
}

static int advanced_chown_callback (Dlg_head * h, int Par, int Msg)
{
    int f_pos = BUTTONS - h->current->dlg_id - single_set - 1;

    switch (Msg) {
    case DLG_POST_KEY:
	if (f_pos < 3)
	    wmove (ch_win, BY, h->current->widget->x + 1 + (flag_pos % 3));
	break;
    case DLG_FOCUS:
	if (f_pos < 3) {
	    if ((flag_pos / 3) != f_pos)
		flag_pos = f_pos * 3;
	    wmove (ch_win, BY, h->current->widget->x + 1 + (flag_pos % 3));
	} else if (f_pos < 5)
	    flag_pos = f_pos + 6;
	break;
    case DLG_KEY:
	switch (Par) {
	case KEY_LEFT:
	case KEY_RIGHT:
	    if (f_pos < 5) {
		if (Par == KEY_LEFT)
		    return (dec_flag_pos (f_pos));
		return (inc_flag_pos (f_pos));
	    }
	    break;
	case 'r':
	case 'w':
	case 'x':
	    if (f_pos > 2)
		break;
	    flag_pos = f_pos * 3 + (strchr (ch_perm, Par) - ch_perm);
	    if (((WButton *) h->current->widget)->text[(flag_pos % 3) + 1] == '-')
		ch_flags[flag_pos] = '+';
	    else
		ch_flags[flag_pos] = '-';
	    print_flags ();
	    send_message (h->current, h, WIDGET_FOCUS, 0);
	    break;

	case '1':
	case '2':
	case '4':
	    if (f_pos > 2)
		break;
	    Par -= '1';
	    if (Par == 3)
		Par--;
	    flag_pos = Par + f_pos * 3;
	    ch_flags[flag_pos] = '=';
	    print_flags ();
	    send_message (h->current, h, WIDGET_FOCUS, 0);
	    break;
	case '=':
	case '-':
	    if (f_pos > 2)
		break;
	case '+':
	    if (f_pos > 4)
		break;
	    ch_flags[flag_pos] = Par;
	    print_flags ();
	    send_message (h->current, h, WIDGET_FOCUS, 0);
	    break;
	}
	return 0;
    }
    return 0;
}

static void chown_refresh (void)
{
    wattrset (ch_win, COLOR_NORMAL);
    wclr (ch_win);

    draw_box (ch_win, 1, 2, 11, 70);

    mvwprintw (ch_win, BY - 1, 8, "owner");
    mvwprintw (ch_win, BY - 1, 16, "group");
    mvwprintw (ch_win, BY - 1, 24, "other");

    mvwprintw (ch_win, BY - 1, 35, "owner");
    mvwprintw (ch_win, BY - 1, 53, "group");

    mvwprintw (ch_win, 3, 4, "On");
    mvwprintw (ch_win, BY + 1, 4, "Flag");
    mvwprintw (ch_win, BY + 2, 4, "Mode");

    if (!single_set)
	mvwprintw (ch_win, 3, 54, "%6d of %d", files_on_begin - (cpanel->marked) + 1,
		   files_on_begin);

    print_flags ();

    wattrset (ch_win, COLOR_HOT_NORMAL);
    mvwprintw (ch_win, 1, 24, " Chown advanced command ");

    wrefresh (ch_win);
}

static int l_call (void)
{
    return 1;
}

static void init_chown_advanced (void)
{
    int i;

    sf_stat = (struct stat *) malloc (sizeof (struct stat));
    do_refresh ();
    end_chown = need_update = c_file = 0;
    single_set = (cpanel->marked < 2) ? 2 : 0;
    memset (ch_flags, '=', 11);
    flag_pos = 0;

#ifdef BUGGY_CURSES
    touchwin (stdscr);
#endif

    ch_win = centerwin (13, 74);
    init_box_colors (ch_colors);

    ch_dlg = dlg_new (ch_win, ch_colors, advanced_chown_callback,
		      winpos (22, 74), "[Chown-advanced]");

#define XTRACT(i) BY+chown_advanced_but[i].y, BX+chown_advanced_but[i].x, \
                  chown_advanced_but[i].ret_cmd, chown_advanced_but[i].text, \
                  chown_advanced_but[i].hkey, chown_advanced_but[i].hpos, 0, 0

    for (i = 0; i < BUTTONS - 5; i++)
	if (!single_set || i < 2)
	    add_widget (ch_dlg, button_new (XTRACT (i)));

    b_own = button_new (XTRACT (8));
    b_grp = button_new (XTRACT (7));
    b_oth = button_new (XTRACT (6));
    b_user = button_new (XTRACT (5));
    b_group = button_new (XTRACT (4));

    add_widget (ch_dlg, b_group);
    add_widget (ch_dlg, b_user);
    add_widget (ch_dlg, b_oth);
    add_widget (ch_dlg, b_grp);
    add_widget (ch_dlg, b_own);

}

void chown_advanced_done (void)
{
    free (sf_stat);
    if (need_update)
	update_panels (UP_OPTIMIZE, UP_KEEPSEL, 0);
    repaint_screen (RP_NOCLEAR);
}

static inline void do_chown (uid_t u, gid_t g)
{
    chown (cpanel->dir.list[c_file].fname, u, g);
    cpanel->dir.list[c_file].f.marked = 0;
}

static void apply_chowns (uid_t u, gid_t g)
{
    char *fname;

    need_update = end_chown = 1;
    do_chown (u, g);
    cpanel->marked--;

    do {
	fname = next_file ();

	do_chown (u, g);
	cpanel->marked--;
    } while (cpanel->marked);
}

void set_perm (char *s, int p)
{
    s[0] = (p & 4) ? 'r' : '-';
    s[1] = (p & 2) ? 'w' : '-';
    s[2] = (p & 1) ? 'x' : '-';
}

void chown_advanced_cmd (void)
{
    char *fname, *name_t;
    struct stat sf_stat;
    WLEntry *fe;
    uid_t new_user;
    gid_t new_group;

    files_on_begin = cpanel->marked;

    do {			/* do while any files remaining */
	init_chown_advanced ();

	if (cpanel->marked)
	    fname = next_file ();	/* next marked file */
	else
	    fname = selection->fname;	/* single file */

	if (!stat_file (fname, &sf_stat))	/* get status of file */
	    break;
	ch_cmode = sf_stat.st_mode;

	chown_refresh ();

	/* display file info */
	wattrset (ch_win, COLOR_NORMAL);

	/* name && mode */
	mvwprintw (ch_win, 3, 8, "%s", name_trunc (fname, 45));
	mvwprintw (ch_win, BY + 2, 9, "%12o", sf_stat.st_mode);

	/* set buttons  - ownership */
	name_t = name_trunc (get_owner (sf_stat.st_uid), 15);
	strncpy (b_user->text + 1, name_t, strlen (name_t));
	name_t = name_trunc (get_group (sf_stat.st_gid), 15);
	strncpy (b_group->text + 1, name_t, strlen (name_t));

	/* permissions */
	set_perm (b_own->text + 1, sf_stat.st_mode >> 6);
	set_perm (b_grp->text + 1, sf_stat.st_mode >> 3);
	set_perm (b_oth->text + 1, sf_stat.st_mode);

	/* game can begin */
	run_dlg (ch_dlg);

	switch (ch_dlg->ret_value) {
	case B_CANCEL:
	    end_chown = 1;
	    break;

/*      case B_SETUSR:
   new_user = (getpwnam (l_user->current->text))->pw_uid;
   apply_chowns (new_user, new_group);
   break;

   case B_SETGRP:
   new_group = (getgrnam (l_group->current->text))->gr_gid;
   apply_chowns (new_user, new_group);
   break;

   case B_SETALL:
   case B_ENTER:
   new_group = (getgrnam (l_group->current->text))->gr_gid;
   new_user = (getpwnam (l_user->current->text))->pw_uid;
   if (ch_dlg->ret_value==B_ENTER) {
   need_update = 1;
   chown (fname, new_user, new_group);
   } else
   apply_chowns (new_user, new_group);
   break; */
	}

	if (cpanel->marked && ch_dlg->ret_value != B_CANCEL) {
	    cpanel->dir.list[c_file].f.marked = 0;
	    cpanel->marked--;
	    need_update = 1;
	}
	destroy_dlg (ch_dlg);
	delwin (ch_win);
    } while (cpanel->marked && !end_chown);

    chown_advanced_done ();
}
