/***************************************************************************
                          JavaSlot.cpp  -  description
                             -------------------
    begin                : Tue Oct 31 2000
    copyright            : (C) 2000-2004 by Richard Dale
    email                : Richard_Dale@tipitina.demon.co.uk
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include <stdio.h>

#include <qmetaobject.h>

#include "JavaSlot.moc"
#include <qtjava/QtSupport.h>

/*
 Converts a Java type signature, from a signal or slot, to the corresponding C++ Qt type.
 Used to connect a Java slot to a signal emitted by the underlying C++ Qt instance.
 Note that all Java slots and signals are of type jobjectArray, and so no conversion is needed for them.
*/
static const char * const javaToQtTypeSignatureMap[][2] = {
	{"()",														"()"},
	{"(boolean)",												"(bool)"},
	{"(boolean[],boolean[])",									"(bool*, bool*)"},
	{"(char)",													"(char)"},
	{"(double)",												"(double)"},
	{"(float)",													"(float)"},
	{"(int)",													"(int)"},
	{"(int,boolean)",											"(int,bool)"},
	{"(int,String,byte[])",										"(int,const QCString&,const QByteArray&)"},
	{"(int,QPoint)",											"(int,const QPoint&)"},
	{"(int,String)",											"(int,const QString&)"},
	{"(int,int)",												"(int,int)"},
	{"(int,int,QPoint)",										"(int,int,const QPoint&)"},
	{"(int,int,int)",											"(int,int,int)"},
	{"(int,int,int,QPoint)",									"(int,int,int,const QPoint&)"},
	{"(int,int,QNetworkOperation)",								"(int,int,QNetworkOperation*)"},
	{"(int,QDropEvent)",										"(int,QDropEvent*)"},
	{"(int,QIconViewItem,QPoint)",								"(int,QIconViewItem*,const QPoint&)"},
	{"(int,QListBoxItem,QPoint)",								"(int,QListBoxItem*,const QPoint)"},
	{"(int,QListViewItem,QPoint,int)",							"(int,QListViewItem*,const QPoint,int)"},
	{"(long)",													"(long)"},
	{"(long,byte[])",											"(long,const QByteArray&)"},
	{"(long,String)",											"(long,const QString&)"},
	{"(short)",													"(short)"},
	{"(byte)",													"(unsigned char)"},
	{"(QAction)",												"(QAction *)"},
	{"(int)",													"(QDir::SortSpec)"},
	{"(QDockWindow)",											"(QDockWindow *)"},
	{"(int)",													"(QDockWindow::Place)"},
	{"(QDropEvent)",											"(QDropEvent*)"},
	{"(QDropEvent,QListViewItem)",								"(QDropEvent*,QListViewItem*)"},
	{"(QDropEvent,ArrayList",									"(QDropEvent*,const QValueList<QIconDragItem>&)"},
	{"(QDropEvent,QListViewItem,QListViewItem)",				"(QDropEvent*,QListViewItem*,QListViewItem*)"},
	{"(QIconViewItem)",											"(QIconViewItem*)"},
	{"(QIconViewItem,QPoint)",									"(QIconViewItem*,const QPoint&)"},
	{"(QIconViewItem,String)",									"(QIconViewItem*,const QString&)"},
	{"(ArrayList,ArrayList,ArrayList)",							"(QList<QListViewItem>,QList<QListViewItem>,QList<QListViewItem>)"},
	{"(QListBoxItem)",											"(QListBoxItem*)"},
	{"(QListBoxItem,QPoint)",									"(QListBoxItem*,const QPoint&)"},
	{"(QListViewItem, int,String)",								"(QListViewItem*, int, const QString&)"},
	{"(QListViewItem,QListViewItem,QListViewItem)",				"(QListViewItem*,QListViewItem*,QListViewItem*)"},
	{"(QListViewItem,QPoint,int)",								"(QListViewItem*,const QPoint&,int)"},
	{"(QListViewItem,String,int)",								"(QListViewItem*,const QString&,int)"},
	{"(QListViewItem,int)",										"(QListViewItem*,int)"},
	{"(QListViewItem)",											"(QListViewItem*)"},
	{"(QNetworkOperation)",										"(QNetworkOperation*)"},
	{"(QObject)",												"(QObject*)"},
	{"(QPopupMenu)",											"(QPopupMenu*)"},
	{"(int)",													"(QSql::Op)"},
	{"(int)",													"(QSqlCursor::Mode)"},
	{"(QSqlRecord)",											"(QSqlRecord*)"},
	{"(int)",													"(QTextEdit::VerticalAlignment)"},
	{"(QToolBar)",												"(QToolBar*)"},
	{"(int)",													"(Orientation)"},
	{"(QWidget)",												"(QWidget*)"},
	{"(QWidget,QPoint)",										"(QWidget*,const QPoint&)"},
	{"(QWidget,String)",										"(QWidget*,const QString&)"},
	{"(QWidget,QDropEvent)",									"(QWidget*,const QDropEvent*)"},
	{"(QWidget,QPixmap)",										"(QWidget*,QPixmap)"},
	{"byte[]",													"(const QByteArray&)"},
	{"(byte[],QNetworkOperation)",								"(const QByteArray&,QNetworkOperation*)"},
	{"(String,boolean)",											"(const char*, bool)"},
	{"(String,String,byte[],String,String,String)",				"(const char*,const QString&,const QByteArray&,const QString&,const QString&,const QString&)"},
	{"(char)",													"(const QChar&)"},
	{"(String)",												"(const QCString&)"},
	{"(QColor)",												"(const QColor&)"},
	{"(QColor,String)",											"(const QColor&,const QString&)"},
	{"(Calendar)",												"(const QDate&)"},
	{"(Calendar,Calendar)",										"(const QDate&,const QDate&)"},
	{"(Calendar)",												"(const QDateTime&)"},
	{"(QDragMoveEvent,boolean)",								"(const QDragMoveEvent*,bool&)"},
	{"(QSize)",													"(const QSize&)"},
	{"(QFont)",													"(const QFont&)"},
	{"(QHttpResponseHeader)",									"(const QHttpResponseHeader&)"},
	{"(QImage,int)",											"(const QImage&, int)"},
	{"(QPixmap)",												"(const QPixmap&)"},
	{"(QPoint)",												"(const QPoint&)"},
	{"(QRect)",													"(const QRect&)"},
	{"(QSqlRecord)",											"(const QSqlRecord*)"},
	{"(String)",												"(const QString&)"},
	{"(String,boolean,String)",									"(const QString&,bool,const QString&)"},
	{"(String,String,String)",									"(const QString&,const QCString&,const QString&)"},
	{"(String,QLibrary)",										"(const QString&,QLibrary*)"},
	{"(String,QPoint)",											"(const QString&,const QPoint&)"},
	{"(String,String)",											"(const QString&,const QString&)"},
	{"(String,ArrayList,int)",									"(const QString&,const QStringList&,unsigned int)"},
	{"(String,String,int)",										"(const QString&,const QString&,unsigned int)"},
	{"(String,int)",											"(const QString&,int)"},
	{"(String,int,int)",										"(const QString&,int,int)"},
	{"(String,int,int,int)",									"(const QString&,int,int,int)"},
	{"(ArrayList)",												"(const QStringList&)"},
	{"(Date)",													"(const QTime&)"},
	{"(String,long,long,long)",									"(const QString&,unsigned long,unsigned long,unsigned long)"},
	{"(QUrlInfo)",												"(const QUrlInfo&)"},
	{"(QUrlInfo,QNetworkOperation)",							"(const QUrlInfo&,QNetworkOperation*)"},
	{"(ArrayList,QNetworkOperation)",							"(const QValueList<QUrlInfo>&,QNetworkOperation*)"},
	{"(QVariant)",												"(const QVariant&)"},
	{"(long,long,long,String)",									"(const unsigned long&,const unsigned long&,const unsigned long&,const QString&)"}
};

JavaSlot::JavaSlot(JNIEnv * env, jobject receiver, jstring slot)
{
	jobject		javaInvocation;
	jmethodID	cid;
	jclass		invocationClass;

	invocationClass = env->FindClass("org/kde/qt/Invocation");
	if (invocationClass == NULL) {
		return;
	}

	cid = env->GetMethodID(invocationClass, "<init>", "(Ljava/lang/Object;Ljava/lang/String;)V");
	if (cid == NULL) {
		return;
	}

	javaInvocation = env->NewObject(invocationClass, cid, receiver, slot);
	invocation = env->NewGlobalRef(javaInvocation);
	env->DeleteLocalRef(invocationClass);
}

JavaSlot::~JavaSlot()
{
	JNIEnv *	env;
	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	env->DeleteGlobalRef(invocation);
}

JavaSlotFactory * JavaSlot::_factory = 0L;
	
void
JavaSlot::setJavaSlotFactory(JavaSlotFactory * factory)
{
	if (_factory != 0L) {
		delete _factory;
	}
	
	_factory = factory;
}

JavaSlot *
JavaSlot::createJavaSlot(JNIEnv * env, jobject receiver, jstring slot)
{
	return _factory->createJavaSlot(env, receiver, slot);
}

const char *
JavaSlot::javaToQtSignalName(JNIEnv * env, jstring signal, QMetaObject * smeta)
{
	char	signalName[200];
	char 	javaTypeSignature[200];
	char *	signalString = (char*) env->GetStringUTFChars(signal, (jboolean *) 0);
	
	if (	signalString[0] == '2'
			&& sscanf(signalString, "%[^(]%s", signalName, javaTypeSignature) == 2 )
	{
		env->ReleaseStringUTFChars(signal, signalString);
		return javaToQtSignalType(signalName, javaTypeSignature, smeta);
	}
	
	env->ReleaseStringUTFChars(signal, signalString);
	return "";
}


const char *
JavaSlot::javaToQtSignalType(const char * signalName, const char * javaTypeSignature, QMetaObject * smeta)
{
static char qtSignalString[200];

	for (	unsigned int index = 0;
			index < sizeof(javaToQtTypeSignatureMap)/sizeof(*javaToQtTypeSignatureMap);
			index++ )
	{
		if (strcmp(javaTypeSignature, javaToQtTypeSignatureMap[index][0]) == 0) {
			(void) sprintf(qtSignalString, "%s%s", signalName, javaToQtTypeSignatureMap[index][1]);

			if (smeta == 0 || smeta->findSignal(((const char *) qtSignalString + 1), TRUE) >= 0) {
				return qtSignalString;
			}
		}
	}

	return "";
}

const char *
JavaSlot::javaToQtSlotName(JNIEnv * env, jstring slot, const char * signalString)
{
static char qtSlotString[200];
	char 	javaTypeSignature[200];
	char *	slotString = (char*) env->GetStringUTFChars(slot, (jboolean *) 0);

	if (sscanf(slotString, "%*[^(]%s", javaTypeSignature) == 1) {
		env->ReleaseStringUTFChars(slot, slotString);
		(void) sprintf(qtSlotString, "1invoke%s", javaToQtSlotType(javaTypeSignature, signalString));
		return qtSlotString;
	}

	(void) sprintf(qtSlotString, "1invoke%s", slotString);
	env->ReleaseStringUTFChars(slot, slotString);
	return qtSlotString;
}

const char *
JavaSlot::javaToQtSlotType(const char * javaTypeSignature, const char * signalString)
{
	for (	unsigned int index = 0;
			index < sizeof(javaToQtTypeSignatureMap)/sizeof(*javaToQtTypeSignatureMap);
			index++ )
	{
		if (	strcmp(javaTypeSignature, javaToQtTypeSignatureMap[index][0]) == 0
				&& (	signalString == 0
						|| QObject::checkConnectArgs(signalString, (const QObject *) 0, javaToQtTypeSignatureMap[index][1]) ) )
		{
			return javaToQtTypeSignatureMap[index][1];
		}
	}
	
	// If no matching C++ type signature is found, then just return the java one
	return javaTypeSignature;
}

void
JavaSlot::invoke()
{
	invoke((jobjectArray) 0);
}

void
JavaSlot::invoke(bool arg)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Z)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, arg);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(bool* arg1, bool* arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "([Z[Z)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	jbooleanArray jarg1 = QtSupport::fromBooleanPtr(env, arg1);
	jbooleanArray jarg2 = QtSupport::fromBooleanPtr(env, arg2);
	result = env->CallObjectMethod(invocation, mid, jarg1, jarg2);
	*arg1 = *(QtSupport::toBooleanPtr(env, jarg1));
	*arg2 = *(QtSupport::toBooleanPtr(env, jarg1));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(char arg)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(C)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, arg);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const char* arg1, bool arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Z)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid, QtSupport::fromCharString(env, (char *)arg1), (jboolean) arg2);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const char* arg1,const QString& arg2,const QByteArray& arg3,const QString& arg4,const QString& arg5,const QString& arg6)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;[BLjava/lang/Object;Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromCharString(env, (char*)arg1),
									QtSupport::fromQString(env, (QString*)&arg2),
									QtSupport::fromQByteArray(env, (QByteArray *) &arg3),
									QtSupport::fromQString(env, (QString*)&arg4),
									QtSupport::fromQString(env, (QString*)&arg5),
									QtSupport::fromQString(env, (QString*)&arg6) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QByteArray& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "([B)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQByteArray(env, (QByteArray *) &arg1) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QByteArray& arg1,QNetworkOperation* arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "([BLjava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQByteArray(env, (QByteArray *) &arg1),
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.QNetworkOperation") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QChar& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(C)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::fromQChar(env, (QChar *) &arg1));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QColor& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.QColor"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QColor& arg1,const QString& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.QColor"),
									QtSupport::fromQString(env, (QString*)&arg2) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QCString& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::fromQCString(env, (QCString*)&arg1));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QDate& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::fromQDate(env, (QDate *) &arg1));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QDate& arg1,const QDate& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQDate(env, (QDate *) &arg1),
									QtSupport::fromQDate(env, (QDate *) &arg2) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QDateTime& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::fromQDateTime(env, (QDateTime *) &arg1));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QDragMoveEvent* arg1,bool& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Z)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.QDragMoveEvent"), (jboolean) arg2);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QFont& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.QFont"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QHttpResponseHeader& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.QHttpResponseHeader"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QImage& arg1, int arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.QImage"), (jint) arg2);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QPixmap& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.QPixmap"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QPoint& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.QPoint"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QRect& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.QRect"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QSize& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.QSize"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QSqlRecord* arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QSqlRecord"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QString& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid, QtSupport::fromQString(env, (QString *) &arg1));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QString& arg1,bool arg2,const QString& arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;ZLjava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQString(env, (QString*) &arg1),
									(jboolean) arg2,
									QtSupport::fromQString(env, (QString*) &arg3) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QString& arg1,const QCString& arg2,const QString& arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQString(env, (QString*) &arg1),
									QtSupport::fromQString(env, (QString*) &arg2),
									QtSupport::fromQString(env, (QString*) &arg3) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QString& arg1,const QPoint& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQString(env, (QString *) &arg1),
									QtSupport::objectForQtKey(env, (void *) &arg2, "org.kde.qt.QPoint") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QString& arg1,const QString& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQString(env, (QString *) &arg1),
									QtSupport::fromQString(env, (QString *) &arg2) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QString& arg1,const QString& arg2,unsigned int arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQString(env, (QString *) &arg1),
									QtSupport::fromQString(env, (QString *) &arg2),
									(jint) arg3 );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QString& arg1,const QStringList& arg2,unsigned int arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQString(env, (QString*) &arg1),
									QtSupport::arrayWithQStringList(env, (QStringList*)&arg2),
									(jint) arg3 );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QString& arg1,int arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQString(env, (QString *) &arg1),
									(jint) arg2 );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QString& arg1,int arg2,int arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;II)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQString(env, (QString *) &arg1),
									(jint) arg2,
									(jint) arg3 );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QString& arg1,int arg2,int arg3,int arg4)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;III)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQString(env, (QString *) &arg1),
									(jint) arg2,
									(jint) arg3,
									(jint) arg4 );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QString& arg1,QLibrary* arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQString(env, (QString *) &arg1),
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.QLibrary") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QString& arg1,unsigned long arg2,unsigned long arg3,unsigned long arg4)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;JJJ)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::fromQString(env, (QString *) &arg1),
									(jlong) arg2, (jlong) arg3,(jlong) arg4 );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QStringList& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::arrayWithQStringList(env, (QStringList *) &arg1));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QTime& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.QTime"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QUrlInfo& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.QUrlInfo") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QUrlInfo& arg1,QNetworkOperation* arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.QUrlInfo"),
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.QNetworkOperation") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QValueList<QUrlInfo>& arg1,QNetworkOperation* arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::arrayWithQUrlInfoList(env, (QValueList<QUrlInfo> *) &arg1),
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.QNetworkOperation") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const QVariant& arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.QVariant"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(const unsigned long& arg1,const unsigned long& arg2,const unsigned long& arg3,const QString& arg4)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(IIILjava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									(jlong) arg1,
									(jlong) arg2,
									(jlong) arg3,
									QtSupport::fromQString(env, (QString*) &arg4) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(double arg)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(D)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, arg);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(float arg)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(F)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, arg);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, arg);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1, bool arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(IZ)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, (jint) arg1, (jboolean) arg2);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,const QCString& arg2,const QByteArray& arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(ILjava/lang/Object;[B)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									(jlong) arg1,
									QtSupport::fromQCString(env, (QCString*)&arg2),
									QtSupport::fromQByteArray(env, (QByteArray *) &arg3) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,const QPoint& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(ILjava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									(jint) arg1,
									QtSupport::objectForQtKey(env, (void *) &arg2, "org.kde.qt.QPoint") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,const QString& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(ILjava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid, (jint) arg1, QtSupport::fromQString(env, (QString *) &arg2));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,int arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(II)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, (jint) arg1, (jint) arg2);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,int arg2,const QPoint& arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(IILjava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									(jint) arg1,
									(jint) arg2,
									QtSupport::objectForQtKey(env, (void *) &arg3, "org.kde.qt.QPoint") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,int arg2,int arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(III)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, (jint) arg1, (jint) arg2, (jint) arg3);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,int arg2,int arg3,const QPoint& arg4)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(IIILjava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									(jint) arg1,
									(jint) arg2,
									(jint) arg3,
									QtSupport::objectForQtKey(env, (void *) &arg4, "org.kde.qt.QPoint"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,int arg2,QNetworkOperation* arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(IILjava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation,
									mid,
									arg1,
									arg2,
									QtSupport::objectForQtKey(env, (void *) arg3, "org.kde.qt.QNetworkOperation") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,QDropEvent* arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(ILjava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation,
									mid,
									arg1,
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.QDropEvent") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,QIconViewItem* arg2,const QPoint& arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(ILjava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									(jint) arg1,
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.QIconViewItem"),
									QtSupport::objectForQtKey(env, (void *) &arg3, "org.kde.qt.QPoint") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,QListBoxItem* arg2,const QPoint arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(ILjava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									(jint) arg1,
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.QListBoxItem"),
									QtSupport::objectForQtKey(env, (void *) &arg3, "org.kde.qt.QPoint") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(int arg1,QListViewItem* arg2,const QPoint arg3,int arg4)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(ILjava/lang/Object;Ljava/lang/Object;I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									(jint) arg1,
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.QListBoxItem"),
									QtSupport::objectForQtKey(env, (void *) &arg3, "org.kde.qt.QPoint"),
									(jint) arg4 );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(long arg)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(J)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, arg);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(long arg1,const QByteArray& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(J[B)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									(jlong) arg1,
									QtSupport::fromQByteArray(env, (QByteArray *) &arg2) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(long arg1,const QString& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(JLjava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid, (jlong) arg1, QtSupport::fromQString(env, (QString *) &arg2));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QAction * arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QAction"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QDir::SortSpec arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, (jint) arg1);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QDockWindow * arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QDockWindow"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QDockWindow::Place arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid, arg1);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QDropEvent* arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QDropEvent"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QDropEvent* arg1,const QValueList<QIconDragItem>& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QDropEvent"),
									QtSupport::arrayWithQIconDragItemList(env, (QValueList<QIconDragItem> *) &arg2) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QDropEvent* arg1,QListViewItem*arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QDropEvent"),
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.QListViewItem") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QDropEvent*arg1,QListViewItem*arg2,QListViewItem*arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QDropEvent"),
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.QListViewItem"),
									QtSupport::objectForQtKey(env, (void *) arg3, "org.kde.qt.QListViewItem") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QIconViewItem* arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QIconViewItem"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QIconViewItem* arg1,const QPoint& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QIconViewItem"),
									QtSupport::objectForQtKey(env, (void *) &arg2, "org.kde.qt.QPoint") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QIconViewItem* arg1,const QString& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QIconViewItem"),
									QtSupport::fromQString(env, (QString *) &arg2) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QListBoxItem* arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QListBoxItem"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QListBoxItem* arg1,const QPoint& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QListBoxItem"),
									QtSupport::objectForQtKey(env, (void *) &arg2, "org.kde.qt.QPoint") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QList<QListViewItem> arg1,QList<QListViewItem> arg2,QList<QListViewItem> arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) &arg1, "org.kde.qt.QList"),
									QtSupport::objectForQtKey(env, (void *) &arg2, "org.kde.qt.QList"),
									QtSupport::objectForQtKey(env, (void *) &arg3, "org.kde.qt.QList") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QListViewItem*arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QListViewItem"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QListViewItem* arg1,const QPoint& arg2,int arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QListBoxItem"),
									QtSupport::objectForQtKey(env, (void *) &arg2, "org.kde.qt.QPoint"),
									(jint) arg3 );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QListViewItem* arg1,const QString& arg2,int arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QListViewItem"),
									QtSupport::fromQString(env, (QString *) &arg2),
									(jint) arg3 );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QListViewItem* arg1,int arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QListViewItem"),
									(jint) arg2 );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QListViewItem* arg1, int arg2, const QString& arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;ILjava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QListViewItem"),
									(jint) arg2,
									QtSupport::fromQString(env, (QString *) &arg3) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QListViewItem* arg1,QListViewItem*arg2,QListViewItem*arg3)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QListViewItem"),
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.QListViewItem"),
									QtSupport::objectForQtKey(env, (void *) arg3, "org.kde.qt.QListViewItem") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QNetworkOperation* arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QNetworkOperation") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QObject* arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QObject") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QPopupMenu* arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QPopupMenu") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QSqlCursor::Mode arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, (jint) arg1);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QSql::Op arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, (jint) arg1);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QSqlRecord* arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QSqlRecord") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QTextEdit::VerticalAlignment arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, (jint) arg1);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QToolBar* arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QToolBar"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(Orientation arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(I)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid, (jint) arg1);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QWidget* arg1)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QWidget"));
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QWidget* arg1,const QDropEvent* arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, 
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QWidget"),
									QtSupport::objectForQtKey(env, (void *) arg2, "org.kde.qt.QDropEvent") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QWidget* arg1,const QPoint& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, 
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QWidget"),
									QtSupport::objectForQtKey(env, (void *) &arg2, "org.kde.qt.QPoint") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QWidget* arg1,const QString& arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(	invocation, mid,
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QWidget"),
									QtSupport::fromQString(env, (QString *) &arg2) );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(QWidget* arg1,QPixmap arg2)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(Ljava/lang/Object;Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, 
									QtSupport::objectForQtKey(env, (void *) arg1, "org.kde.qt.QWidget"),
									QtSupport::objectForQtKey(env, (void *) &arg2, "org.kde.qt.QPixmap") );
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(short arg)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(S)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, arg);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(unsigned char arg)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "(B)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, arg);
	env->PopLocalFrame(0);
}

void
JavaSlot::invoke(jobjectArray args)
{
	JNIEnv *	env;
	jclass		cls;
	jmethodID	mid;
	jobject		result;

	env = QtSupport::GetEnv();
	env->PushLocalFrame(10);

	cls = env->GetObjectClass(invocation);
	mid = env->GetMethodID(cls, "invoke", "([Ljava/lang/Object;)Ljava/lang/Object;");
	if (mid == NULL) {
		return;
	}
	result = env->CallObjectMethod(invocation, mid, args);
	env->PopLocalFrame(0);
}

JavaSlotFactory::JavaSlotFactory()
{
}

JavaSlot *
JavaSlotFactory::createJavaSlot(JNIEnv * env, jobject receiver, jstring slot)
{
	return new JavaSlot(env, receiver, slot);
}

JavaSignal::JavaSignal()
{
}

JavaSignal::~JavaSignal()
{
}

void
JavaSignal::emitArgs(jobjectArray args)
{
	emit signalJava(args);
}


