// -*- c++ -*-

/*
 *  Copyright (C) 2003, Richard J. Moore <rich@kde.org>
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Library General Public
 *  License as published by the Free Software Foundation; either
 *  version 2 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Library General Public License for more details.
 *
 *  You should have received a copy of the GNU Library General Public License
 *  along with this library; see the file COPYING.LIB.  If not, write to
 *  the Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 *  Boston, MA 02111-1307, USA.
 */

#include <kpixmap.h>
#include <qbitmap.h>
#include <kpixmapeffect.h>
#include <kdebug.h>
#include <kjsembed/jsvalueproxy.h>
#include <kjsembed/jsbinding.h>
#include <qvariant.h>

#include "pixmap_imp.h"

namespace KJSEmbed {
namespace Bindings {

Pixmap::Pixmap( KJS::ExecState *exec, int id )
    : JSProxyImp(exec), mid(id)
{
}

Pixmap::~Pixmap()
{
}
void Pixmap::addBindings( KJS::ExecState *exec, KJS::Object &object ) {

    kdDebug() << "Pixmap::addBindings()" << endl;
    JSValueProxy *op = JSProxy::toValueProxy( object.imp() );
    if ( !op ) {
        kdWarning() << "Pixmap::addBindings() failed, not a JSValueProxy" << endl;
        return;
    }

    if ( op->typeName() != "QPixmap" ) {
	kdWarning() << "Pixmap::addBindings() failed, type is " << op->typeName() << endl;
	return;
    }

    JSProxy::MethodTable methods[] = {
	{ MethodisNull, "isNull" },
	{ Methodwidth, "width" },
	{ Methodheight, "height" },
	{ Methodsize, "size" },
	{ Methodrect, "rect" },
	{ Methoddepth, "depth" },
	{ Methodresize, "resize" },
	{ Methodfill, "fill" },
	{ Methodmask, "mask" },
	{ MethodsetMask, "setMask" },
	{ MethodcreateHeuristicMask, "createHeuristicMask" },
	{ 0, 0 }
    };

    int idx = 0;
    do {
        Pixmap *meth = new Pixmap( exec, methods[idx].id );
        object.put( exec , methods[idx].name, KJS::Object(meth) );
        ++idx;
    } while( methods[idx].id );

    //
    // Define the enum constants
    //
    struct EnumValue {
	const char *id;
	int val;
    };

    EnumValue enums[] = {
	// GradiantType
	{ "VerticalGradient", 0 },
	{ "HorizontalGradient", 1 },
	{ "DiagonalGradient", 2 },
	{ "CrossDiagonalGradient", 3 },
	{ "PyramidGradient", 4 },
	{ "RectangleGradient", 5 },
	{ "PipeCrossGradient", 6 },
	{ "EllipticGradient", 7 },
	// RGBComponent
	{ "Red", 0 },
	{ "Green", 1 },
	{ "Blue", 2 },
	{ "Gray", 3 },
	{ "All", 4 },
        // Lighting
	{ "NorthLite", 0 },
	{ "NWLite", 1 },
	{ "WestLite", 2 },
	{ "SWLite", 3 },
	{ "SouthLite", 4 },
	{ "SELite", 5 },
	{ "EastLite", 6 },
	{ "NELite", 7 },
	{ 0, 0 }
    };

    int enumidx = 0;
    do {
        object.put( exec, enums[enumidx].id, KJS::Number(enums[enumidx].val), KJS::ReadOnly );
        ++enumidx;
    } while( enums[enumidx].id );
}

KJS::Value Pixmap::call( KJS::ExecState *exec, KJS::Object &self, const KJS::List &args ) {

    kdDebug() << "Pixmap::call() " << mid << endl;
    JSValueProxy *op = JSProxy::toValueProxy( self.imp() );
    if ( !op ) {
        kdWarning() << "Pixmap::call() failed, not a JSValueProxy" << endl;
        return KJS::Value();
    }

    if ( op->typeName() != "QPixmap" ) {
	kdWarning() << "Pixmap::call() failed, type is " << op->typeName() << endl;
	return KJS::Value();
    }

    pix = op->toVariant().toPixmap();

    KJS::Value retValue = KJS::Value();
    switch ( mid ) {
    case Methodwidth:
        retValue = KJS::Number(width());
        break;
    case Methodheight:
        retValue = KJS::Number(height());
        break;
    case Methoddepth:
        retValue = KJS::Number(depth());
        break;
    case MethodisNull:
        retValue = KJS::Boolean(isNull());
        break;
    case Methodsize: 
        retValue = convertToValue(exec, size());
        break;
    case Methodrect: 
        retValue = convertToValue(exec, rect());
        break;
    case Methodresize: 
    {
        if( args.size() == 2)
		resize(extractInt(exec, args, 0), extractInt(exec, args, 1));
	else if( args.size() == 1)
	    resize(extractSize(exec, args, 0) );
	break;
    }
    case Methodfill:
        fill( extractColor(exec, args, 0));
        break;
    case Methodmask:
    {
        retValue = convertToValue(exec, mask() );
        break;
    }
    case MethodsetMask:
    {
       setMask(extractQPixmap(exec, args, 0));
       break;
    }
    case MethodcreateHeuristicMask:
    {
       retValue = convertToValue(exec, createHeuristicMask(extractBool(exec, args, 0)));
       break;
    }
    default:
        kdWarning() << "Image has no method " << mid << endl;
        break;
    }

    op->setValue(pix);
    return retValue;
}

void Pixmap::resize( int w, int h )
{
    pix.resize( w, h );
}

void Pixmap::resize( const QSize &size )
{
    pix.resize( size );
}

void Pixmap::fill( const QColor &c )
{
    pix.fill( c );
}


void Pixmap::gradient (const QColor &ca, const QColor &cb, KPixmapEffect::GradientType type, int ncols)
{
    pix = KPixmapEffect::gradient(pix, ca, cb, type, ncols);
}

void Pixmap::unbalancedGradient(const QColor &ca, const QColor &cb, KPixmapEffect::GradientType type, int xfactor, int yfactor, int /*ncols*/)
{
    pix = KPixmapEffect::unbalancedGradient(pix, ca, cb, type, xfactor, yfactor);
}

void Pixmap::createTiled ( QSize col)
{
    pix = KPixmapEffect::createTiled(pix, col);
}

void Pixmap::intensity ( float ratio)
{
    pix = KPixmapEffect::intensity(pix, ratio);
}

void Pixmap::channelIntensity ( float ratio, KPixmapEffect::RGBComponent channel)
{
    pix = KPixmapEffect::channelIntensity(pix, ratio,channel);
}

void Pixmap::blend (float initial_intensity, const QColor &bgnd, KPixmapEffect::GradientType eff, bool anti_dir, int ncols)
{
    pix = KPixmapEffect::blend(pix, initial_intensity, bgnd, eff, anti_dir, ncols);
}

void Pixmap::hash ( KPixmapEffect::Lighting lite, unsigned int spacing, int ncols)
{
    pix = KPixmapEffect::hash(pix, lite, spacing, ncols);
}

void Pixmap::pattern (QSize size, const QColor &ca, const QColor &cb, int ncols)
{
    pix = KPixmapEffect::pattern(pix, size, ca, cb, ncols);
}

void Pixmap::fade ( double val, const QColor &color)
{
    pix = KPixmapEffect::fade(pix, val, color);
}

void Pixmap::toGray ( bool fast)
{
    pix = KPixmapEffect::toGray(pix, fast);
}

void Pixmap::desaturate ( float desat)
{
    pix = KPixmapEffect::desaturate(pix, desat);
}

void Pixmap::contrast ( int c)
{
    pix = KPixmapEffect::contrast( pix, c);
}
void Pixmap::dither ( const QColor *palette, int size)
{
    pix = KPixmapEffect::dither(pix, palette, size);
}
void Pixmap::selectedPixmap ( const QColor &col)
{
    pix = KPixmapEffect::selectedPixmap(pix, col);
}

void Pixmap::grabWindow(int winID, int x, int y, int w, int h)
{
	pix = QPixmap::grabWindow(winID, x, y, w, h);
}

QPixmap Pixmap::mask()
{
	return *(pix.mask());
}
void Pixmap::setMask(const QPixmap& mask)
{
	QBitmap bm;
	bm = mask;
	pix.setMask(bm);
}

QPixmap Pixmap::createHeuristicMask ( bool clipTight )
{
	return (QPixmap)pix.createHeuristicMask(clipTight);
}

} // namespace KJSEmbed::Bindings
} // namespace KJSEmbed

