import QtQuick
import QtQuick.Layouts
import QtQuick.Controls.Material
import QtQuick.Dialogs

import Main

StackView {
    id: stackView
    Layout.fillWidth: true
    Layout.fillHeight: true
    initialItem: Page {
        id: appSettingsPage
        title: qsTr("App settings")
        Layout.fillWidth: true
        Layout.fillHeight: true

        CustomListView {
            id: listView
            anchors.fill: parent
            model: ListModel {
                id: model
                ListElement {
                    key: "connection"
                    label: qsTr("Connection to Syncthing backend")
                    title: qsTr("Configure connection with Syncthing backend")
                    iconName: "link"
                }
                ListElement {
                    key: "launcher"
                    label: qsTr("Run conditions of Syncthing backend")
                    title: qsTr("Configure when to run Syncthing backend")
                    iconName: "terminal"
                }
                ListElement {
                    callback: () => App.clearLogfile()
                    label: qsTr("Clear log file")
                    title: qsTr("Disables persistent logging and removes the log file")
                    iconName: "trash-o"
                }
                ListElement {
                    key: "tweaks"
                    label: qsTr("Tweaks")
                    title: qsTr("Configure details of the app's behavior")
                    iconName: "cogs"
                }
                ListElement {
                    callback: () => stackView.push("ErrorsPage.qml", {}, StackView.PushTransition)
                    label: qsTr("Syncthing notifications/errors")
                    iconName: "exclamation-triangle"
                }
                ListElement {
                    callback: () => stackView.push("InternalErrorsPage.qml", {}, StackView.PushTransition)
                    label: qsTr("Log of Syncthing API errors")
                    iconName: "exclamation-circle"
                }
                ListElement {
                    callback: () => stackView.push("StatisticsPage.qml", {stackView: stackView}, StackView.PushTransition)
                    label: qsTr("Statistics")
                    iconName: "area-chart"
                }
                ListElement {
                    functionName: "checkSettings"
                    callback: (availableSettings) => stackView.push("ImportPage.qml", {availableSettings: availableSettings}, StackView.PushTransition)
                    label: qsTr("Import selected settings/secrets/data of app and backend")
                    iconName: "download"
                }
                ListElement {
                    functionName: "exportSettings"
                    label: qsTr("Export all settings/secrets/data of app and backend")
                    iconName: "floppy-o"
                }
                ListElement {
                    callback: () => stackView.push("HomeDirPage.qml", {}, StackView.PushTransition)
                    label: qsTr("Move Syncthing home directory")
                    iconName: "folder-open-o"
                }
            }
            delegate: ItemDelegate {
                width: listView.width
                text: label
                icon.source: App.faUrlBase + iconName // leads to crash when closing UI with Qt < 6.8
                icon.width: App.iconSize
                icon.height: App.iconSize
                onClicked: {
                    if (key.length > 0) {
                        appSettingsPage.openNestedSettings(title, key);
                    } else if (functionName.length > 0) {
                        appSettingsPage.initiateBackup(functionName, callback);
                    } else if (callback !== undefined) {
                        callback();
                    }
                }

            }
        }

        FolderDialog {
            id: backupFolderDialog
            onAccepted: App[appSettingsPage.currentBackupFunction](backupFolderDialog.selectedFolder, appSettingsPage.currentBackupCallback)
        }

        property string currentBackupFunction
        property var currentBackupCallback
        function initiateBackup(functionName, callback) {
            appSettingsPage.currentBackupFunction = functionName;
            appSettingsPage.currentBackupCallback = callback;
            backupFolderDialog.open();
        }
        function openNestedSettings(title, key) {
            if (appSettingsPage.config[key] === undefined) {
                appSettingsPage.config[key] = {};
            }
            stackView.push("ObjectConfigPage.qml", {
                               title: title,
                               parentPage: appSettingsPage,
                               configObject: Qt.binding(() => appSettingsPage.config[key]),
                               specialEntries: appSettingsPage.specialEntries[key] ?? [],
                               specialEntriesByKey: appSettingsPage.specialEntries,
                               specialEntriesOnly: true,
                               stackView: stackView,
                               actions: appSettingsPage.actions},
                           StackView.PushTransition)
        }

        property var config: App.settings
        readonly property var specialEntries: ({
            connection: [
                {key: "useLauncher", type: "boolean", label: qsTr("Automatic"), statusText: qsTr("Connect to the Syncthing backend launched via this app and disregard the settings below.")},
                {key: "syncthingUrl", label: qsTr("Syncthing URL")},
                {key: "apiKey", label: qsTr("API key")},
                {key: "httpsCertPath", label: qsTr("HTTPs certificate path"), type: "filepath"},
                {key: "httpAuth", label: qsTr("HTTP authentication")},
            ],
            httpAuth: [
                {key: "enabled", label: qsTr("Enabled")},
                {key: "userName", label: qsTr("Username")},
                {key: "password", label: qsTr("Password")},
            ],
            launcher: [
                {key: "run", label: qsTr("Run Syncthing"), statusText: Qt.binding(() => App.launcher.runningStatus)},
                {key: "logLevel", label: qsTr("Log level"), type: "options", options: [
                    {value: "debug", label: qsTr("Debug")},
                    {value: "verbose", label: qsTr("Verbose")},
                    {value: "info", label: qsTr("Info")},
                    {value: "warning", label: qsTr("Warning")},
                    {value: "fatal", label: qsTr("Fatal")},
                ]},
                {key: "stopOnMetered", label: qsTr("Stop on metered network connection"), statusText: Qt.binding(() => App.launcher.meteredStatus)},
                {key: "writeLogFile", label: qsTr("Write persistent log file"), statusText: qsTr("Write a persistent log file into the app directory.")},
                {key: "openLogs", label: qsTr("Open logs"), statusText: qsTr("Shows Syncthing logs since app startup"), defaultValue: () => stackView.push("LogPage.qml", {}, StackView.PushTransition)},
            ],
            tweaks: [
                {key: "unloadGuiWhenHidden", type: "boolean", defaultValue: false, label: qsTr("Stop UI when hidden"), statusText: qsTr("Might help save battery live but resets UI state.")},
            ]
        })
        property bool hasUnsavedChanges: false
        property list<Action> actions: [
            Action {
                text: qsTr("Apply")
                icon.source: App.faUrlBase + "check"
                onTriggered: {
                    const cfg = App.settings;
                    for (let i = 0, count = model.count; i !== count; ++i) {
                        const entryKey = model.get(i).key;
                        if (entryKey.length > 0) {
                            cfg[entryKey] = appSettingsPage.config[entryKey];
                        }
                    }
                    App.settings = cfg;
                    return true;
                }
            }
        ]
    }
}
