use core:lang;
use core:asm;
use lang:asm;
use lang:bs:macro;

class CppDtorDecl extends CppDecl {
	FnBody? body;

	init(SStr name, FnBody? body) {
		init(name) {
			body = body;
		}

		if (body)
			body.stmt.pos.end++;
	}

	void create(NameSet inside) : override {
		unless (body)
			return;

		if (name.v != inside.name)
			throw SyntaxError(name.pos, "The destructor should be named ~${inside.name}.");

		CppDtor fn(cppScope(inside), name.pos, inside, body);
		if (visibility)
			fn.visibility = visibility;
		inside.add(fn);

		inside.add(wrapCppDtor(fn));
	}
}

// Find the C++ dtor for a type.
Function cppDtor(Type type) on Compiler {
	if (fn = type.find(SimplePart("__cpp_destroy", [thisPtr(type)]), cppScope(type)) as Function) {
		fn;
	} else {
		named{cppGenericDtor<unsafe:RawPtr>};
	}
}

// Generic destructor for C++ types.
void cppGenericDtor(unsafe:RawPtr obj) {
	// print("Destroying an object ${obj}");
	Nat filled = obj.readFilled;
	filled &= ~AllocFlags:sizeMask.v;
	obj.writeFilled(filled);
}

// Wrap a previously created C++ destructor to make it suitable to call from our array allocations.
// Supports arrays as well!
Function wrapCppDtor(Function dtor) on Compiler {
	Listing l;
	Var param = l.createParam(ptrDesc);
	Var current = l.createVar(l.root, sPtr);
	Var target = l.createVar(l.root, sPtr);

	Label again = l.label();
	Label start = l.label();

	// Size of each element.
	Nat elemSize = dtor.params[0].asRef(false).size.current;

	l << prolog();

	// Initialize the loop.
	l << mov(ptrA, param);
	l << mov(current, ptrConst(0));
	l << mov(target, ptrRel(ptrA, Offset(sPtr)));
	l << and(target, ptrConst(AllocFlags:sizeMask.v));

	l << jmp(start);

	l << again;
	// Compute offset.
	l << mov(ptrB, current);
	l << mul(ptrB, ptrConst(elemSize));
	l << add(ptrB, ptrConst(sPtr * 2));

	// Call the original destructor.
	l << mov(ptrA, param);
	l << add(ptrA, ptrB);
	l << ucast(ecx, ptrB);
	l << fnParam(ptrDesc, ptrA);
	l << fnParam(intDesc, ecx);
	l << fnCall(dtor.ref, true);

	// Next element.
	l << add(current, ptrConst(1));

	// Try again.
	l << start;
	l << cmp(current, target);
	l << jmp(again, CondFlag:ifBelow);

	// Mark it as 'empty' by clearing the size (and keeping other flags).
	l << mov(ptrA, param);
	l << and(ptrRel(ptrA, Offset(sPtr)), ptrConst(~AllocFlags:sizeMask.v));
	l << fnRet();

	Function f(Value(), "__cpp_destroy", [dtor.params[0]]);
	f.setCode(DynamicCode(l));
	f;
}

class CppDtor extends CppFunction {
	private FnBody? body;

	init(Scope scope, SrcPos pos, NameSet inside, FnBody? body) {
		Value[] fTypes;
		SStr[] fNames;

		if (inside as Type) {
			fTypes << Value(thisPtr(inside));
			fNames << SStr("this", pos);

			// We also need a hidden 'offset' ptr, that allows us to reconstruct a Ptr<> object.
			fTypes << Value(named{Nat});
			fNames << SStr("offset", pos);
		}

		init(scope, pos, Value(), "__destroy", fTypes) {
			body = body;
		}

		setCode(LazyCode(&this.code));
	}

	// Generate code on demand.
	private CodeGen code() {
		FnRoot root = if (body) {
			FnRoot fn(body.stmt.pos, this);
			body.stmt.transform(fn);
			fn;
		} else {
			FnRoot(pos, this);
		};

		CodeGen gen(runOn, isMember, result);
		gen.l << prolog();

		if (!root.code(gen))
			throw SyntaxError(pos, "This implementation does not support using 'return' in the destructor!");

		// Destroy all member variables.
		if (type = params[0].type) {
			for (var in type.variables) {
				if (varType = var.type.type) {
					destroyVar(gen, varType, var.offset, root);
				}
			}

			if (s = type.super) {
				// Destroy any super-classes.
				destroyVar(gen, s, Offset(), root);
			}
		}

		gen.l << fnRet();

		// print("Generated code for ${params[0]}:\n${gen.l}");

		gen;
	}

	// Destroy a variable.
	private void destroyVar(CodeGen gen, Type type, Offset offset, FnRoot root) {
		// C++ dtor.
		if (dtor = type.find(SimplePart("__destroy", [thisPtr(type), Value(named{Nat})]), scope) as Function) {
			gen.l << mov(ptrA, root.thisPtr);
			gen.l << add(ptrA, ptrConst(offset));
			gen.l << mov(ecx, root.thisOffset);
			gen.l << add(ecx, intConst(offset));
			gen.l << fnParam(ptrDesc, ptrA);
			gen.l << fnParam(intDesc, ecx);
			gen.l << fnCall(dtor.ref, true);
			return;
		}

		// Regular dtor.
		if (dtor = type.find(SimplePart("__destroy", [thisPtr(type)]), scope) as Function) {
			gen.l << mov(ptrA, root.thisPtr);
			gen.l << add(ptrA, ptrConst(offset));
			gen.l << fnParam(ptrDesc, ptrA);
			gen.l << fnCall(dtor.ref, true);
			return;
		}
	}
}
