import json

import pytest

import stix2.exceptions
import stix2.utils
import stix2.v21

MALWARE_ANALYSIS_JSON = """{
    "type": "malware-analysis",
    "spec_version": "2.1",
    "id": "malware-analysis--f8afc020-f92f-4906-a971-88ee5882eb46",
    "created_by_ref": "identity--e0353ed3-991e-4f71-a332-114c2f10b84f",
    "created": "2017-11-28T09:44:58.418Z",
    "modified": "2017-12-31T21:27:49.754Z",
    "product": "Acme Malware Analyzer",
    "version": "2.5",
    "host_vm_ref": "software--1bda7336-fe67-469f-a8ca-ab6268b0449b",
    "operating_system_ref": "software--c96bfaef-861b-408b-b0f1-b685881725ef",
    "installed_software_refs": [
        "software--7325bf2d-de9e-441e-b3b3-63df43149897",
        "software--46a6a91d-1160-4867-a4d1-b14e080e4e5b"
    ],
    "configuration_version": "1.7",
    "modules": [
        "Super Analyzer"
    ],
    "analysis_engine_version": "1.2",
    "analysis_definition_version": "3.4",
    "submitted": "2018-11-23T06:45:55.747Z",
    "analysis_started": "2018-11-29T07:30:03.895Z",
    "analysis_ended": "2018-11-29T08:30:03.895Z",
    "result_name": "MegaRansom",
    "result": "malicious",
    "analysis_sco_refs": [
        "file--fc27e371-6c88-4c5c-868a-4dda0e60b167",
        "url--6f7a74cd-8eb2-4b88-a4da-aa878e50ac2e"
    ],
    "sample_ref": "email-addr--499a32d7-74c1-4276-ace9-725ac933e243",
    "labels": [
        "label1",
        "label2"
    ]
}"""


MALWARE_ANALYSIS_DICT = json.loads(MALWARE_ANALYSIS_JSON)


def test_malware_analysis_example():
    malware_analysis = stix2.v21.MalwareAnalysis(**MALWARE_ANALYSIS_DICT)

    assert malware_analysis.serialize(pretty=True) == MALWARE_ANALYSIS_JSON


@pytest.mark.parametrize(
    "data", [
        MALWARE_ANALYSIS_JSON,
        MALWARE_ANALYSIS_DICT,
    ],
)
def test_parse_malware_analysis(data):
    ma = stix2.parse(data, version="2.1")

    # timestamp-valued attributes whose values (from JSON) can't be compared
    # directly, since stix2 internally converts them to datetime objects.
    ts_attrs = {
        "created",
        "modified",
        "submitted",
        "analysis_started",
        "analysis_ended",
    }

    for attr_name, attr_value in MALWARE_ANALYSIS_DICT.items():
        cmp_value = stix2.utils.parse_into_datetime(attr_value) \
            if attr_name in ts_attrs else attr_value

        assert getattr(ma, attr_name) == cmp_value


def test_malware_analysis_constraint():
    with pytest.raises(stix2.exceptions.AtLeastOnePropertyError):
        stix2.v21.MalwareAnalysis(
            product="Acme Malware Analyzer",
        )


def test_malware_analysis_custom_sco_refs():
    ma = stix2.v21.MalwareAnalysis(
        product="super scanner",
        analysis_sco_refs=[
            "file--6e8c78cf-4bcc-4729-9265-86a97bfc91ba",
            "some-object--f6bfc147-e844-4578-ae01-847979890239",
        ],
        allow_custom=True,
    )

    assert "some-object--f6bfc147-e844-4578-ae01-847979890239" in \
        ma["analysis_sco_refs"]
    assert ma.has_custom

    with pytest.raises(stix2.exceptions.InvalidValueError):
        stix2.v21.MalwareAnalysis(
            product="super scanner",
            analysis_sco_refs=[
                "file--6e8c78cf-4bcc-4729-9265-86a97bfc91ba",
                "some-object--f6bfc147-e844-4578-ae01-847979890239",
            ],
        )

    with pytest.raises(stix2.exceptions.InvalidValueError):
        stix2.v21.MalwareAnalysis(
            product="super scanner",
            analysis_sco_refs=[
                "file--6e8c78cf-4bcc-4729-9265-86a97bfc91ba",
                # standard object type; wrong category (not SCO)
                "identity--56977a19-49ef-49d7-b259-f733fa4b7bbc",
            ],
            allow_custom=True,
        )
