/*
 * Decompiled with CFR 0.152.
 */
package org.apache.karaf.kar.internal;

import java.io.BufferedOutputStream;
import java.io.BufferedReader;
import java.io.ByteArrayInputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.FileReader;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.io.PrintStream;
import java.net.URI;
import java.net.URISyntaxException;
import java.nio.charset.StandardCharsets;
import java.util.ArrayList;
import java.util.Collections;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.HashSet;
import java.util.Iterator;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.jar.JarEntry;
import java.util.jar.JarOutputStream;
import java.util.jar.Manifest;
import org.apache.karaf.features.BundleInfo;
import org.apache.karaf.features.Conditional;
import org.apache.karaf.features.ConfigFileInfo;
import org.apache.karaf.features.Dependency;
import org.apache.karaf.features.Feature;
import org.apache.karaf.features.FeaturesService;
import org.apache.karaf.features.Repository;
import org.apache.karaf.kar.KarService;
import org.apache.karaf.kar.internal.Kar;
import org.apache.karaf.util.StreamUtils;
import org.apache.karaf.util.maven.Parser;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class KarServiceImpl
implements KarService {
    private static final String FEATURE_CONFIG_FILE = "features.cfg";
    private static final Logger LOGGER = LoggerFactory.getLogger(KarServiceImpl.class);
    private File storage;
    private File base;
    private FeaturesService featuresService;
    private boolean noAutoRefreshBundles;
    private boolean noAutoStartBundles;
    private List<Kar> unsatisfiedKars;
    private AtomicBoolean busy;
    private DelayedDeployerThread delayedDeployerThread;

    public KarServiceImpl(String karafBase, String karStorage, FeaturesService featuresService) {
        this.base = new File(karafBase);
        this.storage = new File(karStorage);
        this.featuresService = featuresService;
        this.storage.mkdirs();
        if (!this.storage.isDirectory()) {
            throw new IllegalStateException("KAR storage " + this.storage + " is not a directory");
        }
        this.unsatisfiedKars = Collections.synchronizedList(new ArrayList());
        this.busy = new AtomicBoolean();
    }

    @Override
    public void install(URI karUri) throws Exception {
        this.install(karUri, false);
    }

    @Override
    public void install(URI karUri, boolean noAutoStartBundles) throws Exception {
        this.install(karUri, noAutoStartBundles, false);
    }

    @Override
    public void install(URI karUri, boolean noAutoStartBundles, boolean noAutoRefreshBundles) throws Exception {
        String karName = new Kar(karUri).getKarName();
        LOGGER.debug("Installing KAR {} from {}", (Object)karName, (Object)karUri);
        File karDir = new File(this.storage, karName);
        this.install(karUri, karDir, this.base, noAutoStartBundles, noAutoRefreshBundles);
    }

    @Override
    public void install(URI karUri, File repoDir, File resourceDir) throws Exception {
        this.install(karUri, repoDir, resourceDir, false);
    }

    @Override
    public void install(URI karUri, File repoDir, File resourceDir, boolean noAutoStartBundles) throws Exception {
        this.install(karUri, repoDir, resourceDir, noAutoStartBundles, false);
    }

    @Override
    public void install(URI karUri, File repoDir, File resourceDir, boolean noAutoStartBundles, boolean noAutoRefreshBundles) throws Exception {
        this.busy.set(true);
        Kar kar = new Kar(karUri);
        try {
            kar.extract(repoDir, resourceDir);
            this.writeToFile(kar.getFeatureRepos(), new File(repoDir, FEATURE_CONFIG_FILE));
            for (URI uri : kar.getFeatureRepos()) {
                this.addToFeaturesRepositories(uri);
            }
            if (kar.isShouldInstallFeatures()) {
                List<URI> featureRepos = kar.getFeatureRepos();
                Dependency missingDependency = this.findMissingDependency(featureRepos);
                if (missingDependency == null) {
                    this.installFeatures(featureRepos, noAutoStartBundles, noAutoRefreshBundles);
                } else {
                    LOGGER.warn("Feature dependency {} is not available. Kar deployment postponed to see if it is about to be deployed", (Object)missingDependency);
                    this.unsatisfiedKars.add(kar);
                    if (this.delayedDeployerThread == null) {
                        this.delayedDeployerThread = new DelayedDeployerThread(noAutoStartBundles, noAutoRefreshBundles);
                        this.delayedDeployerThread.start();
                    }
                }
            }
            if (!this.unsatisfiedKars.isEmpty()) {
                Iterator<Kar> iterator = this.unsatisfiedKars.iterator();
                while (iterator.hasNext()) {
                    Kar delayedKar = iterator.next();
                    if (this.findMissingDependency(delayedKar.getFeatureRepos()) != null) continue;
                    LOGGER.info("Dependencies of kar {} are now satisfied. Installing", (Object)delayedKar.getKarName());
                    iterator.remove();
                    this.installFeatures(delayedKar.getFeatureRepos(), noAutoStartBundles, noAutoRefreshBundles);
                }
            }
            if (this.unsatisfiedKars.isEmpty()) {
                if (this.delayedDeployerThread != null) {
                    this.delayedDeployerThread.cancel();
                }
                this.delayedDeployerThread = null;
            }
        }
        catch (Exception e) {
            this.deleteRecursively(new File(this.storage, kar.getKarName()));
            throw e;
        }
        finally {
            this.busy.set(false);
        }
    }

    private Dependency findMissingDependency(List<URI> featureRepos) throws Exception {
        for (URI uri : featureRepos) {
            Feature[] includedFeatures;
            for (Feature includedFeature : includedFeatures = this.featuresService.getRepository(uri).getFeatures()) {
                List dependencies = includedFeature.getDependencies();
                for (Dependency dependency : dependencies) {
                    Feature feature = this.featuresService.getFeature(dependency.getName(), dependency.getVersion());
                    if (feature != null) continue;
                    return dependency;
                }
            }
        }
        return null;
    }

    private List<URI> readFromFile(File repoListFile) {
        ArrayList<URI> uriList = new ArrayList<URI>();
        FileReader fr = null;
        try {
            String line;
            fr = new FileReader(repoListFile);
            BufferedReader br = new BufferedReader(fr);
            while ((line = br.readLine()) != null) {
                uriList.add(new URI(line));
            }
        }
        catch (Exception e) {
            throw new RuntimeException("Error reading repo list from file " + repoListFile.getAbsolutePath(), e);
        }
        finally {
            try {
                fr.close();
            }
            catch (IOException e) {
                LOGGER.warn("Error closing reader for file " + repoListFile, (Throwable)e);
            }
        }
        return uriList;
    }

    private void writeToFile(List<URI> featuresRepositoriesInKar, File repoListFile) {
        FileOutputStream fos = null;
        PrintStream ps = null;
        try {
            fos = new FileOutputStream(repoListFile);
            ps = new PrintStream(fos);
            for (URI uri : featuresRepositoriesInKar) {
                ps.println(uri);
            }
            ps.close();
            fos.close();
            this.closeStream(ps);
            this.closeStream(fos);
        }
        catch (Exception e) {
            try {
                throw new RuntimeException("Error writing feature repo list to file " + repoListFile.getAbsolutePath(), e);
            }
            catch (Throwable throwable) {
                this.closeStream(ps);
                this.closeStream(fos);
                throw throwable;
            }
        }
    }

    private void deleteRecursively(File dir) {
        if (dir.isDirectory()) {
            File[] children;
            for (File child : children = dir.listFiles()) {
                this.deleteRecursively(child);
            }
        }
        dir.delete();
    }

    @Override
    public void uninstall(String karName) throws Exception {
        this.uninstall(karName, false);
    }

    @Override
    public void uninstall(String karName, boolean noAutoRefreshBundles) throws Exception {
        File karDir = new File(this.storage, karName);
        if (!karDir.exists()) {
            throw new IllegalArgumentException("The KAR " + karName + " is not installed");
        }
        List<URI> featuresRepositories = this.readFromFile(new File(karDir, FEATURE_CONFIG_FILE));
        this.uninstallFeatures(featuresRepositories, noAutoRefreshBundles);
        for (URI featuresRepository : featuresRepositories) {
            this.featuresService.removeRepository(featuresRepository);
        }
        this.deleteRecursively(karDir);
    }

    @Override
    public List<String> list() throws Exception {
        ArrayList<String> kars = new ArrayList<String>();
        for (File kar : this.storage.listFiles()) {
            if (!kar.isDirectory()) continue;
            kars.add(kar.getName());
        }
        return kars;
    }

    private void addToFeaturesRepositories(URI uri) throws Exception {
        try {
            this.featuresService.removeRepository(uri);
            this.featuresService.addRepository(uri, false);
            LOGGER.info("Added feature repository '{}'", (Object)uri);
        }
        catch (Exception e) {
            LOGGER.warn("Unable to add repository '{}'", (Object)uri, (Object)e);
        }
    }

    private void installFeatures(List<URI> featuresRepositories, boolean noAutoStartBundles, boolean noAutoRefreshBundles) throws Exception {
        for (Repository repository : this.featuresService.listRepositories()) {
            for (URI karFeatureRepoUri : featuresRepositories) {
                if (!repository.getURI().equals(karFeatureRepoUri)) continue;
                try {
                    for (Feature feature : repository.getFeatures()) {
                        if (feature.getInstall() == null || "auto".equals(feature.getInstall())) {
                            EnumSet<FeaturesService.Option> options = EnumSet.noneOf(FeaturesService.Option.class);
                            try {
                                LOGGER.debug("noAutoRefreshBundles is {} (default {})", (Object)noAutoRefreshBundles, (Object)this.noAutoRefreshBundles);
                                if (noAutoRefreshBundles || this.noAutoRefreshBundles) {
                                    options.add(FeaturesService.Option.NoAutoRefreshBundles);
                                }
                                LOGGER.debug("noAutoStartBundles is {} (default {})", (Object)noAutoStartBundles, (Object)this.noAutoStartBundles);
                                if (noAutoStartBundles || this.noAutoStartBundles) {
                                    options.add(FeaturesService.Option.NoAutoStartBundles);
                                }
                                this.featuresService.installFeature(feature, options);
                            }
                            catch (Exception e) {
                                LOGGER.warn("Unable to install Kar feature {}", (Object)(feature.getName() + "/" + feature.getVersion()), (Object)e);
                            }
                            continue;
                        }
                        LOGGER.warn("Feature " + feature.getName() + "/" + feature.getVersion() + " has install flag set to \"manual\", so it's not automatically installed");
                    }
                }
                catch (Exception e) {
                    LOGGER.warn("Can't get features for KAR {}", (Object)karFeatureRepoUri, (Object)e);
                }
            }
        }
    }

    @Override
    public void create(String repoName, List<String> features, PrintStream console) {
        FileOutputStream fos = null;
        JarOutputStream jos = null;
        try {
            Repository repo = this.featuresService.getRepository(repoName);
            if (repo == null) {
                throw new RuntimeException("Could not find a repository with name " + repoName);
            }
            String karPath = this.storage + File.separator + repoName + ".kar";
            File karFile = new File(karPath);
            karFile.getParentFile().mkdirs();
            fos = new FileOutputStream(karFile);
            Manifest manifest = this.createNonAutoStartManifest(repo.getURI());
            jos = new JarOutputStream((OutputStream)new BufferedOutputStream(fos, 100000), manifest);
            HashMap<URI, Integer> locationMap = new HashMap<URI, Integer>();
            this.copyResourceToJar(jos, repo.getURI(), locationMap);
            HashMap<String, Feature> featureMap = new HashMap<String, Feature>();
            for (Feature feature : repo.getFeatures()) {
                featureMap.put(feature.getName(), feature);
            }
            Set<Feature> featuresToCopy = this.getFeatures(featureMap, features, 1);
            for (Feature feature : featuresToCopy) {
                if (console != null) {
                    console.println("Adding feature " + feature.getName());
                }
                this.copyFeatureToJar(jos, feature, locationMap);
            }
            if (console != null) {
                console.println("Kar file created : " + karPath);
            }
            this.closeStream(jos);
            this.closeStream(fos);
        }
        catch (Exception e) {
            try {
                throw new RuntimeException("Error creating kar: " + e.getMessage(), e);
            }
            catch (Throwable throwable) {
                this.closeStream(jos);
                this.closeStream(fos);
                throw throwable;
            }
        }
    }

    private Set<Feature> getFeatures(Map<String, Feature> featureMap, List<String> features, int depth) {
        HashSet<Feature> featureSet = new HashSet<Feature>();
        if (depth > 5) {
            return featureSet;
        }
        if (features == null) {
            featureSet.addAll(featureMap.values());
            return featureSet;
        }
        for (String featureName : features) {
            Feature feature = featureMap.get(featureName);
            if (feature == null) {
                System.out.println("Feature " + featureName + " not found in repository.");
                continue;
            }
            featureSet.add(feature);
            List deps = feature.getDependencies();
            ArrayList<String> depNames = new ArrayList<String>();
            for (Dependency dependency : deps) {
                depNames.add(dependency.getName());
            }
            featureSet.addAll(this.getFeatures(featureMap, depNames, depth++));
        }
        return featureSet;
    }

    private Manifest createNonAutoStartManifest(URI repoUri) throws IOException {
        String manifestSt = "Manifest-Version: 1.0\nKaraf-Feature-Start: false\nKaraf-Feature-Repos: " + repoUri.toString() + "\n";
        ByteArrayInputStream manifestIs = new ByteArrayInputStream(manifestSt.getBytes(StandardCharsets.UTF_8));
        Manifest manifest = new Manifest(manifestIs);
        return manifest;
    }

    private void closeStream(OutputStream os) {
        if (os != null) {
            try {
                os.close();
            }
            catch (IOException e) {
                LOGGER.warn("Error closing stream", (Throwable)e);
            }
        }
    }

    private void copyFeatureToJar(JarOutputStream jos, Feature feature, Map<URI, Integer> locationMap) throws URISyntaxException {
        URI location;
        for (BundleInfo bundleInfo : feature.getBundles()) {
            location = new URI(bundleInfo.getLocation().trim());
            this.copyResourceToJar(jos, location, locationMap);
        }
        for (ConfigFileInfo configFileInfo : feature.getConfigurationFiles()) {
            location = new URI(configFileInfo.getLocation().trim());
            this.copyResourceToJar(jos, location, locationMap);
        }
        for (Conditional conditional : feature.getConditional()) {
            URI location2;
            for (BundleInfo bundleInfo : conditional.getBundles()) {
                location2 = new URI(bundleInfo.getLocation().trim());
                this.copyResourceToJar(jos, location2, locationMap);
            }
            for (ConfigFileInfo configFileInfo : conditional.getConfigurationFiles()) {
                location2 = new URI(configFileInfo.getLocation().trim());
                this.copyResourceToJar(jos, location2, locationMap);
            }
        }
    }

    private void copyResourceToJar(JarOutputStream jos, URI location, Map<URI, Integer> locationMap) {
        if (locationMap.containsKey(location)) {
            return;
        }
        try {
            String noPrefixLocation = location.toString().substring(location.toString().lastIndexOf(":") + 1);
            Parser parser = new Parser(noPrefixLocation);
            String path = "repository/" + parser.getArtifactPath();
            jos.putNextEntry(new JarEntry(path));
            try (InputStream is = location.toURL().openStream();){
                StreamUtils.copy(is, jos);
            }
            locationMap.put(location, 1);
        }
        catch (Exception e) {
            LOGGER.error("Error adding " + location, (Throwable)e);
        }
    }

    private void uninstallFeatures(List<URI> featuresRepositories, boolean noAutoRefreshBundles) throws Exception {
        for (Repository repository : this.featuresService.listRepositories()) {
            for (URI karFeatureRepoUri : featuresRepositories) {
                if (!repository.getURI().equals(karFeatureRepoUri)) continue;
                try {
                    for (Feature feature : repository.getFeatures()) {
                        if (feature.getInstall() != null && !"auto".equals(feature.getInstall())) continue;
                        EnumSet<FeaturesService.Option> options = EnumSet.noneOf(FeaturesService.Option.class);
                        try {
                            LOGGER.debug("noAutoRefreshBundles is {} (default {})", (Object)noAutoRefreshBundles, (Object)this.noAutoRefreshBundles);
                            if (noAutoRefreshBundles || this.noAutoRefreshBundles) {
                                options.add(FeaturesService.Option.NoAutoRefreshBundles);
                            }
                            this.featuresService.uninstallFeature(feature.getName(), feature.getVersion(), options);
                        }
                        catch (Exception e) {
                            LOGGER.warn("Unable to uninstall Kar feature {}", (Object)(feature.getName() + "/" + feature.getVersion()), (Object)e);
                        }
                    }
                }
                catch (Exception e) {
                    LOGGER.warn("Can't get features for KAR {}", (Object)karFeatureRepoUri, (Object)e);
                }
            }
        }
    }

    public void setNoAutoRefreshBundles(boolean noAutoRefreshBundles) {
        this.noAutoRefreshBundles = noAutoRefreshBundles;
    }

    public void setNoAutoStartBundles(boolean noAutoStartBundles) {
        this.noAutoStartBundles = noAutoStartBundles;
    }

    private class DelayedDeployerThread
    extends Thread {
        private boolean noAutoStartBundles;
        private boolean noAutoRefreshBundles;
        private AtomicBoolean cancel;

        public DelayedDeployerThread(boolean noAutoStartBundles, boolean noAutoRefreshBundles) {
            super("Delayed kar deployment");
            this.cancel = new AtomicBoolean();
            this.noAutoStartBundles = noAutoStartBundles;
            this.noAutoRefreshBundles = noAutoRefreshBundles;
        }

        public void cancel() {
            this.cancel.set(true);
        }

        @Override
        public void run() {
            try {
                while (KarServiceImpl.this.busy.get() && !this.cancel.get()) {
                    Thread.sleep(TimeUnit.SECONDS.toMillis(2L));
                }
            }
            catch (InterruptedException interruptedException) {
                // empty catch block
            }
            if (!this.cancel.get()) {
                this.installDelayedKars();
            }
        }

        private void installDelayedKars() {
            Iterator<Kar> iterator = KarServiceImpl.this.unsatisfiedKars.iterator();
            while (iterator.hasNext()) {
                Kar kar = iterator.next();
                iterator.remove();
                try {
                    KarServiceImpl.this.installFeatures(kar.getFeatureRepos(), this.noAutoStartBundles, this.noAutoRefreshBundles);
                }
                catch (Exception e) {
                    LOGGER.error("Delayed deployment of kar " + kar.getKarName() + " failed", (Throwable)e);
                }
            }
        }
    }
}

