"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.registerTraceAnalyticsDslRouter = registerTraceAnalyticsDslRouter;
var _configSchema = require("@osd/config-schema");
var _trace_analytics = require("../../common/constants/trace_analytics");
var _metrics_helper = require("../common/metrics/metrics_helper");
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

function registerTraceAnalyticsDslRouter(router, dataSourceEnabled) {
  router.post({
    path: _trace_analytics.TRACE_ANALYTICS_DATA_PREPPER_INDICES_ROUTE,
    validate: {
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    const params = {
      index: [_trace_analytics.DATA_PREPPER_INDEX_NAME, _trace_analytics.DATA_PREPPER_SERVICE_INDEX_NAME],
      allow_no_indices: false
    };
    const {
      dataSourceMDSId
    } = request.query;
    try {
      let resp;
      if (dataSourceEnabled && dataSourceMDSId) {
        const client = context.dataSource.opensearch.legacy.getClient(dataSourceMDSId);
        resp = await client.callAPI('indices.exists', params);
      } else {
        resp = await context.core.opensearch.legacy.client.callAsCurrentUser('indices.exists', params);
      }
      return response.ok({
        body: resp
      });
    } catch (error) {
      console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: _trace_analytics.TRACE_ANALYTICS_JAEGER_INDICES_ROUTE,
    validate: {
      body: _configSchema.schema.any(),
      query: _configSchema.schema.object({
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    const {
      dataSourceMDSId
    } = request.query;
    const params = {
      index: [_trace_analytics.JAEGER_INDEX_NAME, _trace_analytics.JAEGER_SERVICE_INDEX_NAME],
      allow_no_indices: false
    };
    try {
      let resp;
      if (dataSourceEnabled && dataSourceMDSId) {
        const client = context.dataSource.opensearch.legacy.getClient(dataSourceMDSId);
        resp = await client.callAPI('indices.exists', params);
      } else {
        resp = await context.core.opensearch.legacy.client.callAsCurrentUser('indices.exists', params);
      }
      return response.ok({
        body: resp
      });
    } catch (error) {
      console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
  router.post({
    path: _trace_analytics.TRACE_ANALYTICS_DSL_ROUTE,
    validate: {
      body: _configSchema.schema.object({
        index: _configSchema.schema.maybe(_configSchema.schema.string()),
        from: _configSchema.schema.maybe(_configSchema.schema.number()),
        size: _configSchema.schema.number(),
        query: _configSchema.schema.maybe(_configSchema.schema.object({
          bool: _configSchema.schema.object({
            filter: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({}, {
              unknowns: 'allow'
            }))),
            must: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({}, {
              unknowns: 'allow'
            }))),
            should: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({}, {
              unknowns: 'allow'
            }))),
            must_not: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.object({}, {
              unknowns: 'allow'
            }))),
            minimum_should_match: _configSchema.schema.maybe(_configSchema.schema.number()),
            adjust_pure_negative: _configSchema.schema.maybe(_configSchema.schema.boolean()),
            boost: _configSchema.schema.maybe(_configSchema.schema.any())
          })
        })),
        aggs: _configSchema.schema.maybe(_configSchema.schema.any()),
        aggregations: _configSchema.schema.maybe(_configSchema.schema.any()),
        sort: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.any())),
        _source: _configSchema.schema.maybe(_configSchema.schema.object({
          includes: _configSchema.schema.arrayOf(_configSchema.schema.string())
        })),
        script_fields: _configSchema.schema.maybe(_configSchema.schema.any())
      }),
      query: _configSchema.schema.object({
        dataSourceMDSId: _configSchema.schema.maybe(_configSchema.schema.string({
          defaultValue: ''
        }))
      })
    }
  }, async (context, request, response) => {
    (0, _metrics_helper.addRequestToMetric)('trace_analytics', 'get', 'count');
    const {
      index,
      size,
      ...rest
    } = request.body;
    const {
      dataSourceMDSId
    } = request.query;
    const params = {
      index: index || _trace_analytics.DATA_PREPPER_INDEX_NAME,
      size,
      body: rest
    };
    try {
      let resp;
      if (dataSourceEnabled && dataSourceMDSId) {
        const client = context.dataSource.opensearch.legacy.getClient(dataSourceMDSId);
        resp = await client.callAPI('search', params);
      } else {
        resp = await context.core.opensearch.legacy.client.callAsCurrentUser('search', params);
      }
      return response.ok({
        body: resp
      });
    } catch (error) {
      (0, _metrics_helper.addRequestToMetric)('trace_analytics', 'get', error);
      if (error.statusCode !== 404) console.error(error);
      return response.custom({
        statusCode: error.statusCode || 500,
        body: error.message
      });
    }
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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