"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.validateTemplate = exports.validateInstance = void 0;
var _ajv = _interopRequireDefault(require("ajv"));
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

const ajv = new _ajv.default();
const staticAsset = {
  type: 'object',
  properties: {
    path: {
      type: 'string'
    },
    annotation: {
      type: 'string',
      nullable: true
    },
    data: {
      type: 'string',
      nullable: true
    }
  },
  required: ['path'],
  additionalProperties: false
};
const templateSchema = {
  type: 'object',
  properties: {
    name: {
      type: 'string'
    },
    version: {
      type: 'string'
    },
    displayName: {
      type: 'string',
      nullable: true
    },
    license: {
      type: 'string'
    },
    type: {
      type: 'string'
    },
    labels: {
      type: 'array',
      items: {
        type: 'string'
      },
      nullable: true
    },
    tags: {
      type: 'array',
      items: {
        type: 'string'
      },
      nullable: true
    },
    author: {
      type: 'string',
      nullable: true
    },
    description: {
      type: 'string',
      nullable: true
    },
    sourceUrl: {
      type: 'string',
      nullable: true
    },
    workflows: {
      type: 'array',
      items: {
        type: 'object',
        properties: {
          name: {
            type: 'string'
          },
          label: {
            type: 'string'
          },
          description: {
            type: 'string'
          },
          enabled_by_default: {
            type: 'boolean'
          },
          applicable_data_sources: {
            type: 'array',
            items: {
              type: 'string'
            },
            nullable: true
          }
        },
        required: ['name', 'label', 'description', 'enabled_by_default']
      },
      nullable: true
    },
    statics: {
      type: 'object',
      properties: {
        logo: {
          ...staticAsset,
          nullable: true
        },
        gallery: {
          type: 'array',
          items: staticAsset,
          nullable: true
        },
        darkModeLogo: {
          ...staticAsset,
          nullable: true
        },
        darkModeGallery: {
          type: 'array',
          items: staticAsset,
          nullable: true
        }
      },
      additionalProperties: false,
      nullable: true
    },
    components: {
      type: 'array',
      items: {
        type: 'object',
        properties: {
          name: {
            type: 'string'
          },
          version: {
            type: 'string'
          },
          data: {
            type: 'string',
            nullable: true
          }
        },
        required: ['name', 'version']
      }
    },
    assets: {
      type: 'array',
      items: {
        type: 'object',
        properties: {
          name: {
            type: 'string'
          },
          version: {
            type: 'string'
          },
          extension: {
            type: 'string'
          },
          type: {
            type: 'string'
          },
          data: {
            type: 'string',
            nullable: true
          },
          workflows: {
            type: 'array',
            items: {
              type: 'string'
            },
            nullable: true
          }
        },
        required: ['name', 'version', 'extension', 'type'],
        additionalProperties: false
      }
    },
    sampleData: {
      type: 'object',
      properties: {
        path: {
          type: 'string'
        },
        data: {
          type: 'string',
          nullable: true
        }
      },
      required: ['path'],
      additionalProperties: false,
      nullable: true
    }
  },
  required: ['name', 'version', 'license', 'type', 'components', 'assets'],
  additionalProperties: false
};
const instanceSchema = {
  type: 'object',
  properties: {
    name: {
      type: 'string'
    },
    templateName: {
      type: 'string'
    },
    dataSource: {
      type: 'string'
    },
    creationDate: {
      type: 'string'
    },
    status: {
      type: 'string',
      nullable: true
    },
    assets: {
      type: 'array',
      items: {
        type: 'object',
        properties: {
          assetType: {
            type: 'string'
          },
          assetId: {
            type: 'string'
          },
          isDefaultAsset: {
            type: 'boolean'
          },
          description: {
            type: 'string'
          },
          status: {
            type: 'string',
            nullable: true
          }
        },
        required: ['assetType', 'assetId', 'isDefaultAsset', 'description']
      }
    }
  },
  required: ['name', 'templateName', 'dataSource', 'creationDate', 'assets']
};
const templateValidator = ajv.compile(templateSchema);
const instanceValidator = ajv.compile(instanceSchema);

/**
 * Validates an integration template against a predefined schema using the AJV library.
 * Since AJV validators use side effects for errors,
 * this is a more conventional wrapper that simplifies calling.
 *
 * @param data The data to be validated as an IntegrationTemplate.
 * @return A Result indicating whether the validation was successful or not.
 *         If validation succeeds, returns an object with 'ok' set to true and the validated data.
 *         If validation fails, returns an object with 'ok' set to false and an Error object describing the validation error.
 */
const validateTemplate = data => {
  if (!templateValidator(data)) {
    return {
      ok: false,
      error: new Error(ajv.errorsText(templateValidator.errors))
    };
  }
  // We assume an invariant that the type of an integration is connected with its component.
  if (data.components.findIndex(x => x.name === data.type) < 0) {
    return {
      ok: false,
      error: new Error(`The integration type '${data.type}' must be included as a component`)
    };
  }
  return {
    ok: true,
    value: data
  };
};

/**
 * Validates an integration instance against a predefined schema using the AJV library.
 *
 * @param data The data to be validated as an IntegrationInstance.
 * @return A Result indicating whether the validation was successful or not.
 *         If validation succeeds, returns an object with 'ok' set to true and the validated data.
 *         If validation fails, returns an object with 'ok' set to false and an Error object describing the validation error.
 */
exports.validateTemplate = validateTemplate;
const validateInstance = data => {
  if (!instanceValidator(data)) {
    return {
      ok: false,
      error: new Error(ajv.errorsText(instanceValidator.errors))
    };
  }
  return {
    ok: true,
    value: data
  };
};
exports.validateInstance = validateInstance;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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