"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IntegrationInstanceBuilder = void 0;
var _uuid = require("uuid");
var _utils = require("./repository/utils");
function _defineProperty(obj, key, value) { key = _toPropertyKey(key); if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }
function _toPropertyKey(arg) { var key = _toPrimitive(arg, "string"); return typeof key === "symbol" ? key : String(key); }
function _toPrimitive(input, hint) { if (typeof input !== "object" || input === null) return input; var prim = input[Symbol.toPrimitive]; if (prim !== undefined) { var res = prim.call(input, hint || "default"); if (typeof res !== "object") return res; throw new TypeError("@@toPrimitive must return a primitive value."); } return (hint === "string" ? String : Number)(input); } /*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */
class IntegrationInstanceBuilder {
  constructor(client) {
    _defineProperty(this, "client", void 0);
    this.client = client;
  }
  async build(integration, options) {
    const instance = await (0, _utils.deepCheck)(integration);
    if (!instance.ok) {
      return Promise.reject(instance.error);
    }
    const assets = await integration.getAssets();
    if (!assets.ok) {
      return Promise.reject(assets.error);
    }
    const remapped = this.remapIDs(this.getSavedObjectBundles(assets.value, options.workflows));
    const withDataSource = this.remapDataSource(remapped, options.indexPattern);
    const withSubstitutedQueries = this.substituteQueries(withDataSource, options.dataSource, options.tableName);
    const refs = await this.postAssets(withSubstitutedQueries);
    const builtInstance = await this.buildInstance(integration, refs, options);
    return builtInstance;
  }

  // If we have a data source or table specified, hunt for saved queries and update them with the
  // new DS/table.
  substituteQueries(assets, dataSource, tableName) {
    if (!dataSource) {
      return assets;
    }
    assets = assets.map(asset => {
      if (asset.type === 'observability-search') {
        const savedQuery = asset.attributes.savedQuery;
        if (!savedQuery.data_sources) {
          return asset;
        }
        const dataSources = JSON.parse(savedQuery.data_sources);
        for (const ds of dataSources) {
          if (ds.type !== 's3glue') {
            continue; // Nothing to do
          }
          // TODO is there a distinction between these where we should only set one? They're all
          // equivalent in every export I've seen.
          ds.name = dataSource;
          ds.label = dataSource;
          ds.value = dataSource;
        }
        savedQuery.data_sources = JSON.stringify(dataSources);
        if (savedQuery.query_lang === 'SQL' && tableName) {
          savedQuery.query = savedQuery.query.replaceAll('{table_name}', tableName);
        }
      }
      return asset;
    });
    return assets;
  }
  getSavedObjectBundles(assets, includeWorkflows) {
    return assets.filter(asset => {
      // At this stage we only care about installing bundles
      if (asset.type !== 'savedObjectBundle') {
        return false;
      }
      // If no workflows present: default to all workflows
      // Otherwise only install if workflow is present
      if (!asset.workflows || !includeWorkflows) {
        return true;
      }
      return includeWorkflows.some(w => {
        var _asset$workflows;
        return (_asset$workflows = asset.workflows) === null || _asset$workflows === void 0 ? void 0 : _asset$workflows.includes(w);
      });
    }).map(asset => asset.data).flat();
  }
  remapDataSource(assets, dataSource) {
    if (!dataSource) return assets;
    return assets.map(asset => {
      if (asset.type === 'index-pattern') {
        asset.attributes.title = dataSource;
      }
      return asset;
    });
  }
  remapIDs(assets) {
    const toRemap = assets.filter(asset => asset.id);
    const idMap = new Map();
    return toRemap.map(item => {
      if (!idMap.has(item.id)) {
        idMap.set(item.id, (0, _uuid.v4)());
      }
      item.id = idMap.get(item.id);
      for (let ref = 0; ref < item.references.length; ref++) {
        const refId = item.references[ref].id;
        if (!idMap.has(refId)) {
          idMap.set(refId, (0, _uuid.v4)());
        }
        item.references[ref].id = idMap.get(refId);
      }
      return item;
    });
  }
  async postAssets(assets) {
    try {
      const response = await this.client.bulkCreate(assets);
      const refs = response.saved_objects.map(obj => {
        var _obj$attributes;
        return {
          assetType: obj.type,
          assetId: obj.id,
          status: 'available',
          // Assuming a successfully created object is available
          isDefaultAsset: obj.type === 'dashboard',
          // Assuming for now that dashboards are default
          description: (_obj$attributes = obj.attributes) === null || _obj$attributes === void 0 ? void 0 : _obj$attributes.title
        };
      });
      return Promise.resolve(refs);
    } catch (err) {
      return Promise.reject(err);
    }
  }
  async buildInstance(integration, refs, options) {
    const config = await integration.getConfig();
    if (!config.ok) {
      return Promise.reject(new Error('Attempted to create instance with invalid template', config.error));
    }
    return Promise.resolve({
      name: options.name,
      templateName: config.value.name,
      // Before data sources existed we called the index pattern a data source. Now we need the old
      // name for BWC but still use the new data sources in building, so we map the variable only
      // for returned output here
      dataSource: options.indexPattern,
      creationDate: new Date().toISOString(),
      assets: refs
    });
  }
}
exports.IntegrationInstanceBuilder = IntegrationInstanceBuilder;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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