"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.handleWithCallback = void 0;
exports.registerIntegrationsRoute = registerIntegrationsRoute;
var _configSchema = require("@osd/config-schema");
var mime = _interopRequireWildcard(require("mime"));
var _sanitizeFilename = _interopRequireDefault(require("sanitize-filename"));
var _shared = require("../../../common/constants/shared");
var _integrations_manager = require("../../adaptors/integrations/integrations_manager");
function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }
function _getRequireWildcardCache(e) { if ("function" != typeof WeakMap) return null; var r = new WeakMap(), t = new WeakMap(); return (_getRequireWildcardCache = function (e) { return e ? t : r; })(e); }
function _interopRequireWildcard(e, r) { if (!r && e && e.__esModule) return e; if (null === e || "object" != typeof e && "function" != typeof e) return { default: e }; var t = _getRequireWildcardCache(r); if (t && t.has(e)) return t.get(e); var n = { __proto__: null }, a = Object.defineProperty && Object.getOwnPropertyDescriptor; for (var u in e) if ("default" !== u && Object.prototype.hasOwnProperty.call(e, u)) { var i = a ? Object.getOwnPropertyDescriptor(e, u) : null; i && (i.get || i.set) ? Object.defineProperty(n, u, i) : n[u] = e[u]; } return n.default = e, t && t.set(e, n), n; }
/*
 * Copyright OpenSearch Contributors
 * SPDX-License-Identifier: Apache-2.0
 */

/**
 * Handle an `OpenSearchDashboardsRequest` using the provided `callback` function.
 * This is a convenience method that handles common error handling and response formatting.
 * The callback must accept a `IntegrationsManager` as its first argument.
 *
 * If the callback throws an error,
 * the `OpenSearchDashboardsResponse` will be formatted using the error's `statusCode` and `message` properties.
 * Otherwise, the callback's return value will be formatted in a JSON object under the `data` field.
 *
 * @param {IntegrationsManager} manager The integration manager instance to use for making requests.
 * @param {OpenSearchDashboardsResponseFactory} response The factory to use for creating responses.
 * @callback callback A callback that will invoke a request on a provided adaptor.
 * @returns {Promise<OpenSearchDashboardsResponse>} An `OpenSearchDashboardsResponse` with the return data from the callback.
 */
const handleWithCallback = async (manager, response, callback) => {
  try {
    const data = await callback(manager);
    return response.ok({
      body: {
        data
      }
    });
  } catch (err) {
    console.error(`handleWithCallback: callback failed with error "${err.message}"`);
    return response.custom({
      statusCode: err.statusCode || 500,
      body: err.message
    });
  }
};
exports.handleWithCallback = handleWithCallback;
function registerIntegrationsRoute(router) {
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository`,
    validate: false
  }, async (context, request, response) => {
    const adaptor = new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
    return handleWithCallback(adaptor, response, async a => a.getIntegrationTemplates());
  });
  router.post({
    path: `${_shared.INTEGRATIONS_BASE}/store/{templateName}`,
    validate: {
      params: _configSchema.schema.object({
        templateName: _configSchema.schema.string()
      }),
      body: _configSchema.schema.object({
        name: _configSchema.schema.string(),
        indexPattern: _configSchema.schema.string(),
        workflows: _configSchema.schema.maybe(_configSchema.schema.arrayOf(_configSchema.schema.string())),
        dataSource: _configSchema.schema.maybe(_configSchema.schema.string()),
        tableName: _configSchema.schema.maybe(_configSchema.schema.string())
      })
    }
  }, async (context, request, response) => {
    const adaptor = new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
    return handleWithCallback(adaptor, response, async a => {
      return a.loadIntegrationInstance(request.params.templateName, request.body.name, request.body.indexPattern, request.body.workflows, request.body.dataSource, request.body.tableName);
    });
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{name}`,
    validate: {
      params: _configSchema.schema.object({
        name: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
    return handleWithCallback(adaptor, response, async a => (await a.getIntegrationTemplates({
      name: request.params.name
    })).hits[0]);
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/static/{path}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string(),
        path: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
    try {
      const requestPath = (0, _sanitizeFilename.default)(request.params.path);
      const result = await adaptor.getStatic(request.params.id, requestPath);
      return response.ok({
        headers: {
          'Content-Type': mime.getType(request.params.path)
        },
        body: result
      });
    } catch (err) {
      return response.custom({
        statusCode: err.statusCode ? err.statusCode : 500,
        body: err.message
      });
    }
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/schema`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
    return handleWithCallback(adaptor, response, async a => a.getSchemas(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/assets`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
    return handleWithCallback(adaptor, response, async a => a.getAssets(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/repository/{id}/data`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
    return handleWithCallback(adaptor, response, async a => a.getSampleData(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/store`,
    validate: false
  }, async (context, request, response) => {
    const adaptor = new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
    return handleWithCallback(adaptor, response, async a => a.getIntegrationInstances({}));
  });
  router.delete({
    path: `${_shared.INTEGRATIONS_BASE}/store/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
    return handleWithCallback(adaptor, response, async a => a.deleteIntegrationInstance(request.params.id));
  });
  router.get({
    path: `${_shared.INTEGRATIONS_BASE}/store/{id}`,
    validate: {
      params: _configSchema.schema.object({
        id: _configSchema.schema.string()
      })
    }
  }, async (context, request, response) => {
    const adaptor = new _integrations_manager.IntegrationsManager(context.core.savedObjects.client);
    return handleWithCallback(adaptor, response, async a => a.getIntegrationInstance({
      id: request.params.id
    }));
  });
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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