"use strict";
// Copyright Amazon.com Inc. or its affiliates. All Rights Reserved.
// SPDX-License-Identifier: Apache-2.0
Object.defineProperty(exports, "__esModule", { value: true });
exports.AwsKmsMrkAwareSymmetricDiscoveryKeyringClass = void 0;
const material_management_1 = require("@aws-crypto/material-management");
const arn_parsing_1 = require("./arn_parsing");
const helpers_1 = require("./helpers");
function AwsKmsMrkAwareSymmetricDiscoveryKeyringClass(BaseKeyring) {
    class AwsKmsMrkAwareSymmetricDiscoveryKeyring
    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.5
    //# MUST implement that AWS Encryption SDK Keyring interface (../keyring-
    //# interface.md#interface)
     extends BaseKeyring {
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.6
        //# On initialization the caller MUST provide:
        constructor({ client, grantTokens, discoveryFilter, }) {
            super();
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.6
            //# The keyring MUST know what Region the AWS KMS client is in.
            //
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.6
            //# It
            //# SHOULD obtain this information directly from the client as opposed to
            //# having an additional parameter.
            //
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.6
            //# However if it can not, then it MUST
            //# NOT create the client itself.
            //
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.6
            //# It SHOULD have a Region parameter and
            //# SHOULD try to identify mismatched configurations.
            //
            // @ts-ignore the V3 client has set the config to protected
            const clientRegion = client.config.region;
            (0, material_management_1.needs)(clientRegion, 'Client must be configured to a region.');
            /* Precondition: The AwsKmsMrkAwareSymmetricDiscoveryKeyring Discovery filter *must* be able to match something.
             * I am not going to wait to tell you
             * that no CMK can match an empty account list.
             * e.g. [], [''], '' are not valid.
             */
            (0, material_management_1.needs)(!discoveryFilter ||
                (discoveryFilter.accountIDs &&
                    discoveryFilter.accountIDs.length &&
                    !!discoveryFilter.partition &&
                    discoveryFilter.accountIDs.every((a) => typeof a === 'string' && !!a)), 'A discovery filter must be able to match something.');
            (0, material_management_1.readOnlyProperty)(this, 'client', client);
            (0, material_management_1.readOnlyProperty)(this, 'clientRegion', clientRegion);
            (0, material_management_1.readOnlyProperty)(this, 'grantTokens', grantTokens);
            (0, material_management_1.readOnlyProperty)(this, 'discoveryFilter', discoveryFilter
                ? Object.freeze({
                    ...discoveryFilter,
                    accountIDs: Object.freeze(discoveryFilter.accountIDs),
                })
                : discoveryFilter);
        }
        async _onEncrypt() {
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.7
            //# This function MUST fail.
            throw new Error('AwsKmsMrkAwareSymmetricDiscoveryKeyring cannot be used to encrypt');
        }
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
        //# OnDecrypt MUST take decryption materials (structures.md#decryption-
        //# materials) and a list of encrypted data keys
        //# (structures.md#encrypted-data-key) as input.
        async _onDecrypt(material, encryptedDataKeys) {
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
            //# If the decryption materials (structures.md#decryption-materials)
            //# already contained a valid plaintext data key OnDecrypt MUST
            //# immediately return the unmodified decryption materials
            //# (structures.md#decryption-materials).
            if (material.hasValidKey())
                return material;
            const { client, grantTokens, clientRegion } = this;
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
            //# The set of encrypted data keys MUST first be filtered to match this
            //# keyring's configuration.
            const decryptableEDKs = encryptedDataKeys.filter(filterEDKs(this));
            const cmkErrors = [];
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
            //# For each encrypted data key in the filtered set, one at a time, the
            //# OnDecrypt MUST attempt to decrypt the data key.
            for (const edk of decryptableEDKs) {
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                //# Otherwise it MUST
                //# be the provider info.
                let keyId = edk.providerInfo;
                //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                //# *  "KeyId": If the provider info's resource type is "key" and its
                //# resource is a multi-Region key then a new ARN MUST be created
                //# where the region part MUST equal the AWS KMS client region and
                //# every other part MUST equal the provider info.
                const keyArn = (0, arn_parsing_1.parseAwsKmsKeyArn)(edk.providerInfo);
                (0, material_management_1.needs)(keyArn, 'Unexpected EDK ProviderInfo for AWS KMS EDK');
                if ((0, arn_parsing_1.isMultiRegionAwsKmsArn)(keyArn)) {
                    keyId = (0, arn_parsing_1.constructArnInOtherRegion)(keyArn, clientRegion);
                }
                let dataKey = false;
                try {
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                    //# When calling AWS KMS Decrypt
                    //# (https://docs.aws.amazon.com/kms/latest/APIReference/
                    //# API_Decrypt.html), the keyring MUST call with a request constructed
                    //# as follows:
                    dataKey = await (0, helpers_1.decrypt)(
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                    //# To attempt to decrypt a particular encrypted data key
                    //# (structures.md#encrypted-data-key), OnDecrypt MUST call AWS KMS
                    //# Decrypt (https://docs.aws.amazon.com/kms/latest/APIReference/
                    //# API_Decrypt.html) with the configured AWS KMS client.
                    client, {
                        providerId: edk.providerId,
                        providerInfo: keyId,
                        encryptedDataKey: edk.encryptedDataKey,
                    }, material.encryptionContext, grantTokens);
                    /* This should be impossible given that decrypt only returns false if the client supplier does
                     * or if the providerId is not "aws-kms", which we have already filtered out
                     */
                    if (!dataKey)
                        continue;
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                    //# *  The "KeyId" field in the response MUST equal the requested "KeyId"
                    (0, material_management_1.needs)(dataKey.KeyId === keyId, 'KMS Decryption key does not match the requested key id.');
                    const flags = material_management_1.KeyringTraceFlag.WRAPPING_KEY_DECRYPTED_DATA_KEY |
                        material_management_1.KeyringTraceFlag.WRAPPING_KEY_VERIFIED_ENC_CTX;
                    const trace = {
                        keyNamespace: helpers_1.KMS_PROVIDER_ID,
                        keyName: dataKey.KeyId,
                        flags,
                    };
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                    //# *  The length of the response's "Plaintext" MUST equal the key
                    //# derivation input length (algorithm-suites.md#key-derivation-input-
                    //# length) specified by the algorithm suite (algorithm-suites.md)
                    //# included in the input decryption materials
                    //# (structures.md#decryption-materials).
                    //
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                    //# Since the response does satisfies these requirements then OnDecrypt
                    //# MUST do the following with the response:
                    //
                    // setUnencryptedDataKey will throw if the plaintext does not match the algorithm suite requirements.
                    material.setUnencryptedDataKey(dataKey.Plaintext, trace);
                    return material;
                }
                catch (e) {
                    //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
                    //# If the response does not satisfies these requirements then an error
                    //# is collected and the next encrypted data key in the filtered set MUST
                    //# be attempted.
                    cmkErrors.push({ errPlus: e });
                }
            }
            //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
            //# If OnDecrypt fails to successfully decrypt any encrypted data key
            //# (structures.md#encrypted-data-key), then it MUST yield an error that
            //# includes all collected errors.
            (0, material_management_1.needs)(material.hasValidKey(), [
                `Unable to decrypt data key${!decryptableEDKs.length ? ': No EDKs supplied' : ''}.`,
                ...cmkErrors.map((e, i) => `Error #${i + 1}  \n${e.errPlus.stack}`),
            ].join('\n'));
            return material;
        }
    }
    (0, material_management_1.immutableClass)(AwsKmsMrkAwareSymmetricDiscoveryKeyring);
    return AwsKmsMrkAwareSymmetricDiscoveryKeyring;
}
exports.AwsKmsMrkAwareSymmetricDiscoveryKeyringClass = AwsKmsMrkAwareSymmetricDiscoveryKeyringClass;
function filterEDKs({ discoveryFilter, clientRegion, }) {
    return function filter({ providerId, providerInfo }) {
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
        //# *  Its provider ID MUST exactly match the value "aws-kms".
        if (providerId !== helpers_1.KMS_PROVIDER_ID)
            return false;
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
        //# *  The provider info MUST be a valid AWS KMS ARN (aws-kms-key-
        //# arn.md#a-valid-aws-kms-arn) with a resource type of "key" or
        //# OnDecrypt MUST fail.
        const edkArn = (0, arn_parsing_1.parseAwsKmsKeyArn)(providerInfo);
        (0, material_management_1.needs)(edkArn && edkArn.ResourceType === 'key', 'Unexpected EDK ProviderInfo for AWS KMS EDK');
        const { AccountId: account, Partition: partition, Region: edkRegion, } = edkArn;
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
        //# *  If the provider info is not identified as a multi-Region key (aws-
        //# kms-key-arn.md#identifying-an-aws-kms-multi-region-key), then the
        //# provider info's Region MUST match the AWS KMS client region.
        if (!(0, arn_parsing_1.isMultiRegionAwsKmsArn)(edkArn) && clientRegion !== edkRegion) {
            return false;
        }
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
        //# *  If a discovery filter is configured, its partition and the
        //# provider info partition MUST match.
        //
        //= compliance/framework/aws-kms/aws-kms-mrk-aware-symmetric-region-discovery-keyring.txt#2.8
        //# *  If a discovery filter is configured, its set of accounts MUST
        //# contain the provider info account.
        return (!discoveryFilter ||
            (discoveryFilter.partition === partition &&
                discoveryFilter.accountIDs.includes(account)));
    };
}
//# sourceMappingURL=data:application/json;base64,eyJ2ZXJzaW9uIjozLCJmaWxlIjoia21zX21ya19kaXNjb3Zlcnlfa2V5cmluZy5qcyIsInNvdXJjZVJvb3QiOiIiLCJzb3VyY2VzIjpbIi4uLy4uLy4uL3NyYy9rbXNfbXJrX2Rpc2NvdmVyeV9rZXlyaW5nLnRzIl0sIm5hbWVzIjpbXSwibWFwcGluZ3MiOiI7QUFBQSxvRUFBb0U7QUFDcEUsc0NBQXNDOzs7QUFFdEMseUVBYXdDO0FBQ3hDLCtDQUlzQjtBQUN0Qix1Q0FBb0Q7QUF5Q3BELFNBQWdCLDRDQUE0QyxDQUkxRCxXQUFnQztJQUVoQyxNQUFNLHVDQUF1QztJQUMzQyw2RkFBNkY7SUFDN0YseUVBQXlFO0lBQ3pFLDJCQUEyQjtJQUMzQixTQUFRLFdBQVc7UUFXbkIsNkZBQTZGO1FBQzdGLDhDQUE4QztRQUM5QyxZQUFZLEVBQ1YsTUFBTSxFQUNOLFdBQVcsRUFDWCxlQUFlLEdBQ3NDO1lBQ3JELEtBQUssRUFBRSxDQUFBO1lBRVAsNkZBQTZGO1lBQzdGLCtEQUErRDtZQUMvRCxFQUFFO1lBQ0YsNkZBQTZGO1lBQzdGLE1BQU07WUFDTix5RUFBeUU7WUFDekUsbUNBQW1DO1lBQ25DLEVBQUU7WUFDRiw2RkFBNkY7WUFDN0YsdUNBQXVDO1lBQ3ZDLGlDQUFpQztZQUNqQyxFQUFFO1lBQ0YsNkZBQTZGO1lBQzdGLHlDQUF5QztZQUN6QyxxREFBcUQ7WUFDckQsRUFBRTtZQUNGLDJEQUEyRDtZQUMzRCxNQUFNLFlBQVksR0FBRyxNQUFNLENBQUMsTUFBTSxDQUFDLE1BQU0sQ0FBQTtZQUN6QyxJQUFBLDJCQUFLLEVBQUMsWUFBWSxFQUFFLHdDQUF3QyxDQUFDLENBQUE7WUFFN0Q7Ozs7ZUFJRztZQUNILElBQUEsMkJBQUssRUFDSCxDQUFDLGVBQWU7Z0JBQ2QsQ0FBQyxlQUFlLENBQUMsVUFBVTtvQkFDekIsZUFBZSxDQUFDLFVBQVUsQ0FBQyxNQUFNO29CQUNqQyxDQUFDLENBQUMsZUFBZSxDQUFDLFNBQVM7b0JBQzNCLGVBQWUsQ0FBQyxVQUFVLENBQUMsS0FBSyxDQUM5QixDQUFDLENBQUMsRUFBRSxFQUFFLENBQUMsT0FBTyxDQUFDLEtBQUssUUFBUSxJQUFJLENBQUMsQ0FBQyxDQUFDLENBQ3BDLENBQUMsRUFDTixxREFBcUQsQ0FDdEQsQ0FBQTtZQUVELElBQUEsc0NBQWdCLEVBQUMsSUFBSSxFQUFFLFFBQVEsRUFBRSxNQUFNLENBQUMsQ0FBQTtZQUN4QyxJQUFBLHNDQUFnQixFQUFDLElBQUksRUFBRSxjQUFjLEVBQUUsWUFBWSxDQUFDLENBQUE7WUFDcEQsSUFBQSxzQ0FBZ0IsRUFBQyxJQUFJLEVBQUUsYUFBYSxFQUFFLFdBQVcsQ0FBQyxDQUFBO1lBQ2xELElBQUEsc0NBQWdCLEVBQ2QsSUFBSSxFQUNKLGlCQUFpQixFQUNqQixlQUFlO2dCQUNiLENBQUMsQ0FBQyxNQUFNLENBQUMsTUFBTSxDQUFDO29CQUNaLEdBQUcsZUFBZTtvQkFDbEIsVUFBVSxFQUFFLE1BQU0sQ0FBQyxNQUFNLENBQUMsZUFBZSxDQUFDLFVBQVUsQ0FBQztpQkFDdEQsQ0FBQztnQkFDSixDQUFDLENBQUMsZUFBZSxDQUNwQixDQUFBO1FBQ0gsQ0FBQztRQUVELEtBQUssQ0FBQyxVQUFVO1lBQ2QsNkZBQTZGO1lBQzdGLDRCQUE0QjtZQUM1QixNQUFNLElBQUksS0FBSyxDQUNiLG1FQUFtRSxDQUNwRSxDQUFBO1FBQ0gsQ0FBQztRQUVELDZGQUE2RjtRQUM3Rix1RUFBdUU7UUFDdkUsZ0RBQWdEO1FBQ2hELGdEQUFnRDtRQUNoRCxLQUFLLENBQUMsVUFBVSxDQUNkLFFBQStCLEVBQy9CLGlCQUFxQztZQUVyQyw2RkFBNkY7WUFDN0Ysb0VBQW9FO1lBQ3BFLCtEQUErRDtZQUMvRCwwREFBMEQ7WUFDMUQseUNBQXlDO1lBQ3pDLElBQUksUUFBUSxDQUFDLFdBQVcsRUFBRTtnQkFBRSxPQUFPLFFBQVEsQ0FBQTtZQUUzQyxNQUFNLEVBQUUsTUFBTSxFQUFFLFdBQVcsRUFBRSxZQUFZLEVBQUUsR0FBRyxJQUFJLENBQUE7WUFFbEQsNkZBQTZGO1lBQzdGLHVFQUF1RTtZQUN2RSw0QkFBNEI7WUFDNUIsTUFBTSxlQUFlLEdBQUcsaUJBQWlCLENBQUMsTUFBTSxDQUFDLFVBQVUsQ0FBQyxJQUFJLENBQUMsQ0FBQyxDQUFBO1lBQ2xFLE1BQU0sU0FBUyxHQUFnQixFQUFFLENBQUE7WUFFakMsNkZBQTZGO1lBQzdGLHVFQUF1RTtZQUN2RSxtREFBbUQ7WUFDbkQsS0FBSyxNQUFNLEdBQUcsSUFBSSxlQUFlLEVBQUU7Z0JBQ2pDLDZGQUE2RjtnQkFDN0YscUJBQXFCO2dCQUNyQix5QkFBeUI7Z0JBQ3pCLElBQUksS0FBSyxHQUFHLEdBQUcsQ0FBQyxZQUFZLENBQUE7Z0JBQzVCLDZGQUE2RjtnQkFDN0YscUVBQXFFO2dCQUNyRSxpRUFBaUU7Z0JBQ2pFLGtFQUFrRTtnQkFDbEUsa0RBQWtEO2dCQUNsRCxNQUFNLE1BQU0sR0FBRyxJQUFBLCtCQUFpQixFQUFDLEdBQUcsQ0FBQyxZQUFZLENBQUMsQ0FBQTtnQkFDbEQsSUFBQSwyQkFBSyxFQUFDLE1BQU0sRUFBRSw2Q0FBNkMsQ0FBQyxDQUFBO2dCQUM1RCxJQUFJLElBQUEsb0NBQXNCLEVBQUMsTUFBTSxDQUFDLEVBQUU7b0JBQ2xDLEtBQUssR0FBRyxJQUFBLHVDQUF5QixFQUFDLE1BQU0sRUFBRSxZQUFZLENBQUMsQ0FBQTtpQkFDeEQ7Z0JBRUQsSUFBSSxPQUFPLEdBQW9DLEtBQUssQ0FBQTtnQkFDcEQsSUFBSTtvQkFDRiw2RkFBNkY7b0JBQzdGLGdDQUFnQztvQkFDaEMseURBQXlEO29CQUN6RCx1RUFBdUU7b0JBQ3ZFLGVBQWU7b0JBQ2YsT0FBTyxHQUFHLE1BQU0sSUFBQSxpQkFBTztvQkFDckIsNkZBQTZGO29CQUM3Rix5REFBeUQ7b0JBQ3pELG1FQUFtRTtvQkFDbkUsaUVBQWlFO29CQUNqRSx5REFBeUQ7b0JBQ3pELE1BQU0sRUFDTjt3QkFDRSxVQUFVLEVBQUUsR0FBRyxDQUFDLFVBQVU7d0JBQzFCLFlBQVksRUFBRSxLQUFLO3dCQUNuQixnQkFBZ0IsRUFBRSxHQUFHLENBQUMsZ0JBQWdCO3FCQUN2QyxFQUNELFFBQVEsQ0FBQyxpQkFBaUIsRUFDMUIsV0FBVyxDQUNaLENBQUE7b0JBQ0Q7O3VCQUVHO29CQUNILElBQUksQ0FBQyxPQUFPO3dCQUFFLFNBQVE7b0JBRXRCLDZGQUE2RjtvQkFDN0YseUVBQXlFO29CQUN6RSxJQUFBLDJCQUFLLEVBQ0gsT0FBTyxDQUFDLEtBQUssS0FBSyxLQUFLLEVBQ3ZCLHlEQUF5RCxDQUMxRCxDQUFBO29CQUVELE1BQU0sS0FBSyxHQUNULHNDQUFnQixDQUFDLCtCQUErQjt3QkFDaEQsc0NBQWdCLENBQUMsNkJBQTZCLENBQUE7b0JBQ2hELE1BQU0sS0FBSyxHQUFpQjt3QkFDMUIsWUFBWSxFQUFFLHlCQUFlO3dCQUM3QixPQUFPLEVBQUUsT0FBTyxDQUFDLEtBQUs7d0JBQ3RCLEtBQUs7cUJBQ04sQ0FBQTtvQkFFRCw2RkFBNkY7b0JBQzdGLGtFQUFrRTtvQkFDbEUsc0VBQXNFO29CQUN0RSxrRUFBa0U7b0JBQ2xFLDhDQUE4QztvQkFDOUMseUNBQXlDO29CQUN6QyxFQUFFO29CQUNGLDZGQUE2RjtvQkFDN0YsdUVBQXVFO29CQUN2RSw0Q0FBNEM7b0JBQzVDLEVBQUU7b0JBQ0YscUdBQXFHO29CQUNyRyxRQUFRLENBQUMscUJBQXFCLENBQUMsT0FBTyxDQUFDLFNBQVMsRUFBRSxLQUFLLENBQUMsQ0FBQTtvQkFDeEQsT0FBTyxRQUFRLENBQUE7aUJBQ2hCO2dCQUFDLE9BQU8sQ0FBQyxFQUFFO29CQUNWLDZGQUE2RjtvQkFDN0YsdUVBQXVFO29CQUN2RSx5RUFBeUU7b0JBQ3pFLGlCQUFpQjtvQkFDakIsU0FBUyxDQUFDLElBQUksQ0FBQyxFQUFFLE9BQU8sRUFBRSxDQUFDLEVBQUUsQ0FBQyxDQUFBO2lCQUMvQjthQUNGO1lBQ0QsNkZBQTZGO1lBQzdGLHFFQUFxRTtZQUNyRSx3RUFBd0U7WUFDeEUsa0NBQWtDO1lBQ2xDLElBQUEsMkJBQUssRUFDSCxRQUFRLENBQUMsV0FBVyxFQUFFLEVBQ3RCO2dCQUNFLDZCQUNFLENBQUMsZUFBZSxDQUFDLE1BQU0sQ0FBQyxDQUFDLENBQUMsb0JBQW9CLENBQUMsQ0FBQyxDQUFDLEVBQ25ELEdBQUc7Z0JBQ0gsR0FBRyxTQUFTLENBQUMsR0FBRyxDQUFDLENBQUMsQ0FBQyxFQUFFLENBQUMsRUFBRSxFQUFFLENBQUMsVUFBVSxDQUFDLEdBQUcsQ0FBQyxPQUFPLENBQUMsQ0FBQyxPQUFPLENBQUMsS0FBSyxFQUFFLENBQUM7YUFDcEUsQ0FBQyxJQUFJLENBQUMsSUFBSSxDQUFDLENBQ2IsQ0FBQTtZQUNELE9BQU8sUUFBUSxDQUFBO1FBQ2pCLENBQUM7S0FDRjtJQUNELElBQUEsb0NBQWMsRUFBQyx1Q0FBdUMsQ0FBQyxDQUFBO0lBQ3ZELE9BQU8sdUNBQXVDLENBQUE7QUFDaEQsQ0FBQztBQXRORCxvR0FzTkM7QUFFRCxTQUFTLFVBQVUsQ0FHakIsRUFDQSxlQUFlLEVBQ2YsWUFBWSxHQUN3QztJQUNwRCxPQUFPLFNBQVMsTUFBTSxDQUFDLEVBQUUsVUFBVSxFQUFFLFlBQVksRUFBb0I7UUFDbkUsNkZBQTZGO1FBQzdGLDhEQUE4RDtRQUM5RCxJQUFJLFVBQVUsS0FBSyx5QkFBZTtZQUFFLE9BQU8sS0FBSyxDQUFBO1FBRWhELDZGQUE2RjtRQUM3RixrRUFBa0U7UUFDbEUsZ0VBQWdFO1FBQ2hFLHdCQUF3QjtRQUN4QixNQUFNLE1BQU0sR0FBRyxJQUFBLCtCQUFpQixFQUFDLFlBQVksQ0FBQyxDQUFBO1FBQzlDLElBQUEsMkJBQUssRUFDSCxNQUFNLElBQUksTUFBTSxDQUFDLFlBQVksS0FBSyxLQUFLLEVBQ3ZDLDZDQUE2QyxDQUM5QyxDQUFBO1FBQ0QsTUFBTSxFQUNKLFNBQVMsRUFBRSxPQUFPLEVBQ2xCLFNBQVMsRUFBRSxTQUFTLEVBQ3BCLE1BQU0sRUFBRSxTQUFTLEdBQ2xCLEdBQUcsTUFBTSxDQUFBO1FBRVYsNkZBQTZGO1FBQzdGLHlFQUF5RTtRQUN6RSxxRUFBcUU7UUFDckUsZ0VBQWdFO1FBQ2hFLElBQUksQ0FBQyxJQUFBLG9DQUFzQixFQUFDLE1BQU0sQ0FBQyxJQUFJLFlBQVksS0FBSyxTQUFTLEVBQUU7WUFDakUsT0FBTyxLQUFLLENBQUE7U0FDYjtRQUVELDZGQUE2RjtRQUM3RixpRUFBaUU7UUFDakUsdUNBQXVDO1FBQ3ZDLEVBQUU7UUFDRiw2RkFBNkY7UUFDN0Ysb0VBQW9FO1FBQ3BFLHNDQUFzQztRQUN0QyxPQUFPLENBQ0wsQ0FBQyxlQUFlO1lBQ2hCLENBQUMsZUFBZSxDQUFDLFNBQVMsS0FBSyxTQUFTO2dCQUN0QyxlQUFlLENBQUMsVUFBVSxDQUFDLFFBQVEsQ0FBQyxPQUFPLENBQUMsQ0FBQyxDQUNoRCxDQUFBO0lBQ0gsQ0FBQyxDQUFBO0FBQ0gsQ0FBQyJ9