/*
 * Decompiled with CFR 0.152.
 */
package org.keycloak.authentication.authenticators.sessionlimits;

import jakarta.ws.rs.core.Response;
import java.util.Collections;
import java.util.Comparator;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.stream.Collectors;
import org.jboss.logging.Logger;
import org.keycloak.authentication.AuthenticationFlowContext;
import org.keycloak.authentication.AuthenticationFlowError;
import org.keycloak.authentication.AuthenticationFlowException;
import org.keycloak.authentication.Authenticator;
import org.keycloak.models.AuthenticatorConfigModel;
import org.keycloak.models.ClientModel;
import org.keycloak.models.KeycloakSession;
import org.keycloak.models.RealmModel;
import org.keycloak.models.UserModel;
import org.keycloak.models.UserSessionModel;
import org.keycloak.services.managers.AuthenticationManager;
import org.keycloak.utils.StringUtil;

public class UserSessionLimitsAuthenticator
implements Authenticator {
    private static Logger logger = Logger.getLogger(UserSessionLimitsAuthenticator.class);
    public static final String SESSION_LIMIT_EXCEEDED = "sessionLimitExceeded";
    private static String realmEventDetailsTemplate = "Realm session limit exceeded. Realm: %s, Realm limit: %s. Session count: %s, User id: %s";
    private static String clientEventDetailsTemplate = "Client session limit exceeded. Realm: %s, Client limit: %s. Session count: %s, User id: %s";
    protected KeycloakSession session;
    String behavior;

    public UserSessionLimitsAuthenticator(KeycloakSession session) {
        this.session = session;
    }

    public void authenticate(AuthenticationFlowContext context) {
        AuthenticatorConfigModel authenticatorConfig = context.getAuthenticatorConfig();
        if (authenticatorConfig == null) {
            throw new AuthenticationFlowException("No configuration found of 'User Session Count Limiter' authenticator. Please make sure to configure this authenticator in your authentication flow in the realm '" + context.getRealm().getName() + "'!", AuthenticationFlowError.INTERNAL_ERROR);
        }
        Map config = authenticatorConfig.getConfig();
        this.behavior = (String)config.get("behavior");
        int userRealmLimit = this.getIntConfigProperty("userRealmLimit", config);
        int userClientLimit = this.getIntConfigProperty("userClientLimit", config);
        if (context.getRealm() != null && context.getUser() != null) {
            List<UserSessionModel> userSessionsForRealm = this.session.sessions().getUserSessionsStream(context.getRealm(), context.getUser()).collect(Collectors.toList());
            int userSessionCountForRealm = userSessionsForRealm.size();
            ClientModel currentClient = context.getAuthenticationSession().getClient();
            logger.debugf("session-limiter's current keycloak clientId: %s", (Object)currentClient.getClientId());
            List<UserSessionModel> userSessionsForClient = this.getUserSessionsForClientIfEnabled(userSessionsForRealm, currentClient, userClientLimit);
            int userSessionCountForClient = userSessionsForClient.size();
            logger.debugf("session-limiter's configured realm session limit: %s", userRealmLimit);
            logger.debugf("session-limiter's configured client session limit: %s", userClientLimit);
            logger.debugf("session-limiter's count of total user sessions for the entire realm (could be apps other than web apps): %s", userSessionCountForRealm);
            logger.debugf("session-limiter's count of total user sessions for this keycloak client: %s", userSessionCountForClient);
            if (this.exceedsLimit(userSessionCountForRealm, userRealmLimit)) {
                logger.infof("Too many session in this realm for the current user. Session count: %s", (Object)userSessionCountForRealm);
                String eventDetails = String.format(realmEventDetailsTemplate, context.getRealm().getName(), userRealmLimit, userSessionCountForRealm, context.getUser().getId());
                this.handleLimitExceeded(context, userSessionsForRealm, eventDetails, userRealmLimit);
            } else if (this.exceedsLimit(userSessionCountForClient, userClientLimit)) {
                logger.infof("Too many sessions related to the current client for this user. Session count: %s", (Object)userSessionCountForRealm);
                String eventDetails = String.format(clientEventDetailsTemplate, context.getRealm().getName(), userClientLimit, userSessionCountForClient, context.getUser().getId());
                this.handleLimitExceeded(context, userSessionsForClient, eventDetails, userClientLimit);
            } else {
                context.success();
            }
        } else {
            context.success();
        }
    }

    private boolean exceedsLimit(long count, long limit) {
        if (limit <= 0L) {
            return false;
        }
        return this.getNumberOfSessionsThatNeedToBeLoggedOut(count, limit) > 0L;
    }

    private long getNumberOfSessionsThatNeedToBeLoggedOut(long count, long limit) {
        return count - (limit - 1L);
    }

    private int getIntConfigProperty(String key, Map<String, String> config) {
        String value = config.get(key);
        if (StringUtil.isBlank((String)value)) {
            return -1;
        }
        return Integer.parseInt(value);
    }

    private List<UserSessionModel> getUserSessionsForClientIfEnabled(List<UserSessionModel> userSessionsForRealm, ClientModel currentClient, int userClientLimit) {
        if (userClientLimit <= 0) {
            return Collections.EMPTY_LIST;
        }
        logger.debugf("total user sessions for this keycloak client will not be counted. Will be logged as 0 (zero)", new Object[0]);
        List<UserSessionModel> userSessionsForClient = userSessionsForRealm.stream().filter(session -> session.getAuthenticatedClientSessionByClient(currentClient.getId()) != null).collect(Collectors.toList());
        return userSessionsForClient;
    }

    public void action(AuthenticationFlowContext context) {
    }

    public boolean requiresUser() {
        return false;
    }

    public boolean configuredFor(KeycloakSession session, RealmModel realm, UserModel user) {
        return true;
    }

    public void setRequiredActions(KeycloakSession session, RealmModel realm, UserModel user) {
    }

    public void close() {
    }

    private void handleLimitExceeded(AuthenticationFlowContext context, List<UserSessionModel> userSessions, String eventDetails, long limit) {
        switch (this.behavior) {
            case "Deny new session": {
                logger.info((Object)"Denying new session");
                String errorMessage = Optional.ofNullable(context.getAuthenticatorConfig()).map(AuthenticatorConfigModel::getConfig).map(f -> (String)f.get("errorMessage")).orElse(SESSION_LIMIT_EXCEEDED);
                context.getEvent().error("generic_authentication_error");
                Response challenge = context.form().setError(errorMessage, new Object[0]).createErrorPage(Response.Status.FORBIDDEN);
                context.failure(AuthenticationFlowError.GENERIC_AUTHENTICATION_ERROR, challenge, eventDetails, errorMessage);
                break;
            }
            case "Terminate oldest session": {
                logger.info((Object)"Terminating oldest session");
                this.logoutOldestSessions(userSessions, limit);
                context.success();
            }
        }
    }

    private void logoutOldestSessions(List<UserSessionModel> userSessions, long limit) {
        long numberOfSessionsThatNeedToBeLoggedOut = this.getNumberOfSessionsThatNeedToBeLoggedOut(userSessions.size(), limit);
        if (numberOfSessionsThatNeedToBeLoggedOut == 1L) {
            logger.info((Object)"Logging out oldest session");
        } else {
            logger.infof("Logging out oldest %s sessions", (Object)numberOfSessionsThatNeedToBeLoggedOut);
        }
        userSessions.stream().sorted(Comparator.comparingInt(UserSessionModel::getLastSessionRefresh)).limit(numberOfSessionsThatNeedToBeLoggedOut).forEach(userSession -> AuthenticationManager.backchannelLogout(this.session, userSession, true));
    }
}

