"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.IntegrationsKibanaBackend = void 0;

var _path = _interopRequireDefault(require("path"));

var _metrics_helper = require("../../../server/common/metrics/metrics_helper");

var _integrations_builder = require("./integrations_builder");

var _repository = require("./repository/repository");

function _interopRequireDefault(obj) { return obj && obj.__esModule ? obj : { default: obj }; }

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

class IntegrationsKibanaBackend {
  constructor(client, repository) {
    _defineProperty(this, "client", void 0);

    _defineProperty(this, "instanceBuilder", void 0);

    _defineProperty(this, "repository", void 0);

    _defineProperty(this, "deleteIntegrationInstance", async id => {
      let children;

      try {
        children = await this.client.get('integration-instance', id);
      } catch (err) {
        var _err$output;

        return ((_err$output = err.output) === null || _err$output === void 0 ? void 0 : _err$output.statusCode) === 404 ? Promise.resolve([id]) : Promise.reject(err);
      }

      const toDelete = children.attributes.assets.filter(i => i.assetId).map(i => {
        return {
          id: i.assetId,
          type: i.assetType
        };
      });
      toDelete.push({
        id,
        type: 'integration-instance'
      });
      const result = Promise.all(toDelete.map(async asset => {
        try {
          await this.client.delete(asset.type, asset.id);
          return Promise.resolve(asset.id);
        } catch (err) {
          var _err$output2;

          (0, _metrics_helper.addRequestToMetric)('integrations', 'delete', err);
          return ((_err$output2 = err.output) === null || _err$output2 === void 0 ? void 0 : _err$output2.statusCode) === 404 ? Promise.resolve(asset.id) : Promise.reject(err);
        }
      }));
      (0, _metrics_helper.addRequestToMetric)('integrations', 'delete', 'count');
      return result;
    });

    _defineProperty(this, "getIntegrationTemplates", async query => {
      if (query !== null && query !== void 0 && query.name) {
        const integration = await this.repository.getIntegration(query.name);
        const config = await (integration === null || integration === void 0 ? void 0 : integration.getConfig());
        return Promise.resolve({
          hits: config ? [config] : []
        });
      }

      const integrationList = await this.repository.getIntegrationList();
      const configList = await Promise.all(integrationList.map(x => x.getConfig()));
      return Promise.resolve({
        hits: configList.filter(x => x !== null)
      });
    });

    _defineProperty(this, "getIntegrationInstances", async _query => {
      var _result$saved_objects;

      (0, _metrics_helper.addRequestToMetric)('integrations', 'get', 'count');
      const result = await this.client.find({
        type: 'integration-instance'
      });
      return Promise.resolve({
        total: result.total,
        hits: (_result$saved_objects = result.saved_objects) === null || _result$saved_objects === void 0 ? void 0 : _result$saved_objects.map(x => ({ ...x.attributes,
          id: x.id
        }))
      });
    });

    _defineProperty(this, "getIntegrationInstance", async query => {
      (0, _metrics_helper.addRequestToMetric)('integrations', 'get', 'count');
      const result = await this.client.get('integration-instance', `${query.id}`);
      return Promise.resolve(this.buildInstanceResponse(result));
    });

    _defineProperty(this, "buildInstanceResponse", async savedObj => {
      var _savedObj$attributes;

      const assets = (_savedObj$attributes = savedObj.attributes) === null || _savedObj$attributes === void 0 ? void 0 : _savedObj$attributes.assets;
      const status = assets ? await this.getAssetStatus(assets) : 'available';
      return {
        id: savedObj.id,
        status,
        ...savedObj.attributes
      };
    });

    _defineProperty(this, "getAssetStatus", async assets => {
      const statuses = await Promise.all(assets.map(async asset => {
        try {
          await this.client.get(asset.assetType, asset.assetId);
          return {
            id: asset.assetId,
            status: 'available'
          };
        } catch (err) {
          var _err$output3;

          const statusCode = (_err$output3 = err.output) === null || _err$output3 === void 0 ? void 0 : _err$output3.statusCode;

          if (statusCode && 400 <= statusCode && statusCode < 500) {
            return {
              id: asset.assetId,
              status: 'unavailable'
            };
          }

          console.error('Failed to get asset status', err);
          return {
            id: asset.assetId,
            status: 'unknown'
          };
        }
      }));
      const [available, unavailable, unknown] = [statuses.filter(x => x.status === 'available').length, statuses.filter(x => x.status === 'unavailable').length, statuses.filter(x => x.status === 'unknown').length];
      if (unknown > 0) return 'unknown';
      if (unavailable > 0 && available > 0) return 'partially-available';
      if (unavailable > 0) return 'unavailable';
      return 'available';
    });

    _defineProperty(this, "loadIntegrationInstance", async (templateName, name, dataSource) => {
      const template = await this.repository.getIntegration(templateName);

      if (template === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }

      try {
        (0, _metrics_helper.addRequestToMetric)('integrations', 'create', 'count');
        const result = await this.instanceBuilder.build(template, {
          name,
          dataSource
        });
        const test = await this.client.create('integration-instance', result);
        return Promise.resolve({ ...result,
          id: test.id
        });
      } catch (err) {
        (0, _metrics_helper.addRequestToMetric)('integrations', 'create', err);
        return Promise.reject({
          message: err.message,
          statusCode: 500
        });
      }
    });

    _defineProperty(this, "getStatic", async (templateName, staticPath) => {
      var _await$this$repositor;

      const data = await ((_await$this$repositor = await this.repository.getIntegration(templateName)) === null || _await$this$repositor === void 0 ? void 0 : _await$this$repositor.getStatic(staticPath));

      if (!data) {
        return Promise.reject({
          message: `Asset ${staticPath} not found`,
          statusCode: 404
        });
      }

      return Promise.resolve(data);
    });

    _defineProperty(this, "getSchemas", async templateName => {
      const integration = await this.repository.getIntegration(templateName);

      if (integration === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }

      return Promise.resolve(integration.getSchemas());
    });

    _defineProperty(this, "getAssets", async templateName => {
      const integration = await this.repository.getIntegration(templateName);

      if (integration === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }

      return Promise.resolve(integration.getAssets());
    });

    _defineProperty(this, "getSampleData", async templateName => {
      const integration = await this.repository.getIntegration(templateName);

      if (integration === null) {
        return Promise.reject({
          message: `Template ${templateName} not found`,
          statusCode: 404
        });
      }

      return Promise.resolve(integration.getSampleData());
    });

    this.client = client;
    this.repository = repository !== null && repository !== void 0 ? repository : new _repository.Repository(_path.default.join(__dirname, '__data__/repository'));
    this.instanceBuilder = new _integrations_builder.IntegrationInstanceBuilder(this.client);
  }

}

exports.IntegrationsKibanaBackend = IntegrationsKibanaBackend;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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