#------------------------------------------------------------------------------
# File:         PLUS.pm
#
# Description:  PLUS (Picture Licensing Universal System) tags
#
# Revisions:    2016/05/18 - P. Harvey Created
#
# References:   1) http://www.useplus.com/useplus/standards.asp
#------------------------------------------------------------------------------

package Image::ExifTool::PLUS;

use strict;
use vars qw($VERSION);
use Image::ExifTool::XMP;

$VERSION = '1.00';

sub ValidateMediaSummary($);

#------------------------------------------------------------------------------
# PLUS (Picture Licensing Universal System)

# PLUS vocabulary conversions
my %plusVocab = (
    ValueConv => '$val =~ s{http://ns.useplus.org/ldf/vocab/}{}; $val',
    ValueConvInv => '"http://ns.useplus.org/ldf/vocab/$val"',
);

# PLUS License Data Format 1.2.1 structures
# (this seems crazy to me -- why did they define different ID/Name structures
#  for each field rather than just re-using the same structure?)
my %plusLicensee = (
    STRUCT_NAME => 'Licensee',
    NAMESPACE => 'plus',
    # all "rdf:type" properties removed in version 1.2.1
    # (idiots.  Why did they put them there in the first place? -- it required
    # a special patch for ExifTool to support these, and now they are gone!)
  # TYPE => 'plus:LicenseeDetail', (removed in 1.2.1)
    LicenseeID  => { },
    LicenseeName=> { },
);
my %plusEndUser = (
    STRUCT_NAME => 'EndUser',
    NAMESPACE   => 'plus',
  # TYPE => 'plus:EndUserDetail', (removed in 1.2.1)
    EndUserID   => { },
    EndUserName => { },
);
my %plusLicensor = (
    STRUCT_NAME => 'Licensor',
    NAMESPACE   => 'plus',
  # TYPE => 'plus:LicensorDetail', (removed in 1.2.1)
    LicensorID              => { },
    LicensorName            => { },
    LicensorStreetAddress   => { },
    LicensorExtendedAddress => { },
    LicensorCity            => { },
    LicensorRegion          => { },
    LicensorPostalCode      => { },
    LicensorCountry         => { },
    LicensorTelephoneType1  => {
        %plusVocab,
        PrintConv => {
            'work'  => 'Work',
            'cell'  => 'Cell',
            'fax'   => 'FAX',
            'home'  => 'Home',
            'pager' => 'Pager',
        },
    },
    LicensorTelephone1      => { },
    LicensorTelephoneType2  => {
        %plusVocab,
        PrintConv => {
            'work'  => 'Work',
            'cell'  => 'Cell',
            'fax'   => 'FAX',
            'home'  => 'Home',
            'pager' => 'Pager',
        },
    },
    LicensorTelephone2  => { },
    LicensorEmail       => { },
    LicensorURL         => { },
);
my %plusCopyrightOwner = (
    STRUCT_NAME => 'CopyrightOwner',
    NAMESPACE   => 'plus',
  # TYPE        => 'plus:CopyrightOwnerDetail', (removed in 1.2.1)
    CopyrightOwnerID    => { },
    CopyrightOwnerName  => { },
);
my %plusImageCreator = (
    STRUCT_NAME => 'ImageCreator',
    NAMESPACE   => 'plus',
  # TYPE        => 'plus:ImageCreatorDetail', (removed in 1.2.1)
    ImageCreatorID      => { },
    ImageCreatorName    => { },
);
my %plusImageSupplier = (
    STRUCT_NAME => 'ImageSupplier',
    NAMESPACE   => 'plus',
  # TYPE        => 'plus:ImageSupplierDetail', (removed in 1.2.1)
    ImageSupplierID     => { },
    ImageSupplierName   => { },
);

# Media Matrix 1.1.1 decoding
my %mediaMatrix = (
    Notes => q{
        The lookup below is used to add human-readable descriptions to Media Matrix
        ID's in PLUS Media Summary Codes.
    },
    OTHER => sub {
        my ($val, $inv, $conv) = @_;
        $val = uc $val; # convert to upper case
        if ($inv) {
            # prepare MediaSummaryCode for writing
            $val =~ s/\(.*?\)//sg;      # remove descriptions
            $val =~ s/^\s*?PLUS\s+(V[0-9]+)\s+(U[0-9]+)\s*;/|PLUS|$1|$2|/s; # reformat header
            $val =~ tr/;/|/;            # use "|" as separator instead of ";"
            $val =~ tr/0-9A-Z|//dc;     # remove extraneous characters
            $val .= '|' unless $val =~ /\|$/;   # add terminating "|"
            ValidateMediaSummary($val) or return undef;
        } elsif ($val =~ /^\|PLUS\|(.*?)\|(.*?)\|(.*)/s) {
            # add human-readable descriptions
            my ($ver,$num,$code) = ($1,$2,$3);
            $ver .= " (LDF Version $1.$2)" if $ver =~ /V0*(\d+)(\d{2})$/;
            $num .= " ($1 Media Usages:)" if $num =~ /U0*(\d+)/;
            $code =~ tr/0-9A-Z|//dc;    # remove extraneous characters
            $val = "PLUS $ver $num";
            while ($code =~ /(\d[A-Z]{3})/g) {
                my $mmid = $1;
                if (defined $$conv{$mmid}) {
                    $val .= " $mmid ($$conv{$mmid})";
                } elsif ($mmid =~ /^1I([A-Z])([A-Z])/) {    # decode Usage Item count
                    my $n = (ord($1)-65) * 26 + ord($2)-65 + 1;
                    # (add a separator before each new Media Usage Code)
                    $val .= "; $mmid ($n Usage Items:)";
                } elsif ($mmid =~ /^1UN([A-Z])/) {          # decode Usage Number
                    $val .= " (Usage Number $1)";
                } else {
                    $val .= " $mmid";
                }
            }
        }
        return $val;
    },
   # 1 - Usage
    '1IAA' => '1 Usage Item:',
    '1IAB' => '2 Usage Items:',
    '1IAC' => '3 Usage Items:',
    '1IAD' => '4 Usage Items:',
    '1IAE' => '5 Usage Items:',
    # ...
    '1UNA' => 'Usage Number A',
    '1UNB' => 'Usage Number B',
    '1UNC' => 'Usage Number C',
    '1UND' => 'Usage Number D',
    '1UNE' => 'Usage Number E',
    # ...
   # 2 - Media
    '2BOB' => 'Advertising|Periodicals|Magazine, Consumer|Printed',
    '2ACE' => 'Advertising|Periodicals|Magazine, Consumer|Internet Website',
    '2GEN' => 'Advertising|Periodicals|Magazine, Consumer|Internet Downloadable File',
    '2ADS' => 'Advertising|Periodicals|Magazine, Consumer|Internet Email',
    '2GIB' => 'Advertising|Periodicals|Magazine, Consumer|All Internet Distribution Formats',
    '2GHI' => 'Advertising|Periodicals|Magazine, Consumer|Recordable Media',
    '2GEY' => 'Advertising|Periodicals|Magazine, Consumer|All Electronic Distribution Formats',
    '2HAT' => 'Advertising|Periodicals|Magazine, Trade|Printed',
    '2HAG' => 'Advertising|Periodicals|Magazine, Trade|Internet Website',
    '2HAE' => 'Advertising|Periodicals|Magazine, Trade|Internet Downloadable File',
    '2AFT' => 'Advertising|Periodicals|Magazine, Trade|Internet Email',
    '2HAJ' => 'Advertising|Periodicals|Magazine, Trade|All Internet Distribution Formats',
    '2AGE' => 'Advertising|Periodicals|Magazine, Trade|Recordable Media',
    '2HAH' => 'Advertising|Periodicals|Magazine, Trade|All Electronic Distribution Formats',
    '2JAR' => 'Advertising|Periodicals|Magazine, Corporate|Printed',
    '2GIN' => 'Advertising|Periodicals|Magazine, Corporate|Internet Website',
    '2GIE' => 'Advertising|Periodicals|Magazine, Corporate|Internet Downloadable File',
    '2AGO' => 'Advertising|Periodicals|Magazine, Corporate|Internet Email',
    '2GNU' => 'Advertising|Periodicals|Magazine, Corporate|All Internet Distribution Formats',
    '2GIP' => 'Advertising|Periodicals|Magazine, Corporate|Recordable Media',
    '2GIT' => 'Advertising|Periodicals|Magazine, Corporate|All Electronic Distribution Formats',
    '2DEB' => 'Advertising|Periodicals|Magazine, Education|Printed',
    '2GUM' => 'Advertising|Periodicals|Magazine, Education|Internet Website',
    '2GUL' => 'Advertising|Periodicals|Magazine, Education|Internet Downloadable File',
    '2AHA' => 'Advertising|Periodicals|Magazine, Education|Internet Email',
    '2GYP' => 'Advertising|Periodicals|Magazine, Education|All Internet Distribution Formats',
    '2GUV' => 'Advertising|Periodicals|Magazine, Education|Recordable Media',
    '2GUY' => 'Advertising|Periodicals|Magazine, Education|All Electronic Distribution Formats',
    '2CUS' => 'Advertising|Periodicals|Magazine, Custom Published|Printed',
    '2GOR' => 'Advertising|Periodicals|Magazine, Custom Published|Internet Website',
    '2GOS' => 'Advertising|Periodicals|Magazine, Custom Published|Internet Downloadable File',
    '2AIL' => 'Advertising|Periodicals|Magazine, Custom Published|Internet Email',
    '2GOX' => 'Advertising|Periodicals|Magazine, Custom Published|All Internet Distribution Formats',
    '2ALP' => 'Advertising|Periodicals|Magazine, Custom Published|Recordable Media',
    '2AMI' => 'Advertising|Periodicals|Magazine, Custom Published|All Electronic Distribution Formats',
    '2BEN' => 'Advertising|Periodicals|Magazine, Advertorial|Printed',
    '2BAP' => 'Advertising|Periodicals|Magazine, Advertorial|Internet Website',
    '2BAM' => 'Advertising|Periodicals|Magazine, Advertorial|Internet Downloadable File',
    '2BOP' => 'Advertising|Periodicals|Magazine, Advertorial|Internet Email',
    '2GOB' => 'Advertising|Periodicals|Magazine, Advertorial|All Internet Distribution Formats',
    '2DUH' => 'Advertising|Periodicals|Magazine, Advertorial|All Electronic Distribution Formats',
    '2DUP' => 'Advertising|Periodicals|Magazine Reprints, All Types|Printed',
    '2HAS' => 'Advertising|Periodicals|Magazine Reprints, All Types|Internet Website',
    '2HAP' => 'Advertising|Periodicals|Magazine Reprints, All Types|Internet Downloadable File',
    '2AMP' => 'Advertising|Periodicals|Magazine Reprints, All Types|Internet Email',
    '2HEH' => 'Advertising|Periodicals|Magazine Reprints, All Types|All Internet Distribution Formats',
    '2HAW' => 'Advertising|Periodicals|Magazine Reprints, All Types|Recordable Media',
    '2HAY' => 'Advertising|Periodicals|Magazine Reprints, All Types|All Electronic Distribution Formats',
    '2MOO' => 'Advertising|Periodicals|Magazine, All Types|Printed',
    '2ANA' => 'Advertising|Periodicals|Magazine, All Types|Internet Website',
    '2GOO' => 'Advertising|Periodicals|Magazine, All Types|Recordable Media',
    '2ANY' => 'Advertising|Periodicals|Magazine, All Types|Internet Downloadable File',
    '2APE' => 'Advertising|Periodicals|Magazine, All Types|Internet Email',
    '2APT' => 'Advertising|Periodicals|Magazine, All Types|All Internet Distribution Formats',
    '2ARK' => 'Advertising|Periodicals|Magazine, All Types|All Electronic Distribution Formats',
    '2KEG' => 'Advertising|Periodicals|Newspaper, Weekly Supplement|Printed',
    '2HOB' => 'Advertising|Periodicals|Newspaper, Weekly Supplement|Internet Website',
    '2HOD' => 'Advertising|Periodicals|Newspaper, Weekly Supplement|Internet Downloadable File',
    '2ARM' => 'Advertising|Periodicals|Newspaper, Weekly Supplement|Internet Email',
    '2HMM' => 'Advertising|Periodicals|Newspaper, Weekly Supplement|All Internet Distribution Formats',
    '2HOE' => 'Advertising|Periodicals|Newspaper, Weekly Supplement|Recordable Media',
    '2HIS' => 'Advertising|Periodicals|Newspaper, Weekly Supplement|All Electronic Distribution Formats',
    '2KEN' => 'Advertising|Periodicals|Newspaper, Tabloid|Printed',
    '2HOY' => 'Advertising|Periodicals|Newspaper, Tabloid|Internet Website',
    '2HUH' => 'Advertising|Periodicals|Newspaper, Tabloid|All Internet Distribution Formats',
    '2ASH' => 'Advertising|Periodicals|Newspaper, Tabloid|Internet Downloadable File',
    '2BAY' => 'Advertising|Periodicals|Newspaper, Tabloid|Internet Email',
    '2HUE' => 'Advertising|Periodicals|Newspaper, Tabloid|Recordable Media',
    '2HOW' => 'Advertising|Periodicals|Newspaper, Tabloid|All Electronic Distribution Formats',
    '2NEW' => 'Advertising|Periodicals|Newspaper, All Types|Printed',
    '2HID' => 'Advertising|Periodicals|Newspaper, All Types|Internet Website',
    '2HIE' => 'Advertising|Periodicals|Newspaper, All Types|Internet Downloadable File',
    '2BED' => 'Advertising|Periodicals|Newspaper, All Types|Internet Email',
    '2HIC' => 'Advertising|Periodicals|Newspaper, All Types|All Internet Distribution Formats',
    '2HIM' => 'Advertising|Periodicals|Newspaper, All Types|Recordable Media',
    '2HEY' => 'Advertising|Periodicals|Newspaper, All Types|All Electronic Distribution Formats',
    '2JAY' => 'Advertising|Periodicals|Newsletter, All Types|Printed',
    '2HES' => 'Advertising|Periodicals|Newsletter, All Types|Internet Website',
    '2HEP' => 'Advertising|Periodicals|Newsletter, All Types|Internet Downloadable File',
    '2HEW' => 'Advertising|Periodicals|Newsletter, All Types|All Internet Distribution Formats',
    '2BID' => 'Advertising|Periodicals|Newsletter, All Types|Internet Email',
    '2HET' => 'Advertising|Periodicals|Newsletter, All Types|Recordable Media',
    '2HEN' => 'Advertising|Periodicals|Newsletter, All Types|All Electronic Distribution Formats',
    '2BAH' => 'Advertising|Periodicals|Quarterly Report|Printed',
    '2HUT' => 'Advertising|Periodicals|Quarterly Report|Internet Website',
    '2HUP' => 'Advertising|Periodicals|Quarterly Report|Internet Downloadable File',
    '2BAL' => 'Advertising|Periodicals|Quarterly Report|Internet Email',
    '2JEE' => 'Advertising|Periodicals|Quarterly Report|All Internet Distribution Formats',
    '2HYP' => 'Advertising|Periodicals|Quarterly Report|Recordable Media',
    '2JAG' => 'Advertising|Periodicals|Quarterly Report|All Electronic Distribution Formats',
    '2ANN' => 'Advertising|Periodicals|Annual Report|Printed',
    '2FON' => 'Advertising|Periodicals|Annual Report|Internet Website',
    '2FOH' => 'Advertising|Periodicals|Annual Report|Internet Downloadable File',
    '2FOP' => 'Advertising|Periodicals|Annual Report|Internet Email',
    '2FOY' => 'Advertising|Periodicals|Annual Report|All Internet Distribution Formats',
    '2BAS' => 'Advertising|Periodicals|Annual Report|Recordable Media',
    '2FOU' => 'Advertising|Periodicals|Annual Report|All Electronic Distribution Formats',
    '2KIP' => 'Advertising|Periodicals|Program Advertising|Printed',
    '2DEE' => 'Advertising|Periodicals|Wrapper|Printed',
    '2FUD' => 'Advertising|Periodicals|Cover Wrap|Printed',
    '2FUB' => 'Advertising|Periodicals|Belly Band|Printed',
    '2BAT' => 'Advertising|Periodicals|Free Standing Insert, All Insert Types|Printed',
    '2HIP' => 'Advertising|Periodicals|Free Standing Insert, Advertorial Insert|Printed',
    '2BAG' => 'Advertising|Periodicals|All Periodical Types|Printed',
    '2FIZ' => 'Advertising|Periodicals|All Periodical Types|Internet Website',
    '2BOD' => 'Advertising|Periodicals|All Periodical Types|Internet Downloadable File',
    '2BOW' => 'Advertising|Periodicals|All Periodical Types|Internet Email',
    '2FOB' => 'Advertising|Periodicals|All Periodical Types|All Internet Distribution Formats',
    '2FIR' => 'Advertising|Periodicals|All Periodical Types|Recordable Media',
    '2FLU' => 'Advertising|Periodicals|All Periodical Types|All Electronic Distribution Formats',
    '2EDH' => 'Advertising|Marketing Materials|Artist\'s Reference, All Types|Printed',
    '2ECU' => 'Advertising|Marketing Materials|Artist\'s Reference, All Types|All Internet Distribution Formats',
    '2DOT' => 'Advertising|Marketing Materials|Artist\'s Reference, All Types|All Electronic Distribution Formats',
    '2MMA' => 'Advertising|Marketing Materials|Bill Insert|Printed',
    '2MMB' => 'Advertising|Marketing Materials|Blow In Card|Printed',
    '2MMC' => 'Advertising|Marketing Materials|Bound-in Insert|Printed',
    '2MMD' => 'Advertising|Marketing Materials|Broadside|Printed',
    '2EGG' => 'Advertising|Marketing Materials|Brochure|Printed',
    '2EFF' => 'Advertising|Marketing Materials|Brochure|Internet Downloadable File',
    '2EFS' => 'Advertising|Marketing Materials|Brochure|Internet Email',
    '2EFT' => 'Advertising|Marketing Materials|Brochure|Recordable Media',
    '2MME' => 'Advertising|Marketing Materials|Buckslip|Printed',
    '2MMF' => 'Advertising|Marketing Materials|Business Card|Printed',
    '2MMG' => 'Advertising|Marketing Materials|Business Envelope|Printed',
    '2BIZ' => 'Advertising|Marketing Materials|Card, Business Greeting Card|Printed',
    '2MMI' => 'Advertising|Marketing Materials|Business Invitation|Printed',
    '2MMJ' => 'Advertising|Marketing Materials|Business Reply Card|Printed',
    '2MMK' => 'Advertising|Marketing Materials|Business Reply Envelope|Printed',
    '2MML' => 'Advertising|Marketing Materials|Business Stationery|Printed',
    '2FIG' => 'Advertising|Marketing Materials|Catalog|Printed',
    '2ELL' => 'Advertising|Marketing Materials|Catalog|Internet Downloadable File',
    '2ELS' => 'Advertising|Marketing Materials|Catalog|Internet Email',
    '2ELF' => 'Advertising|Marketing Materials|Catalog|Recordable Media',
    '2EDU' => 'Advertising|Marketing Materials|CD ROM|Recordable Media',
    '2MMM' => 'Advertising|Marketing Materials|Compliment Slip|Printed',
    '2MMN' => 'Advertising|Marketing Materials|Coupon|Printed',
    '2EGO' => 'Advertising|Marketing Materials|Coupon|Internet Downloadable File',
    '2ENG' => 'Advertising|Marketing Materials|Coupon|Internet Email',
    '2ENS' => 'Advertising|Marketing Materials|Coupon|Recordable Media',
    '2MMP' => 'Advertising|Marketing Materials|Coupon Packs|Printed',
    '2DVA' => 'Advertising|Marketing Materials|DVD|Recordable Media',
    '2MMQ' => 'Advertising|Marketing Materials|Flyaway Card|Printed',
    '2FLY' => 'Advertising|Marketing Materials|Flyer|Printed',
    '2MMR' => 'Advertising|Marketing Materials|Leaflet|Printed',
    '2MMS' => 'Advertising|Marketing Materials|Magalog|Printed',
    '2END' => 'Advertising|Marketing Materials|Magalog|Internet Downloadable File',
    '2EON' => 'Advertising|Marketing Materials|Magalog|Internet Email',
    '2FAX' => 'Advertising|Marketing Materials|Magalog|Recordable Media',
    '2MMT' => 'Advertising|Marketing Materials|One Sheet|Printed',
    '2MMU' => 'Advertising|Marketing Materials|Onsert|Printed',
    '2MMV' => 'Advertising|Marketing Materials|Polybag|Printed',
    '2BUN' => 'Advertising|Marketing Materials|Promo Card|Printed',
    '2MMW' => 'Advertising|Marketing Materials|Promotional Calendar, One Page|Printed',
    '2FAS' => 'Advertising|Marketing Materials|Promotional Calendar, One Page|Internet Downloadable File',
    '2FAY' => 'Advertising|Marketing Materials|Promotional Calendar, One Page|Internet Email',
    '2GUT' => 'Advertising|Marketing Materials|Promotional Calendar, One Page|Recordable Media',
    '2EWE' => 'Advertising|Marketing Materials|Promotional Calendar, One Page|All Electronic Distribution Formats',
    '2YAM' => 'Advertising|Marketing Materials|Promotional Calendar, Multi-Page|Printed',
    '2ESS' => 'Advertising|Marketing Materials|Promotional Calendar, Multi-Page|Internet Downloadable File',
    '2ETA' => 'Advertising|Marketing Materials|Promotional Calendar, Multi-Page|Internet Email',
    '2HAD' => 'Advertising|Marketing Materials|Promotional Calendar, Multi-Page|Recordable Media',
    '2FET' => 'Advertising|Marketing Materials|Promotional E-card|Internet Email',
    '2FEU' => 'Advertising|Marketing Materials|Promotional E-card|All Internet Distribution Formats',
    '2FEH' => 'Advertising|Marketing Materials|Promotional E-card|Recordable Media',
    '2FEN' => 'Advertising|Marketing Materials|Promotional E-card|Internet Downloadable File',
    '2MMX' => 'Advertising|Marketing Materials|Promotional Envelope|Printed',
    '2NUT' => 'Advertising|Marketing Materials|Promotional Postcard|Printed',
    '2KAF' => 'Advertising|Marketing Materials|Public Relations, Press Kit|Printed',
    '2JUT' => 'Advertising|Marketing Materials|Public Relations, Press Kit|Internet Website',
    '2JUN' => 'Advertising|Marketing Materials|Public Relations, Press Kit|Internet Downloadable File',
    '2JUS' => 'Advertising|Marketing Materials|Public Relations, Press Kit|Internet Email',
    '2JOW' => 'Advertising|Marketing Materials|Public Relations, Press Kit|All Internet Distribution Formats',
    '2JOB' => 'Advertising|Marketing Materials|Public Relations, Press Kit|Recordable Media',
    '2JOE' => 'Advertising|Marketing Materials|Public Relations, Press Kit|All Electronic Distribution Formats',
    '2KAB' => 'Advertising|Marketing Materials|Public Relations, Press Kit|Television Broadcast',
    '2KEX' => 'Advertising|Marketing Materials|Public Relations, Press Release|Printed',
    '2KEA' => 'Advertising|Marketing Materials|Public Relations, Press Release|Internet Website',
    '2KAT' => 'Advertising|Marketing Materials|Public Relations, Press Release|Internet Downloadable File',
    '2KAY' => 'Advertising|Marketing Materials|Public Relations, Press Release|Internet Email',
    '2KAS' => 'Advertising|Marketing Materials|Public Relations, Press Release|Recordable Media',
    '2KID' => 'Advertising|Marketing Materials|Public Relations, Press Release|All Electronic Distribution Formats',
    '2KEF' => 'Advertising|Marketing Materials|Public Relations, Press Release|Television Broadcast',
    '2JIB' => 'Advertising|Marketing Materials|Public Relations, All Types|Printed',
    '2MMY' => 'Advertising|Marketing Materials|Sales Kit|Printed',
    '2MMZ' => 'Advertising|Marketing Materials|Self Mailer|Printed',
    '2JAM' => 'Advertising|Marketing Materials|All Marketing Material Types|Printed',
    '2HAM' => 'Advertising|Marketing Materials|All Marketing Material Types|Internet Downloadable File',
    '2DYE' => 'Advertising|Marketing Materials|All Marketing Material Types|Internet Email',
    '2DUO' => 'Advertising|Marketing Materials|All Marketing Material Types|Recordable Media',
    '2BEG' => 'Advertising|Book|Retail Book, Directory|Printed',
    '2EAC' => 'Advertising|Book|Retail Book, Directory|E-Book in Internet Website',
    '2EAD' => 'Advertising|Book|Retail Book, Directory|E-Book in Internet Downloadable File',
    '2EAE' => 'Advertising|Book|Retail Book, Directory|All E-Book Internet Distribution Formats',
    '2EAF' => 'Advertising|Book|Retail Book, Directory|E-Book on Recordable Media',
    '2EAG' => 'Advertising|Book|Retail Book, Directory|All E-Book Distribution Formats',
    '2ELM' => 'Advertising|Book|Textbook, All Types|Printed',
    '2EAH' => 'Advertising|Book|Textbook, All Types|E-Book in Internet Website',
    '2EAJ' => 'Advertising|Book|Textbook, All Types|E-Book in Internet Downloadable File',
    '2EAK' => 'Advertising|Book|Textbook, All Types|All E-Book Internet Distribution Formats',
    '2EAL' => 'Advertising|Book|Textbook, All Types|E-Book on Recordable Media',
    '2EAM' => 'Advertising|Book|Textbook, All Types|All E-Book Distribution Formats',
    '2HOG' => 'Advertising|Book|All Book Types|Printed',
    '2EAN' => 'Advertising|Book|All Book Types|All E-Book Internet Distribution Formats',
    '2EAP' => 'Advertising|Book|All Book Types|All E-Book Distribution Formats',
    '2HEM' => 'Advertising|Display|Billboard, Bulletin|Printed',
    '2BUR' => 'Advertising|Display|Billboard, Bulletin|Electronic Display',
    '2DAG' => 'Advertising|Display|Billboard, Spectacular|Printed',
    '2DAD' => 'Advertising|Display|Billboard, Spectacular|Electronic Display',
    '2DAK' => 'Advertising|Display|Billboard, Wallscape|Printed',
    '2DAH' => 'Advertising|Display|Billboard, Wallscape|Electronic Display',
    '2DAB' => 'Advertising|Display|Billboard, Rotating Billboard|Printed',
    '2BLD' => 'Advertising|Display|Billboard, Building Wrap|Printed',
    '2BYS' => 'Advertising|Display|Billboard, Mobile Billboard|Printed',
    '2BYE' => 'Advertising|Display|Billboard, Mobile Billboard|Electronic Display',
    '2BIL' => 'Advertising|Display|Billboard, All Types|Printed',
    '2BUD' => 'Advertising|Display|Billboard, All Types|Electronic Display',
    '2BRR' => 'Advertising|Display|Banner, Backdrop|Printed',
    '2BUB' => 'Advertising|Display|Banner, Background|Printed',
    '2BRA' => 'Advertising|Display|Banner, Airborne Display|Printed',
    '2BRO' => 'Advertising|Display|Banner, All Types|Printed',
    '2BAN' => 'Advertising|Display|Banner, All Types|Electronic Display',
    '2CAR' => 'Advertising|Display|Shopping Cart|Printed',
    '2CRT' => 'Advertising|Display|Shopping Cart|Electronic Display',
    '2FAD' => 'Advertising|Display|Poster, Movie Poster|Printed',
    '2HON' => 'Advertising|Display|Poster, Movie Poster|Electronic Display',
    '2DIB' => 'Advertising|Display|Poster, Door Side Poster|Printed',
    '2DEY' => 'Advertising|Display|Poster, Door Side Poster|Electronic Display',
    '2DEV' => 'Advertising|Display|Poster, Corporate Poster|Printed',
    '2DEN' => 'Advertising|Display|Poster, Corporate Poster|Electronic Display',
    '2ELE' => 'Advertising|Display|Poster, Elevator Advertising|Printed',
    '2DID' => 'Advertising|Display|Poster, Elevator Advertising|Electronic Display',
    '2DRY' => 'Advertising|Display|Poster, Restroom Poster|Printed',
    '2WET' => 'Advertising|Display|Poster, Restroom Poster|Electronic Display',
    '2DEL' => 'Advertising|Display|Poster, Backlit Print|Printed',
    '2DEX' => 'Advertising|Display|Poster, Display Chrome|Printed',
    '2FAT' => 'Advertising|Display|Poster, All Types|Printed',
    '2DAW' => 'Advertising|Display|Poster, All Types|Electronic Display',
    '2MOB' => 'Advertising|Display|Store Display, In-Store Poster|Printed',
    '2DIT' => 'Advertising|Display|Store Display, In-Store Poster|Electronic Display',
    '2FOG' => 'Advertising|Display|Store Display, All Display Types|Printed',
    '2DIN' => 'Advertising|Display|Store Display, All Display Types|Electronic Display',
    '2HOT' => 'Advertising|Display|Terminal Advertising, Airport Display|Printed',
    '2JOU' => 'Advertising|Display|Terminal Advertising, Airport Display|Electronic Display',
    '2DOM' => 'Advertising|Display|Terminal Advertising, Bus Stop Advertising|Printed',
    '2DOL' => 'Advertising|Display|Terminal Advertising, Bus Stop Advertising|Electronic Display',
    '2DOR' => 'Advertising|Display|Terminal Advertising, Ferry Terminal Advertising|Printed',
    '2DON' => 'Advertising|Display|Terminal Advertising, Ferry Terminal Advertising|Electronic Display',
    '2TAN' => 'Advertising|Display|Terminal Advertising, Shelter Advertising|Printed',
    '2DOS' => 'Advertising|Display|Terminal Advertising, Shelter Advertising|Electronic Display',
    '2DUB' => 'Advertising|Display|Terminal Advertising, Station Poster|Printed',
    '2DOW' => 'Advertising|Display|Terminal Advertising, Station Poster|Electronic Display',
    '2DUG' => 'Advertising|Display|Terminal Advertising, Subway Terminal Advertising|Printed',
    '2DUE' => 'Advertising|Display|Terminal Advertising, Subway Terminal Advertising|Electronic Display',
    '2DUN' => 'Advertising|Display|Terminal Advertising, Train Terminal Advertising|Printed',
    '2DUI' => 'Advertising|Display|Terminal Advertising, Train Terminal Advertising|Electronic Display',
    '2JOY' => 'Advertising|Display|Terminal Advertising, All Types|Printed',
    '2DOE' => 'Advertising|Display|Terminal Advertising, All Types|Electronic Display',
    '2TAX' => 'Advertising|Display|Transit Advertising, Taxi Advertising|Printed',
    '2BUS' => 'Advertising|Display|Transit Advertising, Bus Panel|Printed',
    '2TRC' => 'Advertising|Display|Transit Advertising, Bus Panel|Electronic Display',
    '2TRE' => 'Advertising|Display|Transit Advertising, Bus Poster|Printed',
    '2BPE' => 'Advertising|Display|Transit Advertising, Bus Poster|Electronic Display',
    '2TRG' => 'Advertising|Display|Transit Advertising, Bus Rear Display|Printed',
    '2TRF' => 'Advertising|Display|Transit Advertising, Bus Rear Display|Electronic Display',
    '2TRN' => 'Advertising|Display|Transit Advertising, Bus Wrap|Printed',
    '2TRM' => 'Advertising|Display|Transit Advertising, Bus Wrap|Electronic Display',
    '2TRJ' => 'Advertising|Display|Transit Advertising, Subway Advertising|Printed',
    '2TRI' => 'Advertising|Display|Transit Advertising, Subway Advertising|Electronic Display',
    '2TRL' => 'Advertising|Display|Transit Advertising, Train Advertising|Printed',
    '2TRK' => 'Advertising|Display|Transit Advertising, Train Advertising|Electronic Display',
    '2TRQ' => 'Advertising|Display|Transit Advertising, Commercial Vehicles|Printed',
    '2TRR' => 'Advertising|Display|Transit Advertising, Commercial Vehicles|Electronic Display',
    '2FER' => 'Advertising|Display|Transit Advertising, Ferry Advertising|Printed',
    '2TRH' => 'Advertising|Display|Transit Advertising, Ferry Advertising|Electronic Display',
    '2TRA' => 'Advertising|Display|Transit Advertising, All Types|Printed',
    '2TRB' => 'Advertising|Display|Transit Advertising, All Types|Electronic Display',
    '2DAM' => 'Advertising|Display|Event, Stadium Advertising|Printed',
    '2DAL' => 'Advertising|Display|Event, Stadium Advertising|Electronic Display',
    '2DIS' => 'Advertising|Display|Event, Trade Show Display|Printed',
    '2DAP' => 'Advertising|Display|Event, Trade Show Display|Electronic Display',
    '2BIG' => 'Advertising|Display|All Display Types|Printed',
    '2BOG' => 'Advertising|Display|All Display Types|Electronic Display',
    '2BOY' => 'Advertising|Art|Art Display, Display Print|Printed',
    '2ART' => 'Advertising|Art|Art Display, All Art Types|Printed',
    '2BEL' => 'Advertising|Art|Art Display, All Art Types|Internet Website',
    '2BEY' => 'Advertising|Art|Art Display, All Art Types|All Internet Distribution Formats',
    '2BOS' => 'Advertising|Art|Art Display, All Art Types|Electronic Display',
    '2BIS' => 'Advertising|Art|Art Display, All Art Types|All Electronic Distribution Formats',
    '2ADH' => 'Advertising|Point of Purchase|Adhesive Tag|Printed',
    '2BOT' => 'Advertising|Point of Purchase|Bottlenecker|Printed',
    '2CAS' => 'Advertising|Point of Purchase|Case Card|Printed',
    '2COU' => 'Advertising|Point of Purchase|Counter Card|Printed',
    '2BUY' => 'Advertising|Point of Purchase|Floor Graphic|Printed',
    '2TAG' => 'Advertising|Point of Purchase|Hang Tag|Printed',
    '2GYM' => 'Advertising|Point of Purchase|Kiosk, Interactive Kiosk|Printed',
    '2GUN' => 'Advertising|Point of Purchase|Kiosk, Interactive Kiosk|Electronic Display',
    '2JUG' => 'Advertising|Point of Purchase|Kiosk, Telephone Kiosk|Printed',
    '2FIN' => 'Advertising|Point of Purchase|Kiosk, All Types|Printed',
    '2KIO' => 'Advertising|Point of Purchase|Kiosk, All Types|Electronic Display',
    '2MEN' => 'Advertising|Point of Purchase|Menu|Printed',
    '2TAL' => 'Advertising|Point of Purchase|Shelf Talker|Printed',
    '2TIN' => 'Advertising|Point of Purchase|Slip Case|Printed',
    '2WAX' => 'Advertising|Point of Purchase|Table Tent|Printed',
    '2BIN' => 'Advertising|Point of Purchase|All Point of Purchase Types|Printed',
    '2BIB' => 'Advertising|Point of Purchase|All Point of Purchase Types|Electronic Display',
    '2BAR' => 'Advertising|Point of Purchase|All Point of Purchase Types|All Electronic Distribution Formats',
    '2MAR' => 'Advertising|Website|Web Page, Design Element|Internet Website',
    '2MAX' => 'Advertising|Website|Web Page, Design Element|All Internet Distribution Formats',
    '2MAS' => 'Advertising|Website|Web Page, Design Element|Recordable Media',
    '2MAW' => 'Advertising|Website|Web Page, Design Element|All Electronic Distribution Formats',
    '2MED' => 'Advertising|Website|Web Page, Web Banner Ad|Internet Website',
    '2MEW' => 'Advertising|Website|Web Page, Web Banner Ad|All Internet Distribution Formats',
    '2MEG' => 'Advertising|Website|Web Page, Web Banner Ad|Recordable Media',
    '2MEM' => 'Advertising|Website|Web Page, Web Banner Ad|All Electronic Distribution Formats',
    '2JWL' => 'Advertising|Website|Web Page, Web Interstitial Ad|Internet Website',
    '2MIG' => 'Advertising|Website|Web Page, Web Interstitial Ad|All Internet Distribution Formats',
    '2MIB' => 'Advertising|Website|Web Page, Web Interstitial Ad|Recordable Media',
    '2MID' => 'Advertising|Website|Web Page, Web Interstitial Ad|All Electronic Distribution Formats',
    '2WEB' => 'Advertising|Website|Web Page, All Types|Internet Website',
    '2KUE' => 'Advertising|Website|Web Page, All Types|All Internet Distribution Formats',
    '2MAC' => 'Advertising|Website|Web Page, All Types|Recordable Media',
    '2MAE' => 'Advertising|Website|Web Page, All Types|All Electronic Distribution Formats',
    '2MIL' => 'Advertising|Website|Webcast, All Types|Internet Website',
    '2EMA' => 'Advertising|Email|All Email Types|Internet Email',
    '2ZUS' => 'Advertising|Mobile|All Mobile Types|Mobile',
    '2FUR' => 'Advertising|Live Presentation|Sales Presentation|Projected Display',
    '2EYE' => 'Advertising|Live Presentation|Panel Presentation|Projected Display',
    '2TOE' => 'Advertising|Live Presentation|Trade Show Presentation|Projected Display',
    '2JAW' => 'Advertising|Live Presentation|Stage Performance|Projected Display',
    '2EAR' => 'Advertising|Live Presentation|All Live Presentation Types|All Electronic Distribution Formats',
    '2FID' => 'Advertising|Merchandise|Apparel, T-Shirts|Printed or Woven',
    '2FEZ' => 'Advertising|Merchandise|Apparel, General Apparel|Printed or Woven',
    '2FIE' => 'Advertising|Merchandise|Folder|Printed',
    '2AAY' => 'Advertising|All Media Types|Promotional Reproduction of Licensed Usage in Context|All Distribution Formats',
    '2AAA' => 'Advertising|All Media Types|All Formats|All Distribution Formats',
    '2GEM' => 'Editorial|Periodicals|Magazine, Consumer|Printed',
    '2GDW' => 'Editorial|Periodicals|Magazine, Consumer|Internet Website',
    '2GDY' => 'Editorial|Periodicals|Magazine, Consumer|Internet Downloadable File',
    '2KIN' => 'Editorial|Periodicals|Magazine, Consumer|Internet Email',
    '2GEA' => 'Editorial|Periodicals|Magazine, Consumer|All Internet Distribution Formats',
    '2GDV' => 'Editorial|Periodicals|Magazine, Consumer|Recordable Media',
    '2GDZ' => 'Editorial|Periodicals|Magazine, Consumer|All Electronic Distribution Formats',
    '2MOM' => 'Editorial|Periodicals|Magazine, Trade|Printed',
    '2MOW' => 'Editorial|Periodicals|Magazine, Trade|Internet Website',
    '2MUT' => 'Editorial|Periodicals|Magazine, Trade|Internet Downloadable File',
    '2NAB' => 'Editorial|Periodicals|Magazine, Trade|Internet Email',
    '2NAG' => 'Editorial|Periodicals|Magazine, Trade|All Internet Distribution Formats',
    '2GFD' => 'Editorial|Periodicals|Magazine, Trade|Recordable Media',
    '2NOB' => 'Editorial|Periodicals|Magazine, Trade|All Electronic Distribution Formats',
    '2MAY' => 'Editorial|Periodicals|Magazine, Education|Printed',
    '2GEP' => 'Editorial|Periodicals|Magazine, Education|Internet Website',
    '2GEQ' => 'Editorial|Periodicals|Magazine, Education|Internet Downloadable File',
    '2NOW' => 'Editorial|Periodicals|Magazine, Education|Internet Email',
    '2GES' => 'Editorial|Periodicals|Magazine, Education|All Internet Distribution Formats',
    '2GEK' => 'Editorial|Periodicals|Magazine, Education|Recordable Media',
    '2GER' => 'Editorial|Periodicals|Magazine, Education|All Electronic Distribution Formats',
    '2MOP' => 'Editorial|Periodicals|Magazine, Custom Published|Printed',
    '2GEF' => 'Editorial|Periodicals|Magazine, Custom Published|Internet Website',
    '2GEG' => 'Editorial|Periodicals|Magazine, Custom Published|Internet Downloadable File',
    '2NUN' => 'Editorial|Periodicals|Magazine, Custom Published|Internet Email',
    '2GEI' => 'Editorial|Periodicals|Magazine, Custom Published|All Internet Distribution Formats',
    '2GEC' => 'Editorial|Periodicals|Magazine, Custom Published|Recordable Media',
    '2GEH' => 'Editorial|Periodicals|Magazine, Custom Published|All Electronic Distribution Formats',
    '2MEL' => 'Editorial|Periodicals|Magazine, Partworks|Printed',
    '2GEV' => 'Editorial|Periodicals|Magazine, Partworks|Internet Website',
    '2GEW' => 'Editorial|Periodicals|Magazine, Partworks|Internet Downloadable File',
    '2TAB' => 'Editorial|Periodicals|Magazine, Partworks|Internet Email',
    '2GFB' => 'Editorial|Periodicals|Magazine, Partworks|All Internet Distribution Formats',
    '2GEZ' => 'Editorial|Periodicals|Magazine, Partworks|Recordable Media',
    '2GFA' => 'Editorial|Periodicals|Magazine, Partworks|All Electronic Distribution Formats',
    '2WIG' => 'Editorial|Periodicals|Scholarly Journal|Printed',
    '2GGF' => 'Editorial|Periodicals|Scholarly Journal|Internet Website',
    '2GGG' => 'Editorial|Periodicals|Scholarly Journal|Internet Downloadable File',
    '2TAD' => 'Editorial|Periodicals|Scholarly Journal|Internet Email',
    '2GGI' => 'Editorial|Periodicals|Scholarly Journal|All Internet Distribution Formats',
    '2GGD' => 'Editorial|Periodicals|Scholarly Journal|Recordable Media',
    '2GGH' => 'Editorial|Periodicals|Scholarly Journal|All Electronic Distribution Formats',
    '2MAG' => 'Editorial|Periodicals|Magazine, All Types|Printed',
    '2GDP' => 'Editorial|Periodicals|Magazine, All Types|Internet Website',
    '2GDM' => 'Editorial|Periodicals|Magazine, All Types|Internet Downloadable File',
    '2GDQ' => 'Editorial|Periodicals|Magazine, All Types|Internet Email',
    '2GDT' => 'Editorial|Periodicals|Magazine, All Types|All Internet Distribution Formats',
    '2GDR' => 'Editorial|Periodicals|Magazine, All Types|Recordable Media',
    '2GDS' => 'Editorial|Periodicals|Magazine, All Types|All Electronic Distribution Formats',
    '2JET' => 'Editorial|Periodicals|Newspaper, All Types|Printed',
    '2GFN' => 'Editorial|Periodicals|Newspaper, All Types|Internet Website',
    '2GFP' => 'Editorial|Periodicals|Newspaper, All Types|Internet Downloadable File',
    '2TAJ' => 'Editorial|Periodicals|Newspaper, All Types|Internet Email',
    '2GFR' => 'Editorial|Periodicals|Newspaper, All Types|All Internet Distribution Formats',
    '2GFM' => 'Editorial|Periodicals|Newspaper, All Types|Recordable Media',
    '2GFQ' => 'Editorial|Periodicals|Newspaper, All Types|All Electronic Distribution Formats',
    '2TUB' => 'Editorial|Periodicals|Newspaper, Weekly Supplement|Printed',
    '2GFW' => 'Editorial|Periodicals|Newspaper, Weekly Supplement|Internet Website',
    '2GFY' => 'Editorial|Periodicals|Newspaper, Weekly Supplement|Internet Downloadable File',
    '2TAO' => 'Editorial|Periodicals|Newspaper, Weekly Supplement|Internet Email',
    '2GFV' => 'Editorial|Periodicals|Newspaper, Weekly Supplement|All Internet Distribution Formats',
    '2GFS' => 'Editorial|Periodicals|Newspaper, Weekly Supplement|Recordable Media',
    '2GFU' => 'Editorial|Periodicals|Newspaper, Weekly Supplement|All Electronic Distribution Formats',
    '2FAN' => 'Editorial|Periodicals|Newspaper, Tabloid|Printed',
    '2TEX' => 'Editorial|Periodicals|Newspaper, Tabloid|Internet Website',
    '2GGA' => 'Editorial|Periodicals|Newspaper, Tabloid|Internet Downloadable File',
    '2TIP' => 'Editorial|Periodicals|Newspaper, Tabloid|Internet Email',
    '2TON' => 'Editorial|Periodicals|Newspaper, Tabloid|All Internet Distribution Formats',
    '2TIE' => 'Editorial|Periodicals|Newspaper, Tabloid|Recordable Media',
    '2TOT' => 'Editorial|Periodicals|Newspaper, Tabloid|All Electronic Distribution Formats',
    '2TOM' => 'Editorial|Periodicals|Newsletter|Printed',
    '2GFH' => 'Editorial|Periodicals|Newsletter|Internet Website',
    '2GFI' => 'Editorial|Periodicals|Newsletter|Internet Downloadable File',
    '2NEL' => 'Editorial|Periodicals|Newsletter|Internet Email',
    '2GFK' => 'Editorial|Periodicals|Newsletter|All Internet Distribution Formats',
    '2GFG' => 'Editorial|Periodicals|Newsletter|Recordable Media',
    '2GFJ' => 'Editorial|Periodicals|Newsletter|All Electronic Distribution Formats',
    '2BUG' => 'Editorial|Book|Retail Book, Children\'s Book|Printed',
    '2EPC' => 'Editorial|Book|Retail Book, Children\'s Book|E-Book in Internet Website',
    '2EBE' => 'Editorial|Book|Retail Book, Children\'s Book|E-Book in Internet Downloadable File',
    '2EGB' => 'Editorial|Book|Retail Book, Children\'s Book|E-Book in Internet Email',
    '2EKB' => 'Editorial|Book|Retail Book, Children\'s Book|All E-Book Internet Distribution Formats',
    '2ERB' => 'Editorial|Book|Retail Book, Children\'s Book|E-Book on Recordable Media',
    '2EGC' => 'Editorial|Book|Retail Book, Children\'s Book|All E-Book Distribution Formats',
    '2TAE' => 'Editorial|Book|Retail Book, Coffee Table Book|Printed',
    '2TAV' => 'Editorial|Book|Retail Book, Concept Book|Printed',
    '2EPD' => 'Editorial|Book|Retail Book, Concept Book|E-Book in Internet Website',
    '2EBF' => 'Editorial|Book|Retail Book, Concept Book|E-Book in Internet Downloadable File',
    '2EGD' => 'Editorial|Book|Retail Book, Concept Book|E-Book in Internet Email',
    '2EKC' => 'Editorial|Book|Retail Book, Concept Book|All E-Book Internet Distribution Formats',
    '2ERC' => 'Editorial|Book|Retail Book, Concept Book|E-Book on Recordable Media',
    '2EGE' => 'Editorial|Book|Retail Book, Concept Book|All E-Book Distribution Formats',
    '2DIR' => 'Editorial|Book|Retail Book, Directory|Printed',
    '2EPE' => 'Editorial|Book|Retail Book, Directory|E-Book in Internet Website',
    '2EBG' => 'Editorial|Book|Retail Book, Directory|E-Book in Internet Downloadable File',
    '2ERD' => 'Editorial|Book|Retail Book, Directory|E-Book on Recordable Media',
    '2YEA' => 'Editorial|Book|Retail Book, Directory|All Electronic Distribution Formats',
    '2EAB' => 'Editorial|Book|Retail Book, Directory|All E-Book Distribution Formats',
    '2HAN' => 'Editorial|Book|Retail Book, Handbook|Printed',
    '2EPF' => 'Editorial|Book|Retail Book, Handbook|E-Book in Internet Website',
    '2EBH' => 'Editorial|Book|Retail Book, Handbook|E-Book in Internet Downloadable File',
    '2EGH' => 'Editorial|Book|Retail Book, Handbook|E-Book in Internet Email',
    '2EKD' => 'Editorial|Book|Retail Book, Handbook|All E-Book Internet Distribution Formats',
    '2ERF' => 'Editorial|Book|Retail Book, Handbook|E-Book on Recordable Media',
    '2EGJ' => 'Editorial|Book|Retail Book, Handbook|All E-Book Distribution Formats',
    '2HIL' => 'Editorial|Book|Retail Book, Hi-lo Book|Printed',
    '2EPG' => 'Editorial|Book|Retail Book, Hi-lo Book|E-Book in Internet Website',
    '2EBI' => 'Editorial|Book|Retail Book, Hi-lo Book|E-Book in Internet Downloadable File',
    '2EGK' => 'Editorial|Book|Retail Book, Hi-lo Book|E-Book in Internet Email',
    '2EKE' => 'Editorial|Book|Retail Book, Hi-lo Book|All E-Book Internet Distribution Formats',
    '2ERG' => 'Editorial|Book|Retail Book, Hi-lo Book|E-Book on Recordable Media',
    '2EGL' => 'Editorial|Book|Retail Book, Hi-lo Book|All E-Book Distribution Formats',
    '2WAB' => 'Editorial|Book|Retail Book, Illustrated Book|Printed',
    '2EPH' => 'Editorial|Book|Retail Book, Illustrated Book|E-Book in Internet Website',
    '2EBJ' => 'Editorial|Book|Retail Book, Illustrated Book|E-Book in Internet Downloadable File',
    '2EGM' => 'Editorial|Book|Retail Book, Illustrated Book|E-Book in Internet Email',
    '2EKF' => 'Editorial|Book|Retail Book, Illustrated Book|All E-Book Internet Distribution Formats',
    '2ERH' => 'Editorial|Book|Retail Book, Illustrated Book|E-Book on Recordable Media',
    '2EGN' => 'Editorial|Book|Retail Book, Illustrated Book|All E-Book Distribution Formats',
    '2WHA' => 'Editorial|Book|Retail Book, Illustrated Guide|Printed',
    '2EPJ' => 'Editorial|Book|Retail Book, Illustrated Guide|E-Book in Internet Website',
    '2EBL' => 'Editorial|Book|Retail Book, Illustrated Guide|E-Book in Internet Downloadable File',
    '2EGP' => 'Editorial|Book|Retail Book, Illustrated Guide|E-Book in Internet Email',
    '2EKG' => 'Editorial|Book|Retail Book, Illustrated Guide|All E-Book Internet Distribution Formats',
    '2ERI' => 'Editorial|Book|Retail Book, Illustrated Guide|E-Book on Recordable Media',
    '2EGQ' => 'Editorial|Book|Retail Book, Illustrated Guide|All E-Book Distribution Formats',
    '2EGR' => 'Editorial|Book|Retail Book, Illustrated Guide|All E-Book Distribution Formats',
    '2MAN' => 'Editorial|Book|Retail Book, Manual|Printed',
    '2EPK' => 'Editorial|Book|Retail Book, Manual|E-Book in Internet Website',
    '2EBM' => 'Editorial|Book|Retail Book, Manual|E-Book in Internet Downloadable File',
    '2EGS' => 'Editorial|Book|Retail Book, Manual|E-Book in Internet Email',
    '2EKH' => 'Editorial|Book|Retail Book, Manual|All E-Book Internet Distribution Formats',
    '2ERJ' => 'Editorial|Book|Retail Book, Manual|E-Book on Recordable Media',
    '2EGT' => 'Editorial|Book|Retail Book, Manual|All E-Book Distribution Formats',
    '2YAP' => 'Editorial|Book|Retail Book, Novelty Book|Printed',
    '2EPL' => 'Editorial|Book|Retail Book, Novelty Book|E-Book in Internet Website',
    '2EBN' => 'Editorial|Book|Retail Book, Novelty Book|E-Book in Internet Downloadable File',
    '2EGV' => 'Editorial|Book|Retail Book, Novelty Book|E-Book in Internet Email',
    '2EKJ' => 'Editorial|Book|Retail Book, Novelty Book|All E-Book Internet Distribution Formats',
    '2ERK' => 'Editorial|Book|Retail Book, Novelty Book|E-Book on Recordable Media',
    '2EGW' => 'Editorial|Book|Retail Book, Novelty Book|All E-Book Distribution Formats',
    '2YEN' => 'Editorial|Book|Retail Book, Postcard Book|Printed',
    '2EPM' => 'Editorial|Book|Retail Book, Postcard Book|E-Book in Internet Website',
    '2EBP' => 'Editorial|Book|Retail Book, Postcard Book|E-Book in Internet Downloadable File',
    '2EGY' => 'Editorial|Book|Retail Book, Postcard Book|E-Book in Internet Email',
    '2EKK' => 'Editorial|Book|Retail Book, Postcard Book|All E-Book Internet Distribution Formats',
    '2ERL' => 'Editorial|Book|Retail Book, Postcard Book|E-Book on Recordable Media',
    '2EGZ' => 'Editorial|Book|Retail Book, Postcard Book|All E-Book Distribution Formats',
    '2YOK' => 'Editorial|Book|Retail Book, Young Adult Book|Printed',
    '2EPN' => 'Editorial|Book|Retail Book, Young Adult Book|E-Book in Internet Website',
    '2EBQ' => 'Editorial|Book|Retail Book, Young Adult Book|E-Book in Internet Downloadable File',
    '2EJB' => 'Editorial|Book|Retail Book, Young Adult Book|E-Book in Internet Email',
    '2EKL' => 'Editorial|Book|Retail Book, Young Adult Book|All E-Book Internet Distribution Formats',
    '2ERM' => 'Editorial|Book|Retail Book, Young Adult Book|E-Book on Recordable Media',
    '2EJC' => 'Editorial|Book|Retail Book, Young Adult Book|All E-Book Distribution Formats',
    '2BOO' => 'Editorial|Book|Retail Book, All Types|Printed',
    '2EPB' => 'Editorial|Book|Retail Book, All Types|E-Book in Internet Website',
    '2EBC' => 'Editorial|Book|Retail Book, All Types|E-Book in Internet Downloadable File',
    '2EJD' => 'Editorial|Book|Retail Book, All Types|E-Book in Internet Email',
    '2EKM' => 'Editorial|Book|Retail Book, All Types|All E-Book Internet Distribution Formats',
    '2ERN' => 'Editorial|Book|Retail Book, All Types|E-Book on Recordable Media',
    '2EJE' => 'Editorial|Book|Retail Book, All Types|All E-Book Distribution Formats',
    '2EEL' => 'Editorial|Book|Textbook, Compendium|Printed',
    '2EPQ' => 'Editorial|Book|Textbook, Compendium|E-Book in Internet Website',
    '2EBS' => 'Editorial|Book|Textbook, Compendium|E-Book in Internet Downloadable File',
    '2EJF' => 'Editorial|Book|Textbook, Compendium|E-Book in Internet Email',
    '2EKN' => 'Editorial|Book|Textbook, Compendium|All E-Book Internet Distribution Formats',
    '2ERP' => 'Editorial|Book|Textbook, Compendium|E-Book on Recordable Media',
    '2EJJ' => 'Editorial|Book|Textbook, Compendium|All E-Book Distribution Formats',
    '2EMU' => 'Editorial|Book|Textbook, Course Pack|Printed',
    '2GBN' => 'Editorial|Book|Textbook, Course Pack|Internet Website',
    '2GBM' => 'Editorial|Book|Textbook, Course Pack|Internet Downloadable File',
    '2ZOO' => 'Editorial|Book|Textbook, Course Pack|Internet Email',
    '2GBQ' => 'Editorial|Book|Textbook, Course Pack|All Internet Distribution Formats',
    '2ZOA' => 'Editorial|Book|Textbook, Course Pack|Recordable Media',
    '2GBP' => 'Editorial|Book|Textbook, Course Pack|All Distribution Formats',
    '2BEE' => 'Editorial|Book|Textbook, Middle Reader|Printed',
    '2EPR' => 'Editorial|Book|Textbook, Middle Reader|E-Book in Internet Website',
    '2EBT' => 'Editorial|Book|Textbook, Middle Reader|E-Book in Internet Downloadable File',
    '2EJK' => 'Editorial|Book|Textbook, Middle Reader|E-Book in Internet Email',
    '2EKP' => 'Editorial|Book|Textbook, Middle Reader|All E-Book Internet Distribution Formats',
    '2ERQ' => 'Editorial|Book|Textbook, Middle Reader|E-Book on Recordable Media',
    '2EJL' => 'Editorial|Book|Textbook, Middle Reader|All E-Book Distribution Formats',
    '2BOA' => 'Editorial|Book|Textbook, Student Edition|Printed',
    '2EPS' => 'Editorial|Book|Textbook, Student Edition|E-Book in Internet Website',
    '2EBV' => 'Editorial|Book|Textbook, Student Edition|E-Book in Internet Downloadable File',
    '2EJM' => 'Editorial|Book|Textbook, Student Edition|E-Book in Internet Email',
    '2EKQ' => 'Editorial|Book|Textbook, Student Edition|All E-Book Internet Distribution Formats',
    '2ERS' => 'Editorial|Book|Textbook, Student Edition|E-Book on Recordable Media',
    '2EJN' => 'Editorial|Book|Textbook, Student Edition|All E-Book Distribution Formats',
    '2FOX' => 'Editorial|Book|Textbook, All Types|Printed',
    '2EPP' => 'Editorial|Book|Textbook, All Types|E-Book in Internet Website',
    '2EBR' => 'Editorial|Book|Textbook, All Types|E-Book in Internet Downloadable File',
    '2EGA' => 'Editorial|Book|Textbook, All Types|E-Book in Internet Email',
    '2EKR' => 'Editorial|Book|Textbook, All Types|All E-Book Internet Distribution Formats',
    '2ERT' => 'Editorial|Book|Textbook, All Types|E-Book on Recordable Media',
    '2EJP' => 'Editorial|Book|Textbook, All Types|All E-Book Distribution Formats',
    '2GBZ' => 'Editorial|Book|Textbook Ancillary Materials, Educational Film Set|Projected Display',
    '2YET' => 'Editorial|Book|Textbook Ancillary Materials, Packaging For Recordable Media|Printed',
    '2YUM' => 'Editorial|Book|Textbook Ancillary Materials, Lab Manual|Printed',
    '2EPZ' => 'Editorial|Book|Textbook Ancillary Materials, Lab Manual|E-Book in Internet Website',
    '2EBA' => 'Editorial|Book|Textbook Ancillary Materials, Lab Manual|E-Book in Internet Downloadable File',
    '2EJQ' => 'Editorial|Book|Textbook Ancillary Materials, Lab Manual|E-Book in Internet Email',
    '2EKS' => 'Editorial|Book|Textbook Ancillary Materials, Lab Manual|All E-Book Internet Distribution Formats',
    '2ERU' => 'Editorial|Book|Textbook Ancillary Materials, Lab Manual|E-Book on Recordable Media',
    '2EJR' => 'Editorial|Book|Textbook Ancillary Materials, Lab Manual|All E-Book Distribution Formats',
    '2YAK' => 'Editorial|Book|Textbook Ancillary Materials, Teachers\' Edition|Printed',
    '2EPT' => 'Editorial|Book|Textbook Ancillary Materials, Teachers\' Edition|E-Book in Internet Website',
    '2EBW' => 'Editorial|Book|Textbook Ancillary Materials, Teachers\' Edition|E-Book in Internet Downloadable File',
    '2EJS' => 'Editorial|Book|Textbook Ancillary Materials, Teachers\' Edition|E-Book in Internet Email',
    '2EKT' => 'Editorial|Book|Textbook Ancillary Materials, Teachers\' Edition|All E-Book Internet Distribution Formats',
    '2ERV' => 'Editorial|Book|Textbook Ancillary Materials, Teachers\' Edition|E-Book on Recordable Media',
    '2EJT' => 'Editorial|Book|Textbook Ancillary Materials, Teachers\' Edition|All E-Book Distribution Formats',
    '2ZOT' => 'Editorial|Book|Textbook Ancillary Materials, Teacher\'s Manual|Printed',
    '2EPW' => 'Editorial|Book|Textbook Ancillary Materials, Teacher\'s Manual|E-Book in Internet Website',
    '2EBD' => 'Editorial|Book|Textbook Ancillary Materials, Teacher\'s Manual|E-Book in Internet Downloadable File',
    '2EJV' => 'Editorial|Book|Textbook Ancillary Materials, Teacher\'s Manual|E-Book in Internet Email',
    '2EKU' => 'Editorial|Book|Textbook Ancillary Materials, Teacher\'s Manual|All E-Book Internet Distribution Formats',
    '2ERW' => 'Editorial|Book|Textbook Ancillary Materials, Teacher\'s Manual|E-Book on Recordable Media',
    '2EJW' => 'Editorial|Book|Textbook Ancillary Materials, Teacher\'s Manual|All E-Book Distribution Formats',
    '2GEE' => 'Editorial|Book|Textbook Ancillary Materials, Workbook|Printed',
    '2EPY' => 'Editorial|Book|Textbook Ancillary Materials, Workbook|E-Book in Internet Website',
    '2EBB' => 'Editorial|Book|Textbook Ancillary Materials, Workbook|E-Book in Internet Downloadable File',
    '2EJY' => 'Editorial|Book|Textbook Ancillary Materials, Workbook|E-Book in Internet Email',
    '2EKV' => 'Editorial|Book|Textbook Ancillary Materials, Workbook|All E-Book Internet Distribution Formats',
    '2ERY' => 'Editorial|Book|Textbook Ancillary Materials, Workbook|E-Book on Recordable Media',
    '2EJZ' => 'Editorial|Book|Textbook Ancillary Materials, Workbook|All E-Book Distribution Formats',
    '2ELK' => 'Editorial|Book|Textbook Ancillary Materials, All Ancillary Types|Printed',
    '2GBY' => 'Editorial|Book|Textbook Ancillary Materials, All Ancillary Types|All Internet Distribution Formats',
    '2GBW' => 'Editorial|Book|Textbook Ancillary Materials, All Ancillary Types|All Distribution Formats',
    '2NAY' => 'Editorial|Book|Reference Book, Encyclopedia|Printed',
    '2EAQ' => 'Editorial|Book|Reference Book, Encyclopedia|E-Book in Internet Website',
    '2EMB' => 'Editorial|Book|Reference Book, Encyclopedia|E-Book in Internet Downloadable File',
    '2EMC' => 'Editorial|Book|Reference Book, Encyclopedia|All E-Book Internet Distribution Formats',
    '2EMD' => 'Editorial|Book|Reference Book, Encyclopedia|E-Book on Recordable Media',
    '2NAH' => 'Editorial|Book|Reference Book, Encyclopedia|All Electronic Distribution Formats',
    '2NIP' => 'Editorial|Book|Reference Book, Telephone Book|Printed',
    '2EMF' => 'Editorial|Book|Reference Book, Telephone Book|E-Book in Internet Website',
    '2EMG' => 'Editorial|Book|Reference Book, Telephone Book|E-Book in Internet Downloadable File',
    '2EMH' => 'Editorial|Book|Reference Book, Telephone Book|All E-Book Internet Distribution Formats',
    '2EMI' => 'Editorial|Book|Reference Book, Telephone Book|E-Book on Recordable Media',
    '2EMJ' => 'Editorial|Book|Reference Book, Telephone Book|All E-Book Distribution Formats',
    '2DOG' => 'Editorial|Book|Reference Book, All Types|Printed',
    '2EMK' => 'Editorial|Book|Reference Book, All Types|E-Book in Internet Website',
    '2EML' => 'Editorial|Book|Reference Book, All Types|E-Book in Internet Downloadable File',
    '2EMM' => 'Editorial|Book|Reference Book, All Types|All E-Book Internet Distribution Formats',
    '2EMN' => 'Editorial|Book|Reference Book, All Types|E-Book on Recordable Media',
    '2EMP' => 'Editorial|Book|Reference Book, All Types|All E-Book Distribution Formats',
    '2DEW' => 'Editorial|Book|Trade Book, All Types|Printed',
    '2EPV' => 'Editorial|Book|Trade Book, All Types|E-Book in Internet Website',
    '2EMQ' => 'Editorial|Book|Trade Book, All Types|E-Book in Internet Downloadable File',
    '2EMR' => 'Editorial|Book|Trade Book, All Types|E-Book in Internet Email',
    '2EMS' => 'Editorial|Book|Trade Book, All Types|All E-Book Internet Distribution Formats',
    '2ERA' => 'Editorial|Book|Trade Book, All Types|E-Book on Recordable Media',
    '2EMT' => 'Editorial|Book|Trade Book, All Types|All E-Book Distribution Formats',
    '2MOG' => 'Editorial|Book|All Book Types|Printed',
    '2EPA' => 'Editorial|Book|All Book Types|E-Book in Internet Website',
    '2EBK' => 'Editorial|Book|All Book Types|E-Book in Internet Downloadable File',
    '2EJU' => 'Editorial|Book|All Book Types|E-Book in Internet Email',
    '2EKA' => 'Editorial|Book|All Book Types|All E-Book Internet Distribution Formats',
    '2ERZ' => 'Editorial|Book|All Book Types|E-Book on Recordable Media',
    '2EJA' => 'Editorial|Book|All Book Types|All E-Book Distribution Formats',
    '2MOR' => 'Editorial|Book|Artist\'s Reference, All Types|Printed',
    '2EKW' => 'Editorial|Book|Artist\'s Reference, All Types|All E-Book Internet Distribution Formats',
    '2EMV' => 'Editorial|Book|Artist\'s Reference, All Types|All E-Book Distribution Formats',
    '2GAL' => 'Editorial|Display|Gallery Exhibition|Printed',
    '2GDJ' => 'Editorial|Display|Gallery Exhibition|Electronic Display',
    '2MUS' => 'Editorial|Display|Museum Display|Printed',
    '2GDK' => 'Editorial|Display|Museum Display|Electronic Display',
    '2EAU' => 'Editorial|Display|Poster, Educational Poster|Printed',
    '2GHG' => 'Editorial|Website|Web Page, Body Content|Internet Website',
    '2GHK' => 'Editorial|Website|Web Page, Body Content|All Internet Distribution Formats',
    '2GHH' => 'Editorial|Website|Web Page, Body Content|Recordable Media',
    '2GHJ' => 'Editorial|Website|Web Page, Body Content|All Electronic Distribution Formats',
    '2GHB' => 'Editorial|Website|Web Page, All Types|Internet Website',
    '2GHF' => 'Editorial|Website|Web Page, All Types|All Internet Distribution Formats',
    '2GHC' => 'Editorial|Website|Web Page, All Types|Recordable Media',
    '2GHD' => 'Editorial|Website|Web Page, All Types|All Electronic Distribution Formats',
    '2GHA' => 'Editorial|Website|Webcast, All Types|Internet Website',
    '2ZAG' => 'Editorial|Mobile|All Mobile Types|Mobile',
    '2GGM' => 'Editorial|Merchandise|CD ROM|Recordable Media',
    '2DVE' => 'Editorial|Merchandise|DVD|Recordable Media',
    '2AAE' => 'Editorial|All Media Types|All Formats|All Distribution Formats',
    '2ZIG' => 'Products|Mobile|Wallpaper|Mobile',
    '2ZIP' => 'Products|Mobile|Game, All Types|Mobile',
    '2ZAP' => 'Products|Mobile|Entertainment Programming|Mobile',
    '2ZEN' => 'Products|Mobile|Computer Software|Mobile',
    '2ZAM' => 'Products|Mobile|All Mobile Types|Mobile',
    '2ADD' => 'Products|Merchandise|Address Book|Printed',
    '2ANT' => 'Products|Merchandise|Anthology|Printed',
    '2APP' => 'Products|Merchandise|Apparel, General Apparel|Printed or Woven',
    '2TST' => 'Products|Merchandise|Apparel, T-Shirts|Printed or Woven',
    '2BIO' => 'Products|Merchandise|Birthday Book|Printed',
    '2BLA' => 'Products|Merchandise|Blank Note Book|Printed',
    '2ATM' => 'Products|Merchandise|Card, ATM Card|Printed',
    '2BNK' => 'Products|Merchandise|Card, Bank Card|Printed',
    '2CRE' => 'Products|Merchandise|Card, Credit Card|Printed',
    '2BIT' => 'Products|Merchandise|Card, Debit Card|Printed',
    '2GFT' => 'Products|Merchandise|Card, Gift Card|Printed',
    '2GRE' => 'Products|Merchandise|Card, Greeting Card|Printed',
    '2HER' => 'Products|Merchandise|Card, Hero Card|Printed',
    '2CRD' => 'Products|Merchandise|Card, Other Card|Printed',
    '2ATT' => 'Products|Merchandise|Card, Phone Card|Printed',
    '2CDR' => 'Products|Merchandise|CD ROM|Recordable Media',
    '2CHK' => 'Products|Merchandise|Check|Printed',
    '2BAK' => 'Products|Merchandise|Computer Software|Internet Downloadable File',
    '2BAU' => 'Products|Merchandise|Computer Software|Internet Email',
    '2BAV' => 'Products|Merchandise|Computer Software|All Internet Distribution Formats',
    '2BAJ' => 'Products|Merchandise|Computer Software|Recordable Media',
    '2WAR' => 'Products|Merchandise|Computer Software|All Electronic Distribution Formats',
    '2GIG' => 'Products|Merchandise|Datebook|Printed',
    '2DIA' => 'Products|Merchandise|Diary|Printed',
    '2BAZ' => 'Products|Merchandise|Diary|Internet Downloadable File',
    '2BBA' => 'Products|Merchandise|Diary|Internet Email',
    '2BAW' => 'Products|Merchandise|Diary|Recordable Media',
    '2HUB' => 'Products|Merchandise|Double Postcard|Printed',
    '2DVD' => 'Products|Merchandise|DVD|Recordable Media',
    '2EAT' => 'Products|Merchandise|Edible Media|Printed',
    '2FOL' => 'Products|Merchandise|Folder|Printed',
    '2GAB' => 'Products|Merchandise|Game, Computer Game|Internet Downloadable File',
    '2GAH' => 'Products|Merchandise|Game, Computer Game|Internet Email',
    '2GAE' => 'Products|Merchandise|Game, Computer Game|All Internet Distribution Formats',
    '2GAG' => 'Products|Merchandise|Game, Computer Game|Recordable Media',
    '2GAN' => 'Products|Merchandise|Game, Computer Game|All Electronic Distribution Formats',
    '2GAP' => 'Products|Merchandise|Game, All Types|Printed',
    '2GAT' => 'Products|Merchandise|Game, All Types|Internet Downloadable File',
    '2GAS' => 'Products|Merchandise|Game, All Types|Internet Email',
    '2GAD' => 'Products|Merchandise|Game, All Types|All Internet Distribution Formats',
    '2GAR' => 'Products|Merchandise|Game, All Types|Recordable Media',
    '2GAM' => 'Products|Merchandise|Game, All Types|All Electronic Distribution Formats',
    '2BOX' => 'Products|Merchandise|Gift Box|Printed',
    '2GIF' => 'Products|Merchandise|Gift Certificate|Printed',
    '2WRP' => 'Products|Merchandise|Gift Wrap|Printed',
    '2JIG' => 'Products|Merchandise|Jigsaw Puzzle|Printed',
    '2JIN' => 'Products|Merchandise|Jigsaw Puzzle|Internet Downloadable File',
    '2JIL' => 'Products|Merchandise|Jigsaw Puzzle|Recordable Media',
    '2JRN' => 'Products|Merchandise|Journal|Printed',
    '2MAP' => 'Products|Merchandise|Map|Printed',
    '2MAD' => 'Products|Merchandise|Map|Internet Downloadable File',
    '2MET' => 'Products|Merchandise|Map|Internet Email',
    '2MIM' => 'Products|Merchandise|Map|All Internet Distribution Formats',
    '2MIR' => 'Products|Merchandise|Map|Recordable Media',
    '2MAB' => 'Products|Merchandise|Map|All Electronic Distribution Formats',
    '2MOU' => 'Products|Merchandise|Mouse Pad|Printed',
    '2MUG' => 'Products|Merchandise|Mugs|Printed',
    '2NOV' => 'Products|Merchandise|Novelty Products|Printed',
    '2MCH' => 'Products|Merchandise|Other Merchandise|Printed',
    '2ALB' => 'Products|Merchandise|Photo Album|Printed',
    '2MAT' => 'Products|Merchandise|Placemat|Printed',
    '2MRP' => 'Products|Merchandise|Plates|Printed',
    '2TOP' => 'Products|Merchandise|Poster, Retail Poster|Printed',
    '2FUN' => 'Products|Merchandise|Playing Cards|Printed',
    '2CLO' => 'Products|Merchandise|Retail Calendar, One Page|Printed',
    '2CAL' => 'Products|Merchandise|Retail Calendar, Multi-Page|Printed',
    '2BCJ' => 'Products|Merchandise|Retail Calendar, Multi-Page|Internet Downloadable File',
    '2BCI' => 'Products|Merchandise|Retail Calendar, Multi-Page|Internet Email',
    '2BCH' => 'Products|Merchandise|Retail Calendar, Multi-Page|All Internet Distribution Formats',
    '2BCF' => 'Products|Merchandise|Retail Calendar, Multi-Page|Recordable Media',
    '2BCG' => 'Products|Merchandise|Retail Calendar, Multi-Page|All Electronic Distribution Formats',
    '2FAR' => 'Products|Merchandise|Retail Postcard|Printed',
    '2FIB' => 'Products|Merchandise|Screen Saver|Internet Downloadable File',
    '2GEL' => 'Products|Merchandise|Screen Saver|Internet Email',
    '2TOD' => 'Products|Merchandise|Screen Saver|All Internet Distribution Formats',
    '2WRY' => 'Products|Merchandise|Screen Saver|Recordable Media',
    '2HEX' => 'Products|Merchandise|Screen Saver|All Electronic Distribution Formats',
    '2KEP' => 'Products|Merchandise|Souvenir|Printed',
    '2WEE' => 'Products|Merchandise|Stamp|Printed',
    '2WIT' => 'Products|Merchandise|Stationery|Printed',
    '2TIC' => 'Products|Merchandise|Sticker|Printed',
    '2FEM' => 'Products|Merchandise|Textiles|Printed or Woven',
    '2TKT' => 'Products|Merchandise|Ticket|Printed',
    '2TOY' => 'Products|Merchandise|Toy|Printed',
    '2TRD' => 'Products|Merchandise|Trading Cards|Printed',
    '2TUA' => 'Products|Merchandise|Virtual Reality|Recordable Media',
    '2WAL' => 'Products|Merchandise|Wallpaper|Printed',
    '2MER' => 'Products|Merchandise|All Merchandise Types|Printed',
    '2BAB' => 'Products|Art|Artist\'s Reference, Tattoo|Printed',
    '2DVL' => 'Products|Product Packaging|Packaging For Recordable Media, Liner Notes|Printed',
    '2DVP' => 'Products|Product Packaging|Packaging For Recordable Media, All Packaging Types|Printed',
    '2FRA' => 'Products|Product Packaging|Picture Frame Insert|Printed',
    '2EVE' => 'Products|Product Packaging|Retail Packaging, All Packaging Types|Printed',
    '2WHO' => 'Products|Product Packaging|Wholesale Packaging, All Packaging Types|Printed',
    '2DAY' => 'Products|Product Packaging|All Product Packaging Types|Printed',
    '2AAP' => 'Products|All Media Types|All Formats|All Distribution Formats',
    '2TUX' => 'Internal Company Use|Periodicals|Magazine, Custom Published|Printed',
    '2BIC' => 'Internal Company Use|Periodicals|Magazine, Custom Published|Intranet and Extranet Website',
    '2BIF' => 'Internal Company Use|Periodicals|Magazine, Custom Published|Intranet and Extranet Downloadable File',
    '2FOE' => 'Internal Company Use|Periodicals|Magazine, Custom Published|Internet Email',
    '2BII' => 'Internal Company Use|Periodicals|Magazine, Custom Published|All Intranet and Extranet Distribution Formats',
    '2BIA' => 'Internal Company Use|Periodicals|Magazine, Custom Published|Recordable Media',
    '2BIH' => 'Internal Company Use|Periodicals|Magazine, Custom Published|All Electronic Distribution Formats',
    '2FRO' => 'Internal Company Use|Comp Use|All Comp Types|Printed',
    '2FPO' => 'Internal Company Use|Comp Use|All Comp Types|All Electronic Distribution Formats',
    '2GED' => 'Internal Company Use|Internal Review|All Review Types|Printed',
    '2GID' => 'Internal Company Use|Internal Review|All Review Types|All Electronic Distribution Formats',
    '2GOA' => 'Internal Company Use|Promotional Materials|Corporate Brochure|Printed',
    '2BHH' => 'Internal Company Use|Promotional Materials|Corporate Brochure|Intranet and Extranet Downloadable File',
    '2BHI' => 'Internal Company Use|Promotional Materials|Corporate Brochure|Intranet and Extranet Email',
    '2BHG' => 'Internal Company Use|Promotional Materials|Corporate Brochure|Recordable Media',
    '2BHN' => 'Internal Company Use|Promotional Materials|Corporate Calendar|Printed',
    '2BHL' => 'Internal Company Use|Promotional Materials|Corporate Calendar|Intranet and Extranet Downloadable File',
    '2BHM' => 'Internal Company Use|Promotional Materials|Corporate Calendar|Intranet and Extranet Email',
    '2BHK' => 'Internal Company Use|Promotional Materials|Corporate Calendar|Recordable Media',
    '2BHY' => 'Internal Company Use|Promotional Materials|Corporate Folder|Printed',
    '2BHQ' => 'Internal Company Use|Promotional Materials|Card, Corporate Card|Printed',
    '2BHT' => 'Internal Company Use|Promotional Materials|Card, Corporate Card|Intranet and Extranet Downloadable File',
    '2BHU' => 'Internal Company Use|Promotional Materials|Card, Corporate Card|Intranet and Extranet Email',
    '2BHS' => 'Internal Company Use|Promotional Materials|Card, Corporate Card|Recordable Media',
    '2BHZ' => 'Internal Company Use|Promotional Materials|CD ROM|Recordable Media',
    '2DVI' => 'Internal Company Use|Promotional Materials|DVD|Recordable Media',
    '2BHV' => 'Internal Company Use|Promotional Materials|Sales Kit|All Electronic Distribution Formats',
    '2BHW' => 'Internal Company Use|Promotional Materials|Training Materials|Printed',
    '2BJN' => 'Internal Company Use|Website|Web Page, Content Body|Intranet and Extranet Website',
    '2FAB' => 'Internal Company Use|Website|Web Page, Content Body|Recordable Media',
    '2FAC' => 'Internal Company Use|Website|Web Page, Content Body|All Electronic Distribution Formats',
    '2FAH' => 'Internal Company Use|Website|Web Page, Design Element|Intranet and Extranet Website',
    '2FAI' => 'Internal Company Use|Website|Web Page, Design Element|Recordable Media',
    '2FAJ' => 'Internal Company Use|Website|Web Page, Design Element|All Electronic Distribution Formats',
    '2NET' => 'Internal Company Use|Website|Web Page, All Types|Intranet and Extranet Website',
    '2BJL' => 'Internal Company Use|Website|Web Page, All Types|Recordable Media',
    '2BJK' => 'Internal Company Use|Website|Web Page, All Types|All Electronic Distribution Formats',
    '2BJH' => 'Internal Company Use|Website|Webcast, All Types|Intranet and Extranet Website',
    '2WAN' => 'Internal Company Use|Website|All Website Types|Intranet and Extranet Website',
    '2BHD' => 'Internal Company Use|Email|All Email Types|Intranet and Extranet Email',
    '2ZUM' => 'Internal Company Use|Mobile|All Mobile Types|Mobile',
    '2BHF' => 'Internal Company Use|Live Presentation|Internal Presentation|Projected Display',
    '2BHB' => 'Internal Company Use|Art|Art Display, Display Print|Printed',
    '2GOY' => 'Internal Company Use|Art|Art Display, All Art Types|Printed',
    '2HAO' => 'Internal Company Use|Art|Art Display, All Art Types|Electronic Display',
    '2AAI' => 'Internal Company Use|All Media Types|All Formats|All Distribution Formats',
    '2TVC' => 'Motion Picture & TV|Television Programming|Commercial|Television Broadcast',
    '2TEB' => 'Motion Picture & TV|Television Programming|Commercial|Internet Downloadable File',
    '2TEC' => 'Motion Picture & TV|Television Programming|Commercial|All Internet Distribution Formats',
    '2TEG' => 'Motion Picture & TV|Television Programming|Commercial|Recordable Media',
    '2TVD' => 'Motion Picture & TV|Television Programming|Commercial|All Electronic Distribution Formats',
    '2TVE' => 'Motion Picture & TV|Television Programming|Infomercial|Television Broadcast',
    '2TEH' => 'Motion Picture & TV|Television Programming|Infomercial|Internet Downloadable File',
    '2TEJ' => 'Motion Picture & TV|Television Programming|Infomercial|All Internet Distribution Formats',
    '2TEK' => 'Motion Picture & TV|Television Programming|Infomercial|Recordable Media',
    '2TVF' => 'Motion Picture & TV|Television Programming|Infomercial|All Electronic Distribution Formats',
    '2TVG' => 'Motion Picture & TV|Television Programming|On-Air Promotion|Television Broadcast',
    '2TEL' => 'Motion Picture & TV|Television Programming|On-Air Promotion|Internet Downloadable File',
    '2TEM' => 'Motion Picture & TV|Television Programming|On-Air Promotion|All Internet Distribution Formats',
    '2TEP' => 'Motion Picture & TV|Television Programming|On-Air Promotion|Recordable Media',
    '2TVH' => 'Motion Picture & TV|Television Programming|On-Air Promotion|All Electronic Distribution Formats',
    '2TVI' => 'Motion Picture & TV|Television Programming|Documentary Program|Television Broadcast',
    '2TER' => 'Motion Picture & TV|Television Programming|Documentary Program|Internet Downloadable File',
    '2TES' => 'Motion Picture & TV|Television Programming|Documentary Program|All Internet Distribution Formats',
    '2TET' => 'Motion Picture & TV|Television Programming|Documentary Program|Recordable Media',
    '2TVJ' => 'Motion Picture & TV|Television Programming|Documentary Program|All Electronic Distribution Formats',
    '2TVA' => 'Motion Picture & TV|Television Programming|All Television Advertising Types|Television Broadcast',
    '2TEA' => 'Motion Picture & TV|Television Programming|All Television Advertising Types|Internet Downloadable File',
    '2TEQ' => 'Motion Picture & TV|Television Programming|All Television Advertising Types|All Internet Distribution Formats',
    '2TED' => 'Motion Picture & TV|Television Programming|All Television Advertising Types|Recordable Media',
    '2TVB' => 'Motion Picture & TV|Television Programming|All Television Advertising Types|All Electronic Distribution Formats',
    '2TVK' => 'Motion Picture & TV|Television Programming|Educational Program|Television Broadcast',
    '2TEU' => 'Motion Picture & TV|Television Programming|Educational Program|Internet Downloadable File',
    '2TEV' => 'Motion Picture & TV|Television Programming|Educational Program|All Internet Distribution Formats',
    '2TEW' => 'Motion Picture & TV|Television Programming|Educational Program|Recordable Media',
    '2TVL' => 'Motion Picture & TV|Television Programming|Educational Program|All Electronic Distribution Formats',
    '2TVM' => 'Motion Picture & TV|Television Programming|Entertainment Program|Television Broadcast',
    '2TEY' => 'Motion Picture & TV|Television Programming|Entertainment Program|Internet Downloadable File',
    '2TEZ' => 'Motion Picture & TV|Television Programming|Entertainment Program|All Internet Distribution Formats',
    '2TLA' => 'Motion Picture & TV|Television Programming|Entertainment Program|Recordable Media',
    '2TVN' => 'Motion Picture & TV|Television Programming|Entertainment Program|All Electronic Distribution Formats',
    '2TVY' => 'Motion Picture & TV|Television Programming|Made For TV Movie|Television Broadcast',
    '2TLC' => 'Motion Picture & TV|Television Programming|Made For TV Movie|Internet Downloadable File',
    '2TLB' => 'Motion Picture & TV|Television Programming|Made For TV Movie|All Internet Distribution Formats',
    '2TLD' => 'Motion Picture & TV|Television Programming|Made For TV Movie|Recordable Media',
    '2TVP' => 'Motion Picture & TV|Television Programming|Made For TV Movie|All Electronic Distribution Formats',
    '2TEF' => 'Motion Picture & TV|Television Programming|News Program, Flash|Television Broadcast',
    '2TVS' => 'Motion Picture & TV|Television Programming|News Program|Television Broadcast',
    '2TLE' => 'Motion Picture & TV|Television Programming|News Program|Internet Downloadable File',
    '2TLF' => 'Motion Picture & TV|Television Programming|News Program|All Internet Distribution Formats',
    '2TLG' => 'Motion Picture & TV|Television Programming|News Program|Recordable Media',
    '2TVT' => 'Motion Picture & TV|Television Programming|News Program|All Electronic Distribution Formats',
    '2TLH' => 'Motion Picture & TV|Television Programming|Non Broadcast Pilot|Recordable Media',
    '2TLJ' => 'Motion Picture & TV|Television Programming|Non Broadcast Pilot|Projected Display',
    '2TVU' => 'Motion Picture & TV|Television Programming|Non-Profit Program|Television Broadcast',
    '2TLK' => 'Motion Picture & TV|Television Programming|Non-Profit Program|Internet Downloadable File',
    '2TLL' => 'Motion Picture & TV|Television Programming|Non-Profit Program|All Internet Distribution Formats',
    '2TLM' => 'Motion Picture & TV|Television Programming|Non-Profit Program|Recordable Media',
    '2TVV' => 'Motion Picture & TV|Television Programming|Non-Profit Program|All Electronic Distribution Formats',
    '2TLR' => 'Motion Picture & TV|Television Programming|Prop|Television Broadcast',
    '2TLN' => 'Motion Picture & TV|Television Programming|Prop|Internet Downloadable File',
    '2TLP' => 'Motion Picture & TV|Television Programming|Prop|All Internet Distribution Formats',
    '2TLQ' => 'Motion Picture & TV|Television Programming|Prop|Recordable Media',
    '2TVW' => 'Motion Picture & TV|Television Programming|Prop|All Electronic Distribution Formats',
    '2TLV' => 'Motion Picture & TV|Television Programming|Set Decor|Television Broadcast',
    '2TLS' => 'Motion Picture & TV|Television Programming|Set Decor|Internet Downloadable File',
    '2TLT' => 'Motion Picture & TV|Television Programming|Set Decor|All Internet Distribution Formats',
    '2TLU' => 'Motion Picture & TV|Television Programming|Set Decor|Recordable Media',
    '2TVQ' => 'Motion Picture & TV|Television Programming|Set Decor|All Electronic Distribution Formats',
    '2KOA' => 'Motion Picture & TV|Television Programming|Artist\'s Reference, All Types|Projected Display',
    '2KOB' => 'Motion Picture & TV|Television Programming|Artist\'s Reference, All Types|Internet Downloadable File',
    '2KOI' => 'Motion Picture & TV|Television Programming|Artist\'s Reference, All Types|All Internet Distribution Formats',
    '2KOP' => 'Motion Picture & TV|Television Programming|Artist\'s Reference, All Types|Recordable Media',
    '2KOR' => 'Motion Picture & TV|Television Programming|Artist\'s Reference, All Types|All Electronic Distribution Formats',
    '2KOS' => 'Motion Picture & TV|Television Programming|Artist\'s Reference, All Types|Television Broadcast',
    '2TEE' => 'Motion Picture & TV|Television Programming|All Editorial Television Types|Television Broadcast',
    '2TLW' => 'Motion Picture & TV|Television Programming|All Editorial Television Types|Internet Downloadable File',
    '2TLY' => 'Motion Picture & TV|Television Programming|All Editorial Television Types|All Internet Distribution Formats',
    '2TLZ' => 'Motion Picture & TV|Television Programming|All Editorial Television Types|Recordable Media',
    '2TVZ' => 'Motion Picture & TV|Television Programming|All Editorial Television Types|All Electronic Distribution Formats',
    '2JOT' => 'Motion Picture & TV|Motion Picture|In Theater Commercial|Projected Display',
    '2ATE' => 'Motion Picture & TV|Motion Picture|Movie Trailer|Projected Display',
    '2MPP' => 'Motion Picture & TV|Motion Picture|Movie Trailer|Internet Downloadable File',
    '2BET' => 'Motion Picture & TV|Motion Picture|Movie Trailer|All Internet Distribution Formats',
    '2DIG' => 'Motion Picture & TV|Motion Picture|Movie Trailer|Recordable Media',
    '2AID' => 'Motion Picture & TV|Motion Picture|Movie Trailer|All Electronic Distribution Formats',
    '2MPF' => 'Motion Picture & TV|Motion Picture|Movie Trailer|Television Broadcast',
    '2JOG' => 'Motion Picture & TV|Motion Picture|Documentary Film|Projected Display',
    '2MPM' => 'Motion Picture & TV|Motion Picture|Documentary Film|Internet Downloadable File',
    '2TUG' => 'Motion Picture & TV|Motion Picture|Documentary Film|All Internet Distribution Formats',
    '2TRY' => 'Motion Picture & TV|Motion Picture|Documentary Film|Recordable Media',
    '2DOC' => 'Motion Picture & TV|Motion Picture|Documentary Film|All Electronic Distribution Formats',
    '2MPD' => 'Motion Picture & TV|Motion Picture|Documentary Film|Television Broadcast',
    '2FEA' => 'Motion Picture & TV|Motion Picture|Feature Film|Projected Display',
    '2MPN' => 'Motion Picture & TV|Motion Picture|Feature Film|Internet Downloadable File',
    '2WAG' => 'Motion Picture & TV|Motion Picture|Feature Film|All Internet Distribution Formats',
    '2WIN' => 'Motion Picture & TV|Motion Picture|Feature Film|Recordable Media',
    '2FIL' => 'Motion Picture & TV|Motion Picture|Feature Film|All Electronic Distribution Formats',
    '2MPE' => 'Motion Picture & TV|Motion Picture|Feature Film|Television Broadcast',
    '2WIZ' => 'Motion Picture & TV|Motion Picture|Short Film|Projected Display',
    '2MPS' => 'Motion Picture & TV|Motion Picture|Short Film|Internet Downloadable File',
    '2ASK' => 'Motion Picture & TV|Motion Picture|Short Film|All Internet Distribution Formats',
    '2AIM' => 'Motion Picture & TV|Motion Picture|Short Film|Recordable Media',
    '2FIT' => 'Motion Picture & TV|Motion Picture|Short Film|All Electronic Distribution Formats',
    '2MPI' => 'Motion Picture & TV|Motion Picture|Short Film|Television Broadcast',
    '2MPG' => 'Motion Picture & TV|Motion Picture|Prop|Television Broadcast',
    '2HOP' => 'Motion Picture & TV|Motion Picture|Prop|Projected Display',
    '2MPQ' => 'Motion Picture & TV|Motion Picture|Prop|Internet Downloadable File',
    '2HIT' => 'Motion Picture & TV|Motion Picture|Prop|All Internet Distribution Formats',
    '2FRY' => 'Motion Picture & TV|Motion Picture|Prop|Recordable Media',
    '2EFX' => 'Motion Picture & TV|Motion Picture|Prop|All Electronic Distribution Formats',
    '2MPH' => 'Motion Picture & TV|Motion Picture|Set Decor|Television Broadcast',
    '2MPR' => 'Motion Picture & TV|Motion Picture|Set Decor|Internet Downloadable File',
    '2JAB' => 'Motion Picture & TV|Motion Picture|Set Decor|All Internet Distribution Formats',
    '2HUG' => 'Motion Picture & TV|Motion Picture|Set Decor|Recordable Media',
    '2HUM' => 'Motion Picture & TV|Motion Picture|Set Decor|All Electronic Distribution Formats',
    '2ARC' => 'Motion Picture & TV|Motion Picture|Set Decor|Projected Display',
    '2TAP' => 'Motion Picture & TV|Motion Picture|Artist\'s Reference, All Types|Projected Display',
    '2MPL' => 'Motion Picture & TV|Motion Picture|Artist\'s Reference, All Types|Internet Downloadable File',
    '2NOD' => 'Motion Picture & TV|Motion Picture|Artist\'s Reference, All Types|All Internet Distribution Formats',
    '2MIX' => 'Motion Picture & TV|Motion Picture|Artist\'s Reference, All Types|Recordable Media',
    '2NAP' => 'Motion Picture & TV|Motion Picture|Artist\'s Reference, All Types|All Electronic Distribution Formats',
    '2MPC' => 'Motion Picture & TV|Motion Picture|Artist\'s Reference, All Types|Television Broadcast',
    '2FIX' => 'Motion Picture & TV|Motion Picture|All Motion Picture Types|Projected Display',
    '2MPJ' => 'Motion Picture & TV|Motion Picture|All Motion Picture Types|Internet Downloadable File',
    '2DIP' => 'Motion Picture & TV|Motion Picture|All Motion Picture Types|All Internet Distribution Formats',
    '2MOV' => 'Motion Picture & TV|Motion Picture|All Motion Picture Types|Recordable Media',
    '2MPA' => 'Motion Picture & TV|Motion Picture|All Motion Picture Types|Television Broadcast',
    '2ACT' => 'Motion Picture & TV|Motion Picture|All Motion Picture Types|All Electronic Distribution Formats',
    '2MPB' => 'Motion Picture & TV|Music Video|All Music Video Types|Television Broadcast',
    '2MPK' => 'Motion Picture & TV|Music Video|All Music Video Types|Internet Downloadable File',
    '2BER' => 'Motion Picture & TV|Music Video|All Music Video Types|Recordable Media',
    '2BES' => 'Motion Picture & TV|Music Video|All Music Video Types|All Internet Distribution Formats',
    '2TVR' => 'Motion Picture & TV|Music Video|All Music Video Types|All Electronic Distribution Formats',
    '2AAM' => 'Motion Picture & TV|All Media Types|All Formats|All Distribution Formats',
    '2WED' => 'Personal Use|Personal Review|All Review Types|Printed',
    '2DIM' => 'Personal Use|Personal Review|All Review Types|All Electronic Distribution Formats',
    '2BFR' => 'Personal Use|Website|Web Page, All Types|Internet Website',
    '2BFU' => 'Personal Use|Website|Web Page, All Types|All Internet Distribution Formats',
    '2BFS' => 'Personal Use|Website|Web Page, All Types|Recordable Media',
    '2BFT' => 'Personal Use|Website|Web Page, All Types|All Electronic Distribution Formats',
    '2ZIT' => 'Personal Use|Mobile|Wallpaper|Mobile',
    '2ZOB' => 'Personal Use|Mobile|All Mobile Types|Mobile',
    '2BFN' => 'Personal Use|Art|Art Display, Display Print|Printed',
    '2KIT' => 'Personal Use|Art|Art Display, All Art Types|Printed',
    '2BFJ' => 'Personal Use|Art|Art Display, All Art Types|Internet Website',
    '2BFI' => 'Personal Use|Art|Art Display, All Art Types|All Internet Distribution Formats',
    '2BFK' => 'Personal Use|Art|Art Display, All Art Types|Electronic Display',
    '2BFH' => 'Personal Use|Art|Art Display, All Art Types|All Electronic Distribution Formats',
    '2TAT' => 'Personal Use|Art|Artist\'s Reference, Tattoo|Printed',
    '2BFP' => 'Personal Use|Art|Study Print, Educational|Printed',
    '2AAU' => 'Personal Use|All Media Types|All Formats|All Distribution Formats',
    '2AAB' => 'All Categories|Book|All Book Types|All Distribution Formats',
    '2AAD' => 'All Categories|Display|All Display Types|All Distribution Formats',
    '2AAT' => 'All Categories|Marketing Materials|All Marketing Material Types|All Distribution Formats',
    '2AAH' => 'All Categories|Merchandise|All Merchandise Types|All Distribution Formats',
    '2AAL' => 'All Categories|Mobile|All Mobile Types|All Distribution Formats',
    '2AAX' => 'All Categories|Motion Picture|All Motion Picture Types|All Distribution Formats',
    '2AAV' => 'All Categories|Music Video|All Music Video Types|All Distribution Formats',
    '2AAC' => 'All Categories|Periodicals|All Periodical Types|All Distribution Formats',
    '2AAR' => 'All Categories|Point of Purchase|All Point Of Purchase Types|All Distribution Formats',
    '2AAG' => 'All Categories|Product Packaging|All Product Packaging Types|All Distribution Formats',
    '2AAN' => 'All Categories|Television Programming|All Television Programming Types|All Distribution Formats',
    '2BAA' => 'All Categories|Website|All Web Page Types|All Distribution Formats',
    '2ALL' => 'All Categories|All Media Types|All Formats|All Distribution Formats',
    '2UNL' => 'Unlicensed|Not Applicable|Not Applicable|Not Applicable',
   # 3P - Placement
    '3PAA' => 'Any Placements on All Pages',
    '3PXX' => 'Not Applicable or None',
    '3PUL' => 'Any Placements',
    '3PNQ' => 'Single Placement in Content Body',
    '3PPU' => 'Single Placement on Title Page',
    '3PQT' => 'Single Placement on Table Of Contents',
    '3PQA' => 'Single Placement as Frontispiece',
    '3PQI' => 'Single Placement on Preface',
    '3PPW' => 'Single Placement as Forward',
    '3PSA' => 'Single Placement as Front Matter',
    '3PNI' => 'Single Placement as Chapter Opener',
    '3PQN' => 'Single Placement as Vignette',
    '3PRT' => 'Single Placement on Pop Up',
    '3PNZ' => 'Single Placement in Bibliography',
    '3PQZ' => 'Single Placement on Index',
    '3PQG' => 'Single Placement on Any Interior Page',
    '3PSR' => 'Single Placement on Inside Cover',
    '3PQE' => 'Single Placement on Spine',
    '3PSL' => 'Single Placement on Flap',
    '3PSP' => 'Single Placement on Back Cover',
    '3PNN' => 'Single Placement on Front Cover',
    '3PTL' => 'Single Placement on Front Cover And Back Cover',
    '3PPA' => 'Single Placement on Dust Jacket',
    '3PTD' => 'Single Placements on Interior, Covers and Jacket',
    '3PRY' => 'Multiple Placements in Content Body',
    '3PRL' => 'Multiple Placements on Any Interior Pages',
    '3PSU' => 'Multiple Placements on Inside Cover',
    '3PQS' => 'Multiple Placements on Back Cover',
    '3PRZ' => 'Multiple Placements on Front Cover',
    '3PTM' => 'Multiple Placements on Front Cover And Back Cover',
    '3PTC' => 'Multiple Placements on Dust Jacket',
    '3PTE' => 'Multiple Placements on Interior, Covers and Jacket',
    '3PST' => 'Single Placement on Any Interior Page',
    '3PPK' => 'Single Placement on Inside Cover',
    '3PNR' => 'Single Placement on Back Cover',
    '3PRB' => 'Single Placement on Front Cover',
    '3PUD' => 'Single Placement on Back Cover Or Interior Page',
    '3PTB' => 'Single Placement on Front Cover And Interior Page',
    '3PUF' => 'Single Placement on Front Cover Or Back Cover',
    '3PNF' => 'Single Placement on Front Cover And Back Cover',
    '3PTP' => 'Single Placement on Any Cover And Interior Page',
    '3PQW' => 'Multiple Placements on Any Interior Pages',
    '3PNY' => 'Multiple Placements on Inside Cover',
    '3PNC' => 'Multiple Placements on Back Cover',
    '3PPV' => 'Multiple Placements on Front Cover',
    '3PRG' => 'Multiple Placements on Front Cover And Back Cover',
    '3PTN' => 'Multiple Placements on Any Covers And Interior Pages',
    '3PQX' => 'Single Placement on Any Interior Page',
    '3PTF' => 'Single Placement on Section Opener Page',
    '3PNT' => 'Single Placement on Front Page',
    '3PTH' => 'Single Placement on Section Opener and Front Page',
    '3PSY' => 'Multiple Placements on Any Interior Pages',
    '3PTG' => 'Multiple Placements on Section Opener Page',
    '3PTI' => 'Multiple Placements on Section Opener and Front Page',
    '3PPM' => 'Multiple Placements on Any Pages',
    '3PQJ' => 'Single Placement on Wrap Around Cover',
    '3PQB' => 'Multiple Placements on Wrap Around Cover',
    '3PRS' => 'Single Placement in Content Body',
    '3PRI' => 'Single Placement as Colophon',
    '3PRC' => 'Single Placement on Table Of Contents',
    '3PQQ' => 'Single Placement as Unit Opener',
    '3PND' => 'Single Placement as Chapter Opener',
    '3PQF' => 'Single Placements in Bibliography',
    '3PRN' => 'Single Placement on Any Interior Page',
    '3PQH' => 'Single Placement on Back Cover',
    '3PSJ' => 'Single Placement on Front Cover',
    '3PTJ' => 'Single Placement on Any Covers And Interior Pages',
    '3PRD' => 'Multiple Placements in Content Body',
    '3PSM' => 'Multiple Placements on Any Interior Pages',
    '3PSN' => 'Multiple Placements on Back Cover',
    '3PPF' => 'Multiple Placements on Front Cover',
    '3PTK' => 'Multiple Placements on Any Covers And Interior Pages',
    '3PNE' => 'Single Placement on Front Side',
    '3PQC' => 'Single Placement on Back Side',
    '3PPL' => 'Single Placement on Both Sides',
    '3PNV' => 'Multiple Placements on Front Side',
    '3PSB' => 'Multiple Placements on Back Side',
    '3PQK' => 'Multiple Placements on Both Sides',
    '3PTA' => 'Single Placement on One Side',
    '3PNB' => 'Multiple Placements on One Side',
    '3PRQ' => 'Single Placement on Screen',
    '3PSD' => 'Multiple Placements on Screen',
    '3PPH' => 'Single Placement on Inside',
    '3PSQ' => 'Single Placement on Back Side',
    '3PRU' => 'Single Placement on Front Side',
    '3PNP' => 'Single Placement on Both Sides',
    '3PRK' => 'Multiple Placements on Inside',
    '3PQM' => 'Multiple Placements on Back Side',
    '3PNK' => 'Multiple Placements on Front Side',
    '3PRV' => 'Multiple Placements on Both Sides',
    '3PRH' => 'Single Placement on Any Interior Page',
    '3PSH' => 'Single Placement on Back Cover',
    '3PQY' => 'Single Placement on Front Cover',
    '3PRM' => 'Single Placement on Front Cover And Back Cover',
    '3PTQ' => 'Single Placement on Any Covers And Interior Pages',
    '3PRF' => 'Multiple Placements on Any Interior Pages',
    '3PSF' => 'Multiple Placements on Back Cover',
    '3PRJ' => 'Multiple Placements on Front Cover',
    '3PPP' => 'Multiple Placements on Front Cover And Interior Pages',
    '3PSS' => 'Multiple Placements on Front Cover And Back Cover',
    '3PTR' => 'Multiple Placements on Any Covers And Interior Pages',
    '3PSW' => 'Single Placement on Item',
    '3PNL' => 'Multiple Placements on Item',
    '3PPX' => 'Single Placement on Packaging Exterior|Front',
    '3PUJ' => 'Single Placement on Packaging Exterior|Back or Side',
    '3PPD' => 'Single Placement in Packaging Interior',
    '3PSI' => 'Multiple Placements on Packaging Exterior|Front',
    '3PUH' => 'Multiple Placements on Packaging Exterior|Back or Side',
    '3PPC' => 'Multiple Placements in Packaging Interior',
    '3PTS' => 'Single Placement Anywhere On Packaging',
    '3PTT' => 'Multiple Placements Anywhere On Packaging',
    '3PNW' => 'Single Placement in Body Of Program',
    '3PQU' => 'Single Placement in Closing Sequence',
    '3PRR' => 'Single Placement in Title Sequence',
    '3PTV' => 'Single Placement in Any Part',
    '3PPZ' => 'Multiple Placements in Body Of Program',
    '3PPN' => 'Multiple Placements in Closing Sequence',
    '3PSV' => 'Multiple Placements in Title Sequence',
    '3PTU' => 'Multiple Placements in Any Part',
    '3PRP' => 'Single Placement in Body Of Advertisement',
    '3PPJ' => 'Multiple Placements in Body Of Advertisement',
    '3PPR' => 'Single Placement in Any Part',
    '3PRW' => 'Multiple Placements in Any Part',
    '3PQD' => 'Single Placement on Secondary Page',
    '3PQP' => 'Single Placement on Pop Up',
    '3PPS' => 'Single Placement on Landing Page',
    '3PPG' => 'Single Placement on Splash Page',
    '3PPY' => 'Single Placement on Home Page',
    '3PTY' => 'Single Placement on Home Page And Secondary Pages',
    '3PTW' => 'Single Placement on Any Pages',
    '3PSZ' => 'Multiple Placements on Secondary Pages',
    '3PNJ' => 'Multiple Placements on Pop Ups',
    '3PPI' => 'Multiple Placements on Landing Pages',
    '3PNS' => 'Multiple Placements on Splash Pages',
    '3PNU' => 'Multiple Placements on Home Page',
    '3PPQ' => 'Multiple Placements on Home Page And Secondary Pages',
    '3PTZ' => 'Multiple Placements on Any Pages',
    '3PPE' => 'Single Placement as Flash',
    '3PNM' => 'Single Placement in Body Of Program',
    '3PPB' => 'Single Placement in Closing Sequence',
    '3PSC' => 'Single Placement in Title Sequence',
    '3PUC' => 'Single Placement in Any Part',
    '3PNX' => 'Multiple Placements as Flash',
    '3PNH' => 'Multiple Placements in Body Of Program',
    '3PQL' => 'Multiple Placements in Closing Sequence',
    '3PSK' => 'Multiple Placements in Title Sequence',
    '3PUB' => 'Multiple Placements in Any Part',
   # 4S - Size
    '4SAA' => 'Any Size Image|Any Size Media',
    '4SXX' => 'Not Applicable or None',
    '4SUL' => 'Any Sizes',
    '4SJA' => 'Up To 1/16 Page|Any Size Page',
    '4SBN' => 'Up To 1/8 Page Image|Any Size Page',
    '4SJB' => 'Up To 1/4 Page Image|Any Size Page',
    '4SEU' => 'Up To 1/3 Page Image|Any Size Page',
    '4SEJ' => 'Up To 1/2 Page Image|Any Size Page',
    '4SMG' => 'Up To 3/4 Page Image|Any Size Page',
    '4SAG' => 'Up To Full Page Image|Any Size Page',
    '4SFI' => 'Up To 2 Page Image|Any Size Pages',
    '4SJZ' => 'Up To 3 Page Image|Any Size Pages',
    '4SFG' => 'Up To 4 Page Image|Any Size Pages',
    '4SBP' => 'Any Size Image|Any Size Page',
    '4SGF' => 'Up To 1/4 Page Image|Up To 1/4 Page Ad',
    '4SCH' => 'Up To 1/4 Page Image|Up To 1/2 Page Ad',
    '4SIY' => 'Up To 1/4 Page Image|Up To Full Page Ad',
    '4SCV' => 'Up To 1/2 Page Image|Up To 1/2 Page Ad',
    '4SLY' => 'Up To 1/2 Page Image|Up To Full Page Ad',
    '4SBU' => 'Up To 1/2 Page Image|Up To 2 Page Ad',
    '4SCP' => 'Up To Full Page Image|Up To Full Page Ad',
    '4SGL' => 'Up To Full Page Image|Up To 2 Page Ad',
    '4SGM' => 'Up To Full Page Image|Any Size Ad',
    '4SMB' => 'Up To 2 Page Image|Up To 2 Page Ad',
    '4SFR' => 'Up To 2 Page Image|Any Size Ad',
    '4SKP' => 'Any Size Image|Any Size Ad',
    '4SBK' => 'Up To 1/16 Page Image|Any Size Page',
    '4SJS' => 'Up To 1/8 Page Image|Any Size Page',
    '4SLV' => 'Up To 1/4 Page Image|Any Size Page',
    '4SGW' => 'Up To 1/2 Page Image|Any Size Page',
    '4SKD' => 'Up To Full Page Image|Any Size Page',
    '4SFJ' => 'Up To 2 Page Image|Any Size Pages',
    '4SLA' => 'Any Size Image|Any Size Pages',
    '4SKE' => 'Up To 1/8 Page Image|Any Size Page',
    '4SAK' => 'Up To 1/4 Page Image|Any Size Page',
    '4SLD' => 'Up To 1/2 Page Image|Any Size Page',
    '4SJV' => 'Up To Full Page Image|Any Size Page',
    '4SJU' => 'Up To 2 Page Image|Any Size Pages',
    '4SIW' => 'Any Size Image|Any Size Pages',
    '4SAL' => 'Up To 1/4 Area Image|Up To Full Area Media',
    '4SJM' => 'Up To 1/2 Area Image|Up To Full Area Media',
    '4SKZ' => 'Up To Full Area Image|Up To Full Area Media',
    '4SCN' => 'Up To 1/8 Page Image|Up To Full Page Media',
    '4SLW' => 'Up To 1/4 Page Image|Up To Full Page Media',
    '4SIZ' => 'Up To 1/2 Page Image|Up To Full Page Media',
    '4SAC' => 'Up To Full Page Image|Up To Full Page Media',
    '4SFX' => 'Up To 8 x 10 Inch Image|Any Size Media',
    '4SFP' => 'Up To 11 x 14 Inch Image|Any Size Media',
    '4SMV' => 'Any Size Image|Any Size Media',
    '4SJD' => 'Up To 21 x 30 cm Image|Any Size Media',
    '4SFQ' => 'Up To 30 x 42 cm Image|Any Size Media',
    '4SJK' => 'Any Size Image|Up To Full Card',
    '4SEQ' => 'Up To 5 x 5 Inch Image|Up To 40 x 40 Inch Media',
    '4SIV' => 'Up To 10 x 10 Inch Image|Up To 40 x 40 Inch Media',
    '4SCE' => 'Up To 20 x 20 Inch Image|Up To 40 x 40 Inch Media',
    '4SDP' => 'Up To 30x 30 Inch Image|Up To 40 x 40 Inch Media',
    '4SCT' => 'Up To 40 x 40 Inch Image|Up To 40 x 40 Inch Media',
    '4SBJ' => 'Up To 20 x 20 cm Image|Up To 100 x 100 cm Media',
    '4SKI' => 'Up To 30 x 30 cm Image|Up To 100 x 100 cm Media',
    '4SJW' => 'Up To 40 x 40 cm Image|Up To 100 x 100 cm Media',
    '4SCB' => 'Up To 50 x 50 cm Image|Up To 100 x 100 cm Media',
    '4SAT' => 'Up To 75 x 75 cm Image|Up To 100 x 100 cm Media',
    '4SCI' => 'Up To 100 x 100 cm Image|Up To 100 x 100 cm Media',
    '4SGP' => 'Any Size Image|Any Size Media',
    '4SIX' => 'Up To 1/4 Area Image|Up To 11 x 36 Foot Display',
    '4SHA' => 'Up To 1/4 Area Image|Up To 10 x 40 Foot Display',
    '4SGU' => 'Up To 1/4 Area Image|Up To 14 x 48 Foot Display',
    '4SAJ' => 'Up To 1/2 Area Image|Up To 11 x 36 Foot Display',
    '4SDX' => 'Up To 1/2 Area Image|Up To 10 x 40 Foot Display',
    '4SCR' => 'Up To 1/2 Area Image|Up To 14 x 48 Foot Display',
    '4SFM' => 'Up To Full Area Image|Up To 11 x 36 Foot Display',
    '4SKV' => 'Up To Full Area Image|Up To 10 x 40 Foot Display',
    '4SGE' => 'Up To Full Area Image|Up To 14 x 48 Foot Display',
    '4SMW' => 'Up To Full Area Image|Any Size Display',
    '4SJI' => 'Up To 1/4 Area Image|Up To 4 x 8 Foot Media',
    '4SBI' => 'Up To 1/4 Area Image|Up To 10 x 8 Foot Media',
    '4SIM' => 'Up To 1/4 Area Image|Up To 12 x 24 Foot Media',
    '4SFF' => 'Up To 1/4 Area Image|Any Size Media',
    '4SAX' => 'Up To 1/2 Area Image|Up To 4 x 8 Foot Media',
    '4SLI' => 'Up To 1/2 Area Image|Up To 10 x 8 Foot Media',
    '4SLS' => 'Up To 1/2 Area Image|Up To 12 x 24 Foot Media',
    '4SMK' => 'Up To 1/2 Area Image|Any Size Media',
    '4SLG' => 'Up To Full Area Image|Up To 4 x 8 Foot Media',
    '4SAN' => 'Up To Full Area Image|Up To 10 x 8 Foot Media',
    '4SDI' => 'Up To Full Area Image|Up To 12 x 24 Foot Media',
    '4SML' => 'Up To Full Area Image|Any Size Media',
    '4SKF' => 'Up To 1/4 Area Image|Up To 672 Square Foot Display',
    '4SKR' => 'Up To 1/4 Area Image|Up To 1,200 Square Foot Display',
    '4SLK' => 'Up To 1/4 Area Image|Up To 2,400 Square Foot Display',
    '4SJQ' => 'Up To 1/4 Area Image|Up To 4,800 Square Foot Display',
    '4SEY' => 'Up To 1/4 Area Image|Up To 10,000 Square Foot Display',
    '4SFE' => 'Up To 1/4 Area Image|Any Size Display',
    '4SKS' => 'Up To 1/2 Area Image|Up To 672 Square Foot Display',
    '4SGS' => 'Up To 1/2 Area Image|Up To 1,200 Square Foot Display',
    '4SMA' => 'Up To 1/2 Area Image|Up To 2,400 Square Foot Display',
    '4SAF' => 'Up To 1/2 Area Image|Up To 4,800 Square Foot Display',
    '4SAW' => 'Up To 1/2 Area Image|Up To 10,000 Square Foot Display',
    '4SAQ' => 'Up To 1/2 Area Image|Any Size Display',
    '4SLN' => 'Up To Full Area Image|Up To 672 Square Foot Display',
    '4SDS' => 'Up To Full Area Image|Up To 1,200 Square Foot Display',
    '4SIT' => 'Up To Full Area Image|Up To 2,400 Square Foot Display',
    '4SGX' => 'Up To Full Area Image|Up To 4,800 Square Foot Display',
    '4SBD' => 'Up To Full Area Image|Up To 10,000 Square Foot Display',
    '4SBF' => 'Up To Full Area Image|Any Size Display',
    '4SDD' => 'Up To 1/4 Area Image|Up To 4 Sheet Display',
    '4SBC' => 'Up To 1/4 Area Image|Up To 8 Sheet Display',
    '4SLJ' => 'Up To 1/4 Area Image|Up To 12 Sheet Display',
    '4SAD' => 'Up To 1/4 Area Image|Up To 16 Sheet Display',
    '4SEA' => 'Up To 1/4 Area Image|Up To 30 Sheet Display',
    '4SCD' => 'Up To 1/4 Area Image|Up To 48 Sheet Display',
    '4SKX' => 'Up To 1/4 Area Image|Up To 96 Sheet Display',
    '4SMN' => 'Up To 1/4 Area Image|Any Size Display',
    '4SDU' => 'Up To 1/2 Area Image|Up To 4 Sheet Display',
    '4SBL' => 'Up To 1/2 Area Image|Up To 8 Sheet Display',
    '4SER' => 'Up To 1/2 Area Image|Up To 12 Sheet Display',
    '4SEN' => 'Up To 1/2 Area Image|Up To 16 Sheet Display',
    '4SLP' => 'Up To 1/2 Area Image|Up To 30 Sheet Display',
    '4SGQ' => 'Up To 1/2 Area Image|Up To 48 Sheet Display',
    '4SDF' => 'Up To 1/2 Area Image|Up To 96 Sheet Display',
    '4SMP' => 'Up To 1/2 Area Image|Any Size Display',
    '4SFL' => 'Up To Full Area Image|Up To 4 Sheet Display',
    '4SFS' => 'Up To Full Area Image|Up To 8 Sheet Display',
    '4SIF' => 'Up To Full Area Image|Up To 12 Sheet Display',
    '4SIS' => 'Up To Full Area Image|Up To 16 Sheet Display',
    '4SFD' => 'Up To Full Area Image|Up To 30 Sheet Display',
    '4SFU' => 'Up To Full Area Image|Up To 48 Sheet Display',
    '4SDK' => 'Up To Full Area Image|Up To 96 Sheet Display',
    '4SMQ' => 'Up To Full Area Image|Any Size Display',
    '4SKA' => 'Up To 1/4 Area Image|Up To 20 x 30 Inch Media',
    '4SIK' => 'Up To 1/4 Area Image|Up To 24 x 36 Inch Media',
    '4SFK' => 'Up To 1/4 Area Image|Up To 30 x 40 Inch Media',
    '4SDV' => 'Up To 1/4 Area Image|Up To 40 x 60 Inch Media',
    '4SMX' => 'Up To 1/4 Area Image|Any Size Media',
    '4SEF' => 'Up To 1/4 Area Image|Up To A1 Media',
    '4SCJ' => 'Up To 1/4 Area Image|Up To B1 Media',
    '4SJG' => 'Up To 1/4 Area Image|Up To A0 Media',
    '4SIP' => 'Up To 1/4 Area Image|Up To B0 Media',
    '4SKU' => 'Up To 1/2 Area Image|Up To 20 x 30 Inch Media',
    '4SAS' => 'Up To 1/2 Area Image|Up To 24 x 36 Inch Media',
    '4SGB' => 'Up To 1/2 Area Image|Up To 30 x 40 Inch Media',
    '4SBQ' => 'Up To 1/2 Area Image|Up To 40 x 60 Inch Media',
    '4SMY' => 'Up To 1/2 Area Image|Any Size Media',
    '4SKJ' => 'Up To 1/2 Area Image|Up To A1 Media',
    '4SAM' => 'Up To 1/2 Area Image|Up To B1 Media',
    '4SDZ' => 'Up To 1/2 Area Image|Up To A0 Media',
    '4SGT' => 'Up To 1/2 Area Image|Up To B0 Media',
    '4SKN' => 'Up To Full Area Image|Up To 20 x 30 Inch Media',
    '4SIQ' => 'Up To Full Area Image|Up To 24 x 36 Inch Media',
    '4SLT' => 'Up To Full Area Image|Up To 30 x 40 Inch Media',
    '4SDN' => 'Up To Full Area Image|Up To 40 x 60 Inch Media',
    '4SMZ' => 'Up To Full Area Image|Any Size Media',
    '4SBH' => 'Up To Full Area Image|Up To A1 Media',
    '4SLU' => 'Up To Full Area Image|Up To B1 Media',
    '4SEL' => 'Up To Full Area Image|Up To A0 Media',
    '4SBM' => 'Up To Full Area Image|Up To B0 Media',
    '4SIH' => 'Up To 1/4 Area Image|Up To 24 x 30 Inch Display',
    '4SKW' => 'Up To 1/4 Area Image|Up To 30 x 40 Inch Display',
    '4SDM' => 'Up To 1/4 Area Image|Up To 27 x 85 Inch Display',
    '4SJP' => 'Up To 1/4 Area Image|Up To 60 x 40 Inch Display',
    '4SIB' => 'Up To 1/4 Area Image|Up To 69 x 48 Inch Display',
    '4SDH' => 'Up To 1/4 Area Image|Up To 27 x 141 Inch Display',
    '4SJL' => 'Up To 1/4 Area Image|Up To 30 x 240 Inch Display',
    '4SLQ' => 'Up To 1/4 Area Image|Up To B1 Display',
    '4SLL' => 'Up To 1/4 Area Image|Up To A0 Display',
    '4SEI' => 'Up To 1/4 Area Image|Up To B0 Display',
    '4SBA' => 'Up To 1/2 Area Image|Up To 24 x 30 Inch Display',
    '4SCL' => 'Up To 1/2 Area Image|Up To 30 x 40 Inch Display',
    '4SIC' => 'Up To 1/2 Area Image|Up To 27 x 85 Inch Display',
    '4SBY' => 'Up To 1/2 Area Image|Up To 60 x 40 Inch Display',
    '4SFZ' => 'Up To 1/2 Area Image|Up To 27 x 141 Inch Display',
    '4SAU' => 'Up To 1/2 Area Image|Up To 26 x 241 Inch Display',
    '4SBR' => 'Up To 1/2 Area Image|Up To 30 x 240 Inch Display',
    '4SCC' => 'Up To 1/2 Area Image|Up To B1 Display',
    '4SAB' => 'Up To 1/2 Area Image|Up To A0 Display',
    '4SLX' => 'Up To 1/2 Area Image|Up To B0 Display',
    '4SGI' => 'Up To Full Area Image|Up To 24 x 30 Inch Display',
    '4SBT' => 'Up To Full Area Image|Up To 30 x 40 Inch Display',
    '4SGK' => 'Up To Full Area Image|Up To 27 x 85 Inch Display',
    '4SIL' => 'Up To Full Area Image|Up To 60 x 40 Inch Display',
    '4SAH' => 'Up To Full Area Image|Up To 69 x 48 Inch Display',
    '4SGA' => 'Up To Full Area Image|Up To 27 x 141 Inch Display',
    '4SDR' => 'Up To Full Area Image|Up To 26 x 241 Inch Display',
    '4SID' => 'Up To Full Area Image|Up To 30 x 240 Inch Display',
    '4SCX' => 'Up To Full Area Image|Any Size Display',
    '4SGR' => 'Up To Full Area Image|Up To B1 Display',
    '4SEC' => 'Up To Full Area Image|Up To A0 Display',
    '4SET' => 'Up To Full Area Image|Up To B0 Display',
    '4SKT' => 'Up To 1/4 Area Image|Up To 43 x 62 Inch Display',
    '4SEW' => 'Up To 1/4 Area Image|Up To 48 x 71 Inch Display',
    '4SEH' => 'Up To 1/4 Area Image|Up To 43 x 126 Inch Display',
    '4SBS' => 'Up To 1/4 Area Image|Up To 83 x 135 Inch Display',
    '4SMR' => 'Up To 1/4 Area Image|Any Size Display',
    '4SJX' => 'Up To 1/4 Area Image|Up To B0 Display',
    '4SAY' => 'Up To 1/2 Area Image|Up To 43 x 62 Inch Display',
    '4SKY' => 'Up To 1/2 Area Image|Up To 48 x 71 Inch Display',
    '4SBV' => 'Up To 1/2 Area Image|Up To 43 x 126 Inch Display',
    '4SEP' => 'Up To 1/2 Area Image|Up To 83 x 135 Inch Display',
    '4SNG' => 'Up To 1/2 Area Image|Any Size Display',
    '4SAZ' => 'Up To 1/2 Area Image|Up To B0 Display',
    '4SDW' => 'Up To Full Area Image|Up To 43 x 62 Inch Display',
    '4SFW' => 'Up To Full Area Image|Up To 48 x 71 Inch Display',
    '4SGN' => 'Up To Full Area Image|Up To 43 x 126 Inch Display',
    '4SBX' => 'Up To Full Area Image|Up To 83 x 135 Inch Display',
    '4SNO' => 'Up To Full Area Image|Any Size Display',
    '4SKK' => 'Up To Full Area Image|Up To B0 Display',
    '4SAP' => 'Up To 1/4 Screen Image|Up To 32 Inch Screen',
    '4SEK' => 'Up To 1/4 Screen Image|Up To 63 Inch Screen',
    '4SLZ' => 'Up To 1/4 Screen Image|Up To 10 Diagonal Foot Screen',
    '4SIE' => 'Up To 1/4 Screen Image|Up To 30 Diagonal Foot Screen',
    '4SEM' => 'Up To 1/4 Screen Image|Up To 100 Diagonal Foot Screen',
    '4SGJ' => 'Up To 1/2 Screen Image|Up To 32 Inch Screen',
    '4SBE' => 'Up To 1/2 Screen Image|Up To 63 Inch Screen',
    '4SIJ' => 'Up To 1/2 Screen Image|Up To 10 Diagonal Foot Screen',
    '4SGY' => 'Up To 1/2 Screen Image|Up To 30 Diagonal Foot Screen',
    '4SDQ' => 'Up To 1/2 Screen Image|Up To 100 Diagonal Foot Screen',
    '4SBZ' => 'Up To Full Screen Image|Up To 32 Inch Screen',
    '4SJF' => 'Up To Full Screen Image|Up To 63 Inch Screen',
    '4SLE' => 'Up To Full Screen Image|Up To 10 Diagonal Foot Screen',
    '4SJY' => 'Up To Full Screen Image|Up To 30 Diagonal Foot Screen',
    '4SKC' => 'Up To Full Screen Image|Up To 100 Diagonal Foot Screen',
    '4SDL' => 'Up To Full Screen Image|Any Size Screen',
    '4SFA' => 'Up To 1/4 Area Image|Any Size Item',
    '4SCG' => 'Up To 1/2 Area Image|Any Size Item',
    '4SLR' => 'Up To Full Area Image|Any Size Item',
    '4SGZ' => 'Up To 3 x 4.5 Inch Image|Any Size Media',
    '4SFH' => 'Up To 5 x 7 Inch Image|Any Size Media',
    '4SCU' => 'Up To 6 x 9 Inch Image|Any Size Media',
    '4SLF' => 'Up To 8 x 10 Inch Image|Any Size Media',
    '4SAV' => 'Up To 8 x 12 Inch Image|Any Size Media',
    '4SKB' => 'Up To 11 x 14 Inch Image|Any Size Media',
    '4SAE' => 'Up To 14 x 20 Inch Image|Any Size Media',
    '4SEZ' => 'Up To 16 x 20 Inch Image|Any Size Media',
    '4SDB' => 'Up To 20 x 24 Inch Image|Any Size Media',
    '4SDG' => 'Up To 24 x 30 Inch Image|Any Size Media',
    '4SDE' => 'Up To 30 x 40 Inch Image|Any Size Media',
    '4SJE' => 'Up To 38 x 50 Inch Image|Any Size Media',
    '4SCM' => 'Up To 40 x 60 Inch Image|Any Size Media',
    '4SIN' => 'Any Size Image|Any Size Media',
    '4SDA' => 'Up To 11 x 15 cm Image|Any Size Media',
    '4SME' => 'Up To 13 x 18 cm Image|Any Size Media',
    '4SFC' => 'Up To 15 x 21 cm Image|Any Size Media',
    '4SIG' => 'Up To 18 x 25 cm Image|Any Size Media',
    '4SEV' => 'Up To 21 x 30 cm Image|Any Size Media',
    '4SJJ' => 'Up To 25 x 36 cm Image|Any Size Media',
    '4SMD' => 'Up To 25 x 36 cm Image|Any Size Media',
    '4SLH' => 'Up To 30 x 42 cm Image|Any Size Media',
    '4SEE' => 'Up To 42 x 60 cm Image|Any Size Media',
    '4SDC' => 'Up To 50 x 71 cm Image|Any Size Media',
    '4SDY' => 'Up To 60 x 85 cm Image|Any Size Media',
    '4SCQ' => 'Up To 70 x 100 cm Image|Any Size Media',
    '4SMF' => 'Up To 85 x 119 cm Image|Any Size Media',
    '4SJC' => 'Up To 100 x 142 cm Image|Any Size Media',
    '4SFN' => 'Up To 1/4 Area Image|Up To 25 x 13 Inch Media',
    '4SLC' => 'Up To 1/4 Area Image|Up To 50 x 24 Inch Media',
    '4SIR' => 'Up To 1/4 Area Image|Up To 26 x 53 Inch Media',
    '4SDT' => 'Up To 1/4 Area Image|Up To 46 x 60 Inch Media',
    '4SFB' => 'Up To 1/4 Area Image|Up To 138 x 53 Inch Media',
    '4SGV' => 'Up To 1/2 Area Image|Up To 25 x 13 Inch Media',
    '4SJR' => 'Up To 1/2 Area Image|Up To 50 x 24 Inch Media',
    '4SKL' => 'Up To 1/2 Area Image|Up To 26 x 53 Inch Media',
    '4SEX' => 'Up To 1/2 Area Image|Up To 46 x 60 Inch Media',
    '4SBW' => 'Up To 1/2 Area Image|Up To 138 x 53 Inch Media',
    '4SJN' => 'Up To Full Area Image|Up To 25 x 13 Inch Media',
    '4SKH' => 'Up To Full Area Image|Up To 50 x 24 Inch Media',
    '4SKM' => 'Up To Full Area Image|Up To 26 x 53 Inch Media',
    '4SGG' => 'Up To Full Area Image|Up To 46 x 60 Inch Media',
    '4SDJ' => 'Up To Full Area Image|Up To 138 x 53 Inch Media',
    '4SCZ' => 'Up To 1/4 Screen Image|Up To 15 Inch Screen',
    '4SCS' => 'Up To 1/4 Screen Image|Up To 21 Inch Screen',
    '4SMS' => 'Up To 1/4 Screen Image|Any Size Screen',
    '4SED' => 'Up To 1/2 Screen Image|Up To 15 Inch Screen',
    '4SFV' => 'Up To 1/2 Screen Image|Up To 21 Inch Screen',
    '4SMT' => 'Up To 1/2 Screen Image|Any Size Screen',
    '4SCY' => 'Up To Full Screen Image|Up To 15 Inch Screen',
    '4SEB' => 'Up To Full Screen Image|Up To 21 Inch Screen',
    '4SMU' => 'Up To Full Screen Image|Any Size Screen',
    '4SGH' => 'Up To 1/4 Screen Image|Any Size Screen',
    '4SGC' => 'Up To 1/2 Screen Image|Any Size Screen',
    '4SES' => 'Up To Full Screen Image|Any Size Screen',
    '4SKQ' => 'Up To 1/4 Area Image|Up To 180 x 150 Pixels Ad',
    '4SCA' => 'Up To 1/4 Area Image|Up To 468 x 60 Pixels Ad',
    '4SCK' => 'Up To 1/4 Area Image|Up To  728 x 90 Pixels Ad',
    '4SCF' => 'Up To 1/4 Area Image|Up To  300 x 600 Pixels Ad',
    '4SIU' => 'Up To 1/4 Area Image|Up To Full Screen Ad',
    '4SAI' => 'Up To 1/2 Area Image|Up To 180 x 150 Pixels Ad',
    '4SII' => 'Up To 1/2 Area Image|Up To 468 x 60 Pixels Ad',
    '4SCW' => 'Up To 1/2 Area Image|Up To  728 x 90 Pixels Ad',
    '4SLM' => 'Up To 1/2 Area Image|Up To  300 x 600 Pixels Ad',
    '4SFT' => 'Up To 1/2 Area Image|Up To Full Screen Ad',
    '4SJT' => 'Up To Full Area Image|Up To 180 x 150 Pixels Ad',
    '4SGD' => 'Up To Full Area Image|Up To 468 x 60 Pixels Ad',
    '4SAR' => 'Up To Full Area Image|Up To  728 x 90 Pixels Ad',
    '4SEG' => 'Up To Full Area Image|Up To  300 x 600 Pixels Ad',
    '4SBG' => 'Any Size Image|Up To Full Screen Ad',
    '4SMH' => 'Up To 150 x 150 Pixels Image|Any Size Screen',
    '4SMJ' => 'Up To 300 x 600 Pixels Image|Any Size Screen',
    '4SKG' => 'Any Size Image|Any Size Screen',
   # 5V - Version
    '5VAA' => 'All Versions',
    '5VXX' => 'Not Applicable or None',
    '5VUL' => 'Any Versions',
    '5VVM' => 'Single Print Version',
    '5VVB' => 'Multiple Print Versions',
    '5VUP' => 'Single Version',
    '5VUY' => 'Multiple Versions',
    '5VVC' => 'Single Paperback Edition',
    '5VUK' => 'Single Hardcover Edition',
    '5VVH' => 'Single Edition in All Binding Formats',
    '5VVK' => 'Multiple Paperback Editions',
    '5VUU' => 'Multiple Hardcover Editions',
    '5VVL' => 'Multiple Editions in All Binding Formats',
    '5VUZ' => 'Single Issue',
    '5VVJ' => 'Multiple Issues',
    '5VUG' => 'Single Edition',
    '5VVG' => 'Multiple Editions',
   # 6Q - Quantity
    '6QAA' => 'Any Quantity',
    '6QXX' => 'Not Applicable or None',
    '6QUL' => 'Any Quantity',
    '6QAU' => 'One|Print Run',
    '6QDA' => 'Up To 10|Print Run',
    '6QEY' => 'Up To 100|Print Run',
    '6QBA' => 'Up To 1,000|Print Run',
    '6QCW' => 'Up To 5,000|Print Run',
    '6QEC' => 'Up To 10,000|Print Run',
    '6QFD' => 'Up To 25,000|Print Run',
    '6QBH' => 'Up To 40,000|Print Run',
    '6QFR' => 'Up To 50,000|Print Run',
    '6QDQ' => 'Up To 100,000|Print Run',
    '6QBG' => 'Up To 250,000|Print Run',
    '6QEN' => 'Up To 500,000|Print Run',
    '6QDL' => 'Any Quantity Of|Print Run',
    '6QBU' => 'One|Copy',
    '6QDR' => 'Up To 10|Total Circulation',
    '6QFU' => 'Up To 25|Total Circulation',
    '6QFV' => 'Up To 50|Total Circulation',
    '6QEJ' => 'Up To 100|Total Circulation',
    '6QFW' => 'Up To 250|Total Circulation',
    '6QFY' => 'Up To 500|Total Circulation',
    '6QCD' => 'Up To 1,000|Total Circulation',
    '6QFZ' => 'Up To 2,500|Total Circulation',
    '6QAT' => 'Up To 5,000|Total Circulation',
    '6QBB' => 'Up To 10,000|Total Circulation',
    '6QFS' => 'Up To 25,000|Total Circulation',
    '6QAN' => 'Up To 50,000|Total Circulation',
    '6QCS' => 'Up To 100,000|Total Circulation',
    '6QDT' => 'Up To 250,000|Total Circulation',
    '6QCI' => 'Up To 500,000|Total Circulation',
    '6QCN' => 'Up To 1 Million|Total Circulation',
    '6QCK' => 'Up To 2 Million|Total Circulation',
    '6QCL' => 'Up To 3 Million|Total Circulation',
    '6QAV' => 'Up To 5 Million|Total Circulation',
    '6QDK' => 'Up To 10 Million|Total Circulation',
    '6QGB' => 'Up To 25 Million|Total Circulation',
    '6QGC' => 'Up To 50 Million|Total Circulation',
    '6QEV' => 'Any Quantity Of|Circulation',
    '6QFI' => 'One|Reprint',
    '6QDG' => 'Up To 10|Reprints',
    '6QCB' => 'Up To 100|Reprints',
    '6QAY' => 'Up To 1,000|Reprints',
    '6QBF' => 'Up To 10,000|Reprints',
    '6QCU' => 'Any Quantity Of|Reprints',
    '6QEH' => 'One|Print Run',
    '6QDH' => 'Up To 10|Print Run',
    '6QAZ' => 'Up To 100|Print Run',
    '6QFB' => 'Up To 1,000|Print Run',
    '6QAQ' => 'Up To 5,000|Print Run',
    '6QEG' => 'Up To 10,000|Print Run',
    '6QBV' => 'Up To 25,000|Print Run',
    '6QCG' => 'Up To 50,000|Print Run',
    '6QAH' => 'Up To 100,000|Print Run',
    '6QDE' => 'Up To 250,000|Print Run',
    '6QDJ' => 'Up To 500,000|Print Run',
    '6QCZ' => 'Up To 1 Million|Print Run',
    '6QCT' => 'Up To 2 Million|Print Run',
    '6QAR' => 'Up To 3 Million|Print Run',
    '6QDC' => 'Up To 5 Million|Print Run',
    '6QDU' => 'Up To 10 Million|Print Run',
    '6QEU' => 'Any Quantity Of|Print Run',
    '6QFT' => 'One|Copy',
    '6QDI' => 'One|Display',
    '6QDY' => 'Up To 5|Displays',
    '6QFJ' => 'Up To 10|Displays',
    '6QEL' => 'Up To 25|Displays',
    '6QBD' => 'Up To 50|Displays',
    '6QAE' => 'Up To 100|Displays',
    '6QCE' => 'Up To 250|Displays',
    '6QAP' => 'Up To 500|Displays',
    '6QBY' => 'Up To 1,000|Displays',
    '6QBC' => 'Up To 2,500|Displays',
    '6QET' => 'Up To 5,000|Displays',
    '6QFF' => 'Up To 10,000|Displays',
    '6QDS' => 'Up To 25,000|Displays',
    '6QES' => 'Up To 50,000|Displays',
    '6QDZ' => 'Up To 100,000|Displays',
    '6QBW' => 'Up To 250,000|Displays',
    '6QAX' => 'Up To 500,000|Displays',
    '6QDB' => 'Up To 1 Million|Displays',
    '6QEK' => 'Up To 2 Million|Displays',
    '6QFA' => 'Up To 3 Million|Displays',
    '6QCJ' => 'Up To 5 Million|Displays',
    '6QFQ' => 'Any Quantity Of|Displays',
    '6QCX' => 'One|Display',
    '6QBP' => 'Up To 5|Displays',
    '6QAL' => 'Up To 10|Displays',
    '6QDX' => 'Up To 25|Displays',
    '6QEM' => 'Up To 50|Displays',
    '6QEA' => 'Up To 100|Displays',
    '6QFG' => 'Up To 250|Displays',
    '6QAG' => 'Up To 500|Displays',
    '6QEI' => 'Any Quantity Of|Displays',
    '6QEP' => 'One|Copy',
    '6QEX' => 'Two|Copies',
    '6QCF' => 'Three|Copies',
    '6QBK' => 'Four|Copies',
    '6QBL' => 'Five|Copies',
    '6QCP' => 'Up To 10|Copies',
    '6QDV' => 'Up To 50|Copies',
    '6QCM' => 'Up To 100|Copies',
    '6QAI' => 'Up To 500|Copies',
    '6QAS' => 'Up To 1,000|Copies',
    '6QDW' => 'Up To 5,000|Copies',
    '6QCQ' => 'Up To 10,000|Copies',
    '6QAM' => 'Any Quantity Of|Copies',
    '6QBE' => 'One|Copy',
    '6QFC' => 'Up To 5|Copies',
    '6QEB' => 'Up To 10|Copies',
    '6QGD' => 'Up To 25|Copies',
    '6QGE' => 'Up To 50|Copies',
    '6QCR' => 'Up To 100|Copies',
    '6QCA' => 'Up To 250|Copies',
    '6QBS' => 'Up To 500|Copies',
    '6QDF' => 'Up To 1,000|Copies',
    '6QEW' => 'Up To 2,500|Copies',
    '6QFE' => 'Up To 5,000|Copies',
    '6QEE' => 'Up To 10,000|Copies',
    '6QBT' => 'Up To 25,000|Copies',
    '6QDP' => 'Up To 50,000|Copies',
    '6QDN' => 'Up To 100,000|Copies',
    '6QFN' => 'Any Quantity Of|Copies',
    '6QFH' => 'Up To 10,000|Viewers',
    '6QCV' => 'Up To 100,000|Viewers',
    '6QBN' => 'Up To 1 Million|Viewers',
    '6QFP' => 'Up To 10 Million|Viewers',
    '6QBI' => 'Up To 100 Million|Viewers',
    '6QBQ' => 'Any Quantity Of|Viewers',
    '6QAB' => 'Up To 10,000|Impressions',
    '6QBR' => 'Up To 100,000|Impressions',
    '6QCY' => 'Up To 1 Million|Impressions',
    '6QBM' => 'Up To 10 Million|Impressions',
    '6QFM' => 'Any Quantity Of|Impressions',
    '6QDD' => 'Up To 100|Viewers',
    '6QEQ' => 'Up To 1,000|Viewers',
    '6QEZ' => 'Up To 10,000|Viewers',
    '6QBZ' => 'Up To 100,000|Viewers',
    '6QCC' => 'Any Quantity Of|Viewers',
    '6QCH' => 'One|Copy',
    '6QAW' => 'Up To 5|Copies',
    '6QAK' => 'Up To 10|Copies',
    '6QFL' => 'Up To 100|Copies',
    '6QAC' => 'Up To 1,000|Copies',
    '6QAD' => 'Up To 10,000|Copies',
    '6QEF' => 'Up To 25,000|Copies',
    '6QAF' => 'Up To 50,000|Copies',
    '6QED' => 'Up To 100,000|Copies',
    '6QFK' => 'Up To 250,000|Copies',
    '6QDM' => 'Up To 500,000|Copies',
    '6QBX' => 'Up To 1 Million|Copies',
    '6QER' => 'Any Quantity Of|Copies',
   # 7D - Duration
    '7DAA' => 'In Perpetuity',
    '7DXX' => 'Not Applicable or None',
    '7DUL' => 'Any Durations',
    '7DUT' => 'Up To 10 Years',
    '7DUS' => 'Up To 6 Months',
    '7DUV' => 'Up To 6 Months',
    '7DUW' => 'Up To 1 Year',
    '7DUY' => 'Up To 3 Years',
    '7DUZ' => 'Up To 5 Years',
    '7DUQ' => 'Up To 10 Years',
    '7DXC' => 'Up To 1 Day',
    '7DXL' => 'Up To 1 Week',
    '7DXH' => 'Life Of Publication',
    '7DYF' => 'Up To 1 Day',
    '7DWL' => 'Up To 1 Week',
    '7DYJ' => 'Up To 2 Weeks',
    '7DXF' => 'Up To 1 Month',
    '7DZF' => 'Up To 2 Months',
    '7DWC' => 'Up To 3 Months',
    '7DWW' => 'Up To 6 Months',
    '7DZA' => 'Up To 1 Year',
    '7DZP' => 'Up To 2 Years',
    '7DZJ' => 'Up To 3 Years',
    '7DZK' => 'Up To 5 Years',
    '7DZL' => 'Up To 10 Years',
    '7DWS' => 'Life Of Publication',
    '7DXZ' => 'Up To 3 Months',
    '7DXW' => 'Up To 6 Months',
    '7DXD' => 'Up To 1 Year',
    '7DYS' => 'Life Of Publication',
    '7DYM' => 'Up To 1 Year',
    '7DWE' => 'Up To 2 Years',
    '7DYL' => 'Up To 3 Years',
    '7DWV' => 'Up To 5 Years',
    '7DWB' => 'Up To 7 Years',
    '7DYY' => 'Up To 10 Years',
    '7DWG' => 'Up To 15 Years',
    '7DXA' => 'Full Term Of Copyright',
    '7DXT' => 'Up To 1 Day',
    '7DXB' => 'Up To 1 Week',
    '7DYI' => 'Up To 1 Month',
    '7DWI' => 'Up To 1 Year',
    '7DZM' => 'Up To 2 Years',
    '7DZN' => 'Up To 3 Years',
    '7DYV' => 'Life Of Event',
    '7DXQ' => 'Up To 1 Year',
    '7DZH' => 'Up To 2 Years',
    '7DXP' => 'Up To 3 Years',
    '7DYD' => 'Up To 5 Years',
    '7DWU' => 'Up To 7 Years',
    '7DXG' => 'Up To 10 Years',
    '7DXV' => 'Up To 15 Years',
    '7DWR' => 'In Perpetuity',
    '7DXM' => 'Up To 1 Year',
    '7DWK' => 'Up To 2 Years',
    '7DXS' => 'In Perpetuity',
    '7DYG' => 'Up To 1 Day',
    '7DZD' => 'Up To 1 Week',
    '7DWP' => 'Up To 13 Weeks',
    '7DYT' => 'Up To 26 Weeks',
    '7DWT' => 'Up To 52 Weeks',
    '7DYA' => 'Up To 5 Years',
    '7DXR' => 'Up To 10 Years',
    '7DXI' => 'In Perpetuity',
    '7DYW' => 'Up To 1 Day',
    '7DYP' => 'Up To 1 Week',
    '7DWJ' => 'Up To 1 Month',
    '7DYE' => 'Up To 3 Months',
    '7DYX' => 'Up To 6 Months',
    '7DXE' => 'Up To 1 Year',
    '7DWZ' => 'Up To 2 Years',
    '7DZB' => 'Up To 3 Years',
    '7DYN' => 'Up To 5 Years',
    '7DYC' => 'Up To 10 Years',
    '7DXY' => 'Up To 15 Years',
    '7DWM' => 'In Perpetuity',
    '7DYZ' => 'Up To 1 Day',
    '7DXK' => 'Up To 1 Month',
    '7DWY' => 'Up To 3 Months',
    '7DYB' => 'Up To 6 Months',
    '7DZG' => 'Up To 1 Year',
    '7DYK' => 'Up To 2 Years',
    '7DYQ' => 'In Perpetuity',
   # 8R - Region
    '8RAA' => 'Worldwide',
    '8RXX' => 'Not Applicable or None',
    '8RUL' => 'Any Regions',
    '8RWA' => 'Broad International Region|Worldwide Excluding Northern America',
    '8RWB' => 'Broad International Region|Worldwide Excluding USA',
    '8RWC' => 'Broad International Region|Worldwide Excluding USA and Europe',
    '8RWD' => 'Broad International Region|Worldwide Excluding Europe',
    '8RWE' => 'Broad International Region|Worldwide Excluding USA and UK',
    '8RWF' => 'Broad International Region|Worldwide Excluding UK',
    '8RWG' => 'Broad International Region|All English Speaking Countries',
    '8RWH' => 'Broad International Region|All English Speaking Countries Excluding USA',
    '8RAH' => 'Broad International Region|All Spanish Speaking Countries',
    '8RWI' => 'Broad International Region|All Spanish Speaking Countries Excluding USA',
    '8RFY' => 'Broad International Region|All Americas',
    '8REK' => 'Broad International Region|Europe, Middle East and Africa',
    '8RWJ' => 'Broad International Region|USA, Canada and Mexico',
    '8RQV' => 'Northern America|One Minor City, Up To 250,000 Population',
    '8RQT' => 'Northern America|One Major City, Over 250,000 Population',
    '8RQU' => 'Northern America|One Metropolitan Area, Adjoining Cities',
    '8RRJ' => 'Northern America|One State Or Province',
    '8RYK' => 'Northern America|Up To 3 States Or Provinces',
    '8RCD' => 'Northern America|Up To 5 States Or Provinces',
    '8RGJ' => 'Northern America|All Northern American Countries',
    '8RCE' => 'Northern America|USA and Canada',
    '8RHQ' => 'Northern America|USA',
    '8RCA' => 'Northern America|Canada',
    '8RHJ' => 'Northern America|USA-Central',
    '8RHP' => 'Northern America|USA-Midwest',
    '8RHR' => 'Northern America|USA-Northeast',
    '8RHS' => 'Northern America|USA-Pacific Northwest',
    '8RHW' => 'Northern America|USA-Southeast',
    '8RHX' => 'Northern America|USA-Southwest',
    '8RIA' => 'Northern America|USA-West',
    '8RHY' => 'Northern America|USA-Minor Outlying Islands',
    '8RUK' => 'Northern America|USA-All Territories, Protectorates, Dependencies, Outposts',
    '8RBK' => 'Northern America|Canada-British Columbia',
    '8RUM' => 'Northern America|Canada-Prairies',
    '8RUN' => 'Northern America|Canada-Atlantic Provinces',
    '8RCJ' => 'Northern America|Canada-Ontario',
    '8RUP' => 'Northern America|Canada-Quebec',
    '8RUQ' => 'Northern America|Canada-Northern Territories',
    '8RBM' => 'Northern America|Bermuda',
    '8RGL' => 'Northern America|Greenland',
    '8RPM' => 'Northern America|Saint Pierre and Miquelon',
    '8RQL' => 'Europe|One Minor City, Up To 250,000 Population',
    '8RQJ' => 'Europe|One Major City, Over 250,000 Population',
    '8RQK' => 'Europe|One Metropolitan Area, Adjoining Cities',
    '8RRF' => 'Europe|One State Or Province',
    '8RYF' => 'Europe|Up To 3 States Or Provinces',
    '8RYG' => 'Europe|Up To 5 States Or Provinces',
    '8RDU' => 'Europe|All Europe',
    '8REL' => 'Europe|All European Union Countries',
    '8REI' => 'Europe|All United Kingdom',
    '8REJ' => 'Europe|All Western Europe',
    '8RED' => 'Europe|All Northern Europe',
    '8REB' => 'Europe|All European Mediterranean Countries',
    '8REA' => 'Europe|All Eastern Europe',
    '8RDW' => 'Europe|All Baltic States',
    '8RDX' => 'Europe|All Benelux',
    '8RDY' => 'Europe|All Caucasian States',
    '8REF' => 'Europe|All Scandinavia',
    '8RAX' => 'Europe|Aland Islands',
    '8RAL' => 'Europe|Albania',
    '8RAD' => 'Europe|Andorra',
    '8RAM' => 'Europe|Armenia',
    '8RAT' => 'Europe|Austria',
    '8RAZ' => 'Europe|Azerbaijan',
    '8RBY' => 'Europe|Belarus',
    '8RBE' => 'Europe|Belgium',
    '8RBA' => 'Europe|Bosnia and Herzegovina',
    '8RBG' => 'Europe|Bulgaria',
    '8RDR' => 'Europe|Croatia',
    '8RCY' => 'Europe|Cyprus',
    '8RCZ' => 'Europe|Czech Republic',
    '8RDK' => 'Europe|Denmark',
    '8REM' => 'Europe|England',
    '8REE' => 'Europe|Estonia',
    '8RDS' => 'Europe|Faeroe Islands',
    '8RFI' => 'Europe|Finland',
    '8RFR' => 'Europe|France',
    '8RGE' => 'Europe|Georgia',
    '8RDE' => 'Europe|Germany',
    '8RGI' => 'Europe|Gibraltar',
    '8RGR' => 'Europe|Greece',
    '8RGG' => 'Europe|Guernsey',
    '8RHU' => 'Europe|Hungary',
    '8RIS' => 'Europe|Iceland',
    '8RIE' => 'Europe|Ireland',
    '8RIM' => 'Europe|Isle Of Man',
    '8RIT' => 'Europe|Italy',
    '8RJE' => 'Europe|Jersey',
    '8RLV' => 'Europe|Latvia',
    '8RLI' => 'Europe|Liechtenstein',
    '8RLT' => 'Europe|Lithuania',
    '8RLU' => 'Europe|Luxembourg',
    '8RMK' => 'Europe|Macedonia',
    '8RMT' => 'Europe|Malta',
    '8RMD' => 'Europe|Moldova',
    '8RMC' => 'Europe|Monaco',
    '8RNL' => 'Europe|Netherlands',
    '8RUH' => 'Europe|Northern Ireland',
    '8RNO' => 'Europe|Norway',
    '8RPL' => 'Europe|Poland',
    '8RPT' => 'Europe|Portugal',
    '8RRO' => 'Europe|Romania',
    '8RRU' => 'Europe|Russian Federation',
    '8RSM' => 'Europe|San Marino',
    '8REN' => 'Europe|Scotland',
    '8RCS' => 'Europe|Serbia and Montenegro',
    '8RSK' => 'Europe|Slovakia',
    '8RSI' => 'Europe|Slovenia',
    '8RES' => 'Europe|Spain',
    '8RSE' => 'Europe|Sweden',
    '8RCH' => 'Europe|Switzerland',
    '8RUA' => 'Europe|Ukraine',
    '8RUI' => 'Europe|Wales',
    '8RDT' => 'Europe|Vatican City State',
    '8RQF' => 'Asia|One Minor City, Up To 250,000 Population',
    '8RQD' => 'Asia|One Major City, Over 250,000 Population',
    '8RQE' => 'Asia|One Metropolitan Area, Adjoining Cities',
    '8RRC' => 'Asia|One State Or Province',
    '8RAV' => 'Asia|Up To 3 States Or Provinces',
    '8RYD' => 'Asia|Up To 5 States Or Provinces',
    '8RDB' => 'Asia|All Asia',
    '8RDC' => 'Asia|All Central Asia',
    '8RCP' => 'Asia|All Eastern Asia',
    '8RDG' => 'Asia|All Southern Asia',
    '8RDH' => 'Asia|All Southeastern Asia',
    '8RBD' => 'Asia|Bangladesh',
    '8RBT' => 'Asia|Bhutan',
    '8RBN' => 'Asia|Brunei Darussalam',
    '8RKH' => 'Asia|Cambodia',
    '8RCN' => 'Asia|All China',
    '8RUB' => 'Asia|China-East',
    '8RAY' => 'Asia|China-Northeast',
    '8RUC' => 'Asia|China-North',
    '8RUD' => 'Asia|China-South Central',
    '8RUF' => 'Asia|China-Southwest',
    '8RHK' => 'Asia|Hong Kong',
    '8RIN' => 'Asia|India',
    '8RID' => 'Asia|Indonesia',
    '8RJP' => 'Asia|Japan',
    '8RKZ' => 'Asia|Kazakhstan',
    '8RKP' => 'Asia|North Korea',
    '8RKR' => 'Asia|South Korea',
    '8RKG' => 'Asia|Kyrgyzstan',
    '8RLA' => 'Asia|Laos',
    '8RMO' => 'Asia|Macao',
    '8RMY' => 'Asia|Malaysia',
    '8RMV' => 'Asia|Maldives',
    '8RMN' => 'Asia|Mongolia',
    '8RMM' => 'Asia|Myanmar',
    '8RNP' => 'Asia|Nepal',
    '8RPK' => 'Asia|Pakistan',
    '8RPH' => 'Asia|Philippines',
    '8RSG' => 'Asia|Singapore',
    '8RTW' => 'Asia|Taiwan',
    '8RTJ' => 'Asia|Tajikistan',
    '8RTH' => 'Asia|Thailand',
    '8RDA' => 'Asia|Tibet',
    '8RTL' => 'Asia|Timor-Leste',
    '8RTM' => 'Asia|Turkmenistan',
    '8RUZ' => 'Asia|Uzbekistan',
    '8RVN' => 'Asia|Viet Nam',
    '8RQO' => 'Latin America and Caribbean|One Minor City, Up To 250,000 Population',
    '8RQM' => 'Latin America and Caribbean|One Major City, Over 250,000 Population',
    '8RQN' => 'Latin America and Caribbean|One Metropolitan Area, Adjoining Cities',
    '8RRG' => 'Latin America and Caribbean|One State Or Province',
    '8RYH' => 'Latin America and Caribbean|Up To 3 States Or Provinces',
    '8RYI' => 'Latin America and Caribbean|Up To 5 States Or Provinces',
    '8RAC' => 'Latin America and Caribbean|All Latin America and Caribbean',
    '8RUJ' => 'Latin America and Caribbean|All Latin America',
    '8RFZ' => 'Latin America and Caribbean|All Caribbean',
    '8RFS' => 'Latin America and Caribbean|All South America',
    '8RGC' => 'Latin America and Caribbean|All Central America',
    '8RFT' => 'Latin America and Caribbean|All Andean Countries',
    '8RFU' => 'Latin America and Caribbean|All Southern Cone',
    '8RFV' => 'Latin America and Caribbean|All Amazonia',
    '8RAI' => 'Latin America and Caribbean|Anguilla',
    '8RAG' => 'Latin America and Caribbean|Antigua and Barbuda',
    '8RAR' => 'Latin America and Caribbean|Argentina',
    '8RAW' => 'Latin America and Caribbean|Aruba',
    '8RBS' => 'Latin America and Caribbean|Bahamas',
    '8RBB' => 'Latin America and Caribbean|Barbados',
    '8RBZ' => 'Latin America and Caribbean|Belize',
    '8RGZ' => 'Latin America and Caribbean|Bequia',
    '8RBO' => 'Latin America and Caribbean|Bolivia',
    '8RHA' => 'Latin America and Caribbean|Bonaire',
    '8RBR' => 'Latin America and Caribbean|Brazil',
    '8RHB' => 'Latin America and Caribbean|British Virgin Islands',
    '8RKY' => 'Latin America and Caribbean|Cayman Islands',
    '8RCL' => 'Latin America and Caribbean|Chile',
    '8RCO' => 'Latin America and Caribbean|Colombia',
    '8RCR' => 'Latin America and Caribbean|Costa Rica',
    '8RCU' => 'Latin America and Caribbean|Cuba',
    '8RHC' => 'Latin America and Caribbean|Curacao',
    '8RDM' => 'Latin America and Caribbean|Dominica',
    '8RDO' => 'Latin America and Caribbean|Dominican Republic',
    '8REC' => 'Latin America and Caribbean|Ecuador',
    '8RSV' => 'Latin America and Caribbean|El Salvador',
    '8RFK' => 'Latin America and Caribbean|Falkland Islands, Malvinas',
    '8RGF' => 'Latin America and Caribbean|French Guiana',
    '8RGD' => 'Latin America and Caribbean|Grenada',
    '8RGP' => 'Latin America and Caribbean|Guadeloupe',
    '8RGT' => 'Latin America and Caribbean|Guatemala',
    '8RGY' => 'Latin America and Caribbean|Guyana',
    '8RHT' => 'Latin America and Caribbean|Haiti',
    '8RHN' => 'Latin America and Caribbean|Honduras',
    '8RJM' => 'Latin America and Caribbean|Jamaica',
    '8RMQ' => 'Latin America and Caribbean|Martinique',
    '8RMX' => 'Latin America and Caribbean|Mexico',
    '8RMS' => 'Latin America and Caribbean|Montserrat',
    '8RAN' => 'Latin America and Caribbean|Netherlands Antilles',
    '8RNI' => 'Latin America and Caribbean|Nicaragua',
    '8RPA' => 'Latin America and Caribbean|Panama',
    '8RPY' => 'Latin America and Caribbean|Paraguay',
    '8RFQ' => 'Latin America and Caribbean|Patagonia',
    '8RPE' => 'Latin America and Caribbean|Peru',
    '8RPR' => 'Latin America and Caribbean|Puerto Rico',
    '8RHD' => 'Latin America and Caribbean|Saba',
    '8RHE' => 'Latin America and Caribbean|Saint Barthelemy',
    '8RHF' => 'Latin America and Caribbean|Saint Eustatius',
    '8RKN' => 'Latin America and Caribbean|Saint Kitts and Nevis',
    '8RLC' => 'Latin America and Caribbean|Saint Lucia',
    '8RHG' => 'Latin America and Caribbean|Saint Martin',
    '8RVC' => 'Latin America and Caribbean|Saint Vincent and The Grenadines',
    '8RSR' => 'Latin America and Caribbean|Suriname',
    '8RTT' => 'Latin America and Caribbean|Trinidad and Tobago',
    '8RTC' => 'Latin America and Caribbean|Turks and Caicos Islands',
    '8RHH' => 'Latin America and Caribbean|U.S. Virgin Islands',
    '8RUY' => 'Latin America and Caribbean|Uruguay',
    '8RVE' => 'Latin America and Caribbean|Venezuela',
    '8RQZ' => 'Oceania|One Minor City, Up To 250,000 Population',
    '8RQW' => 'Oceania|One Major City, Over 250,000 Population',
    '8RQY' => 'Oceania|One Metropolitan Area, Adjoining Cities',
    '8RRK' => 'Oceania|One State Or Province',
    '8RYL' => 'Oceania|Up To 3 States Or Provinces',
    '8RYM' => 'Oceania|Up To 5 States Or Provinces',
    '8RCT' => 'Oceania|All Oceania',
    '8RUR' => 'Oceania|All Australia and New Zealand',
    '8RUS' => 'Oceania|All Oceania excluding Australia and New Zealand',
    '8RAU' => 'Oceania|Australia',
    '8RAS' => 'Oceania|American Samoa',
    '8RCX' => 'Oceania|Christmas Island',
    '8RCC' => 'Oceania|Cocos, Keeling Islands',
    '8RKM' => 'Oceania|Comoros',
    '8RCK' => 'Oceania|Cook Islands',
    '8RFJ' => 'Oceania|Fiji',
    '8RPF' => 'Oceania|French Polynesia',
    '8RGU' => 'Oceania|Guam',
    '8RKI' => 'Oceania|Kiribati',
    '8RMG' => 'Oceania|Madagascar',
    '8RMH' => 'Oceania|Marshall Islands',
    '8RMU' => 'Oceania|Mauritius',
    '8RFM' => 'Oceania|Micronesia',
    '8RFF' => 'Oceania|Midway Islands',
    '8RNR' => 'Oceania|Nauru',
    '8RNC' => 'Oceania|New Caledonia',
    '8RNZ' => 'Oceania|New Zealand',
    '8RNU' => 'Oceania|Niue',
    '8RNF' => 'Oceania|Norfolk Island',
    '8RMP' => 'Oceania|Northern Mariana Islands',
    '8RPW' => 'Oceania|Palau',
    '8RPG' => 'Oceania|Papua New Guinea',
    '8RPN' => 'Oceania|Pitcairn Islands',
    '8RFH' => 'Oceania|Rapa Nui, Easter Island',
    '8RWS' => 'Oceania|Samoa',
    '8RSC' => 'Oceania|Seychelles',
    '8RSB' => 'Oceania|Solomon Islands',
    '8RLK' => 'Oceania|Sri Lanka',
    '8RFL' => 'Oceania|Tahiti',
    '8RTK' => 'Oceania|Tokelau',
    '8RTO' => 'Oceania|Tonga',
    '8RTV' => 'Oceania|Tuvalu',
    '8RVU' => 'Oceania|Vanuatu',
    '8RFP' => 'Oceania|Wallis and Futuna',
    '8RQS' => 'Middle East|One Minor City, Up To 250,000 Population',
    '8RQP' => 'Middle East|One Major City, Over 250,000 Population',
    '8RQR' => 'Middle East|One Metropolitan Area, Adjoining Cities',
    '8RRH' => 'Middle East|One State Or Province',
    '8RYJ' => 'Middle East|Up To 3 States Or Provinces',
    '8RBX' => 'Middle East|Up To 5 States Or Provinces',
    '8REX' => 'Middle East|All Middle East',
    '8REY' => 'Middle East|All Middle Eastern Gulf States',
    '8RIB' => 'Middle East|All Middle Eastern Mediterranean Countries',
    '8RAF' => 'Middle East|Afghanistan',
    '8RBH' => 'Middle East|Bahrain',
    '8RIR' => 'Middle East|Iran',
    '8RIQ' => 'Middle East|Iraq',
    '8RIL' => 'Middle East|Israel',
    '8RJO' => 'Middle East|Jordan',
    '8RKW' => 'Middle East|Kuwait',
    '8RLB' => 'Middle East|Lebanon',
    '8ROM' => 'Middle East|Oman',
    '8REV' => 'Middle East|Palestinian Authority',
    '8RQA' => 'Middle East|Qatar',
    '8RSA' => 'Middle East|Saudi Arabia',
    '8RSY' => 'Middle East|Syria',
    '8RTR' => 'Middle East|Turkey',
    '8RAE' => 'Middle East|United Arab Emirates',
    '8RYE' => 'Middle East|Yemen',
    '8RQC' => 'Africa|One Minor City, Up To 250,000 Population',
    '8RQX' => 'Africa|One Major City, Over 250,000 Population',
    '8RQB' => 'Africa|One Metropolitan Area, Adjoining Cities',
    '8RRB' => 'Africa|One State Or Province',
    '8RYB' => 'Africa|Up To 3 States Or Provinces',
    '8RYC' => 'Africa|Up To 5 States Or Provinces',
    '8RAJ' => 'Africa|All Africa',
    '8RAK' => 'Africa|All African Mediterranean Countries',
    '8RAP' => 'Africa|All Central Africa',
    '8RAQ' => 'Africa|All Eastern Africa',
    '8RBC' => 'Africa|All Southern Africa',
    '8RBL' => 'Africa|All Western Africa',
    '8RDZ' => 'Africa|Algeria',
    '8RAO' => 'Africa|Angola',
    '8RBP' => 'Africa|Ascension Island',
    '8RBJ' => 'Africa|Benin',
    '8RBW' => 'Africa|Botswana',
    '8RBF' => 'Africa|Burkina Faso',
    '8RBI' => 'Africa|Burundi',
    '8RCM' => 'Africa|Cameroon',
    '8RCV' => 'Africa|Cape Verde',
    '8RCF' => 'Africa|Central African Republic',
    '8RTD' => 'Africa|Chad',
    '8RCG' => 'Africa|Congo',
    '8RCI' => 'Africa|Cote D\'Ivoire',
    '8RDJ' => 'Africa|Djibouti',
    '8REG' => 'Africa|Egypt',
    '8RGQ' => 'Africa|Equatorial Guinea',
    '8RER' => 'Africa|Eritrea',
    '8RET' => 'Africa|Ethiopia',
    '8RGA' => 'Africa|Gabon',
    '8RGM' => 'Africa|Gambia',
    '8RGH' => 'Africa|Ghana',
    '8RGN' => 'Africa|Guinea',
    '8RGW' => 'Africa|Guinea-Bissau',
    '8RKE' => 'Africa|Kenya',
    '8RLS' => 'Africa|Lesotho',
    '8RLR' => 'Africa|Liberia',
    '8RLY' => 'Africa|Libyan Arab Jamahiriya',
    '8RMW' => 'Africa|Malawi',
    '8RML' => 'Africa|Mali',
    '8RMR' => 'Africa|Mauritania',
    '8RYT' => 'Africa|Mayotte',
    '8RMA' => 'Africa|Morocco',
    '8RMZ' => 'Africa|Mozambique',
    '8RNA' => 'Africa|Namibia',
    '8RNE' => 'Africa|Niger',
    '8RNG' => 'Africa|Nigeria',
    '8RRE' => 'Africa|Reunion',
    '8RRW' => 'Africa|Rwanda',
    '8RSH' => 'Africa|Saint Helena',
    '8RST' => 'Africa|Sao Tome and Principe',
    '8RSN' => 'Africa|Senegal',
    '8RSL' => 'Africa|Sierra Leone',
    '8RSO' => 'Africa|Somalia',
    '8RZA' => 'Africa|South Africa',
    '8RSD' => 'Africa|Sudan',
    '8RSZ' => 'Africa|Swaziland',
    '8RTZ' => 'Africa|Tanzania, United Republic Of',
    '8RTG' => 'Africa|Togo',
    '8RTN' => 'Africa|Tunisia',
    '8RUG' => 'Africa|Uganda',
    '8REH' => 'Africa|Western Sahara',
    '8RZM' => 'Africa|Zambia',
    '8RZW' => 'Africa|Zimbabwe',
    '8RBQ' => 'Other Regions|Antarctica',
    '8RCB' => 'Other Regions|All Arctic and Arctic Ocean Islands',
    '8RFB' => 'Other Regions|All Northern Atlantic Ocean Islands',
    '8RFW' => 'Other Regions|All Southern Atlantic Ocean Islands',
    '8RFX' => 'Other Regions|All Southern Indian Ocean Islands',
    '8REU' => 'Other Regions|All French Southern Territories',
    '8RDQ' => 'Other Regions|All British Indian Ocean Territories',
   # 8L - Language
    '8LAA' => 'All Languages',
    '8LXX' => 'Not Applicable or None',
    '8LUL' => 'Any Languages',
    '8LOL' => 'Any One Language',
    '8LEN' => 'English',
    '8LAF' => 'Afrikaans',
    '8LAR' => 'Arabic',
    '8LBO' => 'Bosnian',
    '8LBU' => 'Bulgarian',
    '8LCA' => 'Chinese-Cantonese',
    '8LCH' => 'Chinese-Mandarin',
    '8LCP' => 'Chinese-Other',
    '8LCR' => 'Croatian',
    '8LCZ' => 'Czech',
    '8LDA' => 'Danish',
    '8LDU' => 'Dutch',
    '8LES' => 'Estonian',
    '8LFI' => 'Finnish',
    '8LFR' => 'French',
    '8LGE' => 'German',
    '8LGR' => 'Greek',
    '8LHE' => 'Hebrew',
    '8LHI' => 'Hindi',
    '8LHU' => 'Hungarian',
    '8LIC' => 'Icelandic',
    '8LIN' => 'Indonesian',
    '8LIG' => 'Irish Gaelic',
    '8LIT' => 'Italian',
    '8LJA' => 'Japanese',
    '8LKO' => 'Korean',
    '8LLA' => 'Latvian',
    '8LMG' => 'Mongolian',
    '8LNO' => 'Norwegian',
    '8LPO' => 'Polish',
    '8LPR' => 'Portuguese',
    '8LRO' => 'Romanian',
    '8LRU' => 'Russian',
    '8LSG' => 'Scottish Gaelic',
    '8LSE' => 'Serbian',
    '8LSI' => 'Sindhi',
    '8LSV' => 'Slovakian',
    '8LSL' => 'Slovenian',
    '8LSP' => 'Spanish',
    '8LSH' => 'Swahili',
    '8LSZ' => 'Swazi',
    '8LSW' => 'Swedish',
    '8LTA' => 'Tagalog',
    '8LTH' => 'Thai',
    '8LTU' => 'Turkish',
    '8LUR' => 'Ukrainian',
    '8LYI' => 'Yiddish',
    '8LOT' => 'Other Language',
   # 8I - Industry
    '8IAA' => 'All Industries',
    '8IXX' => 'Not Applicable or None',
    '8IUL' => 'Any Industries',
    '8IAD' => 'Advertising and Marketing',
    '8IAG' => 'Agriculture, Farming and Horticulture',
    '8IAT' => 'Airline Transportation',
    '8IAL' => 'Alcohol',
    '8IAR' => 'Architecture and Engineering',
    '8IAE' => 'Arts and Entertainment',
    '8IAU' => 'Automotive',
    '8IAV' => 'Aviation',
    '8IBA' => 'Baby and Childcare',
    '8IBE' => 'Beauty and Personal Care',
    '8IBI' => 'Biotechnology',
    '8IBR' => 'Broadcast Media',
    '8ICO' => 'Business Consulting and Services',
    '8ICH' => 'Chemicals',
    '8ICE' => 'Communications Equipment and Services',
    '8IHS' => 'Computer Hardware, Software and Peripherals',
    '8ICC' => 'Construction and Contracting',
    '8IAP' => 'Consumer Appliances and Electronics',
    '8ICG' => 'Counseling',
    '8IEC' => 'Ecology, Environmental and Conservation',
    '8IED' => 'Education',
    '8IEM' => 'Employment Training and Recruitment',
    '8IEN' => 'Energy, Utilities and Fuel',
    '8IEV' => 'Events and Conventions',
    '8IFA' => 'Fashion',
    '8IFI' => 'Financial Services and Banking',
    '8IFB' => 'Food and Beverage Processing',
    '8IFL' => 'Food and Beverage Retail',
    '8IFS' => 'Food Services',
    '8IFO' => 'Forestry and Wood Products',
    '8IFR' => 'Freight and Warehousing',
    '8IFU' => 'Furniture',
    '8IGA' => 'Games, Toys and Hobbies',
    '8IGI' => 'Gaming Industry',
    '8IGL' => 'Gardening and Landscaping',
    '8IGO' => 'Government and Politics',
    '8IGR' => 'Graphic Design',
    '8IGC' => 'Greeting Card',
    '8IHI' => 'Heavy Industry',
    '8IHO' => 'Home Improvement',
    '8IHH' => 'Hotels and Hospitality',
    '8IHA' => 'Household Appliances',
    '8IHC' => 'Household Cleaning Products',
    '8IIM' => 'Industry and Manufacturing',
    '8IIT' => 'Information Technologies',
    '8IIN' => 'Insurance',
    '8IIS' => 'Internet Services',
    '8ILS' => 'Legal Services',
    '8IME' => 'Medical and Healthcare',
    '8IMS' => 'Microelectronics and Semiconductors',
    '8IMW' => 'Military and Weapons',
    '8IMM' => 'Mining and Metals',
    '8IMU' => 'Music',
    '8INP' => 'Not For Profit, Social, Charitable',
    '8IOP' => 'Office Products',
    '8IOG' => 'Oil and Gas',
    '8IOI' => 'Other Industry',
    '8IPO' => 'Personal Use Only',
    '8IPP' => 'Pet Products and Services',
    '8IPS' => 'Pharmaceuticals and Supplements',
    '8IPT' => 'Printing and Reprographics',
    '8IPR' => 'Public Relations',
    '8IPM' => 'Publishing Media',
    '8IRE' => 'Real Estate',
    '8IRR' => 'Religion and Religious Services',
    '8ISM' => 'Retail Sales and Marketing',
    '8IRM' => 'Retail Merchandise',
    '8ISS' => 'Safety and Security',
    '8ISC' => 'Sciences',
    '8ISH' => 'Shipping',
    '8ISO' => 'Software',
    '8ISF' => 'Sports, Fitness and Recreation',
    '8ITE' => 'Telecommunications',
    '8ITX' => 'Textiles and Apparel',
    '8ITB' => 'Tobacco',
    '8ITR' => 'Travel and Tourism',
   # 9E - Exclusivity
    '9EXX' => 'Not Applicable or None',
    '9ENE' => 'Non-Exclusive',
    '9EXC' => 'All Exclusive',
    '9EIN' => 'Exclusivity For Industry',
    '9EME' => 'Exclusivity For Media',
    '9ELA' => 'Exclusivity For Language',
    '9ERE' => 'Exclusivity For Region'
);

# PLUS License Data Format 1.2.0 (plus) (ref 1)
%Image::ExifTool::PLUS::XMP = (
    %Image::ExifTool::XMP::xmpTableDefaults,
    GROUPS => { 0 => 'XMP', 1 => 'XMP-plus', 2 => 'Author' },
    NAMESPACE => 'plus',
    NOTES => q{
        PLUS (Picture Licensing Universal System) License Data Format 1.2.1 XMP
        tags.  Note that all controlled-vocabulary tags in this table (ie. tags with
        a fixed set of values) have raw values which begin with
        "http://ns.useplus.org/ldf/vocab/", but to reduce clutter this prefix has
        been removed from the values shown below.  See L<http://ns.useplus.org/> for
        the complete specification.
    },
    Version  => { Name => 'PLUSVersion' },
    Licensee => {
        FlatName => '',
        Struct => \%plusLicensee,
        List => 'Seq',
    },
    EndUser => {
        FlatName => '',
        Struct => \%plusEndUser,
        List => 'Seq',
    },
    Licensor => {
        FlatName => '',
        Struct => \%plusLicensor,
        List => 'Seq',
    },
    LicensorNotes               => { Writable => 'lang-alt' },
    MediaSummaryCode => {
        SeparateTable => 'MediaMatrix',
        PrintConv => \%mediaMatrix,
    },
    LicenseStartDate            => { %Image::ExifTool::XMP::dateTimeInfo, Groups => { 2 => 'Time'} },
    LicenseEndDate              => { %Image::ExifTool::XMP::dateTimeInfo, Groups => { 2 => 'Time'} },
    MediaConstraints            => { Writable => 'lang-alt' },
    RegionConstraints           => { Writable => 'lang-alt' },
    ProductOrServiceConstraints => { Writable => 'lang-alt' },
    ImageFileConstraints => {
        List => 'Bag',
        %plusVocab,
        PrintConv => {
            'IF-MFN' => 'Maintain File Name',
            'IF-MID' => 'Maintain ID in File Name',
            'IF-MMD' => 'Maintain Metadata',
            'IF-MFT' => 'Maintain File Type',
        },
    },
    ImageAlterationConstraints => {
        List => 'Bag',
        %plusVocab,
        PrintConv => {
            'AL-CRP' => 'No Cropping',
            'AL-FLP' => 'No Flipping',
            'AL-RET' => 'No Retouching',
            'AL-CLR' => 'No Colorization',
            'AL-DCL' => 'No De-Colorization',
            'AL-MRG' => 'No Merging',
        },
    },
    ImageDuplicationConstraints => {
        %plusVocab,
        PrintConv => {
            'DP-NDC' => 'No Duplication Constraints',
            'DP-LIC' => 'Duplication Only as Necessary Under License',
            'DP-NOD' => 'No Duplication',
        },
    },
    ModelReleaseStatus => {
        %plusVocab,
        PrintConv => {
            'MR-NON' => 'None',
            'MR-NAP' => 'Not Applicable',
            'MR-UMR' => 'Unlimited Model Releases',
            'MR-LMR' => 'Limited or Incomplete Model Releases',
        },
    },
    ModelReleaseID      => { List => 'Bag' },
    MinorModelAgeDisclosure => {
        %plusVocab,
        PrintConv => {
            'AG-UNK' => 'Age Unknown',
            'AG-A25' => 'Age 25 or Over',
            'AG-A24' => 'Age 24',
            'AG-A23' => 'Age 23',
            'AG-A22' => 'Age 22',
            'AG-A21' => 'Age 21',
            'AG-A20' => 'Age 20',
            'AG-A19' => 'Age 19',
            'AG-A18' => 'Age 18',
            'AG-A17' => 'Age 17',
            'AG-A16' => 'Age 16',
            'AG-A15' => 'Age 15',
            'AG-U14' => 'Age 14 or Under',
        },
    },
    PropertyReleaseStatus => {
        %plusVocab,
        PrintConv => {
            'PR-NON' => 'None',
            'PR-NAP' => 'Not Applicable',
            'PR-UPR' => 'Unlimited Property Releases',
            'PR-LPR' => 'Limited or Incomplete Property Releases',
        },
    },
    PropertyReleaseID  => { List => 'Bag' },
    OtherConstraints   => { Writable => 'lang-alt' },
    CreditLineRequired => {
        %plusVocab,
        PrintConv => {
            'CR-NRQ' => 'Not Required',
            'CR-COI' => 'Credit on Image',
            'CR-CAI' => 'Credit Adjacent To Image',
            'CR-CCA' => 'Credit in Credits Area',
        },
    },
    AdultContentWarning => {
        %plusVocab,
        PrintConv => {
            'CW-NRQ' => 'Not Required',
            'CW-AWR' => 'Adult Content Warning Required',
            'CW-UNK' => 'Unknown',
        },
    },
    OtherLicenseRequirements    => { Writable => 'lang-alt' },
    TermsAndConditionsText      => { Writable => 'lang-alt' },
    TermsAndConditionsURL       => { },
    OtherConditions             => { Writable => 'lang-alt' },
    ImageType => {
        %plusVocab,
        PrintConv => {
            'TY-PHO' => 'Photographic Image',
            'TY-ILL' => 'Illustrated Image',
            'TY-MCI' => 'Multimedia or Composited Image',
            'TY-VID' => 'Video',
            'TY-OTR' => 'Other',
        },
    },
    LicensorImageID     => { },
    FileNameAsDelivered => { },
    ImageFileFormatAsDelivered => {
        %plusVocab,
        PrintConv => {
            'FF-JPG' => 'JPEG Interchange Formats (JPG, JIF, JFIF)',
            'FF-TIF' => 'Tagged Image File Format (TIFF)',
            'FF-GIF' => 'Graphics Interchange Format (GIF)',
            'FF-RAW' => 'Proprietary RAW Image Format',
            'FF-DNG' => 'Digital Negative (DNG)',
            'FF-EPS' => 'Encapsulated PostScript (EPS)',
            'FF-BMP' => 'Windows Bitmap (BMP)',
            'FF-PSD' => 'Photoshop Document (PSD)',
            'FF-PIC' => 'Macintosh Picture (PICT)',
            'FF-PNG' => 'Portable Network Graphics (PNG)',
            'FF-WMP' => 'Windows Media Photo (HD Photo)',
            'FF-OTR' => 'Other',
        },
    },
    ImageFileSizeAsDelivered => {
        %plusVocab,
        PrintConv => {
            'SZ-U01' => 'Up to 1 MB',
            'SZ-U10' => 'Up to 10 MB',
            'SZ-U30' => 'Up to 30 MB',
            'SZ-U50' => 'Up to 50 MB',
            'SZ-G50' => 'Greater than 50 MB',
        },
    },
    CopyrightStatus => {
        %plusVocab,
        PrintConv => {
            'CS-PRO' => 'Protected',
            'CS-PUB' => 'Public Domain',
            'CS-UNK' => 'Unknown',
        },
    },
    CopyrightRegistrationNumber => { },
    FirstPublicationDate        => { %Image::ExifTool::XMP::dateTimeInfo, Groups => { 2 => 'Time'} },
    CopyrightOwner => {
        FlatName => '',
        Struct => \%plusCopyrightOwner,
        List => 'Seq',
    },
    CopyrightOwnerImageID   => { },
    ImageCreator => {
        FlatName => '',
        Struct => \%plusImageCreator,
        List => 'Seq',
    },
    ImageCreatorImageID     => { },
    ImageSupplier => {
        FlatName => '',
        Struct => \%plusImageSupplier,
        List => 'Seq',
    },
    ImageSupplierImageID    => { },
    LicenseeImageID         => { },
    LicenseeImageNotes      => { Writable => 'lang-alt' },
    OtherImageInfo          => { Writable => 'lang-alt' },
    LicenseID               => { },
    LicensorTransactionID   => { List => 'Bag' },
    LicenseeTransactionID   => { List => 'Bag' },
    LicenseeProjectReference=> { List => 'Bag' },
    LicenseTransactionDate  => { %Image::ExifTool::XMP::dateTimeInfo, Groups => { 2 => 'Time'} },
    Reuse => {
        %plusVocab,
        PrintConv => {
            'RE-REU' => 'Repeat Use',
            'RE-NAP' => 'Not Applicable',
        },
    },
    OtherLicenseDocuments   => { List => 'Bag' },
    OtherLicenseInfo        => { Writable => 'lang-alt' },
    # Note: these are Bag's of lang-alt lists -- a nested list tag!
    Custom1     => { List => 'Bag', Writable => 'lang-alt' },
    Custom2     => { List => 'Bag', Writable => 'lang-alt' },
    Custom3     => { List => 'Bag', Writable => 'lang-alt' },
    Custom4     => { List => 'Bag', Writable => 'lang-alt' },
    Custom5     => { List => 'Bag', Writable => 'lang-alt' },
    Custom6     => { List => 'Bag', Writable => 'lang-alt' },
    Custom7     => { List => 'Bag', Writable => 'lang-alt' },
    Custom8     => { List => 'Bag', Writable => 'lang-alt' },
    Custom9     => { List => 'Bag', Writable => 'lang-alt' },
    Custom10    => { List => 'Bag', Writable => 'lang-alt' },
);

#------------------------------------------------------------------------------
# Validate Media Summary Code
# Inputs: 0) Media Usage Code
# Returns: true if OK, false on severe error
# - issues warning for detected format problems
# - repairs some repairable problems
sub ValidateMediaSummary($)
{
    my $val = shift;

    my @a = split /\|/, $val;
    @a >= 4 and $a[0] eq '' or warn("Not a valid Media Summary Code\n"), return 0;
    $a[1] eq 'PLUS' or warn("Unrecognized Media Usage standard\n"), return 0;
    $a[2] =~ /^V(\d+)/ or warn("Unrecognized Media Usage version\n");
    $a[3] =~ /^U(\d+)/ or warn("Invalid Media Usage count\n"), return 0;
    my $numUsages = $1;
    my ($i, $j);
    unless ($numUsages == @a - 4) {
        warn("Fixed incorrect number of Media Usages\n");
        $numUsages = @a - 4;
        $a[3] = sprintf('U%.3d', $numUsages);
    }
    for ($i=1; $i<=$numUsages; ++$i) {
        my $usage = $a[$i + 3];
        $usage =~ /^1I([A-Z])([A-Z])/ or warn("Missing Media Usage $i item count\n"), return 0;
        length($usage) % 4 and warn("Incorrect Media Usage $i length\n"), return 0;
        my $numItems = (ord($1)-65) * 26 + ord($2)-65 + 1;
        unless (length($usage) == 4 * ($numItems + 1)) {
            $numItems = length($usage) / 4 - 1;
            warn("Fixed incorrect Media Usage $i item count\n");
            $a[$i+3] = '1I' . chr(65 + int($numItems / 26)) . chr($numItems % 26) . substr($usage, 4);
        }
        for ($j=1; $j<=$numItems; ++$j) {
            my $item = substr($usage, $j*4, 4);
            $item =~ /^\d[A-Z]{3}$/ or warn(qq(Invalid item "$item" for Media Usage $i\n)), return 0;
        }
    }
    $_[0] = join('|', @a) . '|' if $Image::ExifTool::evalWarning;
    return 1;
}

1;  # end

__END__

=head1 NAME

Image::ExifTool::PLUS - PLUS (Picture Licensing Universal System) tags

=head1 DESCRIPTION

Definitions for PLUS (Picture Licensing Universal System) tags.

=head1 AUTHOR

Copyright 2003-2022, Phil Harvey (philharvey66 at gmail.com)

This library is free software; you can redistribute it and/or modify it
under the same terms as Perl itself.

=head1 REFERENCES

=over 4

=item L<http://www.useplus.com/useplus/standards.asp>

=back

=head1 SEE ALSO

L<Image::ExifTool::TagNames/PLUS Tags>,
L<Image::ExifTool(3pm)|Image::ExifTool>

=cut
