"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.BasicAuthRoutes = void 0;

var _configSchema = require("@osd/config-schema");

var _security_cookie = require("../../../session/security_cookie");

var _common = require("../../../../common");

var _tenant_resolver = require("../../../multitenancy/tenant_resolver");

var _encode_uri_query = require("../../../../../../src/plugins/opensearch_dashboards_utils/common/url/encode_uri_query");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
class BasicAuthRoutes {
  constructor(router, config, sessionStorageFactory, securityClient, coreSetup) {
    this.router = router;
    this.config = config;
    this.sessionStorageFactory = sessionStorageFactory;
    this.securityClient = securityClient;
    this.coreSetup = coreSetup;
  }

  setupRoutes() {
    // bootstrap an empty page so that browser app can render the login page
    // using client side routing.
    this.coreSetup.http.resources.register({
      path: _common.LOGIN_PAGE_URI,
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      const clearOldVersionCookie = (0, _security_cookie.clearOldVersionCookieValue)(this.config);
      return response.renderAnonymousCoreApp({
        headers: {
          'set-cookie': clearOldVersionCookie
        }
      });
    }); // login using username and password

    this.router.post({
      path: _common.API_AUTH_LOGIN,
      validate: {
        body: _configSchema.schema.object({
          username: _configSchema.schema.string(),
          password: _configSchema.schema.string()
        })
      },
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      var _this$config$multiten, _this$config$multiten2;

      const forbiddenUsernames = this.config.auth.forbidden_usernames;

      if (forbiddenUsernames.indexOf(request.body.username) > -1) {
        context.security_plugin.logger.error(`Denied login for forbidden username ${request.body.username}`);
        return response.badRequest({
          // Cannot login using forbidden user name.
          body: 'Invalid username or password'
        });
      }

      let user;

      try {
        user = await this.securityClient.authenticate(request, {
          username: request.body.username,
          password: request.body.password
        });
      } catch (error) {
        context.security_plugin.logger.error(`Failed authentication: ${error}`);
        return response.unauthorized({
          headers: {
            'www-authenticate': error.message
          }
        });
      }

      this.sessionStorageFactory.asScoped(request).clear();
      const encodedCredentials = Buffer.from(`${request.body.username}:${request.body.password}`).toString('base64');
      const sessionStorage = {
        username: user.username,
        credentials: {
          authHeaderValue: `Basic ${encodedCredentials}`
        },
        authType: 'basicauth',
        isAnonymousAuth: false,
        expiryTime: Date.now() + this.config.session.ttl
      };

      if ((_this$config$multiten = this.config.multitenancy) !== null && _this$config$multiten !== void 0 && _this$config$multiten.enabled) {
        const selectTenant = (0, _tenant_resolver.resolveTenant)(request, user.username, user.roles, user.tenants, this.config, sessionStorage);
        sessionStorage.tenant = selectTenant;
      }

      this.sessionStorageFactory.asScoped(request).set(sessionStorage);
      return response.ok({
        body: {
          username: user.username,
          tenants: user.tenants,
          roles: user.roles,
          backendroles: user.backendRoles,
          selectedTenants: (_this$config$multiten2 = this.config.multitenancy) !== null && _this$config$multiten2 !== void 0 && _this$config$multiten2.enabled ? sessionStorage.tenant : undefined
        }
      });
    }); // logout

    this.router.post({
      path: _common.API_AUTH_LOGOUT,
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      this.sessionStorageFactory.asScoped(request).clear();
      return response.ok({
        body: {}
      });
    }); // anonymous auth

    this.router.get({
      path: `/auth/anonymous`,
      validate: false,
      options: {
        authRequired: false
      }
    }, async (context, request, response) => {
      if (this.config.auth.anonymous_auth_enabled) {
        var _this$config$multiten3;

        let user; // If the request contains no redirect path, simply redirect to basepath.

        let redirectUrl = this.coreSetup.http.basePath.serverBasePath ? this.coreSetup.http.basePath.serverBasePath : '/';
        const requestQuery = request.url.searchParams;
        const nextUrl = requestQuery === null || requestQuery === void 0 ? void 0 : requestQuery.get('nextUrl');

        if (nextUrl) {
          redirectUrl = nextUrl;
        }

        context.security_plugin.logger.info('The Redirect Path is ' + redirectUrl);

        try {
          user = await this.securityClient.authenticateWithHeaders(request, {});
        } catch (error) {
          context.security_plugin.logger.error(`Failed authentication: ${error}. Redirecting to Login Page`);
          return response.redirected({
            headers: {
              location: `${this.coreSetup.http.basePath.serverBasePath}${_common.LOGIN_PAGE_URI}${nextUrl ? '?nextUrl=' + (0, _encode_uri_query.encodeUriQuery)(redirectUrl) : ''}`
            }
          });
        }

        this.sessionStorageFactory.asScoped(request).clear();
        const sessionStorage = {
          username: user.username,
          authType: 'basicauth',
          isAnonymousAuth: true,
          expiryTime: Date.now() + this.config.session.ttl
        };

        if ((_this$config$multiten3 = this.config.multitenancy) !== null && _this$config$multiten3 !== void 0 && _this$config$multiten3.enabled) {
          const selectTenant = (0, _tenant_resolver.resolveTenant)(request, user.username, user.roles, user.tenants, this.config, sessionStorage);
          sessionStorage.tenant = selectTenant;
        }

        this.sessionStorageFactory.asScoped(request).set(sessionStorage);
        return response.redirected({
          headers: {
            location: `${redirectUrl}`
          }
        });
      } else {
        context.security_plugin.logger.error('Anonymous auth is disabled. Redirecting to Login Page');
        return response.redirected({
          headers: {
            location: `${this.coreSetup.http.basePath.serverBasePath}${_common.LOGIN_PAGE_URI}`
          }
        });
      }
    });
  }

}

exports.BasicAuthRoutes = BasicAuthRoutes;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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