"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.default = void 0;
exports.registerAlertingRoutes = registerAlertingRoutes;

var _lodash = require("lodash");

var _constants = require("../utils/constants");

var _adHelpers = require("./utils/adHelpers");

function _defineProperty(obj, key, value) { if (key in obj) { Object.defineProperty(obj, key, { value: value, enumerable: true, configurable: true, writable: true }); } else { obj[key] = value; } return obj; }

function registerAlertingRoutes(apiRouter, alertingService) {
  apiRouter.post('/monitors/_search', alertingService.searchMonitors);
  apiRouter.get('/monitors/alerts', alertingService.searchAlerts);
}

class AlertingService {
  constructor(client) {
    _defineProperty(this, "client", void 0);

    _defineProperty(this, "searchMonitors", async (context, request, opensearchDashboardsResponse) => {
      try {
        const requestBody = {
          size: _constants.MAX_MONITORS,
          query: {
            nested: {
              path: 'monitor.inputs',
              query: {
                bool: {
                  should: [{
                    term: {
                      'monitor.inputs.search.indices.keyword': {
                        value: '.opendistro-anomaly-results*'
                      }
                    }
                  }, {
                    match_phrase_prefix: {
                      'monitor.inputs.search.indices': 'opensearch-ad-plugin-result-'
                    }
                  }]
                }
              }
            }
          }
        };
        const response = await this.client.asScoped(request).callAsCurrentUser('alerting.searchMonitors', {
          body: requestBody
        });
        const totalMonitors = (0, _lodash.get)(response, 'hits.total.value', 0);
        const allMonitors = (0, _lodash.get)(response, 'hits.hits', []).reduce((acc, monitor) => ({ ...acc,
          [monitor._id]: {
            id: monitor._id,
            name: (0, _lodash.get)(monitor, '_source.name'),
            enabled: (0, _lodash.get)(monitor, '_source.enabled', false),
            enabledTime: (0, _lodash.get)(monitor, '_source.enabled_time'),
            schedule: (0, _lodash.get)(monitor, '_source.schedule'),
            inputs: (0, _lodash.get)(monitor, '_source.inputs'),
            triggers: (0, _lodash.get)(monitor, '_source.triggers'),
            lastUpdateTime: (0, _lodash.get)(monitor, '_source.last_update_time')
          }
        }), {});
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response: {
              totalMonitors,
              monitors: Object.values(allMonitors)
            }
          }
        });
      } catch (err) {
        console.log('Unable to get monitor on top of detector', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    _defineProperty(this, "searchAlerts", async (context, request, opensearchDashboardsResponse) => {
      try {
        const {
          monitorId,
          startTime,
          endTime
        } = request.query;
        const response = await this.client.asScoped(request).callAsCurrentUser('alerting.searchAlerts', {
          monitorId: monitorId,
          startTime: startTime,
          endTime: endTime
        });
        return opensearchDashboardsResponse.ok({
          body: {
            ok: true,
            response
          }
        });
      } catch (err) {
        console.log('Unable to search alerts', err);
        return opensearchDashboardsResponse.ok({
          body: {
            ok: false,
            error: (0, _adHelpers.getErrorMessage)(err)
          }
        });
      }
    });

    this.client = client;
  }

}

exports.default = AlertingService;
//# sourceMappingURL=data:application/json;charset=utf-8;base64,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