"use strict";
/*
 * Licensed to Elasticsearch B.V. under one or more contributor
 * license agreements. See the NOTICE file distributed with
 * this work for additional information regarding copyright
 * ownership. Elasticsearch B.V. licenses this file to you under
 * the Apache License, Version 2.0 (the "License"); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *    http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */
Object.defineProperty(exports, "__esModule", { value: true });
exports.ApmConfiguration = void 0;
const path_1 = require("path");
const lodash_1 = require("lodash");
const child_process_1 = require("child_process");
// deep import to avoid loading the whole package
const path_2 = require("@kbn/utils/target/path");
const fs_1 = require("fs");
const getDefaultConfig = (isDistributable) => {
    if (isDistributable) {
        return {
            active: false,
            globalLabels: {},
            // Do not use a centralized controlled config
            centralConfig: false,
            // Capture all exceptions that are not caught
            logUncaughtExceptions: true,
            // Can be performance intensive, disabling by default
            breakdownMetrics: false,
        };
    }
    return {
        active: false,
        serverUrl: 'https://f1542b814f674090afd914960583265f.apm.us-central1.gcp.cloud.es.io:443',
        // The secretToken below is intended to be hardcoded in this file even though
        // it makes it public. This is not a security/privacy issue. Normally we'd
        // instead disable the need for a secretToken in the APM Server config where
        // the data is transmitted to, but due to how it's being hosted, it's easier,
        // for now, to simply leave it in.
        secretToken: 'R0Gjg46pE9K9wGestd',
        globalLabels: {},
        breakdownMetrics: true,
        centralConfig: false,
        logUncaughtExceptions: true,
    };
};
class ApmConfiguration {
    constructor(rootDir, rawKibanaConfig, isDistributable) {
        this.rootDir = rootDir;
        this.rawKibanaConfig = rawKibanaConfig;
        this.isDistributable = isDistributable;
        // eslint-disable-next-line @typescript-eslint/no-var-requires
        const { version, build } = require(path_1.join(this.rootDir, 'package.json'));
        this.kibanaVersion = version;
        this.pkgBuild = build;
    }
    getConfig(serviceName) {
        return {
            ...this.getBaseConfig(),
            serviceName,
        };
    }
    getBaseConfig() {
        if (!this.baseConfig) {
            const apmConfig = lodash_1.merge(getDefaultConfig(this.isDistributable), this.getConfigFromKibanaConfig(), this.getDevConfig(), this.getDistConfig());
            const rev = this.getGitRev();
            if (rev !== null) {
                apmConfig.globalLabels.git_rev = rev;
            }
            const uuid = this.getKibanaUuid();
            if (uuid) {
                apmConfig.globalLabels.kibana_uuid = uuid;
            }
            apmConfig.serviceVersion = this.kibanaVersion;
            this.baseConfig = apmConfig;
        }
        return this.baseConfig;
    }
    getConfigFromKibanaConfig() {
        return lodash_1.get(this.rawKibanaConfig, 'elastic.apm', {});
    }
    getKibanaUuid() {
        // try to access the `server.uuid` value from the config file first.
        // if not manually defined, we will then read the value from the `{DATA_FOLDER}/uuid` file.
        // note that as the file is created by the platform AFTER apm init, the file
        // will not be present at first startup, but there is nothing we can really do about that.
        if (lodash_1.get(this.rawKibanaConfig, 'server.uuid')) {
            return this.rawKibanaConfig.server.uuid;
        }
        const dataPath = lodash_1.get(this.rawKibanaConfig, 'path.data') || path_2.getDataPath();
        try {
            const filename = path_1.join(dataPath, 'uuid');
            return fs_1.readFileSync(filename, 'utf-8');
        }
        catch (e) { } // eslint-disable-line no-empty
    }
    getDevConfig() {
        try {
            const apmDevConfigPath = path_1.join(this.rootDir, 'config', 'apm.dev.js');
            return require(apmDevConfigPath);
        }
        catch (e) {
            return {};
        }
    }
    /** Config keys that cannot be overridden in production builds */
    getDistConfig() {
        if (!this.isDistributable) {
            return {};
        }
        return {
            // Headers & body may contain sensitive info
            captureHeaders: false,
            captureBody: 'off',
        };
    }
    getGitRev() {
        if (this.isDistributable) {
            return this.pkgBuild.sha;
        }
        try {
            return child_process_1.execSync('git rev-parse --short HEAD', {
                encoding: 'utf-8',
                stdio: ['ignore', 'pipe', 'ignore'],
            }).trim();
        }
        catch (e) {
            return null;
        }
    }
}
exports.ApmConfiguration = ApmConfiguration;
