"use strict";

Object.defineProperty(exports, "__esModule", {
  value: true
});
exports.PRIVATE_TENANTS = exports.GLOBAL_TENANTS = void 0;
exports.isMultitenantPath = isMultitenantPath;
exports.isValidTenant = isValidTenant;
exports.resolveTenant = resolveTenant;

var _lodash = require("lodash");

/*
 *   Copyright OpenSearch Contributors
 *
 *   Licensed under the Apache License, Version 2.0 (the "License").
 *   You may not use this file except in compliance with the License.
 *   A copy of the License is located at
 *
 *       http://www.apache.org/licenses/LICENSE-2.0
 *
 *   or in the "license" file accompanying this file. This file is distributed
 *   on an "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either
 *   express or implied. See the License for the specific language governing
 *   permissions and limitations under the License.
 */
const PRIVATE_TENANT_SYMBOL = '__user__';
const GLOBAL_TENANT_SYMBOL = '';
const PRIVATE_TENANTS = [PRIVATE_TENANT_SYMBOL, 'private'];
exports.PRIVATE_TENANTS = PRIVATE_TENANTS;
const GLOBAL_TENANTS = ['global', GLOBAL_TENANT_SYMBOL];
/**
 * Resovles the tenant the user is using.
 *
 * @param request OpenSearchDashboards request.
 * @param config security plugin config.
 * @param cookie cookie extracted from the request. The cookie should have been parsed by AuthenticationHandler.
 * pass it as parameter instead of extracting again.
 * @param authInfo authentication info, the Elasticsearch authinfo API response.
 *
 * @returns user preferred tenant of the request.
 */

exports.GLOBAL_TENANTS = GLOBAL_TENANTS;

function resolveTenant(request, username, roles, availabeTenants, config, cookie) {
  var _request$url, _request$url$searchPa, _request$url2, _request$url2$searchP, _request$url3, _request$url3$searchP, _config$multitenancy, _config$multitenancy2, _config$multitenancy3;

  const DEFAULT_READONLY_ROLES = ['kibana_read_only'];
  let selectedTenant;
  const securityTenant_ = request === null || request === void 0 ? void 0 : (_request$url = request.url) === null || _request$url === void 0 ? void 0 : (_request$url$searchPa = _request$url.searchParams) === null || _request$url$searchPa === void 0 ? void 0 : _request$url$searchPa.get('securityTenant_');
  const securitytenant = request === null || request === void 0 ? void 0 : (_request$url2 = request.url) === null || _request$url2 === void 0 ? void 0 : (_request$url2$searchP = _request$url2.searchParams) === null || _request$url2$searchP === void 0 ? void 0 : _request$url2$searchP.get('securitytenant'); // eslint-disable-next-line @typescript-eslint/naming-convention

  const security_tenant = request === null || request === void 0 ? void 0 : (_request$url3 = request.url) === null || _request$url3 === void 0 ? void 0 : (_request$url3$searchP = _request$url3.searchParams) === null || _request$url3$searchP === void 0 ? void 0 : _request$url3$searchP.get('security_tenant');

  if (securityTenant_) {
    selectedTenant = securityTenant_;
  } else if (securitytenant) {
    selectedTenant = securitytenant;
  } else if (security_tenant) {
    selectedTenant = security_tenant;
  } else if (request.headers.securitytenant || request.headers.securityTenant_) {
    selectedTenant = request.headers.securitytenant ? request.headers.securitytenant : request.headers.securityTenant_;
  } else if (isValidTenant(cookie.tenant)) {
    selectedTenant = cookie.tenant;
  } else {
    selectedTenant = undefined;
  }

  const isReadonly = roles === null || roles === void 0 ? void 0 : roles.some(role => {
    var _config$readonly_mode;

    return ((_config$readonly_mode = config.readonly_mode) === null || _config$readonly_mode === void 0 ? void 0 : _config$readonly_mode.roles.includes(role)) || DEFAULT_READONLY_ROLES.includes(role);
  });
  const preferredTenants = (_config$multitenancy = config.multitenancy) === null || _config$multitenancy === void 0 ? void 0 : _config$multitenancy.tenants.preferred;
  const globalTenantEnabled = (_config$multitenancy2 = config.multitenancy) === null || _config$multitenancy2 === void 0 ? void 0 : _config$multitenancy2.tenants.enable_global;
  const privateTenantEnabled = ((_config$multitenancy3 = config.multitenancy) === null || _config$multitenancy3 === void 0 ? void 0 : _config$multitenancy3.tenants.enable_private) && !isReadonly;
  return resolve(username, selectedTenant, preferredTenants, availabeTenants, globalTenantEnabled, privateTenantEnabled);
}
/**
 * Determines whether the request requires tenant info.
 * @param request opensearch-dashboards request.
 *
 * @returns true if the request requires tenant info, otherwise false.
 */


function isMultitenantPath(request) {
  var _request$url$pathname, _request$url$pathname2, _request$url$pathname3, _request$url$pathname4, _request$url$pathname5;

  return ((_request$url$pathname = request.url.pathname) === null || _request$url$pathname === void 0 ? void 0 : _request$url$pathname.startsWith('/opensearch')) || ((_request$url$pathname2 = request.url.pathname) === null || _request$url$pathname2 === void 0 ? void 0 : _request$url$pathname2.startsWith('/api')) || ((_request$url$pathname3 = request.url.pathname) === null || _request$url$pathname3 === void 0 ? void 0 : _request$url$pathname3.startsWith('/app')) || ( // short url path
  (_request$url$pathname4 = request.url.pathname) === null || _request$url$pathname4 === void 0 ? void 0 : _request$url$pathname4.startsWith('/goto')) || // bootstrap.js depends on tenant info to fetch opensearch-dashboards configs in tenant index
  (((_request$url$pathname5 = request.url.pathname) === null || _request$url$pathname5 === void 0 ? void 0 : _request$url$pathname5.indexOf('bootstrap.js')) || -1) > -1 || request.url.pathname === '/';
}

function resolve(username, requestedTenant, preferredTenants, availableTenants, // is an object like { tenant_name_1: true, tenant_name_2: false, ... }
globalTenantEnabled, privateTenantEnabled) {
  const availableTenantsClone = (0, _lodash.cloneDeep)(availableTenants);
  delete availableTenantsClone[username];

  if (!globalTenantEnabled && !privateTenantEnabled && (0, _lodash.isEmpty)(availableTenantsClone)) {
    return undefined;
  }

  if (isValidTenant(requestedTenant)) {
    requestedTenant = requestedTenant;

    if (requestedTenant in availableTenants) {
      return requestedTenant;
    }

    if (privateTenantEnabled && username in availableTenants && PRIVATE_TENANTS.indexOf(requestedTenant) > -1) {
      return PRIVATE_TENANT_SYMBOL;
    }

    if (globalTenantEnabled && GLOBAL_TENANTS.indexOf(requestedTenant) > -1) {
      return GLOBAL_TENANT_SYMBOL;
    }
  }

  if (preferredTenants && !(0, _lodash.isEmpty)(preferredTenants)) {
    for (const element of preferredTenants) {
      const tenant = element.toLowerCase();

      if (globalTenantEnabled && GLOBAL_TENANTS.indexOf(tenant) > -1) {
        return GLOBAL_TENANT_SYMBOL;
      }

      if (privateTenantEnabled && PRIVATE_TENANTS.indexOf(tenant) > -1 && username in availableTenants) {
        return PRIVATE_TENANT_SYMBOL;
      }

      if (tenant in availableTenants) {
        return tenant;
      }
    }
  }

  if (globalTenantEnabled) {
    return GLOBAL_TENANT_SYMBOL;
  }

  if (privateTenantEnabled) {
    return PRIVATE_TENANT_SYMBOL;
  } // fall back to the first tenant in the available tenants


  return (0, _lodash.findKey)(availableTenantsClone, () => true);
}
/**
 * Return true if tenant parameter is a valid tenent.
 *
 * Note: empty string '' is valid, which means global tenant.
 *
 * @param tenant
 */


function isValidTenant(tenant) {
  return tenant !== undefined && tenant !== null;
}
//# sourceMappingURL=data:application/json;charset=utf-8;base64,eyJ2ZXJzaW9uIjozLCJzb3VyY2VzIjpbInRlbmFudF9yZXNvbHZlci50cyJdLCJuYW1lcyI6WyJQUklWQVRFX1RFTkFOVF9TWU1CT0wiLCJHTE9CQUxfVEVOQU5UX1NZTUJPTCIsIlBSSVZBVEVfVEVOQU5UUyIsIkdMT0JBTF9URU5BTlRTIiwicmVzb2x2ZVRlbmFudCIsInJlcXVlc3QiLCJ1c2VybmFtZSIsInJvbGVzIiwiYXZhaWxhYmVUZW5hbnRzIiwiY29uZmlnIiwiY29va2llIiwiREVGQVVMVF9SRUFET05MWV9ST0xFUyIsInNlbGVjdGVkVGVuYW50Iiwic2VjdXJpdHlUZW5hbnRfIiwidXJsIiwic2VhcmNoUGFyYW1zIiwiZ2V0Iiwic2VjdXJpdHl0ZW5hbnQiLCJzZWN1cml0eV90ZW5hbnQiLCJoZWFkZXJzIiwiaXNWYWxpZFRlbmFudCIsInRlbmFudCIsInVuZGVmaW5lZCIsImlzUmVhZG9ubHkiLCJzb21lIiwicm9sZSIsInJlYWRvbmx5X21vZGUiLCJpbmNsdWRlcyIsInByZWZlcnJlZFRlbmFudHMiLCJtdWx0aXRlbmFuY3kiLCJ0ZW5hbnRzIiwicHJlZmVycmVkIiwiZ2xvYmFsVGVuYW50RW5hYmxlZCIsImVuYWJsZV9nbG9iYWwiLCJwcml2YXRlVGVuYW50RW5hYmxlZCIsImVuYWJsZV9wcml2YXRlIiwicmVzb2x2ZSIsImlzTXVsdGl0ZW5hbnRQYXRoIiwicGF0aG5hbWUiLCJzdGFydHNXaXRoIiwiaW5kZXhPZiIsInJlcXVlc3RlZFRlbmFudCIsImF2YWlsYWJsZVRlbmFudHMiLCJhdmFpbGFibGVUZW5hbnRzQ2xvbmUiLCJlbGVtZW50IiwidG9Mb3dlckNhc2UiXSwibWFwcGluZ3MiOiI7Ozs7Ozs7Ozs7QUFlQTs7QUFmQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBT0EsTUFBTUEscUJBQTZCLEdBQUcsVUFBdEM7QUFDQSxNQUFNQyxvQkFBNEIsR0FBRyxFQUFyQztBQUVPLE1BQU1DLGVBQXlCLEdBQUcsQ0FBQ0YscUJBQUQsRUFBd0IsU0FBeEIsQ0FBbEM7O0FBQ0EsTUFBTUcsY0FBd0IsR0FBRyxDQUFDLFFBQUQsRUFBV0Ysb0JBQVgsQ0FBakM7QUFDUDtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOzs7O0FBQ08sU0FBU0csYUFBVCxDQUNMQyxPQURLLEVBRUxDLFFBRkssRUFHTEMsS0FISyxFQUlMQyxlQUpLLEVBS0xDLE1BTEssRUFNTEMsTUFOSyxFQU9lO0FBQUE7O0FBQ3BCLFFBQU1DLHNCQUFzQixHQUFHLENBQUMsa0JBQUQsQ0FBL0I7QUFDQSxNQUFJQyxjQUFKO0FBQ0EsUUFBTUMsZUFBZSxHQUFHUixPQUFILGFBQUdBLE9BQUgsdUNBQUdBLE9BQU8sQ0FBRVMsR0FBWiwwRUFBRyxhQUFjQyxZQUFqQiwwREFBRyxzQkFBNEJDLEdBQTVCLENBQWdDLGlCQUFoQyxDQUF4QjtBQUNBLFFBQU1DLGNBQWMsR0FBR1osT0FBSCxhQUFHQSxPQUFILHdDQUFHQSxPQUFPLENBQUVTLEdBQVosMkVBQUcsY0FBY0MsWUFBakIsMERBQUcsc0JBQTRCQyxHQUE1QixDQUFnQyxnQkFBaEMsQ0FBdkIsQ0FKb0IsQ0FLcEI7O0FBQ0EsUUFBTUUsZUFBZSxHQUFHYixPQUFILGFBQUdBLE9BQUgsd0NBQUdBLE9BQU8sQ0FBRVMsR0FBWiwyRUFBRyxjQUFjQyxZQUFqQiwwREFBRyxzQkFBNEJDLEdBQTVCLENBQWdDLGlCQUFoQyxDQUF4Qjs7QUFDQSxNQUFJSCxlQUFKLEVBQXFCO0FBQ25CRCxJQUFBQSxjQUFjLEdBQUdDLGVBQWpCO0FBQ0QsR0FGRCxNQUVPLElBQUlJLGNBQUosRUFBb0I7QUFDekJMLElBQUFBLGNBQWMsR0FBR0ssY0FBakI7QUFDRCxHQUZNLE1BRUEsSUFBSUMsZUFBSixFQUFxQjtBQUMxQk4sSUFBQUEsY0FBYyxHQUFHTSxlQUFqQjtBQUNELEdBRk0sTUFFQSxJQUFJYixPQUFPLENBQUNjLE9BQVIsQ0FBZ0JGLGNBQWhCLElBQWtDWixPQUFPLENBQUNjLE9BQVIsQ0FBZ0JOLGVBQXRELEVBQXVFO0FBQzVFRCxJQUFBQSxjQUFjLEdBQUdQLE9BQU8sQ0FBQ2MsT0FBUixDQUFnQkYsY0FBaEIsR0FDWlosT0FBTyxDQUFDYyxPQUFSLENBQWdCRixjQURKLEdBRVpaLE9BQU8sQ0FBQ2MsT0FBUixDQUFnQk4sZUFGckI7QUFHRCxHQUpNLE1BSUEsSUFBSU8sYUFBYSxDQUFDVixNQUFNLENBQUNXLE1BQVIsQ0FBakIsRUFBa0M7QUFDdkNULElBQUFBLGNBQWMsR0FBR0YsTUFBTSxDQUFDVyxNQUF4QjtBQUNELEdBRk0sTUFFQTtBQUNMVCxJQUFBQSxjQUFjLEdBQUdVLFNBQWpCO0FBQ0Q7O0FBQ0QsUUFBTUMsVUFBVSxHQUFHaEIsS0FBSCxhQUFHQSxLQUFILHVCQUFHQSxLQUFLLENBQUVpQixJQUFQLENBQ2hCQyxJQUFEO0FBQUE7O0FBQUEsV0FBVSwwQkFBQWhCLE1BQU0sQ0FBQ2lCLGFBQVAsZ0ZBQXNCbkIsS0FBdEIsQ0FBNEJvQixRQUE1QixDQUFxQ0YsSUFBckMsTUFBOENkLHNCQUFzQixDQUFDZ0IsUUFBdkIsQ0FBZ0NGLElBQWhDLENBQXhEO0FBQUEsR0FEaUIsQ0FBbkI7QUFJQSxRQUFNRyxnQkFBZ0IsMkJBQUduQixNQUFNLENBQUNvQixZQUFWLHlEQUFHLHFCQUFxQkMsT0FBckIsQ0FBNkJDLFNBQXREO0FBQ0EsUUFBTUMsbUJBQW1CLDRCQUFHdkIsTUFBTSxDQUFDb0IsWUFBViwwREFBRyxzQkFBcUJDLE9BQXJCLENBQTZCRyxhQUF6RDtBQUNBLFFBQU1DLG9CQUFvQixHQUFHLDBCQUFBekIsTUFBTSxDQUFDb0IsWUFBUCxnRkFBcUJDLE9BQXJCLENBQTZCSyxjQUE3QixLQUErQyxDQUFDWixVQUE3RTtBQUVBLFNBQU9hLE9BQU8sQ0FDWjlCLFFBRFksRUFFWk0sY0FGWSxFQUdaZ0IsZ0JBSFksRUFJWnBCLGVBSlksRUFLWndCLG1CQUxZLEVBTVpFLG9CQU5ZLENBQWQ7QUFRRDtBQUVEO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTs7O0FBQ08sU0FBU0csaUJBQVQsQ0FBMkJoQyxPQUEzQixFQUEwRTtBQUFBOztBQUMvRSxTQUNFLDBCQUFBQSxPQUFPLENBQUNTLEdBQVIsQ0FBWXdCLFFBQVosZ0ZBQXNCQyxVQUF0QixDQUFpQyxhQUFqQyxpQ0FDQWxDLE9BQU8sQ0FBQ1MsR0FBUixDQUFZd0IsUUFEWiwyREFDQSx1QkFBc0JDLFVBQXRCLENBQWlDLE1BQWpDLENBREEsZ0NBRUFsQyxPQUFPLENBQUNTLEdBQVIsQ0FBWXdCLFFBRlosMkRBRUEsdUJBQXNCQyxVQUF0QixDQUFpQyxNQUFqQyxDQUZBLE9BR0E7QUFIQSw0QkFJQWxDLE9BQU8sQ0FBQ1MsR0FBUixDQUFZd0IsUUFKWiwyREFJQSx1QkFBc0JDLFVBQXRCLENBQWlDLE9BQWpDLENBSkEsS0FLQTtBQUNBLEdBQUMsMkJBQUFsQyxPQUFPLENBQUNTLEdBQVIsQ0FBWXdCLFFBQVosa0ZBQXNCRSxPQUF0QixDQUE4QixjQUE5QixNQUFpRCxDQUFDLENBQW5ELElBQXdELENBQUMsQ0FOekQsSUFPQW5DLE9BQU8sQ0FBQ1MsR0FBUixDQUFZd0IsUUFBWixLQUF5QixHQVIzQjtBQVVEOztBQUVELFNBQVNGLE9BQVQsQ0FDRTlCLFFBREYsRUFFRW1DLGVBRkYsRUFHRWIsZ0JBSEYsRUFJRWMsZ0JBSkYsRUFJeUI7QUFDdkJWLG1CQUxGLEVBTUVFLG9CQU5GLEVBT3NCO0FBQ3BCLFFBQU1TLHFCQUFxQixHQUFHLHVCQUFVRCxnQkFBVixDQUE5QjtBQUNBLFNBQU9DLHFCQUFxQixDQUFDckMsUUFBRCxDQUE1Qjs7QUFFQSxNQUFJLENBQUMwQixtQkFBRCxJQUF3QixDQUFDRSxvQkFBekIsSUFBaUQscUJBQVFTLHFCQUFSLENBQXJELEVBQXFGO0FBQ25GLFdBQU9yQixTQUFQO0FBQ0Q7O0FBRUQsTUFBSUYsYUFBYSxDQUFDcUIsZUFBRCxDQUFqQixFQUFvQztBQUNsQ0EsSUFBQUEsZUFBZSxHQUFHQSxlQUFsQjs7QUFDQSxRQUFJQSxlQUFlLElBQUlDLGdCQUF2QixFQUF5QztBQUN2QyxhQUFPRCxlQUFQO0FBQ0Q7O0FBRUQsUUFDRVAsb0JBQW9CLElBQ3BCNUIsUUFBUSxJQUFJb0MsZ0JBRFosSUFFQXhDLGVBQWUsQ0FBQ3NDLE9BQWhCLENBQXdCQyxlQUF4QixJQUEyQyxDQUFDLENBSDlDLEVBSUU7QUFDQSxhQUFPekMscUJBQVA7QUFDRDs7QUFFRCxRQUFJZ0MsbUJBQW1CLElBQUk3QixjQUFjLENBQUNxQyxPQUFmLENBQXVCQyxlQUF2QixJQUEwQyxDQUFDLENBQXRFLEVBQXlFO0FBQ3ZFLGFBQU94QyxvQkFBUDtBQUNEO0FBQ0Y7O0FBRUQsTUFBSTJCLGdCQUFnQixJQUFJLENBQUMscUJBQVFBLGdCQUFSLENBQXpCLEVBQW9EO0FBQ2xELFNBQUssTUFBTWdCLE9BQVgsSUFBc0JoQixnQkFBdEIsRUFBd0M7QUFDdEMsWUFBTVAsTUFBTSxHQUFHdUIsT0FBTyxDQUFDQyxXQUFSLEVBQWY7O0FBRUEsVUFBSWIsbUJBQW1CLElBQUk3QixjQUFjLENBQUNxQyxPQUFmLENBQXVCbkIsTUFBdkIsSUFBaUMsQ0FBQyxDQUE3RCxFQUFnRTtBQUM5RCxlQUFPcEIsb0JBQVA7QUFDRDs7QUFFRCxVQUNFaUMsb0JBQW9CLElBQ3BCaEMsZUFBZSxDQUFDc0MsT0FBaEIsQ0FBd0JuQixNQUF4QixJQUFrQyxDQUFDLENBRG5DLElBRUFmLFFBQVEsSUFBSW9DLGdCQUhkLEVBSUU7QUFDQSxlQUFPMUMscUJBQVA7QUFDRDs7QUFFRCxVQUFJcUIsTUFBTSxJQUFJcUIsZ0JBQWQsRUFBZ0M7QUFDOUIsZUFBT3JCLE1BQVA7QUFDRDtBQUNGO0FBQ0Y7O0FBRUQsTUFBSVcsbUJBQUosRUFBeUI7QUFDdkIsV0FBTy9CLG9CQUFQO0FBQ0Q7O0FBRUQsTUFBSWlDLG9CQUFKLEVBQTBCO0FBQ3hCLFdBQU9sQyxxQkFBUDtBQUNELEdBdkRtQixDQXlEcEI7OztBQUNBLFNBQU8scUJBQVEyQyxxQkFBUixFQUErQixNQUFNLElBQXJDLENBQVA7QUFDRDtBQUVEO0FBQ0E7QUFDQTtBQUNBO0FBQ0E7QUFDQTtBQUNBOzs7QUFDTyxTQUFTdkIsYUFBVCxDQUF1QkMsTUFBdkIsRUFBbUU7QUFDeEUsU0FBT0EsTUFBTSxLQUFLQyxTQUFYLElBQXdCRCxNQUFNLEtBQUssSUFBMUM7QUFDRCIsInNvdXJjZXNDb250ZW50IjpbIi8qXG4gKiAgIENvcHlyaWdodCBPcGVuU2VhcmNoIENvbnRyaWJ1dG9yc1xuICpcbiAqICAgTGljZW5zZWQgdW5kZXIgdGhlIEFwYWNoZSBMaWNlbnNlLCBWZXJzaW9uIDIuMCAodGhlIFwiTGljZW5zZVwiKS5cbiAqICAgWW91IG1heSBub3QgdXNlIHRoaXMgZmlsZSBleGNlcHQgaW4gY29tcGxpYW5jZSB3aXRoIHRoZSBMaWNlbnNlLlxuICogICBBIGNvcHkgb2YgdGhlIExpY2Vuc2UgaXMgbG9jYXRlZCBhdFxuICpcbiAqICAgICAgIGh0dHA6Ly93d3cuYXBhY2hlLm9yZy9saWNlbnNlcy9MSUNFTlNFLTIuMFxuICpcbiAqICAgb3IgaW4gdGhlIFwibGljZW5zZVwiIGZpbGUgYWNjb21wYW55aW5nIHRoaXMgZmlsZS4gVGhpcyBmaWxlIGlzIGRpc3RyaWJ1dGVkXG4gKiAgIG9uIGFuIFwiQVMgSVNcIiBCQVNJUywgV0lUSE9VVCBXQVJSQU5USUVTIE9SIENPTkRJVElPTlMgT0YgQU5ZIEtJTkQsIGVpdGhlclxuICogICBleHByZXNzIG9yIGltcGxpZWQuIFNlZSB0aGUgTGljZW5zZSBmb3IgdGhlIHNwZWNpZmljIGxhbmd1YWdlIGdvdmVybmluZ1xuICogICBwZXJtaXNzaW9ucyBhbmQgbGltaXRhdGlvbnMgdW5kZXIgdGhlIExpY2Vuc2UuXG4gKi9cblxuaW1wb3J0IHsgaXNFbXB0eSwgZmluZEtleSwgY2xvbmVEZWVwIH0gZnJvbSAnbG9kYXNoJztcbmltcG9ydCB7IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVxdWVzdCB9IGZyb20gJy4uLy4uLy4uLy4uL3NyYy9jb3JlL3NlcnZlcic7XG5pbXBvcnQgeyBTZWN1cml0eVNlc3Npb25Db29raWUgfSBmcm9tICcuLi9zZXNzaW9uL3NlY3VyaXR5X2Nvb2tpZSc7XG5pbXBvcnQgeyBTZWN1cml0eVBsdWdpbkNvbmZpZ1R5cGUgfSBmcm9tICcuLic7XG5cbmNvbnN0IFBSSVZBVEVfVEVOQU5UX1NZTUJPTDogc3RyaW5nID0gJ19fdXNlcl9fJztcbmNvbnN0IEdMT0JBTF9URU5BTlRfU1lNQk9MOiBzdHJpbmcgPSAnJztcblxuZXhwb3J0IGNvbnN0IFBSSVZBVEVfVEVOQU5UUzogc3RyaW5nW10gPSBbUFJJVkFURV9URU5BTlRfU1lNQk9MLCAncHJpdmF0ZSddO1xuZXhwb3J0IGNvbnN0IEdMT0JBTF9URU5BTlRTOiBzdHJpbmdbXSA9IFsnZ2xvYmFsJywgR0xPQkFMX1RFTkFOVF9TWU1CT0xdO1xuLyoqXG4gKiBSZXNvdmxlcyB0aGUgdGVuYW50IHRoZSB1c2VyIGlzIHVzaW5nLlxuICpcbiAqIEBwYXJhbSByZXF1ZXN0IE9wZW5TZWFyY2hEYXNoYm9hcmRzIHJlcXVlc3QuXG4gKiBAcGFyYW0gY29uZmlnIHNlY3VyaXR5IHBsdWdpbiBjb25maWcuXG4gKiBAcGFyYW0gY29va2llIGNvb2tpZSBleHRyYWN0ZWQgZnJvbSB0aGUgcmVxdWVzdC4gVGhlIGNvb2tpZSBzaG91bGQgaGF2ZSBiZWVuIHBhcnNlZCBieSBBdXRoZW50aWNhdGlvbkhhbmRsZXIuXG4gKiBwYXNzIGl0IGFzIHBhcmFtZXRlciBpbnN0ZWFkIG9mIGV4dHJhY3RpbmcgYWdhaW4uXG4gKiBAcGFyYW0gYXV0aEluZm8gYXV0aGVudGljYXRpb24gaW5mbywgdGhlIEVsYXN0aWNzZWFyY2ggYXV0aGluZm8gQVBJIHJlc3BvbnNlLlxuICpcbiAqIEByZXR1cm5zIHVzZXIgcHJlZmVycmVkIHRlbmFudCBvZiB0aGUgcmVxdWVzdC5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIHJlc29sdmVUZW5hbnQoXG4gIHJlcXVlc3Q6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVxdWVzdCxcbiAgdXNlcm5hbWU6IHN0cmluZyxcbiAgcm9sZXM6IHN0cmluZ1tdIHwgdW5kZWZpbmVkLFxuICBhdmFpbGFiZVRlbmFudHM6IGFueSxcbiAgY29uZmlnOiBTZWN1cml0eVBsdWdpbkNvbmZpZ1R5cGUsXG4gIGNvb2tpZTogU2VjdXJpdHlTZXNzaW9uQ29va2llXG4pOiBzdHJpbmcgfCB1bmRlZmluZWQge1xuICBjb25zdCBERUZBVUxUX1JFQURPTkxZX1JPTEVTID0gWydraWJhbmFfcmVhZF9vbmx5J107XG4gIGxldCBzZWxlY3RlZFRlbmFudDogc3RyaW5nIHwgdW5kZWZpbmVkO1xuICBjb25zdCBzZWN1cml0eVRlbmFudF8gPSByZXF1ZXN0Py51cmw/LnNlYXJjaFBhcmFtcz8uZ2V0KCdzZWN1cml0eVRlbmFudF8nKTtcbiAgY29uc3Qgc2VjdXJpdHl0ZW5hbnQgPSByZXF1ZXN0Py51cmw/LnNlYXJjaFBhcmFtcz8uZ2V0KCdzZWN1cml0eXRlbmFudCcpO1xuICAvLyBlc2xpbnQtZGlzYWJsZS1uZXh0LWxpbmUgQHR5cGVzY3JpcHQtZXNsaW50L25hbWluZy1jb252ZW50aW9uXG4gIGNvbnN0IHNlY3VyaXR5X3RlbmFudCA9IHJlcXVlc3Q/LnVybD8uc2VhcmNoUGFyYW1zPy5nZXQoJ3NlY3VyaXR5X3RlbmFudCcpO1xuICBpZiAoc2VjdXJpdHlUZW5hbnRfKSB7XG4gICAgc2VsZWN0ZWRUZW5hbnQgPSBzZWN1cml0eVRlbmFudF87XG4gIH0gZWxzZSBpZiAoc2VjdXJpdHl0ZW5hbnQpIHtcbiAgICBzZWxlY3RlZFRlbmFudCA9IHNlY3VyaXR5dGVuYW50O1xuICB9IGVsc2UgaWYgKHNlY3VyaXR5X3RlbmFudCkge1xuICAgIHNlbGVjdGVkVGVuYW50ID0gc2VjdXJpdHlfdGVuYW50O1xuICB9IGVsc2UgaWYgKHJlcXVlc3QuaGVhZGVycy5zZWN1cml0eXRlbmFudCB8fCByZXF1ZXN0LmhlYWRlcnMuc2VjdXJpdHlUZW5hbnRfKSB7XG4gICAgc2VsZWN0ZWRUZW5hbnQgPSByZXF1ZXN0LmhlYWRlcnMuc2VjdXJpdHl0ZW5hbnRcbiAgICAgID8gKHJlcXVlc3QuaGVhZGVycy5zZWN1cml0eXRlbmFudCBhcyBzdHJpbmcpXG4gICAgICA6IChyZXF1ZXN0LmhlYWRlcnMuc2VjdXJpdHlUZW5hbnRfIGFzIHN0cmluZyk7XG4gIH0gZWxzZSBpZiAoaXNWYWxpZFRlbmFudChjb29raWUudGVuYW50KSkge1xuICAgIHNlbGVjdGVkVGVuYW50ID0gY29va2llLnRlbmFudDtcbiAgfSBlbHNlIHtcbiAgICBzZWxlY3RlZFRlbmFudCA9IHVuZGVmaW5lZDtcbiAgfVxuICBjb25zdCBpc1JlYWRvbmx5ID0gcm9sZXM/LnNvbWUoXG4gICAgKHJvbGUpID0+IGNvbmZpZy5yZWFkb25seV9tb2RlPy5yb2xlcy5pbmNsdWRlcyhyb2xlKSB8fCBERUZBVUxUX1JFQURPTkxZX1JPTEVTLmluY2x1ZGVzKHJvbGUpXG4gICk7XG5cbiAgY29uc3QgcHJlZmVycmVkVGVuYW50cyA9IGNvbmZpZy5tdWx0aXRlbmFuY3k/LnRlbmFudHMucHJlZmVycmVkO1xuICBjb25zdCBnbG9iYWxUZW5hbnRFbmFibGVkID0gY29uZmlnLm11bHRpdGVuYW5jeT8udGVuYW50cy5lbmFibGVfZ2xvYmFsO1xuICBjb25zdCBwcml2YXRlVGVuYW50RW5hYmxlZCA9IGNvbmZpZy5tdWx0aXRlbmFuY3k/LnRlbmFudHMuZW5hYmxlX3ByaXZhdGUgJiYgIWlzUmVhZG9ubHk7XG5cbiAgcmV0dXJuIHJlc29sdmUoXG4gICAgdXNlcm5hbWUsXG4gICAgc2VsZWN0ZWRUZW5hbnQsXG4gICAgcHJlZmVycmVkVGVuYW50cyxcbiAgICBhdmFpbGFiZVRlbmFudHMsXG4gICAgZ2xvYmFsVGVuYW50RW5hYmxlZCxcbiAgICBwcml2YXRlVGVuYW50RW5hYmxlZFxuICApO1xufVxuXG4vKipcbiAqIERldGVybWluZXMgd2hldGhlciB0aGUgcmVxdWVzdCByZXF1aXJlcyB0ZW5hbnQgaW5mby5cbiAqIEBwYXJhbSByZXF1ZXN0IG9wZW5zZWFyY2gtZGFzaGJvYXJkcyByZXF1ZXN0LlxuICpcbiAqIEByZXR1cm5zIHRydWUgaWYgdGhlIHJlcXVlc3QgcmVxdWlyZXMgdGVuYW50IGluZm8sIG90aGVyd2lzZSBmYWxzZS5cbiAqL1xuZXhwb3J0IGZ1bmN0aW9uIGlzTXVsdGl0ZW5hbnRQYXRoKHJlcXVlc3Q6IE9wZW5TZWFyY2hEYXNoYm9hcmRzUmVxdWVzdCk6IGJvb2xlYW4ge1xuICByZXR1cm4gKFxuICAgIHJlcXVlc3QudXJsLnBhdGhuYW1lPy5zdGFydHNXaXRoKCcvb3BlbnNlYXJjaCcpIHx8XG4gICAgcmVxdWVzdC51cmwucGF0aG5hbWU/LnN0YXJ0c1dpdGgoJy9hcGknKSB8fFxuICAgIHJlcXVlc3QudXJsLnBhdGhuYW1lPy5zdGFydHNXaXRoKCcvYXBwJykgfHxcbiAgICAvLyBzaG9ydCB1cmwgcGF0aFxuICAgIHJlcXVlc3QudXJsLnBhdGhuYW1lPy5zdGFydHNXaXRoKCcvZ290bycpIHx8XG4gICAgLy8gYm9vdHN0cmFwLmpzIGRlcGVuZHMgb24gdGVuYW50IGluZm8gdG8gZmV0Y2ggb3BlbnNlYXJjaC1kYXNoYm9hcmRzIGNvbmZpZ3MgaW4gdGVuYW50IGluZGV4XG4gICAgKHJlcXVlc3QudXJsLnBhdGhuYW1lPy5pbmRleE9mKCdib290c3RyYXAuanMnKSB8fCAtMSkgPiAtMSB8fFxuICAgIHJlcXVlc3QudXJsLnBhdGhuYW1lID09PSAnLydcbiAgKTtcbn1cblxuZnVuY3Rpb24gcmVzb2x2ZShcbiAgdXNlcm5hbWU6IHN0cmluZyxcbiAgcmVxdWVzdGVkVGVuYW50OiBzdHJpbmcgfCB1bmRlZmluZWQsXG4gIHByZWZlcnJlZFRlbmFudHM6IHN0cmluZ1tdIHwgdW5kZWZpbmVkLFxuICBhdmFpbGFibGVUZW5hbnRzOiBhbnksIC8vIGlzIGFuIG9iamVjdCBsaWtlIHsgdGVuYW50X25hbWVfMTogdHJ1ZSwgdGVuYW50X25hbWVfMjogZmFsc2UsIC4uLiB9XG4gIGdsb2JhbFRlbmFudEVuYWJsZWQ6IGJvb2xlYW4sXG4gIHByaXZhdGVUZW5hbnRFbmFibGVkOiBib29sZWFuXG4pOiBzdHJpbmcgfCB1bmRlZmluZWQge1xuICBjb25zdCBhdmFpbGFibGVUZW5hbnRzQ2xvbmUgPSBjbG9uZURlZXAoYXZhaWxhYmxlVGVuYW50cyk7XG4gIGRlbGV0ZSBhdmFpbGFibGVUZW5hbnRzQ2xvbmVbdXNlcm5hbWVdO1xuXG4gIGlmICghZ2xvYmFsVGVuYW50RW5hYmxlZCAmJiAhcHJpdmF0ZVRlbmFudEVuYWJsZWQgJiYgaXNFbXB0eShhdmFpbGFibGVUZW5hbnRzQ2xvbmUpKSB7XG4gICAgcmV0dXJuIHVuZGVmaW5lZDtcbiAgfVxuXG4gIGlmIChpc1ZhbGlkVGVuYW50KHJlcXVlc3RlZFRlbmFudCkpIHtcbiAgICByZXF1ZXN0ZWRUZW5hbnQgPSByZXF1ZXN0ZWRUZW5hbnQhO1xuICAgIGlmIChyZXF1ZXN0ZWRUZW5hbnQgaW4gYXZhaWxhYmxlVGVuYW50cykge1xuICAgICAgcmV0dXJuIHJlcXVlc3RlZFRlbmFudDtcbiAgICB9XG5cbiAgICBpZiAoXG4gICAgICBwcml2YXRlVGVuYW50RW5hYmxlZCAmJlxuICAgICAgdXNlcm5hbWUgaW4gYXZhaWxhYmxlVGVuYW50cyAmJlxuICAgICAgUFJJVkFURV9URU5BTlRTLmluZGV4T2YocmVxdWVzdGVkVGVuYW50KSA+IC0xXG4gICAgKSB7XG4gICAgICByZXR1cm4gUFJJVkFURV9URU5BTlRfU1lNQk9MO1xuICAgIH1cblxuICAgIGlmIChnbG9iYWxUZW5hbnRFbmFibGVkICYmIEdMT0JBTF9URU5BTlRTLmluZGV4T2YocmVxdWVzdGVkVGVuYW50KSA+IC0xKSB7XG4gICAgICByZXR1cm4gR0xPQkFMX1RFTkFOVF9TWU1CT0w7XG4gICAgfVxuICB9XG5cbiAgaWYgKHByZWZlcnJlZFRlbmFudHMgJiYgIWlzRW1wdHkocHJlZmVycmVkVGVuYW50cykpIHtcbiAgICBmb3IgKGNvbnN0IGVsZW1lbnQgb2YgcHJlZmVycmVkVGVuYW50cykge1xuICAgICAgY29uc3QgdGVuYW50ID0gZWxlbWVudC50b0xvd2VyQ2FzZSgpO1xuXG4gICAgICBpZiAoZ2xvYmFsVGVuYW50RW5hYmxlZCAmJiBHTE9CQUxfVEVOQU5UUy5pbmRleE9mKHRlbmFudCkgPiAtMSkge1xuICAgICAgICByZXR1cm4gR0xPQkFMX1RFTkFOVF9TWU1CT0w7XG4gICAgICB9XG5cbiAgICAgIGlmIChcbiAgICAgICAgcHJpdmF0ZVRlbmFudEVuYWJsZWQgJiZcbiAgICAgICAgUFJJVkFURV9URU5BTlRTLmluZGV4T2YodGVuYW50KSA+IC0xICYmXG4gICAgICAgIHVzZXJuYW1lIGluIGF2YWlsYWJsZVRlbmFudHNcbiAgICAgICkge1xuICAgICAgICByZXR1cm4gUFJJVkFURV9URU5BTlRfU1lNQk9MO1xuICAgICAgfVxuXG4gICAgICBpZiAodGVuYW50IGluIGF2YWlsYWJsZVRlbmFudHMpIHtcbiAgICAgICAgcmV0dXJuIHRlbmFudDtcbiAgICAgIH1cbiAgICB9XG4gIH1cblxuICBpZiAoZ2xvYmFsVGVuYW50RW5hYmxlZCkge1xuICAgIHJldHVybiBHTE9CQUxfVEVOQU5UX1NZTUJPTDtcbiAgfVxuXG4gIGlmIChwcml2YXRlVGVuYW50RW5hYmxlZCkge1xuICAgIHJldHVybiBQUklWQVRFX1RFTkFOVF9TWU1CT0w7XG4gIH1cblxuICAvLyBmYWxsIGJhY2sgdG8gdGhlIGZpcnN0IHRlbmFudCBpbiB0aGUgYXZhaWxhYmxlIHRlbmFudHNcbiAgcmV0dXJuIGZpbmRLZXkoYXZhaWxhYmxlVGVuYW50c0Nsb25lLCAoKSA9PiB0cnVlKTtcbn1cblxuLyoqXG4gKiBSZXR1cm4gdHJ1ZSBpZiB0ZW5hbnQgcGFyYW1ldGVyIGlzIGEgdmFsaWQgdGVuZW50LlxuICpcbiAqIE5vdGU6IGVtcHR5IHN0cmluZyAnJyBpcyB2YWxpZCwgd2hpY2ggbWVhbnMgZ2xvYmFsIHRlbmFudC5cbiAqXG4gKiBAcGFyYW0gdGVuYW50XG4gKi9cbmV4cG9ydCBmdW5jdGlvbiBpc1ZhbGlkVGVuYW50KHRlbmFudDogc3RyaW5nIHwgdW5kZWZpbmVkIHwgbnVsbCk6IGJvb2xlYW4ge1xuICByZXR1cm4gdGVuYW50ICE9PSB1bmRlZmluZWQgJiYgdGVuYW50ICE9PSBudWxsO1xufVxuIl19