\echo Use "CREATE EXTENSION postgis" to load this file. \quit
-- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
--
----
-- PostGIS - Spatial Types for PostgreSQL
-- http://postgis.net
--
-- Copyright (C) 2011 Regina Obe <lr@pcorp.us>
--
-- This is free software; you can redistribute and/or modify it under
-- the terms of the GNU General Public Licence. See the COPYING file.
--
-- Author: Regina Obe <lr@pcorp.us>
--
-- This is a suite of SQL helper functions for use during a PostGIS extension install/upgrade
-- The functions get uninstalled after the extention install/upgrade process
---------------------------
-- postgis_extension_remove_objects: This function removes objects of a particular class from an extension
-- this is needed because there is no ALTER EXTENSION DROP FUNCTION/AGGREGATE command
-- and we can't CREATE OR REPALCe functions whose signatures have changed and we can drop them if they are part of an extention
-- So we use this to remove it from extension first before we drop
CREATE OR REPLACE FUNCTION postgis_extension_remove_objects(param_extension text, param_type text)
  RETURNS boolean AS
$$
DECLARE
	var_sql text := '';
	var_r record;
	var_result boolean := false;
	var_class text := '';
	var_is_aggregate boolean := false;
	var_sql_list text := '';
	var_pgsql_version integer := current_setting('server_version_num');
BEGIN
		var_class := CASE WHEN lower(param_type) = 'function' OR lower(param_type) = 'aggregate' THEN 'pg_proc' ELSE '' END;
		var_is_aggregate := CASE WHEN lower(param_type) = 'aggregate' THEN true ELSE false END;

		IF var_pgsql_version < 110000 THEN
			var_sql_list := $sql$SELECT 'ALTER EXTENSION ' || e.extname || ' DROP ' || $3 || ' ' || COALESCE(proc.proname || '(' || oidvectortypes(proc.proargtypes) || ')' ,typ.typname, cd.relname, op.oprname,
					cs.typname || ' AS ' || ct.typname || ') ', opcname, opfname) || ';' AS remove_command
			FROM pg_depend As d INNER JOIN pg_extension As e
				ON d.refobjid = e.oid INNER JOIN pg_class As c ON
					c.oid = d.classid
					LEFT JOIN pg_proc AS proc ON proc.oid = d.objid
					LEFT JOIN pg_type AS typ ON typ.oid = d.objid
					LEFT JOIN pg_class As cd ON cd.oid = d.objid
					LEFT JOIN pg_operator As op ON op.oid = d.objid
					LEFT JOIN pg_cast AS ca ON ca.oid = d.objid
					LEFT JOIN pg_type AS cs ON ca.castsource = cs.oid
					LEFT JOIN pg_type AS ct ON ca.casttarget = ct.oid
					LEFT JOIN pg_opclass As oc ON oc.oid = d.objid
					LEFT JOIN pg_opfamily As ofa ON ofa.oid = d.objid
			WHERE d.deptype = 'e' and e.extname = $1 and c.relname = $2 AND COALESCE(proc.proisagg, false) = $4;$sql$;
		ELSE -- for PostgreSQL 11 and above, they removed proc.proisagg among others and replaced with some func type thing
			var_sql_list := $sql$SELECT 'ALTER EXTENSION ' || e.extname || ' DROP ' || $3 || ' ' || COALESCE(proc.proname || '(' || oidvectortypes(proc.proargtypes) || ')' ,typ.typname, cd.relname, op.oprname,
					cs.typname || ' AS ' || ct.typname || ') ', opcname, opfname) || ';' AS remove_command
			FROM pg_depend As d INNER JOIN pg_extension As e
				ON d.refobjid = e.oid INNER JOIN pg_class As c ON
					c.oid = d.classid
					LEFT JOIN pg_proc AS proc ON proc.oid = d.objid
					LEFT JOIN pg_type AS typ ON typ.oid = d.objid
					LEFT JOIN pg_class As cd ON cd.oid = d.objid
					LEFT JOIN pg_operator As op ON op.oid = d.objid
					LEFT JOIN pg_cast AS ca ON ca.oid = d.objid
					LEFT JOIN pg_type AS cs ON ca.castsource = cs.oid
					LEFT JOIN pg_type AS ct ON ca.casttarget = ct.oid
					LEFT JOIN pg_opclass As oc ON oc.oid = d.objid
					LEFT JOIN pg_opfamily As ofa ON ofa.oid = d.objid
			WHERE d.deptype = 'e' and e.extname = $1 and c.relname = $2 AND (proc.prokind = 'a')  = $4;$sql$;
		END IF;

		FOR var_r IN EXECUTE var_sql_list  USING param_extension, var_class, param_type, var_is_aggregate
		LOOP
			var_sql := var_sql || var_r.remove_command || ';';
		END LOOP;
		IF var_sql > '' THEN
			EXECUTE var_sql;
			var_result := true;
		END IF;

		RETURN var_result;
END;
$$
LANGUAGE plpgsql VOLATILE;

CREATE OR REPLACE FUNCTION postgis_extension_drop_if_exists(param_extension text, param_statement text)
  RETURNS boolean AS
$$
DECLARE
	var_sql_ext text := 'ALTER EXTENSION ' || quote_ident(param_extension) || ' ' || replace(param_statement, 'IF EXISTS', '');
	var_result boolean := false;
BEGIN
	BEGIN
		EXECUTE var_sql_ext;
		var_result := true;
	EXCEPTION
		WHEN OTHERS THEN
			--this is to allow ignoring if the object does not exist in extension
			var_result := false;
	END;
	RETURN var_result;
END;
$$
LANGUAGE plpgsql VOLATILE;

CREATE OR REPLACE FUNCTION postgis_extension_AddToSearchPath(a_schema_name varchar)
RETURNS text
AS
$$
DECLARE
	var_result text;
	var_cur_search_path text;
BEGIN
	SELECT reset_val INTO var_cur_search_path FROM pg_settings WHERE name = 'search_path';
	IF var_cur_search_path LIKE '%' || quote_ident(a_schema_name) || '%' THEN
		var_result := a_schema_name || ' already in database search_path';
	ELSE
		var_cur_search_path := var_cur_search_path || ', '
                        || quote_ident(a_schema_name);
		EXECUTE 'ALTER DATABASE ' || quote_ident(current_database())
                              || ' SET search_path = ' || var_cur_search_path;
		var_result := a_schema_name || ' has been added to end of database search_path ';
	END IF;

	EXECUTE 'SET search_path = ' || var_cur_search_path;

  RETURN var_result;
END
$$
LANGUAGE 'plpgsql' VOLATILE STRICT;
DO $unpackage_if_needed$
BEGIN
	IF EXISTS (
		select t.typname from pg_depend d, pg_extension e, pg_type t where
		e.extname = 'postgis' and d.refclassid =
		'pg_catalog.pg_extension'::pg_catalog.regclass and d.refobjid = e.oid
		and d.classid = 'pg_type'::regclass and d.objid = t.oid
		and t.typname = 'raster'
	) THEN

		RAISE WARNING 'unpackaging raster';

		EXECUTE $unpackage$
		-- UNPACKAGE_CODE --
DO $$ BEGIN ALTER EXTENSION postgis DROP CAST (raster AS bytea);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_bytea(raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP CAST (raster AS box2d);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION box2d(raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP CAST (raster AS geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_MapAlgebra(raster, integer, text, text, nodatavaluerepl text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_MapAlgebra(raster, pixeltype text, expression text, nodatavaluerepl text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_MapAlgebraExpr(raster, integer, text, text, text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_MapAlgebraExpr(raster, text, text, text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_MapalgebraFct(raster, regprocedure);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_MapAlgebraFct(raster, text, regprocedure, VARIADIC text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_MapAlgebraFct(raster, text, regprocedure);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_MapAlgebraFct(raster, regprocedure, VARIADIC text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_MapAlgebraFct(raster, integer, regprocedure, variadic text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_MapAlgebraFct(raster, integer, text, regprocedure, VARIADIC text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_MapAlgebraFct(raster, integer, text, regprocedure);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_MapAlgebraFct(raster, integer, regprocedure, variadic text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_MapalgebraFct(raster, integer, regprocedure);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_MapAlgebraFct(raster, raster, regprocedure, text, text, VARIADIC text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_MapAlgebraFct(raster, integer, raster, integer, regprocedure, text, text, VARIADIC text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_MapAlgebraFctNgb(raster, integer, text, integer, integer, regprocedure, text,  VARIADIC text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION  ST_MapAlgebraFct(raster, raster, regprocedure, VARIADIC text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_AsRaster(geometry , integer , integer , double precision , double precision , text , double precision , double precision , double precision , double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_AsRaster(geometry , integer , integer , text[] , double precision[] , double precision[] , double precision , double precision , double precision , double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_AsRaster(geometry , integer , integer , text , double precision , double precision , double precision , double precision , double precision , double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_AsRaster(geometry , integer , integer , double precision , double precision , text[] , double precision[] , double precision[] , double precision , double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_AsRaster(geometry , integer , integer , double precision , double precision , text[] , double precision[] , double precision[] , double precision , double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_AsRaster(geometry , double precision , double precision , text , double precision , double precision , double precision , double precision , double precision , double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_AsRaster(geometry , raster , text , double precision , double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _ST_AsRaster(geometry,double precision , double precision, integer , integer,text[] , double precision[] ,double precision[] ,  double precision,  double precision, double precision,double precision, double precision, double precision,touched boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _ST_Resample(raster, text, double precision, integer, double precision, double precision, double precision, double precision, double precision, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_Resample(raster, raster, text, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_HasNoBand(raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_BandMetaData(raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_BandNoDataValue(raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_BandNoDataValue(raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_SetGeoReference(raster, text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_SetGeoReference(raster, text, text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_setbandisnodata(raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_setbandisnodata(raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_setbandnodatavalue(raster, integer, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_setbandnodatavalue(raster, integer, double precision, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_dumpaspolygons(raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_dumpaspolygons(raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_polygon(raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_polygon(raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_makeemptyraster(int, int, float8, float8, float8, float8, float8, float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_makeemptyraster(int, int, float8, float8, float8, float8, float8, float8, int4);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_addband(raster, text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_addband(raster, text, float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_addband(raster, int, text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_addband(raster, int, text, float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_addband(raster, raster, int);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_addband(raster, raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_addband(raster, text, float8, float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_addband(raster, int, text, float8, float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_addband(raster, raster, int, int);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_bandisnodata(raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_bandisnodata(raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_bandisnodata(raster, integer, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_bandpath(raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_bandpath(raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_bandpixeltype(raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_bandpixeltype(raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_value(raster, integer, integer, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_value(raster, integer, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_value(raster, integer, geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_value(raster, geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_value(raster, integer, integer, integer, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_value(raster, integer, integer, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_value(raster, integer, geometry, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_value(raster, geometry, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_value(raster, integer, geometry, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_value(raster, geometry, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_georeference(raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_georeference(raster, text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION dumpaswktpolygons(raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_bandmetadata(raster, VARIADIC int[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_PixelAsPolygons(raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_PixelAsPolygons(raster,integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_summarystats(raster,int, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_summarystats(raster, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxsummarystats(raster,int, boolean, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxsummarystats(raster,int, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxsummarystats(raster, boolean, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxsummarystats(raster, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_summarystats(text, text,integer, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_summarystats(text, text, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxsummarystats(text, text,integer, boolean, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxsummarystats(text, text,integer, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxsummarystats(text, text, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxsummarystats(text, text, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_summarystats(raster,int, boolean, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_summarystats(text, text,integer, boolean, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_quantile(raster, int, boolean, double precision[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_quantile(raster, int, double precision[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_quantile(raster, double precision[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile(raster, int, boolean, double precision, double precision[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile(raster, int, double precision, double precision[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile(raster, double precision, double precision[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile(raster, double precision[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_quantile(text, text, int, boolean, double precision[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_quantile(text, text, int, double precision[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_quantile(text, text, double precision[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile(text, text, int, boolean, double precision, double precision[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile(text, text, int, double precision, double precision[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile(text, text, double precision, double precision[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile(text, text, double precision[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_quantile(raster, int, boolean, double precision, double precision[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_quantile(text, text, int, boolean, double precision, double precision[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP TYPE quantile;EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuecount(text, text, integer, double precision, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuecount(text, text, integer, boolean, double precision[], double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuecount(text, text, double precision[], double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuecount(text, text, integer, double precision[], double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuecount(text, text, integer, boolean, double precision, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuecount(text, text, double precision, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuecount(raster, integer, boolean, double precision[], double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuecount(raster, integer, double precision[], double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuecount(raster, double precision[], double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_valuecount(text, text, integer, boolean, double precision[], double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_valuecount(raster, integer, boolean, double precision[], double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP TYPE valuecount;EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_histogram(raster, int, boolean, int, double precision[], boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_histogram(raster, int, boolean, int, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_histogram(raster, int, int, double precision[], boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_histogram(raster, int, int, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram( raster, int, boolean, double precision, int, double precision[], boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram(raster, int, boolean, double precision, int, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram(raster, int, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram(raster, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram(raster, int, double precision, int, double precision[], boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram(raster, int, double precision, int, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_histogram(text, text, int, boolean, int, double precision[], boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_histogram(text, text, int, boolean, int, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_histogram(text, text, int, int, double precision[], boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_histogram(text, text, int, int, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram( text, text, int, boolean, double precision, int, double precision[], boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram(text, text, int, boolean, double precision, int, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram(text, text, int, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram(text, text, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram(text, text, int, double precision, int, double precision[], boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram(text, text, int, double precision, int, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_histogram( raster, int, boolean, double precision, int, double precision[], boolean, double precision, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_histogram( text, text, int, boolean, double precision, int, double precision[], boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP TYPE histogram;EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP TYPE bandmetadata;EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP TYPE geomvalxy;EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _rename_raster_tables();EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _rename_raster_tables();EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP VIEW raster_columns;EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION AddRasterColumn(varchar, varchar, varchar, varchar, integer, varchar[], boolean, boolean, double precision[], double precision, double precision, integer, integer, geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION AddRasterColumn(varchar, varchar, varchar, integer, varchar[], boolean, boolean, double precision[], double precision, double precision, integer, integer, geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION AddRasterColumn(varchar, varchar, integer, varchar[], boolean, boolean, double precision[], double precision, double precision, integer, integer, geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION DropRasterColumn(varchar, varchar, varchar, varchar);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION DropRasterColumn(varchar, varchar, varchar);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION DropRasterColumn(varchar, varchar);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION DropRasterTable(varchar, varchar, varchar);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION DropRasterTable(varchar, varchar);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION DropRasterTable(varchar);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION AddRasterConstraints(name, name, name, boolean, boolean, boolean, boolean, boolean, boolean, boolean, boolean, boolean, boolean, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION AddRasterConstraints(name, name, boolean, boolean, boolean, boolean, boolean, boolean, boolean, boolean, boolean, boolean, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION DropRasterConstraints(name, name, name, boolean, boolean, boolean, boolean, boolean, boolean, boolean, boolean, boolean, boolean, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION DropRasterConstraints(name, name, boolean, boolean, boolean, boolean, boolean, boolean, boolean, boolean, boolean, boolean, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _drop_st_samealignment();EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_intersects(raster, integer, raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersects(raster, integer, raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersects(raster, raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersection(raster, integer, geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersection(raster, geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_minpossibleval(text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_pixelaspolygon(raster, integer, integer, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersection(raster, int, geometry, text, regprocedure);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersection(raster, int, geometry, regprocedure);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersection(raster, geometry, text, regprocedure);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersection(raster, geometry, regprocedure);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_clip(raster, integer, geometry, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_clip(raster, geometry, float8, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_clip(raster, geometry, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_clip(raster, int, geometry, float8, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_clip(raster, geometry, float8[], boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_clip(raster, integer, geometry, float8[], boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_dumpaswktpolygons(raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP TYPE wktgeomval;EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_nearestvalue(raster, integer, integer, integer, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_nearestvalue(raster, integer, integer, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_neighborhood(raster, integer, integer, integer, integer, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_neighborhood(raster, integer, integer, integer, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_neighborhood(raster, integer, geometry, integer, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_neighborhood(raster, geometry, integer, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersection(raster, integer, raster, integer, text, regprocedure);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersection(raster, integer, raster, integer, regprocedure);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersection(raster, raster, text, regprocedure);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersection(raster, raster, regprocedure);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_pixelaspolygons(raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_bandsurface(raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersects(raster, integer, geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersects(raster, geometry, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersects(geometry, raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_intersects(raster, geometry, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_intersects(geometry, raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_overlaps(geometry, raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_overlaps(raster, integer, geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_overlaps(raster, geometry, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_overlaps(raster, geometry, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_overlaps(geometry, raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_touches(geometry, raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_touches(raster, geometry, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_touches(raster, integer, geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_touches(geometry, raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_touches(raster, geometry, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_contains(raster, geometry, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_contains(raster, integer, geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_contains(geometry, raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_contains(raster, geometry, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_contains(geometry, raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_addband(raster, raster[], integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_slope(raster, integer, text, text, double precision, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_slope(raster, integer, text, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_slope(raster, integer, text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_aspect(raster, integer, text, text, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_aspect(raster, integer, text, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_aspect(raster, integer, text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_hillshade(raster, integer, text, double precision, double precision, double precision, double precision, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_hillshade(raster, integer, text, float, float, float, float, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_hillshade(raster, integer, text, float, float, float, float);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_tile(raster, integer, integer, integer[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_tile(raster, integer, integer, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_setvalue(raster, integer, geometry, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_setvalue(raster, geometry, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_world2rastercoord(raster, double precision, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_world2rastercoord(raster, geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_world2rastercoord(raster, double precision, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_world2rastercoordx(raster, float8, float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_world2rastercoordx(raster, float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_world2rastercoordx(raster, geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_world2rastercoordy(raster, float8, float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_world2rastercoordy(raster, float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_world2rastercoordy(raster, geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_raster2worldcoord( raster, integer, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_raster2worldcoord(raster, integer, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_raster2worldcoordx(raster, int, int);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_raster2worldcoordx(raster, int);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_raster2worldcoordy(raster, int, int);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_raster2worldcoordy(raster, int);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_resample(raster, text, double precision, integer, double precision, double precision, double precision, double precision, double precision, double precision, integer, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_resample(raster, integer, double precision, double precision, double precision, double precision, double precision, double precision, text, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_resample(raster, integer, integer, integer, double precision, double precision, double precision, double precision, text, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_tile(raster, integer, integer, int[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_tile(raster, integer[], integer, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_tile(raster, integer, integer, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_tile(raster, integer, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _add_raster_constraint_regular_blocking(name, name, name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_asbinary(raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_aspect4ma(float8[], text, text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_hillshade4ma(float8[], text, text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_slope4ma(float8[], text, text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_gdaldrivers();EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_bandmetadata(raster, int[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_bandmetadata(raster, int);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP AGGREGATE ST_Union(raster, text, text, text, double precision, text, text, text, double precision, text, text, text, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP AGGREGATE ST_Union(raster, text, text, text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP AGGREGATE ST_Union(raster, text, text, text, double precision, text, text, text, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP AGGREGATE ST_Union(raster, text, text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP AGGREGATE ST_Union(raster, text, text, text, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP AGGREGATE ST_Union(raster, record[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_Intersects(raster,boolean,geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_Intersects(geometry,raster,boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_Intersects(raster,geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_Intersects(geometry,raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_Intersects(raster, integer, boolean, geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_Intersects(geometry, raster, integer, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_Intersection(raster,raster, integer, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_Intersection(geometry,raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_mapalgebra4unionfinal1(raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_mapalgebra4unionstate(raster, raster, int4);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_mapalgebra4unionstate(raster, raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_mapalgebra4unionstate(raster, raster, text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_mapalgebra4unionstate(raster, raster, int4, text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_mapalgebra4unionstate(raster, raster, text, text, text, float8, text, text, text, float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_mapalgebra(rastbandarg[],regprocedure,text,integer,integer,text,raster,text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_count(text, text, integer, boolean, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_count(text, text, int, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_count(text, text, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxcount(text, text, int, boolean, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxcount(text, text, int, boolean, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxcount(text, text, int, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxcount(text, text, boolean, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxcount(text, text, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_summarystats(text, text, integer, boolean, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_summarystats(text, text, integer, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_summarystats(text, text, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxsummarystats(text, text, integer, boolean, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxsummarystats(text, text, integer, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxsummarystats(text, text, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxsummarystats(text, text, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_histogram(text, text, int, boolean, double precision, int,double precision[], boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_histogram(text, text, int, boolean, int, double precision[], boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_histogram(text, text, int, boolean, int, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_histogram(text, text, int, int, double precision[], boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_histogram( ext, text, int, int, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram(text, text, int, boolean, double precision, int, double precision[], boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram(text, text, int, boolean, double precision, int, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram(text, text, int,double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram(text, text, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram(text, text, int, double precision, int, double precision[], boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram(text, text, int, double precision, int, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_quantile( rastertable text, rastercolumn text, nband int, exclude_nodata_value boolean, sample_percent double precision, quantiles double precision[], OUT quantile double precision, OUT value double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_quantile( rastertable text, rastercolumn text, nband int, exclude_nodata_value boolean, quantiles double precision[], OUT quantile double precision, OUT value double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_quantile( rastertable text, rastercolumn text, nband int, quantiles double precision[], OUT quantile double precision, OUT value double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_quantile( rastertable text, rastercolumn text, quantiles double precision[], OUT quantile double precision, OUT value double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_quantile(rastertable text, rastercolumn text, nband int, exclude_nodata_value boolean, quantile double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_quantile(rastertable text, rastercolumn text, nband int, quantile double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_quantile(rastertable text, rastercolumn text, exclude_nodata_value boolean, quantile double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_quantile(rastertable text, rastercolumn text, quantile double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile(rastertable text, rastercolumn text, nband int, exclude_nodata_value boolean, sample_percent double precision, quantiles double precision[], OUT quantile double precision, OUT value double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile( rastertable text, rastercolumn text, nband int, sample_percent double precision, quantiles double precision[], OUT quantile double precision, OUT value double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile( rastertable text, rastercolumn text, sample_percent double precision, quantiles double precision[], OUT quantile double precision, OUT value double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile( rastertable text, rastercolumn text, quantiles double precision[], OUT quantile double precision, OUT value double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile(rastertable text, rastercolumn text, nband int, exclude_nodata_value boolean, sample_percent double precision, quantile double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile(rastertable text, rastercolumn text, nband int, sample_percent double precision, quantile double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile(rastertable text, rastercolumn text, sample_percent double precision, quantile double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile(rastertable text, rastercolumn text, exclude_nodata_value boolean, quantile double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile(rastertable text, rastercolumn text, quantile double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP VIEW raster_columns;EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP VIEW raster_overviews;EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP AGGREGATE st_summarystatsagg (raster, integer, boolean, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP AGGREGATE st_summarystatsagg (raster, boolean, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP AGGREGATE st_summarystatsagg (raster, int, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP AGGREGATE st_countagg (raster, integer, boolean, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP AGGREGATE st_countagg (raster, integer, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP AGGREGATE st_countagg (raster, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP AGGREGATE st_samealignment (raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP AGGREGATE st_union (raster, unionarg[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP AGGREGATE st_union (raster, integer, text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP AGGREGATE st_union (raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP AGGREGATE st_union (raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP AGGREGATE st_union (raster, text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP OPERATOR CLASS hash_raster_ops USING hash;EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP OPERATOR FAMILY hash_raster_ops USING hash;EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP OPERATOR =  (raster,raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP OPERATOR <<  (raster,raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP OPERATOR &<  (raster,raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP OPERATOR <<|  (raster,raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP OPERATOR &<|  (raster,raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP OPERATOR &&  (raster,raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP OPERATOR &>  (raster,raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP OPERATOR >>  (raster,raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP OPERATOR |&>  (raster,raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP OPERATOR |>>  (raster,raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP OPERATOR ~=  (raster,raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP OPERATOR @  (raster,raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP OPERATOR ~  (raster,raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP OPERATOR ~  (raster,geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP OPERATOR @  (raster,geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP OPERATOR &&  (raster,geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP OPERATOR ~  (geometry,raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP OPERATOR @  (geometry,raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP OPERATOR &&  (geometry,raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP CAST (raster AS box3d);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP CAST (raster AS geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP CAST (raster AS bytea);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION postgis_raster_lib_version ();EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION postgis_raster_scripts_installed ();EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION postgis_raster_lib_build_date ();EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION postgis_gdal_version ();EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_envelope (raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_convexhull (raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_minconvexhull (	rast raster,	nband integer );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION box3d (raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_height (raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_numbands (raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_scalex (raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_scaley (raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_skewx (raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_skewy (raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_srid (raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_upperleftx (raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_upperlefty (raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_width (raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_pixelwidth (raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_pixelheight (raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_geotransform (raster,    OUT imag double precision,    OUT jmag double precision,    OUT theta_i double precision,    OUT theta_ij double precision,    OUT xoffset double precision,    OUT yoffset double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_rotation (raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_metadata (	rast raster,	OUT upperleftx double precision,	OUT upperlefty double precision,	OUT width int,	OUT height int,	OUT scalex double precision,	OUT scaley double precision,	OUT skewx double precision,	OUT skewy double precision,	OUT srid int,	OUT numbands int);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_summary (rast raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_MemSize (raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_makeemptyraster (width int, height int, upperleftx float8, upperlefty float8, scalex float8, scaley float8, skewx float8, skewy float8, srid int4 );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_makeemptyraster (width int, height int, upperleftx float8, upperlefty float8, pixelsize float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_makeemptyraster (rast raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_addband (rast raster, addbandargset addbandarg[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_addband (	rast raster,	index int,	pixeltype text,	initialvalue float8 ,	nodataval float8 );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_addband (	rast raster,	pixeltype text,	initialvalue float8 ,	nodataval float8 );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_addband (	torast raster,	fromrast raster,	fromband int ,	torastindex int );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_addband (	torast raster,	fromrasts raster[], fromband integer ,	torastindex int );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_addband (	rast raster,	index int,	outdbfile text, outdbindex int[],	nodataval double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_addband (	rast raster,	outdbfile text, outdbindex int[],	index int ,	nodataval double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_band (rast raster, nbands int[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_band (rast raster, nband int);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_band (rast raster, nbands text, delimiter char );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_summarystats (	rast raster,	nband int ,	exclude_nodata_value boolean ,	sample_percent double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_summarystats (	rast raster,	nband int ,	exclude_nodata_value boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_summarystats (	rast raster,	exclude_nodata_value boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxsummarystats (	rast raster,	nband int ,	exclude_nodata_value boolean ,	sample_percent double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxsummarystats (	rast raster,	nband int,	sample_percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxsummarystats (	rast raster,	exclude_nodata_value boolean,	sample_percent double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxsummarystats (	rast raster,	sample_percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_summarystats_finalfn (internal);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_summarystats_transfn (	internal,	raster, integer,	boolean, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_summarystats_transfn (	internal,	raster, boolean, double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_summarystats_transfn (	internal,	raster, int, boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_count (rast raster, nband int , exclude_nodata_value boolean , sample_percent double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_count (rast raster, nband int , exclude_nodata_value boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_count (rast raster, exclude_nodata_value boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxcount (rast raster, nband int , exclude_nodata_value boolean , sample_percent double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxcount (rast raster, nband int, sample_percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxcount (rast raster, exclude_nodata_value boolean, sample_percent double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxcount (rast raster, sample_percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_countagg_finalfn (agg agg_count);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION __st_countagg_transfn (	agg agg_count,	rast raster, 	nband integer , exclude_nodata_value boolean ,	sample_percent double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_countagg_transfn (	agg agg_count,	rast raster, 	nband integer, exclude_nodata_value boolean,	sample_percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_countagg_transfn (	agg agg_count,	rast raster, 	nband integer, exclude_nodata_value boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_countagg_transfn (	agg agg_count,	rast raster, 	exclude_nodata_value boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_histogram (	rast raster, nband int ,	exclude_nodata_value boolean ,	sample_percent double precision ,	bins int , width double precision[] ,	right boolean ,	min double precision , max double precision ,	OUT min double precision,	OUT max double precision,	OUT count bigint,	OUT percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_histogram (	rast raster, nband int ,	exclude_nodata_value boolean ,	bins int , width double precision[] ,	right boolean ,	OUT min double precision,	OUT max double precision,	OUT count bigint,	OUT percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_histogram (	rast raster, nband int,	exclude_nodata_value boolean,	bins int,	right boolean,	OUT min double precision,	OUT max double precision,	OUT count bigint,	OUT percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_histogram (	rast raster, nband int,	bins int, width double precision[] ,	right boolean ,	OUT min double precision,	OUT max double precision,	OUT count bigint,	OUT percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_histogram (	rast raster, nband int,	bins int,	right boolean,	OUT min double precision,	OUT max double precision,	OUT count bigint,	OUT percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram (	rast raster, nband int ,	exclude_nodata_value boolean ,	sample_percent double precision ,	bins int , width double precision[] ,	right boolean ,	OUT min double precision,	OUT max double precision,	OUT count bigint,	OUT percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram (	rast raster, nband int,	exclude_nodata_value boolean,	sample_percent double precision,	bins int,	right boolean,	OUT min double precision,	OUT max double precision,	OUT count bigint,	OUT percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram (	rast raster, nband int,	sample_percent double precision,	OUT min double precision,	OUT max double precision,	OUT count bigint,	OUT percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram (	rast raster,	sample_percent double precision,	OUT min double precision,	OUT max double precision,	OUT count bigint,	OUT percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram (	rast raster, nband int,	sample_percent double precision,	bins int, width double precision[] ,	right boolean ,	OUT min double precision,	OUT max double precision,	OUT count bigint,	OUT percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxhistogram (	rast raster, nband int,	sample_percent double precision,	bins int, right boolean,	OUT min double precision,	OUT max double precision,	OUT count bigint,	OUT percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_quantile (	rast raster,	nband int ,	exclude_nodata_value boolean ,	sample_percent double precision ,	quantiles double precision[] ,	OUT quantile double precision,	OUT value double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_quantile (	rast raster,	nband int ,	exclude_nodata_value boolean ,	quantiles double precision[] ,	OUT quantile double precision,	OUT value double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_quantile (	rast raster,	nband int,	quantiles double precision[],	OUT quantile double precision,	OUT value double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_quantile (	rast raster,	quantiles double precision[],	OUT quantile double precision,	OUT value double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_quantile (rast raster, nband int, exclude_nodata_value boolean, quantile double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_quantile (rast raster, nband int, quantile double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_quantile (rast raster, exclude_nodata_value boolean, quantile double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_quantile (rast raster, quantile double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile (	rast raster,	nband int ,	exclude_nodata_value boolean ,	sample_percent double precision ,	quantiles double precision[] ,	OUT quantile double precision,	OUT value double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile (	rast raster,	nband int,	sample_percent double precision,	quantiles double precision[] ,	OUT quantile double precision,	OUT value double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile (	rast raster,	sample_percent double precision,	quantiles double precision[] ,	OUT quantile double precision,	OUT value double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile (	rast raster,	quantiles double precision[],	OUT quantile double precision,	OUT value double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile (rast raster, nband int, exclude_nodata_value boolean, sample_percent double precision, quantile double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile (rast raster, nband int, sample_percent double precision, quantile double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile (rast raster, sample_percent double precision, quantile double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile (rast raster, exclude_nodata_value boolean, quantile double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_approxquantile (rast raster, quantile double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_valuecount (	rast raster, nband integer ,	exclude_nodata_value boolean ,	searchvalues double precision[] ,	roundto double precision ,	OUT value double precision,	OUT count integer,	OUT percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuecount (	rast raster, nband integer ,	exclude_nodata_value boolean ,	searchvalues double precision[] ,	roundto double precision ,	OUT value double precision, OUT count integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuecount (rast raster, nband integer, searchvalues double precision[], roundto double precision , OUT value double precision, OUT count integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuecount (rast raster, searchvalues double precision[], roundto double precision , OUT value double precision, OUT count integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuecount (rast raster, nband integer, exclude_nodata_value boolean, searchvalue double precision, roundto double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuecount (rast raster, nband integer, searchvalue double precision, roundto double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuecount (rast raster, searchvalue double precision, roundto double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuepercent (	rast raster, nband integer ,	exclude_nodata_value boolean ,	searchvalues double precision[] ,	roundto double precision ,	OUT value double precision, OUT percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuepercent (rast raster, nband integer, searchvalues double precision[], roundto double precision , OUT value double precision, OUT percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuepercent (rast raster, searchvalues double precision[], roundto double precision , OUT value double precision, OUT percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuepercent (rast raster, nband integer, exclude_nodata_value boolean, searchvalue double precision, roundto double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuepercent (rast raster, nband integer, searchvalue double precision, roundto double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuepercent (rast raster, searchvalue double precision, roundto double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_valuecount (	rastertable text,	rastercolumn text,	nband integer ,	exclude_nodata_value boolean ,	searchvalues double precision[] ,	roundto double precision ,	OUT value double precision,	OUT count integer,	OUT percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuecount (	rastertable text, rastercolumn text,	nband integer ,	exclude_nodata_value boolean ,	searchvalues double precision[] ,	roundto double precision ,	OUT value double precision, OUT count integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuecount (rastertable text, rastercolumn text, nband integer, searchvalues double precision[], roundto double precision , OUT value double precision, OUT count integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuecount (rastertable text, rastercolumn text, searchvalues double precision[], roundto double precision , OUT value double precision, OUT count integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuecount (rastertable text, rastercolumn text, nband integer, exclude_nodata_value boolean, searchvalue double precision, roundto double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuecount (rastertable text, rastercolumn text, nband integer, searchvalue double precision, roundto double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuecount (rastertable text, rastercolumn text, searchvalue double precision, roundto double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuepercent (	rastertable text, rastercolumn text,	nband integer ,	exclude_nodata_value boolean ,	searchvalues double precision[] ,	roundto double precision ,	OUT value double precision, OUT percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuepercent (rastertable text, rastercolumn text, nband integer, searchvalues double precision[], roundto double precision , OUT value double precision, OUT percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuepercent (rastertable text, rastercolumn text, searchvalues double precision[], roundto double precision , OUT value double precision, OUT percent double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuepercent (rastertable text, rastercolumn text, nband integer, exclude_nodata_value boolean, searchvalue double precision, roundto double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuepercent (rastertable text, rastercolumn text, nband integer, searchvalue double precision, roundto double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_valuepercent (rastertable text, rastercolumn text, searchvalue double precision, roundto double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_reclass (rast raster, VARIADIC reclassargset reclassarg[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_reclass (rast raster, VARIADIC reclassargset reclassarg[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_reclass (rast raster, nband int, reclassexpr text, pixeltype text, nodataval double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_reclass (rast raster, reclassexpr text, pixeltype text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_colormap (	rast raster, nband int,	colormap text,	method text );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_colormap (	rast raster, nband int ,	colormap text ,	method text );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_colormap (	rast raster,	colormap text,	method text );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_fromgdalraster (gdaldata bytea, srid integer );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_gdaldrivers (	OUT idx int,	OUT short_name text,	OUT long_name text,	OUT can_read boolean,	OUT can_write boolean,	OUT create_options text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_asgdalraster (rast raster, format text, options text[] , srid integer );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_astiff (rast raster, options text[] , srid integer );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_astiff (rast raster, nbands int[], options text[] , srid integer );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_astiff (rast raster, compression text, srid integer );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_astiff (rast raster, nbands int[], compression text, srid integer );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_asjpeg (rast raster, options text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_asjpeg (rast raster, nbands int[], options text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_asjpeg (rast raster, nbands int[], quality int);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_asjpeg (rast raster, nband int, options text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_asjpeg (rast raster, nband int, quality int);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_aspng (rast raster, options text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_aspng (rast raster, nbands int[], options text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_aspng (rast raster, nbands int[], compression int);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_aspng (rast raster, nband int, options text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_aspng (rast raster, nband int, compression int);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_asraster (	geom geometry,	scalex double precision , scaley double precision ,	width integer , height integer ,	pixeltype text[] ,	value double precision[] ,	nodataval double precision[] ,	upperleftx double precision , upperlefty double precision ,	gridx double precision , gridy double precision ,	skewx double precision , skewy double precision ,	touched boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_asraster (	geom geometry,	scalex double precision, scaley double precision,	gridx double precision , gridy double precision ,	pixeltype text[] ,	value double precision[] ,	nodataval double precision[] ,	skewx double precision , skewy double precision ,	touched boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_asraster (	geom geometry,	scalex double precision, scaley double precision,	pixeltype text[],	value double precision[] ,	nodataval double precision[] ,	upperleftx double precision , upperlefty double precision ,	skewx double precision , skewy double precision ,	touched boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_asraster (	geom geometry,	width integer, height integer,	gridx double precision , gridy double precision ,	pixeltype text[] ,	value double precision[] ,	nodataval double precision[] ,	skewx double precision , skewy double precision ,	touched boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_asraster (	geom geometry,	width integer, height integer,	pixeltype text[],	value double precision[] ,	nodataval double precision[] ,	upperleftx double precision , upperlefty double precision ,	skewx double precision , skewy double precision ,	touched boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_asraster (	geom geometry,	scalex double precision, scaley double precision,	gridx double precision, gridy double precision,	pixeltype text,	value double precision ,	nodataval double precision ,	skewx double precision , skewy double precision ,	touched boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_asraster (	geom geometry,	scalex double precision, scaley double precision,	pixeltype text,	value double precision ,	nodataval double precision ,	upperleftx double precision , upperlefty double precision ,	skewx double precision , skewy double precision ,	touched boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_asraster (	geom geometry,	width integer, height integer,	gridx double precision, gridy double precision,	pixeltype text,	value double precision ,	nodataval double precision ,	skewx double precision , skewy double precision ,	touched boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_asraster (	geom geometry,	width integer, height integer,	pixeltype text,	value double precision ,	nodataval double precision ,	upperleftx double precision , upperlefty double precision ,	skewx double precision , skewy double precision ,	touched boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_asraster (	geom geometry,	ref raster,	pixeltype text[] ,	value double precision[] ,	nodataval double precision[] ,	touched boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_asraster (	geom geometry,	ref raster,	pixeltype text,	value double precision ,	nodataval double precision ,	touched boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _ST_gdalwarp (	rast raster,	algorithm text , maxerr double precision ,	srid integer ,	scalex double precision , scaley double precision ,	gridx double precision , gridy double precision ,	skewx double precision , skewy double precision ,	width integer , height integer );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_resample (	rast raster,	scalex double precision , scaley double precision ,	gridx double precision , gridy double precision ,	skewx double precision , skewy double precision ,	algorithm text , maxerr double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_resample (	rast raster,	width integer, height integer,	gridx double precision , gridy double precision ,	skewx double precision , skewy double precision ,	algorithm text , maxerr double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_resample (	rast raster,	ref raster,	algorithm text ,	maxerr double precision ,	usescale boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_resample (	rast raster,	ref raster,	usescale boolean,	algorithm text ,	maxerr double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_transform (rast raster, srid integer, algorithm text , maxerr double precision , scalex double precision , scaley double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_transform (rast raster, srid integer, scalex double precision, scaley double precision, algorithm text , maxerr double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_transform (rast raster, srid integer, scalexy double precision, algorithm text , maxerr double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_transform (	rast raster,	alignto raster,	algorithm text , maxerr double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_rescale (rast raster, scalex double precision, scaley double precision, algorithm text , maxerr double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_rescale (rast raster, scalexy double precision, algorithm text , maxerr double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_reskew (rast raster, skewx double precision, skewy double precision, algorithm text , maxerr double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_reskew (rast raster, skewxy double precision, algorithm text , maxerr double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_snaptogrid (	rast raster,	gridx double precision, gridy double precision,	algorithm text , maxerr double precision ,	scalex double precision , scaley double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_snaptogrid (	rast raster,	gridx double precision, gridy double precision,	scalex double precision, scaley double precision,	algorithm text , maxerr double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_snaptogrid (	rast raster,	gridx double precision, gridy double precision,	scalexy double precision,	algorithm text , maxerr double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_resize (	rast raster,	width text, height text,	algorithm text , maxerr double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_resize (	rast raster,	width integer, height integer,	algorithm text , maxerr double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_resize (	rast raster,	percentwidth double precision, percentheight double precision,	algorithm text , maxerr double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebraexpr (rast raster, band integer, pixeltype text,        expression text, nodataval double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebraexpr (rast raster, pixeltype text, expression text,        nodataval double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebrafct (rast raster, band integer,        pixeltype text, onerastuserfunc regprocedure, variadic args text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebrafct (rast raster, band integer,        pixeltype text, onerastuserfunc regprocedure);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebrafct (rast raster, band integer,        onerastuserfunc regprocedure, variadic args text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebrafct (rast raster, band integer,        onerastuserfunc regprocedure);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebrafct (rast raster, pixeltype text,        onerastuserfunc regprocedure, variadic args text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebrafct (rast raster, pixeltype text,        onerastuserfunc regprocedure);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebrafct (rast raster, onerastuserfunc regprocedure,        variadic args text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebrafct (rast raster, onerastuserfunc regprocedure);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebraexpr (	rast1 raster, band1 integer,	rast2 raster, band2 integer,	expression text,	pixeltype text , extenttype text ,	nodata1expr text , nodata2expr text ,	nodatanodataval double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebraexpr (	rast1 raster,	rast2 raster,	expression text,	pixeltype text , extenttype text ,	nodata1expr text , nodata2expr text ,	nodatanodataval double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebrafct (	rast1 raster, band1 integer,	rast2 raster, band2 integer,	tworastuserfunc regprocedure,	pixeltype text , extenttype text ,	VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebrafct (	rast1 raster,	rast2 raster,	tworastuserfunc regprocedure,	pixeltype text , extenttype text ,	VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebrafctngb (    rast raster,    band integer,    pixeltype text,    ngbwidth integer,    ngbheight integer,    onerastngbuserfunc regprocedure,    nodatamode text,    variadic args text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_max4ma (matrix float[][], nodatamode text, variadic args text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_min4ma (matrix float[][], nodatamode text, variadic args text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_sum4ma (matrix float[][], nodatamode text, variadic args text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mean4ma (matrix float[][], nodatamode text, variadic args text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_range4ma (matrix float[][], nodatamode text, variadic args text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_distinct4ma (matrix float[][], nodatamode TEXT, VARIADIC args TEXT[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_stddev4ma (matrix float[][], nodatamode TEXT, VARIADIC args TEXT[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_mapalgebra (	rastbandargset rastbandarg[],	callbackfunc regprocedure,	pixeltype text ,	distancex integer , distancey integer ,	extenttype text , customextent raster ,	mask double precision[][] , weighted boolean ,	VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebra (	rastbandargset rastbandarg[],	callbackfunc regprocedure,	pixeltype text ,	extenttype text , customextent raster ,	distancex integer , distancey integer ,	VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebra (	rast raster, nband int[],	callbackfunc regprocedure,	pixeltype text ,	extenttype text , customextent raster ,	distancex integer , distancey integer ,	VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebra (	rast raster, nband int,	callbackfunc regprocedure,	pixeltype text ,	extenttype text , customextent raster ,	distancex integer , distancey integer ,	VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebra (	rast1 raster, nband1 int,	rast2 raster, nband2 int,	callbackfunc regprocedure,	pixeltype text ,	extenttype text , customextent raster ,	distancex integer , distancey integer ,	VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebra (	rast raster, nband int,	callbackfunc regprocedure,	mask double precision [][], weighted boolean,	pixeltype text ,	extenttype text , customextent raster ,	VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_mapalgebra (	rastbandargset rastbandarg[],	expression text,	pixeltype text , extenttype text ,	nodata1expr text , nodata2expr text ,	nodatanodataval double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebra (	rast raster, nband integer,	pixeltype text,	expression text, nodataval double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebra (	rast raster,	pixeltype text,	expression text, nodataval double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebra (	rast1 raster, band1 integer,	rast2 raster, band2 integer,	expression text,	pixeltype text , extenttype text ,	nodata1expr text , nodata2expr text ,	nodatanodataval double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mapalgebra (	rast1 raster,	rast2 raster,	expression text,	pixeltype text , extenttype text ,	nodata1expr text , nodata2expr text ,	nodatanodataval double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_convertarray4ma (value double precision[][]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_max4ma (value double precision[][][], pos integer[][], VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_min4ma (value double precision[][][], pos integer[][], VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_sum4ma (value double precision[][][], pos integer[][], VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mean4ma (value double precision[][][], pos integer[][], VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_range4ma (value double precision[][][], pos integer[][], VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_distinct4ma (value double precision[][][], pos integer[][], VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_stddev4ma (value double precision[][][], pos integer[][], VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_invdistweight4ma (value double precision[][][], pos integer[][], VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_mindist4ma (value double precision[][][], pos integer[][], VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_slope4ma (value double precision[][][], pos integer[][], VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_slope (	rast raster, nband integer,	customextent raster,	pixeltype text , units text ,	scale double precision ,	interpolate_nodata boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_slope (	rast raster, nband integer ,	pixeltype text , units text ,	scale double precision ,	interpolate_nodata boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_aspect4ma (value double precision[][][], pos integer[][], VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_aspect (	rast raster, nband integer,	customextent raster,	pixeltype text , units text ,	interpolate_nodata boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_aspect (	rast raster, nband integer ,	pixeltype text , units text ,	interpolate_nodata boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_hillshade4ma (value double precision[][][], pos integer[][], VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_hillshade (	rast raster, nband integer,	customextent raster,	pixeltype text ,	azimuth double precision , altitude double precision ,	max_bright double precision , scale double precision ,	interpolate_nodata boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_hillshade (	rast raster, nband integer ,	pixeltype text ,	azimuth double precision , altitude double precision ,	max_bright double precision , scale double precision ,	interpolate_nodata boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_tpi4ma (value double precision[][][], pos integer[][], VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_tpi (	rast raster, nband integer,	customextent raster,	pixeltype text , interpolate_nodata boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_tpi (	rast raster, nband integer ,	pixeltype text , interpolate_nodata boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_roughness4ma (value double precision[][][], pos integer[][], VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_roughness (	rast raster, nband integer,	customextent raster,	pixeltype text , interpolate_nodata boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_roughness (	rast raster, nband integer ,	pixeltype text , interpolate_nodata boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_tri4ma (value double precision[][][], pos integer[][], VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_tri (	rast raster, nband integer,	customextent raster,	pixeltype text ,	interpolate_nodata boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_tri (	rast raster, nband integer ,	pixeltype text , interpolate_nodata boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_grayscale4ma (value double precision[][][], pos integer[][], VARIADIC userargs text[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_grayscale (	rastbandargset rastbandarg[],	extenttype text );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_grayscale (	rast raster, 	redband integer , 	greenband integer , 	blueband integer ,	extenttype text );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_isempty (rast raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_hasnoband (rast raster, nband int );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_bandnodatavalue (rast raster, band integer );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_bandisnodata (rast raster, band integer , forceChecking boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_bandisnodata (rast raster, forceChecking boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_bandpath (rast raster, band integer );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_BandPixelType (rast raster, band integer );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_BandMetaData (	rast raster,	band int[],	OUT bandnum int,	OUT pixeltype text,	OUT nodatavalue double precision,	OUT isoutdb boolean,	OUT path text,	OUT outdbbandnum integer,	OUT filesize bigint,	OUT filetimestamp bigint);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_BandMetaData (	rast raster,	band int ,	OUT pixeltype text,	OUT nodatavalue double precision,	OUT isoutdb boolean,	OUT path text,	OUT outdbbandnum integer,	OUT filesize bigint,	OUT filetimestamp bigint);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_BandFileSize (rast raster, band integer );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION St_BandFileTimestamp (rast raster, band integer );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_value (rast raster, band integer, x integer, y integer, exclude_nodata_value boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_value (rast raster, x integer, y integer, exclude_nodata_value boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_value (rast raster, band integer, pt geometry, exclude_nodata_value boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_Value (rast raster, pt geometry, exclude_nodata_value boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_pixelofvalue (	rast raster,	nband integer,	search double precision[],	exclude_nodata_value boolean ,	OUT val double precision,	OUT x integer,	OUT y integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_PixelofValue (	rast raster,	search double precision[],	exclude_nodata_value boolean ,	OUT val double precision,	OUT x integer,	OUT y integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_pixelofvalue (	rast raster,	nband integer,	search double precision,	exclude_nodata_value boolean ,	OUT x integer,	OUT y integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_pixelofvalue (	rast raster,	search double precision,	exclude_nodata_value boolean ,	OUT x integer,	OUT y integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_georeference (rast raster, format text );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_setscale (rast raster, scale float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_setscale (rast raster, scalex float8, scaley float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_setskew (rast raster, skew float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_setskew (rast raster, skewx float8, skewy float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_setsrid (rast raster, srid integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_setupperleft (rast raster, upperleftx float8, upperlefty float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_setrotation (rast raster, rotation float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_setgeotransform (rast raster,    imag double precision,    jmag double precision,    theta_i double precision,    theta_ij double precision,    xoffset double precision,    yoffset double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_setgeoreference (rast raster, georef text, format text );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_setgeoreference (	rast raster,	upperleftx double precision, upperlefty double precision,	scalex double precision, scaley double precision,	skewx double precision, skewy double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_tile (	rast raster,	width integer, height integer,	nband integer[] ,	padwithnodata boolean , nodataval double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_tile (	rast raster, nband integer[],	width integer, height integer,	padwithnodata boolean , nodataval double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_tile (	rast raster, nband integer,	width integer, height integer,	padwithnodata boolean , nodataval double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_tile (	rast raster,	width integer, height integer,	padwithnodata boolean , nodataval double precision );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_setbandnodatavalue (rast raster, band integer, nodatavalue float8, forceChecking boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_setbandnodatavalue (rast raster, nodatavalue float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_setbandisnodata (rast raster, band integer );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_setbandpath (rast raster, band integer, outdbpath text, outdbindex integer, force boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_setbandindex (rast raster, band integer, outdbindex integer, force boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _ST_SetValues (	rast raster, nband integer,	x integer, y integer,	newvalueset double precision[][],	noset boolean[][] ,	hasnosetvalue boolean ,	nosetvalue double precision ,	keepnodata boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_SetValues (	rast raster, nband integer,	x integer, y integer,	newvalueset double precision[][],	noset boolean[][] ,	keepnodata boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_SetValues (	rast raster, nband integer,	x integer, y integer,	newvalueset double precision[][],	nosetvalue double precision,	keepnodata boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_SetValues (	rast raster, nband integer,	x integer, y integer,	width integer, height integer,	newvalue double precision,	keepnodata boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_SetValues (	rast raster,	x integer, y integer,	width integer, height integer,	newvalue double precision,	keepnodata boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_SetValues (	rast raster, nband integer,	geomvalset geomval[],	keepnodata boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_SetValue (rast raster, band integer, x integer, y integer, newvalue float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_SetValue (rast raster, x integer, y integer, newvalue float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_SetValue (	rast raster, nband integer,	geom geometry, newvalue double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_SetValue (	rast raster,	geom geometry, newvalue double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_dumpaspolygons (rast raster, band integer , exclude_nodata_value boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_dumpvalues (	rast raster, nband integer[] , exclude_nodata_value boolean ,	OUT nband integer, OUT valarray double precision[][]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_dumpvalues (rast raster, nband integer, exclude_nodata_value boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_polygon (rast raster, band integer );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_pixelaspolygons (	rast raster,	band integer ,	columnx integer ,	rowy integer ,	exclude_nodata_value boolean ,	OUT geom geometry,	OUT val double precision,	OUT x integer,	OUT y integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_pixelaspolygons (	rast raster,	band integer ,	exclude_nodata_value boolean ,	OUT geom geometry,	OUT val double precision,	OUT x int,	OUT y int);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_pixelaspolygon (rast raster, x integer, y integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_pixelaspoints (	rast raster,	band integer ,	exclude_nodata_value boolean ,	OUT geom geometry,	OUT val double precision,	OUT x int,	OUT y int);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_pixelaspoint (rast raster, x integer, y integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_pixelascentroids (	rast raster,	band integer ,	exclude_nodata_value boolean ,	OUT geom geometry,	OUT val double precision,	OUT x int,	OUT y int);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_pixelascentroid (rast raster, x integer, y integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_worldtorastercoord (	rast raster,	longitude double precision , latitude double precision ,	OUT columnx integer,	OUT rowy integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_worldtorastercoord (	rast raster,	longitude double precision, latitude double precision,	OUT columnx integer,	OUT rowy integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_worldtorastercoord (	rast raster, pt geometry,	OUT columnx integer,	OUT rowy integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_worldtorastercoordx (rast raster, xw float8, yw float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_worldtorastercoordx (rast raster, xw float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_worldtorastercoordx (rast raster, pt geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_worldtorastercoordy (rast raster, xw float8, yw float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_worldtorastercoordy (rast raster, yw float8);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_worldtorastercoordy (rast raster, pt geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_rastertoworldcoord (	rast raster,	columnx integer , rowy integer ,	OUT longitude double precision,	OUT latitude double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_rastertoworldcoord (	rast raster,	columnx integer, rowy integer,	OUT longitude double precision,	OUT latitude double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_rastertoworldcoordx (rast raster, xr int, yr int);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_rastertoworldcoordx (rast raster, xr int);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_rastertoworldcoordy (rast raster, xr int, yr int);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_rastertoworldcoordy (rast raster, yr int);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_minpossiblevalue (pixeltype text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_rastfromwkb (bytea);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_rastfromhexwkb (text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_aswkb (raster, outasin boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_asbinary (raster, outasin boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_ashexwkb (raster, outasin boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION bytea (raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION raster_hash (raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION raster_eq (raster, raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION raster_overleft (raster, raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION raster_overright (raster, raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION raster_left (raster, raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION raster_right (raster, raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION raster_overabove (raster, raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION raster_overbelow (raster, raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION raster_above (raster, raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION raster_below (raster, raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION raster_same (raster, raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION raster_contained (raster, raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION raster_contain (raster, raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION raster_overlap (raster, raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION raster_geometry_contain (raster, geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION raster_contained_by_geometry (raster, geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION raster_geometry_overlap (raster, geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION geometry_raster_contain (geometry, raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION geometry_contained_by_raster (geometry, raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION geometry_raster_overlap (geometry, raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_samealignment (rast1 raster, rast2 raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_samealignment (	ulx1 double precision, uly1 double precision, scalex1 double precision, scaley1 double precision, skewx1 double precision, skewy1 double precision,	ulx2 double precision, uly2 double precision, scalex2 double precision, scaley2 double precision, skewx2 double precision, skewy2 double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_samealignment_transfn (agg agg_samealignment, rast raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _ST_samealignment_finalfn (agg agg_samealignment);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_notsamealignmentreason (rast1 raster, rast2 raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_iscoveragetile (rast raster, coverage raster, tilewidth integer, tileheight integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_intersects (rast1 raster, nband1 integer, rast2 raster, nband2 integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersects (rast1 raster, nband1 integer, rast2 raster, nband2 integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersects (rast1 raster, rast2 raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_intersects (geom geometry, rast raster, nband integer );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersects (geom geometry, rast raster, nband integer );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersects (rast raster, geom geometry, nband integer );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersects (rast raster, nband integer, geom geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_overlaps (rast1 raster, nband1 integer, rast2 raster, nband2 integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_overlaps (rast1 raster, nband1 integer, rast2 raster, nband2 integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_overlaps (rast1 raster, rast2 raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_touches (rast1 raster, nband1 integer, rast2 raster, nband2 integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_touches (rast1 raster, nband1 integer, rast2 raster, nband2 integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_touches (rast1 raster, rast2 raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_contains (rast1 raster, nband1 integer, rast2 raster, nband2 integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_contains (rast1 raster, nband1 integer, rast2 raster, nband2 integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_contains (rast1 raster, rast2 raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_containsproperly (rast1 raster, nband1 integer, rast2 raster, nband2 integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_containsproperly (rast1 raster, nband1 integer, rast2 raster, nband2 integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_containsproperly (rast1 raster, rast2 raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_covers (rast1 raster, nband1 integer, rast2 raster, nband2 integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_covers (rast1 raster, nband1 integer, rast2 raster, nband2 integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_covers (rast1 raster, rast2 raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_coveredby (rast1 raster, nband1 integer, rast2 raster, nband2 integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_coveredby (rast1 raster, nband1 integer, rast2 raster, nband2 integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_coveredby (rast1 raster, rast2 raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_within (rast1 raster, nband1 integer, rast2 raster, nband2 integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_within (rast1 raster, nband1 integer, rast2 raster, nband2 integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_within (rast1 raster, rast2 raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _ST_DWithin (rast1 raster, nband1 integer, rast2 raster, nband2 integer, distance double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_DWithin (rast1 raster, nband1 integer, rast2 raster, nband2 integer, distance double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_DWithin (rast1 raster, rast2 raster, distance double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _ST_DFullyWithin (rast1 raster, nband1 integer, rast2 raster, nband2 integer, distance double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_DFullyWithin (rast1 raster, nband1 integer, rast2 raster, nband2 integer, distance double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_DFullyWithin (rast1 raster, rast2 raster, distance double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_disjoint (rast1 raster, nband1 integer, rast2 raster, nband2 integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_Disjoint (rast1 raster, rast2 raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_Intersection (geomin geometry, rast raster, band integer );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersection (rast raster, band integer, geomin geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_Intersection (rast raster, geomin geometry);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_Intersection (	rast1 raster, band1 int,	rast2 raster, band2 int,	returnband text ,	nodataval double precision[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersection (	rast1 raster, band1 int,	rast2 raster, band2 int,	returnband text,	nodataval double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersection (	rast1 raster, band1 int,	rast2 raster, band2 int,	nodataval double precision[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersection (	rast1 raster, band1 int,	rast2 raster, band2 int,	nodataval double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersection (	rast1 raster,	rast2 raster,	returnband text ,	nodataval double precision[] );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersection (	rast1 raster,	rast2 raster,	returnband text,	nodataval double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersection (	rast1 raster,	rast2 raster,	nodataval double precision[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_intersection (	rast1 raster,	rast2 raster,	nodataval double precision);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_union_finalfn (internal);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_union_transfn (internal, raster, unionarg[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_union_transfn (internal, raster, integer, text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_union_transfn (internal, raster, integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_union_transfn (internal, raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_union_transfn (internal, raster, text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_clip (	rast raster, nband integer[],	geom geometry,	nodataval double precision[] , crop boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_clip (	rast raster, nband integer[],	geom geometry,	nodataval double precision[] , crop boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_clip (	rast raster, nband integer,	geom geometry,	nodataval double precision, crop boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_clip (	rast raster, nband integer,	geom geometry,	crop boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_clip (	rast raster,	geom geometry,	nodataval double precision[] , crop boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_clip (	rast raster,	geom geometry,	nodataval double precision, crop boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_clip (	rast raster,	geom geometry,	crop boolean);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_nearestvalue (	rast raster, band integer,	pt geometry,	exclude_nodata_value boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_nearestvalue (	rast raster,	pt geometry,	exclude_nodata_value boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_nearestvalue (	rast raster, band integer,	columnx integer, rowy integer,	exclude_nodata_value boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_nearestvalue (	rast raster,	columnx integer, rowy integer,	exclude_nodata_value boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _st_neighborhood (	rast raster, band integer,	columnx integer, rowy integer,	distancex integer, distancey integer,	exclude_nodata_value boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_neighborhood (	rast raster, band integer,	columnx integer, rowy integer,	distancex integer, distancey integer,	exclude_nodata_value boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_neighborhood (	rast raster,	columnx integer, rowy integer,	distancex integer, distancey integer,	exclude_nodata_value boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_neighborhood (	rast raster, band integer,	pt geometry,	distancex integer, distancey integer,	exclude_nodata_value boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_neighborhood (	rast raster,	pt geometry,	distancex integer, distancey integer,	exclude_nodata_value boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _add_raster_constraint (cn name, sql text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _drop_raster_constraint (rastschema name, rasttable name, cn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _raster_constraint_info_srid (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _add_raster_constraint_srid (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _drop_raster_constraint_srid (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _raster_constraint_info_scale (rastschema name, rasttable name, rastcolumn name, axis char);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _add_raster_constraint_scale (rastschema name, rasttable name, rastcolumn name, axis char);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _drop_raster_constraint_scale (rastschema name, rasttable name, rastcolumn name, axis char);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _raster_constraint_info_blocksize (rastschema name, rasttable name, rastcolumn name, axis text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _add_raster_constraint_blocksize (rastschema name, rasttable name, rastcolumn name, axis text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _drop_raster_constraint_blocksize (rastschema name, rasttable name, rastcolumn name, axis text);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _raster_constraint_info_extent (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _add_raster_constraint_extent (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _drop_raster_constraint_extent (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _raster_constraint_info_alignment (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _add_raster_constraint_alignment (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _drop_raster_constraint_alignment (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _raster_constraint_info_spatially_unique (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _add_raster_constraint_spatially_unique (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _drop_raster_constraint_spatially_unique (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _raster_constraint_info_coverage_tile (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _add_raster_constraint_coverage_tile (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _drop_raster_constraint_coverage_tile (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _raster_constraint_info_regular_blocking (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _drop_raster_constraint_regular_blocking (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _raster_constraint_info_num_bands (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _add_raster_constraint_num_bands (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _drop_raster_constraint_num_bands (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _raster_constraint_info_pixel_types (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _raster_constraint_pixel_types (rast raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _add_raster_constraint_pixel_types (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _drop_raster_constraint_pixel_types (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _raster_constraint_info_nodata_values (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _raster_constraint_nodata_values (rast raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _add_raster_constraint_nodata_values (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _drop_raster_constraint_nodata_values (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _raster_constraint_info_out_db (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _raster_constraint_out_db (rast raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _add_raster_constraint_out_db (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _drop_raster_constraint_out_db (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _raster_constraint_info_index (rastschema name, rasttable name, rastcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION AddRasterConstraints  (	rastschema name,	rasttable name,	rastcolumn name,	VARIADIC constraints text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION AddRasterConstraints  (	rasttable name,	rastcolumn name,	VARIADIC constraints text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION AddRasterConstraints  (	rastschema name,	rasttable name,	rastcolumn name,	srid boolean ,	scale_x boolean ,	scale_y boolean ,	blocksize_x boolean ,	blocksize_y boolean ,	same_alignment boolean ,	regular_blocking boolean , 	num_bands boolean ,	pixel_types boolean ,	nodata_values boolean ,	out_db boolean ,	extent boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION AddRasterConstraints  (	rasttable name,	rastcolumn name,	srid boolean ,	scale_x boolean ,	scale_y boolean ,	blocksize_x boolean ,	blocksize_y boolean ,	same_alignment boolean ,	regular_blocking boolean , 	num_bands boolean ,	pixel_types boolean ,	nodata_values boolean ,	out_db boolean ,	extent boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION DropRasterConstraints  (	rastschema name,	rasttable name,	rastcolumn name,	VARIADIC constraints text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION DropRasterConstraints  (	rasttable name,	rastcolumn name,	VARIADIC constraints text[]);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION DropRasterConstraints  (	rastschema name,	rasttable name,	rastcolumn name,	srid boolean ,	scale_x boolean ,	scale_y boolean ,	blocksize_x boolean ,	blocksize_y boolean ,	same_alignment boolean ,	regular_blocking boolean ,	num_bands boolean ,	pixel_types boolean ,	nodata_values boolean ,	out_db boolean ,	extent boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION DropRasterConstraints  (	rasttable name,	rastcolumn name,	srid boolean ,	scale_x boolean ,	scale_y boolean ,	blocksize_x boolean ,	blocksize_y boolean ,	same_alignment boolean ,	regular_blocking boolean ,	num_bands boolean ,	pixel_types boolean ,	nodata_values boolean ,	out_db boolean ,	extent boolean );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _overview_constraint (ov raster, factor integer, refschema name, reftable name, refcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _overview_constraint_info (	ovschema name, ovtable name, ovcolumn name,	OUT refschema name, OUT reftable name, OUT refcolumn name, OUT factor integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _add_overview_constraint (	ovschema name, ovtable name, ovcolumn name,	refschema name, reftable name, refcolumn name,	factor integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _drop_overview_constraint (ovschema name, ovtable name, ovcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION AddOverviewConstraints  (	ovschema name, ovtable name, ovcolumn name,	refschema name, reftable name, refcolumn name,	ovfactor int);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION AddOverviewConstraints  (	ovtable name, ovcolumn name,	reftable name, refcolumn name,	ovfactor int);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION DropOverviewConstraints  (	ovschema name,	ovtable name,	ovcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION DropOverviewConstraints  (	ovtable name,	ovcolumn name);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION _UpdateRasterSRID (	schema_name name, table_name name, column_name name,	new_srid integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION UpdateRasterSRID (	schema_name name, table_name name, column_name name,	new_srid integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION UpdateRasterSRID (	table_name name, column_name name,	new_srid integer);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_Retile (tab regclass, col name, ext geometry, sfx float8, sfy float8, tw int, th int, algo text );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION ST_CreateOverview (tab regclass, col name, factor int, algo text );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION st_makeemptycoverage (tilewidth int, tileheight int, width int, height int, upperleftx float8, upperlefty float8, scalex float8, scaley float8, skewx float8, skewy float8, srid int4 );EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION postgis_noop (raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP TYPE raster;EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP TYPE rastbandarg;EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP TYPE geomval;EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP TYPE addbandarg;EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP TYPE summarystats;EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP TYPE agg_count;EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP TYPE reclassarg;EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP TYPE agg_samealignment;EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP TYPE unionarg;EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION raster_in (cstring);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
DO $$ BEGIN ALTER EXTENSION postgis DROP FUNCTION raster_out (raster);EXCEPTION WHEN object_not_in_prerequisite_state OR undefined_function OR undefined_object THEN  RAISE NOTICE '%', SQLERRM; WHEN OTHERS THEN  RAISE EXCEPTION 'Got % - %', SQLSTATE, SQLERRM; END $$ LANGUAGE 'plpgsql';
		$unpackage$;

		RAISE WARNING 'PostGIS Raster functionality has been unpackaged'
		USING HINT = 'type `SELECT postgis_extensions_upgrade();` to finish the upgrade.'
						' After upgrading, if you want to drop raster, run: DROP EXTENSION postgis_raster;';
	END IF;
END
$unpackage_if_needed$ LANGUAGE 'plpgsql';

-- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
--
--
-- PostGIS - Spatial Types for PostgreSQL
-- http://postgis.net
--
-- Copyright (C) 2011-2012 Sandro Santilli <strk@kbt.io>
-- Copyright (C) 2010-2013 Regina Obe <lr@pcorp.us>
-- Copyright (C) 2009      Paul Ramsey <pramsey@cleverelephant.ca>
--
-- This is free software; you can redistribute and/or modify it under
-- the terms of the GNU General Public Licence. See the COPYING file.
--
-- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
--
-- This file contains drop commands for obsoleted items that need
-- to be dropped _before_ upgrade of old functions.
-- Changes to this file affect postgis_upgrade*.sql script.
--
-- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -


-- Helper function to drop functions when they match the full signature
-- Requires schema, name and __exact arguments__ as extracted from pg_catalog.pg_get_function_arguments
-- You can extract the old function arguments using a query like:
-- SELECT  p.oid as oid,
--                 n.nspname as schema,
--                 p.proname as name,
--                 pg_catalog.pg_get_function_arguments(p.oid) as arguments
--         FROM pg_catalog.pg_proc p
--         LEFT JOIN pg_catalog.pg_namespace n ON n.oid = p.pronamespace
--         WHERE
--                 LOWER(n.nspname) = LOWER('public') AND
--                 LOWER(p.proname) = LOWER('ST_AsGeoJson')
--         ORDER BY 1, 2, 3, 4;
CREATE OR REPLACE FUNCTION _postgis_drop_function_if_needed(
	function_name text,
	function_arguments text) RETURNS void AS $$
DECLARE
	frec RECORD;
	sql_drop text;
BEGIN
	FOR frec IN
		SELECT  p.oid as oid,
				n.nspname as schema,
				n.oid as schema_oid,
				p.proname as name,
				pg_catalog.pg_get_function_arguments(p.oid) as arguments,
				pg_catalog.pg_get_function_identity_arguments(p.oid) as identity_arguments
			FROM pg_catalog.pg_proc p
			LEFT JOIN pg_catalog.pg_namespace n ON n.oid = p.pronamespace
			WHERE
				n.oid = (
					SELECT n.oid
					FROM pg_proc p
					JOIN pg_namespace n ON p.pronamespace = n.oid
					WHERE proname = 'postgis_full_version'
					) AND
				LOWER(p.proname) = LOWER(function_name) AND
				LOWER(pg_catalog.pg_get_function_arguments(p.oid)) ~ LOWER(function_arguments) AND
				pg_catalog.pg_function_is_visible(p.oid)
			ORDER BY 1, 2, 4
	LOOP
		sql_drop := 'DROP FUNCTION ' || quote_ident(frec.schema) || '.' || quote_ident(frec.name) || ' ( ' || frec.identity_arguments || ' ) ';
		RAISE DEBUG 'Name (%): %', frec.oid, frec.name;
		RAISE DEBUG 'Arguments: %', frec.arguments;
		RAISE DEBUG 'Identity arguments: %', frec.identity_arguments;
		RAISE DEBUG 'SQL query: %', sql_drop;
		BEGIN
			EXECUTE sql_drop;
		EXCEPTION
			WHEN OTHERS THEN
				RAISE EXCEPTION 'Could not drop function %. You might need to drop dependant objects. Postgres error: %', function_name, SQLERRM;
		END;
	END LOOP;
END;
$$ LANGUAGE plpgsql;


-- FUNCTION AddGeometryColumn signature dropped
-- (catalog_name character varying, schema_name character varying, table_name character varying, column_name character varying, new_srid integer, new_type character varying, new_dim integer, use_typmod boolean)
SELECT _postgis_drop_function_if_needed
	(
	'AddGeometryColumn',
	'catalog_name character varying, schema_name character varying, table_name character varying, column_name character varying, new_srid integer, new_type character varying, new_dim integer, use_typmod boolean'
	);

-- FUNCTION ST_AsX3D was changed to add versioning for 2.0
-- (geom geometry, prec integer, options integer)
SELECT _postgis_drop_function_if_needed
	(
	'ST_AsX3D',
	'geom geometry, prec integer, options integer'
	);

-- FUNCTION UpdateGeometrySRID changed the name of the args (http://trac.osgeo.org/postgis/ticket/1606) for 2.0
-- It changed the paramenter `new_srid` to `new_srid_in`
-- (catalogn_name character varying, schema_name character varying, table_name character varying, column_name character varying, new_srid integer)
-- Dropping it conditionally since the same signature still exists.
SELECT _postgis_drop_function_if_needed
	(
	'UpdateGeometrySRID',
	'catalogn_name character varying, schema_name character varying, table_name character varying, column_name character varying, new_srid integer'
	);


--deprecated and removed in 2.1
-- Hack to fix 2.0 naming
-- We can't just drop it since its bound to opclass
-- See ticket 2279 for why we need to do this
-- We can get rid of this DO code when 3.0 comes along
DO  language 'plpgsql' $$
BEGIN
	-- fix geometry ops --
	IF EXISTS(SELECT oprname from pg_operator where oprname = '&&' AND oprrest::text = 'geometry_gist_sel_2d') THEN
	--it is bound to old name, drop new, rename old to new, install will fix body of code
		DROP FUNCTION IF EXISTS gserialized_gist_sel_2d(internal, oid, internal, int4) ;
		ALTER FUNCTION geometry_gist_sel_2d(internal, oid, internal, int4) RENAME TO gserialized_gist_sel_2d;
	END IF;
	IF EXISTS(SELECT oprname from pg_operator where oprname = '&&' AND oprjoin::text = 'geometry_gist_joinsel_2d') THEN
	--it is bound to old name, drop new, rename old to new,  install will fix body of code
		DROP FUNCTION IF EXISTS gserialized_gist_joinsel_2d(internal, oid, internal, smallint) ;
		ALTER FUNCTION geometry_gist_joinsel_2d(internal, oid, internal, smallint) RENAME TO gserialized_gist_joinsel_2d;
	END IF;
	-- fix geography ops --
	IF EXISTS(SELECT oprname from pg_operator where oprname = '&&' AND oprrest::text = 'geography_gist_selectivity') THEN
	--it is bound to old name, drop new, rename old to new, install will fix body of code
		DROP FUNCTION IF EXISTS gserialized_gist_sel_nd(internal, oid, internal, int4) ;
		ALTER FUNCTION geography_gist_selectivity(internal, oid, internal, int4) RENAME TO gserialized_gist_sel_nd;
	END IF;

	IF EXISTS(SELECT oprname from pg_operator where oprname = '&&' AND oprjoin::text = 'geography_gist_join_selectivity') THEN
	--it is bound to old name, drop new, rename old to new, install will fix body of code
		DROP FUNCTION IF EXISTS gserialized_gist_joinsel_nd(internal, oid, internal, smallint) ;
		ALTER FUNCTION geography_gist_join_selectivity(internal, oid, internal, smallint) RENAME TO gserialized_gist_joinsel_nd;
	END IF;
END;
$$ ;


-- FUNCTION ST_AsLatLonText went from multiple signatures to a single one with defaults for 2.2.0
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_AsLatLonText(geometry)');
DROP FUNCTION IF EXISTS ST_AsLatLonText(geometry); -- Does not conflict

SELECT _postgis_drop_function_if_needed
	(
	'ST_AsLatLonText',
	'geometry, text'
	);

-- FUNCTION ST_LineCrossingDirection changed argument names in 3.0
-- Was (geom1 geometry, geom2 geometry) and now (line1 geometry, line2 geometry)
SELECT _postgis_drop_function_if_needed
	(
	'ST_LineCrossingDirection',
	'geom1 geometry, geom2 geometry'
	);

-- FUNCTION _st_linecrossingdirection changed argument names in 3.0
-- Was (geom1 geometry, geom2 geometry) and now (line1 geometry, line2 geometry)
SELECT _postgis_drop_function_if_needed
	(
	'_ST_LineCrossingDirection',
	'geom1 geometry, geom2 geometry'
	);

-- FUNCTION ST_AsGeoJson changed argument names
-- (pretty_print => pretty_bool) in 3.0alpha4
SELECT _postgis_drop_function_if_needed
	(
	'ST_AsGeoJson',
	$args$r record, geom_column text DEFAULT ''::text, maxdecimaldigits integer DEFAULT 15, pretty_print boolean DEFAULT false$args$
	);

-- FUNCTION _st_orderingequals changed argument names in 3.0
-- Was (GeometryA geometry, GeometryB geometry) and now (geom1 geometry, geom2 geometry)
SELECT _postgis_drop_function_if_needed
	(
	'_st_orderingequals',
	'GeometryA geometry, GeometryB geometry'
	);

-- FUNCTION st_orderingequals changed argument names in 3.0
-- Was (GeometryA geometry, GeometryB geometry) and now (geom1 geometry, geom2 geometry)
SELECT _postgis_drop_function_if_needed
	(
	'st_orderingequals',
	'GeometryA geometry, GeometryB geometry'
	);

-- FUNCTION st_tileenvelope added a new default argument in 3.1
SELECT _postgis_drop_function_if_needed
    (
    'st_tileenvelope',
    'zoom integer, x integer, y integer, bounds geometry DEFAULT ''0102000020110F00000200000052107C45F81B73C152107C45F81B73C152107C45F81B734152107C45F81B7341''::geometry'
    );

-- FUNCTION ST_Force3D, ST_Force3DZ, ST_Force3DM, ST_Force4DZ changed to add defaults in 3.1
-- These signatures were superseeded
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_Force3D(geometry)');
DROP FUNCTION IF EXISTS ST_Force3D(geometry); -- Does not conflict
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_Force3DZ(geometry)');
DROP FUNCTION IF EXISTS ST_Force3DZ(geometry); -- Does not conflict
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_Force3DM(geometry)');
DROP FUNCTION IF EXISTS ST_Force3DM(geometry); -- Does not conflict
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_Force4D(geometry)');
DROP FUNCTION IF EXISTS ST_Force4D(geometry); -- Does not conflict


-- FUNCTION st_askml changed to add defaults in 3.0 / r17357
-- These signatures were superseeded
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_askml(geometry, integer)');
DROP FUNCTION IF EXISTS st_askml(geometry, integer); -- Does not conflict
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_askml(geography, integer)');
DROP FUNCTION IF EXISTS st_askml(geography, integer); -- Does not conflict


-- FUNCTION st_buffer changed to add defaults in 3.0
-- This signature was superseeded
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_buffer(geometry, double precision)');
DROP FUNCTION IF EXISTS st_buffer(geometry, double precision); -- Does not conflict

-- FUNCTION ST_CurveToLine changed to add defaults in 2.5
-- These signatures were superseeded
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_CurveToLine(geometry, integer)');
DROP FUNCTION IF EXISTS ST_CurveToLine(geometry, integer); -- Does not conflict
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_CurveToLine(geometry)');
DROP FUNCTION IF EXISTS ST_CurveToLine(geometry); -- Does not conflict

SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_intersection(geometry, geometry)');
DROP FUNCTION IF EXISTS st_intersection(geometry, geometry); -- replaced in 3.1.0 by 3 args version
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_subdivide(geometry, integer)');
DROP FUNCTION IF EXISTS st_subdivide(geometry, integer); -- replaced in 3.1.0 by 3 args version
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_difference(geometry, geometry)');
DROP FUNCTION IF EXISTS st_difference(geometry, geometry); -- replaced in 3.1.0 by 3 args version
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_symdifference(geometry, geometry)');
DROP FUNCTION IF EXISTS st_symdifference(geometry, geometry); -- replaced in 3.1.0 by 3 args version
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_unaryunion(geometry)');
DROP FUNCTION IF EXISTS st_unaryunion(geometry); -- replaced in 3.1.0 by 3 args version

-- geometry_columns changed parameter types so we verify if it needs to be dropped
-- We check the catalog to see if the view (geometry_columns) has a column
-- with name `f_table_schema` and type `character varying(256)` as it was
-- changed to type `name` in 2.2
DO  language 'plpgsql' $$
BEGIN
	IF EXISTS
		(
			WITH oids AS
			(
				SELECT c.oid as oid,
					n.nspname,
					c.relname
					FROM pg_catalog.pg_class c
					LEFT JOIN pg_catalog.pg_namespace n ON n.oid = c.relnamespace
					WHERE c.relname = 'geometry_columns' AND
						n.nspname = 'public'
					AND pg_catalog.pg_table_is_visible(c.oid)
					ORDER BY 2, 3

			),
			name_attribute AS
			(
				SELECT  a.attname as attname,
						pg_catalog.format_type(a.atttypid, a.atttypmod) as format_type
						FROM pg_catalog.pg_attribute a, oids
						WHERE a.attrelid = oids.oid AND a.attnum > 0 AND NOT a.attisdropped
						ORDER BY a.attnum
			)
			SELECT attname, format_type
			FROM name_attribute
			WHERE attname = 'f_table_schema' AND format_type = 'character varying(256)'
		)
		THEN
			DROP VIEW geometry_columns;
		END IF;
END;
$$;


-- DROP auxiliar function (created above)
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION _postgis_drop_function_if_needed(text, text)');
DROP FUNCTION _postgis_drop_function_if_needed(text, text);


--
-- UPGRADE SCRIPT TO PostGIS 3.1
--

LOAD '$libdir/postgis-3';

DO $$
DECLARE
    old_scripts text;
    new_scripts text;
    old_maj text;
    new_maj text;
BEGIN
    --
    -- This uses postgis_lib_version() rather then
    -- postgis_scripts_installed() as in 1.0 because
    -- in the 1.0 => 1.1 transition that would result
    -- in an impossible upgrade:
    --
    --   from 0.3.0 to 1.1.0
    --
    -- Next releases will still be ok as
    -- postgis_lib_version() and postgis_scripts_installed()
    -- would both return actual PostGIS release number.
    --
    BEGIN
        SELECT into old_scripts postgis_lib_version();
    EXCEPTION WHEN OTHERS THEN
        RAISE DEBUG 'Got %', SQLERRM;
        SELECT into old_scripts postgis_scripts_installed();
    END;
    SELECT into new_scripts '3.1';
    SELECT into old_maj substring(old_scripts from 1 for 1);
    SELECT into new_maj substring(new_scripts from 1 for 1);

    -- 2.x to 3.x was upgrade-compatible, see
    -- https://trac.osgeo.org/postgis/ticket/4170#comment:1
    IF new_maj = '3' AND old_maj = '2' THEN
        old_maj = '3'; -- let's pretend old major = new major
    END IF;

    IF old_maj != new_maj THEN
        RAISE EXCEPTION 'Upgrade of postgis from version % to version % requires a dump/reload. See PostGIS manual for instructions', old_scripts, new_scripts;
    END IF;
END
$$
LANGUAGE 'plpgsql';

CREATE TEMPORARY TABLE _postgis_upgrade_info AS WITH versions AS (
  SELECT '3.1'::text as upgraded,
  postgis_scripts_installed() as installed
) SELECT
  upgraded as scripts_upgraded,
  installed as scripts_installed,
  substring(upgraded from '([0-9]*)\.')::int * 100 +
  substring(upgraded from '[0-9]*\.([0-9]*)\.')::int
    as version_to_num,
  substring(installed from '([0-9]*)\.')::int * 100 +
  substring(installed from '[0-9]*\.([0-9]*)\.')::int
    as version_from_num,
  installed ~ 'dev|alpha|beta'
    as version_from_isdev
  FROM versions
;

CREATE OR REPLACE FUNCTION _postgis_deprecate(oldname text, newname text, version text)
RETURNS void AS
$$
DECLARE
  curver_text text;
BEGIN
  --
  -- Raises a NOTICE if it was deprecated in this version,
  -- a WARNING if in a previous version (only up to minor version checked)
  --
    curver_text := '3.1.1';
    IF split_part(curver_text,'.',1)::int > split_part(version,'.',1)::int OR
       ( split_part(curver_text,'.',1) = split_part(version,'.',1) AND
         split_part(curver_text,'.',2) != split_part(version,'.',2) )
    THEN
      RAISE WARNING '% signature was deprecated in %. Please use %', oldname, version, newname;
    ELSE
      RAISE DEBUG '% signature was deprecated in %. Please use %', oldname, version, newname;
    END IF;
END;
$$ LANGUAGE 'plpgsql' IMMUTABLE STRICT COST 500;
CREATE OR REPLACE FUNCTION spheroid_in(cstring)
	RETURNS spheroid
	AS '$libdir/postgis-3','ellipsoid_in'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION spheroid_out(spheroid)
	RETURNS cstring
	AS '$libdir/postgis-3','ellipsoid_out'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
-- Type spheroid -- LastUpdated: 5
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF 5 > version_from_num
     FROM _postgis_upgrade_info
  THEN
      EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE TYPE spheroid (
	alignment = double,
	internallength = 65,
	input = spheroid_in,
	output = spheroid_out
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_in(cstring)
	RETURNS geometry
	AS '$libdir/postgis-3','LWGEOM_in'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_out(geometry)
	RETURNS cstring
	AS '$libdir/postgis-3','LWGEOM_out'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_typmod_in(cstring[])
	RETURNS integer
	AS '$libdir/postgis-3','geometry_typmod_in'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_typmod_out(integer)
	RETURNS cstring
	AS '$libdir/postgis-3','postgis_typmod_out'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_analyze(internal)
	RETURNS bool
	AS '$libdir/postgis-3', 'gserialized_analyze_nd'
	LANGUAGE 'c' VOLATILE STRICT;
CREATE OR REPLACE FUNCTION geometry_recv(internal)
	RETURNS geometry
	AS '$libdir/postgis-3','LWGEOM_recv'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_send(geometry)
	RETURNS bytea
	AS '$libdir/postgis-3','LWGEOM_send'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
-- Type geometry -- LastUpdated: 1
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF 1 > version_from_num
     FROM _postgis_upgrade_info
  THEN
      EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE TYPE geometry (
	internallength = variable,
	input = geometry_in,
	output = geometry_out,
	send = geometry_send,
	receive = geometry_recv,
	typmod_in = geometry_typmod_in,
	typmod_out = geometry_typmod_out,
	delimiter = ':',
	alignment = double,
	analyze = geometry_analyze,
	storage = main
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry(geometry, integer, boolean)
	RETURNS geometry
	AS '$libdir/postgis-3','geometry_enforce_typmod'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (geometry AS geometry)');
DROP CAST IF EXISTS (geometry AS geometry);
CREATE CAST (geometry AS geometry) WITH FUNCTION geometry(geometry, integer, boolean) AS IMPLICIT;
CREATE OR REPLACE FUNCTION geometry(point)
	RETURNS geometry
	AS '$libdir/postgis-3','point_to_geometry'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION point(geometry)
	RETURNS point
	AS '$libdir/postgis-3','geometry_to_point'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry(path)
	RETURNS geometry
	AS '$libdir/postgis-3','path_to_geometry'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION path(geometry)
	RETURNS path
	AS '$libdir/postgis-3','geometry_to_path'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry(polygon)
	RETURNS geometry
	AS '$libdir/postgis-3','polygon_to_geometry'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION polygon(geometry)
	RETURNS polygon
	AS '$libdir/postgis-3','geometry_to_polygon'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (geometry AS point)');
DROP CAST IF EXISTS (geometry AS point);
CREATE CAST (geometry AS point) WITH FUNCTION point(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (point AS geometry)');
DROP CAST IF EXISTS (point AS geometry);
CREATE CAST (point AS geometry) WITH FUNCTION geometry(point);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (geometry AS path)');
DROP CAST IF EXISTS (geometry AS path);
CREATE CAST (geometry AS path) WITH FUNCTION path(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (path AS geometry)');
DROP CAST IF EXISTS (path AS geometry);
CREATE CAST (path AS geometry) WITH FUNCTION geometry(path);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (geometry AS polygon)');
DROP CAST IF EXISTS (geometry AS polygon);
CREATE CAST (geometry AS polygon) WITH FUNCTION polygon(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (polygon AS geometry)');
DROP CAST IF EXISTS (polygon AS geometry);
CREATE CAST (polygon AS geometry) WITH FUNCTION geometry(polygon);
CREATE OR REPLACE FUNCTION ST_X(geometry)
	RETURNS float8
	AS '$libdir/postgis-3','LWGEOM_x_point'
	LANGUAGE 'c' IMMUTABLE STRICT
	PARALLEL SAFE COST 1;
CREATE OR REPLACE FUNCTION ST_Y(geometry)
	RETURNS float8
	AS '$libdir/postgis-3','LWGEOM_y_point'
	LANGUAGE 'c' IMMUTABLE STRICT
	PARALLEL SAFE COST 1;
CREATE OR REPLACE FUNCTION ST_Z(geometry)
	RETURNS float8
	AS '$libdir/postgis-3','LWGEOM_z_point'
	LANGUAGE 'c' IMMUTABLE STRICT
	PARALLEL SAFE COST 1;
CREATE OR REPLACE FUNCTION ST_M(geometry)
	RETURNS float8
	AS '$libdir/postgis-3','LWGEOM_m_point'
	LANGUAGE 'c' IMMUTABLE STRICT
	PARALLEL SAFE COST 1;
CREATE OR REPLACE FUNCTION box3d_in(cstring)
	RETURNS box3d
	AS '$libdir/postgis-3', 'BOX3D_in'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION box3d_out(box3d)
	RETURNS cstring
	AS '$libdir/postgis-3', 'BOX3D_out'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
-- Type box3d -- LastUpdated: 1
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF 1 > version_from_num
     FROM _postgis_upgrade_info
  THEN
      EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE TYPE box3d (
	alignment = double,
	internallength = 52,
	input = box3d_in,
	output = box3d_out
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION box2d_in(cstring)
	RETURNS box2d
	AS '$libdir/postgis-3','BOX2D_in'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION box2d_out(box2d)
	RETURNS cstring
	AS '$libdir/postgis-3','BOX2D_out'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
-- Type box2d -- LastUpdated: 8
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF 8 > version_from_num
     FROM _postgis_upgrade_info
  THEN
      EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE TYPE box2d (
	internallength = 65,
	input = box2d_in,
	output = box2d_out,
	storage = plain
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION box2df_in(cstring)
	RETURNS box2df
	AS '$libdir/postgis-3','box2df_in'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION box2df_out(box2df)
	RETURNS cstring
	AS '$libdir/postgis-3','box2df_out'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
-- Type box2df -- LastUpdated: 200
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF 200 > version_from_num
     FROM _postgis_upgrade_info
  THEN
      EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE TYPE box2df (
	internallength = 16,
	input = box2df_in,
	output = box2df_out,
	storage = plain,
	alignment = double
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION gidx_in(cstring)
	RETURNS gidx
	AS '$libdir/postgis-3','gidx_in'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION gidx_out(gidx)
	RETURNS cstring
	AS '$libdir/postgis-3','gidx_out'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
-- Type gidx -- LastUpdated: 105
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF 105 > version_from_num
     FROM _postgis_upgrade_info
  THEN
      EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE TYPE gidx (
	internallength = variable,
	input = gidx_in,
	output = gidx_out,
	storage = plain,
	alignment = double
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_lt(geom1 geometry, geom2 geometry)
	RETURNS bool
	AS '$libdir/postgis-3', 'lwgeom_lt'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION geometry_le(geom1 geometry, geom2 geometry)
	RETURNS bool
	AS '$libdir/postgis-3', 'lwgeom_le'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION geometry_gt(geom1 geometry, geom2 geometry)
	RETURNS bool
	AS '$libdir/postgis-3', 'lwgeom_gt'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION geometry_ge(geom1 geometry, geom2 geometry)
	RETURNS bool
	AS '$libdir/postgis-3', 'lwgeom_ge'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION geometry_eq(geom1 geometry, geom2 geometry)
	RETURNS bool
	AS '$libdir/postgis-3', 'lwgeom_eq'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION geometry_cmp(geom1 geometry, geom2 geometry)
	RETURNS integer
	AS '$libdir/postgis-3', 'lwgeom_cmp'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION geometry_sortsupport(internal)
	RETURNS void
	AS '$libdir/postgis-3', 'lwgeom_sortsupport'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
-- Operator geometry < geometry -- LastUpdated: 9
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 9 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '<' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR < (
	LEFTARG = geometry, RIGHTARG = geometry, PROCEDURE = geometry_lt,
	COMMUTATOR = '>', NEGATOR = '>=',
	RESTRICT = contsel, JOIN = contjoinsel
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator geometry <= geometry -- LastUpdated: 9
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 9 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '<=' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR <= (
	LEFTARG = geometry, RIGHTARG = geometry, PROCEDURE = geometry_le,
	COMMUTATOR = '>=', NEGATOR = '>',
	RESTRICT = contsel, JOIN = contjoinsel
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator geometry = geometry -- LastUpdated: 9
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 9 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '=' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR = (
	LEFTARG = geometry, RIGHTARG = geometry, PROCEDURE = geometry_eq,
	COMMUTATOR = '=', -- we might implement a faster negator here
	RESTRICT = contsel, JOIN = contjoinsel, HASHES, MERGES
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator geometry >= geometry -- LastUpdated: 9
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 9 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '>=' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR >= (
	LEFTARG = geometry, RIGHTARG = geometry, PROCEDURE = geometry_ge,
	COMMUTATOR = '<=', NEGATOR = '<',
	RESTRICT = contsel, JOIN = contjoinsel
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator geometry > geometry -- LastUpdated: 9
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 9 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '>' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR > (
	LEFTARG = geometry, RIGHTARG = geometry, PROCEDURE = geometry_gt,
	COMMUTATOR = '<', NEGATOR = '<=',
	RESTRICT = contsel, JOIN = contjoinsel
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator class btree_geometry_ops -- LastUpdated: 9
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN

  IF 9 > version_from_num FROM _postgis_upgrade_info
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$
    CREATE OPERATOR CLASS btree_geometry_ops
	DEFAULT FOR TYPE geometry USING btree AS
	OPERATOR	1	< ,
	OPERATOR	2	<= ,
	OPERATOR	3	= ,
	OPERATOR	4	>= ,
	OPERATOR	5	> ,
	FUNCTION	1	geometry_cmp (geom1 geometry, geom2 geometry),
	FUNCTION	2	geometry_sortsupport(internal);
    $postgis_proc_upgrade_parsed_def$;
  END IF; -- version_from >= 9
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_hash(geometry)
	RETURNS integer
	AS '$libdir/postgis-3','lwgeom_hash'
	LANGUAGE 'c' STRICT IMMUTABLE PARALLEL SAFE;
-- Operator class hash_geometry_ops -- LastUpdated: 205
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN

  IF 205 > version_from_num FROM _postgis_upgrade_info
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$
    CREATE OPERATOR CLASS hash_geometry_ops
	DEFAULT FOR TYPE geometry USING hash AS
	OPERATOR	1   = ,
	FUNCTION	1   geometry_hash(geometry);
    $postgis_proc_upgrade_parsed_def$;
  END IF; -- version_from >= 205
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_gist_distance_2d(internal,geometry,int4)
	RETURNS float8
	AS '$libdir/postgis-3' ,'gserialized_gist_distance_2d'
	LANGUAGE 'c' PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_gist_consistent_2d(internal,geometry,int4)
	RETURNS bool
	AS '$libdir/postgis-3' ,'gserialized_gist_consistent_2d'
	LANGUAGE 'c' PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_gist_compress_2d(internal)
	RETURNS internal
	AS '$libdir/postgis-3','gserialized_gist_compress_2d'
	LANGUAGE 'c' PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_gist_penalty_2d(internal,internal,internal)
	RETURNS internal
	AS '$libdir/postgis-3' ,'gserialized_gist_penalty_2d'
	LANGUAGE 'c' PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_gist_picksplit_2d(internal, internal)
	RETURNS internal
	AS '$libdir/postgis-3' ,'gserialized_gist_picksplit_2d'
	LANGUAGE 'c' PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_gist_union_2d(bytea, internal)
	RETURNS internal
	AS '$libdir/postgis-3' ,'gserialized_gist_union_2d'
	LANGUAGE 'c' PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_gist_same_2d(geom1 geometry, geom2 geometry, internal)
	RETURNS internal
	AS '$libdir/postgis-3' ,'gserialized_gist_same_2d'
	LANGUAGE 'c' PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_gist_decompress_2d(internal)
	RETURNS internal
	AS '$libdir/postgis-3' ,'gserialized_gist_decompress_2d'
	LANGUAGE 'c' PARALLEL SAFE;
CREATE OR REPLACE FUNCTION _postgis_selectivity(tbl regclass, att_name text, geom geometry, mode text default '2')
	RETURNS float8
	AS '$libdir/postgis-3', '_postgis_gserialized_sel'
	LANGUAGE 'c' STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION _postgis_join_selectivity(regclass, text, regclass, text, text default '2')
	RETURNS float8
	AS '$libdir/postgis-3', '_postgis_gserialized_joinsel'
	LANGUAGE 'c' STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION _postgis_stats(tbl regclass, att_name text, text default '2')
	RETURNS text
	AS '$libdir/postgis-3', '_postgis_gserialized_stats'
	LANGUAGE 'c' STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION _postgis_index_extent(tbl regclass, col text)
	RETURNS box2d
	AS '$libdir/postgis-3','_postgis_gserialized_index_extent'
	LANGUAGE 'c' STABLE STRICT;
CREATE OR REPLACE FUNCTION gserialized_gist_sel_2d (internal, oid, internal, int4)
	RETURNS float8
	AS '$libdir/postgis-3', 'gserialized_gist_sel_2d'
	LANGUAGE 'c' PARALLEL SAFE;
CREATE OR REPLACE FUNCTION gserialized_gist_sel_nd (internal, oid, internal, int4)
	RETURNS float8
	AS '$libdir/postgis-3', 'gserialized_gist_sel_nd'
	LANGUAGE 'c' PARALLEL SAFE;
CREATE OR REPLACE FUNCTION gserialized_gist_joinsel_2d (internal, oid, internal, smallint)
	RETURNS float8
	AS '$libdir/postgis-3', 'gserialized_gist_joinsel_2d'
	LANGUAGE 'c' PARALLEL SAFE;
CREATE OR REPLACE FUNCTION gserialized_gist_joinsel_nd (internal, oid, internal, smallint)
	RETURNS float8
	AS '$libdir/postgis-3', 'gserialized_gist_joinsel_nd'
	LANGUAGE 'c' PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_overlaps(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3' ,'gserialized_overlaps_2d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
-- Operator geometry && geometry -- LastUpdated: 200
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 200 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '&&' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR && (
	LEFTARG = geometry, RIGHTARG = geometry, PROCEDURE = geometry_overlaps,
	COMMUTATOR = '&&',
	RESTRICT = gserialized_gist_sel_2d,
	JOIN = gserialized_gist_joinsel_2d
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_same(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3' ,'gserialized_same_2d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
-- Operator geometry ~= geometry -- LastUpdated: 1
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 1 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '~=' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR ~= (
	LEFTARG = geometry, RIGHTARG = geometry, PROCEDURE = geometry_same,
	RESTRICT = contsel, JOIN = contjoinsel
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_distance_centroid(geom1 geometry, geom2 geometry)
	RETURNS float8
	AS '$libdir/postgis-3', 'ST_Distance'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION geometry_distance_box(geom1 geometry, geom2 geometry)
	RETURNS float8
	AS '$libdir/postgis-3', 'gserialized_distance_box_2d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
-- Operator geometry <-> geometry -- LastUpdated: 200
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 200 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '<->' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR <-> (
	LEFTARG = geometry, RIGHTARG = geometry, PROCEDURE = geometry_distance_centroid,
	COMMUTATOR = '<->'
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator geometry <#> geometry -- LastUpdated: 200
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 200 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '<#>' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR <#> (
	LEFTARG = geometry, RIGHTARG = geometry, PROCEDURE = geometry_distance_box,
	COMMUTATOR = '<#>'
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_contains(geom1 geometry, geom2 geometry)
	RETURNS bool
	AS '$libdir/postgis-3', 'gserialized_contains_2d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION geometry_within(geom1 geometry, geom2 geometry)
	RETURNS bool
	AS '$libdir/postgis-3', 'gserialized_within_2d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
-- Operator geometry @ geometry -- LastUpdated: 1
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 1 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '@' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR @ (
	LEFTARG = geometry, RIGHTARG = geometry, PROCEDURE = geometry_within,
	COMMUTATOR = '~',
	RESTRICT = contsel, JOIN = contjoinsel
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator geometry ~ geometry -- LastUpdated: 1
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 1 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '~' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR ~ (
	LEFTARG = geometry, RIGHTARG = geometry, PROCEDURE = geometry_contains,
	COMMUTATOR = '@',
	RESTRICT = contsel, JOIN = contjoinsel
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_left(geom1 geometry, geom2 geometry)
	RETURNS bool
	AS '$libdir/postgis-3', 'gserialized_left_2d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
-- Operator geometry << geometry -- LastUpdated: 1
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 1 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '<<' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR << (
	LEFTARG = geometry, RIGHTARG = geometry, PROCEDURE = geometry_left,
	COMMUTATOR = '>>',
	RESTRICT = positionsel, JOIN = positionjoinsel
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_overleft(geom1 geometry, geom2 geometry)
	RETURNS bool
	AS '$libdir/postgis-3', 'gserialized_overleft_2d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
-- Operator geometry &< geometry -- LastUpdated: 1
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 1 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '&<' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR &< (
	LEFTARG = geometry, RIGHTARG = geometry, PROCEDURE = geometry_overleft,
	RESTRICT = positionsel, JOIN = positionjoinsel
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_below(geom1 geometry, geom2 geometry)
	RETURNS bool
	AS '$libdir/postgis-3', 'gserialized_below_2d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
-- Operator geometry <<| geometry -- LastUpdated: 1
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 1 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '<<|' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR <<| (
	LEFTARG = geometry, RIGHTARG = geometry, PROCEDURE = geometry_below,
	COMMUTATOR = '|>>',
	RESTRICT = positionsel, JOIN = positionjoinsel
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_overbelow(geom1 geometry, geom2 geometry)
	RETURNS bool
	AS '$libdir/postgis-3', 'gserialized_overbelow_2d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
-- Operator geometry &<| geometry -- LastUpdated: 1
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 1 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '&<|' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR &<| (
	LEFTARG = geometry, RIGHTARG = geometry, PROCEDURE = geometry_overbelow,
	RESTRICT = positionsel, JOIN = positionjoinsel
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_overright(geom1 geometry, geom2 geometry)
	RETURNS bool
	AS '$libdir/postgis-3', 'gserialized_overright_2d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
-- Operator geometry &> geometry -- LastUpdated: 1
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 1 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '&>' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR &> (
	LEFTARG = geometry, RIGHTARG = geometry, PROCEDURE = geometry_overright,
	RESTRICT = positionsel, JOIN = positionjoinsel
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_right(geom1 geometry, geom2 geometry)
	RETURNS bool
	AS '$libdir/postgis-3', 'gserialized_right_2d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
-- Operator geometry >> geometry -- LastUpdated: 1
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 1 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '>>' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR >> (
	LEFTARG = geometry, RIGHTARG = geometry, PROCEDURE = geometry_right,
	COMMUTATOR = '<<',
	RESTRICT = positionsel, JOIN = positionjoinsel
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_overabove(geom1 geometry, geom2 geometry)
	RETURNS bool
	AS '$libdir/postgis-3', 'gserialized_overabove_2d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
-- Operator geometry |&> geometry -- LastUpdated: 1
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 1 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '|&>' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR |&> (
	LEFTARG = geometry, RIGHTARG = geometry, PROCEDURE = geometry_overabove,
	RESTRICT = positionsel, JOIN = positionjoinsel
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_above(geom1 geometry, geom2 geometry)
	RETURNS bool
	AS '$libdir/postgis-3', 'gserialized_above_2d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
-- Operator geometry |>> geometry -- LastUpdated: 1
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 1 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '|>>' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR |>> (
	LEFTARG = geometry, RIGHTARG = geometry, PROCEDURE = geometry_above,
	COMMUTATOR = '<<|',
	RESTRICT = positionsel, JOIN = positionjoinsel
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator class gist_geometry_ops_2d -- LastUpdated: 200
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN

  IF 200 > version_from_num FROM _postgis_upgrade_info
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$
    CREATE OPERATOR CLASS gist_geometry_ops_2d
	DEFAULT FOR TYPE geometry USING GIST AS
	STORAGE box2df,
	OPERATOR        1        <<  ,
	OPERATOR        2        &<	 ,
	OPERATOR        3        &&  ,
	OPERATOR        4        &>	 ,
	OPERATOR        5        >>	 ,
	OPERATOR        6        ~=	 ,
	OPERATOR        7        ~	 ,
	OPERATOR        8        @	 ,
	OPERATOR        9        &<| ,
	OPERATOR        10       <<| ,
	OPERATOR        11       |>> ,
	OPERATOR        12       |&> ,
	OPERATOR        13       <-> FOR ORDER BY pg_catalog.float_ops,
	OPERATOR        14       <#> FOR ORDER BY pg_catalog.float_ops,
	FUNCTION        8        geometry_gist_distance_2d (internal, geometry, int4),
	FUNCTION        1        geometry_gist_consistent_2d (internal, geometry, int4),
	FUNCTION        2        geometry_gist_union_2d (bytea, internal),
	FUNCTION        3        geometry_gist_compress_2d (internal),
	FUNCTION        4        geometry_gist_decompress_2d (internal),
	FUNCTION        5        geometry_gist_penalty_2d (internal, internal, internal),
	FUNCTION        6        geometry_gist_picksplit_2d (internal, internal),
	FUNCTION        7        geometry_gist_same_2d (geom1 geometry, geom2 geometry, internal);
    $postgis_proc_upgrade_parsed_def$;
  END IF; -- version_from >= 200
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_gist_consistent_nd(internal,geometry,int4)
	RETURNS bool
	AS '$libdir/postgis-3' ,'gserialized_gist_consistent'
	LANGUAGE 'c' PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION geometry_gist_compress_nd(internal)
	RETURNS internal
	AS '$libdir/postgis-3','gserialized_gist_compress'
	LANGUAGE 'c' PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION geometry_gist_penalty_nd(internal,internal,internal)
	RETURNS internal
	AS '$libdir/postgis-3' ,'gserialized_gist_penalty'
	LANGUAGE 'c' PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION geometry_gist_picksplit_nd(internal, internal)
	RETURNS internal
	AS '$libdir/postgis-3' ,'gserialized_gist_picksplit'
	LANGUAGE 'c' PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION geometry_gist_union_nd(bytea, internal)
	RETURNS internal
	AS '$libdir/postgis-3' ,'gserialized_gist_union'
	LANGUAGE 'c' PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION geometry_gist_same_nd(geometry, geometry, internal)
	RETURNS internal
	AS '$libdir/postgis-3' ,'gserialized_gist_same'
	LANGUAGE 'c' PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION geometry_gist_decompress_nd(internal)
	RETURNS internal
	AS '$libdir/postgis-3' ,'gserialized_gist_decompress'
	LANGUAGE 'c' PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION geometry_overlaps_nd(geometry, geometry)
	RETURNS boolean
	AS '$libdir/postgis-3' ,'gserialized_overlaps'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
-- Operator geometry &&& geometry -- LastUpdated: 200
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 200 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '&&&' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR &&& (
	LEFTARG = geometry, RIGHTARG = geometry, PROCEDURE = geometry_overlaps_nd,
	COMMUTATOR = '&&&',
	RESTRICT = gserialized_gist_sel_nd,
	JOIN = gserialized_gist_joinsel_nd
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_contains_nd(geometry, geometry)
	RETURNS boolean
	AS '$libdir/postgis-3' ,'gserialized_contains'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
-- Operator geometry ~~ geometry -- LastUpdated: 300
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 300 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '~~' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR ~~ (
	LEFTARG = geometry, RIGHTARG = geometry, PROCEDURE = geometry_contains_nd,
	COMMUTATOR = '@@',
	RESTRICT = gserialized_gist_sel_nd,
	JOIN = gserialized_gist_joinsel_nd
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_within_nd(geometry, geometry)
	RETURNS boolean
	AS '$libdir/postgis-3' ,'gserialized_within'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
-- Operator geometry @@ geometry -- LastUpdated: 300
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 300 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '@@' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR @@ (
	LEFTARG = geometry, RIGHTARG = geometry, PROCEDURE = geometry_within_nd,
	COMMUTATOR = '~~',
	RESTRICT = gserialized_gist_sel_nd,
	JOIN = gserialized_gist_joinsel_nd
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_same_nd(geometry, geometry)
	RETURNS boolean
	AS '$libdir/postgis-3' ,'gserialized_same'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
-- Operator geometry ~~= geometry -- LastUpdated: 300
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 300 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '~~=' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR ~~= (
	LEFTARG = geometry, RIGHTARG = geometry, PROCEDURE = geometry_same_nd,
	COMMUTATOR = '~~=',
	RESTRICT = gserialized_gist_sel_nd,
	JOIN = gserialized_gist_joinsel_nd
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_distance_centroid_nd(geometry,geometry)
	RETURNS float8
	AS '$libdir/postgis-3', 'gserialized_distance_nd'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
-- Operator geometry <<->> geometry -- LastUpdated: 202
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 202 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '<<->>' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR <<->> (
	LEFTARG = geometry, RIGHTARG = geometry,
	PROCEDURE = geometry_distance_centroid_nd,
	COMMUTATOR = '<<->>'
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_distance_cpa(geometry, geometry)
	RETURNS float8
	AS '$libdir/postgis-3', 'ST_DistanceCPA'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
-- Operator geometry |=| geometry -- LastUpdated: 202
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 202 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '|=|' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR |=| (
	LEFTARG = geometry, RIGHTARG = geometry,
	PROCEDURE = geometry_distance_cpa,
	COMMUTATOR = '|=|'
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_gist_distance_nd(internal,geometry,int4)
	RETURNS float8
	AS '$libdir/postgis-3', 'gserialized_gist_distance'
	LANGUAGE 'c' PARALLEL SAFE
	COST 1;
-- Operator class gist_geometry_ops_nd -- LastUpdated: 200
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN

  IF 200 > version_from_num FROM _postgis_upgrade_info
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$
    CREATE OPERATOR CLASS gist_geometry_ops_nd
	FOR TYPE geometry USING GIST AS
	STORAGE 	gidx,
	OPERATOR        3        &&&	,
	OPERATOR        6        ~~=	,
	OPERATOR        7        ~~	,
	OPERATOR        8        @@	,
	OPERATOR        13       <<->> FOR ORDER BY pg_catalog.float_ops,
	OPERATOR        20       |=| FOR ORDER BY pg_catalog.float_ops,
	FUNCTION        8        geometry_gist_distance_nd (internal, geometry, int4),
	FUNCTION        1        geometry_gist_consistent_nd (internal, geometry, int4),
	FUNCTION        2        geometry_gist_union_nd (bytea, internal),
	FUNCTION        3        geometry_gist_compress_nd (internal),
	FUNCTION        4        geometry_gist_decompress_nd (internal),
	FUNCTION        5        geometry_gist_penalty_nd (internal, internal, internal),
	FUNCTION        6        geometry_gist_picksplit_nd (internal, internal),
	FUNCTION        7        geometry_gist_same_nd (geometry, geometry, internal);
    $postgis_proc_upgrade_parsed_def$;
  ELSE -- version_from >= 200
    -- Last Updated: 300
    IF 300 > version_from_num FROM _postgis_upgrade_info THEN
      EXECUTE $postgis_proc_upgrade_parsed_def$
        ALTER OPERATOR FAMILY gist_geometry_ops_nd USING gist
          ADD OPERATOR        6        ~~=	(geometry,geometry) ;
      $postgis_proc_upgrade_parsed_def$;
    END IF;
  
    -- Last Updated: 300
    IF 300 > version_from_num FROM _postgis_upgrade_info THEN
      EXECUTE $postgis_proc_upgrade_parsed_def$
        ALTER OPERATOR FAMILY gist_geometry_ops_nd USING gist
          ADD OPERATOR        7        ~~	(geometry,geometry) ;
      $postgis_proc_upgrade_parsed_def$;
    END IF;
  
    -- Last Updated: 300
    IF 300 > version_from_num FROM _postgis_upgrade_info THEN
      EXECUTE $postgis_proc_upgrade_parsed_def$
        ALTER OPERATOR FAMILY gist_geometry_ops_nd USING gist
          ADD OPERATOR        8        @@	(geometry,geometry) ;
      $postgis_proc_upgrade_parsed_def$;
    END IF;
  
    -- Last Updated: 202
    IF 202 > version_from_num FROM _postgis_upgrade_info THEN
      EXECUTE $postgis_proc_upgrade_parsed_def$
        ALTER OPERATOR FAMILY gist_geometry_ops_nd USING gist
          ADD OPERATOR        13       <<->> (geometry,geometry) FOR ORDER BY pg_catalog.float_ops;
      $postgis_proc_upgrade_parsed_def$;
    END IF;
  
    -- Last Updated: 202
    IF 202 > version_from_num FROM _postgis_upgrade_info THEN
      EXECUTE $postgis_proc_upgrade_parsed_def$
        ALTER OPERATOR FAMILY gist_geometry_ops_nd USING gist
          ADD OPERATOR        20       |=| (geometry,geometry) FOR ORDER BY pg_catalog.float_ops;
      $postgis_proc_upgrade_parsed_def$;
    END IF;
  
    -- Last Updated: 202
    IF 202 > version_from_num FROM _postgis_upgrade_info THEN
      EXECUTE $postgis_proc_upgrade_parsed_def$
        ALTER OPERATOR FAMILY gist_geometry_ops_nd USING gist
          ADD FUNCTION        8 (geometry,geometry)        geometry_gist_distance_nd (internal, geometry, int4);
      $postgis_proc_upgrade_parsed_def$;
    END IF;
  END IF; -- version_from >= 202
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION ST_ShiftLongitude(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_longitude_shift'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_WrapX(geom geometry, wrap float8, move float8)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_WrapX'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_XMin(box3d)
	RETURNS FLOAT8
	AS '$libdir/postgis-3','BOX3D_xmin'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION ST_YMin(box3d)
	RETURNS FLOAT8
	AS '$libdir/postgis-3','BOX3D_ymin'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION ST_ZMin(box3d)
	RETURNS FLOAT8
	AS '$libdir/postgis-3','BOX3D_zmin'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION ST_XMax(box3d)
	RETURNS FLOAT8
	AS '$libdir/postgis-3','BOX3D_xmax'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION ST_YMax(box3d)
	RETURNS FLOAT8
	AS '$libdir/postgis-3','BOX3D_ymax'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION ST_ZMax(box3d)
	RETURNS FLOAT8
	AS '$libdir/postgis-3','BOX3D_zmax'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION ST_Expand(box2d,float8)
	RETURNS box2d
	AS '$libdir/postgis-3', 'BOX2D_expand'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION ST_Expand(box box2d, dx float8, dy float8)
	RETURNS box2d
	AS '$libdir/postgis-3', 'BOX2D_expand'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION postgis_getbbox(geometry)
	RETURNS box2d
	AS '$libdir/postgis-3','LWGEOM_to_BOX2DF'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION ST_MakeBox2d(geom1 geometry, geom2 geometry)
	RETURNS box2d
	AS '$libdir/postgis-3', 'BOX2D_construct'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION ST_EstimatedExtent(text,text,text,boolean) RETURNS box2d AS
	'$libdir/postgis-3', 'gserialized_estimated_extent'
	LANGUAGE 'c' STABLE STRICT SECURITY DEFINER;
CREATE OR REPLACE FUNCTION ST_EstimatedExtent(text,text,text) RETURNS box2d AS
	'$libdir/postgis-3', 'gserialized_estimated_extent'
	LANGUAGE 'c' STABLE STRICT SECURITY DEFINER;
CREATE OR REPLACE FUNCTION ST_EstimatedExtent(text,text) RETURNS box2d AS
	'$libdir/postgis-3', 'gserialized_estimated_extent'
	LANGUAGE 'c' STABLE STRICT SECURITY DEFINER;
CREATE OR REPLACE FUNCTION ST_FindExtent(text,text,text) RETURNS box2d AS
$$
DECLARE
	schemaname alias for $1;
	tablename alias for $2;
	columnname alias for $3;
	myrec RECORD;
BEGIN
	FOR myrec IN EXECUTE 'SELECT @extschema@.ST_Extent("' || columnname || '") As extent FROM "' || schemaname || '"."' || tablename || '"' LOOP
		return myrec.extent;
	END LOOP;
END;
$$
LANGUAGE 'plpgsql' STABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION ST_FindExtent(text,text) RETURNS box2d AS
$$
DECLARE
	tablename alias for $1;
	columnname alias for $2;
	myrec RECORD;

BEGIN
	FOR myrec IN EXECUTE 'SELECT @extschema@.ST_Extent("' || columnname || '") As extent FROM "' || tablename || '"' LOOP
		return myrec.extent;
	END LOOP;
END;
$$
LANGUAGE 'plpgsql' STABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION postgis_addbbox(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3','LWGEOM_addBBOX'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION postgis_dropbbox(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3','LWGEOM_dropBBOX'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION postgis_hasbbox(geometry)
	RETURNS bool
	AS '$libdir/postgis-3', 'LWGEOM_hasBBOX'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION ST_QuantizeCoordinates(g geometry, prec_x int, prec_y int DEFAULT NULL, prec_z int DEFAULT NULL, prec_m int DEFAULT NULL)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_QuantizeCoordinates'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_MemSize(geometry)
	RETURNS int4
	AS '$libdir/postgis-3', 'LWGEOM_mem_size'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION ST_Summary(geometry)
	RETURNS text
	AS '$libdir/postgis-3', 'LWGEOM_summary'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_NPoints(geometry)
	RETURNS int4
	AS '$libdir/postgis-3', 'LWGEOM_npoints'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_NRings(geometry)
	RETURNS int4
	AS '$libdir/postgis-3', 'LWGEOM_nrings'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_3DLength(geometry)
	RETURNS FLOAT8
	AS '$libdir/postgis-3', 'LWGEOM_length_linestring'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Length2d(geometry)
	RETURNS FLOAT8
	AS '$libdir/postgis-3', 'LWGEOM_length2d_linestring'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Length(geometry)
	RETURNS FLOAT8
	AS '$libdir/postgis-3', 'LWGEOM_length2d_linestring'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_LengthSpheroid(geometry, spheroid)
	RETURNS FLOAT8
	AS '$libdir/postgis-3','LWGEOM_length_ellipsoid_linestring'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_Length2DSpheroid(geometry, spheroid)
	RETURNS FLOAT8
	AS '$libdir/postgis-3','LWGEOM_length2d_ellipsoid'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_3DPerimeter(geometry)
	RETURNS FLOAT8
	AS '$libdir/postgis-3', 'LWGEOM_perimeter_poly'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_perimeter2d(geometry)
	RETURNS FLOAT8
	AS '$libdir/postgis-3', 'LWGEOM_perimeter2d_poly'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Perimeter(geometry)
	RETURNS FLOAT8
	AS '$libdir/postgis-3', 'LWGEOM_perimeter2d_poly'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Area2D(geometry)
	RETURNS FLOAT8
	AS '$libdir/postgis-3', 'ST_Area'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Area(geometry)
	RETURNS FLOAT8
	AS '$libdir/postgis-3','ST_Area'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_IsPolygonCW(geometry)
	RETURNS boolean
	AS '$libdir/postgis-3','ST_IsPolygonCW'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_IsPolygonCCW(geometry)
	RETURNS boolean
	AS '$libdir/postgis-3','ST_IsPolygonCCW'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_DistanceSpheroid(geom1 geometry, geom2 geometry,spheroid)
	RETURNS FLOAT8
	AS '$libdir/postgis-3','LWGEOM_distance_ellipsoid'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500; --upped this
CREATE OR REPLACE FUNCTION ST_Distance(geom1 geometry, geom2 geometry)
	RETURNS float8
	AS '$libdir/postgis-3', 'ST_Distance'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_PointInsideCircle(geometry,float8,float8,float8)
	RETURNS bool
	AS '$libdir/postgis-3', 'LWGEOM_inside_circle_point'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_azimuth(geom1 geometry, geom2 geometry)
	RETURNS float8
	AS '$libdir/postgis-3', 'LWGEOM_azimuth'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Angle(pt1 geometry, pt2 geometry, pt3 geometry, pt4 geometry default 'POINT EMPTY'::geometry)
	RETURNS float8
	AS '$libdir/postgis-3', 'LWGEOM_angle'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Force2D(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_force_2d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Force3DZ(geom geometry, zvalue float8 default 0.0)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_force_3dz'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Force3D(geom geometry, zvalue float8 default 0.0)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_Force3DZ($1, $2)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Force3DM(geom geometry, mvalue float8 default 0.0)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_force_3dm'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Force4D(geom geometry, zvalue float8 default 0.0, mvalue float8 default 0.0)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_force_4d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_ForceCollection(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_force_collection'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_CollectionExtract(geometry, integer)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_CollectionExtract'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_CollectionExtract(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_CollectionExtract'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_CollectionHomogenize(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_CollectionHomogenize'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Multi(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_force_multi'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_ForceCurve(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_force_curve'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_ForceSFS(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_force_sfs'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_ForceSFS(geometry, version text)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_force_sfs'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_Expand(box3d,float8)
	RETURNS box3d
	AS '$libdir/postgis-3', 'BOX3D_expand'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Expand(box box3d, dx float8, dy float8, dz float8 DEFAULT 0)
	RETURNS box3d
	AS '$libdir/postgis-3', 'BOX3D_expand'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Expand(geometry,float8)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_expand'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Expand(geom geometry, dx float8, dy float8, dz float8 DEFAULT 0, dm float8 DEFAULT 0)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_expand'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Envelope(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_envelope'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_BoundingDiagonal(geom geometry, fits boolean DEFAULT false)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_BoundingDiagonal'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION ST_Reverse(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_reverse'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_ForcePolygonCW(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_force_clockwise_poly'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_ForcePolygonCCW(geometry)
	RETURNS geometry
	AS $$ SELECT @extschema@.ST_Reverse(@extschema@.ST_ForcePolygonCW($1)) $$
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_ForceRHR(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_force_clockwise_poly'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION postgis_noop(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_noop'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION postgis_geos_noop(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'GEOSnoop'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION ST_Normalize(geom geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_Normalize'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_zmflag(geometry)
	RETURNS smallint
	AS '$libdir/postgis-3', 'LWGEOM_zmflag'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION ST_NDims(geometry)
	RETURNS smallint
	AS '$libdir/postgis-3', 'LWGEOM_ndims'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION ST_AsEWKT(geometry)
	RETURNS TEXT
	AS '$libdir/postgis-3','LWGEOM_asEWKT'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_AsEWKT(geometry, int4)
	RETURNS TEXT
	AS '$libdir/postgis-3','LWGEOM_asEWKT'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_AsTWKB(geom geometry, prec int4 default NULL, prec_z int4 default NULL, prec_m int4 default NULL, with_sizes boolean default NULL, with_boxes boolean default NULL)
	RETURNS bytea
	AS '$libdir/postgis-3','TWKBFromLWGEOM'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_AsTWKB(geom geometry[], ids bigint[], prec int4 default NULL, prec_z int4 default NULL, prec_m int4 default NULL, with_sizes boolean default NULL, with_boxes boolean default NULL)
	RETURNS bytea
	AS '$libdir/postgis-3','TWKBFromLWGEOMArray'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_AsEWKB(geometry)
	RETURNS BYTEA
	AS '$libdir/postgis-3','WKBFromLWGEOM'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_AsHEXEWKB(geometry)
	RETURNS TEXT
	AS '$libdir/postgis-3','LWGEOM_asHEXEWKB'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_AsHEXEWKB(geometry, text)
	RETURNS TEXT
	AS '$libdir/postgis-3','LWGEOM_asHEXEWKB'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_AsEWKB(geometry,text)
	RETURNS bytea
	AS '$libdir/postgis-3','WKBFromLWGEOM'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_AsLatLonText(geom geometry, tmpl text DEFAULT '')
	RETURNS text
	AS '$libdir/postgis-3','LWGEOM_to_latlon'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION GeomFromEWKB(bytea)
	RETURNS geometry
	AS '$libdir/postgis-3','LWGEOMFromEWKB'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_GeomFromEWKB(bytea)
	RETURNS geometry
	AS '$libdir/postgis-3','LWGEOMFromEWKB'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_GeomFromTWKB(bytea)
	RETURNS geometry
	AS '$libdir/postgis-3','LWGEOMFromTWKB'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION GeomFromEWKT(text)
	RETURNS geometry
	AS '$libdir/postgis-3','parse_WKT_lwgeom'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_GeomFromEWKT(text)
	RETURNS geometry
	AS '$libdir/postgis-3','parse_WKT_lwgeom'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION postgis_cache_bbox()
	RETURNS trigger
	AS '$libdir/postgis-3', 'cache_bbox'
	LANGUAGE 'c';
CREATE OR REPLACE FUNCTION ST_MakePoint(float8, float8)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_makepoint'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_MakePoint(float8, float8, float8)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_makepoint'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_MakePoint(float8, float8, float8, float8)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_makepoint'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_MakePointM(float8, float8, float8)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_makepoint3dm'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_3DMakeBox(geom1 geometry, geom2 geometry)
	RETURNS box3d
	AS '$libdir/postgis-3', 'BOX3D_construct'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_MakeLine (geometry[])
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_makeline_garray'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_LineFromMultiPoint(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_line_from_mpoint'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_MakeLine(geom1 geometry, geom2 geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_makeline'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_AddPoint(geom1 geometry, geom2 geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_addpoint'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_AddPoint(geom1 geometry, geom2 geometry, integer)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_addpoint'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_RemovePoint(geometry, integer)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_removepoint'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_SetPoint(geometry, integer, geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_setpoint_linestring'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_MakeEnvelope(float8, float8, float8, float8, integer DEFAULT 0)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_MakeEnvelope'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_TileEnvelope(zoom integer, x integer, y integer, bounds geometry DEFAULT 'SRID=3857;LINESTRING(-20037508.342789244 -20037508.342789244, 20037508.342789244 20037508.342789244)'::geometry, margin float8 DEFAULT 0.0)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_TileEnvelope'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_MakePolygon(geometry, geometry[])
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_makepoly'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_MakePolygon(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_makepoly'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_BuildArea(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_BuildArea'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Polygonize (geometry[])
	RETURNS geometry
	AS '$libdir/postgis-3', 'polygonize_garray'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_ClusterIntersecting(geometry[])
	RETURNS geometry[]
	AS '$libdir/postgis-3',  'clusterintersecting_garray'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_ClusterWithin(geometry[], float8)
	RETURNS geometry[]
	AS '$libdir/postgis-3',  'cluster_within_distance_garray'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_ClusterDBSCAN (geometry, eps float8, minpoints int)
	RETURNS int
	AS '$libdir/postgis-3', 'ST_ClusterDBSCAN'
	LANGUAGE 'c' IMMUTABLE STRICT WINDOW PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_LineMerge(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'linemerge'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Affine(geometry,float8,float8,float8,float8,float8,float8,float8,float8,float8,float8,float8,float8)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_affine'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Affine(geometry,float8,float8,float8,float8,float8,float8)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_Affine($1,  $2, $3, 0,  $4, $5, 0,  0, 0, 1,  $6, $7, 0)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Rotate(geometry,float8)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_Affine($1,  cos($2), -sin($2), 0,  sin($2), cos($2), 0,  0, 0, 1,  0, 0, 0)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Rotate(geometry,float8,float8,float8)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_Affine($1,  cos($2), -sin($2), 0,  sin($2),  cos($2), 0, 0, 0, 1,	$3 - cos($2) * $3 + sin($2) * $4, $4 - sin($2) * $3 - cos($2) * $4, 0)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Rotate(geometry,float8,geometry)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_Affine($1,  cos($2), -sin($2), 0,  sin($2),  cos($2), 0, 0, 0, 1, @extschema@.ST_X($3) - cos($2) * @extschema@.ST_X($3) + sin($2) * @extschema@.ST_Y($3), @extschema@.ST_Y($3) - sin($2) * @extschema@.ST_X($3) - cos($2) * @extschema@.ST_Y($3), 0)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_RotateZ(geometry,float8)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_Rotate($1, $2)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_RotateX(geometry,float8)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_Affine($1, 1, 0, 0, 0, cos($2), -sin($2), 0, sin($2), cos($2), 0, 0, 0)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_RotateY(geometry,float8)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_Affine($1,  cos($2), 0, sin($2),  0, 1, 0,  -sin($2), 0, cos($2), 0,  0, 0)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Translate(geometry,float8,float8,float8)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_Affine($1, 1, 0, 0, 0, 1, 0, 0, 0, 1, $2, $3, $4)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Translate(geometry,float8,float8)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_Translate($1, $2, $3, 0)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Scale(geometry,geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_Scale'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Scale(geometry,geometry,origin geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_Scale'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Scale(geometry,float8,float8,float8)
	RETURNS geometry
	--AS 'SELECT ST_Affine($1,  $2, 0, 0,  0, $3, 0,  0, 0, $4,  0, 0, 0)'
	AS 'SELECT @extschema@.ST_Scale($1, @extschema@.ST_MakePoint($2, $3, $4))'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Scale(geometry,float8,float8)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_Scale($1, $2, $3, 1)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Transscale(geometry,float8,float8,float8,float8)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_Affine($1,  $4, 0, 0,  0, $5, 0,
		0, 0, 1,  $2 * $4, $3 * $5, 0)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
-- Type geometry_dump -- LastUpdated: 100
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF 100 > version_from_num
     FROM _postgis_upgrade_info
  THEN
      EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE TYPE geometry_dump AS (
	path integer[],
	geom geometry
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION ST_Dump(geometry)
	RETURNS SETOF geometry_dump
	AS '$libdir/postgis-3', 'LWGEOM_dump'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_DumpRings(geometry)
	RETURNS SETOF geometry_dump
	AS '$libdir/postgis-3', 'LWGEOM_dump_rings'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_DumpPoints(geometry)
	RETURNS SETOF geometry_dump
	AS '$libdir/postgis-3', 'LWGEOM_dumppoints'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION populate_geometry_columns(use_typmod boolean DEFAULT true)
	RETURNS text AS
$$
DECLARE
	inserted	integer;
	oldcount	integer;
	probed	  integer;
	stale	   integer;
	gcs		 RECORD;
	gc		  RECORD;
	gsrid	   integer;
	gndims	  integer;
	gtype	   text;
	query	   text;
	gc_is_valid boolean;

BEGIN
	SELECT count(*) INTO oldcount FROM @extschema@.geometry_columns;
	inserted := 0;

	-- Count the number of geometry columns in all tables and views
	SELECT count(DISTINCT c.oid) INTO probed
	FROM pg_class c,
		 pg_attribute a,
		 pg_type t,
		 pg_namespace n
	WHERE c.relkind IN('r','v','f', 'p')
		AND t.typname = 'geometry'
		AND a.attisdropped = false
		AND a.atttypid = t.oid
		AND a.attrelid = c.oid
		AND c.relnamespace = n.oid
		AND n.nspname NOT ILIKE 'pg_temp%' AND c.relname != 'raster_columns' ;

	-- Iterate through all non-dropped geometry columns
	RAISE DEBUG 'Processing Tables.....';

	FOR gcs IN
	SELECT DISTINCT ON (c.oid) c.oid, n.nspname, c.relname
		FROM pg_class c,
			 pg_attribute a,
			 pg_type t,
			 pg_namespace n
		WHERE c.relkind IN( 'r', 'f', 'p')
		AND t.typname = 'geometry'
		AND a.attisdropped = false
		AND a.atttypid = t.oid
		AND a.attrelid = c.oid
		AND c.relnamespace = n.oid
		AND n.nspname NOT ILIKE 'pg_temp%' AND c.relname != 'raster_columns'
	LOOP

		inserted := inserted + @extschema@.populate_geometry_columns(gcs.oid, use_typmod);
	END LOOP;

	IF oldcount > inserted THEN
		stale = oldcount-inserted;
	ELSE
		stale = 0;
	END IF;

	RETURN 'probed:' ||probed|| ' inserted:'||inserted;
END

$$
LANGUAGE 'plpgsql' VOLATILE;
CREATE OR REPLACE FUNCTION populate_geometry_columns(tbl_oid oid, use_typmod boolean DEFAULT true)
	RETURNS integer AS
$$
DECLARE
	gcs		 RECORD;
	gc		  RECORD;
	gc_old	  RECORD;
	gsrid	   integer;
	gndims	  integer;
	gtype	   text;
	query	   text;
	gc_is_valid boolean;
	inserted	integer;
	constraint_successful boolean := false;

BEGIN
	inserted := 0;

	-- Iterate through all geometry columns in this table
	FOR gcs IN
	SELECT n.nspname, c.relname, a.attname, c.relkind
		FROM pg_class c,
			 pg_attribute a,
			 pg_type t,
			 pg_namespace n
		WHERE c.relkind IN('r', 'f', 'p')
		AND t.typname = 'geometry'
		AND a.attisdropped = false
		AND a.atttypid = t.oid
		AND a.attrelid = c.oid
		AND c.relnamespace = n.oid
		AND n.nspname NOT ILIKE 'pg_temp%'
		AND c.oid = tbl_oid
	LOOP

        RAISE DEBUG 'Processing column %.%.%', gcs.nspname, gcs.relname, gcs.attname;

        gc_is_valid := true;
        -- Find the srid, coord_dimension, and type of current geometry
        -- in geometry_columns -- which is now a view

        SELECT type, srid, coord_dimension, gcs.relkind INTO gc_old
            FROM geometry_columns
            WHERE f_table_schema = gcs.nspname AND f_table_name = gcs.relname AND f_geometry_column = gcs.attname;

        IF upper(gc_old.type) = 'GEOMETRY' THEN
        -- This is an unconstrained geometry we need to do something
        -- We need to figure out what to set the type by inspecting the data
            EXECUTE 'SELECT @extschema@.ST_srid(' || quote_ident(gcs.attname) || ') As srid, @extschema@.GeometryType(' || quote_ident(gcs.attname) || ') As type, @extschema@.ST_NDims(' || quote_ident(gcs.attname) || ') As dims ' ||
                     ' FROM ONLY ' || quote_ident(gcs.nspname) || '.' || quote_ident(gcs.relname) ||
                     ' WHERE ' || quote_ident(gcs.attname) || ' IS NOT NULL LIMIT 1;'
                INTO gc;
            IF gc IS NULL THEN -- there is no data so we can not determine geometry type
            	RAISE WARNING 'No data in table %.%, so no information to determine geometry type and srid', gcs.nspname, gcs.relname;
            	RETURN 0;
            END IF;
            gsrid := gc.srid; gtype := gc.type; gndims := gc.dims;

            IF use_typmod THEN
                BEGIN
                    EXECUTE 'ALTER TABLE ' || quote_ident(gcs.nspname) || '.' || quote_ident(gcs.relname) || ' ALTER COLUMN ' || quote_ident(gcs.attname) ||
                        ' TYPE geometry(' || postgis_type_name(gtype, gndims, true) || ', ' || gsrid::text  || ') ';
                    inserted := inserted + 1;
                EXCEPTION
                        WHEN invalid_parameter_value OR feature_not_supported THEN
                        RAISE WARNING 'Could not convert ''%'' in ''%.%'' to use typmod with srid %, type %: %', quote_ident(gcs.attname), quote_ident(gcs.nspname), quote_ident(gcs.relname), gsrid, postgis_type_name(gtype, gndims, true), SQLERRM;
                            gc_is_valid := false;
                END;

            ELSE
                -- Try to apply srid check to column
            	constraint_successful = false;
                IF (gsrid > 0 AND postgis_constraint_srid(gcs.nspname, gcs.relname,gcs.attname) IS NULL ) THEN
                    BEGIN
                        EXECUTE 'ALTER TABLE ONLY ' || quote_ident(gcs.nspname) || '.' || quote_ident(gcs.relname) ||
                                 ' ADD CONSTRAINT ' || quote_ident('enforce_srid_' || gcs.attname) ||
                                 ' CHECK (ST_srid(' || quote_ident(gcs.attname) || ') = ' || gsrid || ')';
                        constraint_successful := true;
                    EXCEPTION
                        WHEN check_violation THEN
                            RAISE WARNING 'Not inserting ''%'' in ''%.%'' into geometry_columns: could not apply constraint CHECK (st_srid(%) = %)', quote_ident(gcs.attname), quote_ident(gcs.nspname), quote_ident(gcs.relname), quote_ident(gcs.attname), gsrid;
                            gc_is_valid := false;
                    END;
                END IF;

                -- Try to apply ndims check to column
                IF (gndims IS NOT NULL AND postgis_constraint_dims(gcs.nspname, gcs.relname,gcs.attname) IS NULL ) THEN
                    BEGIN
                        EXECUTE 'ALTER TABLE ONLY ' || quote_ident(gcs.nspname) || '.' || quote_ident(gcs.relname) || '
                                 ADD CONSTRAINT ' || quote_ident('enforce_dims_' || gcs.attname) || '
                                 CHECK (st_ndims(' || quote_ident(gcs.attname) || ') = '||gndims||')';
                        constraint_successful := true;
                    EXCEPTION
                        WHEN check_violation THEN
                            RAISE WARNING 'Not inserting ''%'' in ''%.%'' into geometry_columns: could not apply constraint CHECK (st_ndims(%) = %)', quote_ident(gcs.attname), quote_ident(gcs.nspname), quote_ident(gcs.relname), quote_ident(gcs.attname), gndims;
                            gc_is_valid := false;
                    END;
                END IF;

                -- Try to apply geometrytype check to column
                IF (gtype IS NOT NULL AND postgis_constraint_type(gcs.nspname, gcs.relname,gcs.attname) IS NULL ) THEN
                    BEGIN
                        EXECUTE 'ALTER TABLE ONLY ' || quote_ident(gcs.nspname) || '.' || quote_ident(gcs.relname) || '
                        ADD CONSTRAINT ' || quote_ident('enforce_geotype_' || gcs.attname) || '
                        CHECK (geometrytype(' || quote_ident(gcs.attname) || ') = ' || quote_literal(gtype) || ')';
                        constraint_successful := true;
                    EXCEPTION
                        WHEN check_violation THEN
                            -- No geometry check can be applied. This column contains a number of geometry types.
                            RAISE WARNING 'Could not add geometry type check (%) to table column: %.%.%', gtype, quote_ident(gcs.nspname),quote_ident(gcs.relname),quote_ident(gcs.attname);
                    END;
                END IF;
                 --only count if we were successful in applying at least one constraint
                IF constraint_successful THEN
                	inserted := inserted + 1;
                END IF;
            END IF;
	    END IF;

	END LOOP;

	RETURN inserted;
END

$$
LANGUAGE 'plpgsql' VOLATILE;
CREATE OR REPLACE FUNCTION AddGeometryColumn(catalog_name varchar,schema_name varchar,table_name varchar,column_name varchar,new_srid_in integer,new_type varchar,new_dim integer, use_typmod boolean DEFAULT true)
	RETURNS text
	AS
$$
DECLARE
	rec RECORD;
	sr varchar;
	real_schema name;
	sql text;
	new_srid integer;

BEGIN

	-- Verify geometry type
	IF (postgis_type_name(new_type,new_dim) IS NULL )
	THEN
		RAISE EXCEPTION 'Invalid type name "%(%)" - valid ones are:
	POINT, MULTIPOINT,
	LINESTRING, MULTILINESTRING,
	POLYGON, MULTIPOLYGON,
	CIRCULARSTRING, COMPOUNDCURVE, MULTICURVE,
	CURVEPOLYGON, MULTISURFACE,
	GEOMETRY, GEOMETRYCOLLECTION,
	POINTM, MULTIPOINTM,
	LINESTRINGM, MULTILINESTRINGM,
	POLYGONM, MULTIPOLYGONM,
	CIRCULARSTRINGM, COMPOUNDCURVEM, MULTICURVEM
	CURVEPOLYGONM, MULTISURFACEM, TRIANGLE, TRIANGLEM,
	POLYHEDRALSURFACE, POLYHEDRALSURFACEM, TIN, TINM
	or GEOMETRYCOLLECTIONM', new_type, new_dim;
		RETURN 'fail';
	END IF;

	-- Verify dimension
	IF ( (new_dim >4) OR (new_dim <2) ) THEN
		RAISE EXCEPTION 'invalid dimension';
		RETURN 'fail';
	END IF;

	IF ( (new_type LIKE '%M') AND (new_dim!=3) ) THEN
		RAISE EXCEPTION 'TypeM needs 3 dimensions';
		RETURN 'fail';
	END IF;

	-- Verify SRID
	IF ( new_srid_in > 0 ) THEN
		IF new_srid_in > 998999 THEN
			RAISE EXCEPTION 'AddGeometryColumn() - SRID must be <= %', 998999;
		END IF;
		new_srid := new_srid_in;
		SELECT SRID INTO sr FROM spatial_ref_sys WHERE SRID = new_srid;
		IF NOT FOUND THEN
			RAISE EXCEPTION 'AddGeometryColumn() - invalid SRID';
			RETURN 'fail';
		END IF;
	ELSE
		new_srid := @extschema@.ST_SRID('POINT EMPTY'::@extschema@.geometry);
		IF ( new_srid_in != new_srid ) THEN
			RAISE NOTICE 'SRID value % converted to the officially unknown SRID value %', new_srid_in, new_srid;
		END IF;
	END IF;

	-- Verify schema
	IF ( schema_name IS NOT NULL AND schema_name != '' ) THEN
		sql := 'SELECT nspname FROM pg_namespace ' ||
			'WHERE text(nspname) = ' || quote_literal(schema_name) ||
			'LIMIT 1';
		RAISE DEBUG '%', sql;
		EXECUTE sql INTO real_schema;

		IF ( real_schema IS NULL ) THEN
			RAISE EXCEPTION 'Schema % is not a valid schemaname', quote_literal(schema_name);
			RETURN 'fail';
		END IF;
	END IF;

	IF ( real_schema IS NULL ) THEN
		RAISE DEBUG 'Detecting schema';
		sql := 'SELECT n.nspname AS schemaname ' ||
			'FROM pg_catalog.pg_class c ' ||
			  'JOIN pg_catalog.pg_namespace n ON n.oid = c.relnamespace ' ||
			'WHERE c.relkind = ' || quote_literal('r') ||
			' AND n.nspname NOT IN (' || quote_literal('pg_catalog') || ', ' || quote_literal('pg_toast') || ')' ||
			' AND pg_catalog.pg_table_is_visible(c.oid)' ||
			' AND c.relname = ' || quote_literal(table_name);
		RAISE DEBUG '%', sql;
		EXECUTE sql INTO real_schema;

		IF ( real_schema IS NULL ) THEN
			RAISE EXCEPTION 'Table % does not occur in the search_path', quote_literal(table_name);
			RETURN 'fail';
		END IF;
	END IF;

	-- Add geometry column to table
	IF use_typmod THEN
	     sql := 'ALTER TABLE ' ||
            quote_ident(real_schema) || '.' || quote_ident(table_name)
            || ' ADD COLUMN ' || quote_ident(column_name) ||
            ' geometry(' || @extschema@.postgis_type_name(new_type, new_dim) || ', ' || new_srid::text || ')';
        RAISE DEBUG '%', sql;
	ELSE
        sql := 'ALTER TABLE ' ||
            quote_ident(real_schema) || '.' || quote_ident(table_name)
            || ' ADD COLUMN ' || quote_ident(column_name) ||
            ' geometry ';
        RAISE DEBUG '%', sql;
    END IF;
	EXECUTE sql;

	IF NOT use_typmod THEN
        -- Add table CHECKs
        sql := 'ALTER TABLE ' ||
            quote_ident(real_schema) || '.' || quote_ident(table_name)
            || ' ADD CONSTRAINT '
            || quote_ident('enforce_srid_' || column_name)
            || ' CHECK (st_srid(' || quote_ident(column_name) ||
            ') = ' || new_srid::text || ')' ;
        RAISE DEBUG '%', sql;
        EXECUTE sql;

        sql := 'ALTER TABLE ' ||
            quote_ident(real_schema) || '.' || quote_ident(table_name)
            || ' ADD CONSTRAINT '
            || quote_ident('enforce_dims_' || column_name)
            || ' CHECK (st_ndims(' || quote_ident(column_name) ||
            ') = ' || new_dim::text || ')' ;
        RAISE DEBUG '%', sql;
        EXECUTE sql;

        IF ( NOT (new_type = 'GEOMETRY')) THEN
            sql := 'ALTER TABLE ' ||
                quote_ident(real_schema) || '.' || quote_ident(table_name) || ' ADD CONSTRAINT ' ||
                quote_ident('enforce_geotype_' || column_name) ||
                ' CHECK (GeometryType(' ||
                quote_ident(column_name) || ')=' ||
                quote_literal(new_type) || ' OR (' ||
                quote_ident(column_name) || ') is null)';
            RAISE DEBUG '%', sql;
            EXECUTE sql;
        END IF;
    END IF;

	RETURN
		real_schema || '.' ||
		table_name || '.' || column_name ||
		' SRID:' || new_srid::text ||
		' TYPE:' || new_type ||
		' DIMS:' || new_dim::text || ' ';
END;
$$
LANGUAGE 'plpgsql' VOLATILE STRICT;
CREATE OR REPLACE FUNCTION AddGeometryColumn(schema_name varchar,table_name varchar,column_name varchar,new_srid integer,new_type varchar,new_dim integer, use_typmod boolean DEFAULT true) RETURNS text AS $$
DECLARE
	ret  text;
BEGIN
	SELECT @extschema@.AddGeometryColumn('',$1,$2,$3,$4,$5,$6,$7) into ret;
	RETURN ret;
END;
$$
LANGUAGE 'plpgsql' STABLE STRICT;
CREATE OR REPLACE FUNCTION AddGeometryColumn(table_name varchar,column_name varchar,new_srid integer,new_type varchar,new_dim integer, use_typmod boolean DEFAULT true) RETURNS text AS $$
DECLARE
	ret  text;
BEGIN
	SELECT @extschema@.AddGeometryColumn('','',$1,$2,$3,$4,$5, $6) into ret;
	RETURN ret;
END;
$$
LANGUAGE 'plpgsql' VOLATILE STRICT;
CREATE OR REPLACE FUNCTION DropGeometryColumn(catalog_name varchar, schema_name varchar,table_name varchar,column_name varchar)
	RETURNS text
	AS
$$
DECLARE
	myrec RECORD;
	okay boolean;
	real_schema name;

BEGIN

	-- Find, check or fix schema_name
	IF ( schema_name != '' ) THEN
		okay = false;

		FOR myrec IN SELECT nspname FROM pg_namespace WHERE text(nspname) = schema_name LOOP
			okay := true;
		END LOOP;

		IF ( okay <>  true ) THEN
			RAISE NOTICE 'Invalid schema name - using current_schema()';
			SELECT current_schema() into real_schema;
		ELSE
			real_schema = schema_name;
		END IF;
	ELSE
		SELECT current_schema() into real_schema;
	END IF;

	-- Find out if the column is in the geometry_columns table
	okay = false;
	FOR myrec IN SELECT * from @extschema@.geometry_columns where f_table_schema = text(real_schema) and f_table_name = table_name and f_geometry_column = column_name LOOP
		okay := true;
	END LOOP;
	IF (okay <> true) THEN
		RAISE EXCEPTION 'column not found in geometry_columns table';
		RETURN false;
	END IF;

	-- Remove table column
	EXECUTE 'ALTER TABLE ' || quote_ident(real_schema) || '.' ||
		quote_ident(table_name) || ' DROP COLUMN ' ||
		quote_ident(column_name);

	RETURN real_schema || '.' || table_name || '.' || column_name ||' effectively removed.';

END;
$$
LANGUAGE 'plpgsql' VOLATILE STRICT;
CREATE OR REPLACE FUNCTION DropGeometryColumn(schema_name varchar, table_name varchar,column_name varchar)
	RETURNS text
	AS
$$
DECLARE
	ret text;
BEGIN
	SELECT @extschema@.DropGeometryColumn('',$1,$2,$3) into ret;
	RETURN ret;
END;
$$
LANGUAGE 'plpgsql' VOLATILE STRICT;
CREATE OR REPLACE FUNCTION DropGeometryColumn(table_name varchar, column_name varchar)
	RETURNS text
	AS
$$
DECLARE
	ret text;
BEGIN
	SELECT @extschema@.DropGeometryColumn('','',$1,$2) into ret;
	RETURN ret;
END;
$$
LANGUAGE 'plpgsql' VOLATILE STRICT;
CREATE OR REPLACE FUNCTION DropGeometryTable(catalog_name varchar, schema_name varchar, table_name varchar)
	RETURNS text
	AS
$$
DECLARE
	real_schema name;

BEGIN

	IF ( schema_name = '' ) THEN
		SELECT current_schema() into real_schema;
	ELSE
		real_schema = schema_name;
	END IF;

	-- TODO: Should we warn if table doesn't exist probably instead just saying dropped
	-- Remove table
	EXECUTE 'DROP TABLE IF EXISTS '
		|| quote_ident(real_schema) || '.' ||
		quote_ident(table_name) || ' RESTRICT';

	RETURN
		real_schema || '.' ||
		table_name ||' dropped.';

END;
$$
LANGUAGE 'plpgsql' VOLATILE STRICT;
CREATE OR REPLACE FUNCTION DropGeometryTable(schema_name varchar, table_name varchar) RETURNS text AS
$$ SELECT @extschema@.DropGeometryTable('',$1,$2) $$
LANGUAGE 'sql' VOLATILE STRICT;
CREATE OR REPLACE FUNCTION DropGeometryTable(table_name varchar) RETURNS text AS
$$ SELECT @extschema@.DropGeometryTable('','',$1) $$
LANGUAGE 'sql' VOLATILE STRICT;
CREATE OR REPLACE FUNCTION UpdateGeometrySRID(catalogn_name varchar,schema_name varchar,table_name varchar,column_name varchar,new_srid_in integer)
	RETURNS text
	AS
$$
DECLARE
	myrec RECORD;
	okay boolean;
	cname varchar;
	real_schema name;
	unknown_srid integer;
	new_srid integer := new_srid_in;

BEGIN

	-- Find, check or fix schema_name
	IF ( schema_name != '' ) THEN
		okay = false;

		FOR myrec IN SELECT nspname FROM pg_namespace WHERE text(nspname) = schema_name LOOP
			okay := true;
		END LOOP;

		IF ( okay <> true ) THEN
			RAISE EXCEPTION 'Invalid schema name';
		ELSE
			real_schema = schema_name;
		END IF;
	ELSE
		SELECT INTO real_schema current_schema()::text;
	END IF;

	-- Ensure that column_name is in geometry_columns
	okay = false;
	FOR myrec IN SELECT type, coord_dimension FROM @extschema@.geometry_columns WHERE f_table_schema = text(real_schema) and f_table_name = table_name and f_geometry_column = column_name LOOP
		okay := true;
	END LOOP;
	IF (NOT okay) THEN
		RAISE EXCEPTION 'column not found in geometry_columns table';
		RETURN false;
	END IF;

	-- Ensure that new_srid is valid
	IF ( new_srid > 0 ) THEN
		IF ( SELECT count(*) = 0 from spatial_ref_sys where srid = new_srid ) THEN
			RAISE EXCEPTION 'invalid SRID: % not found in spatial_ref_sys', new_srid;
			RETURN false;
		END IF;
	ELSE
		unknown_srid := @extschema@.ST_SRID('POINT EMPTY'::@extschema@.geometry);
		IF ( new_srid != unknown_srid ) THEN
			new_srid := unknown_srid;
			RAISE NOTICE 'SRID value % converted to the officially unknown SRID value %', new_srid_in, new_srid;
		END IF;
	END IF;

	IF postgis_constraint_srid(real_schema, table_name, column_name) IS NOT NULL THEN
	-- srid was enforced with constraints before, keep it that way.
        -- Make up constraint name
        cname = 'enforce_srid_'  || column_name;

        -- Drop enforce_srid constraint
        EXECUTE 'ALTER TABLE ' || quote_ident(real_schema) ||
            '.' || quote_ident(table_name) ||
            ' DROP constraint ' || quote_ident(cname);

        -- Update geometries SRID
        EXECUTE 'UPDATE ' || quote_ident(real_schema) ||
            '.' || quote_ident(table_name) ||
            ' SET ' || quote_ident(column_name) ||
            ' = @extschema@.ST_SetSRID(' || quote_ident(column_name) ||
            ', ' || new_srid::text || ')';

        -- Reset enforce_srid constraint
        EXECUTE 'ALTER TABLE ' || quote_ident(real_schema) ||
            '.' || quote_ident(table_name) ||
            ' ADD constraint ' || quote_ident(cname) ||
            ' CHECK (st_srid(' || quote_ident(column_name) ||
            ') = ' || new_srid::text || ')';
    ELSE
        -- We will use typmod to enforce if no srid constraints
        -- We are using postgis_type_name to lookup the new name
        -- (in case Paul changes his mind and flips geometry_columns to return old upper case name)
        EXECUTE 'ALTER TABLE ' || quote_ident(real_schema) || '.' || quote_ident(table_name) ||
        ' ALTER COLUMN ' || quote_ident(column_name) || ' TYPE  geometry(' || @extschema@.postgis_type_name(myrec.type, myrec.coord_dimension, true) || ', ' || new_srid::text || ') USING @extschema@.ST_SetSRID(' || quote_ident(column_name) || ',' || new_srid::text || ');' ;
    END IF;

	RETURN real_schema || '.' || table_name || '.' || column_name ||' SRID changed to ' || new_srid::text;

END;
$$
LANGUAGE 'plpgsql' VOLATILE STRICT;
CREATE OR REPLACE FUNCTION UpdateGeometrySRID(varchar,varchar,varchar,integer)
	RETURNS text
	AS $$
DECLARE
	ret  text;
BEGIN
	SELECT @extschema@.UpdateGeometrySRID('',$1,$2,$3,$4) into ret;
	RETURN ret;
END;
$$
LANGUAGE 'plpgsql' VOLATILE STRICT;
CREATE OR REPLACE FUNCTION UpdateGeometrySRID(varchar,varchar,integer)
	RETURNS text
	AS $$
DECLARE
	ret  text;
BEGIN
	SELECT @extschema@.UpdateGeometrySRID('','',$1,$2,$3) into ret;
	RETURN ret;
END;
$$
LANGUAGE 'plpgsql' VOLATILE STRICT;
CREATE OR REPLACE FUNCTION find_srid(varchar,varchar,varchar) RETURNS int4 AS
$$
DECLARE
	schem varchar =  $1;
	tabl varchar = $2;
	sr int4;
BEGIN
-- if the table contains a . and the schema is empty
-- split the table into a schema and a table
-- otherwise drop through to default behavior
	IF ( schem = '' and strpos(tabl,'.') > 0 ) THEN
	 schem = substr(tabl,1,strpos(tabl,'.')-1);
	 tabl = substr(tabl,length(schem)+2);
	END IF;

	select SRID into sr from @extschema@.geometry_columns where (f_table_schema = schem or schem = '') and f_table_name = tabl and f_geometry_column = $3;
	IF NOT FOUND THEN
	   RAISE EXCEPTION 'find_srid() - could not find the corresponding SRID - is the geometry registered in the GEOMETRY_COLUMNS table?  Is there an uppercase/lowercase mismatch?';
	END IF;
	return sr;
END;
$$
LANGUAGE 'plpgsql' STABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION get_proj4_from_srid(integer) RETURNS text AS
	$$
	BEGIN
	RETURN proj4text::text FROM @extschema@.spatial_ref_sys WHERE srid= $1;
	END;
	$$
	LANGUAGE 'plpgsql' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION ST_SetSRID(geom geometry, srid int4)
	RETURNS geometry
	AS '$libdir/postgis-3','LWGEOM_set_srid'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION ST_SRID(geom geometry)
	RETURNS int4
	AS '$libdir/postgis-3','LWGEOM_get_srid'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION postgis_transform_geometry(geom geometry, text, text, int)
	RETURNS geometry
	AS '$libdir/postgis-3','transform_geom'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_Transform(geometry,integer)
	RETURNS geometry
	AS '$libdir/postgis-3','transform'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Transform(geom geometry, to_proj text)
	RETURNS geometry AS
	'SELECT @extschema@.postgis_transform_geometry($1, proj4text, $2, 0)
	FROM spatial_ref_sys WHERE srid=@extschema@.ST_SRID($1);'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Transform(geom geometry, from_proj text, to_proj text)
	RETURNS geometry AS
	'SELECT @extschema@.postgis_transform_geometry($1, $2, $3, 0)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Transform(geom geometry, from_proj text, to_srid integer)
	RETURNS geometry AS
	'SELECT @extschema@.postgis_transform_geometry($1, $2, proj4text, $3)
	FROM spatial_ref_sys WHERE srid=$3;'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION postgis_version() RETURNS text
	AS '$libdir/postgis-3'
	LANGUAGE 'c' IMMUTABLE
	COST 1;
CREATE OR REPLACE FUNCTION postgis_liblwgeom_version() RETURNS text
	AS '$libdir/postgis-3'
	LANGUAGE 'c' IMMUTABLE
	COST 1;
CREATE OR REPLACE FUNCTION postgis_proj_version() RETURNS text
	AS '$libdir/postgis-3'
	LANGUAGE 'c' IMMUTABLE
	COST 1;
CREATE OR REPLACE FUNCTION postgis_wagyu_version() RETURNS text
	AS '$libdir/postgis-3'
	LANGUAGE 'c' IMMUTABLE
	COST 1;
CREATE OR REPLACE FUNCTION postgis_scripts_installed() RETURNS text
	AS $$ SELECT trim('3.1.1'::text || $rev$ aaf4c79 $rev$) AS version $$
	LANGUAGE 'sql' IMMUTABLE;
CREATE OR REPLACE FUNCTION postgis_lib_version() RETURNS text
	AS '$libdir/postgis-3'
	LANGUAGE 'c' IMMUTABLE; -- a new lib will require a new session
CREATE OR REPLACE FUNCTION postgis_scripts_released() RETURNS text
	AS '$libdir/postgis-3'
	LANGUAGE 'c' IMMUTABLE;
CREATE OR REPLACE FUNCTION postgis_geos_version() RETURNS text
	AS '$libdir/postgis-3'
	LANGUAGE 'c' IMMUTABLE;
CREATE OR REPLACE FUNCTION postgis_lib_revision() RETURNS text
	AS '$libdir/postgis-3'
	LANGUAGE 'c' IMMUTABLE;
CREATE OR REPLACE FUNCTION postgis_svn_version()
RETURNS text AS $$
	SELECT @extschema@._postgis_deprecate(
		'postgis_svn_version', 'postgis_lib_revision', '3.1.0');
	SELECT @extschema@.postgis_lib_revision();
$$
LANGUAGE 'sql' IMMUTABLE SECURITY INVOKER;
CREATE OR REPLACE FUNCTION postgis_libxml_version() RETURNS text
	AS '$libdir/postgis-3'
	LANGUAGE 'c' IMMUTABLE;
CREATE OR REPLACE FUNCTION postgis_scripts_build_date() RETURNS text
	AS 'SELECT ''2021-04-21 04:42:55''::text AS version'
	LANGUAGE 'sql' IMMUTABLE;
CREATE OR REPLACE FUNCTION postgis_lib_build_date() RETURNS text
	AS '$libdir/postgis-3'
	LANGUAGE 'c' IMMUTABLE;
CREATE OR REPLACE FUNCTION _postgis_scripts_pgsql_version() RETURNS text
	AS 'SELECT ''130''::text AS version'
	LANGUAGE 'sql' IMMUTABLE;
CREATE OR REPLACE FUNCTION _postgis_pgsql_version() RETURNS text
AS $$
	SELECT CASE WHEN split_part(s,'.',1)::integer > 9 THEN split_part(s,'.',1) || '0' ELSE split_part(s,'.', 1) || split_part(s,'.', 2) END AS v
	FROM substring(version(), 'PostgreSQL ([0-9\.]+)') AS s;
$$ LANGUAGE 'sql' STABLE;
CREATE OR REPLACE FUNCTION postgis_extensions_upgrade() RETURNS text
AS $$
DECLARE rec record; sql text; var_schema text;
BEGIN

	FOR rec IN
		SELECT name, default_version, installed_version
		FROM pg_catalog.pg_available_extensions
		WHERE name IN (
			'postgis',
			'postgis_raster',
			'postgis_sfcgal',
			'postgis_topology',
			'postgis_tiger_geocoder'
		)
		ORDER BY length(name) -- this is to make sure 'postgis' is first !
	LOOP
		IF rec.installed_version IS NULL THEN
			-- If the support installed by available extension
			-- is found unpackaged, we package it
			IF
				 -- PostGIS is always available (this function is part of it)
				 rec.name = 'postgis'

				 -- PostGIS raster is available if type 'raster' exists
				 OR ( rec.name = 'postgis_raster' AND EXISTS (
							SELECT 1 FROM pg_catalog.pg_type
							WHERE typname = 'raster' ) )

				 -- PostGIS SFCGAL is availble if
				 -- 'postgis_sfcgal_version' function exists
				 OR ( rec.name = 'postgis_sfcgal' AND EXISTS (
							SELECT 1 FROM pg_catalog.pg_proc
							WHERE proname = 'postgis_sfcgal_version' ) )

				 -- PostGIS Topology is available if
				 -- 'topology.topology' table exists
				 -- NOTE: watch out for https://trac.osgeo.org/postgis/ticket/2503
				 OR ( rec.name = 'postgis_topology' AND EXISTS (
							SELECT 1 FROM pg_catalog.pg_class c
							JOIN pg_catalog.pg_namespace n ON (c.relnamespace = n.oid )
							WHERE n.nspname = 'topology' AND c.relname = 'topology') )

				 OR ( rec.name = 'postgis_tiger_geocoder' AND EXISTS (
							SELECT 1 FROM pg_catalog.pg_class c
							JOIN pg_catalog.pg_namespace n ON (c.relnamespace = n.oid )
							WHERE n.nspname = 'tiger' AND c.relname = 'geocode_settings') )
			THEN
				-- Force install in same schema as postgis
				SELECT INTO var_schema n.nspname
				  FROM pg_namespace n, pg_proc p
				  WHERE p.proname = 'postgis_full_version'
				    AND n.oid = p.pronamespace
				  LIMIT 1;
				IF rec.name NOT IN('postgis_topology', 'postgis_tiger_geocoder')
				THEN
					sql := format(
					          'CREATE EXTENSION %1$I SCHEMA %2$I VERSION unpackaged;'
					          'ALTER EXTENSION %1$I UPDATE TO %3$I',
					          rec.name, var_schema, rec.default_version);
				ELSE
					sql := format(
					         'CREATE EXTENSION %1$I VERSION unpackaged;'
					         'ALTER EXTENSION %1$I UPDATE TO %2$I',
					         rec.name, rec.default_version);
				END IF;
				RAISE NOTICE 'Packaging extension %', rec.name;
				RAISE DEBUG '%', sql;
				EXECUTE sql;
			ELSE
				RAISE NOTICE 'Extension % is not available or not packagable for some reason', rec.name;
			END IF;
		ELSIF rec.default_version != rec.installed_version
		THEN
			sql = 'ALTER EXTENSION ' || rec.name || ' UPDATE TO ' ||
						quote_ident(rec.default_version)   || ';';
			RAISE NOTICE 'Updating extension % from % to %',
				rec.name, rec.installed_version, rec.default_version;
			RAISE DEBUG '%', sql;
			EXECUTE sql;
		ELSIF (rec.default_version = rec.installed_version AND rec.installed_version ILIKE '%dev') OR
			(@extschema@._postgis_pgsql_version() != @extschema@._postgis_scripts_pgsql_version())
		THEN
			-- we need to upgrade to next and back
			RAISE NOTICE 'Updating extension % %',
				rec.name, rec.installed_version;
			sql = 'ALTER EXTENSION ' || rec.name || ' UPDATE TO ' ||
						quote_ident(rec.default_version || 'next')   || ';';
			RAISE DEBUG '%', sql;
			EXECUTE sql;
			sql = 'ALTER EXTENSION ' || rec.name || ' UPDATE TO ' ||
						quote_ident(rec.default_version)   || ';';
			RAISE DEBUG '%', sql;
			EXECUTE sql;
		END IF;

	END LOOP;

	RETURN 'Upgrade completed, run SELECT postgis_full_version(); for details';

END
$$ LANGUAGE plpgsql VOLATILE;
CREATE OR REPLACE FUNCTION postgis_full_version() RETURNS text
AS $$
DECLARE
	libver text;
	librev text;
	projver text;
	geosver text;
	sfcgalver text;
	gdalver text := NULL;
	libxmlver text;
	liblwgeomver text;
	dbproc text;
	relproc text;
	fullver text;
	rast_lib_ver text := NULL;
	rast_scr_ver text := NULL;
	topo_scr_ver text := NULL;
	json_lib_ver text;
	protobuf_lib_ver text;
	wagyu_lib_ver text;
	sfcgal_lib_ver text;
	sfcgal_scr_ver text;
	pgsql_scr_ver text;
	pgsql_ver text;
	core_is_extension bool;
BEGIN
	SELECT @extschema@.postgis_lib_version() INTO libver;
	SELECT @extschema@.postgis_proj_version() INTO projver;
	SELECT @extschema@.postgis_geos_version() INTO geosver;
	SELECT @extschema@.postgis_libjson_version() INTO json_lib_ver;
	SELECT @extschema@.postgis_libprotobuf_version() INTO protobuf_lib_ver;
	SELECT @extschema@.postgis_wagyu_version() INTO wagyu_lib_ver;
	SELECT @extschema@._postgis_scripts_pgsql_version() INTO pgsql_scr_ver;
	SELECT @extschema@._postgis_pgsql_version() INTO pgsql_ver;
	BEGIN
		SELECT @extschema@.postgis_gdal_version() INTO gdalver;
	EXCEPTION
		WHEN undefined_function THEN
			RAISE DEBUG 'Function postgis_gdal_version() not found.  Is raster support enabled and rtpostgis.sql installed?';
	END;
	BEGIN
		SELECT @extschema@.postgis_sfcgal_version() INTO sfcgalver;
		BEGIN
			SELECT @extschema@.postgis_sfcgal_scripts_installed() INTO sfcgal_scr_ver;
		EXCEPTION
			WHEN undefined_function THEN
				sfcgal_scr_ver := 'missing';
		END;
	EXCEPTION
		WHEN undefined_function THEN
			RAISE DEBUG 'Function postgis_sfcgal_scripts_installed() not found. Is sfcgal support enabled and sfcgal.sql installed?';
	END;
	SELECT @extschema@.postgis_liblwgeom_version() INTO liblwgeomver;
	SELECT @extschema@.postgis_libxml_version() INTO libxmlver;
	SELECT @extschema@.postgis_scripts_installed() INTO dbproc;
	SELECT @extschema@.postgis_scripts_released() INTO relproc;
	SELECT @extschema@.postgis_lib_revision() INTO librev;
	BEGIN
		SELECT topology.postgis_topology_scripts_installed() INTO topo_scr_ver;
	EXCEPTION
		WHEN undefined_function OR invalid_schema_name THEN
			RAISE DEBUG 'Function postgis_topology_scripts_installed() not found. Is topology support enabled and topology.sql installed?';
		WHEN insufficient_privilege THEN
			RAISE NOTICE 'Topology support cannot be inspected. Is current user granted USAGE on schema "topology" ?';
		WHEN OTHERS THEN
			RAISE NOTICE 'Function postgis_topology_scripts_installed() could not be called: % (%)', SQLERRM, SQLSTATE;
	END;

	BEGIN
		SELECT postgis_raster_scripts_installed() INTO rast_scr_ver;
	EXCEPTION
		WHEN undefined_function THEN
			RAISE DEBUG 'Function postgis_raster_scripts_installed() not found. Is raster support enabled and rtpostgis.sql installed?';
		WHEN OTHERS THEN
			RAISE NOTICE 'Function postgis_raster_scripts_installed() could not be called: % (%)', SQLERRM, SQLSTATE;
	END;

	BEGIN
		SELECT @extschema@.postgis_raster_lib_version() INTO rast_lib_ver;
	EXCEPTION
		WHEN undefined_function THEN
			RAISE DEBUG 'Function postgis_raster_lib_version() not found. Is raster support enabled and rtpostgis.sql installed?';
		WHEN OTHERS THEN
			RAISE NOTICE 'Function postgis_raster_lib_version() could not be called: % (%)', SQLERRM, SQLSTATE;
	END;

	fullver = 'POSTGIS="' || libver;

	IF  librev IS NOT NULL THEN
		fullver = fullver || ' ' || librev;
	END IF;

	fullver = fullver || '"';

	IF EXISTS (
		SELECT * FROM pg_catalog.pg_extension
		WHERE extname = 'postgis')
	THEN
			fullver = fullver || ' [EXTENSION]';
			core_is_extension := true;
	ELSE
			core_is_extension := false;
	END IF;

	IF liblwgeomver != relproc THEN
		fullver = fullver || ' (liblwgeom version mismatch: "' || liblwgeomver || '")';
	END IF;

	fullver = fullver || ' PGSQL="' || pgsql_scr_ver || '"';
	IF pgsql_scr_ver != pgsql_ver THEN
		fullver = fullver || ' (procs need upgrade for use with PostgreSQL "' || pgsql_ver || '")';
	END IF;

	IF  geosver IS NOT NULL THEN
		fullver = fullver || ' GEOS="' || geosver || '"';
	END IF;

	IF  sfcgalver IS NOT NULL THEN
		fullver = fullver || ' SFCGAL="' || sfcgalver || '"';
	END IF;

	IF  projver IS NOT NULL THEN
		fullver = fullver || ' PROJ="' || projver || '"';
	END IF;

	IF  gdalver IS NOT NULL THEN
		fullver = fullver || ' GDAL="' || gdalver || '"';
	END IF;

	IF  libxmlver IS NOT NULL THEN
		fullver = fullver || ' LIBXML="' || libxmlver || '"';
	END IF;

	IF json_lib_ver IS NOT NULL THEN
		fullver = fullver || ' LIBJSON="' || json_lib_ver || '"';
	END IF;

	IF protobuf_lib_ver IS NOT NULL THEN
		fullver = fullver || ' LIBPROTOBUF="' || protobuf_lib_ver || '"';
	END IF;

	IF wagyu_lib_ver IS NOT NULL THEN
		fullver = fullver || ' WAGYU="' || wagyu_lib_ver || '"';
	END IF;

	IF dbproc != relproc THEN
		fullver = fullver || ' (core procs from "' || dbproc || '" need upgrade)';
	END IF;

	IF topo_scr_ver IS NOT NULL THEN
		fullver = fullver || ' TOPOLOGY';
		IF topo_scr_ver != relproc THEN
			fullver = fullver || ' (topology procs from "' || topo_scr_ver || '" need upgrade)';
		END IF;
		IF core_is_extension AND NOT EXISTS (
			SELECT * FROM pg_catalog.pg_extension
			WHERE extname = 'postgis_topology')
		THEN
				fullver = fullver || ' [UNPACKAGED!]';
		END IF;
	END IF;

	IF rast_lib_ver IS NOT NULL THEN
		fullver = fullver || ' RASTER';
		IF rast_lib_ver != relproc THEN
			fullver = fullver || ' (raster lib from "' || rast_lib_ver || '" need upgrade)';
		END IF;
		IF core_is_extension AND NOT EXISTS (
			SELECT * FROM pg_catalog.pg_extension
			WHERE extname = 'postgis_raster')
		THEN
				fullver = fullver || ' [UNPACKAGED!]';
		END IF;
	END IF;

	IF rast_scr_ver IS NOT NULL AND rast_scr_ver != relproc THEN
		fullver = fullver || ' (raster procs from "' || rast_scr_ver || '" need upgrade)';
	END IF;

	IF sfcgal_scr_ver IS NOT NULL AND sfcgal_scr_ver != relproc THEN
		fullver = fullver || ' (sfcgal procs from "' || sfcgal_scr_ver || '" need upgrade)';
	END IF;

	RETURN fullver;
END
$$
LANGUAGE 'plpgsql' IMMUTABLE;
CREATE OR REPLACE FUNCTION box2d(geometry)
	RETURNS box2d
	AS '$libdir/postgis-3','LWGEOM_to_BOX2D'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION box3d(geometry)
	RETURNS box3d
	AS '$libdir/postgis-3','LWGEOM_to_BOX3D'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION box(geometry)
	RETURNS box
	AS '$libdir/postgis-3','LWGEOM_to_BOX'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION box2d(box3d)
	RETURNS box2d
	AS '$libdir/postgis-3','BOX3D_to_BOX2D'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION box3d(box2d)
	RETURNS box3d
	AS '$libdir/postgis-3','BOX2D_to_BOX3D'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION box(box3d)
	RETURNS box
	AS '$libdir/postgis-3','BOX3D_to_BOX'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION text(geometry)
	RETURNS text
	AS '$libdir/postgis-3','LWGEOM_to_text'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION box3dtobox(box3d)
	RETURNS box
	AS '$libdir/postgis-3','BOX3D_to_BOX'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION geometry(box2d)
	RETURNS geometry
	AS '$libdir/postgis-3','BOX2D_to_LWGEOM'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION geometry(box3d)
	RETURNS geometry
	AS '$libdir/postgis-3','BOX3D_to_LWGEOM'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION geometry(text)
	RETURNS geometry
	AS '$libdir/postgis-3','parse_WKT_lwgeom'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION geometry(bytea)
	RETURNS geometry
	AS '$libdir/postgis-3','LWGEOM_from_bytea'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION bytea(geometry)
	RETURNS bytea
	AS '$libdir/postgis-3','LWGEOM_to_bytea'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (geometry AS box2d)');
DROP CAST IF EXISTS (geometry AS box2d);
CREATE CAST (geometry AS box2d) WITH FUNCTION box2d(geometry) AS IMPLICIT;
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (geometry AS box3d)');
DROP CAST IF EXISTS (geometry AS box3d);
CREATE CAST (geometry AS box3d) WITH FUNCTION box3d(geometry) AS IMPLICIT;
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (geometry AS box)');
DROP CAST IF EXISTS (geometry AS box);
CREATE CAST (geometry AS box) WITH FUNCTION box(geometry) AS ASSIGNMENT;
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (box3d AS box2d)');
DROP CAST IF EXISTS (box3d AS box2d);
CREATE CAST (box3d AS box2d) WITH FUNCTION box2d(box3d) AS IMPLICIT;
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (box2d AS box3d)');
DROP CAST IF EXISTS (box2d AS box3d);
CREATE CAST (box2d AS box3d) WITH FUNCTION box3d(box2d) AS IMPLICIT;
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (box2d AS geometry)');
DROP CAST IF EXISTS (box2d AS geometry);
CREATE CAST (box2d AS geometry) WITH FUNCTION geometry(box2d) AS IMPLICIT;
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (box3d AS box)');
DROP CAST IF EXISTS (box3d AS box);
CREATE CAST (box3d AS box) WITH FUNCTION box(box3d) AS IMPLICIT;
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (box3d AS geometry)');
DROP CAST IF EXISTS (box3d AS geometry);
CREATE CAST (box3d AS geometry) WITH FUNCTION geometry(box3d) AS IMPLICIT;
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (text AS geometry)');
DROP CAST IF EXISTS (text AS geometry);
CREATE CAST (text AS geometry) WITH FUNCTION geometry(text) AS IMPLICIT;
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (geometry AS text)');
DROP CAST IF EXISTS (geometry AS text);
CREATE CAST (geometry AS text) WITH FUNCTION text(geometry) AS IMPLICIT;
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (bytea AS geometry)');
DROP CAST IF EXISTS (bytea AS geometry);
CREATE CAST (bytea AS geometry) WITH FUNCTION geometry(bytea) AS IMPLICIT;
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (geometry AS bytea)');
DROP CAST IF EXISTS (geometry AS bytea);
CREATE CAST (geometry AS bytea) WITH FUNCTION bytea(geometry) AS IMPLICIT;
CREATE OR REPLACE FUNCTION ST_Simplify(geometry, float8)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_simplify2d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Simplify(geometry, float8, boolean)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_simplify2d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_SimplifyVW(geometry, float8)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_SetEffectiveArea'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_SetEffectiveArea(geometry,  float8 default -1, integer default 1)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_SetEffectiveArea'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_FilterByM(geometry, double precision, double precision default null, boolean default false)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_FilterByM'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_ChaikinSmoothing(geometry, integer default 1, boolean default false)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_ChaikinSmoothing'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_SnapToGrid(geometry, float8, float8, float8, float8)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_snaptogrid'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_SnapToGrid(geometry, float8, float8)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_SnapToGrid($1, 0, 0, $2, $3)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_SnapToGrid(geometry, float8)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_SnapToGrid($1, 0, 0, $2, $2)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_SnapToGrid(geom1 geometry, geom2 geometry, float8, float8, float8, float8)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_snaptogrid_pointoff'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Segmentize(geometry, float8)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_segmentize2d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_LineInterpolatePoint(geometry, float8)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_line_interpolate_point'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_LineInterpolatePoints(geometry, float8, repeat boolean DEFAULT true)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_line_interpolate_point'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_LineSubstring(geometry, float8, float8)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_line_substring'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_LineLocatePoint(geom1 geometry, geom2 geometry)
	RETURNS float8
	AS '$libdir/postgis-3', 'LWGEOM_line_locate_point'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_AddMeasure(geometry, float8, float8)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_AddMeasure'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_ClosestPointOfApproach(geometry, geometry)
	RETURNS float8
	AS '$libdir/postgis-3', 'ST_ClosestPointOfApproach'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_DistanceCPA(geometry, geometry)
	RETURNS float8
	AS '$libdir/postgis-3', 'ST_DistanceCPA'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_CPAWithin(geometry, geometry, float8)
	RETURNS bool
	AS '$libdir/postgis-3', 'ST_CPAWithin'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_IsValidTrajectory(geometry)
	RETURNS bool
	AS '$libdir/postgis-3', 'ST_IsValidTrajectory'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Intersection(geom1 geometry, geom2 geometry, gridSize float8 DEFAULT -1)
	RETURNS geometry
	AS '$libdir/postgis-3','ST_Intersection'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Buffer(geom geometry, radius float8, options text DEFAULT '')
	RETURNS geometry
	AS '$libdir/postgis-3','buffer'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Buffer(geom geometry, radius float8, quadsegs integer)
	RETURNS geometry
	AS $$ SELECT @extschema@.ST_Buffer($1, $2, CAST('quad_segs='||CAST($3 AS text) as text)) $$
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_MinimumBoundingRadius(geometry, OUT center geometry, OUT radius double precision)
	AS '$libdir/postgis-3', 'ST_MinimumBoundingRadius'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_MinimumBoundingCircle(inputgeom geometry, segs_per_quarter integer DEFAULT 48)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_MinimumBoundingCircle'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_OrientedEnvelope(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_OrientedEnvelope'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_OffsetCurve(line geometry, distance float8, params text DEFAULT '')
RETURNS geometry
	AS '$libdir/postgis-3','ST_OffsetCurve'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_GeneratePoints(area geometry, npoints integer)
RETURNS geometry
	AS '$libdir/postgis-3','ST_GeneratePoints'
	LANGUAGE 'c' VOLATILE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_GeneratePoints(area geometry, npoints integer, seed integer)
RETURNS geometry
	AS '$libdir/postgis-3','ST_GeneratePoints'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_ConvexHull(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3','convexhull'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_SimplifyPreserveTopology(geometry, float8)
	RETURNS geometry
	AS '$libdir/postgis-3','topologypreservesimplify'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_IsValidReason(geometry)
	RETURNS text
	AS '$libdir/postgis-3', 'isvalidreason'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
-- Type valid_detail -- LastUpdated: 200
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF 200 > version_from_num
     FROM _postgis_upgrade_info
  THEN
      EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE TYPE valid_detail AS (
	valid bool,
	reason varchar,
	location geometry
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION ST_IsValidDetail(geom geometry, flags int4 DEFAULT 0)
	RETURNS valid_detail
	AS '$libdir/postgis-3', 'isvaliddetail'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_IsValidReason(geometry, int4)
	RETURNS text
	AS $$
	SELECT CASE WHEN valid THEN 'Valid Geometry' ELSE reason END FROM (
		SELECT (@extschema@.ST_isValidDetail($1, $2)).*
	) foo
	$$
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_IsValid(geometry, int4)
	RETURNS boolean
	AS 'SELECT (@extschema@.ST_isValidDetail($1, $2)).valid'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_HausdorffDistance(geom1 geometry, geom2 geometry)
	RETURNS FLOAT8
	AS '$libdir/postgis-3', 'hausdorffdistance'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_HausdorffDistance(geom1 geometry, geom2 geometry, float8)
	RETURNS FLOAT8
	AS '$libdir/postgis-3', 'hausdorffdistancedensify'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_FrechetDistance(geom1 geometry, geom2 geometry, float8 default -1)
	RETURNS FLOAT8
	AS '$libdir/postgis-3', 'ST_FrechetDistance'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_MaximumInscribedCircle(geometry, OUT center geometry, OUT nearest geometry, OUT radius double precision)
	AS '$libdir/postgis-3', 'ST_MaximumInscribedCircle'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Difference(geom1 geometry, geom2 geometry, gridSize float8 DEFAULT -1.0)
	RETURNS geometry
	AS '$libdir/postgis-3','ST_Difference'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Boundary(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3','boundary'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_Points(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_Points'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_SymDifference(geom1 geometry, geom2 geometry, gridSize float8 DEFAULT -1.0)
	RETURNS geometry
	AS '$libdir/postgis-3','ST_SymDifference'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_SymmetricDifference(geom1 geometry, geom2 geometry)
	RETURNS geometry
	AS 'SELECT ST_SymDifference(geom1, geom2, -1.0);'
	LANGUAGE 'sql';
CREATE OR REPLACE FUNCTION ST_Union(geom1 geometry, geom2 geometry)
	RETURNS geometry
	AS '$libdir/postgis-3','ST_Union'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Union(geom1 geometry, geom2 geometry, gridSize float8)
	RETURNS geometry
	AS '$libdir/postgis-3','ST_Union'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_UnaryUnion(geometry, gridSize float8 DEFAULT -1.0)
	RETURNS geometry
	AS '$libdir/postgis-3','ST_UnaryUnion'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_RemoveRepeatedPoints(geom geometry, tolerance float8 default 0.0)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_RemoveRepeatedPoints'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_ClipByBox2d(geom geometry, box box2d)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_ClipByBox2d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_Subdivide(geom geometry, maxvertices integer DEFAULT 256, gridSize float8 DEFAULT -1.0)
	RETURNS setof geometry
	AS '$libdir/postgis-3', 'ST_Subdivide'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_ReducePrecision(geom geometry, gridsize float8)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_ReducePrecision'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_MakeValid(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_MakeValid'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_CleanGeometry(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_CleanGeometry'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Split(geom1 geometry, geom2 geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_Split'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_SharedPaths(geom1 geometry, geom2 geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_SharedPaths'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Snap(geom1 geometry, geom2 geometry, float8)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_Snap'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_RelateMatch(text, text)
	RETURNS bool
	AS '$libdir/postgis-3', 'ST_RelateMatch'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_Node(g geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_Node'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_DelaunayTriangles(g1 geometry, tolerance float8 DEFAULT 0.0, flags int4 DEFAULT 0)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_DelaunayTriangles'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION _ST_Voronoi(g1 geometry, clip geometry DEFAULT NULL, tolerance float8 DEFAULT 0.0, return_polygons boolean DEFAULT true)
	   RETURNS geometry
	   AS '$libdir/postgis-3', 'ST_Voronoi'
	   LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	   COST 10000;
CREATE OR REPLACE FUNCTION ST_VoronoiPolygons(g1 geometry, tolerance float8 DEFAULT 0.0, extend_to geometry DEFAULT NULL)
	   RETURNS geometry
	   AS $$ SELECT @extschema@._ST_Voronoi(g1, extend_to, tolerance, true) $$
	   LANGUAGE 'sql' IMMUTABLE PARALLEL SAFE;
CREATE OR REPLACE FUNCTION ST_VoronoiLines(g1 geometry, tolerance float8 DEFAULT 0.0, extend_to geometry DEFAULT NULL)
	   RETURNS geometry
	   AS $$ SELECT @extschema@._ST_Voronoi(g1, extend_to, tolerance, false) $$
	   LANGUAGE 'sql' IMMUTABLE PARALLEL SAFE;
CREATE OR REPLACE FUNCTION ST_CombineBBox(box3d,geometry)
	RETURNS box3d
	AS '$libdir/postgis-3', 'BOX3D_combine'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_CombineBBox(box3d,box3d)
	RETURNS box3d
	AS '$libdir/postgis-3', 'BOX3D_combine_BOX3D'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_CombineBbox(box2d,geometry)
	RETURNS box2d
	AS '$libdir/postgis-3', 'BOX2D_combine'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 1;
-- Aggregate ST_Extent(geometry) -- LastUpdated: 203
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF current_setting('server_version_num')::integer >= 120000
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OR REPLACE AGGREGATE ST_Extent(geometry) (
	sfunc = ST_CombineBBox,
	stype = box3d,
	combinefunc = ST_CombineBBox,
	parallel = safe,
	finalfunc = box2d
	);
 $postgis_proc_upgrade_parsed_def$;
  ELSIF 203 > version_from_num OR (
      203 = version_from_num AND version_from_isdev
    ) FROM _postgis_upgrade_info
  THEN
    EXECUTE 'DROP AGGREGATE IF EXISTS ST_Extent(geometry)';
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE AGGREGATE ST_Extent(geometry) (
	sfunc = ST_CombineBBox,
	stype = box3d,
	combinefunc = ST_CombineBBox,
	parallel = safe,
	finalfunc = box2d
	);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Aggregate ST_3DExtent(geometry) -- LastUpdated: 203
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF current_setting('server_version_num')::integer >= 120000
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OR REPLACE AGGREGATE ST_3DExtent(geometry)(
	sfunc = ST_CombineBBox,
	combinefunc = ST_CombineBBox,
	parallel = safe,
	stype = box3d
	);
 $postgis_proc_upgrade_parsed_def$;
  ELSIF 203 > version_from_num OR (
      203 = version_from_num AND version_from_isdev
    ) FROM _postgis_upgrade_info
  THEN
    EXECUTE 'DROP AGGREGATE IF EXISTS ST_3DExtent(geometry)';
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE AGGREGATE ST_3DExtent(geometry)(
	sfunc = ST_CombineBBox,
	combinefunc = ST_CombineBBox,
	parallel = safe,
	stype = box3d
	);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION ST_Collect(geom1 geometry, geom2 geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_collect'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 50;
-- Aggregate ST_MemCollect(geometry) -- LastUpdated: 203
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF current_setting('server_version_num')::integer >= 120000
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OR REPLACE AGGREGATE ST_MemCollect(geometry)(
	sfunc = ST_collect,
	combinefunc = ST_collect,
	parallel = safe,
	stype = geometry
	);
 $postgis_proc_upgrade_parsed_def$;
  ELSIF 203 > version_from_num OR (
      203 = version_from_num AND version_from_isdev
    ) FROM _postgis_upgrade_info
  THEN
    EXECUTE 'DROP AGGREGATE IF EXISTS ST_MemCollect(geometry)';
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE AGGREGATE ST_MemCollect(geometry)(
	sfunc = ST_collect,
	combinefunc = ST_collect,
	parallel = safe,
	stype = geometry
	);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION ST_Collect(geometry[])
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_collect_garray'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
-- Aggregate ST_MemUnion(geometry) -- LastUpdated: 203
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF current_setting('server_version_num')::integer >= 120000
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OR REPLACE AGGREGATE ST_MemUnion(geometry) (
	sfunc = ST_Union,
	combinefunc = ST_Union,
	parallel = safe,
	stype = geometry
	);
 $postgis_proc_upgrade_parsed_def$;
  ELSIF 203 > version_from_num OR (
      203 = version_from_num AND version_from_isdev
    ) FROM _postgis_upgrade_info
  THEN
    EXECUTE 'DROP AGGREGATE IF EXISTS ST_MemUnion(geometry)';
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE AGGREGATE ST_MemUnion(geometry) (
	sfunc = ST_Union,
	combinefunc = ST_Union,
	parallel = safe,
	stype = geometry
	);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION pgis_geometry_accum_transfn(internal, geometry)
	RETURNS internal
	AS '$libdir/postgis-3'
	LANGUAGE 'c' PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION pgis_geometry_accum_transfn(internal, geometry, float8)
	RETURNS internal
	AS '$libdir/postgis-3'
	LANGUAGE 'c' PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION pgis_geometry_accum_transfn(internal, geometry, float8, int)
	RETURNS internal
	AS '$libdir/postgis-3'
	LANGUAGE 'c' PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION pgis_geometry_union_finalfn(internal)
	RETURNS geometry
	AS '$libdir/postgis-3'
	LANGUAGE 'c' PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION pgis_geometry_collect_finalfn(internal)
	RETURNS geometry
	AS '$libdir/postgis-3'
	LANGUAGE 'c' PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION pgis_geometry_polygonize_finalfn(internal)
	RETURNS geometry
	AS '$libdir/postgis-3'
	LANGUAGE 'c' PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION pgis_geometry_clusterintersecting_finalfn(internal)
	RETURNS geometry[]
	AS '$libdir/postgis-3'
	LANGUAGE 'c' PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION pgis_geometry_clusterwithin_finalfn(internal)
	RETURNS geometry[]
	AS '$libdir/postgis-3'
	LANGUAGE 'c' PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION pgis_geometry_makeline_finalfn(internal)
	RETURNS geometry
	AS '$libdir/postgis-3'
	LANGUAGE 'c' PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_Union (geometry[])
	RETURNS geometry
	AS '$libdir/postgis-3','pgis_union_geometry_array'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
-- Aggregate ST_Union (geometry) -- LastUpdated: 205
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF current_setting('server_version_num')::integer >= 120000
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OR REPLACE AGGREGATE ST_Union (geometry) (
	sfunc = pgis_geometry_accum_transfn,
	stype = internal,
	parallel = safe,
	finalfunc = pgis_geometry_union_finalfn
	);
 $postgis_proc_upgrade_parsed_def$;
  ELSIF 205 > version_from_num OR (
      205 = version_from_num AND version_from_isdev
    ) FROM _postgis_upgrade_info
  THEN
    EXECUTE 'DROP AGGREGATE IF EXISTS ST_Union (geometry)';
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE AGGREGATE ST_Union (geometry) (
	sfunc = pgis_geometry_accum_transfn,
	stype = internal,
	parallel = safe,
	finalfunc = pgis_geometry_union_finalfn
	);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Aggregate ST_Union (geometry,gridSize float8) -- LastUpdated: 301
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF current_setting('server_version_num')::integer >= 120000
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OR REPLACE AGGREGATE ST_Union (geometry, gridSize float8) (
	sfunc = pgis_geometry_accum_transfn,
	stype = internal,
	parallel = safe,
	finalfunc = pgis_geometry_union_finalfn
	);
 $postgis_proc_upgrade_parsed_def$;
  ELSIF 301 > version_from_num OR (
      301 = version_from_num AND version_from_isdev
    ) FROM _postgis_upgrade_info
  THEN
    EXECUTE 'DROP AGGREGATE IF EXISTS ST_Union (geometry,gridSize float8)';
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE AGGREGATE ST_Union (geometry, gridSize float8) (
	sfunc = pgis_geometry_accum_transfn,
	stype = internal,
	parallel = safe,
	finalfunc = pgis_geometry_union_finalfn
	);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Aggregate ST_Collect (geometry) -- LastUpdated: 205
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF current_setting('server_version_num')::integer >= 120000
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OR REPLACE AGGREGATE ST_Collect (geometry) (
	SFUNC = pgis_geometry_accum_transfn,
	STYPE = internal,
	parallel = safe,
	FINALFUNC = pgis_geometry_collect_finalfn
	);
 $postgis_proc_upgrade_parsed_def$;
  ELSIF 205 > version_from_num OR (
      205 = version_from_num AND version_from_isdev
    ) FROM _postgis_upgrade_info
  THEN
    EXECUTE 'DROP AGGREGATE IF EXISTS ST_Collect (geometry)';
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE AGGREGATE ST_Collect (geometry) (
	SFUNC = pgis_geometry_accum_transfn,
	STYPE = internal,
	parallel = safe,
	FINALFUNC = pgis_geometry_collect_finalfn
	);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Aggregate ST_ClusterIntersecting (geometry) -- LastUpdated: 205
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF current_setting('server_version_num')::integer >= 120000
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OR REPLACE AGGREGATE ST_ClusterIntersecting (geometry) (
	SFUNC = pgis_geometry_accum_transfn,
	STYPE = internal,
	parallel = safe,
	FINALFUNC = pgis_geometry_clusterintersecting_finalfn
	);
 $postgis_proc_upgrade_parsed_def$;
  ELSIF 205 > version_from_num OR (
      205 = version_from_num AND version_from_isdev
    ) FROM _postgis_upgrade_info
  THEN
    EXECUTE 'DROP AGGREGATE IF EXISTS ST_ClusterIntersecting (geometry)';
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE AGGREGATE ST_ClusterIntersecting (geometry) (
	SFUNC = pgis_geometry_accum_transfn,
	STYPE = internal,
	parallel = safe,
	FINALFUNC = pgis_geometry_clusterintersecting_finalfn
	);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Aggregate ST_ClusterWithin (geometry,float8) -- LastUpdated: 205
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF current_setting('server_version_num')::integer >= 120000
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OR REPLACE AGGREGATE ST_ClusterWithin (geometry, float8) (
	SFUNC = pgis_geometry_accum_transfn,
	STYPE = internal,
	parallel = safe,
	FINALFUNC = pgis_geometry_clusterwithin_finalfn
	);
 $postgis_proc_upgrade_parsed_def$;
  ELSIF 205 > version_from_num OR (
      205 = version_from_num AND version_from_isdev
    ) FROM _postgis_upgrade_info
  THEN
    EXECUTE 'DROP AGGREGATE IF EXISTS ST_ClusterWithin (geometry,float8)';
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE AGGREGATE ST_ClusterWithin (geometry, float8) (
	SFUNC = pgis_geometry_accum_transfn,
	STYPE = internal,
	parallel = safe,
	FINALFUNC = pgis_geometry_clusterwithin_finalfn
	);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Aggregate ST_Polygonize (geometry) -- LastUpdated: 205
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF current_setting('server_version_num')::integer >= 120000
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OR REPLACE AGGREGATE ST_Polygonize (geometry) (
	SFUNC = pgis_geometry_accum_transfn,
	STYPE = internal,
	parallel = safe,
	FINALFUNC = pgis_geometry_polygonize_finalfn
	);
 $postgis_proc_upgrade_parsed_def$;
  ELSIF 205 > version_from_num OR (
      205 = version_from_num AND version_from_isdev
    ) FROM _postgis_upgrade_info
  THEN
    EXECUTE 'DROP AGGREGATE IF EXISTS ST_Polygonize (geometry)';
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE AGGREGATE ST_Polygonize (geometry) (
	SFUNC = pgis_geometry_accum_transfn,
	STYPE = internal,
	parallel = safe,
	FINALFUNC = pgis_geometry_polygonize_finalfn
	);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Aggregate ST_MakeLine (geometry) -- LastUpdated: 205
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF current_setting('server_version_num')::integer >= 120000
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OR REPLACE AGGREGATE ST_MakeLine (geometry) (
	SFUNC = pgis_geometry_accum_transfn,
	STYPE = internal,
	parallel = safe,
	FINALFUNC = pgis_geometry_makeline_finalfn
	);
 $postgis_proc_upgrade_parsed_def$;
  ELSIF 205 > version_from_num OR (
      205 = version_from_num AND version_from_isdev
    ) FROM _postgis_upgrade_info
  THEN
    EXECUTE 'DROP AGGREGATE IF EXISTS ST_MakeLine (geometry)';
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE AGGREGATE ST_MakeLine (geometry) (
	SFUNC = pgis_geometry_accum_transfn,
	STYPE = internal,
	parallel = safe,
	FINALFUNC = pgis_geometry_makeline_finalfn
	);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION ST_ClusterKMeans(geom geometry, k integer)
	RETURNS integer
	AS '$libdir/postgis-3', 'ST_ClusterKMeans'
	LANGUAGE 'c' VOLATILE STRICT WINDOW
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Relate(geom1 geometry, geom2 geometry)
	RETURNS text
	AS '$libdir/postgis-3','relate_full'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Relate(geom1 geometry, geom2 geometry, int4)
	RETURNS text
	AS '$libdir/postgis-3','relate_full'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Relate(geom1 geometry, geom2 geometry,text)
	RETURNS boolean
	AS '$libdir/postgis-3','relate_pattern'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Disjoint(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3','disjoint'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION _ST_LineCrossingDirection(line1 geometry, line2 geometry)
	RETURNS integer
	AS '$libdir/postgis-3', 'ST_LineCrossingDirection'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION _ST_DWithin(geom1 geometry, geom2 geometry,float8)
	RETURNS boolean
	AS '$libdir/postgis-3', 'LWGEOM_dwithin'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION _ST_Touches(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3','touches'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION _ST_Intersects(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3','ST_Intersects'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION _ST_Crosses(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3','crosses'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION _ST_Contains(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3','contains'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION _ST_ContainsProperly(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3','containsproperly'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION _ST_Covers(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3', 'covers'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION _ST_CoveredBy(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3', 'coveredby'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION _ST_Within(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS 'SELECT @extschema@._ST_Contains($2,$1)'
	LANGUAGE 'sql' IMMUTABLE PARALLEL SAFE;
CREATE OR REPLACE FUNCTION _ST_Overlaps(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3','overlaps'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION _ST_DFullyWithin(geom1 geometry, geom2 geometry,float8)
	RETURNS boolean
	AS '$libdir/postgis-3', 'LWGEOM_dfullywithin'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION _ST_3DDWithin(geom1 geometry, geom2 geometry,float8)
	RETURNS boolean
	AS '$libdir/postgis-3', 'LWGEOM_dwithin3d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION _ST_3DDFullyWithin(geom1 geometry, geom2 geometry,float8)
	RETURNS boolean
	AS '$libdir/postgis-3', 'LWGEOM_dfullywithin3d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION _ST_3DIntersects(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3', 'ST_3DIntersects'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION _ST_OrderingEquals(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3', 'LWGEOM_same'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION _ST_Equals(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3','ST_Equals'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION postgis_index_supportfn (internal)
	RETURNS internal
	AS '$libdir/postgis-3', 'postgis_index_supportfn'
	LANGUAGE 'c';
CREATE OR REPLACE FUNCTION ST_LineCrossingDirection(line1 geometry, line2 geometry)
	RETURNS integer
	AS '$libdir/postgis-3', 'ST_LineCrossingDirection'
	SUPPORT postgis_index_supportfn
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_DWithin(geom1 geometry, geom2 geometry,float8)
	RETURNS boolean
	AS '$libdir/postgis-3', 'LWGEOM_dwithin'
	SUPPORT postgis_index_supportfn
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Touches(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3','touches'
	SUPPORT postgis_index_supportfn
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Intersects(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3','ST_Intersects'
	SUPPORT postgis_index_supportfn
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Crosses(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3','crosses'
	SUPPORT postgis_index_supportfn
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Contains(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3','contains'
	SUPPORT postgis_index_supportfn
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_ContainsProperly(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3','containsproperly'
	SUPPORT postgis_index_supportfn
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Within(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3','within'
	SUPPORT postgis_index_supportfn
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Covers(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3', 'covers'
	SUPPORT postgis_index_supportfn
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_CoveredBy(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3', 'coveredby'
	SUPPORT postgis_index_supportfn
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Overlaps(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3','overlaps'
	SUPPORT postgis_index_supportfn
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_DFullyWithin(geom1 geometry, geom2 geometry,float8)
	RETURNS boolean
	AS '$libdir/postgis-3', 'LWGEOM_dfullywithin'
	SUPPORT postgis_index_supportfn
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_3DDWithin(geom1 geometry, geom2 geometry,float8)
	RETURNS boolean
	AS '$libdir/postgis-3', 'LWGEOM_dwithin3d'
	SUPPORT postgis_index_supportfn
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_3DDFullyWithin(geom1 geometry, geom2 geometry,float8)
	RETURNS boolean
	AS '$libdir/postgis-3', 'LWGEOM_dfullywithin3d'
	SUPPORT postgis_index_supportfn
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_3DIntersects(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3', 'ST_3DIntersects'
	SUPPORT postgis_index_supportfn
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_OrderingEquals(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3', 'LWGEOM_same'
	SUPPORT postgis_index_supportfn
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Equals(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3','ST_Equals'
	SUPPORT postgis_index_supportfn
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_IsValid(geometry)
	RETURNS boolean
	AS '$libdir/postgis-3', 'isvalid'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_MinimumClearance(geometry)
	RETURNS float8
	AS '$libdir/postgis-3', 'ST_MinimumClearance'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_MinimumClearanceLine(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_MinimumClearanceLine'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Centroid(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'centroid'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_GeometricMedian(g geometry, tolerance float8 DEFAULT NULL, max_iter int DEFAULT 10000, fail_if_not_converged boolean DEFAULT false)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_GeometricMedian'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_IsRing(geometry)
	RETURNS boolean
	AS '$libdir/postgis-3', 'isring'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_PointOnSurface(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'pointonsurface'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_IsSimple(geometry)
	RETURNS boolean
	AS '$libdir/postgis-3', 'issimple'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_IsCollection(geometry)
	RETURNS boolean
	AS '$libdir/postgis-3', 'ST_IsCollection'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION Equals(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3','ST_Equals'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION _ST_GeomFromGML(text, int4)
	RETURNS geometry
	AS '$libdir/postgis-3','geom_from_gml'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_GeomFromGML(text, int4)
	RETURNS geometry
	AS '$libdir/postgis-3','geom_from_gml'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_GeomFromGML(text)
	RETURNS geometry
	AS 'SELECT @extschema@._ST_GeomFromGML($1, 0)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_GMLToSQL(text)
	RETURNS geometry
	AS 'SELECT @extschema@._ST_GeomFromGML($1, 0)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_GMLToSQL(text, int4)
	RETURNS geometry
	AS '$libdir/postgis-3','geom_from_gml'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_GeomFromKML(text)
	RETURNS geometry
	AS '$libdir/postgis-3','geom_from_kml'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_GeomFromGeoJson(text)
	RETURNS geometry
	AS '$libdir/postgis-3','geom_from_geojson'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_GeomFromGeoJson(json)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_GeomFromGeoJson($1::text)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_GeomFromGeoJson(jsonb)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_GeomFromGeoJson($1::text)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION postgis_libjson_version()
	RETURNS text
	AS '$libdir/postgis-3','postgis_libjson_version'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION ST_LineFromEncodedPolyline(txtin text, nprecision int4 DEFAULT 5)
	RETURNS geometry
	AS '$libdir/postgis-3','line_from_encoded_polyline'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_AsEncodedPolyline(geom geometry, nprecision int4 DEFAULT 5)
	RETURNS TEXT
	AS '$libdir/postgis-3','LWGEOM_asEncodedPolyline'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_AsSVG(geom geometry, rel int4 DEFAULT 0, maxdecimaldigits int4 DEFAULT 15)
	RETURNS TEXT
	AS '$libdir/postgis-3','LWGEOM_asSVG'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION _ST_AsGML(int4, geometry, int4, int4, text, text)
	RETURNS TEXT
	AS '$libdir/postgis-3','LWGEOM_asGML'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_AsGML(geom geometry, maxdecimaldigits int4 DEFAULT 15, options int4 DEFAULT 0)
	RETURNS TEXT
	AS '$libdir/postgis-3','LWGEOM_asGML'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_AsGML(version int4, geom geometry, maxdecimaldigits int4 DEFAULT 15, options int4 DEFAULT 0, nprefix text DEFAULT null, id text DEFAULT null)
	RETURNS TEXT
	AS '$libdir/postgis-3','LWGEOM_asGML'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_AsKML(geom geometry, maxdecimaldigits int4 DEFAULT 15, nprefix TEXT default '')
	RETURNS TEXT
	AS '$libdir/postgis-3','LWGEOM_asKML'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_AsGeoJson(geom geometry, maxdecimaldigits int4 DEFAULT 9, options int4 DEFAULT 8)
	RETURNS text
	AS '$libdir/postgis-3','LWGEOM_asGeoJson'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_AsGeoJson(r record, geom_column text DEFAULT '', maxdecimaldigits int4 DEFAULT 9, pretty_bool bool DEFAULT false)
	RETURNS text
	AS '$libdir/postgis-3','ST_AsGeoJsonRow'
	LANGUAGE 'c' STABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION json(geometry)
	RETURNS json
	AS '$libdir/postgis-3','geometry_to_json'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION jsonb(geometry)
	RETURNS jsonb
	AS '$libdir/postgis-3','geometry_to_jsonb'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (geometry AS json)');
DROP CAST IF EXISTS (geometry AS json);
CREATE CAST (geometry AS json) WITH FUNCTION json(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (geometry AS jsonb)');
DROP CAST IF EXISTS (geometry AS jsonb);
CREATE CAST (geometry AS jsonb) WITH FUNCTION jsonb(geometry);
CREATE OR REPLACE FUNCTION pgis_asmvt_transfn(internal, anyelement)
	RETURNS internal
	AS '$libdir/postgis-3', 'pgis_asmvt_transfn'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION pgis_asmvt_transfn(internal, anyelement, text)
	RETURNS internal
	AS '$libdir/postgis-3', 'pgis_asmvt_transfn'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION pgis_asmvt_transfn(internal, anyelement, text, int4)
	RETURNS internal
	AS '$libdir/postgis-3', 'pgis_asmvt_transfn'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION pgis_asmvt_transfn(internal, anyelement, text, int4, text)
	RETURNS internal
	AS '$libdir/postgis-3', 'pgis_asmvt_transfn'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION pgis_asmvt_transfn(internal, anyelement, text, int4, text, text)
	RETURNS internal
	AS '$libdir/postgis-3', 'pgis_asmvt_transfn'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION pgis_asmvt_finalfn(internal)
	RETURNS bytea
	AS '$libdir/postgis-3', 'pgis_asmvt_finalfn'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION pgis_asmvt_combinefn(internal, internal)
	RETURNS internal
	AS '$libdir/postgis-3', 'pgis_asmvt_combinefn'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION pgis_asmvt_serialfn(internal)
	RETURNS bytea
	AS '$libdir/postgis-3', 'pgis_asmvt_serialfn'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION pgis_asmvt_deserialfn(bytea, internal)
	RETURNS internal
	AS '$libdir/postgis-3', 'pgis_asmvt_deserialfn'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 500;
-- Aggregate ST_AsMVT(anyelement) -- LastUpdated: 205
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF current_setting('server_version_num')::integer >= 120000
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OR REPLACE AGGREGATE ST_AsMVT(anyelement)
(
	sfunc = pgis_asmvt_transfn,
	stype = internal,
	parallel = safe,
	serialfunc = pgis_asmvt_serialfn,
	deserialfunc = pgis_asmvt_deserialfn,
	combinefunc = pgis_asmvt_combinefn,
	finalfunc = pgis_asmvt_finalfn
);
 $postgis_proc_upgrade_parsed_def$;
  ELSIF 205 > version_from_num OR (
      205 = version_from_num AND version_from_isdev
    ) FROM _postgis_upgrade_info
  THEN
    EXECUTE 'DROP AGGREGATE IF EXISTS ST_AsMVT(anyelement)';
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE AGGREGATE ST_AsMVT(anyelement)
(
	sfunc = pgis_asmvt_transfn,
	stype = internal,
	parallel = safe,
	serialfunc = pgis_asmvt_serialfn,
	deserialfunc = pgis_asmvt_deserialfn,
	combinefunc = pgis_asmvt_combinefn,
	finalfunc = pgis_asmvt_finalfn
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Aggregate ST_AsMVT(anyelement,text) -- LastUpdated: 205
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF current_setting('server_version_num')::integer >= 120000
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OR REPLACE AGGREGATE ST_AsMVT(anyelement, text)
(
	sfunc = pgis_asmvt_transfn,
	stype = internal,
	parallel = safe,
	serialfunc = pgis_asmvt_serialfn,
	deserialfunc = pgis_asmvt_deserialfn,
	combinefunc = pgis_asmvt_combinefn,
	finalfunc = pgis_asmvt_finalfn
);
 $postgis_proc_upgrade_parsed_def$;
  ELSIF 205 > version_from_num OR (
      205 = version_from_num AND version_from_isdev
    ) FROM _postgis_upgrade_info
  THEN
    EXECUTE 'DROP AGGREGATE IF EXISTS ST_AsMVT(anyelement,text)';
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE AGGREGATE ST_AsMVT(anyelement, text)
(
	sfunc = pgis_asmvt_transfn,
	stype = internal,
	parallel = safe,
	serialfunc = pgis_asmvt_serialfn,
	deserialfunc = pgis_asmvt_deserialfn,
	combinefunc = pgis_asmvt_combinefn,
	finalfunc = pgis_asmvt_finalfn
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Aggregate ST_AsMVT(anyelement,text,int4) -- LastUpdated: 205
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF current_setting('server_version_num')::integer >= 120000
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OR REPLACE AGGREGATE ST_AsMVT(anyelement, text, int4)
(
	sfunc = pgis_asmvt_transfn,
	stype = internal,
	parallel = safe,
	serialfunc = pgis_asmvt_serialfn,
	deserialfunc = pgis_asmvt_deserialfn,
	combinefunc = pgis_asmvt_combinefn,
	finalfunc = pgis_asmvt_finalfn
);
 $postgis_proc_upgrade_parsed_def$;
  ELSIF 205 > version_from_num OR (
      205 = version_from_num AND version_from_isdev
    ) FROM _postgis_upgrade_info
  THEN
    EXECUTE 'DROP AGGREGATE IF EXISTS ST_AsMVT(anyelement,text,int4)';
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE AGGREGATE ST_AsMVT(anyelement, text, int4)
(
	sfunc = pgis_asmvt_transfn,
	stype = internal,
	parallel = safe,
	serialfunc = pgis_asmvt_serialfn,
	deserialfunc = pgis_asmvt_deserialfn,
	combinefunc = pgis_asmvt_combinefn,
	finalfunc = pgis_asmvt_finalfn
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Aggregate ST_AsMVT(anyelement,text,int4,text) -- LastUpdated: 205
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF current_setting('server_version_num')::integer >= 120000
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OR REPLACE AGGREGATE ST_AsMVT(anyelement, text, int4, text)
(
	sfunc = pgis_asmvt_transfn,
	stype = internal,
	parallel = safe,
	serialfunc = pgis_asmvt_serialfn,
	deserialfunc = pgis_asmvt_deserialfn,
	combinefunc = pgis_asmvt_combinefn,
	finalfunc = pgis_asmvt_finalfn
);
 $postgis_proc_upgrade_parsed_def$;
  ELSIF 205 > version_from_num OR (
      205 = version_from_num AND version_from_isdev
    ) FROM _postgis_upgrade_info
  THEN
    EXECUTE 'DROP AGGREGATE IF EXISTS ST_AsMVT(anyelement,text,int4,text)';
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE AGGREGATE ST_AsMVT(anyelement, text, int4, text)
(
	sfunc = pgis_asmvt_transfn,
	stype = internal,
	parallel = safe,
	serialfunc = pgis_asmvt_serialfn,
	deserialfunc = pgis_asmvt_deserialfn,
	combinefunc = pgis_asmvt_combinefn,
	finalfunc = pgis_asmvt_finalfn
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Aggregate ST_AsMVT(anyelement,text,int4,text,text) -- LastUpdated: 300
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF current_setting('server_version_num')::integer >= 120000
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OR REPLACE AGGREGATE ST_AsMVT(anyelement, text, int4, text, text)
(
	sfunc = pgis_asmvt_transfn,
	stype = internal,
	parallel = safe,
	serialfunc = pgis_asmvt_serialfn,
	deserialfunc = pgis_asmvt_deserialfn,
	combinefunc = pgis_asmvt_combinefn,
	finalfunc = pgis_asmvt_finalfn
);
 $postgis_proc_upgrade_parsed_def$;
  ELSIF 300 > version_from_num OR (
      300 = version_from_num AND version_from_isdev
    ) FROM _postgis_upgrade_info
  THEN
    EXECUTE 'DROP AGGREGATE IF EXISTS ST_AsMVT(anyelement,text,int4,text,text)';
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE AGGREGATE ST_AsMVT(anyelement, text, int4, text, text)
(
	sfunc = pgis_asmvt_transfn,
	stype = internal,
	parallel = safe,
	serialfunc = pgis_asmvt_serialfn,
	deserialfunc = pgis_asmvt_deserialfn,
	combinefunc = pgis_asmvt_combinefn,
	finalfunc = pgis_asmvt_finalfn
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION ST_AsMVTGeom(geom geometry, bounds box2d, extent int4 default 4096, buffer int4 default 256, clip_geom bool default true)
	RETURNS geometry
	AS '$libdir/postgis-3','ST_AsMVTGeom'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION postgis_libprotobuf_version()
	RETURNS text
	AS '$libdir/postgis-3','postgis_libprotobuf_version'
	LANGUAGE 'c' IMMUTABLE STRICT;
CREATE OR REPLACE FUNCTION pgis_asgeobuf_transfn(internal, anyelement)
	RETURNS internal
	AS '$libdir/postgis-3', 'pgis_asgeobuf_transfn'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION pgis_asgeobuf_transfn(internal, anyelement, text)
	RETURNS internal
	AS '$libdir/postgis-3', 'pgis_asgeobuf_transfn'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION pgis_asgeobuf_finalfn(internal)
	RETURNS bytea
	AS '$libdir/postgis-3', 'pgis_asgeobuf_finalfn'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 500;
-- Aggregate ST_AsGeobuf(anyelement) -- LastUpdated: 204
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF current_setting('server_version_num')::integer >= 120000
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OR REPLACE AGGREGATE ST_AsGeobuf(anyelement)
(
	sfunc = pgis_asgeobuf_transfn,
	stype = internal,
	parallel = safe,
	finalfunc = pgis_asgeobuf_finalfn
);
 $postgis_proc_upgrade_parsed_def$;
  ELSIF 204 > version_from_num OR (
      204 = version_from_num AND version_from_isdev
    ) FROM _postgis_upgrade_info
  THEN
    EXECUTE 'DROP AGGREGATE IF EXISTS ST_AsGeobuf(anyelement)';
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE AGGREGATE ST_AsGeobuf(anyelement)
(
	sfunc = pgis_asgeobuf_transfn,
	stype = internal,
	parallel = safe,
	finalfunc = pgis_asgeobuf_finalfn
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Aggregate ST_AsGeobuf(anyelement,text) -- LastUpdated: 204
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF current_setting('server_version_num')::integer >= 120000
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OR REPLACE AGGREGATE ST_AsGeobuf(anyelement, text)
(
	sfunc = pgis_asgeobuf_transfn,
	stype = internal,
	parallel = safe,
	finalfunc = pgis_asgeobuf_finalfn
);
 $postgis_proc_upgrade_parsed_def$;
  ELSIF 204 > version_from_num OR (
      204 = version_from_num AND version_from_isdev
    ) FROM _postgis_upgrade_info
  THEN
    EXECUTE 'DROP AGGREGATE IF EXISTS ST_AsGeobuf(anyelement,text)';
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE AGGREGATE ST_AsGeobuf(anyelement, text)
(
	sfunc = pgis_asgeobuf_transfn,
	stype = internal,
	parallel = safe,
	finalfunc = pgis_asgeobuf_finalfn
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION ST_GeoHash(geom geometry, maxchars int4 DEFAULT 0)
	RETURNS TEXT
	AS '$libdir/postgis-3', 'ST_GeoHash'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION _ST_SortableHash(geom geometry)
	RETURNS bigint
	AS '$libdir/postgis-3', '_ST_SortableHash'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Box2dFromGeoHash(text, int4 DEFAULT NULL)
	RETURNS box2d
	AS '$libdir/postgis-3','box2d_from_geohash'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_PointFromGeoHash(text, int4 DEFAULT NULL)
	RETURNS geometry
	AS '$libdir/postgis-3','point_from_geohash'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_GeomFromGeoHash(text, int4 DEFAULT NULL)
	RETURNS geometry
	AS $$ SELECT CAST(@extschema@.ST_Box2dFromGeoHash($1, $2) AS geometry); $$
	LANGUAGE 'sql' IMMUTABLE PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_NumPoints(geometry)
	RETURNS int4
	AS '$libdir/postgis-3', 'LWGEOM_numpoints_linestring'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_NumGeometries(geometry)
	RETURNS int4
	AS '$libdir/postgis-3', 'LWGEOM_numgeometries_collection'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_GeometryN(geometry,integer)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_geometryn_collection'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Dimension(geometry)
	RETURNS int4
	AS '$libdir/postgis-3', 'LWGEOM_dimension'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_ExteriorRing(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3','LWGEOM_exteriorring_polygon'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_NumInteriorRings(geometry)
	RETURNS integer
	AS '$libdir/postgis-3','LWGEOM_numinteriorrings_polygon'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_NumInteriorRing(geometry)
	RETURNS integer
	AS '$libdir/postgis-3','LWGEOM_numinteriorrings_polygon'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_InteriorRingN(geometry,integer)
	RETURNS geometry
	AS '$libdir/postgis-3','LWGEOM_interiorringn_polygon'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION GeometryType(geometry)
	RETURNS text
	AS '$libdir/postgis-3', 'LWGEOM_getTYPE'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION ST_GeometryType(geometry)
	RETURNS text
	AS '$libdir/postgis-3', 'geometry_geometrytype'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION ST_PointN(geometry,integer)
	RETURNS geometry
	AS '$libdir/postgis-3','LWGEOM_pointn_linestring'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_NumPatches(geometry)
	RETURNS int4
	AS '
	SELECT CASE WHEN @extschema@.ST_GeometryType($1) = ''ST_PolyhedralSurface''
	THEN @extschema@.ST_NumGeometries($1)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_PatchN(geometry, integer)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.ST_GeometryType($1) = ''ST_PolyhedralSurface''
	THEN @extschema@.ST_GeometryN($1, $2)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_StartPoint(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_startpoint_linestring'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_EndPoint(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_endpoint_linestring'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_IsClosed(geometry)
	RETURNS boolean
	AS '$libdir/postgis-3', 'LWGEOM_isclosed'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_IsEmpty(geometry)
	RETURNS boolean
	AS '$libdir/postgis-3', 'LWGEOM_isempty'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_AsBinary(geometry,text)
	RETURNS bytea
	AS '$libdir/postgis-3','LWGEOM_asBinary'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_AsBinary(geometry)
	RETURNS bytea
	AS '$libdir/postgis-3','LWGEOM_asBinary'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_AsText(geometry)
	RETURNS TEXT
	AS '$libdir/postgis-3','LWGEOM_asText'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_AsText(geometry, int4)
	RETURNS TEXT
	AS '$libdir/postgis-3','LWGEOM_asText'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_GeometryFromText(text)
	RETURNS geometry
	AS '$libdir/postgis-3','LWGEOM_from_text'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_GeometryFromText(text, int4)
	RETURNS geometry
	AS '$libdir/postgis-3','LWGEOM_from_text'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_GeomFromText(text)
	RETURNS geometry
	AS '$libdir/postgis-3','LWGEOM_from_text'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_GeomFromText(text, int4)
	RETURNS geometry
	AS '$libdir/postgis-3','LWGEOM_from_text'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_WKTToSQL(text)
	RETURNS geometry
	AS '$libdir/postgis-3','LWGEOM_from_text'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_PointFromText(text)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromText($1)) = ''POINT''
	THEN @extschema@.ST_GeomFromText($1)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_PointFromText(text, int4)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromText($1, $2)) = ''POINT''
	THEN @extschema@.ST_GeomFromText($1, $2)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_LineFromText(text)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromText($1)) = ''LINESTRING''
	THEN @extschema@.ST_GeomFromText($1)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_LineFromText(text, int4)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromText($1, $2)) = ''LINESTRING''
	THEN @extschema@.ST_GeomFromText($1,$2)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_PolyFromText(text)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromText($1)) = ''POLYGON''
	THEN @extschema@.ST_GeomFromText($1)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_PolyFromText(text, int4)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromText($1, $2)) = ''POLYGON''
	THEN @extschema@.ST_GeomFromText($1, $2)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_PolygonFromText(text, int4)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_PolyFromText($1, $2)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_PolygonFromText(text)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_PolyFromText($1)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_MLineFromText(text, int4)
	RETURNS geometry
	AS '
	SELECT CASE
	WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromText($1, $2)) = ''MULTILINESTRING''
	THEN @extschema@.ST_GeomFromText($1,$2)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_MLineFromText(text)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromText($1)) = ''MULTILINESTRING''
	THEN @extschema@.ST_GeomFromText($1)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_MultiLineStringFromText(text)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_MLineFromText($1)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_MultiLineStringFromText(text, int4)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_MLineFromText($1, $2)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_MPointFromText(text, int4)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromText($1, $2)) = ''MULTIPOINT''
	THEN ST_GeomFromText($1, $2)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_MPointFromText(text)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromText($1)) = ''MULTIPOINT''
	THEN @extschema@.ST_GeomFromText($1)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_MultiPointFromText(text)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_MPointFromText($1)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_MPolyFromText(text, int4)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromText($1, $2)) = ''MULTIPOLYGON''
	THEN @extschema@.ST_GeomFromText($1,$2)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_MPolyFromText(text)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromText($1)) = ''MULTIPOLYGON''
	THEN @extschema@.ST_GeomFromText($1)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_MultiPolygonFromText(text, int4)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_MPolyFromText($1, $2)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_MultiPolygonFromText(text)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_MPolyFromText($1)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_GeomCollFromText(text, int4)
	RETURNS geometry
	AS '
	SELECT CASE
	WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromText($1, $2)) = ''GEOMETRYCOLLECTION''
	THEN @extschema@.ST_GeomFromText($1,$2)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_GeomCollFromText(text)
	RETURNS geometry
	AS '
	SELECT CASE
	WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromText($1)) = ''GEOMETRYCOLLECTION''
	THEN @extschema@.ST_GeomFromText($1)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_GeomFromWKB(bytea)
	RETURNS geometry
	AS '$libdir/postgis-3','LWGEOM_from_WKB'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_GeomFromWKB(bytea, int)
	RETURNS geometry
	AS 'SELECT @extschema@.ST_SetSRID(@extschema@.ST_GeomFromWKB($1), $2)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_PointFromWKB(bytea, int)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromWKB($1, $2)) = ''POINT''
	THEN @extschema@.ST_GeomFromWKB($1, $2)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_PointFromWKB(bytea)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromWKB($1)) = ''POINT''
	THEN @extschema@.ST_GeomFromWKB($1)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_LineFromWKB(bytea, int)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromWKB($1, $2)) = ''LINESTRING''
	THEN @extschema@.ST_GeomFromWKB($1, $2)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_LineFromWKB(bytea)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromWKB($1)) = ''LINESTRING''
	THEN @extschema@.ST_GeomFromWKB($1)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_LinestringFromWKB(bytea, int)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromWKB($1, $2)) = ''LINESTRING''
	THEN @extschema@.ST_GeomFromWKB($1, $2)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_LinestringFromWKB(bytea)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromWKB($1)) = ''LINESTRING''
	THEN @extschema@.ST_GeomFromWKB($1)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_PolyFromWKB(bytea, int)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromWKB($1, $2)) = ''POLYGON''
	THEN @extschema@.ST_GeomFromWKB($1, $2)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_PolyFromWKB(bytea)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromWKB($1)) = ''POLYGON''
	THEN @extschema@.ST_GeomFromWKB($1)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_PolygonFromWKB(bytea, int)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromWKB($1,$2)) = ''POLYGON''
	THEN @extschema@.ST_GeomFromWKB($1, $2)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_PolygonFromWKB(bytea)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromWKB($1)) = ''POLYGON''
	THEN @extschema@.ST_GeomFromWKB($1)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_MPointFromWKB(bytea, int)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromWKB($1, $2)) = ''MULTIPOINT''
	THEN @extschema@.ST_GeomFromWKB($1, $2)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_MPointFromWKB(bytea)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromWKB($1)) = ''MULTIPOINT''
	THEN @extschema@.ST_GeomFromWKB($1)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_MultiPointFromWKB(bytea, int)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromWKB($1,$2)) = ''MULTIPOINT''
	THEN @extschema@.ST_GeomFromWKB($1, $2)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_MultiPointFromWKB(bytea)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromWKB($1)) = ''MULTIPOINT''
	THEN @extschema@.ST_GeomFromWKB($1)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_MultiLineFromWKB(bytea)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromWKB($1)) = ''MULTILINESTRING''
	THEN @extschema@.ST_GeomFromWKB($1)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_MLineFromWKB(bytea, int)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromWKB($1, $2)) = ''MULTILINESTRING''
	THEN @extschema@.ST_GeomFromWKB($1, $2)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_MLineFromWKB(bytea)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromWKB($1)) = ''MULTILINESTRING''
	THEN @extschema@.ST_GeomFromWKB($1)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_MPolyFromWKB(bytea, int)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromWKB($1, $2)) = ''MULTIPOLYGON''
	THEN @extschema@.ST_GeomFromWKB($1, $2)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_MPolyFromWKB(bytea)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromWKB($1)) = ''MULTIPOLYGON''
	THEN @extschema@.ST_GeomFromWKB($1)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION ST_MultiPolyFromWKB(bytea, int)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromWKB($1, $2)) = ''MULTIPOLYGON''
	THEN @extschema@.ST_GeomFromWKB($1, $2)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_MultiPolyFromWKB(bytea)
	RETURNS geometry
	AS '
	SELECT CASE WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromWKB($1)) = ''MULTIPOLYGON''
	THEN @extschema@.ST_GeomFromWKB($1)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_GeomCollFromWKB(bytea, int)
	RETURNS geometry
	AS '
	SELECT CASE
	WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromWKB($1, $2)) = ''GEOMETRYCOLLECTION''
	THEN @extschema@.ST_GeomFromWKB($1, $2)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_GeomCollFromWKB(bytea)
	RETURNS geometry
	AS '
	SELECT CASE
	WHEN @extschema@.geometrytype(@extschema@.ST_GeomFromWKB($1)) = ''GEOMETRYCOLLECTION''
	THEN @extschema@.ST_GeomFromWKB($1)
	ELSE NULL END
	'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION _ST_MaxDistance(geom1 geometry, geom2 geometry)
	RETURNS float8
	AS '$libdir/postgis-3', 'LWGEOM_maxdistance2d_linestring'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_MaxDistance(geom1 geometry, geom2 geometry)
	RETURNS float8
	AS 'SELECT @extschema@._ST_MaxDistance(@extschema@.ST_ConvexHull($1), @extschema@.ST_ConvexHull($2))'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_ClosestPoint(geom1 geometry, geom2 geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_closestpoint'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_ShortestLine(geom1 geometry, geom2 geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_shortestline2d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION _ST_LongestLine(geom1 geometry, geom2 geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_longestline2d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_LongestLine(geom1 geometry, geom2 geometry)
	RETURNS geometry
	AS 'SELECT @extschema@._ST_LongestLine(@extschema@.ST_ConvexHull($1), @extschema@.ST_ConvexHull($2))'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_SwapOrdinates(geom geometry, ords cstring)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_SwapOrdinates'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_FlipCoordinates(geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_FlipCoordinates'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_BdPolyFromText(text, integer)
RETURNS geometry
AS $$
DECLARE
	geomtext alias for $1;
	srid alias for $2;
	mline @extschema@.geometry;
	geom @extschema@.geometry;
BEGIN
	mline := @extschema@.ST_MultiLineStringFromText(geomtext, srid);

	IF mline IS NULL
	THEN
		RAISE EXCEPTION 'Input is not a MultiLinestring';
	END IF;

	geom := @extschema@.ST_BuildArea(mline);

	IF @extschema@.GeometryType(geom) != 'POLYGON'
	THEN
		RAISE EXCEPTION 'Input returns more then a single polygon, try using BdMPolyFromText instead';
	END IF;

	RETURN geom;
END;
$$
LANGUAGE 'plpgsql' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION ST_BdMPolyFromText(text, integer)
RETURNS geometry
AS $$
DECLARE
	geomtext alias for $1;
	srid alias for $2;
	mline @extschema@.geometry;
	geom @extschema@.geometry;
BEGIN
	mline := @extschema@.ST_MultiLineStringFromText(geomtext, srid);

	IF mline IS NULL
	THEN
		RAISE EXCEPTION 'Input is not a MultiLinestring';
	END IF;

	geom := @extschema@.ST_Multi(@extschema@.ST_BuildArea(mline));

	RETURN geom;
END;
$$
LANGUAGE 'plpgsql' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION UnlockRows(text)
	RETURNS int
	AS $$
DECLARE
	ret int;
BEGIN

	IF NOT LongTransactionsEnabled() THEN
		RAISE EXCEPTION 'Long transaction support disabled, use EnableLongTransaction() to enable.';
	END IF;

	EXECUTE 'DELETE FROM authorization_table where authid = ' ||
		quote_literal($1);

	GET DIAGNOSTICS ret = ROW_COUNT;

	RETURN ret;
END;
$$
LANGUAGE 'plpgsql'  VOLATILE STRICT;
CREATE OR REPLACE FUNCTION LockRow(text, text, text, text, timestamp)
	RETURNS int
	AS $$
DECLARE
	myschema alias for $1;
	mytable alias for $2;
	myrid   alias for $3;
	authid alias for $4;
	expires alias for $5;
	ret int;
	mytoid oid;
	myrec RECORD;

BEGIN

	IF NOT LongTransactionsEnabled() THEN
		RAISE EXCEPTION 'Long transaction support disabled, use EnableLongTransaction() to enable.';
	END IF;

	EXECUTE 'DELETE FROM authorization_table WHERE expires < now()';

	SELECT c.oid INTO mytoid FROM pg_class c, pg_namespace n
		WHERE c.relname = mytable
		AND c.relnamespace = n.oid
		AND n.nspname = myschema;

	-- RAISE NOTICE 'toid: %', mytoid;

	FOR myrec IN SELECT * FROM authorization_table WHERE
		toid = mytoid AND rid = myrid
	LOOP
		IF myrec.authid != authid THEN
			RETURN 0;
		ELSE
			RETURN 1;
		END IF;
	END LOOP;

	EXECUTE 'INSERT INTO authorization_table VALUES ('||
		quote_literal(mytoid::text)||','||quote_literal(myrid)||
		','||quote_literal(expires::text)||
		','||quote_literal(authid) ||')';

	GET DIAGNOSTICS ret = ROW_COUNT;

	RETURN ret;
END;
$$
LANGUAGE 'plpgsql'  VOLATILE STRICT;
CREATE OR REPLACE FUNCTION LockRow(text, text, text, text)
	RETURNS int
	AS
$$ SELECT LockRow($1, $2, $3, $4, now()::timestamp+'1:00'); $$
	LANGUAGE 'sql'  VOLATILE STRICT;
CREATE OR REPLACE FUNCTION LockRow(text, text, text)
	RETURNS int
	AS
$$ SELECT LockRow(current_schema(), $1, $2, $3, now()::timestamp+'1:00'); $$
	LANGUAGE 'sql'  VOLATILE STRICT;
CREATE OR REPLACE FUNCTION LockRow(text, text, text, timestamp)
	RETURNS int
	AS
$$ SELECT LockRow(current_schema(), $1, $2, $3, $4); $$
	LANGUAGE 'sql'  VOLATILE STRICT;
CREATE OR REPLACE FUNCTION AddAuth(text)
	RETURNS BOOLEAN
	AS $$
DECLARE
	lockid alias for $1;
	okay boolean;
	myrec record;
BEGIN
	-- check to see if table exists
	--  if not, CREATE TEMP TABLE mylock (transid xid, lockcode text)
	okay := 'f';
	FOR myrec IN SELECT * FROM pg_class WHERE relname = 'temp_lock_have_table' LOOP
		okay := 't';
	END LOOP;
	IF (okay <> 't') THEN
		CREATE TEMP TABLE temp_lock_have_table (transid xid, lockcode text);
			-- this will only work from pgsql7.4 up
			-- ON COMMIT DELETE ROWS;
	END IF;

	--  INSERT INTO mylock VALUES ( $1)
--	EXECUTE 'INSERT INTO temp_lock_have_table VALUES ( '||
--		quote_literal(getTransactionID()) || ',' ||
--		quote_literal(lockid) ||')';

	INSERT INTO temp_lock_have_table VALUES (getTransactionID(), lockid);

	RETURN true::boolean;
END;
$$
LANGUAGE PLPGSQL;
CREATE OR REPLACE FUNCTION CheckAuth(text, text, text)
	RETURNS INT
	AS $$
DECLARE
	schema text;
BEGIN
	IF NOT LongTransactionsEnabled() THEN
		RAISE EXCEPTION 'Long transaction support disabled, use EnableLongTransaction() to enable.';
	END IF;

	if ( $1 != '' ) THEN
		schema = $1;
	ELSE
		SELECT current_schema() into schema;
	END IF;

	-- TODO: check for an already existing trigger ?

	EXECUTE 'CREATE TRIGGER check_auth BEFORE UPDATE OR DELETE ON '
		|| quote_ident(schema) || '.' || quote_ident($2)
		||' FOR EACH ROW EXECUTE PROCEDURE CheckAuthTrigger('
		|| quote_literal($3) || ')';

	RETURN 0;
END;
$$
LANGUAGE 'plpgsql';
CREATE OR REPLACE FUNCTION CheckAuth(text, text)
	RETURNS INT
	AS
	$$ SELECT CheckAuth('', $1, $2) $$
	LANGUAGE 'sql';
CREATE OR REPLACE FUNCTION CheckAuthTrigger()
	RETURNS trigger AS
	'$libdir/postgis-3', 'check_authorization'
	LANGUAGE C;
CREATE OR REPLACE FUNCTION GetTransactionID()
	RETURNS xid AS
	'$libdir/postgis-3', 'getTransactionID'
	LANGUAGE C;
CREATE OR REPLACE FUNCTION EnableLongTransactions()
	RETURNS TEXT
	AS $$
DECLARE
	"query" text;
	exists bool;
	rec RECORD;

BEGIN

	exists = 'f';
	FOR rec IN SELECT * FROM pg_class WHERE relname = 'authorization_table'
	LOOP
		exists = 't';
	END LOOP;

	IF NOT exists
	THEN
		"query" = 'CREATE TABLE authorization_table (
			toid oid, -- table oid
			rid text, -- row id
			expires timestamp,
			authid text
		)';
		EXECUTE "query";
	END IF;

	exists = 'f';
	FOR rec IN SELECT * FROM pg_class WHERE relname = 'authorized_tables'
	LOOP
		exists = 't';
	END LOOP;

	IF NOT exists THEN
		"query" = 'CREATE VIEW authorized_tables AS ' ||
			'SELECT ' ||
			'n.nspname as schema, ' ||
			'c.relname as table, trim(' ||
			quote_literal(chr(92) || '000') ||
			' from t.tgargs) as id_column ' ||
			'FROM pg_trigger t, pg_class c, pg_proc p ' ||
			', pg_namespace n ' ||
			'WHERE p.proname = ' || quote_literal('checkauthtrigger') ||
			' AND c.relnamespace = n.oid' ||
			' AND t.tgfoid = p.oid and t.tgrelid = c.oid';
		EXECUTE "query";
	END IF;

	RETURN 'Long transactions support enabled';
END;
$$
LANGUAGE 'plpgsql';
CREATE OR REPLACE FUNCTION LongTransactionsEnabled()
	RETURNS bool
AS $$
DECLARE
	rec RECORD;
BEGIN
	FOR rec IN SELECT oid FROM pg_class WHERE relname = 'authorized_tables'
	LOOP
		return 't';
	END LOOP;
	return 'f';
END;
$$
LANGUAGE 'plpgsql';
CREATE OR REPLACE FUNCTION DisableLongTransactions()
	RETURNS TEXT
	AS $$
DECLARE
	rec RECORD;

BEGIN

	--
	-- Drop all triggers applied by CheckAuth()
	--
	FOR rec IN
		SELECT c.relname, t.tgname, t.tgargs FROM pg_trigger t, pg_class c, pg_proc p
		WHERE p.proname = 'checkauthtrigger' and t.tgfoid = p.oid and t.tgrelid = c.oid
	LOOP
		EXECUTE 'DROP TRIGGER ' || quote_ident(rec.tgname) ||
			' ON ' || quote_ident(rec.relname);
	END LOOP;

	--
	-- Drop the authorization_table table
	--
	FOR rec IN SELECT * FROM pg_class WHERE relname = 'authorization_table' LOOP
		DROP TABLE authorization_table;
	END LOOP;

	--
	-- Drop the authorized_tables view
	--
	FOR rec IN SELECT * FROM pg_class WHERE relname = 'authorized_tables' LOOP
		DROP VIEW authorized_tables;
	END LOOP;

	RETURN 'Long transactions support disabled';
END;
$$
LANGUAGE 'plpgsql';
CREATE OR REPLACE FUNCTION geography_typmod_in(cstring[])
	RETURNS integer
	AS '$libdir/postgis-3','geography_typmod_in'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geography_typmod_out(integer)
	RETURNS cstring
	AS '$libdir/postgis-3','postgis_typmod_out'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geography_in(cstring, oid, integer)
	RETURNS geography
	AS '$libdir/postgis-3','geography_in'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geography_out(geography)
	RETURNS cstring
	AS '$libdir/postgis-3','geography_out'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geography_recv(internal, oid, integer)
	RETURNS geography
	AS '$libdir/postgis-3','geography_recv'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geography_send(geography)
	RETURNS bytea
	AS '$libdir/postgis-3','geography_send'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geography_analyze(internal)
	RETURNS bool
	AS '$libdir/postgis-3','gserialized_analyze_nd'
	LANGUAGE 'c' VOLATILE STRICT;
-- Type geography -- LastUpdated: 105
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  IF 105 > version_from_num
     FROM _postgis_upgrade_info
  THEN
      EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE TYPE geography (
	internallength = variable,
	input = geography_in,
	output = geography_out,
	receive = geography_recv,
	send = geography_send,
	typmod_in = geography_typmod_in,
	typmod_out = geography_typmod_out,
	delimiter = ':',
	analyze = geography_analyze,
	storage = main,
	alignment = double
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geography(geography, integer, boolean)
	RETURNS geography
	AS '$libdir/postgis-3','geography_enforce_typmod'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (geography AS geography)');
DROP CAST IF EXISTS (geography AS geography);
CREATE CAST (geography AS geography) WITH FUNCTION geography(geography, integer, boolean) AS IMPLICIT;
CREATE OR REPLACE FUNCTION geography(bytea)
	RETURNS geography
	AS '$libdir/postgis-3','geography_from_binary'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION bytea(geography)
	RETURNS bytea
	AS '$libdir/postgis-3','LWGEOM_to_bytea'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (bytea AS geography)');
DROP CAST IF EXISTS (bytea AS geography);
CREATE CAST (bytea AS geography) WITH FUNCTION geography(bytea) AS IMPLICIT;
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (geography AS bytea)');
DROP CAST IF EXISTS (geography AS bytea);
CREATE CAST (geography AS bytea) WITH FUNCTION bytea(geography) AS IMPLICIT;
CREATE OR REPLACE FUNCTION ST_AsText(geography)
	RETURNS TEXT
	AS '$libdir/postgis-3','LWGEOM_asText'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_AsText(geography, int4)
	RETURNS TEXT
	AS '$libdir/postgis-3','LWGEOM_asText'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_AsText(text)
	RETURNS text AS
	$$ SELECT @extschema@.ST_AsText($1::@extschema@.geometry);  $$
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
        COST 500;
CREATE OR REPLACE FUNCTION ST_GeographyFromText(text)
	RETURNS geography
	AS '$libdir/postgis-3','geography_from_text'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_GeogFromText(text)
	RETURNS geography
	AS '$libdir/postgis-3','geography_from_text'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_GeogFromWKB(bytea)
	RETURNS geography
	AS '$libdir/postgis-3','geography_from_binary'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION postgis_typmod_dims(integer)
	RETURNS integer
	AS '$libdir/postgis-3','postgis_typmod_dims'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION postgis_typmod_srid(integer)
	RETURNS integer
	AS '$libdir/postgis-3','postgis_typmod_srid'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION postgis_typmod_type(integer)
	RETURNS text
	AS '$libdir/postgis-3','postgis_typmod_type'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE VIEW geography_columns AS
	SELECT
		current_database() AS f_table_catalog,
		n.nspname AS f_table_schema,
		c.relname AS f_table_name,
		a.attname AS f_geography_column,
		postgis_typmod_dims(a.atttypmod) AS coord_dimension,
		postgis_typmod_srid(a.atttypmod) AS srid,
		postgis_typmod_type(a.atttypmod) AS type
	FROM
		pg_class c,
		pg_attribute a,
		pg_type t,
		pg_namespace n
	WHERE t.typname = 'geography'
		AND a.attisdropped = false
		AND a.atttypid = t.oid
		AND a.attrelid = c.oid
		AND c.relnamespace = n.oid
		AND c.relkind = ANY (ARRAY['r'::"char", 'v'::"char", 'm'::"char", 'f'::"char", 'p'::"char"] )
		AND NOT pg_is_other_temp_schema(c.relnamespace)
		AND has_table_privilege( c.oid, 'SELECT'::text );
CREATE OR REPLACE FUNCTION geography(geometry)
	RETURNS geography
	AS '$libdir/postgis-3','geography_from_geometry'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (geometry AS geography)');
DROP CAST IF EXISTS (geometry AS geography);
CREATE CAST (geometry AS geography) WITH FUNCTION geography(geometry) AS IMPLICIT;
CREATE OR REPLACE FUNCTION geometry(geography)
	RETURNS geometry
	AS '$libdir/postgis-3','geometry_from_geography'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
SELECT postgis_extension_drop_if_exists('postgis', 'DROP CAST IF EXISTS (geography AS geometry)');
DROP CAST IF EXISTS (geography AS geometry);
CREATE CAST (geography AS geometry) WITH FUNCTION geometry(geography) ;
CREATE OR REPLACE FUNCTION geography_gist_consistent(internal,geography,int4)
	RETURNS bool
	AS '$libdir/postgis-3' ,'gserialized_gist_consistent'
	LANGUAGE 'c';
CREATE OR REPLACE FUNCTION geography_gist_compress(internal)
	RETURNS internal
	AS '$libdir/postgis-3','gserialized_gist_compress'
	LANGUAGE 'c';
CREATE OR REPLACE FUNCTION geography_gist_penalty(internal,internal,internal)
	RETURNS internal
	AS '$libdir/postgis-3' ,'gserialized_gist_penalty'
	LANGUAGE 'c';
CREATE OR REPLACE FUNCTION geography_gist_picksplit(internal, internal)
	RETURNS internal
	AS '$libdir/postgis-3' ,'gserialized_gist_picksplit'
	LANGUAGE 'c';
CREATE OR REPLACE FUNCTION geography_gist_union(bytea, internal)
	RETURNS internal
	AS '$libdir/postgis-3' ,'gserialized_gist_union'
	LANGUAGE 'c';
CREATE OR REPLACE FUNCTION geography_gist_same(box2d, box2d, internal)
	RETURNS internal
	AS '$libdir/postgis-3' ,'gserialized_gist_same'
	LANGUAGE 'c';
CREATE OR REPLACE FUNCTION geography_gist_decompress(internal)
	RETURNS internal
	AS '$libdir/postgis-3' ,'gserialized_gist_decompress'
	LANGUAGE 'c';
CREATE OR REPLACE FUNCTION geography_overlaps(geography, geography)
	RETURNS boolean
	AS '$libdir/postgis-3' ,'gserialized_overlaps'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
-- Operator geography && geography -- LastUpdated: 105
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 105 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '&&' AND
            tl.typname = 'geography' AND
            tr.typname = 'geography'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR && (
	LEFTARG = geography, RIGHTARG = geography, PROCEDURE = geography_overlaps,
	COMMUTATOR = '&&',
	RESTRICT = gserialized_gist_sel_nd,
	JOIN = gserialized_gist_joinsel_nd
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geography_distance_knn(geography, geography)
  RETURNS float8
  AS '$libdir/postgis-3','geography_distance_knn'
  LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
  COST 100;
-- Operator geography <-> geography -- LastUpdated: 202
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 202 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '<->' AND
            tl.typname = 'geography' AND
            tr.typname = 'geography'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR <-> (
  LEFTARG = geography, RIGHTARG = geography, PROCEDURE = geography_distance_knn,
  COMMUTATOR = '<->'
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geography_gist_distance(internal, geography, int4)
	RETURNS float8
	AS '$libdir/postgis-3' ,'gserialized_gist_geog_distance'
	LANGUAGE 'c';
-- Operator class gist_geography_ops -- LastUpdated: 105
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN

  IF 105 > version_from_num FROM _postgis_upgrade_info
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$
    CREATE OPERATOR CLASS gist_geography_ops
	DEFAULT FOR TYPE geography USING GIST AS
	STORAGE 	gidx,
	OPERATOR        3        &&	,
	OPERATOR        13       <-> FOR ORDER BY pg_catalog.float_ops,
	FUNCTION        8        geography_gist_distance (internal, geography, int4),
	FUNCTION        1        geography_gist_consistent (internal, geography, int4),
	FUNCTION        2        geography_gist_union (bytea, internal),
	FUNCTION        3        geography_gist_compress (internal),
	FUNCTION        4        geography_gist_decompress (internal),
	FUNCTION        5        geography_gist_penalty (internal, internal, internal),
	FUNCTION        6        geography_gist_picksplit (internal, internal),
	FUNCTION        7        geography_gist_same (box2d, box2d, internal);
    $postgis_proc_upgrade_parsed_def$;
  ELSE -- version_from >= 105
    -- Last Updated: 202
    IF 202 > version_from_num FROM _postgis_upgrade_info THEN
      EXECUTE $postgis_proc_upgrade_parsed_def$
        ALTER OPERATOR FAMILY gist_geography_ops USING gist
          ADD OPERATOR        13       <-> (geography,geography) FOR ORDER BY pg_catalog.float_ops;
      $postgis_proc_upgrade_parsed_def$;
    END IF;
  
    -- Last Updated: 202
    IF 202 > version_from_num FROM _postgis_upgrade_info THEN
      EXECUTE $postgis_proc_upgrade_parsed_def$
        ALTER OPERATOR FAMILY gist_geography_ops USING gist
          ADD FUNCTION        8 (geography,geography)        geography_gist_distance (internal, geography, int4);
      $postgis_proc_upgrade_parsed_def$;
    END IF;
  END IF; -- version_from >= 202
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION overlaps_geog(gidx, geography)
RETURNS boolean
AS '$libdir/postgis-3','gserialized_gidx_geog_overlaps'
LANGUAGE 'c' IMMUTABLE STRICT;
CREATE OR REPLACE FUNCTION overlaps_geog(gidx, gidx)
RETURNS boolean
AS '$libdir/postgis-3','gserialized_gidx_gidx_overlaps'
LANGUAGE 'c' IMMUTABLE STRICT;
-- Operator gidx && geography -- LastUpdated: 203
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 203 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '&&' AND
            tl.typname = 'gidx' AND
            tr.typname = 'geography'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR && (
  LEFTARG    = gidx,
  RIGHTARG   = geography,
  PROCEDURE  = overlaps_geog,
  COMMUTATOR = &&
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION overlaps_geog(geography, gidx)
RETURNS boolean
AS
  'SELECT $2 OPERATOR(@extschema@.&&) $1;'
 LANGUAGE SQL IMMUTABLE STRICT;
-- Operator geography && gidx -- LastUpdated: 203
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 203 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '&&' AND
            tl.typname = 'geography' AND
            tr.typname = 'gidx'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR && (
  LEFTARG    = geography,
  RIGHTARG   = gidx,
  PROCEDURE  = overlaps_geog,
  COMMUTATOR = &&
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator gidx && gidx -- LastUpdated: 203
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 203 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '&&' AND
            tl.typname = 'gidx' AND
            tr.typname = 'gidx'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR && (
  LEFTARG   = gidx,
  RIGHTARG  = gidx,
  PROCEDURE = overlaps_geog,
  COMMUTATOR = &&
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geog_brin_inclusion_add_value(internal, internal, internal, internal) RETURNS boolean
        AS '$libdir/postgis-3','geog_brin_inclusion_add_value'
        LANGUAGE 'c';
-- Operator class brin_geography_inclusion_ops -- LastUpdated: 203
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN

  IF 203 > version_from_num FROM _postgis_upgrade_info
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$
    CREATE OPERATOR CLASS brin_geography_inclusion_ops
  DEFAULT FOR TYPE geography
  USING brin AS
    FUNCTION      1        brin_inclusion_opcinfo(internal),
    FUNCTION      2        geog_brin_inclusion_add_value(internal, internal, internal, internal),
    FUNCTION      3        brin_inclusion_consistent(internal, internal, internal),
    FUNCTION      4        brin_inclusion_union(internal, internal, internal),
    OPERATOR      3        &&(geography, geography),
    OPERATOR      3        &&(geography, gidx),
    OPERATOR      3        &&(gidx, geography),
    OPERATOR      3        &&(gidx, gidx),
  STORAGE gidx;
    $postgis_proc_upgrade_parsed_def$;
  END IF; -- version_from >= 203
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geography_lt(geography, geography)
	RETURNS bool
	AS '$libdir/postgis-3', 'geography_lt'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geography_le(geography, geography)
	RETURNS bool
	AS '$libdir/postgis-3', 'geography_le'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geography_gt(geography, geography)
	RETURNS bool
	AS '$libdir/postgis-3', 'geography_gt'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geography_ge(geography, geography)
	RETURNS bool
	AS '$libdir/postgis-3', 'geography_ge'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geography_eq(geography, geography)
	RETURNS bool
	AS '$libdir/postgis-3', 'geography_eq'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geography_cmp(geography, geography)
	RETURNS integer
	AS '$libdir/postgis-3', 'geography_cmp'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
-- Operator geography < geography -- LastUpdated: 105
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 105 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '<' AND
            tl.typname = 'geography' AND
            tr.typname = 'geography'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR < (
	LEFTARG = geography, RIGHTARG = geography, PROCEDURE = geography_lt,
	COMMUTATOR = '>', NEGATOR = '>=',
	RESTRICT = contsel, JOIN = contjoinsel
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator geography <= geography -- LastUpdated: 105
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 105 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '<=' AND
            tl.typname = 'geography' AND
            tr.typname = 'geography'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR <= (
	LEFTARG = geography, RIGHTARG = geography, PROCEDURE = geography_le,
	COMMUTATOR = '>=', NEGATOR = '>',
	RESTRICT = contsel, JOIN = contjoinsel
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator geography = geography -- LastUpdated: 105
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 105 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '=' AND
            tl.typname = 'geography' AND
            tr.typname = 'geography'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR = (
	LEFTARG = geography, RIGHTARG = geography, PROCEDURE = geography_eq,
	COMMUTATOR = '=', -- we might implement a faster negator here
	RESTRICT = contsel, JOIN = contjoinsel
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator geography >= geography -- LastUpdated: 105
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 105 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '>=' AND
            tl.typname = 'geography' AND
            tr.typname = 'geography'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR >= (
	LEFTARG = geography, RIGHTARG = geography, PROCEDURE = geography_ge,
	COMMUTATOR = '<=', NEGATOR = '<',
	RESTRICT = contsel, JOIN = contjoinsel
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator geography > geography -- LastUpdated: 105
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 105 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '>' AND
            tl.typname = 'geography' AND
            tr.typname = 'geography'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR > (
	LEFTARG = geography, RIGHTARG = geography, PROCEDURE = geography_gt,
	COMMUTATOR = '<', NEGATOR = '<=',
	RESTRICT = contsel, JOIN = contjoinsel
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator class btree_geography_ops -- LastUpdated: 105
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN

  IF 105 > version_from_num FROM _postgis_upgrade_info
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$
    CREATE OPERATOR CLASS btree_geography_ops
	DEFAULT FOR TYPE geography USING btree AS
	OPERATOR	1	< ,
	OPERATOR	2	<= ,
	OPERATOR	3	= ,
	OPERATOR	4	>= ,
	OPERATOR	5	> ,
	FUNCTION	1	geography_cmp (geography, geography);
    $postgis_proc_upgrade_parsed_def$;
  END IF; -- version_from >= 105
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION ST_AsSVG(geog geography, rel int4 DEFAULT 0, maxdecimaldigits int4 DEFAULT 15)
	RETURNS text
	AS '$libdir/postgis-3','geography_as_svg'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_AsSVG(text)
	RETURNS text AS
	$$ SELECT @extschema@.ST_AsSVG($1::@extschema@.geometry,0,15);  $$
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
        COST 500;
CREATE OR REPLACE FUNCTION ST_AsGML(version int4, geog geography, maxdecimaldigits int4 DEFAULT 15, options int4 DEFAULT 0, nprefix text DEFAULT 'gml', id text DEFAULT '')
	RETURNS text
	AS '$libdir/postgis-3','geography_as_gml'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_AsGML(geog geography, maxdecimaldigits int4 DEFAULT 15, options int4 DEFAULT 0, nprefix text DEFAULT 'gml', id text DEFAULT '')
	RETURNS text
	AS '$libdir/postgis-3','geography_as_gml'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_AsGML(text)
	RETURNS text AS
	$$ SELECT @extschema@._ST_AsGML(2,$1::@extschema@.geometry,15,0, NULL, NULL);  $$
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
        COST 500;
CREATE OR REPLACE FUNCTION ST_AsKML(geog geography, maxdecimaldigits int4 DEFAULT 15, nprefix text DEFAULT '')
	RETURNS text
	AS '$libdir/postgis-3','geography_as_kml'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_AsKML(text)
	RETURNS text AS
	$$ SELECT @extschema@.ST_AsKML($1::@extschema@.geometry, 15);  $$
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
        COST 500;
CREATE OR REPLACE FUNCTION ST_AsGeoJson(geog geography, maxdecimaldigits int4 DEFAULT 9, options int4 DEFAULT 0)
	RETURNS text
	AS '$libdir/postgis-3','geography_as_geojson'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_AsGeoJson(text)
	RETURNS text AS
	$$ SELECT @extschema@.ST_AsGeoJson($1::@extschema@.geometry, 9, 0);  $$
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
        COST 500;
CREATE OR REPLACE FUNCTION ST_Distance(geog1 geography, geog2 geography, use_spheroid boolean DEFAULT true)
	RETURNS float8
	AS '$libdir/postgis-3','geography_distance'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Distance(text, text)
	RETURNS float8 AS
	$$ SELECT @extschema@.ST_Distance($1::@extschema@.geometry, $2::@extschema@.geometry);  $$
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION _ST_Expand(geography, float8)
	RETURNS geography
	AS '$libdir/postgis-3','geography_expand'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION _ST_DistanceUnCached(geography, geography, float8, boolean)
	RETURNS float8
	AS '$libdir/postgis-3','geography_distance_uncached'
	LANGUAGE 'c' IMMUTABLE STRICT
	COST 10000;
CREATE OR REPLACE FUNCTION _ST_DistanceUnCached(geography, geography, boolean)
	RETURNS float8
	AS 'SELECT @extschema@._ST_DistanceUnCached($1, $2, 0.0, $3)'
	LANGUAGE 'sql' IMMUTABLE STRICT;
CREATE OR REPLACE FUNCTION _ST_DistanceUnCached(geography, geography)
	RETURNS float8
	AS 'SELECT @extschema@._ST_DistanceUnCached($1, $2, 0.0, true)'
	LANGUAGE 'sql' IMMUTABLE STRICT;
CREATE OR REPLACE FUNCTION _ST_DistanceTree(geography, geography, float8, boolean)
	RETURNS float8
	AS '$libdir/postgis-3','geography_distance_tree'
	LANGUAGE 'c' IMMUTABLE STRICT
	COST 10000;
CREATE OR REPLACE FUNCTION _ST_DistanceTree(geography, geography)
	RETURNS float8
	AS 'SELECT @extschema@._ST_DistanceTree($1, $2, 0.0, true)'
	LANGUAGE 'sql' IMMUTABLE STRICT;
CREATE OR REPLACE FUNCTION _ST_DWithinUnCached(geography, geography, float8, boolean)
	RETURNS boolean
	AS '$libdir/postgis-3','geography_dwithin_uncached'
	LANGUAGE 'c' IMMUTABLE STRICT
	COST 10000;
CREATE OR REPLACE FUNCTION _ST_DWithinUnCached(geography, geography, float8)
	RETURNS boolean
	AS 'SELECT $1 OPERATOR(@extschema@.&&) @extschema@._ST_Expand($2,$3) AND $2 OPERATOR(@extschema@.&&) @extschema@._ST_Expand($1,$3) AND @extschema@._ST_DWithinUnCached($1, $2, $3, true)'
	LANGUAGE 'sql' IMMUTABLE;
CREATE OR REPLACE FUNCTION ST_Area(geog geography, use_spheroid boolean DEFAULT true)
	RETURNS float8
	AS '$libdir/postgis-3','geography_area'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Area(text)
	RETURNS float8 AS
	$$ SELECT @extschema@.ST_Area($1::@extschema@.geometry);  $$
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION ST_Length(geog geography, use_spheroid boolean DEFAULT true)
	RETURNS float8
	AS '$libdir/postgis-3','geography_length'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_Length(text)
	RETURNS float8 AS
	$$ SELECT @extschema@.ST_Length($1::@extschema@.geometry);  $$
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION ST_Project(geog geography, distance float8, azimuth float8)
	RETURNS geography
	AS '$libdir/postgis-3','geography_project'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_Azimuth(geog1 geography, geog2 geography)
	RETURNS float8
	AS '$libdir/postgis-3','geography_azimuth'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_Perimeter(geog geography, use_spheroid boolean DEFAULT true)
	RETURNS float8
	AS '$libdir/postgis-3','geography_perimeter'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION _ST_PointOutside(geography)
	RETURNS geography
	AS '$libdir/postgis-3','geography_point_outside'
	LANGUAGE 'c' IMMUTABLE STRICT
	COST 1;
CREATE OR REPLACE FUNCTION ST_Segmentize(geog geography, max_segment_length float8)
	RETURNS geography
	AS '$libdir/postgis-3','geography_segmentize'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION _ST_BestSRID(geography, geography)
	RETURNS integer
	AS '$libdir/postgis-3','geography_bestsrid'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION _ST_BestSRID(geography)
	RETURNS integer
	AS '$libdir/postgis-3','geography_bestsrid'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Buffer(geography, float8)
	RETURNS geography
	AS 'SELECT @extschema@.geography(@extschema@.ST_Transform(@extschema@.ST_Buffer(@extschema@.ST_Transform(@extschema@.geometry($1), @extschema@._ST_BestSRID($1)), $2), 4326))'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION ST_Buffer(geography, float8, integer)
	RETURNS geography
	AS 'SELECT @extschema@.geography(@extschema@.ST_Transform(@extschema@.ST_Buffer(@extschema@.ST_Transform(@extschema@.geometry($1), @extschema@._ST_BestSRID($1)), $2, $3), 4326))'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION ST_Buffer(geography, float8, text)
	RETURNS geography
	AS 'SELECT @extschema@.geography(@extschema@.ST_Transform(@extschema@.ST_Buffer(@extschema@.ST_Transform(@extschema@.geometry($1), @extschema@._ST_BestSRID($1)), $2, $3), 4326))'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION ST_Buffer(text, float8)
	RETURNS geometry AS
	$$ SELECT @extschema@.ST_Buffer($1::@extschema@.geometry, $2);  $$
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION ST_Buffer(text, float8, integer)
	RETURNS geometry AS
	$$ SELECT @extschema@.ST_Buffer($1::@extschema@.geometry, $2, $3);  $$
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION ST_Buffer(text, float8, text)
	RETURNS geometry AS
	$$ SELECT @extschema@.ST_Buffer($1::@extschema@.geometry, $2, $3);  $$
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION ST_Intersection(geography, geography)
	RETURNS geography
	AS 'SELECT @extschema@.geography(@extschema@.ST_Transform(@extschema@.ST_Intersection(@extschema@.ST_Transform(@extschema@.geometry($1), @extschema@._ST_BestSRID($1, $2)), @extschema@.ST_Transform(@extschema@.geometry($2), @extschema@._ST_BestSRID($1, $2))), 4326))'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION ST_Intersection(text, text)
	RETURNS geometry AS
	$$ SELECT @extschema@.ST_Intersection($1::@extschema@.geometry, $2::@extschema@.geometry);  $$
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_AsBinary(geography)
	RETURNS bytea
	AS '$libdir/postgis-3','LWGEOM_asBinary'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_AsBinary(geography, text)
	RETURNS bytea
	AS '$libdir/postgis-3','LWGEOM_asBinary'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_AsEWKT(geography)
	RETURNS TEXT
	AS '$libdir/postgis-3','LWGEOM_asEWKT'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_AsEWKT(geography, int4)
	RETURNS TEXT
	AS '$libdir/postgis-3','LWGEOM_asEWKT'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_AsEWKT(text)
	RETURNS text AS
	$$ SELECT @extschema@.ST_AsEWKT($1::@extschema@.geometry);  $$
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
        COST 500;
CREATE OR REPLACE FUNCTION GeometryType(geography)
	RETURNS text
	AS '$libdir/postgis-3', 'LWGEOM_getTYPE'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION ST_Summary(geography)
	RETURNS text
	AS '$libdir/postgis-3', 'LWGEOM_summary'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_GeoHash(geog geography, maxchars int4 DEFAULT 0)
	RETURNS TEXT
	AS '$libdir/postgis-3', 'ST_GeoHash'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_SRID(geog geography)
	RETURNS int4
	AS '$libdir/postgis-3', 'LWGEOM_get_srid'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_SetSRID(geog geography, srid int4)
	RETURNS geography
	AS '$libdir/postgis-3', 'LWGEOM_set_srid'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Centroid(geography, use_spheroid boolean DEFAULT true)
	RETURNS geography
	AS '$libdir/postgis-3','geography_centroid'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_Centroid(text)
	RETURNS geometry AS
	$$ SELECT @extschema@.ST_Centroid($1::@extschema@.geometry);  $$
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION _ST_Covers(geog1 geography, geog2 geography)
	RETURNS boolean
	AS '$libdir/postgis-3','geography_covers'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION _ST_DWithin(geog1 geography, geog2 geography, tolerance float8, use_spheroid boolean DEFAULT true)
	RETURNS boolean
	AS '$libdir/postgis-3','geography_dwithin'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION _ST_CoveredBy(geog1 geography, geog2 geography)
	RETURNS boolean
	AS '$libdir/postgis-3','geography_coveredby'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Covers(geog1 geography, geog2 geography)
	RETURNS boolean
	AS '$libdir/postgis-3','geography_covers'
	SUPPORT postgis_index_supportfn
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_DWithin(geog1 geography, geog2 geography, tolerance float8, use_spheroid boolean DEFAULT true)
	RETURNS boolean
	AS '$libdir/postgis-3','geography_dwithin'
	SUPPORT postgis_index_supportfn
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_CoveredBy(geog1 geography, geog2 geography)
	RETURNS boolean
	AS '$libdir/postgis-3','geography_coveredby'
	SUPPORT postgis_index_supportfn
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Intersects(geog1 geography, geog2 geography)
	RETURNS boolean
	AS '$libdir/postgis-3','geography_intersects'
	SUPPORT postgis_index_supportfn
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Covers(text, text)
	RETURNS boolean AS
	$$ SELECT @extschema@.ST_Covers($1::@extschema@.geometry, $2::@extschema@.geometry);  $$
	LANGUAGE 'sql' IMMUTABLE PARALLEL SAFE;
CREATE OR REPLACE FUNCTION ST_CoveredBy(text, text)
	RETURNS boolean AS
	$$ SELECT @extschema@.ST_CoveredBy($1::@extschema@.geometry, $2::@extschema@.geometry);  $$
	LANGUAGE 'sql' IMMUTABLE PARALLEL SAFE;
CREATE OR REPLACE FUNCTION ST_DWithin(text, text, float8)
	RETURNS boolean AS
	$$ SELECT @extschema@.ST_DWithin($1::@extschema@.geometry, $2::@extschema@.geometry, $3);  $$
	LANGUAGE 'sql' IMMUTABLE PARALLEL SAFE;
CREATE OR REPLACE FUNCTION ST_Intersects(text, text)
	RETURNS boolean AS
	$$ SELECT @extschema@.ST_Intersects($1::@extschema@.geometry, $2::@extschema@.geometry);  $$
	LANGUAGE 'sql' IMMUTABLE PARALLEL SAFE;
CREATE OR REPLACE FUNCTION ST_DistanceSphere(geom1 geometry, geom2 geometry)
	RETURNS FLOAT8 AS
	'select @extschema@.ST_distance( @extschema@.geography($1), @extschema@.geography($2),false)'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION postgis_type_name(geomname varchar, coord_dimension integer, use_new_name boolean DEFAULT true)
	RETURNS varchar
AS
$$
	SELECT CASE WHEN $3 THEN new_name ELSE old_name END As geomname
	FROM
	( VALUES
			('GEOMETRY', 'Geometry', 2),
			('GEOMETRY', 'GeometryZ', 3),
			('GEOMETRYM', 'GeometryM', 3),
			('GEOMETRY', 'GeometryZM', 4),

			('GEOMETRYCOLLECTION', 'GeometryCollection', 2),
			('GEOMETRYCOLLECTION', 'GeometryCollectionZ', 3),
			('GEOMETRYCOLLECTIONM', 'GeometryCollectionM', 3),
			('GEOMETRYCOLLECTION', 'GeometryCollectionZM', 4),

			('POINT', 'Point', 2),
			('POINT', 'PointZ', 3),
			('POINTM','PointM', 3),
			('POINT', 'PointZM', 4),

			('MULTIPOINT','MultiPoint', 2),
			('MULTIPOINT','MultiPointZ', 3),
			('MULTIPOINTM','MultiPointM', 3),
			('MULTIPOINT','MultiPointZM', 4),

			('POLYGON', 'Polygon', 2),
			('POLYGON', 'PolygonZ', 3),
			('POLYGONM', 'PolygonM', 3),
			('POLYGON', 'PolygonZM', 4),

			('MULTIPOLYGON', 'MultiPolygon', 2),
			('MULTIPOLYGON', 'MultiPolygonZ', 3),
			('MULTIPOLYGONM', 'MultiPolygonM', 3),
			('MULTIPOLYGON', 'MultiPolygonZM', 4),

			('MULTILINESTRING', 'MultiLineString', 2),
			('MULTILINESTRING', 'MultiLineStringZ', 3),
			('MULTILINESTRINGM', 'MultiLineStringM', 3),
			('MULTILINESTRING', 'MultiLineStringZM', 4),

			('LINESTRING', 'LineString', 2),
			('LINESTRING', 'LineStringZ', 3),
			('LINESTRINGM', 'LineStringM', 3),
			('LINESTRING', 'LineStringZM', 4),

			('CIRCULARSTRING', 'CircularString', 2),
			('CIRCULARSTRING', 'CircularStringZ', 3),
			('CIRCULARSTRINGM', 'CircularStringM' ,3),
			('CIRCULARSTRING', 'CircularStringZM', 4),

			('COMPOUNDCURVE', 'CompoundCurve', 2),
			('COMPOUNDCURVE', 'CompoundCurveZ', 3),
			('COMPOUNDCURVEM', 'CompoundCurveM', 3),
			('COMPOUNDCURVE', 'CompoundCurveZM', 4),

			('CURVEPOLYGON', 'CurvePolygon', 2),
			('CURVEPOLYGON', 'CurvePolygonZ', 3),
			('CURVEPOLYGONM', 'CurvePolygonM', 3),
			('CURVEPOLYGON', 'CurvePolygonZM', 4),

			('MULTICURVE', 'MultiCurve', 2),
			('MULTICURVE', 'MultiCurveZ', 3),
			('MULTICURVEM', 'MultiCurveM', 3),
			('MULTICURVE', 'MultiCurveZM', 4),

			('MULTISURFACE', 'MultiSurface', 2),
			('MULTISURFACE', 'MultiSurfaceZ', 3),
			('MULTISURFACEM', 'MultiSurfaceM', 3),
			('MULTISURFACE', 'MultiSurfaceZM', 4),

			('POLYHEDRALSURFACE', 'PolyhedralSurface', 2),
			('POLYHEDRALSURFACE', 'PolyhedralSurfaceZ', 3),
			('POLYHEDRALSURFACEM', 'PolyhedralSurfaceM', 3),
			('POLYHEDRALSURFACE', 'PolyhedralSurfaceZM', 4),

			('TRIANGLE', 'Triangle', 2),
			('TRIANGLE', 'TriangleZ', 3),
			('TRIANGLEM', 'TriangleM', 3),
			('TRIANGLE', 'TriangleZM', 4),

			('TIN', 'Tin', 2),
			('TIN', 'TinZ', 3),
			('TINM', 'TinM', 3),
			('TIN', 'TinZM', 4) )
			 As g(old_name, new_name, coord_dimension)
	WHERE (upper(old_name) = upper($1) OR upper(new_name) = upper($1))
		AND coord_dimension = $2;
$$
LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE COST 10000;
CREATE OR REPLACE FUNCTION postgis_constraint_srid(geomschema text, geomtable text, geomcolumn text) RETURNS integer AS
$$
SELECT replace(replace(split_part(s.consrc, ' = ', 2), ')', ''), '(', '')::integer
		 FROM pg_class c, pg_namespace n, pg_attribute a
		 , (SELECT connamespace, conrelid, conkey, pg_get_constraintdef(oid) As consrc
			FROM pg_constraint) AS s
		 WHERE n.nspname = $1
		 AND c.relname = $2
		 AND a.attname = $3
		 AND a.attrelid = c.oid
		 AND s.connamespace = n.oid
		 AND s.conrelid = c.oid
		 AND a.attnum = ANY (s.conkey)
		 AND s.consrc LIKE '%srid(% = %';
$$
LANGUAGE 'sql' STABLE STRICT PARALLEL SAFE COST 500;
CREATE OR REPLACE FUNCTION postgis_constraint_dims(geomschema text, geomtable text, geomcolumn text) RETURNS integer AS
$$
SELECT  replace(split_part(s.consrc, ' = ', 2), ')', '')::integer
		 FROM pg_class c, pg_namespace n, pg_attribute a
		 , (SELECT connamespace, conrelid, conkey, pg_get_constraintdef(oid) As consrc
			FROM pg_constraint) AS s
		 WHERE n.nspname = $1
		 AND c.relname = $2
		 AND a.attname = $3
		 AND a.attrelid = c.oid
		 AND s.connamespace = n.oid
		 AND s.conrelid = c.oid
		 AND a.attnum = ANY (s.conkey)
		 AND s.consrc LIKE '%ndims(% = %';
$$
LANGUAGE 'sql' STABLE STRICT PARALLEL SAFE COST 500;
CREATE OR REPLACE FUNCTION postgis_constraint_type(geomschema text, geomtable text, geomcolumn text) RETURNS varchar AS
$$
SELECT  replace(split_part(s.consrc, '''', 2), ')', '')::varchar
		 FROM pg_class c, pg_namespace n, pg_attribute a
		 , (SELECT connamespace, conrelid, conkey, pg_get_constraintdef(oid) As consrc
			FROM pg_constraint) AS s
		 WHERE n.nspname = $1
		 AND c.relname = $2
		 AND a.attname = $3
		 AND a.attrelid = c.oid
		 AND s.connamespace = n.oid
		 AND s.conrelid = c.oid
		 AND a.attnum = ANY (s.conkey)
		 AND s.consrc LIKE '%geometrytype(% = %';
$$
LANGUAGE 'sql' STABLE STRICT PARALLEL SAFE COST 500;
CREATE OR REPLACE VIEW geometry_columns AS
 SELECT current_database()::character varying(256) AS f_table_catalog,
    n.nspname AS f_table_schema,
    c.relname AS f_table_name,
    a.attname AS f_geometry_column,
    COALESCE(postgis_typmod_dims(a.atttypmod), sn.ndims, 2) AS coord_dimension,
    COALESCE(NULLIF(postgis_typmod_srid(a.atttypmod), 0), sr.srid, 0) AS srid,
    replace(replace(COALESCE(NULLIF(upper(postgis_typmod_type(a.atttypmod)), 'GEOMETRY'::text), st.type, 'GEOMETRY'::text), 'ZM'::text, ''::text), 'Z'::text, ''::text)::character varying(30) AS type
   FROM pg_class c
     JOIN pg_attribute a ON a.attrelid = c.oid AND NOT a.attisdropped
     JOIN pg_namespace n ON c.relnamespace = n.oid
     JOIN pg_type t ON a.atttypid = t.oid
     LEFT JOIN ( SELECT s.connamespace,
            s.conrelid,
            s.conkey, replace(split_part(s.consrc, ''''::text, 2), ')'::text, ''::text) As type
           FROM (SELECT connamespace, conrelid, conkey, pg_get_constraintdef(oid) As consrc
				FROM pg_constraint) AS s
          WHERE s.consrc ~~* '%geometrytype(% = %'::text

) st ON st.connamespace = n.oid AND st.conrelid = c.oid AND (a.attnum = ANY (st.conkey))
     LEFT JOIN ( SELECT s.connamespace,
            s.conrelid,
            s.conkey, replace(split_part(s.consrc, ' = '::text, 2), ')'::text, ''::text)::integer As ndims
           FROM (SELECT connamespace, conrelid, conkey, pg_get_constraintdef(oid) As consrc
		    FROM pg_constraint) AS s
          WHERE s.consrc ~~* '%ndims(% = %'::text

) sn ON sn.connamespace = n.oid AND sn.conrelid = c.oid AND (a.attnum = ANY (sn.conkey))
     LEFT JOIN ( SELECT s.connamespace,
            s.conrelid,
            s.conkey, replace(replace(split_part(s.consrc, ' = '::text, 2), ')'::text, ''::text), '('::text, ''::text)::integer As srid
           FROM (SELECT connamespace, conrelid, conkey, pg_get_constraintdef(oid) As consrc
		    FROM pg_constraint) AS s
          WHERE s.consrc ~~* '%srid(% = %'::text

) sr ON sr.connamespace = n.oid AND sr.conrelid = c.oid AND (a.attnum = ANY (sr.conkey))
  WHERE (c.relkind = ANY (ARRAY['r'::"char", 'v'::"char", 'm'::"char", 'f'::"char", 'p'::"char"]))
  AND NOT c.relname = 'raster_columns'::name AND t.typname = 'geometry'::name
  AND NOT pg_is_other_temp_schema(c.relnamespace) AND has_table_privilege(c.oid, 'SELECT'::text);
CREATE OR REPLACE RULE geometry_columns_insert AS
        ON INSERT TO geometry_columns
        DO INSTEAD NOTHING;
CREATE OR REPLACE RULE geometry_columns_update AS
        ON UPDATE TO geometry_columns
        DO INSTEAD NOTHING;
CREATE OR REPLACE RULE geometry_columns_delete AS
        ON DELETE TO geometry_columns
        DO INSTEAD NOTHING;
CREATE OR REPLACE FUNCTION ST_3DDistance(geom1 geometry, geom2 geometry)
	RETURNS float8
	AS '$libdir/postgis-3', 'ST_3DDistance'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_3DMaxDistance(geom1 geometry, geom2 geometry)
	RETURNS float8
	AS '$libdir/postgis-3', 'LWGEOM_maxdistance3d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_3DClosestPoint(geom1 geometry, geom2 geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_closestpoint3d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_3DShortestLine(geom1 geometry, geom2 geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_shortestline3d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_3DLongestLine(geom1 geometry, geom2 geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_longestline3d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_CoordDim(Geometry geometry)
	RETURNS smallint
	AS '$libdir/postgis-3', 'LWGEOM_ndims'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 1;
CREATE OR REPLACE FUNCTION ST_CurveToLine(geom geometry, tol float8 DEFAULT 32, toltype integer DEFAULT 0, flags integer DEFAULT 0)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_CurveToLine'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_HasArc(Geometry geometry)
	RETURNS boolean
	AS '$libdir/postgis-3', 'LWGEOM_has_arc'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_LineToCurve(Geometry geometry)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_line_desegmentize'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 10000;
CREATE OR REPLACE FUNCTION ST_Point(float8, float8)
	RETURNS geometry
	AS '$libdir/postgis-3', 'LWGEOM_makepoint'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Polygon(geometry, int)
	RETURNS geometry
	AS $$
	SELECT @extschema@.ST_SetSRID(@extschema@.ST_MakePolygon($1), $2)
	$$
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_WKBToSQL(WKB bytea)
	RETURNS geometry
	AS '$libdir/postgis-3','LWGEOM_from_WKB'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_LocateBetween(Geometry geometry, FromMeasure float8, ToMeasure float8, LeftRightOffset float8 default 0.0)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_LocateBetween'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_LocateAlong(Geometry geometry, Measure float8, LeftRightOffset float8 default 0.0)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_LocateAlong'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_LocateBetweenElevations(Geometry geometry, FromElevation float8, ToElevation float8)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_LocateBetweenElevations'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_InterpolatePoint(Line geometry, Point geometry)
	RETURNS float8
	AS '$libdir/postgis-3', 'ST_InterpolatePoint'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_Hexagon(size float8, cell_i integer, cell_j integer, origin geometry DEFAULT 'POINT(0 0)')
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_Hexagon'
	LANGUAGE 'c' IMMUTABLE STRICT
	PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_Square(size float8, cell_i integer, cell_j integer, origin geometry DEFAULT 'POINT(0 0)')
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_Square'
	LANGUAGE 'c' IMMUTABLE STRICT
	PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION ST_HexagonGrid(size float8, bounds geometry, OUT geom geometry, OUT i integer, OUT j integer)
	RETURNS SETOF record
	AS '$libdir/postgis-3', 'ST_ShapeGrid'
	LANGUAGE 'c' IMMUTABLE STRICT
	PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_SquareGrid(size float8, bounds geometry, OUT geom geometry, OUT i integer, OUT j integer)
	RETURNS SETOF record
	AS '$libdir/postgis-3', 'ST_ShapeGrid'
	LANGUAGE 'c' IMMUTABLE STRICT
	PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION contains_2d(box2df, geometry)
RETURNS boolean
AS '$libdir/postgis-3','gserialized_contains_box2df_geom_2d'
LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE COST 1;
CREATE OR REPLACE FUNCTION is_contained_2d(box2df, geometry)
RETURNS boolean
AS '$libdir/postgis-3','gserialized_within_box2df_geom_2d'
LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE COST 1;
CREATE OR REPLACE FUNCTION overlaps_2d(box2df, geometry)
RETURNS boolean
AS '$libdir/postgis-3','gserialized_overlaps_box2df_geom_2d'
LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE COST 1;
CREATE OR REPLACE FUNCTION overlaps_2d(box2df, box2df)
RETURNS boolean
AS '$libdir/postgis-3','gserialized_contains_box2df_box2df_2d'
LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE COST 1;
CREATE OR REPLACE FUNCTION contains_2d(box2df, box2df)
RETURNS boolean
AS '$libdir/postgis-3','gserialized_contains_box2df_box2df_2d'
LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE COST 1;
CREATE OR REPLACE FUNCTION is_contained_2d(box2df, box2df)
RETURNS boolean
AS '$libdir/postgis-3','gserialized_contains_box2df_box2df_2d'
LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE COST 1;
-- Operator box2df ~ geometry -- LastUpdated: 203
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 203 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '~' AND
            tl.typname = 'box2df' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR ~ (
	LEFTARG    = box2df,
	RIGHTARG   = geometry,
	PROCEDURE  = contains_2d,
	COMMUTATOR = @
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator box2df @ geometry -- LastUpdated: 203
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 203 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '@' AND
            tl.typname = 'box2df' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR @ (
	LEFTARG    = box2df,
	RIGHTARG   = geometry,
	PROCEDURE  = is_contained_2d,
	COMMUTATOR = ~
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator box2df && geometry -- LastUpdated: 203
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 203 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '&&' AND
            tl.typname = 'box2df' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR && (
	LEFTARG    = box2df,
	RIGHTARG   = geometry,
	PROCEDURE  = overlaps_2d,
	COMMUTATOR = &&
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION contains_2d(geometry, box2df)
RETURNS boolean
AS
	'SELECT $2 OPERATOR(@extschema@.@) $1;'
LANGUAGE SQL IMMUTABLE STRICT PARALLEL SAFE COST 1;
CREATE OR REPLACE FUNCTION is_contained_2d(geometry, box2df)
RETURNS boolean
AS
	'SELECT $2 OPERATOR(@extschema@.~) $1;'
LANGUAGE SQL IMMUTABLE STRICT PARALLEL SAFE COST 1;
CREATE OR REPLACE FUNCTION overlaps_2d(geometry, box2df)
RETURNS boolean
AS
	'SELECT $2 OPERATOR(@extschema@.&&) $1;'
LANGUAGE SQL IMMUTABLE STRICT PARALLEL SAFE COST 1;
-- Operator geometry ~ box2df -- LastUpdated: 203
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 203 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '~' AND
            tl.typname = 'geometry' AND
            tr.typname = 'box2df'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR ~ (
	LEFTARG = geometry,
	RIGHTARG = box2df,
	COMMUTATOR = @,
	PROCEDURE  = contains_2d
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator geometry @ box2df -- LastUpdated: 203
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 203 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '@' AND
            tl.typname = 'geometry' AND
            tr.typname = 'box2df'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR @ (
	LEFTARG = geometry,
	RIGHTARG = box2df,
	COMMUTATOR = ~,
	PROCEDURE = is_contained_2d
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator geometry && box2df -- LastUpdated: 203
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 203 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '&&' AND
            tl.typname = 'geometry' AND
            tr.typname = 'box2df'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR && (
	LEFTARG    = geometry,
	RIGHTARG   = box2df,
	PROCEDURE  = overlaps_2d,
	COMMUTATOR = &&
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator box2df && box2df -- LastUpdated: 203
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 203 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '&&' AND
            tl.typname = 'box2df' AND
            tr.typname = 'box2df'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR && (
	LEFTARG   = box2df,
	RIGHTARG  = box2df,
	PROCEDURE = overlaps_2d,
	COMMUTATOR = &&
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator box2df @ box2df -- LastUpdated: 203
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 203 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '@' AND
            tl.typname = 'box2df' AND
            tr.typname = 'box2df'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR @ (
	LEFTARG   = box2df,
	RIGHTARG  = box2df,
	PROCEDURE = is_contained_2d,
	COMMUTATOR = ~
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator box2df ~ box2df -- LastUpdated: 203
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 203 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '~' AND
            tl.typname = 'box2df' AND
            tr.typname = 'box2df'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR ~ (
	LEFTARG   = box2df,
	RIGHTARG  = box2df,
	PROCEDURE = contains_2d,
	COMMUTATOR = @
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION overlaps_nd(gidx, geometry)
RETURNS boolean
AS '$libdir/postgis-3','gserialized_gidx_geom_overlaps'
LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE COST 1;
CREATE OR REPLACE FUNCTION overlaps_nd(gidx, gidx)
RETURNS boolean
AS '$libdir/postgis-3','gserialized_gidx_gidx_overlaps'
LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE COST 1;
-- Operator gidx &&& geometry -- LastUpdated: 203
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 203 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '&&&' AND
            tl.typname = 'gidx' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR &&& (
	LEFTARG    = gidx,
	RIGHTARG   = geometry,
	PROCEDURE  = overlaps_nd,
	COMMUTATOR = &&&
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION overlaps_nd(geometry, gidx)
RETURNS boolean
AS
	'SELECT $2 OPERATOR(@extschema@.&&&) $1;'
LANGUAGE SQL IMMUTABLE STRICT PARALLEL SAFE COST 1;
-- Operator geometry &&& gidx -- LastUpdated: 203
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 203 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '&&&' AND
            tl.typname = 'geometry' AND
            tr.typname = 'gidx'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR &&& (
	LEFTARG    = geometry,
	RIGHTARG   = gidx,
	PROCEDURE  = overlaps_nd,
	COMMUTATOR = &&&
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator gidx &&& gidx -- LastUpdated: 203
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 203 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '&&&' AND
            tl.typname = 'gidx' AND
            tr.typname = 'gidx'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR &&& (
	LEFTARG   = gidx,
	RIGHTARG  = gidx,
	PROCEDURE = overlaps_nd,
	COMMUTATOR = &&&
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geom2d_brin_inclusion_add_value(internal, internal, internal, internal)
RETURNS boolean
AS '$libdir/postgis-3','geom2d_brin_inclusion_add_value'
LANGUAGE 'c' PARALLEL SAFE COST 1;
CREATE OR REPLACE FUNCTION geom3d_brin_inclusion_add_value(internal, internal, internal, internal)
RETURNS boolean
AS '$libdir/postgis-3','geom3d_brin_inclusion_add_value'
LANGUAGE 'c' PARALLEL SAFE COST 1;
CREATE OR REPLACE FUNCTION geom4d_brin_inclusion_add_value(internal, internal, internal, internal)
RETURNS boolean
AS '$libdir/postgis-3','geom4d_brin_inclusion_add_value'
LANGUAGE 'c' PARALLEL SAFE COST 1;
-- Operator class brin_geometry_inclusion_ops_2d -- LastUpdated: 203
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN

  IF 203 > version_from_num FROM _postgis_upgrade_info
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$
    CREATE OPERATOR CLASS brin_geometry_inclusion_ops_2d
  DEFAULT FOR TYPE geometry
  USING brin AS
    FUNCTION      1        brin_inclusion_opcinfo(internal),
    FUNCTION      2        geom2d_brin_inclusion_add_value(internal, internal, internal, internal),
    FUNCTION      3        brin_inclusion_consistent(internal, internal, internal),
    FUNCTION      4        brin_inclusion_union(internal, internal, internal),
    OPERATOR      3         &&(box2df, box2df),
    OPERATOR      3         &&(box2df, geometry),
    OPERATOR      3         &&(geometry, box2df),
    OPERATOR      3        &&(geometry, geometry),
    OPERATOR      7         ~(box2df, box2df),
    OPERATOR      7         ~(box2df, geometry),
    OPERATOR      7         ~(geometry, box2df),
    OPERATOR      7        ~(geometry, geometry),
    OPERATOR      8         @(box2df, box2df),
    OPERATOR      8         @(box2df, geometry),
    OPERATOR      8         @(geometry, box2df),
    OPERATOR      8        @(geometry, geometry),
  STORAGE box2df;
    $postgis_proc_upgrade_parsed_def$;
  END IF; -- version_from >= 203
END
$postgis_proc_upgrade$;
-- Operator class brin_geometry_inclusion_ops_3d -- LastUpdated: 203
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN

  IF 203 > version_from_num FROM _postgis_upgrade_info
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$
    CREATE OPERATOR CLASS brin_geometry_inclusion_ops_3d
  FOR TYPE geometry
  USING brin AS
    FUNCTION      1        brin_inclusion_opcinfo(internal) ,
    FUNCTION      2        geom3d_brin_inclusion_add_value(internal, internal, internal, internal),
    FUNCTION      3        brin_inclusion_consistent(internal, internal, internal),
    FUNCTION      4        brin_inclusion_union(internal, internal, internal),
    OPERATOR      3        &&&(geometry, geometry),
    OPERATOR      3        &&&(geometry, gidx),
    OPERATOR      3        &&&(gidx, geometry),
    OPERATOR      3        &&&(gidx, gidx),
  STORAGE gidx;
    $postgis_proc_upgrade_parsed_def$;
  END IF; -- version_from >= 203
END
$postgis_proc_upgrade$;
-- Operator class brin_geometry_inclusion_ops_4d -- LastUpdated: 203
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN

  IF 203 > version_from_num FROM _postgis_upgrade_info
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$
    CREATE OPERATOR CLASS brin_geometry_inclusion_ops_4d
  FOR TYPE geometry
  USING brin AS
    FUNCTION      1        brin_inclusion_opcinfo(internal),
    FUNCTION      2        geom4d_brin_inclusion_add_value(internal, internal, internal, internal),
    FUNCTION      3        brin_inclusion_consistent(internal, internal, internal),
    FUNCTION      4        brin_inclusion_union(internal, internal, internal),
    OPERATOR      3        &&&(geometry, geometry),
    OPERATOR      3        &&&(geometry, gidx),
    OPERATOR      3        &&&(gidx, geometry),
    OPERATOR      3        &&&(gidx, gidx),
  STORAGE gidx;
    $postgis_proc_upgrade_parsed_def$;
  END IF; -- version_from >= 203
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION _st_concavehull(param_inputgeom geometry)
  RETURNS geometry AS
$$
	DECLARE
	vexhull @extschema@.geometry;
	var_resultgeom @extschema@.geometry;
	var_inputgeom @extschema@.geometry;
	vexring @extschema@.geometry;
	cavering @extschema@.geometry;
	cavept @extschema@.geometry[];
	seglength double precision;
	var_tempgeom @extschema@.geometry;
	scale_factor float := 1;
	i integer;
	BEGIN
		-- First compute the ConvexHull of the geometry
		vexhull := @extschema@.ST_ConvexHull(param_inputgeom);
		var_inputgeom := param_inputgeom;
		--A point really has no concave hull
		IF @extschema@.ST_GeometryType(vexhull) = 'ST_Point' OR @extschema@.ST_GeometryType(vexHull) = 'ST_LineString' THEN
			RETURN vexhull;
		END IF;

		-- convert the hull perimeter to a linestring so we can manipulate individual points
		vexring := CASE WHEN @extschema@.ST_GeometryType(vexhull) = 'ST_LineString' THEN vexhull ELSE @extschema@.ST_ExteriorRing(vexhull) END;
		IF abs(@extschema@.ST_X(@extschema@.ST_PointN(vexring,1))) < 1 THEN --scale the geometry to prevent stupid precision errors - not sure it works so make low for now
			scale_factor := 100;
			vexring := @extschema@.ST_Scale(vexring, scale_factor,scale_factor);
			var_inputgeom := @extschema@.ST_Scale(var_inputgeom, scale_factor, scale_factor);
			--RAISE NOTICE 'Scaling';
		END IF;
		seglength := @extschema@.ST_Length(vexring)/least(@extschema@.ST_NPoints(vexring)*2,1000) ;

		vexring := @extschema@.ST_Segmentize(vexring, seglength);
		-- find the point on the original geom that is closest to each point of the convex hull and make a new linestring out of it.
		cavering := @extschema@.ST_Collect(
			ARRAY(

				SELECT
					@extschema@.ST_ClosestPoint(var_inputgeom, pt ) As the_geom
					FROM (
						SELECT  @extschema@.ST_PointN(vexring, n ) As pt, n
							FROM
							generate_series(1, @extschema@.ST_NPoints(vexring) ) As n
						) As pt

				)
			)
		;

		var_resultgeom := @extschema@.ST_MakeLine(geom)
			FROM @extschema@.ST_Dump(cavering) As foo;

		IF @extschema@.ST_IsSimple(var_resultgeom) THEN
			var_resultgeom := @extschema@.ST_MakePolygon(var_resultgeom);
			--RAISE NOTICE 'is Simple: %', var_resultgeom;
		ELSE 
			--RAISE NOTICE 'is not Simple: %', var_resultgeom;
			var_resultgeom := @extschema@.ST_ConvexHull(var_resultgeom);
		END IF;

		IF scale_factor > 1 THEN -- scale the result back
			var_resultgeom := @extschema@.ST_Scale(var_resultgeom, 1/scale_factor, 1/scale_factor);
		END IF;

		-- make sure result covers original (#3638)
		-- Using ST_UnaryUnion since SFCGAL doesn't replace with its own implementation
		-- and SFCGAL one chokes for some reason
		var_resultgeom := @extschema@.ST_UnaryUnion(@extschema@.ST_Collect(param_inputgeom, var_resultgeom) );
		RETURN var_resultgeom;

	END;
$$
LANGUAGE 'plpgsql' IMMUTABLE STRICT PARALLEL SAFE COST 10000;
CREATE OR REPLACE FUNCTION ST_ConcaveHull(param_geom geometry, param_pctconvex float, param_allow_holes boolean DEFAULT false) RETURNS geometry AS
$$
	DECLARE
		var_convhull @extschema@.geometry := @extschema@.ST_ForceSFS(@extschema@.ST_ConvexHull(param_geom));
		var_param_geom @extschema@.geometry := @extschema@.ST_ForceSFS(param_geom);
		var_initarea float := @extschema@.ST_Area(var_convhull);
		var_newarea float := var_initarea;
		var_div integer := 6; 
		var_tempgeom @extschema@.geometry;
		var_tempgeom2 @extschema@.geometry;
		var_cent @extschema@.geometry;
		var_geoms @extschema@.geometry[4]; 
		var_enline @extschema@.geometry;
		var_resultgeom @extschema@.geometry;
		var_atempgeoms @extschema@.geometry[];
		var_buf float := 1; 
	BEGIN
		-- We start with convex hull as our base
		var_resultgeom := var_convhull;

		IF param_pctconvex = 1 THEN
			-- this is the same as asking for the convex hull
			return var_resultgeom;
		ELSIF @extschema@.ST_GeometryType(var_param_geom) = 'ST_Polygon' THEN -- it is as concave as it is going to get
			IF param_allow_holes THEN -- leave the holes
				RETURN var_param_geom;
			ELSE -- remove the holes
				var_resultgeom := @extschema@.ST_MakePolygon(@extschema@.ST_ExteriorRing(var_param_geom));
				RETURN var_resultgeom;
			END IF;
		END IF;
		IF @extschema@.ST_Dimension(var_resultgeom) > 1 AND param_pctconvex BETWEEN 0 and 0.98 THEN
		-- get linestring that forms envelope of geometry
			var_enline := @extschema@.ST_Boundary(@extschema@.ST_Envelope(var_param_geom));
			var_buf := @extschema@.ST_Length(var_enline)/1000.0;
			IF @extschema@.ST_GeometryType(var_param_geom) = 'ST_MultiPoint' AND @extschema@.ST_NumGeometries(var_param_geom) BETWEEN 4 and 200 THEN
			-- we make polygons out of points since they are easier to cave in.
			-- Note we limit to between 4 and 200 points because this process is slow and gets quadratically slow
				var_buf := sqrt(@extschema@.ST_Area(var_convhull)*0.8/(@extschema@.ST_NumGeometries(var_param_geom)*@extschema@.ST_NumGeometries(var_param_geom)));
				var_atempgeoms := ARRAY(SELECT geom FROM @extschema@.ST_DumpPoints(var_param_geom));
				-- 5 and 10 and just fudge factors
				var_tempgeom := @extschema@.ST_Union(ARRAY(SELECT geom
						FROM (
						-- fuse near neighbors together
						SELECT DISTINCT ON (i) i,  @extschema@.ST_Distance(var_atempgeoms[i],var_atempgeoms[j]), @extschema@.ST_Buffer(@extschema@.ST_MakeLine(var_atempgeoms[i], var_atempgeoms[j]) , var_buf*5, 'quad_segs=3') As geom
								FROM generate_series(1,array_upper(var_atempgeoms, 1)) As i
									INNER JOIN generate_series(1,array_upper(var_atempgeoms, 1)) As j
										ON (
								 NOT @extschema@.ST_Intersects(var_atempgeoms[i],var_atempgeoms[j])
									AND @extschema@.ST_DWithin(var_atempgeoms[i],var_atempgeoms[j], var_buf*10)
									)
								UNION ALL
						-- catch the ones with no near neighbors
								SELECT i, 0, @extschema@.ST_Buffer(var_atempgeoms[i] , var_buf*10, 'quad_segs=3') As geom
								FROM generate_series(1,array_upper(var_atempgeoms, 1)) As i
									LEFT JOIN generate_series(ceiling(array_upper(var_atempgeoms,1)/2)::integer,array_upper(var_atempgeoms, 1)) As j
										ON (
								 NOT @extschema@.ST_Intersects(var_atempgeoms[i],var_atempgeoms[j])
									AND @extschema@.ST_DWithin(var_atempgeoms[i],var_atempgeoms[j], var_buf*10)
									)
									WHERE j IS NULL
								ORDER BY 1, 2
							) As foo	) );
				IF @extschema@.ST_IsValid(var_tempgeom) AND @extschema@.ST_GeometryType(var_tempgeom) = 'ST_Polygon' THEN
					var_tempgeom := @extschema@.ST_ForceSFS(@extschema@.ST_Intersection(var_tempgeom, var_convhull));
					IF param_allow_holes THEN
						var_param_geom := var_tempgeom;
					ELSIF @extschema@.ST_GeometryType(var_tempgeom) = 'ST_Polygon' THEN
						var_param_geom := @extschema@.ST_ForceSFS(@extschema@.ST_MakePolygon(@extschema@.ST_ExteriorRing(var_tempgeom)));
					ELSE
						var_param_geom := @extschema@.ST_ForceSFS(@extschema@.ST_ConvexHull(var_param_geom));
					END IF;
					-- make sure result covers original (#3638)
					var_param_geom := @extschema@.ST_Union(param_geom, var_param_geom);
					return var_param_geom;
				ELSIF @extschema@.ST_IsValid(var_tempgeom) THEN
					var_param_geom := @extschema@.ST_ForceSFS(@extschema@.ST_Intersection(var_tempgeom, var_convhull));
				END IF;
			END IF;

			IF @extschema@.ST_GeometryType(var_param_geom) = 'ST_Polygon' THEN
				IF NOT param_allow_holes THEN
					var_param_geom := @extschema@.ST_ForceSFS(@extschema@.ST_MakePolygon(@extschema@.ST_ExteriorRing(var_param_geom)));
				END IF;
				-- make sure result covers original (#3638)
				--var_param_geom := @extschema@.ST_Union(param_geom, var_param_geom);
				return var_param_geom;
			END IF;
            var_cent := @extschema@.ST_Centroid(var_param_geom);
            IF (@extschema@.ST_XMax(var_enline) - @extschema@.ST_XMin(var_enline) ) > var_buf AND (@extschema@.ST_YMax(var_enline) - @extschema@.ST_YMin(var_enline) ) > var_buf THEN
                    IF @extschema@.ST_Dwithin(@extschema@.ST_Centroid(var_convhull) , @extschema@.ST_Centroid(@extschema@.ST_Envelope(var_param_geom)), var_buf/2) THEN
                -- If the geometric dimension is > 1 and the object is symettric (cutting at centroid will not work -- offset a bit)
                        var_cent := @extschema@.ST_Translate(var_cent, (@extschema@.ST_XMax(var_enline) - @extschema@.ST_XMin(var_enline))/1000,  (@extschema@.ST_YMAX(var_enline) - @extschema@.ST_YMin(var_enline))/1000);
                    ELSE
                        -- uses closest point on geometry to centroid. I can't explain why we are doing this
                        var_cent := @extschema@.ST_ClosestPoint(var_param_geom,var_cent);
                    END IF;
                    IF @extschema@.ST_DWithin(var_cent, var_enline,var_buf) THEN
                        var_cent := @extschema@.ST_centroid(@extschema@.ST_Envelope(var_param_geom));
                    END IF;
                    -- break envelope into 4 triangles about the centroid of the geometry and returned the clipped geometry in each quadrant
                    FOR i in 1 .. 4 LOOP
                       var_geoms[i] := @extschema@.ST_MakePolygon(@extschema@.ST_MakeLine(ARRAY[@extschema@.ST_PointN(var_enline,i), @extschema@.ST_PointN(var_enline,i+1), var_cent, @extschema@.ST_PointN(var_enline,i)]));
                       var_geoms[i] := @extschema@.ST_ForceSFS(@extschema@.ST_Intersection(var_param_geom, @extschema@.ST_Buffer(var_geoms[i],var_buf)));
                       IF @extschema@.ST_IsValid(var_geoms[i]) THEN

                       ELSE
                            var_geoms[i] := @extschema@.ST_BuildArea(@extschema@.ST_MakeLine(ARRAY[@extschema@.ST_PointN(var_enline,i), @extschema@.ST_PointN(var_enline,i+1), var_cent, @extschema@.ST_PointN(var_enline,i)]));
                       END IF;
                    END LOOP;
                    var_tempgeom := @extschema@.ST_Union(ARRAY[@extschema@.ST_ConvexHull(var_geoms[1]), @extschema@.ST_ConvexHull(var_geoms[2]) , @extschema@.ST_ConvexHull(var_geoms[3]), @extschema@.ST_ConvexHull(var_geoms[4])]);
                    --RAISE NOTICE 'Curr vex % ', @extschema@.ST_AsText(var_tempgeom);
                    IF @extschema@.ST_Area(var_tempgeom) <= var_newarea AND @extschema@.ST_IsValid(var_tempgeom)  THEN --AND @extschema@.ST_GeometryType(var_tempgeom) ILIKE '%Polygon'

                        var_tempgeom := @extschema@.ST_Buffer(@extschema@.ST_ConcaveHull(var_geoms[1],least(param_pctconvex + param_pctconvex/var_div),true),var_buf, 'quad_segs=2');
                        FOR i IN 1 .. 4 LOOP
                            var_geoms[i] := @extschema@.ST_Buffer(@extschema@.ST_ConcaveHull(var_geoms[i],least(param_pctconvex + param_pctconvex/var_div),true), var_buf, 'quad_segs=2');
                            IF @extschema@.ST_IsValid(var_geoms[i]) Then
                                var_tempgeom := @extschema@.ST_Union(var_tempgeom, var_geoms[i]);
                            ELSE
                                RAISE NOTICE 'Not valid % %', i, @extschema@.ST_AsText(var_tempgeom);
                                var_tempgeom := @extschema@.ST_Union(var_tempgeom, @extschema@.ST_ConvexHull(var_geoms[i]));
                            END IF;
                        END LOOP;

                        --RAISE NOTICE 'Curr concave % ', @extschema@.ST_AsText(var_tempgeom);
                        IF @extschema@.ST_IsValid(var_tempgeom) THEN
                            var_resultgeom := var_tempgeom;
                        END IF;
                        var_newarea := @extschema@.ST_Area(var_resultgeom);
                    ELSIF @extschema@.ST_IsValid(var_tempgeom) THEN
                        var_resultgeom := var_tempgeom;
                    END IF;

                    IF @extschema@.ST_NumGeometries(var_resultgeom) > 1  THEN
                        var_tempgeom := @extschema@._ST_ConcaveHull(var_resultgeom);
                        IF @extschema@.ST_IsValid(var_tempgeom) AND @extschema@.ST_GeometryType(var_tempgeom) ILIKE 'ST_Polygon' THEN
                            var_resultgeom := var_tempgeom;
                        ELSE
                            var_resultgeom := @extschema@.ST_Buffer(var_tempgeom,var_buf, 'quad_segs=2');
                        END IF;
                    END IF;
                    IF param_allow_holes = false THEN
                    -- only keep exterior ring since we do not want holes
                        var_resultgeom := @extschema@.ST_MakePolygon(@extschema@.ST_ExteriorRing(var_resultgeom));
                    END IF;
                ELSE
                    var_resultgeom := @extschema@.ST_Buffer(var_resultgeom,var_buf);
                END IF;
                var_resultgeom := @extschema@.ST_ForceSFS(@extschema@.ST_Intersection(var_resultgeom, @extschema@.ST_ConvexHull(var_param_geom)));
            ELSE
                -- dimensions are too small to cut
                var_resultgeom := @extschema@._ST_ConcaveHull(var_param_geom);
            END IF;

            RETURN var_resultgeom;
	END;
$$
LANGUAGE 'plpgsql' IMMUTABLE STRICT PARALLEL SAFE COST 10000;
CREATE OR REPLACE FUNCTION _ST_AsX3D(int4, geometry, int4, int4, text)
	RETURNS TEXT
	AS '$libdir/postgis-3','LWGEOM_asX3D'
	LANGUAGE 'c' IMMUTABLE PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_AsX3D(geom geometry, maxdecimaldigits integer DEFAULT 15, options integer DEFAULT 0)
	RETURNS TEXT
	AS $$SELECT @extschema@._ST_AsX3D(3,$1,$2,$3,'');$$
	LANGUAGE 'sql' IMMUTABLE PARALLEL SAFE
	COST 500;
CREATE OR REPLACE FUNCTION ST_Angle(line1 geometry, line2 geometry)
	RETURNS float8 AS 'SELECT ST_Angle(St_StartPoint($1), ST_EndPoint($1), St_StartPoint($2), ST_EndPoint($2))'
	LANGUAGE 'sql' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
GRANT SELECT ON TABLE geography_columns TO public;
GRANT SELECT ON TABLE geometry_columns TO public;
GRANT SELECT ON TABLE spatial_ref_sys TO public;
CREATE OR REPLACE FUNCTION ST_3DLineInterpolatePoint(geometry, float8)
	RETURNS geometry
	AS '$libdir/postgis-3', 'ST_3DLineInterpolatePoint'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE
	COST 50;
CREATE OR REPLACE FUNCTION geometry_spgist_config_2d(internal, internal)
	RETURNS void
	AS '$libdir/postgis-3' ,'gserialized_spgist_config_2d'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_spgist_choose_2d(internal, internal)
	RETURNS void
	AS '$libdir/postgis-3' ,'gserialized_spgist_choose_2d'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_spgist_picksplit_2d(internal, internal)
	RETURNS void
	AS '$libdir/postgis-3' ,'gserialized_spgist_picksplit_2d'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_spgist_inner_consistent_2d(internal, internal)
	RETURNS void
	AS '$libdir/postgis-3' ,'gserialized_spgist_inner_consistent_2d'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_spgist_leaf_consistent_2d(internal, internal)
	RETURNS bool
	AS '$libdir/postgis-3' ,'gserialized_spgist_leaf_consistent_2d'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_spgist_compress_2d(internal)
	RETURNS internal
	AS '$libdir/postgis-3' ,'gserialized_spgist_compress_2d'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
-- Operator class spgist_geometry_ops_2d -- LastUpdated: 205
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN

  IF 205 > version_from_num FROM _postgis_upgrade_info
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$
    CREATE OPERATOR CLASS spgist_geometry_ops_2d
	DEFAULT FOR TYPE geometry USING SPGIST AS
	OPERATOR        1        <<  ,
	OPERATOR        2        &<	 ,
	OPERATOR        3        &&  ,
	OPERATOR        4        &>	 ,
	OPERATOR        5        >>	 ,
	OPERATOR        6        ~=	 ,
	OPERATOR        7        ~	 ,
	OPERATOR        8        @	 ,
	OPERATOR        9        &<| ,
	OPERATOR        10       <<| ,
	OPERATOR        11       |>> ,
	OPERATOR        12       |&> ,
	FUNCTION		1		geometry_spgist_config_2d(internal, internal),
	FUNCTION		2		geometry_spgist_choose_2d(internal, internal),
	FUNCTION		3		geometry_spgist_picksplit_2d(internal, internal),
	FUNCTION		4		geometry_spgist_inner_consistent_2d(internal, internal),
	FUNCTION		5		geometry_spgist_leaf_consistent_2d(internal, internal),
	FUNCTION		6		geometry_spgist_compress_2d(internal);
    $postgis_proc_upgrade_parsed_def$;
  END IF; -- version_from >= 205
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_overlaps_3d(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3' ,'gserialized_overlaps_3d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_contains_3d(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3' ,'gserialized_contains_3d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_contained_3d(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3' ,'gserialized_contained_3d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_same_3d(geom1 geometry, geom2 geometry)
	RETURNS boolean
	AS '$libdir/postgis-3' ,'gserialized_same_3d'
	LANGUAGE 'c' IMMUTABLE STRICT PARALLEL SAFE;
-- Operator geometry &/& geometry -- LastUpdated: 205
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 205 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '&/&' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR &/& (
	PROCEDURE = geometry_overlaps_3d,
	LEFTARG = geometry, RIGHTARG = geometry,
	COMMUTATOR = &/&,
	RESTRICT = gserialized_gist_sel_nd, JOIN = gserialized_gist_joinsel_nd
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator geometry @>> geometry -- LastUpdated: 205
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 205 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '@>>' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR @>> (
	PROCEDURE = geometry_contains_3d,
	LEFTARG = geometry, RIGHTARG = geometry,
	COMMUTATOR = <<@,
	RESTRICT = gserialized_gist_sel_nd, JOIN = gserialized_gist_joinsel_nd
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator geometry <<@ geometry -- LastUpdated: 205
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 205 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '<<@' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR <<@ (
	PROCEDURE = geometry_contained_3d,
	LEFTARG = geometry, RIGHTARG = geometry,
	COMMUTATOR = @>>,
	RESTRICT = gserialized_gist_sel_nd, JOIN = gserialized_gist_joinsel_nd
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
-- Operator geometry ~== geometry -- LastUpdated: 205
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN
  --IF 205 > version_from_num FROM _postgis_upgrade_info
    --We trust presence of operator rather than version info
    IF NOT EXISTS (
        SELECT o.oprname
        FROM
            pg_catalog.pg_operator o,
            pg_catalog.pg_type tl,
            pg_catalog.pg_type tr
        WHERE
            o.oprleft = tl.oid AND
            o.oprright = tr.oid AND
            o.oprcode != 0 AND
            o.oprname = '~==' AND
            tl.typname = 'geometry' AND
            tr.typname = 'geometry'
    )
    THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$ CREATE OPERATOR ~== (
	PROCEDURE = geometry_same_3d,
	LEFTARG = geometry, RIGHTARG = geometry,
	COMMUTATOR = ~==,
	RESTRICT = gserialized_gist_sel_nd, JOIN = gserialized_gist_joinsel_nd
);
 $postgis_proc_upgrade_parsed_def$;
  END IF;
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_spgist_config_3d(internal, internal)
	RETURNS void
	AS '$libdir/postgis-3', 'gserialized_spgist_config_3d'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_spgist_choose_3d(internal, internal)
	RETURNS void
	AS '$libdir/postgis-3', 'gserialized_spgist_choose_3d'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_spgist_picksplit_3d(internal, internal)
	RETURNS void
	AS '$libdir/postgis-3', 'gserialized_spgist_picksplit_3d'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_spgist_inner_consistent_3d(internal, internal)
	RETURNS void
	AS '$libdir/postgis-3', 'gserialized_spgist_inner_consistent_3d'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_spgist_leaf_consistent_3d(internal, internal)
	RETURNS bool
	AS '$libdir/postgis-3', 'gserialized_spgist_leaf_consistent_3d'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_spgist_compress_3d(internal)
	RETURNS internal
	AS '$libdir/postgis-3', 'gserialized_spgist_compress_3d'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
-- Operator class spgist_geometry_ops_3d -- LastUpdated: 205
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN

  IF 205 > version_from_num FROM _postgis_upgrade_info
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$
    CREATE OPERATOR CLASS spgist_geometry_ops_3d
	FOR TYPE geometry USING SPGIST AS
	OPERATOR        3        &/&	,
	OPERATOR        6        ~==	,
	OPERATOR        7        @>>	,
	OPERATOR        8        <<@	,
	FUNCTION	1	geometry_spgist_config_3d(internal, internal),
	FUNCTION	2	geometry_spgist_choose_3d(internal, internal),
	FUNCTION	3	geometry_spgist_picksplit_3d(internal, internal),
	FUNCTION	4	geometry_spgist_inner_consistent_3d(internal, internal),
	FUNCTION	5	geometry_spgist_leaf_consistent_3d(internal, internal),
	FUNCTION	6	geometry_spgist_compress_3d(internal);
    $postgis_proc_upgrade_parsed_def$;
  END IF; -- version_from >= 205
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geometry_spgist_config_nd(internal, internal)
	RETURNS void
	AS '$libdir/postgis-3' ,'gserialized_spgist_config_nd'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_spgist_choose_nd(internal, internal)
	RETURNS void
	AS '$libdir/postgis-3' ,'gserialized_spgist_choose_nd'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_spgist_picksplit_nd(internal, internal)
	RETURNS void
	AS '$libdir/postgis-3' ,'gserialized_spgist_picksplit_nd'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_spgist_inner_consistent_nd(internal, internal)
	RETURNS void
	AS '$libdir/postgis-3' ,'gserialized_spgist_inner_consistent_nd'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_spgist_leaf_consistent_nd(internal, internal)
	RETURNS bool
	AS '$libdir/postgis-3' ,'gserialized_spgist_leaf_consistent_nd'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geometry_spgist_compress_nd(internal)
	RETURNS internal
	AS '$libdir/postgis-3' ,'gserialized_spgist_compress_nd'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
-- Operator class spgist_geometry_ops_nd -- LastUpdated: 300
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN

  IF 300 > version_from_num FROM _postgis_upgrade_info
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$
    CREATE OPERATOR CLASS spgist_geometry_ops_nd
	FOR TYPE geometry USING SPGIST AS
	OPERATOR        3        &&& ,
	OPERATOR        6        ~~=	,
	OPERATOR        7        ~~	,
	OPERATOR        8       @@ 	,
	FUNCTION		1		geometry_spgist_config_nd(internal, internal),
	FUNCTION		2		geometry_spgist_choose_nd(internal, internal),
	FUNCTION		3		geometry_spgist_picksplit_nd(internal, internal),
	FUNCTION		4		geometry_spgist_inner_consistent_nd(internal, internal),
	FUNCTION		5		geometry_spgist_leaf_consistent_nd(internal, internal),
	FUNCTION		6		geometry_spgist_compress_nd(internal);
    $postgis_proc_upgrade_parsed_def$;
  END IF; -- version_from >= 300
END
$postgis_proc_upgrade$;
CREATE OR REPLACE FUNCTION geography_spgist_config_nd(internal, internal)
	RETURNS void
	AS '$libdir/postgis-3' ,'gserialized_spgist_config_nd'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geography_spgist_choose_nd(internal, internal)
	RETURNS void
	AS '$libdir/postgis-3' ,'gserialized_spgist_choose_nd'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geography_spgist_picksplit_nd(internal, internal)
	RETURNS void
	AS '$libdir/postgis-3' ,'gserialized_spgist_picksplit_nd'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geography_spgist_inner_consistent_nd(internal, internal)
	RETURNS void
	AS '$libdir/postgis-3' ,'gserialized_spgist_inner_consistent_nd'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geography_spgist_leaf_consistent_nd(internal, internal)
	RETURNS bool
	AS '$libdir/postgis-3' ,'gserialized_spgist_leaf_consistent_nd'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
CREATE OR REPLACE FUNCTION geography_spgist_compress_nd(internal)
	RETURNS internal
	AS '$libdir/postgis-3' ,'gserialized_spgist_compress_nd'
	LANGUAGE C IMMUTABLE STRICT PARALLEL SAFE;
-- Operator class spgist_geography_ops_nd -- LastUpdated: 300
DO LANGUAGE 'plpgsql'
$postgis_proc_upgrade$
BEGIN

  IF 300 > version_from_num FROM _postgis_upgrade_info
  THEN
    EXECUTE $postgis_proc_upgrade_parsed_def$
    CREATE OPERATOR CLASS spgist_geography_ops_nd
	DEFAULT FOR TYPE geography USING SPGIST AS
	OPERATOR        3        && ,
	FUNCTION		1		geography_spgist_config_nd(internal, internal),
	FUNCTION		2		geography_spgist_choose_nd(internal, internal),
	FUNCTION		3		geography_spgist_picksplit_nd(internal, internal),
	FUNCTION		4		geography_spgist_inner_consistent_nd(internal, internal),
	FUNCTION		5		geography_spgist_leaf_consistent_nd(internal, internal),
	FUNCTION		6		geography_spgist_compress_nd(internal);
    $postgis_proc_upgrade_parsed_def$;
  END IF; -- version_from >= 300
END
$postgis_proc_upgrade$;
SELECT postgis_extension_drop_if_exists('postgis', 'DROP TABLE _postgis_upgrade_info');
DROP TABLE _postgis_upgrade_info;
-- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
--
--
-- PostGIS - Spatial Types for PostgreSQL
-- http://postgis.net
--
-- Copyright (C) 2011-2020 Sandro Santilli <strk@kbt.io>
-- Copyright (C) 2010-2012 Regina Obe <lr@pcorp.us>
-- Copyright (C) 2009      Paul Ramsey <pramsey@cleverelephant.ca>
--
-- This is free software; you can redistribute and/or modify it under
-- the terms of the GNU General Public Licence. See the COPYING file.
--
-- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
--
-- This file contains drop commands for obsoleted items that need
-- to be dropped _after_ upgrade of old functions.
-- Changes to this file affect postgis_upgrade*.sql script.
--
-- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -

-- First drop old aggregates
SELECT postgis_extension_drop_if_exists('postgis', 'DROP AGGREGATE IF EXISTS memgeomunion(geometry)');
DROP AGGREGATE IF EXISTS memgeomunion(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP AGGREGATE IF EXISTS geomunion(geometry)');
DROP AGGREGATE IF EXISTS geomunion(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP AGGREGATE IF EXISTS polygonize(geometry)');
DROP AGGREGATE IF EXISTS polygonize(geometry); -- Deprecated in 1.2.3, Dropped in 2.0.0
SELECT postgis_extension_drop_if_exists('postgis', 'DROP AGGREGATE IF EXISTS collect(geometry)');
DROP AGGREGATE IF EXISTS collect(geometry); -- Deprecated in 1.2.3, Dropped in 2.0.0
SELECT postgis_extension_drop_if_exists('postgis', 'DROP AGGREGATE IF EXISTS st_geomunion(geometry)');
DROP AGGREGATE IF EXISTS st_geomunion(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP AGGREGATE IF EXISTS accum_old(geometry)');
DROP AGGREGATE IF EXISTS accum_old(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP AGGREGATE IF EXISTS st_accum_old(geometry)');
DROP AGGREGATE IF EXISTS st_accum_old(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP AGGREGATE IF EXISTS st_accum(geometry)');
DROP AGGREGATE IF EXISTS st_accum(geometry); -- Dropped in 3.0.0
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS pgis_geometry_accum_finalfn(internal)');
DROP FUNCTION IF EXISTS pgis_geometry_accum_finalfn(internal);

SELECT postgis_extension_drop_if_exists('postgis', 'DROP AGGREGATE IF EXISTS st_astwkb_agg(geometry, integer)');
DROP AGGREGATE IF EXISTS st_astwkb_agg(geometry, integer); -- temporarely introduced before 2.2.0 final
SELECT postgis_extension_drop_if_exists('postgis', 'DROP AGGREGATE IF EXISTS st_astwkb_agg(geometry, integer, bigint)');
DROP AGGREGATE IF EXISTS st_astwkb_agg(geometry, integer, bigint); -- temporarely introduced before 2.2.0 final
SELECT postgis_extension_drop_if_exists('postgis', 'DROP AGGREGATE IF EXISTS st_astwkbagg(geometry, integer)');
DROP AGGREGATE IF EXISTS st_astwkbagg(geometry, integer); -- temporarely introduced before 2.2.0 final
SELECT postgis_extension_drop_if_exists('postgis', 'DROP AGGREGATE IF EXISTS st_astwkbagg(geometry, integer, bigint)');
DROP AGGREGATE IF EXISTS st_astwkbagg(geometry, integer, bigint); -- temporarely introduced before 2.2.0 final
SELECT postgis_extension_drop_if_exists('postgis', 'DROP AGGREGATE IF EXISTS st_astwkbagg(geometry, integer, bigint, boolean)');
DROP AGGREGATE IF EXISTS st_astwkbagg(geometry, integer, bigint, boolean); -- temporarely introduced before 2.2.0 final
SELECT postgis_extension_drop_if_exists('postgis', 'DROP AGGREGATE IF EXISTS st_astwkbagg(geometry, integer, bigint, boolean, boolean)');
DROP AGGREGATE IF EXISTS st_astwkbagg(geometry, integer, bigint, boolean, boolean); -- temporarely introduced before 2.2.0 final

-- BEGIN Management functions that now have default param for typmod --
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS AddGeometryColumn(varchar, varchar, varchar, varchar, integer, varchar, integer)');
DROP FUNCTION IF EXISTS AddGeometryColumn(varchar, varchar, varchar, varchar, integer, varchar, integer);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS AddGeometryColumn(varchar, varchar, varchar, integer, varchar, integer)');
DROP FUNCTION IF EXISTS AddGeometryColumn(varchar, varchar, varchar, integer, varchar, integer);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS AddGeometryColumn(varchar, varchar, integer, varchar, integer)');
DROP FUNCTION IF EXISTS AddGeometryColumn(varchar, varchar, integer, varchar, integer);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS populate_geometry_columns()');
DROP FUNCTION IF EXISTS populate_geometry_columns();
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS populate_geometry_columns(oid)');
DROP FUNCTION IF EXISTS populate_geometry_columns(oid);

-- END Management functions now have default parameter for typmod --
-- Then drop old functions
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS box2d_overleft(box2d, box2d)');
DROP FUNCTION IF EXISTS box2d_overleft(box2d, box2d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS box2d_overright(box2d, box2d)');
DROP FUNCTION IF EXISTS box2d_overright(box2d, box2d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS box2d_left(box2d, box2d)');
DROP FUNCTION IF EXISTS box2d_left(box2d, box2d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS box2d_right(box2d, box2d)');
DROP FUNCTION IF EXISTS box2d_right(box2d, box2d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS box2d_contain(box2d, box2d)');
DROP FUNCTION IF EXISTS box2d_contain(box2d, box2d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS box2d_contained(box2d, box2d)');
DROP FUNCTION IF EXISTS box2d_contained(box2d, box2d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS box2d_overlap(box2d, box2d)');
DROP FUNCTION IF EXISTS box2d_overlap(box2d, box2d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS box2d_same(box2d, box2d)');
DROP FUNCTION IF EXISTS box2d_same(box2d, box2d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS box2d_intersects(box2d, box2d)');
DROP FUNCTION IF EXISTS box2d_intersects(box2d, box2d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_area(geography)');
DROP FUNCTION IF EXISTS st_area(geography); -- this one changed to use default parameters
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_AsGeoJson(geometry)');
DROP FUNCTION IF EXISTS ST_AsGeoJson(geometry); -- this one changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_AsGeoJson(geography)');
DROP FUNCTION IF EXISTS ST_AsGeoJson(geography); -- this one changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_AsGeoJson(geometry, int4)');
DROP FUNCTION IF EXISTS ST_AsGeoJson(geometry, int4); -- this one changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_AsGeoJson(geography, int4)');
DROP FUNCTION IF EXISTS ST_AsGeoJson(geography, int4); -- this one changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_AsGeoJson(int4, geometry)');
DROP FUNCTION IF EXISTS ST_AsGeoJson(int4, geometry); -- this one changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_AsGeoJson(int4, geography)');
DROP FUNCTION IF EXISTS ST_AsGeoJson(int4, geography); -- this one changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_AsGeoJson(int4, geometry, int4)');
DROP FUNCTION IF EXISTS ST_AsGeoJson(int4, geometry, int4); -- this one changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_AsGeoJson(int4, geography, int4)');
DROP FUNCTION IF EXISTS ST_AsGeoJson(int4, geography, int4); -- this one changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_AsGeoJson(int4, geography, int4, int4)');
DROP FUNCTION IF EXISTS ST_AsGeoJson(int4, geography, int4, int4); -- dropped because the version-first signature is dumb
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS _ST_AsGeoJson(int4, geometry, int4, int4)');
DROP FUNCTION IF EXISTS _ST_AsGeoJson(int4, geometry, int4, int4); -- dropped in PostGIS-3.0 (r17300)
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS _ST_AsGeoJson(int4, geography, int4, int4)');
DROP FUNCTION IF EXISTS _ST_AsGeoJson(int4, geography, int4, int4); -- dropped in PostGIS-3.0 (r17300)
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_asgml(geometry)');
DROP FUNCTION IF EXISTS st_asgml(geometry); -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_asgml(geometry, int4)');
DROP FUNCTION IF EXISTS st_asgml(geometry, int4);  -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_asgml(int4, geometry)');
DROP FUNCTION IF EXISTS st_asgml(int4, geometry);  -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_asgml(int4, geometry, int4)');
DROP FUNCTION IF EXISTS st_asgml(int4, geometry, int4);  -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_asgml(int4, geometry, int4, int4)');
DROP FUNCTION IF EXISTS st_asgml(int4, geometry, int4, int4);  -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_asgml(int4, geometry, int4, int4, text)');
DROP FUNCTION IF EXISTS st_asgml(int4, geometry, int4, int4, text); -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_asgml(geography)');
DROP FUNCTION IF EXISTS st_asgml(geography); -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_asgml(geography, int4)');
DROP FUNCTION IF EXISTS st_asgml(geography, int4);  -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_asgml(int4, geography)');
DROP FUNCTION IF EXISTS st_asgml(int4, geography);  -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_asgml(int4, geography, int4)');
DROP FUNCTION IF EXISTS st_asgml(int4, geography, int4);  -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_asgml(int4, geography, int4, int4)');
DROP FUNCTION IF EXISTS st_asgml(int4, geography, int4, int4);  -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_asgml(int4, geography, int4, int4, text)');
DROP FUNCTION IF EXISTS st_asgml(int4, geography, int4, int4, text); -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS _st_asgml(int4, geometry, int4, int4, text)');
DROP FUNCTION IF EXISTS _st_asgml(int4, geometry, int4, int4, text); -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS _st_asgml(int4, geography, int4, int4, text)');
DROP FUNCTION IF EXISTS _st_asgml(int4, geography, int4, int4, text); -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS _st_asgml(int4, geography, int4, int4, text, text)');
DROP FUNCTION IF EXISTS _st_asgml(int4, geography, int4, int4, text, text); -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_asgml(geography, int4, int4)');
DROP FUNCTION IF EXISTS st_asgml(geography, int4, int4);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS _st_askml(int4, geography, int4, text)');
DROP FUNCTION IF EXISTS _st_askml(int4, geography, int4, text); -- dropped in PostGIS-3.0 (r17300)
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS _st_askml(int4, geometry, int4, text)');
DROP FUNCTION IF EXISTS _st_askml(int4, geometry, int4, text); -- dropped in PostGIS-3.0 (r17300)
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_askml(geometry)');
DROP FUNCTION IF EXISTS st_askml(geometry); -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_askml(geography)');
DROP FUNCTION IF EXISTS st_askml(geography); -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_askml(int4, geometry, int4)');
DROP FUNCTION IF EXISTS st_askml(int4, geometry, int4); -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_askml(int4, geography, int4)');
DROP FUNCTION IF EXISTS st_askml(int4, geography, int4); -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_askml(int4, geography, int4, text)');
DROP FUNCTION IF EXISTS st_askml(int4, geography, int4, text); -- dropped because the version-first signature is dumb

SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_asx3d(geometry)');
DROP FUNCTION IF EXISTS st_asx3d(geometry); -- this one changed to use default parameters so full function deals with it
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_asx3d(geometry, int4)');
DROP FUNCTION IF EXISTS st_asx3d(geometry, int4); -- introduce variant with opts so get rid of other without ops
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_assvg(geometry)');
DROP FUNCTION IF EXISTS st_assvg(geometry); -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_assvg(geometry, int4)');
DROP FUNCTION IF EXISTS st_assvg(geometry, int4); -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_assvg(geography)');
DROP FUNCTION IF EXISTS st_assvg(geography); -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_assvg(geography, int4)');
DROP FUNCTION IF EXISTS st_assvg(geography, int4); -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_box2d_overleft(box2d, box2d)');
DROP FUNCTION IF EXISTS st_box2d_overleft(box2d, box2d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_box2d_overright(box2d, box2d)');
DROP FUNCTION IF EXISTS st_box2d_overright(box2d, box2d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_box2d_left(box2d, box2d)');
DROP FUNCTION IF EXISTS st_box2d_left(box2d, box2d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_box2d_right(box2d, box2d)');
DROP FUNCTION IF EXISTS st_box2d_right(box2d, box2d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_box2d_contain(box2d, box2d)');
DROP FUNCTION IF EXISTS st_box2d_contain(box2d, box2d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_box2d_contained(box2d, box2d)');
DROP FUNCTION IF EXISTS st_box2d_contained(box2d, box2d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_box2d_overlap(box2d, box2d)');
DROP FUNCTION IF EXISTS st_box2d_overlap(box2d, box2d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_box2d_same(box2d, box2d)');
DROP FUNCTION IF EXISTS st_box2d_same(box2d, box2d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_box2d_intersects(box2d, box2d)');
DROP FUNCTION IF EXISTS st_box2d_intersects(box2d, box2d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_box2d_in(cstring)');
DROP FUNCTION IF EXISTS st_box2d_in(cstring);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_box2d_out(box2d)');
DROP FUNCTION IF EXISTS st_box2d_out(box2d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_box2d(geometry)');
DROP FUNCTION IF EXISTS st_box2d(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_box2d(box3d)');
DROP FUNCTION IF EXISTS st_box2d(box3d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_box3d(box2d)');
DROP FUNCTION IF EXISTS st_box3d(box2d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_box(box3d)');
DROP FUNCTION IF EXISTS st_box(box3d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_box3d(geometry)');
DROP FUNCTION IF EXISTS st_box3d(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_box(geometry)');
DROP FUNCTION IF EXISTS st_box(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS _st_buffer(geometry, float8, cstring)');
DROP FUNCTION IF EXISTS _st_buffer(geometry, float8, cstring); -- dropped in PostGIS-3.0 (r17300)
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_ConcaveHull(geometry,float)');
DROP FUNCTION IF EXISTS ST_ConcaveHull(geometry,float); -- this one changed to use default parameters
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_DWithin(geography, geography, float8)');
DROP FUNCTION IF EXISTS ST_DWithin(geography, geography, float8); -- this one changed to use default parameters
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_text(geometry)');
DROP FUNCTION IF EXISTS st_text(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_geometry(box2d)');
DROP FUNCTION IF EXISTS st_geometry(box2d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_geometry(box3d)');
DROP FUNCTION IF EXISTS st_geometry(box3d);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_geometry(text)');
DROP FUNCTION IF EXISTS st_geometry(text);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_geometry(bytea)');
DROP FUNCTION IF EXISTS st_geometry(bytea);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_bytea(geometry)');
DROP FUNCTION IF EXISTS st_bytea(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_addbbox(geometry)');
DROP FUNCTION IF EXISTS st_addbbox(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS _st_distance(geography, geography, float8, boolean)');
DROP FUNCTION IF EXISTS _st_distance(geography, geography, float8, boolean); -- dropped in PostGIS-3.0 (r17300)
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_dropbbox(geometry)');
DROP FUNCTION IF EXISTS st_dropbbox(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_hasbbox(geometry)');
DROP FUNCTION IF EXISTS st_hasbbox(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS cache_bbox()');
DROP FUNCTION IF EXISTS cache_bbox();
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_cache_bbox()');
DROP FUNCTION IF EXISTS st_cache_bbox();
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_GeoHash(geometry)');
DROP FUNCTION IF EXISTS ST_GeoHash(geometry); -- changed to use default args
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_length(geography)');
DROP FUNCTION IF EXISTS st_length(geography); -- this one changed to use default parameters
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_perimeter(geography)');
DROP FUNCTION IF EXISTS st_perimeter(geography); -- this one changed to use default parameters
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS transform_geometry(geometry, text, text, int)');
DROP FUNCTION IF EXISTS transform_geometry(geometry, text, text, int);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS collector(geometry, geometry)');
DROP FUNCTION IF EXISTS collector(geometry, geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_collector(geometry, geometry)');
DROP FUNCTION IF EXISTS st_collector(geometry, geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS geom_accum (geometry[],geometry)');
DROP FUNCTION IF EXISTS geom_accum (geometry[],geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_geom_accum (geometry[],geometry)');
DROP FUNCTION IF EXISTS st_geom_accum (geometry[],geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS collect_garray (geometry[])');
DROP FUNCTION IF EXISTS collect_garray (geometry[]);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_collect_garray (geometry[])');
DROP FUNCTION IF EXISTS st_collect_garray (geometry[]);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS geosnoop(geometry)');
DROP FUNCTION IF EXISTS geosnoop(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS jtsnoop(geometry)');
DROP FUNCTION IF EXISTS jtsnoop(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_noop(geometry)');
DROP FUNCTION IF EXISTS st_noop(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_max_distance(geometry, geometry)');
DROP FUNCTION IF EXISTS st_max_distance(geometry, geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS  ST_MinimumBoundingCircle(geometry)');
DROP FUNCTION IF EXISTS  ST_MinimumBoundingCircle(geometry); --changed to use default parameters
-- Drop internals that should never have existed --
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_geometry_analyze(internal)');
DROP FUNCTION IF EXISTS st_geometry_analyze(internal);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_geometry_in(cstring)');
DROP FUNCTION IF EXISTS st_geometry_in(cstring);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_geometry_out(geometry)');
DROP FUNCTION IF EXISTS st_geometry_out(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_geometry_recv(internal)');
DROP FUNCTION IF EXISTS st_geometry_recv(internal);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_geometry_send(geometry)');
DROP FUNCTION IF EXISTS st_geometry_send(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_spheroid_in(cstring)');
DROP FUNCTION IF EXISTS st_spheroid_in(cstring);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_spheroid_out(spheroid)');
DROP FUNCTION IF EXISTS st_spheroid_out(spheroid);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_geometry_lt(geometry, geometry)');
DROP FUNCTION IF EXISTS st_geometry_lt(geometry, geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_geometry_gt(geometry, geometry)');
DROP FUNCTION IF EXISTS st_geometry_gt(geometry, geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_geometry_ge(geometry, geometry)');
DROP FUNCTION IF EXISTS st_geometry_ge(geometry, geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_geometry_eq(geometry, geometry)');
DROP FUNCTION IF EXISTS st_geometry_eq(geometry, geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_geometry_cmp(geometry, geometry)');
DROP FUNCTION IF EXISTS st_geometry_cmp(geometry, geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS SnapToGrid(geometry, float8, float8)');
DROP FUNCTION IF EXISTS SnapToGrid(geometry, float8, float8);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_removerepeatedpoints(geometry)');
DROP FUNCTION IF EXISTS st_removerepeatedpoints(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_voronoi(geometry, geometry, double precision, boolean)');
DROP FUNCTION IF EXISTS st_voronoi(geometry, geometry, double precision, boolean); --temporarely introduced before 2.3.0 final

SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS geometry_gist_sel_2d (internal, oid, internal, int4)');
DROP FUNCTION IF EXISTS geometry_gist_sel_2d (internal, oid, internal, int4);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS geometry_gist_joinsel_2d(internal, oid, internal, smallint)');
DROP FUNCTION IF EXISTS geometry_gist_joinsel_2d(internal, oid, internal, smallint);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS geography_gist_selectivity (internal, oid, internal, int4)');
DROP FUNCTION IF EXISTS geography_gist_selectivity (internal, oid, internal, int4);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS geography_gist_join_selectivity(internal, oid, internal, smallint)');
DROP FUNCTION IF EXISTS geography_gist_join_selectivity(internal, oid, internal, smallint);

SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_AsBinary(text)');
DROP FUNCTION IF EXISTS ST_AsBinary(text); -- deprecated in 2.0
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS postgis_uses_stats()');
DROP FUNCTION IF EXISTS postgis_uses_stats(); -- deprecated in 2.0
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_GeneratePoints(geometry, numeric)');
DROP FUNCTION IF EXISTS ST_GeneratePoints(geometry, numeric); -- numeric -> integer

-- Old accum aggregate support type, removed in 2.5.0
-- See #4035
SELECT postgis_extension_drop_if_exists('postgis', 'DROP TYPE IF EXISTS pgis_abs CASCADE');
DROP TYPE IF EXISTS pgis_abs CASCADE;

SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_astwkb(geometry, integer, bigint, bool, bool)');
DROP FUNCTION IF EXISTS st_astwkb(geometry, integer, bigint, bool, bool); -- temporarely introduced before 2.2.0 final
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS pgis_twkb_accum_transfn(internal, geometry, integer)');
DROP FUNCTION IF EXISTS pgis_twkb_accum_transfn(internal, geometry, integer); -- temporarely introduced before 2.2.0 final
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS pgis_twkb_accum_transfn(internal, geometry, integer, bigint)');
DROP FUNCTION IF EXISTS pgis_twkb_accum_transfn(internal, geometry, integer, bigint); -- temporarely introduced before 2.2.0 final
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS pgis_twkb_accum_transfn(internal, geometry, integer, bigint, bool)');
DROP FUNCTION IF EXISTS pgis_twkb_accum_transfn(internal, geometry, integer, bigint, bool); -- temporarely introduced before 2.2.0 final
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS pgis_twkb_accum_transfn(internal, geometry, integer, bigint, bool, bool)');
DROP FUNCTION IF EXISTS pgis_twkb_accum_transfn(internal, geometry, integer, bigint, bool, bool); -- temporarely introduced before 2.2.0 final
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS pgis_twkb_accum_finalfn(internal)');
DROP FUNCTION IF EXISTS pgis_twkb_accum_finalfn(internal); -- temporarely introduced before 2.2.0 final

SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_seteffectivearea(geometry, double precision)');
DROP FUNCTION IF EXISTS st_seteffectivearea(geometry, double precision); -- temporarely introduced before 2.2.0 final

SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS geometry_distance_box_nd(geometry, geometry)');
DROP FUNCTION IF EXISTS geometry_distance_box_nd(geometry, geometry); -- temporarely introduced before 2.2.0 final

SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS _ST_DumpPoints(geometry, integer[])');
DROP FUNCTION IF EXISTS _ST_DumpPoints(geometry, integer[]); -- removed 2.4.0, but really should have been removed 2.1.0 when ST_DumpPoints got reimpmented in C

-- Temporary clean-up while we wait to return these to action in dev
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS _ST_DistanceRectTree(g1 geometry, g2 geometry)');
DROP FUNCTION IF EXISTS _ST_DistanceRectTree(g1 geometry, g2 geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS _ST_DistanceRectTreeCached(g1 geometry, g2 geometry)');
DROP FUNCTION IF EXISTS _ST_DistanceRectTreeCached(g1 geometry, g2 geometry);

-- Deplicative signatures removed
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_Distance(geography, geography)');
DROP FUNCTION IF EXISTS ST_Distance(geography, geography);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_Distance(geography, geography, float8, boolean)');
DROP FUNCTION IF EXISTS ST_Distance(geography, geography, float8, boolean);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_Buffer(geometry, float8, cstring)');
DROP FUNCTION IF EXISTS ST_Buffer(geometry, float8, cstring);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_IsValidDetail(geometry)');
DROP FUNCTION IF EXISTS ST_IsValidDetail(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_AsKML(int4, geometry, int4, text)');
DROP FUNCTION IF EXISTS ST_AsKML(int4, geometry, int4, text);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_AsKML(geometry, int4)');
DROP FUNCTION IF EXISTS ST_AsKML(geometry, int4);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS ST_AsGeoJson(int4, geometry, int4, int4)');
DROP FUNCTION IF EXISTS ST_AsGeoJson(int4, geometry, int4, int4);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS _ST_AsGeoJson(int4, geometry, int4, int4)');
DROP FUNCTION IF EXISTS _ST_AsGeoJson(int4, geometry, int4, int4);

-- Underscore_signatures removed for CamelCase
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_shift_longitude(geometry)');
DROP FUNCTION IF EXISTS st_shift_longitude(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_estimated_extent(text,text,text)');
DROP FUNCTION IF EXISTS st_estimated_extent(text,text,text);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_estimated_extent(text,text)');
DROP FUNCTION IF EXISTS st_estimated_extent(text,text);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_find_extent(text,text,text)');
DROP FUNCTION IF EXISTS st_find_extent(text,text,text);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_find_extent(text,text)');
DROP FUNCTION IF EXISTS st_find_extent(text,text);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_mem_size(geometry)');
DROP FUNCTION IF EXISTS st_mem_size(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_3dlength_spheroid(geometry, spheroid)');
DROP FUNCTION IF EXISTS st_3dlength_spheroid(geometry, spheroid);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_length_spheroid(geometry, spheroid)');
DROP FUNCTION IF EXISTS st_length_spheroid(geometry, spheroid);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_length2d_spheroid(geometry, spheroid)');
DROP FUNCTION IF EXISTS st_length2d_spheroid(geometry, spheroid);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_distance_spheroid(geometry, geometry, spheroid)');
DROP FUNCTION IF EXISTS st_distance_spheroid(geometry, geometry, spheroid);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_point_inside_circle(geometry, float8, float8, float8)');
DROP FUNCTION IF EXISTS st_point_inside_circle(geometry, float8, float8, float8);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_force_2d(geometry)');
DROP FUNCTION IF EXISTS st_force_2d(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_force_3dz(geometry)');
DROP FUNCTION IF EXISTS st_force_3dz(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_force_3dm(geometry)');
DROP FUNCTION IF EXISTS st_force_3dm(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_force_collection(geometry)');
DROP FUNCTION IF EXISTS st_force_collection(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_force_4d(geometry)');
DROP FUNCTION IF EXISTS st_force_4d(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_force_3d(geometry)');
DROP FUNCTION IF EXISTS st_force_3d(geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_line_interpolate_point(geometry, float8)');
DROP FUNCTION IF EXISTS st_line_interpolate_point(geometry, float8);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_line_substring(geometry, float8, float8)');
DROP FUNCTION IF EXISTS st_line_substring(geometry, float8, float8);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_line_locate_point(geometry, geometry)');
DROP FUNCTION IF EXISTS st_line_locate_point(geometry, geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_locate_between_measures(geometry, float8, float8)');
DROP FUNCTION IF EXISTS st_locate_between_measures(geometry, float8, float8);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_locate_along_measure(geometry, float8)');
DROP FUNCTION IF EXISTS st_locate_along_measure(geometry, float8);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_combine_bbox(box3d, geometry)');
DROP FUNCTION IF EXISTS st_combine_bbox(box3d, geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_combine_bbox(box2d, geometry)');
DROP FUNCTION IF EXISTS st_combine_bbox(box2d, geometry);
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS st_distance_sphere(geometry, geometry)');
DROP FUNCTION IF EXISTS st_distance_sphere(geometry, geometry);

-- dev function 3.0 cycle
SELECT postgis_extension_drop_if_exists('postgis', 'DROP FUNCTION IF EXISTS pgis_geometry_union_transfn(internal, geometry)');
DROP FUNCTION IF EXISTS pgis_geometry_union_transfn(internal, geometry);

-- #4394
update pg_operator set oprcanhash = true, oprcanmerge = true where oprname = '=' and oprcode = 'geometry_eq'::regproc;


DO language 'plpgsql'
$$
BEGIN
IF _postgis_scripts_pgsql_version()::integer >= 96 THEN
-- mark ST_Union agg as parallel safe if it is not already
        BEGIN
            UPDATE pg_proc SET proparallel = 's'
            WHERE oid = 'st_union(geometry)'::regprocedure AND proparallel = 'u';
        EXCEPTION WHEN OTHERS THEN
            RAISE DEBUG 'Could not update st_union(geometry): %', SQLERRM;
        END;
END IF;
END;
$$;

	COMMENT ON TYPE box2d IS 'postgis type: A 2-dimensional bounding box. Used to describe the 2D extent of a geometry or collection of geometries.';

	COMMENT ON TYPE box3d IS 'postgis type: A 3-dimensional bounding box. Used to describe the 3D extent of a geometry or collection of geometries.';

	COMMENT ON TYPE geometry IS 'postgis type: The type representing spatial features with planar coordinate systems.';

	COMMENT ON TYPE geometry_dump IS 'postgis type: A composite type used to describe the parts of complex geometry.';

	COMMENT ON TYPE geography IS 'postgis type: The type representing spatial features with geodetic (ellipsoidal) coordinate systems.';

COMMENT ON FUNCTION AddGeometryColumn(varchar , varchar , integer , varchar , integer , boolean ) IS 'args: table_name, column_name, srid, type, dimension, use_typmod=true - Adds a geometry column to an existing table.';
			
COMMENT ON FUNCTION AddGeometryColumn(varchar , varchar , varchar , integer , varchar , integer , boolean ) IS 'args: schema_name, table_name, column_name, srid, type, dimension, use_typmod=true - Adds a geometry column to an existing table.';
			
COMMENT ON FUNCTION AddGeometryColumn(varchar , varchar , varchar , varchar , integer , varchar , integer , boolean ) IS 'args: catalog_name, schema_name, table_name, column_name, srid, type, dimension, use_typmod=true - Adds a geometry column to an existing table.';
			
COMMENT ON FUNCTION DropGeometryColumn(varchar , varchar ) IS 'args: table_name, column_name - Removes a geometry column from a spatial table.';
			
COMMENT ON FUNCTION DropGeometryColumn(varchar , varchar , varchar ) IS 'args: schema_name, table_name, column_name - Removes a geometry column from a spatial table.';
			
COMMENT ON FUNCTION DropGeometryColumn(varchar , varchar , varchar , varchar ) IS 'args: catalog_name, schema_name, table_name, column_name - Removes a geometry column from a spatial table.';
			
COMMENT ON FUNCTION DropGeometryTable(varchar ) IS 'args: table_name - Drops a table and all its references in geometry_columns.';
			
COMMENT ON FUNCTION DropGeometryTable(varchar , varchar ) IS 'args: schema_name, table_name - Drops a table and all its references in geometry_columns.';
			
COMMENT ON FUNCTION DropGeometryTable(varchar , varchar , varchar ) IS 'args: catalog_name, schema_name, table_name - Drops a table and all its references in geometry_columns.';
			
COMMENT ON FUNCTION Find_SRID(varchar , varchar , varchar ) IS 'args: a_schema_name, a_table_name, a_geomfield_name - Returns the SRID defined for a geometry column.';
			
COMMENT ON FUNCTION Populate_Geometry_Columns(boolean ) IS 'args: use_typmod=true - Ensures geometry columns are defined with type modifiers or have appropriate spatial constraints.';
			
COMMENT ON FUNCTION Populate_Geometry_Columns(oid, boolean ) IS 'args: relation_oid, use_typmod=true - Ensures geometry columns are defined with type modifiers or have appropriate spatial constraints.';
			
COMMENT ON FUNCTION UpdateGeometrySRID(varchar , varchar , integer ) IS 'args: table_name, column_name, srid - Updates the SRID of all features in a geometry column, and the table metadata.';
			
COMMENT ON FUNCTION UpdateGeometrySRID(varchar , varchar , varchar , integer ) IS 'args: schema_name, table_name, column_name, srid - Updates the SRID of all features in a geometry column, and the table metadata.';
			
COMMENT ON FUNCTION UpdateGeometrySRID(varchar , varchar , varchar , varchar , integer ) IS 'args: catalog_name, schema_name, table_name, column_name, srid - Updates the SRID of all features in a geometry column, and the table metadata.';
			
COMMENT ON FUNCTION ST_Collect(geometry, geometry) IS 'args: g1, g2 - Creates a GeometryCollection or Multi* geometry from a set of geometries.';
			
COMMENT ON FUNCTION ST_Collect(geometry[]) IS 'args: g1_array - Creates a GeometryCollection or Multi* geometry from a set of geometries.';
			
COMMENT ON AGGREGATE ST_Collect(geometry) IS 'args: g1field - Creates a GeometryCollection or Multi* geometry from a set of geometries.';
			
COMMENT ON FUNCTION ST_LineFromMultiPoint(geometry ) IS 'args: aMultiPoint - Creates a LineString from a MultiPoint geometry.';
			
COMMENT ON FUNCTION ST_MakeEnvelope(float, float, float, float, integer) IS 'args: xmin, ymin, xmax, ymax, srid=unknown - Creates a rectangular Polygon from minimum and maximum coordinates.';
			
COMMENT ON FUNCTION ST_MakeLine(geometry, geometry) IS 'args: geom1, geom2 - Creates a Linestring from Point, MultiPoint, or LineString geometries.';
			
COMMENT ON FUNCTION ST_MakeLine(geometry[]) IS 'args: geoms_array - Creates a Linestring from Point, MultiPoint, or LineString geometries.';
			
COMMENT ON AGGREGATE ST_MakeLine(geometry) IS 'args: geoms - Creates a Linestring from Point, MultiPoint, or LineString geometries.';
			
COMMENT ON FUNCTION ST_MakePoint(float, float) IS 'args: x, y - Creates a 2D, 3DZ or 4D Point.';
			
COMMENT ON FUNCTION ST_MakePoint(float, float, float) IS 'args: x, y, z - Creates a 2D, 3DZ or 4D Point.';
			
COMMENT ON FUNCTION ST_MakePoint(float, float, float, float) IS 'args: x, y, z, m - Creates a 2D, 3DZ or 4D Point.';
			
COMMENT ON FUNCTION ST_MakePointM(float, float, float) IS 'args: x, y, m - Creates a Point from X, Y and M values.';
			
COMMENT ON FUNCTION ST_MakePolygon(geometry) IS 'args: linestring - Creates a Polygon from a shell and optional list of holes.';
			
COMMENT ON FUNCTION ST_MakePolygon(geometry, geometry[]) IS 'args: outerlinestring, interiorlinestrings - Creates a Polygon from a shell and optional list of holes.';
			
COMMENT ON FUNCTION ST_Point(float, float) IS 'args: x, y - Creates a Point with the given coordinate values. Alias for ST_MakePoint.';
			
COMMENT ON FUNCTION ST_Polygon(geometry , integer ) IS 'args: lineString, srid - Creates a Polygon from a LineString with a specified SRID.';
			
COMMENT ON FUNCTION ST_TileEnvelope(integer, integer, integer, geometry, float) IS 'args: tileZoom, tileX, tileY, bounds=SRID=3857;LINESTRING(-20037508.342789 -20037508.342789,20037508.342789 20037508.342789), margin=0.0 - Creates a rectangular Polygon in Web Mercator (SRID:3857) using the XYZ tile system.';
			
COMMENT ON FUNCTION ST_HexagonGrid(float8, geometry) IS 'args: size, bounds - Returns a set of hexagons and cell indices that completely cover the bounds of the geometry argument.';
			
COMMENT ON FUNCTION ST_SquareGrid(float8, geometry) IS 'args: size, bounds - Returns a set of grid squares and cell indices that completely cover the bounds of the geometry argument.';
			
COMMENT ON FUNCTION ST_Hexagon(float8, integer, integer, geometry) IS 'args: size, cell_i, cell_j, origin - Returns a single hexagon, using the provided edge size and cell coordinate within the hexagon grid space.';
			
COMMENT ON FUNCTION ST_Square(float8, integer, integer, geometry) IS 'args: size, cell_i, cell_j, origin - Returns a single square, using the provided edge size and cell coordinate within the hexagon grid space.';
			
COMMENT ON FUNCTION GeometryType(geometry ) IS 'args: geomA - Returns the type of a geometry as text.';
			
COMMENT ON FUNCTION ST_Boundary(geometry ) IS 'args: geomA - Returns the boundary of a geometry.';
			
COMMENT ON FUNCTION ST_CoordDim(geometry ) IS 'args: geomA - Return the coordinate dimension of a geometry.';
			
COMMENT ON FUNCTION ST_Dimension(geometry ) IS 'args: g - Returns the topological dimension of a geometry.';
			
COMMENT ON FUNCTION ST_Dump(geometry ) IS 'args: g1 - Returns a set of geometry_dump rows for the components of a geometry.';
			
COMMENT ON FUNCTION ST_DumpPoints(geometry ) IS 'args: geom - Returns a set of geometry_dump rows for the points in a geometry.';
			
COMMENT ON FUNCTION ST_DumpRings(geometry ) IS 'args: a_polygon - Returns a set of geometry_dump rows for the exterior and interior rings of a Polygon.';
			
COMMENT ON FUNCTION ST_EndPoint(geometry ) IS 'args: g - Returns the last point of a LineString or CircularLineString.';
			
COMMENT ON FUNCTION ST_Envelope(geometry ) IS 'args: g1 - Returns a geometry representing the bounding box of a geometry.';
			
COMMENT ON FUNCTION ST_BoundingDiagonal(geometry , boolean ) IS 'args: geom, fits=false - Returns the diagonal of a geometrys bounding box.';
			
COMMENT ON FUNCTION ST_ExteriorRing(geometry ) IS 'args: a_polygon - Returns a LineString representing the exterior ring of a Polygon.';
			
COMMENT ON FUNCTION ST_GeometryN(geometry , integer ) IS 'args: geomA, n - Return the Nth geometry element of a geometry collection.';
			
COMMENT ON FUNCTION ST_GeometryType(geometry ) IS 'args: g1 - Returns the SQL-MM type of a geometry as text.';
			
COMMENT ON FUNCTION ST_HasArc(geometry ) IS 'args: geomA - Tests if a geometry contains a circular arc';
			
COMMENT ON FUNCTION ST_InteriorRingN(geometry , integer ) IS 'args: a_polygon, n - Returns the Nth interior ring (hole) of a Polygon.';
			
COMMENT ON FUNCTION ST_IsPolygonCCW(geometry) IS 'args: geom - Tests if Polygons have exterior rings oriented counter-clockwise and interior rings oriented clockwise.';
			
COMMENT ON FUNCTION ST_IsPolygonCW(geometry) IS 'args: geom - Tests if Polygons have exterior rings oriented clockwise and interior rings oriented counter-clockwise.';
			
COMMENT ON FUNCTION ST_IsClosed(geometry ) IS 'args: g - Tests if a LineStringss start and end points are coincident. For a PolyhedralSurface tests if it is closed (volumetric).';
			
COMMENT ON FUNCTION ST_IsCollection(geometry ) IS 'args: g - Tests if a geometry is a geometry collection type.';
			
COMMENT ON FUNCTION ST_IsEmpty(geometry ) IS 'args: geomA - Tests if a geometry is empty.';
			
COMMENT ON FUNCTION ST_IsRing(geometry ) IS 'args: g - Tests if a LineString is closed and simple.';
			
COMMENT ON FUNCTION ST_IsSimple(geometry ) IS 'args: geomA - Tests if a geometry has no points of self-intersection or self-tangency.';
			
COMMENT ON FUNCTION ST_M(geometry ) IS 'args: a_point - Returns the M coordinate of a Point.';
			
COMMENT ON FUNCTION ST_MemSize(geometry ) IS 'args: geomA - Returns the amount of memory space a geometry takes.';
			
COMMENT ON FUNCTION ST_NDims(geometry ) IS 'args: g1 - Returns the coordinate dimension of a geometry.';
			
COMMENT ON FUNCTION ST_NPoints(geometry ) IS 'args: g1 - Returns the number of points (vertices) in a geometry.';
			
COMMENT ON FUNCTION ST_NRings(geometry ) IS 'args: geomA - Returns the number of rings in a polygonal geometry.';
			
COMMENT ON FUNCTION ST_NumGeometries(geometry ) IS 'args: geom - Returns the number of elements in a geometry collection.';
			
COMMENT ON FUNCTION ST_NumInteriorRings(geometry ) IS 'args: a_polygon - Returns the number of interior rings (holes) of a Polygon.';
			
COMMENT ON FUNCTION ST_NumInteriorRing(geometry ) IS 'args: a_polygon - Returns the number of interior rings (holes) of a Polygon. Aias for ST_NumInteriorRings';
			
COMMENT ON FUNCTION ST_NumPatches(geometry ) IS 'args: g1 - Return the number of faces on a Polyhedral Surface. Will return null for non-polyhedral geometries.';
			
COMMENT ON FUNCTION ST_NumPoints(geometry ) IS 'args: g1 - Returns the number of points in a LineString or CircularString.';
			
COMMENT ON FUNCTION ST_PatchN(geometry , integer ) IS 'args: geomA, n - Returns the Nth geometry (face) of a PolyhedralSurface.';
			
COMMENT ON FUNCTION ST_PointN(geometry , integer ) IS 'args: a_linestring, n - Returns the Nth point in the first LineString or circular LineString in a geometry.';
			
COMMENT ON FUNCTION ST_Points(geometry) IS 'args: geom - Returns a MultiPoint containing all the coordinates of a geometry.';
			
COMMENT ON FUNCTION ST_StartPoint(geometry ) IS 'args: geomA - Returns the first point of a LineString.';
			
COMMENT ON FUNCTION ST_Summary(geometry ) IS 'args: g - Returns a text summary of the contents of a geometry.';
			
COMMENT ON FUNCTION ST_Summary(geography ) IS 'args: g - Returns a text summary of the contents of a geometry.';
			
COMMENT ON FUNCTION ST_X(geometry ) IS 'args: a_point - Returns the X coordinate of a Point.';
			
COMMENT ON FUNCTION ST_Y(geometry ) IS 'args: a_point - Returns the Y coordinate of a Point.';
			
COMMENT ON FUNCTION ST_Z(geometry ) IS 'args: a_point - Returns the Z coordinate of a Point.';
			
COMMENT ON FUNCTION ST_Zmflag(geometry ) IS 'args: geomA - Returns a code indicating the ZM coordinate dimension of a geometry.';
			
COMMENT ON FUNCTION ST_AddPoint(geometry, geometry) IS 'args: linestring, point - Add a point to a LineString.';
			
COMMENT ON FUNCTION ST_AddPoint(geometry, geometry, integer) IS 'args: linestring, point, position - Add a point to a LineString.';
			
COMMENT ON FUNCTION ST_CollectionExtract(geometry ) IS 'args: collection - Given a (multi)geometry, return a (multi)geometry consisting only of elements of the specified type.';
			
COMMENT ON FUNCTION ST_CollectionExtract(geometry , integer ) IS 'args: collection, type - Given a (multi)geometry, return a (multi)geometry consisting only of elements of the specified type.';
			
COMMENT ON FUNCTION ST_CollectionHomogenize(geometry ) IS 'args: collection - Given a geometry collection, return the "simplest" representation of the contents.';
			
COMMENT ON FUNCTION ST_CurveToLine(geometry, float, integer, integer) IS 'args: curveGeom, tolerance, tolerance_type, flags - Converts a geometry containing curves to a linear geometry.';
			
COMMENT ON FUNCTION ST_FlipCoordinates(geometry) IS 'args: geom - Returns a version of a geometry with X and Y axis flipped.';
			
COMMENT ON FUNCTION ST_Force2D(geometry ) IS 'args: geomA - Force the geometries into a "2-dimensional mode".';
			
COMMENT ON FUNCTION ST_Force3D(geometry , float ) IS 'args: geomA, Zvalue = 0.0 - Force the geometries into XYZ mode. This is an alias for ST_Force3DZ.';
			
COMMENT ON FUNCTION ST_Force3DZ(geometry , float ) IS 'args: geomA, Zvalue = 0.0 - Force the geometries into XYZ mode.';
			
COMMENT ON FUNCTION ST_Force3DM(geometry , float ) IS 'args: geomA, Mvalue = 0.0 - Force the geometries into XYM mode.';
			
COMMENT ON FUNCTION ST_Force4D(geometry , float , float ) IS 'args: geomA, Zvalue = 0.0, Mvalue = 0.0 - Force the geometries into XYZM mode.';
			
COMMENT ON FUNCTION ST_ForcePolygonCCW(geometry) IS 'args: geom - Orients all exterior rings counter-clockwise and all interior rings clockwise.';
			
COMMENT ON FUNCTION ST_ForceCollection(geometry ) IS 'args: geomA - Convert the geometry into a GEOMETRYCOLLECTION.';
			
COMMENT ON FUNCTION ST_ForcePolygonCW(geometry) IS 'args: geom - Orients all exterior rings clockwise and all interior rings counter-clockwise.';
			
COMMENT ON FUNCTION ST_ForceSFS(geometry ) IS 'args: geomA - Force the geometries to use SFS 1.1 geometry types only.';
			
COMMENT ON FUNCTION ST_ForceSFS(geometry , text ) IS 'args: geomA, version - Force the geometries to use SFS 1.1 geometry types only.';
			
COMMENT ON FUNCTION ST_ForceRHR(geometry) IS 'args: g - Force the orientation of the vertices in a polygon to follow the Right-Hand-Rule.';
			
COMMENT ON FUNCTION ST_ForceCurve(geometry) IS 'args: g - Upcast a geometry into its curved type, if applicable.';
			
COMMENT ON FUNCTION ST_LineMerge(geometry ) IS 'args: amultilinestring - Return a (set of) LineString(s) formed by sewing together a MULTILINESTRING.';
			
COMMENT ON FUNCTION ST_LineToCurve(geometry ) IS 'args: geomANoncircular - Converts a linear geometry to a curved geometry.';
			
COMMENT ON FUNCTION ST_Multi(geometry ) IS 'args: g1 - Return the geometry as a MULTI* geometry.';
			
COMMENT ON FUNCTION ST_Normalize(geometry ) IS 'args: geom - Return the geometry in its canonical form.';
			
COMMENT ON FUNCTION ST_QuantizeCoordinates(geometry, int, int, int, int) IS 'args: g, prec_x, prec_y, prec_z, prec_m - Sets least significant bits of coordinates to zero';
			
COMMENT ON FUNCTION ST_RemovePoint(geometry, integer) IS 'args: linestring, offset - Remove point from a linestring.';
			
COMMENT ON FUNCTION ST_RemoveRepeatedPoints(geometry, float8) IS 'args: geom, tolerance - Returns a version of the given geometry with duplicated points removed.';
			
COMMENT ON FUNCTION ST_Reverse(geometry ) IS 'args: g1 - Return the geometry with vertex order reversed.';
			
COMMENT ON FUNCTION ST_Segmentize(geometry , float ) IS 'args: geom, max_segment_length - Return a modified geometry/geography having no segment longer than the given distance.';
			
COMMENT ON FUNCTION ST_Segmentize(geography , float ) IS 'args: geog, max_segment_length - Return a modified geometry/geography having no segment longer than the given distance.';
			
COMMENT ON FUNCTION ST_SetPoint(geometry, integer, geometry) IS 'args: linestring, zerobasedposition, point - Replace point of a linestring with a given point.';
			
COMMENT ON FUNCTION ST_ShiftLongitude(geometry ) IS 'args: geomA - Shifts a geometry with geographic coordinates between -180..180 and 0..360.';
			
COMMENT ON FUNCTION ST_WrapX(geometry , float8 , float8 ) IS 'args: geom, wrap, move - Wrap a geometry around an X value.';
			
COMMENT ON FUNCTION ST_SnapToGrid(geometry , float , float , float , float ) IS 'args: geomA, originX, originY, sizeX, sizeY - Snap all points of the input geometry to a regular grid.';
			
COMMENT ON FUNCTION ST_SnapToGrid(geometry , float , float ) IS 'args: geomA, sizeX, sizeY - Snap all points of the input geometry to a regular grid.';
			
COMMENT ON FUNCTION ST_SnapToGrid(geometry , float ) IS 'args: geomA, size - Snap all points of the input geometry to a regular grid.';
			
COMMENT ON FUNCTION ST_SnapToGrid(geometry , geometry , float , float , float , float ) IS 'args: geomA, pointOrigin, sizeX, sizeY, sizeZ, sizeM - Snap all points of the input geometry to a regular grid.';
			
COMMENT ON FUNCTION ST_Snap(geometry , geometry , float ) IS 'args: input, reference, tolerance - Snap segments and vertices of input geometry to vertices of a reference geometry.';
			
COMMENT ON FUNCTION ST_SwapOrdinates(geometry, cstring) IS 'args: geom, ords - Returns a version of the given geometry with given ordinate values swapped.';
			
COMMENT ON FUNCTION ST_IsValid(geometry ) IS 'args: g - Tests if a geometry is well-formed in 2D.';
			
COMMENT ON FUNCTION ST_IsValid(geometry , integer ) IS 'args: g, flags - Tests if a geometry is well-formed in 2D.';
			
COMMENT ON FUNCTION ST_IsValidDetail(geometry , integer ) IS 'args: geom, flags - Returns a valid_detail row stating if a geometry is valid, and if not a reason why and a location.';
			
COMMENT ON FUNCTION ST_IsValidReason(geometry , integer ) IS 'args: geomA, flags - Returns text stating if a geometry is valid, or a reason for invalidity.';
			
COMMENT ON FUNCTION ST_IsValidReason(geometry ) IS 'args: geomA - Returns text stating if a geometry is valid, or a reason for invalidity.';
			
COMMENT ON FUNCTION ST_MakeValid(geometry) IS 'args: input - Attempts to make an invalid geometry valid without losing vertices.';
			
COMMENT ON FUNCTION ST_SetSRID(geometry , integer ) IS 'args: geom, srid - Set the SRID on a geometry to a particular integer value.';
			
COMMENT ON FUNCTION ST_SRID(geometry ) IS 'args: g1 - Returns the spatial reference identifier for the ST_Geometry as defined in spatial_ref_sys table.';
			
COMMENT ON FUNCTION ST_Transform(geometry , integer ) IS 'args: g1, srid - Return a new geometry with its coordinates transformed to a different spatial reference system.';
			
COMMENT ON FUNCTION ST_Transform(geometry , text ) IS 'args: geom, to_proj - Return a new geometry with its coordinates transformed to a different spatial reference system.';
			
COMMENT ON FUNCTION ST_Transform(geometry , text , text ) IS 'args: geom, from_proj, to_proj - Return a new geometry with its coordinates transformed to a different spatial reference system.';
			
COMMENT ON FUNCTION ST_Transform(geometry , text , integer ) IS 'args: geom, from_proj, to_srid - Return a new geometry with its coordinates transformed to a different spatial reference system.';
			
COMMENT ON FUNCTION ST_Area(geometry ) IS 'args: g1 - Returns the area of a polygonal geometry.';
			
COMMENT ON FUNCTION ST_Area(geography , boolean ) IS 'args: geog, use_spheroid=true - Returns the area of a polygonal geometry.';
			
COMMENT ON FUNCTION ST_Azimuth(geometry , geometry ) IS 'args: pointA, pointB - Returns the north-based azimuth as the angle in radians measured clockwise from the vertical on pointA to pointB.';
			
COMMENT ON FUNCTION ST_Azimuth(geography , geography ) IS 'args: pointA, pointB - Returns the north-based azimuth as the angle in radians measured clockwise from the vertical on pointA to pointB.';
			
COMMENT ON FUNCTION ST_Angle(geometry , geometry , geometry , geometry ) IS 'args: point1, point2, point3, point4 - Returns the angle between 3 points, or between 2 vectors (4 points or 2 lines).';
			
COMMENT ON FUNCTION ST_Angle(geometry , geometry ) IS 'args: line1, line2 - Returns the angle between 3 points, or between 2 vectors (4 points or 2 lines).';
			
COMMENT ON FUNCTION ST_ClosestPoint(geometry , geometry ) IS 'args: g1, g2 - Returns the 2D point on g1 that is closest to g2. This is the first point of the shortest line.';
			
COMMENT ON FUNCTION ST_3DClosestPoint(geometry , geometry ) IS 'args: g1, g2 - Returns the 3D point on g1 that is closest to g2. This is the first point of the 3D shortest line.';
			
COMMENT ON FUNCTION ST_Distance(geometry , geometry ) IS 'args: g1, g2 - Returns the distance between two geometry or geography values.';
			
COMMENT ON FUNCTION ST_Distance(geography , geography , boolean ) IS 'args: geog1, geog2, use_spheroid=true - Returns the distance between two geometry or geography values.';
			
COMMENT ON FUNCTION ST_3DDistance(geometry , geometry ) IS 'args: g1, g2 - Returns the 3D cartesian minimum distance (based on spatial ref) between two geometries in projected units.';
			
COMMENT ON FUNCTION ST_DistanceSphere(geometry , geometry ) IS 'args: geomlonlatA, geomlonlatB - Returns minimum distance in meters between two lon/lat geometries using a spherical earth model.';
			
COMMENT ON FUNCTION ST_DistanceSpheroid(geometry , geometry , spheroid ) IS 'args: geomlonlatA, geomlonlatB, measurement_spheroid - Returns the minimum distance between two lon/lat geometries using a spheroidal earth model.';
			
COMMENT ON FUNCTION ST_FrechetDistance(geometry , geometry , float) IS 'args: g1, g2, densifyFrac = -1 - Returns the Fréchet distance between two geometries.';
			
COMMENT ON FUNCTION ST_HausdorffDistance(geometry , geometry ) IS 'args: g1, g2 - Returns the Hausdorff distance between two geometries.';
			
COMMENT ON FUNCTION ST_HausdorffDistance(geometry , geometry , float) IS 'args: g1, g2, densifyFrac - Returns the Hausdorff distance between two geometries.';
			
COMMENT ON FUNCTION ST_Length(geometry ) IS 'args: a_2dlinestring - Returns the 2D length of a linear geometry.';
			
COMMENT ON FUNCTION ST_Length(geography , boolean ) IS 'args: geog, use_spheroid=true - Returns the 2D length of a linear geometry.';
			
COMMENT ON FUNCTION ST_Length2D(geometry ) IS 'args: a_2dlinestring - Returns the 2D length of a linear geometry. Alias for ST_Length';
			
COMMENT ON FUNCTION ST_3DLength(geometry ) IS 'args: a_3dlinestring - Returns the 3D length of a linear geometry.';
			
COMMENT ON FUNCTION ST_LengthSpheroid(geometry , spheroid ) IS 'args: a_geometry, a_spheroid - Returns the 2D or 3D length/perimeter of a lon/lat geometry on a spheroid.';
			
COMMENT ON FUNCTION ST_LongestLine(geometry , geometry ) IS 'args: g1, g2 - Returns the 2D longest line between two geometries.';
			
COMMENT ON FUNCTION ST_3DLongestLine(geometry , geometry ) IS 'args: g1, g2 - Returns the 3D longest line between two geometries';
			
COMMENT ON FUNCTION ST_MaxDistance(geometry , geometry ) IS 'args: g1, g2 - Returns the 2D largest distance between two geometries in projected units.';
			
COMMENT ON FUNCTION ST_3DMaxDistance(geometry , geometry ) IS 'args: g1, g2 - Returns the 3D cartesian maximum distance (based on spatial ref) between two geometries in projected units.';
			
COMMENT ON FUNCTION ST_MinimumClearance(geometry ) IS 'args: g - Returns the minimum clearance of a geometry, a measure of a geometrys robustness.';
			
COMMENT ON FUNCTION ST_MinimumClearanceLine(geometry ) IS 'args: g - Returns the two-point LineString spanning a geometrys minimum clearance.';
			
COMMENT ON FUNCTION ST_Perimeter(geometry ) IS 'args: g1 - Returns the length of the boundary of a polygonal geometry or geography.';
			
COMMENT ON FUNCTION ST_Perimeter(geography , boolean ) IS 'args: geog, use_spheroid=true - Returns the length of the boundary of a polygonal geometry or geography.';
			
COMMENT ON FUNCTION ST_Perimeter2D(geometry ) IS 'args: geomA - Returns the 2D perimeter of a polygonal geometry. Alias for ST_Perimeter.';
			
COMMENT ON FUNCTION ST_3DPerimeter(geometry ) IS 'args: geomA - Returns the 3D perimeter of a polygonal geometry.';
			
COMMENT ON FUNCTION ST_Project(geography , float , float ) IS 'args: g1, distance, azimuth - Returns a point projected from a start point by a distance and bearing (azimuth).';
			
COMMENT ON FUNCTION ST_ShortestLine(geometry , geometry ) IS 'args: g1, g2 - Returns the 2D shortest line between two geometries';
			
COMMENT ON FUNCTION ST_3DShortestLine(geometry , geometry ) IS 'args: g1, g2 - Returns the 3D shortest line between two geometries';
			
COMMENT ON FUNCTION ST_ClipByBox2D(geometry, box2d) IS 'args: geom, box - Returns the portion of a geometry falling within a rectangle.';
			
COMMENT ON FUNCTION ST_Difference(geometry , geometry , float8 ) IS 'args: geomA, geomB, gridSize = -1 - Returns a geometry representing the part of geometry A that does not intersect geometry B.';
			
COMMENT ON FUNCTION ST_Intersection(geometry, geometry, float8) IS 'args: geomA, geomB, gridSize = -1 - Returns a geometry representing the shared portion of geometries A and B.';
			
COMMENT ON FUNCTION ST_Intersection(geography, geography) IS 'args: geogA, geogB - Returns a geometry representing the shared portion of geometries A and B.';
			
COMMENT ON AGGREGATE ST_MemUnion(geometry) IS 'args: geomfield - Aggregate function which unions geometry in a memory-efficent but slower way';
			
COMMENT ON FUNCTION ST_Node(geometry ) IS 'args: geom - Nodes a collection of lines.';
			
COMMENT ON FUNCTION ST_Split(geometry, geometry) IS 'args: input, blade - Returns a collection of geometries created by splitting a geometry by another geometry.';
			
COMMENT ON FUNCTION ST_Subdivide(geometry, integer, float8) IS 'args: geom, max_vertices=256, gridSize = -1 - Computes a rectilinear subdivision of a geometry.';
			
COMMENT ON FUNCTION ST_SymDifference(geometry , geometry , float8 ) IS 'args: geomA, geomB, gridSize = -1 - Returns a geometry representing the portions of geometries A and B that do not intersect.';
			
COMMENT ON FUNCTION ST_Union(geometry, geometry) IS 'args: g1, g2 - Returns a geometry representing the point-set union of the input geometries.';
			
COMMENT ON FUNCTION ST_Union(geometry, geometry, float8) IS 'args: g1, g2, gridSize - Returns a geometry representing the point-set union of the input geometries.';
			
COMMENT ON FUNCTION ST_Union(geometry[]) IS 'args: g1_array - Returns a geometry representing the point-set union of the input geometries.';
			
COMMENT ON AGGREGATE ST_Union(geometry) IS 'args: g1field - Returns a geometry representing the point-set union of the input geometries.';
			
COMMENT ON AGGREGATE ST_Union(geometry, float8) IS 'args: g1field, gridSize - Returns a geometry representing the point-set union of the input geometries.';
			
COMMENT ON FUNCTION ST_UnaryUnion(geometry , float8 ) IS 'args: geom, gridSize = -1 - Computes the union of the components of a single geometry.';
			
COMMENT ON FUNCTION ST_Buffer(geometry , float , text ) IS 'args: g1, radius_of_buffer, buffer_style_parameters = '' - Returns a geometry covering all points within a given distance from a geometry.';
			
COMMENT ON FUNCTION ST_Buffer(geometry , float , integer ) IS 'args: g1, radius_of_buffer, num_seg_quarter_circle - Returns a geometry covering all points within a given distance from a geometry.';
			
COMMENT ON FUNCTION ST_Buffer(geography , float , text ) IS 'args: g1, radius_of_buffer, buffer_style_parameters - Returns a geometry covering all points within a given distance from a geometry.';
			
COMMENT ON FUNCTION ST_Buffer(geography , float , integer ) IS 'args: g1, radius_of_buffer, num_seg_quarter_circle - Returns a geometry covering all points within a given distance from a geometry.';
			
COMMENT ON FUNCTION ST_BuildArea(geometry ) IS 'args: geom - Creates a polygonal geometry formed by the linework of a geometry.';
			
COMMENT ON FUNCTION ST_Centroid(geometry ) IS 'args: g1 - Returns the geometric center of a geometry.';
			
COMMENT ON FUNCTION ST_Centroid(geography , boolean ) IS 'args: g1, use_spheroid=true - Returns the geometric center of a geometry.';
			
COMMENT ON FUNCTION ST_ConcaveHull(geometry , float , boolean ) IS 'args: geom, target_percent, allow_holes = false - Computes a possibly concave geometry that encloses all input geometry vertices';
			
COMMENT ON FUNCTION ST_ConvexHull(geometry ) IS 'args: geomA - Computes the convex hull of a geometry.';
			
COMMENT ON FUNCTION ST_DelaunayTriangles(geometry , float , int4 ) IS 'args: g1, tolerance, flags - Returns the Delaunay triangulation of the vertices of a geometry.';
			
COMMENT ON FUNCTION ST_FilterByM(geometry, double precision, double precision, boolean) IS 'args: geom, min, max = null, returnM = false - Removes vertices based on their M value';
			
COMMENT ON FUNCTION ST_GeneratePoints(geometry, integer) IS 'args: g, npoints - Generates random points contained in a Polygon or MultiPolygon.';
			
COMMENT ON FUNCTION ST_GeneratePoints(geometry, integer, integer) IS 'args: g, npoints, seed - Generates random points contained in a Polygon or MultiPolygon.';
			
COMMENT ON FUNCTION ST_GeometricMedian(geometry, float8, int, boolean) IS 'args: geom, tolerance = NULL, max_iter = 10000, fail_if_not_converged = false - Returns the geometric median of a MultiPoint.';
			
COMMENT ON FUNCTION ST_MaximumInscribedCircle(geometry ) IS 'args: geom - Computes the largest circle that is fully contained within a geometry.';
			
COMMENT ON FUNCTION ST_MinimumBoundingCircle(geometry , integer ) IS 'args: geomA, num_segs_per_qt_circ=48 - Returns the smallest circle polygon that contains a geometry.';
			
COMMENT ON FUNCTION ST_MinimumBoundingRadius(geometry) IS 'args: geom - Returns the center point and radius of the smallest circle that contains a geometry.';
			
COMMENT ON FUNCTION ST_OrientedEnvelope(geometry) IS 'args: geom - Returns a minimum-area rectangle containing a geometry.';
			
COMMENT ON FUNCTION ST_OffsetCurve(geometry , float , text ) IS 'args: line, signed_distance, style_parameters='' - Returns an offset line at a given distance and side from an input line.';
			
COMMENT ON FUNCTION ST_PointOnSurface(geometry ) IS 'args: g1 - Returns a point guaranteed to lie in a polygon or on a geometry.';
			
COMMENT ON AGGREGATE ST_Polygonize(geometry) IS 'args: geomfield - Computes a collection of polygons formed from the linework of a set of geometries.';
			
COMMENT ON FUNCTION ST_Polygonize(geometry[]) IS 'args: geom_array - Computes a collection of polygons formed from the linework of a set of geometries.';
			
COMMENT ON FUNCTION ST_ReducePrecision(geometry , float8 ) IS 'args: g, gridsize - Returns a valid geometry with all points rounded to the provided grid tolerance.';
			
COMMENT ON FUNCTION ST_SharedPaths(geometry, geometry) IS 'args: lineal1, lineal2 - Returns a collection containing paths shared by the two input linestrings/multilinestrings.';
			
COMMENT ON FUNCTION ST_Simplify(geometry, float, boolean) IS 'args: geomA, tolerance, preserveCollapsed - Returns a simplified version of a geometry, using the Douglas-Peucker algorithm.';
			
COMMENT ON FUNCTION ST_SimplifyPreserveTopology(geometry, float) IS 'args: geomA, tolerance - Returns a simplified and valid version of a geometry, using the Douglas-Peucker algorithm.';
			
COMMENT ON FUNCTION ST_SimplifyVW(geometry, float) IS 'args: geomA, tolerance - Returns a simplified version of a geometry, using the Visvalingam-Whyatt algorithm';
			
COMMENT ON FUNCTION ST_ChaikinSmoothing(geometry, integer, boolean) IS 'args: geom, nIterations = 1, preserveEndPoints = false - Returns a smoothed version of a geometry, using the Chaikin algorithm';
			
COMMENT ON FUNCTION ST_SetEffectiveArea(geometry, float, integer) IS 'args: geomA, threshold = 0, set_area = 1 - Sets the effective area for each vertex, using the Visvalingam-Whyatt algorithm.';
			
COMMENT ON FUNCTION ST_VoronoiLines(geometry, float8, geometry) IS 'args: g1, tolerance, extend_to - Returns the boundaries of the Voronoi diagram of the vertices of a geometry.';
			
COMMENT ON FUNCTION ST_VoronoiPolygons(geometry, float8, geometry) IS 'args: g1, tolerance, extend_to - Returns the cells of the Voronoi diagram of the vertices of a geometry.';
			
COMMENT ON FUNCTION ST_Affine(geometry , float , float , float , float , float , float , float , float , float , float , float , float ) IS 'args: geomA, a, b, c, d, e, f, g, h, i, xoff, yoff, zoff - Apply a 3D affine transformation to a geometry.';
			
COMMENT ON FUNCTION ST_Affine(geometry , float , float , float , float , float , float ) IS 'args: geomA, a, b, d, e, xoff, yoff - Apply a 3D affine transformation to a geometry.';
			
COMMENT ON FUNCTION ST_Rotate(geometry, float) IS 'args: geomA, rotRadians - Rotates a geometry about an origin point.';
			
COMMENT ON FUNCTION ST_Rotate(geometry, float, float, float) IS 'args: geomA, rotRadians, x0, y0 - Rotates a geometry about an origin point.';
			
COMMENT ON FUNCTION ST_Rotate(geometry, float, geometry) IS 'args: geomA, rotRadians, pointOrigin - Rotates a geometry about an origin point.';
			
COMMENT ON FUNCTION ST_RotateX(geometry, float) IS 'args: geomA, rotRadians - Rotates a geometry about the X axis.';
			
COMMENT ON FUNCTION ST_RotateY(geometry, float) IS 'args: geomA, rotRadians - Rotates a geometry about the Y axis.';
			
COMMENT ON FUNCTION ST_RotateZ(geometry, float) IS 'args: geomA, rotRadians - Rotates a geometry about the Z axis.';
			
COMMENT ON FUNCTION ST_Scale(geometry , float, float, float) IS 'args: geomA, XFactor, YFactor, ZFactor - Scales a geometry by given factors.';
			
COMMENT ON FUNCTION ST_Scale(geometry , float, float) IS 'args: geomA, XFactor, YFactor - Scales a geometry by given factors.';
			
COMMENT ON FUNCTION ST_Scale(geometry , geometry) IS 'args: geom, factor - Scales a geometry by given factors.';
			
COMMENT ON FUNCTION ST_Scale(geometry , geometry, geometry) IS 'args: geom, factor, origin - Scales a geometry by given factors.';
			
COMMENT ON FUNCTION ST_Translate(geometry , float , float ) IS 'args: g1, deltax, deltay - Translates a geometry by given offsets.';
			
COMMENT ON FUNCTION ST_Translate(geometry , float , float , float ) IS 'args: g1, deltax, deltay, deltaz - Translates a geometry by given offsets.';
			
COMMENT ON FUNCTION ST_TransScale(geometry , float, float, float, float) IS 'args: geomA, deltaX, deltaY, XFactor, YFactor - Translates and scales a geometry by given offsets and factors.';
			
COMMENT ON FUNCTION ST_ClusterDBSCAN(geometry, float8 , integer ) IS 'args: geom, eps, minpoints - Window function that returns a cluster id for each input geometry using the DBSCAN algorithm.';
			
COMMENT ON AGGREGATE ST_ClusterIntersecting(geometry) IS 'args: g - Aggregate function that clusters the input geometries into connected sets.';
			
COMMENT ON FUNCTION ST_ClusterKMeans(geometry, integer ) IS 'args: geom, number_of_clusters - Window function that returns a cluster id for each input geometry using the K-means algorithm.';
			
COMMENT ON AGGREGATE ST_ClusterWithin(geometry, float8 ) IS 'args: g, distance - Aggregate function that clusters the input geometries by separation distance.';
			
COMMENT ON FUNCTION Box2D(geometry ) IS 'args: geomA - Returns a BOX2D representing the 2D extent of the geometry.';
			
COMMENT ON FUNCTION Box3D(geometry ) IS 'args: geomA - Returns a BOX3D representing the 3D extent of the geometry.';
			
COMMENT ON FUNCTION ST_EstimatedExtent(text , text , text , boolean ) IS 'args: schema_name, table_name, geocolumn_name, parent_only - Return the estimated extent of a spatial table.';
			
COMMENT ON FUNCTION ST_EstimatedExtent(text , text , text ) IS 'args: schema_name, table_name, geocolumn_name - Return the estimated extent of a spatial table.';
			
COMMENT ON FUNCTION ST_EstimatedExtent(text , text ) IS 'args: table_name, geocolumn_name - Return the estimated extent of a spatial table.';
			
COMMENT ON FUNCTION ST_Expand(geometry , float) IS 'args: geom, units_to_expand - Returns a bounding box expanded from another bounding box or a geometry.';
			
COMMENT ON FUNCTION ST_Expand(geometry , float, float, float, float) IS 'args: geom, dx, dy, dz=0, dm=0 - Returns a bounding box expanded from another bounding box or a geometry.';
			
COMMENT ON FUNCTION ST_Expand(box2d , float) IS 'args: box, units_to_expand - Returns a bounding box expanded from another bounding box or a geometry.';
			
COMMENT ON FUNCTION ST_Expand(box2d , float, float) IS 'args: box, dx, dy - Returns a bounding box expanded from another bounding box or a geometry.';
			
COMMENT ON FUNCTION ST_Expand(box3d , float) IS 'args: box, units_to_expand - Returns a bounding box expanded from another bounding box or a geometry.';
			
COMMENT ON FUNCTION ST_Expand(box3d , float, float, float) IS 'args: box, dx, dy, dz=0 - Returns a bounding box expanded from another bounding box or a geometry.';
			
COMMENT ON AGGREGATE ST_Extent(geometry) IS 'args: geomfield - an aggregate function that returns the bounding box that bounds rows of geometries.';
			
COMMENT ON AGGREGATE ST_3DExtent(geometry) IS 'args: geomfield - an aggregate function that returns the 3D bounding box that bounds rows of geometries.';
			
COMMENT ON FUNCTION ST_MakeBox2D(geometry , geometry ) IS 'args: pointLowLeft, pointUpRight - Creates a BOX2D defined by two 2D point geometries.';
			
COMMENT ON FUNCTION ST_3DMakeBox(geometry , geometry ) IS 'args: point3DLowLeftBottom, point3DUpRightTop - Creates a BOX3D defined by two 3D point geometries.';
			
COMMENT ON FUNCTION ST_XMax(box3d ) IS 'args: aGeomorBox2DorBox3D - Returns the X maxima of a 2D or 3D bounding box or a geometry.';
			
COMMENT ON FUNCTION ST_XMin(box3d ) IS 'args: aGeomorBox2DorBox3D - Returns the X minima of a 2D or 3D bounding box or a geometry.';
			
COMMENT ON FUNCTION ST_YMax(box3d ) IS 'args: aGeomorBox2DorBox3D - Returns the Y maxima of a 2D or 3D bounding box or a geometry.';
			
COMMENT ON FUNCTION ST_YMin(box3d ) IS 'args: aGeomorBox2DorBox3D - Returns the Y minima of a 2D or 3D bounding box or a geometry.';
			
COMMENT ON FUNCTION ST_ZMax(box3d ) IS 'args: aGeomorBox2DorBox3D - Returns the Z maxima of a 2D or 3D bounding box or a geometry.';
			
COMMENT ON FUNCTION ST_ZMin(box3d ) IS 'args: aGeomorBox2DorBox3D - Returns the Z minima of a 2D or 3D bounding box or a geometry.';
			
COMMENT ON FUNCTION ST_LineInterpolatePoint(geometry , float8 ) IS 'args: a_linestring, a_fraction - Returns a point interpolated along a line. Second argument is a float8 between 0 and 1 representing fraction of total length of linestring the point has to be located.';
			
COMMENT ON FUNCTION ST_3DLineInterpolatePoint(geometry , float8 ) IS 'args: a_linestring, a_fraction - Returns a point interpolated along a line in 3D. Second argument is a float8 between 0 and 1 representing fraction of total length of linestring the point has to be located.';
			
COMMENT ON FUNCTION ST_LineInterpolatePoints(geometry , float8 , boolean ) IS 'args: a_linestring, a_fraction, repeat - Returns one or more points interpolated along a line.';
			
COMMENT ON FUNCTION ST_LineLocatePoint(geometry , geometry ) IS 'args: a_linestring, a_point - Returns a float between 0 and 1 representing the location of the closest point on LineString to the given Point, as a fraction of total 2d line length.';
			
COMMENT ON FUNCTION ST_LineSubstring(geometry , float8 , float8 ) IS 'args: a_linestring, startfraction, endfraction - Return a linestring being a substring of the input one starting and ending at the given fractions of total 2d length. Second and third arguments are float8 values between 0 and 1.';
			
COMMENT ON FUNCTION ST_LocateAlong(geometry , float8 , float8 ) IS 'args: ageom_with_measure, a_measure, offset - Return a derived geometry collection value with elements that match the specified measure. Polygonal elements are not supported.';
			
COMMENT ON FUNCTION ST_LocateBetween(geometry , float8 , float8 , float8 ) IS 'args: geom, measure_start, measure_end, offset - Return a derived geometry collection value with elements that match the specified range of measures inclusively.';
			
COMMENT ON FUNCTION ST_LocateBetweenElevations(geometry , float8 , float8 ) IS 'args: geom, elevation_start, elevation_end - Return a derived geometry (collection) value with elements that intersect the specified range of elevations inclusively.';
			
COMMENT ON FUNCTION ST_InterpolatePoint(geometry , geometry ) IS 'args: line, point - Return the value of the measure dimension of a geometry at the point closed to the provided point.';
			
COMMENT ON FUNCTION ST_AddMeasure(geometry , float8 , float8 ) IS 'args: geom_mline, measure_start, measure_end - Return a derived geometry with measure elements linearly interpolated between the start and end points.';
			
COMMENT ON FUNCTION ST_IsValidTrajectory(geometry ) IS 'args: line - Returns true if the geometry is a valid trajectory.';
			
COMMENT ON FUNCTION ST_ClosestPointOfApproach(geometry , geometry ) IS 'args: track1, track2 - Returns the measure at which points interpolated along two trajectories are closest.';
			
COMMENT ON FUNCTION ST_DistanceCPA(geometry , geometry ) IS 'args: track1, track2 - Returns the distance between the closest point of approach of two trajectories.';
			
COMMENT ON FUNCTION ST_CPAWithin(geometry , geometry , float8 ) IS 'args: track1, track2, maxdist - Returns true if the closest point of approach of two trajectoriesis within the specified distance.';
			
COMMENT ON FUNCTION AddAuth(text ) IS 'args: auth_token - Adds an authorization token to be used in the current transaction.';
			
COMMENT ON FUNCTION CheckAuth(text , text , text ) IS 'args: a_schema_name, a_table_name, a_key_column_name - Creates a trigger on a table to prevent/allow updates and deletes of rows based on authorization token.';
			
COMMENT ON FUNCTION CheckAuth(text , text ) IS 'args: a_table_name, a_key_column_name - Creates a trigger on a table to prevent/allow updates and deletes of rows based on authorization token.';
			
COMMENT ON FUNCTION DisableLongTransactions() IS 'Disables long transaction support.';
			
COMMENT ON FUNCTION EnableLongTransactions() IS 'Enables long transaction support.';
			
COMMENT ON FUNCTION LockRow(text , text , text , text, timestamp) IS 'args: a_schema_name, a_table_name, a_row_key, an_auth_token, expire_dt - Sets lock/authorization for a row in a table.';
			
COMMENT ON FUNCTION LockRow(text , text , text, timestamp) IS 'args: a_table_name, a_row_key, an_auth_token, expire_dt - Sets lock/authorization for a row in a table.';
			
COMMENT ON FUNCTION LockRow(text , text , text) IS 'args: a_table_name, a_row_key, an_auth_token - Sets lock/authorization for a row in a table.';
			
COMMENT ON FUNCTION UnlockRows(text ) IS 'args: auth_token - Removes all locks held by an authorization token.';
			
COMMENT ON FUNCTION PostGIS_Extensions_Upgrade() IS 'Packages and upgrades postgis extensions (e.g. postgis_raster,postgis_topology, postgis_sfcgal) to latest available version.';
			
COMMENT ON FUNCTION PostGIS_Full_Version() IS 'Reports full postgis version and build configuration infos.';
			
COMMENT ON FUNCTION PostGIS_GEOS_Version() IS 'Returns the version number of the GEOS library.';
			
COMMENT ON FUNCTION PostGIS_Liblwgeom_Version() IS 'Returns the version number of the liblwgeom library. This should match the version of PostGIS.';
			
COMMENT ON FUNCTION PostGIS_LibXML_Version() IS 'Returns the version number of the libxml2 library.';
			
COMMENT ON FUNCTION PostGIS_Lib_Build_Date() IS 'Returns build date of the PostGIS library.';
			
COMMENT ON FUNCTION PostGIS_Lib_Version() IS 'Returns the version number of the PostGIS library.';
			
COMMENT ON FUNCTION PostGIS_PROJ_Version() IS 'Returns the version number of the PROJ4 library.';
			
COMMENT ON FUNCTION PostGIS_Wagyu_Version() IS 'Returns the version number of the internal Wagyu library.';
			
COMMENT ON FUNCTION PostGIS_Scripts_Build_Date() IS 'Returns build date of the PostGIS scripts.';
			
COMMENT ON FUNCTION PostGIS_Scripts_Installed() IS 'Returns version of the postgis scripts installed in this database.';
			
COMMENT ON FUNCTION PostGIS_Scripts_Released() IS 'Returns the version number of the postgis.sql script released with the installed postgis lib.';
			
COMMENT ON FUNCTION PostGIS_Version() IS 'Returns PostGIS version number and compile-time options.';
			
COMMENT ON FUNCTION PostGIS_AddBBox(geometry ) IS 'args: geomA - Add bounding box to the geometry.';
			
COMMENT ON FUNCTION PostGIS_DropBBox(geometry ) IS 'args: geomA - Drop the bounding box cache from the geometry.';
			
COMMENT ON FUNCTION PostGIS_HasBBox(geometry ) IS 'args: geomA - Returns TRUE if the bbox of this geometry is cached, FALSE otherwise.';
			-- - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
--
----
-- PostGIS - Spatial Types for PostgreSQL
-- http://postgis.net
--
-- Copyright (C) 2011 Regina Obe <lr@pcorp.us>
--
-- This is free software; you can redistribute and/or modify it under
-- the terms of the GNU General Public Licence. See the COPYING file.
--
-- Author: Regina Obe <lr@pcorp.us>
--
-- This drops extension helper functions
-- and should be called at the end of the extension upgrade file
DROP FUNCTION postgis_extension_remove_objects(text, text);
DROP FUNCTION postgis_extension_drop_if_exists(text, text);
DROP FUNCTION postgis_extension_AddToSearchPath(varchar);
