/* ###
 * IP: GHIDRA
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 * 
 *      http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package ghidra.app.plugin.core.decompile;

import ghidra.app.decompiler.component.ClangTextField;
import ghidra.app.services.ClipboardContentProviderService;
import ghidra.app.util.ByteCopier;
import ghidra.app.util.ClipboardType;
import ghidra.util.task.TaskMonitor;

import java.awt.FontMetrics;
import java.awt.datatransfer.DataFlavor;
import java.awt.datatransfer.Transferable;
import java.math.BigInteger;
import java.util.*;
import java.util.concurrent.CopyOnWriteArraySet;

import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;

import docking.ActionContext;
import docking.ComponentProvider;
import docking.widgets.fieldpanel.Layout;
import docking.widgets.fieldpanel.LayoutModel;
import docking.widgets.fieldpanel.internal.PaintContext;
import docking.widgets.fieldpanel.support.FieldRange;
import docking.widgets.fieldpanel.support.FieldSelection;

public class DecompileClipboardProvider extends ByteCopier implements
		ClipboardContentProviderService {

	private static final PaintContext PAINT_CONTEXT = new PaintContext();
	private static final ClipboardType TEXT_TYPE = new ClipboardType(DataFlavor.stringFlavor,
		"Text");
	private static final List<ClipboardType> COPY_TYPES = new LinkedList<ClipboardType>();

	static {
		COPY_TYPES.add(TEXT_TYPE);
	}

	private DecompilerProvider provider;
	private FieldSelection selection;

	private boolean copyEnabled;
	private Set<ChangeListener> listeners = new CopyOnWriteArraySet<ChangeListener>();
	private int spaceCharWidthInPixels = 7;

	public DecompileClipboardProvider(DecompilePlugin plugin, DecompilerProvider provider) {
		this.provider = provider;
		this.tool = plugin.getTool();
		PAINT_CONTEXT.setTextCopying(true);
	}

	public void addChangeListener(ChangeListener listener) {
		listeners.add(listener);
	}

	public void removeChangeListener(ChangeListener listener) {
		listeners.remove(listener);
	}

	private void notifyStateChanged() {
		ChangeEvent event = new ChangeEvent(this);
		for (ChangeListener listener : listeners) {
			listener.stateChanged(event);
		}
	}

	public Transferable copy(TaskMonitor monitor) {
		return copyText(monitor);
	}

	public List<ClipboardType> getCurrentCopyTypes() {
		if (copyEnabled) {
			return COPY_TYPES;
		}
		return EMPTY_LIST;
	}

	public List<ClipboardType> getCurrentPasteTypes(Transferable t) {
		return null;
	}

	public Transferable copySpecial(ClipboardType copyType, TaskMonitor monitor) {
		if (copyType == TEXT_TYPE) {
			return copyText(monitor);
		}

		return null;
	}

	public boolean isValidContext(ActionContext context) {
		return context.getComponentProvider() == provider;
	}

	public void selectionChanged(FieldSelection sel) {
		this.selection = sel;
		copyEnabled = (selection != null && selection.getNumRanges() > 0);
		notifyStateChanged();
	}

	public ComponentProvider getComponentProvider() {
		return provider;
	}

	public boolean enableCopy() {
		return true;
	}

	public boolean enableCopySpecial() {
		return false;
	}

	public boolean canCopy() {
		return copyEnabled;
	}

	@Override
	public boolean canCopySpecial() {
		return false;
	}

	protected Transferable copyText(TaskMonitor monitor) {

//		TextLayoutGraphics g = new TextLayoutGraphics();
//		Rectangle rect = new Rectangle(2048, 2048);
//		LayoutBackgroundColorManager layoutColorMap = new EmptyLayoutBackgroundColorManager(PAINT_CONTEXT.background);
//		
//		LayoutModel model = provider.getDecompilerPanel().getLayoutModel();
//		int numRanges = selection.getNumRanges();
//		for (int i=0; i<numRanges; i++) {
//			FieldRange range = selection.getFieldRange(i);
//			for (int j=range.getStart().getIndex(); j<range.getEnd().getIndex(); j++) {
//				Layout layout = model.getLayout(j);
//				if (layout != null) {
//					layout.paint(g, PAINT_CONTEXT, rect, layoutColorMap, null);
//					g.setHeight(layout.getHeight());
//					g.flush();
//				}
//			}
//		}
//
//		return createStringTransferable(g.getBuffer().toString());
		return createStringTransferable(getText());
	}

	String getText() {
		StringBuffer buffer = new StringBuffer();
		int numRanges = selection.getNumRanges();
		for (int i = 0; i < numRanges; i++) {
			appendText(buffer, selection.getFieldRange(i));
		}
		return buffer.toString();
	}

	void appendText(StringBuffer buffer, FieldRange fieldRange) {
		int startIndex = fieldRange.getStart().getIndex().intValue();
		int endIndex = fieldRange.getEnd().getIndex().intValue();
		if (startIndex == endIndex) { // single line selection (don't include padding)
			appendTextSingleLine(buffer, startIndex, selection.intersect(startIndex));
			return;
		}
//		if (fieldRange.getEnd().getCol() == 0) {
//			endIndex--;
//		}
		appendText(buffer, startIndex, selection.intersect(startIndex));
		for (int line = startIndex + 1; line <= endIndex; line++) {
			buffer.append('\n');
			appendText(buffer, line, selection.intersect(line));
		}
	}

	private void appendText(StringBuffer buffer, int lineNumber, FieldSelection singleLineSelection) {
		if (singleLineSelection.isEmpty()) {
			return;
		}
		FieldRange fieldRange = singleLineSelection.getFieldRange(0);
		int startColumn = fieldRange.getStart().getCol();
		int endColumn = Integer.MAX_VALUE;
		int startRow = fieldRange.getStart().getRow();
		int endRow = Integer.MAX_VALUE;
		int startIndex = fieldRange.getStart().getIndex().intValue();
		int endIndex = fieldRange.getEnd().getIndex().intValue();
		if (startIndex == endIndex) {
			endColumn = fieldRange.getEnd().getCol();
			endRow = fieldRange.getEnd().getRow();
		}

		LayoutModel model = provider.getDecompilerPanel().getLayoutModel();
		Layout layout = model.getLayout(BigInteger.valueOf(lineNumber));
		ClangTextField field = (ClangTextField) layout.getField(0);
		int numSpaces = (field.getStartX() - field.getLineNumberWidth()) / spaceCharWidthInPixels;
		for (int i = 0; i < numSpaces; i++) {
			buffer.append(' ');
		}

		int startPos = field.screenLocationToTextOffset(startRow, startColumn);
		int endPos = field.screenLocationToTextOffset(endRow, endColumn);
		for (int i = 0; i < startPos; i++) {
			buffer.append(' ');
		}
		if (startPos >= 0 && endPos >= startPos) {
			buffer.append(field.getText().substring(startPos, endPos));
		}
	}

	private void appendTextSingleLine(StringBuffer buffer, int lineNumber,
			FieldSelection singleLineSelection) {
		if (singleLineSelection.isEmpty()) {
			return;
		}
		FieldRange fieldRange = singleLineSelection.getFieldRange(0);
		int startColumn = fieldRange.getStart().getCol();
		int endColumn = fieldRange.getEnd().getCol();
		int startRow = fieldRange.getStart().getRow();
		int endRow = fieldRange.getEnd().getRow();

		LayoutModel model = provider.getDecompilerPanel().getLayoutModel();
		Layout layout = model.getLayout(BigInteger.valueOf(lineNumber));
		ClangTextField field = (ClangTextField) layout.getField(0);

		int startPos = field.screenLocationToTextOffset(startRow, startColumn);
		int endPos = field.screenLocationToTextOffset(endRow, endColumn);

		if (startPos >= 0 && endPos >= startPos) {
			buffer.append(field.getText().substring(startPos, endPos));
		}
	}

//==================================================================================================
// Unsupported Operations
//==================================================================================================    

	public boolean enablePaste() {
		return false;
	}

	public boolean canPaste(DataFlavor[] availableFlavors) {
		return false;
	}

	public boolean paste(Transferable pasteData) {
		return false;
	}

	public boolean pasteSpecial(Transferable pasteData, ClipboardType pasteType) {
		return false;
	}

	public void lostOwnership(Transferable transferable) {
		// no-op
	}

	public void setFontMetrics(FontMetrics metrics) {
		spaceCharWidthInPixels = metrics.charWidth(' ');
	}
}
