"use strict";
var __assign = (this && this.__assign) || function () {
    __assign = Object.assign || function(t) {
        for (var s, i = 1, n = arguments.length; i < n; i++) {
            s = arguments[i];
            for (var p in s) if (Object.prototype.hasOwnProperty.call(s, p))
                t[p] = s[p];
        }
        return t;
    };
    return __assign.apply(this, arguments);
};
var __read = (this && this.__read) || function (o, n) {
    var m = typeof Symbol === "function" && o[Symbol.iterator];
    if (!m) return o;
    var i = m.call(o), r, ar = [], e;
    try {
        while ((n === void 0 || n-- > 0) && !(r = i.next()).done) ar.push(r.value);
    }
    catch (error) { e = { error: error }; }
    finally {
        try {
            if (r && !r.done && (m = i["return"])) m.call(i);
        }
        finally { if (e) throw e.error; }
    }
    return ar;
};
var __spread = (this && this.__spread) || function () {
    for (var ar = [], i = 0; i < arguments.length; i++) ar = ar.concat(__read(arguments[i]));
    return ar;
};
Object.defineProperty(exports, "__esModule", { value: true });
var d3_shape_1 = require("d3-shape");
var canvas_text_bbox_calculator_1 = require("../../../utils/bbox/canvas_text_bbox_calculator");
var scale_continuous_1 = require("../../../utils/scales/scale_continuous");
var scales_1 = require("../../../utils/scales/scales");
var curves_1 = require("../../../utils/curves");
var geometry_1 = require("../../../utils/geometry");
var commons_1 = require("../../../utils/commons");
function mutableIndexedGeometryMapUpsert(mutableGeometriesIndex, key, geometry) {
    var existing = mutableGeometriesIndex.get(key);
    var upsertGeometry = Array.isArray(geometry) ? geometry : [geometry];
    if (existing === undefined) {
        mutableGeometriesIndex.set(key, upsertGeometry);
    }
    else {
        mutableGeometriesIndex.set(key, __spread(upsertGeometry, existing));
    }
}
exports.mutableIndexedGeometryMapUpsert = mutableIndexedGeometryMapUpsert;
function getPointStyleOverrides(datum, seriesIdentifier, pointStyleAccessor) {
    var styleOverride = pointStyleAccessor && pointStyleAccessor(datum, seriesIdentifier);
    if (!styleOverride) {
        return;
    }
    if (typeof styleOverride === 'string') {
        return {
            stroke: styleOverride,
        };
    }
    return styleOverride;
}
exports.getPointStyleOverrides = getPointStyleOverrides;
function getBarStyleOverrides(datum, seriesIdentifier, seriesStyle, styleAccessor) {
    var styleOverride = styleAccessor && styleAccessor(datum, seriesIdentifier);
    if (!styleOverride) {
        return seriesStyle;
    }
    if (typeof styleOverride === 'string') {
        return __assign(__assign({}, seriesStyle), { rect: __assign(__assign({}, seriesStyle.rect), { fill: styleOverride }) });
    }
    return commons_1.mergePartial(seriesStyle, styleOverride, {
        mergeOptionalPartialValues: true,
    });
}
exports.getBarStyleOverrides = getBarStyleOverrides;
function renderPoints(shift, dataSeries, xScale, yScale, color, hasY0Accessors, styleAccessor) {
    var indexedGeometries = new Map();
    var isLogScale = scale_continuous_1.isLogarithmicScale(yScale);
    var pointGeometries = dataSeries.data.reduce(function (acc, datum) {
        var xValue = datum.x, y0 = datum.y0, y1 = datum.y1, initialY0 = datum.initialY0, initialY1 = datum.initialY1, filled = datum.filled;
        // don't create the point if not within the xScale domain or it that point was filled
        if (!xScale.isValueInDomain(xValue) || (filled && filled.y1 !== undefined)) {
            return acc;
        }
        var x = xScale.scale(xValue);
        var points = [];
        var yDatums = hasY0Accessors ? [y0, y1] : [y1];
        yDatums.forEach(function (yDatum, index) {
            // skip rendering point if y1 is null
            if (y1 === null) {
                return;
            }
            var y;
            var radius = 10;
            // we fix 0 and negative values at y = 0
            if (yDatum === null || (isLogScale && yDatum <= 0)) {
                y = yScale.range[0];
                radius = 0;
            }
            else {
                y = yScale.scale(yDatum);
            }
            var originalY = hasY0Accessors && index === 0 ? initialY0 : initialY1;
            var seriesIdentifier = {
                key: dataSeries.key,
                specId: dataSeries.specId,
                yAccessor: dataSeries.yAccessor,
                splitAccessors: dataSeries.splitAccessors,
                seriesKeys: dataSeries.seriesKeys,
            };
            var styleOverrides = getPointStyleOverrides(datum, seriesIdentifier, styleAccessor);
            var pointGeometry = {
                radius: radius,
                x: x,
                y: y,
                color: color,
                value: {
                    x: xValue,
                    y: originalY,
                    accessor: hasY0Accessors && index === 0 ? geometry_1.BandedAccessorType.Y0 : geometry_1.BandedAccessorType.Y1,
                },
                transform: {
                    x: shift,
                    y: 0,
                },
                seriesIdentifier: seriesIdentifier,
                styleOverrides: styleOverrides,
            };
            mutableIndexedGeometryMapUpsert(indexedGeometries, xValue, pointGeometry);
            // use the geometry only if the yDatum in contained in the current yScale domain
            var isHidden = yDatum === null || (isLogScale && yDatum <= 0);
            if (!isHidden && yScale.isValueInDomain(yDatum)) {
                points.push(pointGeometry);
            }
        });
        return __spread(acc, points);
    }, []);
    return {
        pointGeometries: pointGeometries,
        indexedGeometries: indexedGeometries,
    };
}
function renderBars(orderIndex, dataSeries, xScale, yScale, color, sharedSeriesStyle, displayValueSettings, styleAccessor, minBarHeight) {
    var indexedGeometries = new Map();
    var barGeometries = [];
    var bboxCalculator = new canvas_text_bbox_calculator_1.CanvasTextBBoxCalculator();
    // default padding to 1 for now
    var padding = 1;
    var fontSize = sharedSeriesStyle.displayValue.fontSize;
    var fontFamily = sharedSeriesStyle.displayValue.fontFamily;
    var absMinHeight = minBarHeight && Math.abs(minBarHeight);
    dataSeries.data.forEach(function (datum) {
        var y0 = datum.y0, y1 = datum.y1, initialY1 = datum.initialY1, filled = datum.filled;
        // don't create a bar if the initialY1 value is null.
        if (y1 === null || initialY1 === null || (filled && filled.y1 !== undefined)) {
            return;
        }
        // don't create a bar if not within the xScale domain
        if (!xScale.isValueInDomain(datum.x)) {
            return;
        }
        var y = 0;
        var y0Scaled;
        if (yScale.type === scales_1.ScaleType.Log) {
            y = y1 === 0 || y1 === null ? yScale.range[0] : yScale.scale(y1);
            if (yScale.isInverted) {
                y0Scaled = y0 === 0 || y0 === null ? yScale.range[1] : yScale.scale(y0);
            }
            else {
                y0Scaled = y0 === 0 || y0 === null ? yScale.range[0] : yScale.scale(y0);
            }
        }
        else {
            y = yScale.scale(y1);
            if (yScale.isInverted) {
                // use always zero as baseline if y0 is null
                y0Scaled = y0 === null ? yScale.scale(0) : yScale.scale(y0);
            }
            else {
                y0Scaled = y0 === null ? yScale.scale(0) : yScale.scale(y0);
            }
        }
        var height = y0Scaled - y;
        // handle minBarHeight adjustment
        if (absMinHeight !== undefined && height !== 0 && Math.abs(height) < absMinHeight) {
            var heightDelta = absMinHeight - Math.abs(height);
            if (height < 0) {
                height = -absMinHeight;
                y = y + heightDelta;
            }
            else {
                height = absMinHeight;
                y = y - heightDelta;
            }
        }
        var x = xScale.scale(datum.x) + xScale.bandwidth * orderIndex;
        var width = xScale.bandwidth;
        var formattedDisplayValue = displayValueSettings && displayValueSettings.valueFormatter
            ? displayValueSettings.valueFormatter(initialY1)
            : undefined;
        // only show displayValue for even bars if showOverlappingValue
        var displayValueText = displayValueSettings && displayValueSettings.isAlternatingValueLabel
            ? barGeometries.length % 2 === 0
                ? formattedDisplayValue
                : undefined
            : formattedDisplayValue;
        var computedDisplayValueWidth = bboxCalculator.compute(displayValueText || '', padding, fontSize, fontFamily)
            .width;
        var displayValueWidth = displayValueSettings && displayValueSettings.isValueContainedInElement ? width : computedDisplayValueWidth;
        var hideClippedValue = displayValueSettings ? displayValueSettings.hideClippedValue : undefined;
        var displayValue = displayValueSettings && displayValueSettings.showValueLabel
            ? {
                text: displayValueText,
                width: displayValueWidth,
                height: fontSize,
                hideClippedValue: hideClippedValue,
                isValueContainedInElement: displayValueSettings.isValueContainedInElement,
            }
            : undefined;
        var seriesIdentifier = {
            key: dataSeries.key,
            specId: dataSeries.specId,
            yAccessor: dataSeries.yAccessor,
            splitAccessors: dataSeries.splitAccessors,
            seriesKeys: dataSeries.seriesKeys,
        };
        var seriesStyle = getBarStyleOverrides(datum, seriesIdentifier, sharedSeriesStyle, styleAccessor);
        var barGeometry = {
            displayValue: displayValue,
            x: x,
            y: y,
            width: width,
            height: height,
            color: color,
            value: {
                x: datum.x,
                y: initialY1,
                accessor: geometry_1.BandedAccessorType.Y1,
            },
            seriesIdentifier: seriesIdentifier,
            seriesStyle: seriesStyle,
        };
        mutableIndexedGeometryMapUpsert(indexedGeometries, datum.x, barGeometry);
        barGeometries.push(barGeometry);
    });
    bboxCalculator.destroy();
    return {
        barGeometries: barGeometries,
        indexedGeometries: indexedGeometries,
    };
}
exports.renderBars = renderBars;
function renderLine(shift, dataSeries, xScale, yScale, color, curve, hasY0Accessors, xScaleOffset, seriesStyle, pointStyleAccessor, hasFit) {
    var isLogScale = scale_continuous_1.isLogarithmicScale(yScale);
    var pathGenerator = d3_shape_1.line()
        .x(function (_a) {
        var x = _a.x;
        return xScale.scale(x) - xScaleOffset;
    })
        .y(function (datum) {
        var yValue = exports.getYValue(datum);
        if (yValue !== null) {
            return yScale.scale(yValue);
        }
        // this should never happen thanks to the defined function
        return yScale.isInverted ? yScale.range[1] : yScale.range[0];
    })
        .defined(function (datum) {
        var yValue = exports.getYValue(datum);
        return yValue !== null && !(isLogScale && yValue <= 0) && xScale.isValueInDomain(datum.x);
    })
        .curve(curves_1.getCurveFactory(curve));
    var y = 0;
    var x = shift;
    var _a = renderPoints(shift - xScaleOffset, dataSeries, xScale, yScale, color, hasY0Accessors, pointStyleAccessor), pointGeometries = _a.pointGeometries, indexedGeometries = _a.indexedGeometries;
    var clippedRanges = hasFit && !hasY0Accessors ? getClippedRanges(dataSeries.data, xScale, xScaleOffset) : [];
    var lineGeometry = {
        line: pathGenerator(dataSeries.data) || '',
        points: pointGeometries,
        color: color,
        transform: {
            x: x,
            y: y,
        },
        seriesIdentifier: {
            key: dataSeries.key,
            specId: dataSeries.specId,
            yAccessor: dataSeries.yAccessor,
            splitAccessors: dataSeries.splitAccessors,
            seriesKeys: dataSeries.seriesKeys,
        },
        seriesLineStyle: seriesStyle.line,
        seriesPointStyle: seriesStyle.point,
        clippedRanges: clippedRanges,
    };
    return {
        lineGeometry: lineGeometry,
        indexedGeometries: indexedGeometries,
    };
}
exports.renderLine = renderLine;
/**
 * Returns value of `y1` or `filled.y1` or null
 */
exports.getYValue = function (_a) {
    var y1 = _a.y1, filled = _a.filled;
    if (y1 !== null) {
        return y1;
    }
    if (filled && filled.y1 !== undefined) {
        return filled.y1;
    }
    return null;
};
function renderArea(shift, dataSeries, xScale, yScale, color, curve, hasY0Accessors, xScaleOffset, seriesStyle, isStacked, pointStyleAccessor, hasFit) {
    if (isStacked === void 0) { isStacked = false; }
    var isLogScale = scale_continuous_1.isLogarithmicScale(yScale);
    var pathGenerator = d3_shape_1.area()
        .x(function (_a) {
        var x = _a.x;
        return xScale.scale(x) - xScaleOffset;
    })
        .y1(function (datum) {
        var yValue = exports.getYValue(datum);
        if (yValue !== null) {
            return yScale.scale(yValue);
        }
        // this should never happen thanks to the defined function
        return yScale.isInverted ? yScale.range[1] : yScale.range[0];
    })
        .y0(function (_a) {
        var y0 = _a.y0;
        if (y0 === null || (isLogScale && y0 <= 0)) {
            return yScale.range[0];
        }
        return yScale.scale(y0);
    })
        .defined(function (datum) {
        var yValue = exports.getYValue(datum);
        return yValue !== null && !(isLogScale && yValue <= 0) && xScale.isValueInDomain(datum.x);
    })
        .curve(curves_1.getCurveFactory(curve));
    var clippedRanges = hasFit && !hasY0Accessors && !isStacked ? getClippedRanges(dataSeries.data, xScale, xScaleOffset) : [];
    var y1Line = pathGenerator.lineY1()(dataSeries.data);
    var lines = [];
    if (y1Line) {
        lines.push(y1Line);
    }
    if (hasY0Accessors) {
        var y0Line = pathGenerator.lineY0()(dataSeries.data);
        if (y0Line) {
            lines.push(y0Line);
        }
    }
    var _a = renderPoints(shift - xScaleOffset, dataSeries, xScale, yScale, color, hasY0Accessors, pointStyleAccessor), pointGeometries = _a.pointGeometries, indexedGeometries = _a.indexedGeometries;
    var areaGeometry = {
        area: pathGenerator(dataSeries.data) || '',
        lines: lines,
        points: pointGeometries,
        color: color,
        transform: {
            y: 0,
            x: shift,
        },
        seriesIdentifier: {
            key: dataSeries.key,
            specId: dataSeries.specId,
            yAccessor: dataSeries.yAccessor,
            splitAccessors: dataSeries.splitAccessors,
            seriesKeys: dataSeries.seriesKeys,
        },
        seriesAreaStyle: seriesStyle.area,
        seriesAreaLineStyle: seriesStyle.line,
        seriesPointStyle: seriesStyle.point,
        isStacked: isStacked,
        clippedRanges: clippedRanges,
    };
    return {
        areaGeometry: areaGeometry,
        indexedGeometries: indexedGeometries,
    };
}
exports.renderArea = renderArea;
/**
 * Gets clipped ranges that have been fitted to values
 * @param dataset
 * @param xScale
 * @param xScaleOffset
 */
function getClippedRanges(dataset, xScale, xScaleOffset) {
    var firstNonNullX = null;
    var hasNull = false;
    return dataset.reduce(function (acc, _a) {
        var x = _a.x, y1 = _a.y1;
        var xValue = xScale.scale(x) - xScaleOffset + xScale.bandwidth / 2;
        if (y1 !== null) {
            if (hasNull) {
                if (firstNonNullX !== null) {
                    acc.push([firstNonNullX, xValue]);
                }
                else {
                    acc.push([0, xValue]);
                }
                hasNull = false;
            }
            firstNonNullX = xValue;
        }
        else {
            var endXValue = xScale.range[1] - xScale.bandwidth * (2 / 3);
            if (firstNonNullX !== null && xValue === endXValue) {
                acc.push([firstNonNullX, xValue]);
            }
            hasNull = true;
        }
        return acc;
    }, []);
}
exports.getClippedRanges = getClippedRanges;
function getGeometryStateStyle(seriesIdentifier, highlightedLegendItem, sharedGeometryStyle, individualHighlight) {
    var defaultStyles = sharedGeometryStyle.default, highlighted = sharedGeometryStyle.highlighted, unhighlighted = sharedGeometryStyle.unhighlighted;
    if (highlightedLegendItem != null) {
        var isPartOfHighlightedSeries = seriesIdentifier.key === highlightedLegendItem.seriesIdentifier.key;
        return isPartOfHighlightedSeries ? highlighted : unhighlighted;
    }
    if (individualHighlight) {
        var hasHighlight = individualHighlight.hasHighlight, hasGeometryHover = individualHighlight.hasGeometryHover;
        if (!hasGeometryHover) {
            return highlighted;
        }
        return hasHighlight ? highlighted : unhighlighted;
    }
    return defaultStyles;
}
exports.getGeometryStateStyle = getGeometryStateStyle;
function isPointOnGeometry(xCoordinate, yCoordinate, indexedGeometry) {
    var x = indexedGeometry.x, y = indexedGeometry.y;
    if (geometry_1.isPointGeometry(indexedGeometry)) {
        var radius = indexedGeometry.radius, transform = indexedGeometry.transform;
        return (yCoordinate >= y - radius &&
            yCoordinate <= y + radius &&
            xCoordinate >= x + transform.x - radius &&
            xCoordinate <= x + transform.x + radius);
    }
    var width = indexedGeometry.width, height = indexedGeometry.height;
    return yCoordinate >= y && yCoordinate <= y + height && xCoordinate >= x && xCoordinate <= x + width;
}
exports.isPointOnGeometry = isPointOnGeometry;
function getSeriesIdentifierPrefixedKey(seriesIdentifier, prefix, postfix) {
    return "" + (prefix || '') + seriesIdentifier.key + (postfix || '');
}
exports.getSeriesIdentifierPrefixedKey = getSeriesIdentifierPrefixedKey;
//# sourceMappingURL=rendering.js.map