# Copyright 2016 Google Inc. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
"""Changes a user's password in a given instance.

Changes a user's password in a given instance with specified username and host.
"""

import getpass

from googlecloudsdk.api_lib.sql import api_util
from googlecloudsdk.api_lib.sql import operations
from googlecloudsdk.calliope import base
from googlecloudsdk.command_lib.sql import flags
from googlecloudsdk.core import properties


@base.ReleaseTracks(base.ReleaseTrack.GA, base.ReleaseTrack.BETA)
class SetPassword(base.CreateCommand):
  """Changes a user's password in a given instance.

  Changes a user's password in a given instance with specified username and
  host.
  """

  @staticmethod
  def Args(parser):
    """Args is called by calliope to gather arguments for this command.

    Args:
      parser: An argparse parser that you can use it to add arguments that go
          on the command line after this command. Positional arguments are
          allowed.
    """
    flags.INSTANCE_FLAG.AddToParser(parser)
    flags.USERNAME_FLAG.AddToParser(parser)
    flags.HOST_FLAG.AddToParser(parser)
    password_group = parser.add_mutually_exclusive_group()
    flags.PASSWORD_FLAG.AddToParser(password_group)
    flags.PROMPT_FOR_PASSWORD_FLAG.AddToParser(password_group)
    base.ASYNC_FLAG.AddToParser(parser)

  def Run(self, args):
    """Changes a user's password in a given instance.

    Args:
      args: argparse.Namespace, The arguments that this command was invoked
          with.

    Returns:
      SQL user resource iterator.
    Raises:
      HttpException: An http error response was received while executing api
          request.
      ToolException: An error other than an http error occured while executing
          the command.
    """
    client = api_util.SqlClient(api_util.API_VERSION_DEFAULT)
    sql_client = client.sql_client
    sql_messages = client.sql_messages

    if args.prompt_for_password:
      args.password = getpass.getpass('Instance Password: ')

    instance_ref = client.resource_parser.Parse(
        args.instance,
        params={'project': properties.VALUES.core.project.GetOrFail},
        collection='sql.instances')
    operation_ref = None
    result_operation = sql_client.users.Update(
        sql_messages.SqlUsersUpdateRequest(
            project=instance_ref.project,
            instance=args.instance,
            name=args.username,
            host=args.host,
            user=sql_messages.User(
                project=instance_ref.project,
                instance=args.instance,
                name=args.username,
                host=args.host,
                password=args.password)))
    operation_ref = client.resource_parser.Create(
        'sql.operations',
        operation=result_operation.name,
        project=instance_ref.project)
    if args.async:
      return sql_client.operations.Get(
          sql_messages.SqlOperationsGetRequest(
              project=operation_ref.project, operation=operation_ref.operation))
    operations.OperationsV1Beta4.WaitForOperation(sql_client, operation_ref,
                                                  'Updating Cloud SQL user')
