# This file was automatically generated by bifcl from bro.bif.

##! A collection of built-in functions that implement a variety of things
##! such as general programming algorithms, string processing, math functions,
##! introspection, type conversion, file/directory manipulation, packet
##! filtering, interprocess communication and controlling protocol analyzer
##! behavior.
##!
##! You'll find most of Bro's built-in functions that aren't protocol-specific
##! in this file.

export {








## Returns the current wall-clock time.
##
## In general, you should use :bro:id:`network_time` instead
## unless you are using Bro for non-networking uses (such as general
## scripting; not particularly recommended), because otherwise your script
## may behave very differently on live traffic versus played-back traffic
## from a save file.
##
## Returns: The wall-clock time.
##
## .. bro:see:: network_time
global current_time: function(): time ;


## Returns the timestamp of the last packet processed. This function returns
## the timestamp of the most recently read packet, whether read from a
## live network interface or from a save file.
##
## Returns: The timestamp of the packet processed.
##
## .. bro:see:: current_time
global network_time: function(): time ;


## Returns a system environment variable.
##
## var: The name of the variable whose value to request.
##
## Returns: The system environment variable identified by *var*, or an empty
##          string if it is not defined.
##
## .. bro:see:: setenv
global getenv: function(var: string ): string ;


## Sets a system environment variable.
##
## var: The name of the variable.
##
## val: The (new) value of the variable *var*.
##
## Returns: True on success.
##
## .. bro:see:: getenv
global setenv: function(var: string , val: string ): bool ;


## Shuts down the Bro process immediately.
##
## code: The exit code to return with.
##
## .. bro:see:: terminate
global exit: function(code: int ): any ;


## Gracefully shut down Bro by terminating outstanding processing.
##
## Returns: True after successful termination and false when Bro is still in
##          the process of shutting down.
##
## .. bro:see:: exit bro_is_terminating
global terminate: function(): bool ;




## Invokes a command via the ``system`` function of the OS.
## The command runs in the background with ``stdout`` redirecting to
## ``stderr``. Here is a usage example:
## ``system(fmt("rm \"%s\"", str_shell_escape(sniffed_data)));``
##
## str: The command to execute.
##
## Returns: The return value from the OS ``system`` function.
##
## .. bro:see:: system_env str_shell_escape piped_exec
##
## .. note::
##
##      Note that this corresponds to the status of backgrounding the
##      given command, not to the exit status of the command itself. A
##      value of 127 corresponds to a failure to execute ``sh``, and -1
##      to an internal system failure.
global system: function(str: string ): int ;


## Invokes a command via the ``system`` function of the OS with a prepared
## environment. The function is essentially the same as :bro:id:`system`,
## but changes the environment before invoking the command.
##
## str: The command to execute.
##
## env: A :bro:type:`table` with the environment variables in the form
##      of key-value pairs. Each specified environment variable name
##      will be automatically prepended with ``BRO_ARG_``.
##
## Returns: The return value from the OS ``system`` function.
##
## .. bro:see:: system str_shell_escape piped_exec
global system_env: function(str: string , env: table_string_of_string ): int ;


## Opens a program with ``popen`` and writes a given string to the returned
## stream to send it to the opened process's stdin.
##
## program: The program to execute.
##
## to_write: Data to pipe to the opened program's process via ``stdin``.
##
## Returns: True on success.
##
## .. bro:see:: system system_env
global piped_exec: function(program: string , to_write: string ): bool ;




## Computes the MD5 hash value of the provided list of arguments.
##
## Returns: The MD5 hash value of the concatenated arguments.
##
## .. bro:see:: md5_hmac md5_hash_init md5_hash_update md5_hash_finish
##    sha1_hash sha1_hash_init sha1_hash_update sha1_hash_finish
##    sha256_hash sha256_hash_init sha256_hash_update sha256_hash_finish
##
## .. note::
##
##      This function performs a one-shot computation of its arguments.
##      For incremental hash computation, see :bro:id:`md5_hash_init` and
##      friends.
global md5_hash: function(va_args: any): string ;


## Computes the SHA1 hash value of the provided list of arguments.
##
## Returns: The SHA1 hash value of the concatenated arguments.
##
## .. bro:see:: md5_hash md5_hmac md5_hash_init md5_hash_update md5_hash_finish
##    sha1_hash_init sha1_hash_update sha1_hash_finish
##    sha256_hash sha256_hash_init sha256_hash_update sha256_hash_finish
##
## .. note::
##
##      This function performs a one-shot computation of its arguments.
##      For incremental hash computation, see :bro:id:`sha1_hash_init` and
##      friends.
global sha1_hash: function(va_args: any): string ;


## Computes the SHA256 hash value of the provided list of arguments.
##
## Returns: The SHA256 hash value of the concatenated arguments.
##
## .. bro:see:: md5_hash md5_hmac md5_hash_init md5_hash_update md5_hash_finish
##    sha1_hash sha1_hash_init sha1_hash_update sha1_hash_finish
##    sha256_hash_init sha256_hash_update sha256_hash_finish
##
## .. note::
##
##      This function performs a one-shot computation of its arguments.
##      For incremental hash computation, see :bro:id:`sha256_hash_init` and
##      friends.
global sha256_hash: function(va_args: any): string ;


## Computes an HMAC-MD5 hash value of the provided list of arguments. The HMAC
## secret key is generated from available entropy when Bro starts up, or it can
## be specified for repeatability using the ``-K`` command line flag.
##
## Returns: The HMAC-MD5 hash value of the concatenated arguments.
##
## .. bro:see:: md5_hash md5_hash_init md5_hash_update md5_hash_finish
##    sha1_hash sha1_hash_init sha1_hash_update sha1_hash_finish
##    sha256_hash sha256_hash_init sha256_hash_update sha256_hash_finish
global md5_hmac: function(va_args: any): string ;


## Constructs an MD5 handle to enable incremental hash computation. You can
## feed data to the returned opaque value with :bro:id:`md5_hash_update` and
## eventually need to call :bro:id:`md5_hash_finish` to finish the computation
## and get the hash digest.
##
## For example, when computing incremental MD5 values of transferred files in
## multiple concurrent HTTP connections, one keeps an optional handle in the
## HTTP session record. Then, one would call
## ``c$http$md5_handle = md5_hash_init()`` once before invoking
## ``md5_hash_update(c$http$md5_handle, some_more_data)`` in the
## :bro:id:`http_entity_data` event handler. When all data has arrived, a call
## to :bro:id:`md5_hash_finish` returns the final hash value.
##
## Returns: The opaque handle associated with this hash computation.
##
## .. bro:see:: md5_hmac md5_hash md5_hash_update md5_hash_finish
##    sha1_hash sha1_hash_init sha1_hash_update sha1_hash_finish
##    sha256_hash sha256_hash_init sha256_hash_update sha256_hash_finish
global md5_hash_init: function(): opaque of md5 ;


## Constructs an SHA1 handle to enable incremental hash computation. You can
## feed data to the returned opaque value with :bro:id:`sha1_hash_update` and
## finally need to call :bro:id:`sha1_hash_finish` to finish the computation
## and get the hash digest.
##
## For example, when computing incremental SHA1 values of transferred files in
## multiple concurrent HTTP connections, one keeps an optional handle in the
## HTTP session record. Then, one would call
## ``c$http$sha1_handle = sha1_hash_init()`` once before invoking
## ``sha1_hash_update(c$http$sha1_handle, some_more_data)`` in the
## :bro:id:`http_entity_data` event handler. When all data has arrived, a call
## to :bro:id:`sha1_hash_finish` returns the final hash value.
##
## Returns: The opaque handle associated with this hash computation.
##
## .. bro:see:: md5_hmac md5_hash md5_hash_init md5_hash_update md5_hash_finish
##    sha1_hash sha1_hash_update sha1_hash_finish
##    sha256_hash sha256_hash_init sha256_hash_update sha256_hash_finish
global sha1_hash_init: function(): opaque of sha1 ;


## Constructs an SHA256 handle to enable incremental hash computation. You can
## feed data to the returned opaque value with :bro:id:`sha256_hash_update` and
## finally need to call :bro:id:`sha256_hash_finish` to finish the computation
## and get the hash digest.
##
## For example, when computing incremental SHA256 values of transferred files in
## multiple concurrent HTTP connections, one keeps an optional handle in the
## HTTP session record. Then, one would call
## ``c$http$sha256_handle = sha256_hash_init()`` once before invoking
## ``sha256_hash_update(c$http$sha256_handle, some_more_data)`` in the
## :bro:id:`http_entity_data` event handler. When all data has arrived, a call
## to :bro:id:`sha256_hash_finish` returns the final hash value.
##
## Returns: The opaque handle associated with this hash computation.
##
## .. bro:see:: md5_hmac md5_hash md5_hash_init md5_hash_update md5_hash_finish
##    sha1_hash sha1_hash_init sha1_hash_update sha1_hash_finish
##    sha256_hash sha256_hash_update sha256_hash_finish
global sha256_hash_init: function(): opaque of sha256 ;


## Updates the MD5 value associated with a given index. It is required to
## call :bro:id:`md5_hash_init` once before calling this
## function.
##
## handle: The opaque handle associated with this hash computation.
##
## data: The data to add to the hash computation.
##
## Returns: True on success.
##
## .. bro:see:: md5_hmac md5_hash md5_hash_init md5_hash_finish
##    sha1_hash sha1_hash_init sha1_hash_update sha1_hash_finish
##    sha256_hash sha256_hash_init sha256_hash_update sha256_hash_finish
global md5_hash_update: function(handle: opaque of md5 , data: string ): bool ;


## Updates the SHA1 value associated with a given index. It is required to
## call :bro:id:`sha1_hash_init` once before calling this
## function.
##
## handle: The opaque handle associated with this hash computation.
##
## data: The data to add to the hash computation.
##
## Returns: True on success.
##
## .. bro:see:: md5_hmac md5_hash md5_hash_init md5_hash_update md5_hash_finish
##    sha1_hash sha1_hash_init sha1_hash_finish
##    sha256_hash sha256_hash_init sha256_hash_update sha256_hash_finish
global sha1_hash_update: function(handle: opaque of sha1 , data: string ): bool ;


## Updates the SHA256 value associated with a given index. It is required to
## call :bro:id:`sha256_hash_init` once before calling this
## function.
##
## handle: The opaque handle associated with this hash computation.
##
## data: The data to add to the hash computation.
##
## Returns: True on success.
##
## .. bro:see:: md5_hmac md5_hash md5_hash_init md5_hash_update md5_hash_finish
##    sha1_hash sha1_hash_init sha1_hash_update sha1_hash_finish
##    sha256_hash sha256_hash_init sha256_hash_finish
global sha256_hash_update: function(handle: opaque of sha256 , data: string ): bool ;


## Returns the final MD5 digest of an incremental hash computation.
##
## handle: The opaque handle associated with this hash computation.
##
## Returns: The hash value associated with the computation of *handle*.
##
## .. bro:see:: md5_hmac md5_hash md5_hash_init md5_hash_update
##    sha1_hash sha1_hash_init sha1_hash_update sha1_hash_finish
##    sha256_hash sha256_hash_init sha256_hash_update sha256_hash_finish
global md5_hash_finish: function(handle: opaque of md5 ): string ;


## Returns the final SHA1 digest of an incremental hash computation.
##
## handle: The opaque handle associated with this hash computation.
##
## Returns: The hash value associated with the computation of *handle*.
##
## .. bro:see:: md5_hmac md5_hash md5_hash_init md5_hash_update md5_hash_finish
##    sha1_hash sha1_hash_init sha1_hash_update
##    sha256_hash sha256_hash_init sha256_hash_update sha256_hash_finish
global sha1_hash_finish: function(handle: opaque of sha1 ): string ;


## Returns the final SHA256 digest of an incremental hash computation.
##
## handle: The opaque handle associated with this hash computation.
##
## Returns: The hash value associated with the computation of *handle*.
##
## .. bro:see:: md5_hmac md5_hash md5_hash_init md5_hash_update md5_hash_finish
##    sha1_hash sha1_hash_init sha1_hash_update sha1_hash_finish
##    sha256_hash sha256_hash_init sha256_hash_update
global sha256_hash_finish: function(handle: opaque of sha256 ): string ;


## Generates a random number.
##
## max: The maximum value of the random number.
##
## Returns: a random positive integer in the interval *[0, max)*.
##
## .. bro:see:: srand
##
## .. note::
##
##      This function is a wrapper about the function ``random``
##      provided by the OS.
global rand: function(max: count ): count ;


## Sets the seed for subsequent :bro:id:`rand` calls.
##
## seed: The seed for the PRNG.
##
## .. bro:see:: rand
##
## .. note::
##
##      This function is a wrapper about the function ``srandom``
##      provided by the OS.
global srand: function(seed: count ): any ;




## Send a string to syslog.
##
## s: The string to log via syslog
global syslog: function(s: string ): any ;


## Determines the MIME type of a piece of data using Bro's file magic
## signatures.
##
## data: The data to find the MIME type for.
##
## return_mime: Deprecated argument; does nothing, except emit a warning
##              when false.
##
## Returns: The MIME type of *data*, or "<unknown>" if there was an error
##          or no match.  This is the strongest signature match.
##
## .. bro:see:: file_magic
global identify_data: function(data: string , return_mime: bool &default=T): string ;


## Determines the MIME type of a piece of data using Bro's file magic
## signatures.
##
## data: The data for which to find matching MIME types.
##
## Returns: All matching signatures, in order of strength.
##
## .. bro:see:: identify_data
global file_magic: function(data: string ): mime_matches ;


## Performs an entropy test on the given data.
## See http://www.fourmilab.ch/random.
##
## data: The data to compute the entropy for.
##
## Returns: The result of the entropy test, which contains the following
##          fields.
##
##              - ``entropy``: The information density expressed as a number of
##                bits per character.
##
##              - ``chi_square``: The chi-square test value expressed as an
##                absolute number and a percentage which indicates how
##                frequently a truly random sequence would exceed the value
##                calculated, i.e., the degree to which the sequence tested is
##                suspected of being non-random.
##
##                If the percentage is greater than 99% or less than 1%, the
##                sequence is almost certainly not random. If the percentage is
##                between 99% and 95% or between 1% and 5%, the sequence is
##                suspect. Percentages between 90\% and 95\% and 5\% and 10\%
##                indicate the sequence is "almost suspect."
##
##              - ``mean``: The arithmetic mean of all the bytes. If the data
##                are close to random, it should be around 127.5.
##
##              - ``monte_carlo_pi``: Each successive sequence of six bytes is
##                used as 24-bit *x* and *y* coordinates within a square. If
##                the distance of the randomly-generated point is less than the
##                radius of a circle inscribed within the square, the six-byte
##                sequence is considered a "hit." The percentage of hits can
##                be used to calculate the value of pi. For very large streams
##                the value will approach the correct value of pi if the
##                sequence is close to random.
##
##              - ``serial_correlation``: This quantity measures the extent to
##                which each byte in the file depends upon the previous byte.
##                For random sequences this value will be close to zero.
##
## .. bro:see:: entropy_test_init entropy_test_add entropy_test_finish
global find_entropy: function(data: string ): entropy_test_result ;


## Initializes data structures for incremental entropy calculation.
##
## Returns: An opaque handle to be used in subsequent operations.
##
## .. bro:see:: find_entropy entropy_test_add entropy_test_finish
global entropy_test_init: function(): opaque of entropy ;


## Adds data to an incremental entropy calculation.
##
## handle: The opaque handle representing the entropy calculation state.
##
## data: The data to add to the entropy calculation.
##
## Returns: True on success.
##
## .. bro:see:: find_entropy entropy_test_add entropy_test_finish
global entropy_test_add: function(handle: opaque of entropy , data: string ): bool ;


## Finishes an incremental entropy calculation. Before using this function,
## one needs to obtain an opaque handle with :bro:id:`entropy_test_init` and
## add data to it via :bro:id:`entropy_test_add`.
##
## handle: The opaque handle representing the entropy calculation state.
##
## Returns: The result of the entropy test. See :bro:id:`find_entropy` for a
##          description of the individual components.
##
## .. bro:see:: find_entropy entropy_test_init entropy_test_add
global entropy_test_finish: function(handle: opaque of entropy ): entropy_test_result ;


## Creates an identifier that is unique with high probability.
##
## prefix: A custom string prepended to the result.
##
## Returns: A string identifier that is unique.
##
## .. bro:see:: unique_id_from
global unique_id: function(prefix: string ) : string ;


## Creates an identifier that is unique with high probability.
##
## pool: A seed for determinism.
##
## prefix: A custom string prepended to the result.
##
## Returns: A string identifier that is unique.
##
## .. bro:see:: unique_id
global unique_id_from: function(pool: int , prefix: string ) : string ;








## Removes all elements from a set or table.
##
## v: The set or table
global clear_table: function(v: any ): any ;


## Gets all subnets that contain a given subnet from a set/table[subnet].
##
## search: the subnet to search for.
##
## t: the set[subnet] or table[subnet].
##
## Returns: All the keys of the set or table that cover the subnet searched for.
global matching_subnets: function(search: subnet , t: any ): subnet_vec ;


## For a set[subnet]/table[subnet], create a new table that contains all entries
## that contain a given subnet.
##
## search: the subnet to search for.
##
## t: the set[subnet] or table[subnet].
##
## Returns: A new table that contains all the entries that cover the subnet searched for.
global filter_subnet_table: function(search: subnet , t: any ): any ;


## Checks if a specific subnet is a member of a set/table[subnet].
## In contrast to the ``in`` operator, this performs an exact match, not
## a longest prefix match.
##
## search: the subnet to search for.
##
## t: the set[subnet] or table[subnet].
##
## Returns: True if the exact subnet is a member, false otherwise.
global check_subnet: function(search: subnet , t: any ): bool ;


## Checks whether two objects reference the same internal object. This function
## uses equality comparison of C++ raw pointer values to determine if the two
## objects are the same.
##
## o1: The first object.
##
## o2: The second object.
##
## Returns: True if *o1* and *o2* are equal.
global same_object: function(o1: any , o2: any ): bool ;


## Returns the number of bytes that a value occupies in memory.
##
## v: The value
##
## Returns: The number of bytes that *v* occupies.
global val_size: function(v: any ): count ;


## Resizes a vector.
##
## aggr: The vector instance.
##
## newsize: The new size of *aggr*.
##
## Returns: The old size of *aggr*, or 0 if *aggr* is not a :bro:type:`vector`.
global resize: function(aggr: any , newsize: count ) : count ;


## Tests whether a boolean vector (``vector of bool``) has *any* true
## element.
##
## v: The boolean vector instance.
##
## Returns: True if any element in *v* is true.
##
## .. bro:see:: all_set
global any_set: function(v: any ) : bool ;


## Tests whether *all* elements of a boolean vector (``vector of bool``) are
## true.
##
## v: The boolean vector instance.
##
## Returns: True iff all elements in *v* are true.
##
## .. bro:see:: any_set
##
## .. note::
##
##      Missing elements count as false.
global all_set: function(v: any ) : bool ;




## Sorts a vector in place. The second argument is a comparison function that
## takes two arguments: if the vector type is ``vector of T``, then the
## comparison function must be ``function(a: T, b: T): int``, which returns
## a value less than zero if ``a < b`` for some type-specific notion of the
## less-than operator.  The comparison function is optional if the type
## is an integral type (int, count, etc.).
##
## v: The vector instance to sort.
##
## Returns: The vector, sorted from minimum to maximum value. If the vector
##          could not be sorted, then the original vector is returned instead.
##
## .. bro:see:: order
global sort: function(va_args: any) : any ;


## Returns the order of the elements in a vector according to some
## comparison function. See :bro:id:`sort` for details about the comparison
## function.
##
## v: The vector whose order to compute.
##
## Returns: A ``vector of count`` with the indices of the ordered elements.
##          For example, the elements of *v* in order are (assuming ``o``
##          is the vector returned by ``order``):  v[o[0]], v[o[1]], etc.
##
## .. bro:see:: sort
global order: function(va_args: any) : index_vec ;








## Returns the concatenation of the string representation of its arguments. The
## arguments can be of any type. For example, ``cat("foo", 3, T)`` returns
## ``"foo3T"``.
##
## Returns: A string concatentation of all arguments.
global cat: function(va_args: any): string ;


## Concatenates all arguments, with a separator placed between each one. This
## function is similar to :bro:id:`cat`, but places a separator between each
## given argument. If any of the variable arguments is an empty string it is
## replaced by a given default string instead.
##
## sep: The separator to place between each argument.
##
## def: The default string to use when an argument is the empty string.
##
## Returns: A concatenation of all arguments with *sep* between each one and
##          empty strings replaced with *def*.
##
## .. bro:see:: cat string_cat cat_string_array cat_string_array_n
global cat_sep: function(va_args: any): string ;


## Produces a formatted string à la ``printf``. The first argument is the
## *format string* and specifies how subsequent arguments are converted for
## output. It is composed of zero or more directives: ordinary characters (not
## ``%``), which are copied unchanged to the output, and conversion
## specifications, each of which fetches zero or more subsequent arguments.
## Conversion specifications begin with ``%`` and the arguments must properly
## correspond to the specifier. After the ``%``, the following characters
## may appear in sequence:
##
##    - ``%``: Literal ``%``
##
##    - ``-``: Left-align field
##
##    - ``[0-9]+``: The field width (< 128)
##
##    - ``.``: Precision of floating point specifiers ``[efg]`` (< 128)
##
##    - ``[DTdxsefg]``: Format specifier
##
##        - ``[DT]``: ISO timestamp with microsecond precision
##
##        - ``d``: Signed/Unsigned integer (using C-style ``%lld``/``%llu``
##                 for ``int``/``count``)
##
##        - ``x``: Unsigned hexadecimal (using C-style ``%llx``);
##                 addresses/ports are converted to host-byte order
##
##        - ``s``: String (byte values less than 32 or greater than 126
##                 will be escaped)
##
##        - ``[efg]``: Double
##
## Returns: Returns the formatted string. Given no arguments, :bro:id:`fmt`
##          returns an empty string. Given no format string or the wrong
##          number of additional arguments for the given format specifier,
##          :bro:id:`fmt` generates a run-time error.
##
## .. bro:see:: cat cat_sep string_cat cat_string_array cat_string_array_n
global fmt: function(va_args: any): string ;








## Computes the greatest integer less than the given :bro:type:`double` value.
## For example, ``floor(3.14)`` returns ``3.0``, and ``floor(-3.14)``
## returns ``-4.0``.
##
## d: The :bro:type:`double` to manipulate.
##
## Returns: The next lowest integer of *d* as :bro:type:`double`.
##
## .. bro:see:: sqrt exp ln log10
global floor: function(d: double ): double ;


## Computes the square root of a :bro:type:`double`.
##
## x: The number to compute the square root of.
##
## Returns: The square root of *x*.
##
## .. bro:see:: floor exp ln log10
global sqrt: function(x: double ): double ;


## Computes the exponential function.
##
## d: The argument to the exponential function.
##
## Returns: *e* to the power of *d*.
##
## .. bro:see:: floor sqrt ln log10
global exp: function(d: double ): double ;


## Computes the natural logarithm of a number.
##
## d: The argument to the logarithm.
##
## Returns: The natural logarithm of *d*.
##
## .. bro:see:: exp floor sqrt log10
global ln: function(d: double ): double ;


## Computes the common logarithm of a number.
##
## d: The argument to the logarithm.
##
## Returns: The common logarithm of *d*.
##
## .. bro:see:: exp floor sqrt ln
global log10: function(d: double ): double ;








## Determines whether a connection has been received externally. For example,
## Broccoli or the Time Machine can send packets to Bro via a mechanism that is
## one step lower than sending events. This function checks whether the packets
## of a connection stem from one of these external *packet sources*.
##
## c: The connection to test.
##
## Returns: True if *c* has been received externally.
global is_external_connection: function(c: connection ) : bool ;


## Returns the ID of the analyzer which raised the current event.
##
## Returns: The ID of the analyzer which raised the current event, or 0 if
##          none.
global current_analyzer: function() : count ;


## Returns Bro's process ID.
##
## Returns: Bro's process ID.
global getpid: function() : count ;




## Returns the Bro version string.
##
## Returns: Bro's version, e.g., 2.0-beta-47-debug.
global bro_version: function(): string ;


## Converts a record type name to a vector of strings, where each element is
## the name of a record field. Nested records are flattened.
##
## rt: The name of the record type.
##
## Returns: A string vector with the field names of *rt*.
global record_type_to_vector: function(rt: string ): string_vec ;


## Returns the type name of an arbitrary Bro variable.
##
## t: An arbitrary object.
##
## Returns: The type name of *t*.
global type_name: function(t: any ): string ;


## Checks whether Bro reads traffic from one or more network interfaces (as
## opposed to from a network trace in a file). Note that this function returns
## true even after Bro has stopped reading network traffic, for example due to
## receiving a termination signal.
##
## Returns: True if reading traffic from a network interface.
##
## .. bro:see:: reading_traces
global reading_live_traffic: function(): bool ;


## Checks whether Bro reads traffic from a trace file (as opposed to from a
## network interface).
##
## Returns: True if reading traffic from a network trace.
##
## .. bro:see:: reading_live_traffic
global reading_traces: function(): bool ;



## Generates a table of the size of all global variables. The table index is
## the variable name and the value is the variable size in bytes.
##
## Returns: A table that maps variable names to their sizes.
##
## .. bro:see:: global_ids
global global_sizes: function(): var_sizes ;


## Generates a table with information about all global identifiers. The table
## value is a record containing the type name of the identifier, whether it is
## exported, a constant, an enum constant, redefinable, and its value (if it
## has one).
##
## Returns: A table that maps identifier names to information about them.
##
## .. bro:see:: global_sizes
global global_ids: function(): id_table ;


## Returns the value of a global identifier.
##
## id: The global identifier.
##
## Returns: The value of *id*. If *id* does not describe a valid identifier,
##          the string ``"<unknown id>"`` or ``"<no ID value>"`` is returned.
global lookup_ID: function(id: string ) : any ;


## Generates metadata about a record's fields. The returned information
## includes the field name, whether it is logged, its value (if it has one),
## and its default value (if specified).
##
## rec: The record to inspect.
##
## Returns: A table that describes the fields of a record.
global record_fields: function(rec: any ): record_field_table ;


## Enables detailed collection of profiling statistics. Statistics include
## CPU/memory usage, connections, TCP states/reassembler, DNS lookups,
## timers, and script-level state. The script variable :bro:id:`profiling_file`
## holds the name of the file.
##
## .. bro:see:: get_conn_stats
##              get_dns_stats
##              get_event_stats
##              get_file_analysis_stats
##              get_gap_stats
##              get_matcher_stats
##              get_net_stats
##              get_proc_stats
##              get_reassembler_stats
##              get_thread_stats
##              get_timer_stats
global do_profiling: function() : any ;


## Checks whether a given IP address belongs to a local interface.
##
## ip: The IP address to check.
##
## Returns: True if *ip* belongs to a local interface.
global is_local_interface: function(ip: addr ) : bool ;


## Write rule matcher statistics (DFA states, transitions, memory usage, cache
## hits/misses) to a file.
##
## f: The file to write to.
##
## Returns: True (unconditionally).
##
## .. bro:see:: get_matcher_stats
global dump_rule_stats: function(f: file ): bool ;


## Checks if Bro is terminating.
##
## Returns: True if Bro is in the process of shutting down.
##
## .. bro:see:: terminate
global bro_is_terminating: function(): bool ;


## Returns the hostname of the machine Bro runs on.
##
## Returns: The hostname of the machine Bro runs on.
global gethostname: function() : string ;


## Returns whether an address is IPv4 or not.
##
## a: the address to check.
##
## Returns: true if *a* is an IPv4 address, else false.
global is_v4_addr: function(a: addr ): bool ;


## Returns whether an address is IPv6 or not.
##
## a: the address to check.
##
## Returns: true if *a* is an IPv6 address, else false.
global is_v6_addr: function(a: addr ): bool ;


## Returns whether a subnet specification is IPv4 or not.
##
## s: the subnet to check.
##
## Returns: true if *s* is an IPv4 subnet, else false.
global is_v4_subnet: function(s: subnet ): bool ;


## Returns whether a subnet specification is IPv6 or not.
##
## s: the subnet to check.
##
## Returns: true if *s* is an IPv6 subnet, else false.
global is_v6_subnet: function(s: subnet ): bool ;









## Converts the *data* field of :bro:type:`ip6_routing` records that have
## *rtype* of 0 into a vector of addresses.
##
## s: The *data* field of an :bro:type:`ip6_routing` record that has
##    an *rtype* of 0.
##
## Returns: The vector of addresses contained in the routing header data.
global routing0_data_to_addrs: function(s: string ): addr_vec ;


## Converts an :bro:type:`addr` to an :bro:type:`index_vec`.
##
## a: The address to convert into a vector of counts.
##
## Returns: A vector containing the host-order address representation,
##          four elements in size for IPv6 addresses, or one element for IPv4.
##
## .. bro:see:: counts_to_addr
global addr_to_counts: function(a: addr ): index_vec ;


## Converts an :bro:type:`index_vec` to an :bro:type:`addr`.
##
## v: The vector containing host-order IP address representation,
##    one element for IPv4 addresses, four elements for IPv6 addresses.
##
## Returns: An IP address.
##
## .. bro:see:: addr_to_counts
global counts_to_addr: function(v: index_vec ): addr ;


## Converts an :bro:type:`enum` to an :bro:type:`int`.
##
## e: The :bro:type:`enum` to convert.
##
## Returns: The :bro:type:`int` value that corresponds to the :bro:type:`enum`.
global enum_to_int: function(e: any ): int ;


## Converts a :bro:type:`string` to an :bro:type:`int`.
##
## str: The :bro:type:`string` to convert.
##
## Returns: The :bro:type:`string` *str* as :bro:type:`int`.
##
## .. bro:see:: to_addr to_port to_subnet
global to_int: function(str: string ): int ;



## Converts a (positive) :bro:type:`int` to a :bro:type:`count`.
##
## n: The :bro:type:`int` to convert.
##
## Returns: The :bro:type:`int` *n* as unsigned integer, or 0 if *n* < 0.
global int_to_count: function(n: int ): count ;


## Converts a :bro:type:`double` to a :bro:type:`count`.
##
## d: The :bro:type:`double` to convert.
##
## Returns: The :bro:type:`double` *d* as unsigned integer, or 0 if *d* < 0.0.
##
## .. bro:see:: double_to_time
global double_to_count: function(d: double ): count ;


## Converts a :bro:type:`string` to a :bro:type:`count`.
##
## str: The :bro:type:`string` to convert.
##
## Returns: The :bro:type:`string` *str* as unsigned integer, or 0 if *str* has
##          an invalid format.
##
## .. bro:see:: to_addr to_int to_port to_subnet
global to_count: function(str: string ): count ;


## Converts an :bro:type:`interval` to a :bro:type:`double`.
##
## i: The :bro:type:`interval` to convert.
##
## Returns: The :bro:type:`interval` *i* as :bro:type:`double`.
##
## .. bro:see:: double_to_interval
global interval_to_double: function(i: interval ): double ;


## Converts a :bro:type:`time` value to a :bro:type:`double`.
##
## t: The :bro:type:`time` to convert.
##
## Returns: The :bro:type:`time` value *t* as :bro:type:`double`.
##
## .. bro:see:: double_to_time
global time_to_double: function(t: time ): double ;


## Converts a :bro:type:`double` value to a :bro:type:`time`.
##
## d: The :bro:type:`double` to convert.
##
## Returns: The :bro:type:`double` value *d* as :bro:type:`time`.
##
## .. bro:see:: time_to_double double_to_count
global double_to_time: function(d: double ): time ;


## Converts a :bro:type:`double` to an :bro:type:`interval`.
##
## d: The :bro:type:`double` to convert.
##
## Returns: The :bro:type:`double` *d* as :bro:type:`interval`.
##
## .. bro:see:: interval_to_double
global double_to_interval: function(d: double ): interval ;


## Converts a :bro:type:`port` to a :bro:type:`count`.
##
## p: The :bro:type:`port` to convert.
##
## Returns: The :bro:type:`port` *p* as :bro:type:`count`.
##
## .. bro:see:: count_to_port
global port_to_count: function(p: port ): count ;


## Converts a :bro:type:`count` and ``transport_proto`` to a :bro:type:`port`.
##
## num: The :bro:type:`port` number.
##
## proto: The transport protocol.
##
## Returns: The :bro:type:`count` *num* as :bro:type:`port`.
##
## .. bro:see:: port_to_count
global count_to_port: function(num: count , proto: transport_proto ): port ;


## Converts a :bro:type:`string` to an :bro:type:`addr`.
##
## ip: The :bro:type:`string` to convert.
##
## Returns: The :bro:type:`string` *ip* as :bro:type:`addr`, or the unspecified
##          address ``::`` if the input string does not parse correctly.
##
## .. bro:see:: to_count to_int to_port count_to_v4_addr raw_bytes_to_v4_addr
##    to_subnet
global to_addr: function(ip: string ): addr ;


## Converts a :bro:type:`string` to a :bro:type:`subnet`.
##
## sn: The subnet to convert.
##
## Returns: The *sn* string as a :bro:type:`subnet`, or the unspecified subnet
##          ``::/0`` if the input string does not parse correctly.
##
## .. bro:see:: to_count to_int to_port count_to_v4_addr raw_bytes_to_v4_addr
##    to_addr
global to_subnet: function(sn: string ): subnet ;


## Converts a :bro:type:`addr` to a :bro:type:`subnet`.
##
## a: The address to convert.
##
## Returns: The address as a :bro:type:`subnet`.
##
## .. bro:see:: to_subnet
global addr_to_subnet: function(a: addr ): subnet ;


## Converts a :bro:type:`subnet` to an :bro:type:`addr` by
## extracting the prefix.
##
## sn: The subnet to convert.
##
## Returns: The subnet as an :bro:type:`addr`.
##
## .. bro:see:: to_subnet
global subnet_to_addr: function(sn: subnet ): addr ;


## Returns the width of a :bro:type:`subnet`.
##
## sn: The subnet.
##
## Returns: The width of the subnet.
##
## .. bro:see:: to_subnet
global subnet_width: function(sn: subnet ): count ;


## Converts a :bro:type:`string` to a :bro:type:`double`.
##
## str: The :bro:type:`string` to convert.
##
## Returns: The :bro:type:`string` *str* as double, or 0 if *str* has
##          an invalid format.
##
global to_double: function(str: string ): double ;


## Converts a :bro:type:`count` to an :bro:type:`addr`.
##
## ip: The :bro:type:`count` to convert.
##
## Returns: The :bro:type:`count` *ip* as :bro:type:`addr`.
##
## .. bro:see:: raw_bytes_to_v4_addr to_addr to_subnet
global count_to_v4_addr: function(ip: count ): addr ;


## Converts a :bro:type:`string` of bytes into an IPv4 address. In particular,
## this function interprets the first 4 bytes of the string as an IPv4 address
## in network order.
##
## b: The raw bytes (:bro:type:`string`) to convert.
##
## Returns: The byte :bro:type:`string` *b* as :bro:type:`addr`.
##
## .. bro:see:: raw_bytes_to_v4_addr to_addr to_subnet
global raw_bytes_to_v4_addr: function(b: string ): addr ;


## Converts a :bro:type:`string` to a :bro:type:`port`.
##
## s: The :bro:type:`string` to convert.
##
## Returns: A :bro:type:`port` converted from *s*.
##
## .. bro:see:: to_addr to_count to_int to_subnet
global to_port: function(s: string ): port ;


## Converts a string of bytes (in network byte order) to a :bro:type:`double`.
##
## s: A string of bytes containing the binary representation of a double value.
##
## Returns: The double value contained in *s*, or 0 if the conversion
##          failed.
##
global bytestring_to_double: function(s: string ): double ;


## Converts a string of bytes to a :bro:type:`count`.
##
## s: A string of bytes containing the binary representation of the value.
##
## is_le: If true, *s* is assumed to be in little endian format, else it's big endian.
##
## Returns: The value contained in *s*, or 0 if the conversion failed.
##
global bytestring_to_count: function(s: string , is_le: bool &default=F): count ;


## Converts a reverse pointer name to an address. For example,
## ``1.0.168.192.in-addr.arpa`` to ``192.168.0.1``.
##
## s: The string with the reverse pointer name.
##
## Returns: The IP address corresponding to *s*.
##
## .. bro:see:: addr_to_ptr_name to_addr
global ptr_name_to_addr: function(s: string ): addr ;


## Converts an IP address to a reverse pointer name. For example,
## ``192.168.0.1`` to ``1.0.168.192.in-addr.arpa``.
##
## a: The IP address to convert to a reverse pointer name.
##
## Returns: The reverse pointer representation of *a*.
##
## .. bro:see:: ptr_name_to_addr to_addr
global addr_to_ptr_name: function(a: addr ): string ;


## Converts a string of bytes into its hexadecimal representation.
## For example, ``"04"`` would be converted to ``"3034"``.
##
## bytestring: The string of bytes.
##
## Returns: The hexadecimal representation of *bytestring*.
##
## .. bro:see:: hexdump hexstr_to_bytestring
global bytestring_to_hexstr: function(bytestring: string ): string ;


## Converts a hex-string into its binary representation.
## For example, ``"3034"`` would be converted to ``"04"``.
##
## The input string is assumed to contain an even number of hexadecimal digits
## (0-9, a-f, or A-F), otherwise behavior is undefined.
##
## hexstr: The hexadecimal string representation.
##
## Returns: The binary representation of *hexstr*.
##
## .. bro:see:: hexdump bytestring_to_hexstr
global hexstr_to_bytestring: function(hexstr: string ): string ;


## Encodes a Base64-encoded string.
##
## s: The string to encode.
##
## a: An optional custom alphabet. The empty string indicates the default
##    alphabet. If given, the string must consist of 64 unique characters.
##
## Returns: The encoded version of *s*.
##
## .. bro:see:: decode_base64
global encode_base64: function(s: string , a: string &default=""): string ;



## Encodes a Base64-encoded string with a custom alphabet.
##
## s: The string to encode.
##
## a: The custom alphabet. The string must consist of 64 unique
##    characters. The empty string indicates the default alphabet.
##
## Returns: The encoded version of *s*.
##
## .. bro:see:: encode_base64
global encode_base64_custom: function(s: string , a: string ): string &deprecated;


## Decodes a Base64-encoded string.
##
## s: The Base64-encoded string.
##
## a: An optional custom alphabet. The empty string indicates the default
##    alphabet. If given, the string must consist of 64 unique characters.
##
## Returns: The decoded version of *s*.
##
## .. bro:see:: decode_base64_conn encode_base64
global decode_base64: function(s: string , a: string &default=""): string ;


## Decodes a Base64-encoded string that was derived from processing a connection.
## If an error is encountered decoding the string, that will be logged to
## ``weird.log`` with the associated connection.
##
## cid: The identifier of the connection that the encoding originates from.
##
## s: The Base64-encoded string.
##
## a: An optional custom alphabet. The empty string indicates the default
##    alphabet. If given, the string must consist of 64 unique characters.
##
## Returns: The decoded version of *s*.
##
## .. bro:see:: decode_base64
global decode_base64_conn: function(cid: conn_id , s: string , a: string &default=""): string ;


## Decodes a Base64-encoded string with a custom alphabet.
##
## s: The Base64-encoded string.
##
## a: The custom alphabet. The string must consist of 64 unique characters.
##    The empty string indicates the default alphabet.
##
## Returns: The decoded version of *s*.
##
## .. bro:see:: decode_base64 decode_base64_conn
global decode_base64_custom: function(s: string , a: string ): string &deprecated;




## Converts a bytes representation of a UUID into its string form. For example,
## given a string of 16 bytes, it produces an output string in this format:
## ``550e8400-e29b-41d4-a716-446655440000``.
## See `<http://en.wikipedia.org/wiki/Universally_unique_identifier>`_.
##
## uuid: The 16 bytes of the UUID.
##
## Returns: The string representation of *uuid*.
global uuid_to_string: function(uuid: string ): string ;


## Merges and compiles two regular expressions at initialization time.
##
## p1: The first pattern.
##
## p2: The second pattern.
##
## Returns: The compiled pattern of the concatenation of *p1* and *p2*.
##
## .. bro:see:: convert_for_pattern string_to_pattern
##
## .. note::
##
##      This function must be called at Bro startup time, e.g., in the event
##      :bro:id:`bro_init`.
global merge_pattern: function(p1: pattern , p2: pattern ): pattern ;




## Escapes a string so that it becomes a valid :bro:type:`pattern` and can be
## used with the :bro:id:`string_to_pattern`. Any character from the set
## ``^$-:"\/|*+?.(){}[]`` is prefixed with a ``\``.
##
## s: The string to escape.
##
## Returns: An escaped version of *s* that has the structure of a valid
##          :bro:type:`pattern`.
##
## .. bro:see:: merge_pattern string_to_pattern
##
global convert_for_pattern: function(s: string ): string ;


## Converts a :bro:type:`string` into a :bro:type:`pattern`.
##
## s: The string to convert.
##
## convert: If true, *s* is first passed through the function
##          :bro:id:`convert_for_pattern` to escape special characters of
##          patterns.
##
## Returns: *s* as :bro:type:`pattern`.
##
## .. bro:see:: convert_for_pattern merge_pattern
##
## .. note::
##
##      This function must be called at Bro startup time, e.g., in the event
##      :bro:id:`bro_init`.
global string_to_pattern: function(s: string , convert: bool ): pattern ;


## Formats a given time value according to a format string.
##
## fmt: The format string. See ``man strftime`` for the syntax.
##
## d: The time value.
##
## Returns: The time *d* formatted according to *fmt*.
global strftime: function(fmt: string , d: time ) : string ;



## Parse a textual representation of a date/time value into a ``time`` type value.
##
## fmt: The format string used to parse the following *d* argument. See ``man strftime``
##      for the syntax.
##
## d: The string representing the time.
##
## Returns: The time value calculated from parsing *d* with *fmt*.
global strptime: function(fmt: string , d: string ) : time ;









## Masks an address down to the number of given upper bits. For example,
## ``mask_addr(1.2.3.4, 18)`` returns ``1.2.0.0``.
##
## a: The address to mask.
##
## top_bits_to_keep: The number of top bits to keep in *a*; must be greater
##                   than 0 and less than 33 for IPv4, or 129 for IPv6.
##
## Returns: The address *a* masked down to *top_bits_to_keep* bits.
##
## .. bro:see:: remask_addr
global mask_addr: function(a: addr , top_bits_to_keep: count ): subnet ;


## Takes some top bits (such as a subnet address) from one address and the other
## bits (intra-subnet part) from a second address and merges them to get a new
## address. This is useful for anonymizing at subnet level while preserving
## serial scans.
##
## a1: The address to mask with *top_bits_from_a1*.
##
## a2: The address to take the remaining bits from.
##
## top_bits_from_a1: The number of top bits to keep in *a1*; must be greater
##                   than 0 and less than 129.  This value is always interpreted
##                   relative to the IPv6 bit width (v4-mapped addresses start
##                   at bit number 96).
##
## Returns: The address *a* masked down to *top_bits_to_keep* bits.
##
## .. bro:see:: mask_addr
global remask_addr: function(a1: addr , a2: addr , top_bits_from_a1: count ): addr ;


## Checks whether a given :bro:type:`port` has TCP as transport protocol.
##
## p: The :bro:type:`port` to check.
##
## Returns: True iff *p* is a TCP port.
##
## .. bro:see:: is_udp_port is_icmp_port
global is_tcp_port: function(p: port ): bool ;


## Checks whether a given :bro:type:`port` has UDP as transport protocol.
##
## p: The :bro:type:`port` to check.
##
## Returns: True iff *p* is a UDP port.
##
## .. bro:see:: is_icmp_port is_tcp_port
global is_udp_port: function(p: port ): bool ;


## Checks whether a given :bro:type:`port` has ICMP as transport protocol.
##
## p: The :bro:type:`port` to check.
##
## Returns: True iff *p* is an ICMP port.
##
## .. bro:see:: is_tcp_port is_udp_port
global is_icmp_port: function(p: port ): bool ;




## Extracts the transport protocol from a connection.
##
## cid: The connection identifier.
##
## Returns: The transport protocol of the connection identified by *cid*.
##
## .. bro:see:: get_port_transport_proto
##              get_orig_seq get_resp_seq
global get_conn_transport_proto: function(cid: conn_id ): transport_proto ;


## Extracts the transport protocol from a :bro:type:`port`.
##
## p: The port.
##
## Returns: The transport protocol of the port *p*.
##
## .. bro:see:: get_conn_transport_proto
##              get_orig_seq get_resp_seq
global get_port_transport_proto: function(p: port ): transport_proto ;


## Checks whether a connection is (still) active.
##
## c: The connection id to check.
##
## Returns: True if the connection identified by *c* exists.
##
## .. bro:see:: lookup_connection
global connection_exists: function(c: conn_id ): bool ;


## Returns the :bro:type:`connection` record for a given connection identifier.
##
## cid: The connection ID.
##
## Returns: The :bro:type:`connection` record for *cid*. If *cid* does not point
##          to an existing connection, the function generates a run-time error
##          and returns a dummy value.
##
## .. bro:see:: connection_exists
global lookup_connection: function(cid: conn_id ): connection ;




## Writes the current packet to a file.
##
## file_name: The name of the file to write the packet to.
##
## Returns: True on success.
##
## .. bro:see:: dump_packet get_current_packet send_current_packet
global dump_current_packet: function(file_name: string ) : bool ;


## Returns the currently processed PCAP packet.
##
## Returns: The currently processed packet, which is a record
##          containing the timestamp, ``snaplen``, and packet data.
##
## .. bro:see:: dump_current_packet dump_packet send_current_packet
global get_current_packet: function() : pcap_packet ;


## Function to get the raw headers of the currently processed packet.
##
## Returns: The :bro:type:`raw_pkt_hdr` record containing the Layer 2, 3 and
##          4 headers of the currently processed packet.
##
## .. bro:see:: raw_pkt_hdr get_current_packet
global get_current_packet_header: function() : raw_pkt_hdr ;


## Writes a given packet to a file.
##
## pkt: The PCAP packet.
##
## file_name: The name of the file to write *pkt* to.
##
## Returns: True on success
##
## .. bro:see:: get_current_packet dump_current_packet send_current_packet
global dump_packet: function(pkt: pcap_packet , file_name: string ) : bool ;




## Issues an asynchronous reverse DNS lookup and delays the function result.
## This function can therefore only be called inside a ``when`` condition,
## e.g., ``when ( local host = lookup_addr(10.0.0.1) ) { f(host); }``.
##
## host: The IP address to lookup.
##
## Returns: The DNS name of *host*.
##
## .. bro:see:: lookup_hostname
global lookup_addr: function(host: addr ) : string ;


## Issues an asynchronous TEXT DNS lookup and delays the function result.
## This function can therefore only be called inside a ``when`` condition,
## e.g., ``when ( local h = lookup_hostname_txt("www.bro.org") ) { f(h); }``.
##
## host: The hostname to lookup.
##
## Returns: The DNS TXT record associated with *host*.
##
## .. bro:see:: lookup_hostname
global lookup_hostname_txt: function(host: string ) : string ;


## Issues an asynchronous DNS lookup and delays the function result.
## This function can therefore only be called inside a ``when`` condition,
## e.g., ``when ( local h = lookup_hostname("www.bro.org") ) { f(h); }``.
##
## host: The hostname to lookup.
##
## Returns: A set of DNS A and AAAA records associated with *host*.
##
## .. bro:see:: lookup_addr
global lookup_hostname: function(host: string ) : addr_set ;




## Performs a geo-lookup of an IP address.
## Requires Bro to be built with ``libgeoip``.
##
## a: The IP address to lookup.
##
## Returns: A record with country, region, city, latitude, and longitude.
##
## .. bro:see:: lookup_asn
global lookup_location: function(a: addr ) : geo_location ;


## Performs an AS lookup of an IP address.
## Requires Bro to be built with ``libgeoip``.
##
## a: The IP address to lookup.
##
## Returns: The number of the AS that contains *a*.
##
## .. bro:see:: lookup_location
global lookup_asn: function(a: addr ) : count ;


## Calculates distance between two geographic locations using the haversine
## formula.  Latitudes and longitudes must be given in degrees, where southern
## hemispere latitudes are negative and western hemisphere longitudes are
## negative.
##
## lat1: Latitude (in degrees) of location 1.
##
## long1: Longitude (in degrees) of location 1.
##
## lat2: Latitude (in degrees) of location 2.
##
## long2: Longitude (in degrees) of location 2.
##
## Returns: Distance in miles.
##
## .. bro:see:: haversine_distance_ip
global haversine_distance: function(lat1: double , long1: double , lat2: double , long2: double ): double ;


## Converts UNIX file permissions given by a mode to an ASCII string.
##
## mode: The permissions (an octal number like 0644 converted to decimal).
##
## Returns: A string representation of *mode* in the format
##          ``rw[xsS]rw[xsS]rw[xtT]``.
global file_mode: function(mode: count ): string ;










## Disables the analyzer which raised the current event (if the analyzer
## belongs to the given connection).
##
## cid: The connection identifier.
##
## aid: The analyzer ID.
##
## Returns: True if the connection identified by *cid* exists and has analyzer
##          *aid*.
##
## .. bro:see:: Analyzer::schedule_analyzer Analyzer::name
global disable_analyzer: function(cid: conn_id , aid: count , err_if_no_conn: bool &default=T) : bool ;


## Informs Bro that it should skip any further processing of the contents of
## a given connection. In particular, Bro will refrain from reassembling the
## TCP byte stream and from generating events relating to any analyzers that
## have been processing the connection.
##
## cid: The connection ID.
##
## Returns: False if *cid* does not point to an active connection, and true
##          otherwise.
##
## .. note::
##
##     Bro will still generate connection-oriented events such as
##     :bro:id:`connection_finished`.
global skip_further_processing: function(cid: conn_id ): bool ;


## Controls whether packet contents belonging to a connection should be
## recorded (when ``-w`` option is provided on the command line).
##
## cid: The connection identifier.
##
## do_record: True to enable packet contents, and false to disable for the
##            connection identified by *cid*.
##
## Returns: False if *cid* does not point to an active connection, and true
##          otherwise.
##
## .. bro:see:: skip_further_processing
##
## .. note::
##
##     This is independent of whether Bro processes the packets of this
##     connection, which is controlled separately by
##     :bro:id:`skip_further_processing`.
##
## .. bro:see:: get_contents_file set_contents_file
global set_record_packets: function(cid: conn_id , do_record: bool ): bool ;


## Sets an individual inactivity timeout for a connection and thus
## overrides the global inactivity timeout.
##
## cid: The connection ID.
##
## t: The new inactivity timeout for the connection identified by *cid*.
##
## Returns: The previous timeout interval.
global set_inactivity_timeout: function(cid: conn_id , t: interval ): interval ;








## Opens a file for writing. If a file with the same name already exists, this
## function overwrites it (as opposed to :bro:id:`open_for_append`).
##
## f: The path to the file.
##
## Returns: A :bro:type:`file` handle for subsequent operations.
##
## .. bro:see:: active_file open_for_append close write_file
##              get_file_name set_buf flush_all mkdir enable_raw_output
global open: function(f: string ): file ;


## Opens a file for writing or appending. If a file with the same name already
## exists, this function appends to it (as opposed to :bro:id:`open`).
##
## f: The path to the file.
##
## Returns: A :bro:type:`file` handle for subsequent operations.
##
## .. bro:see:: active_file open close write_file
##              get_file_name set_buf flush_all mkdir enable_raw_output
global open_for_append: function(f: string ): file ;


## Closes an open file and flushes any buffered content.
##
## f: A :bro:type:`file` handle to an open file.
##
## Returns: True on success.
##
## .. bro:see:: active_file open open_for_append write_file
##              get_file_name set_buf flush_all mkdir enable_raw_output
global close: function(f: file ): bool ;


## Writes data to an open file.
##
## f: A :bro:type:`file` handle to an open file.
##
## data: The data to write to *f*.
##
## Returns: True on success.
##
## .. bro:see:: active_file open open_for_append close
##              get_file_name set_buf flush_all mkdir enable_raw_output
global write_file: function(f: file , data: string ): bool ;


## Alters the buffering behavior of a file.
##
## f: A :bro:type:`file` handle to an open file.
##
## buffered: When true, *f* is fully buffered, i.e., bytes are saved in a
##           buffer until the block size has been reached. When
##           false, *f* is line buffered, i.e., bytes are saved up until a
##           newline occurs.
##
## .. bro:see:: active_file open open_for_append close
##              get_file_name write_file flush_all mkdir enable_raw_output
global set_buf: function(f: file , buffered: bool ): any ;


## Flushes all open files to disk.
##
## Returns: True on success.
##
## .. bro:see:: active_file open open_for_append close
##              get_file_name write_file set_buf mkdir enable_raw_output
global flush_all: function(): bool ;


## Creates a new directory.
##
## f: The directory name.
##
## Returns: Returns true if the operation succeeds or if *f* already exists,
##          and false if the file creation fails.
##
## .. bro:see:: active_file open_for_append close write_file
##              get_file_name set_buf flush_all enable_raw_output
global mkdir: function(f: string ): bool ;


## Checks whether a given file is open.
##
## f: The file to check.
##
## Returns: True if *f* is an open :bro:type:`file`.
##
## .. todo:: Rename to ``is_open``.
global active_file: function(f: file ): bool ;


## Gets the filename associated with a file handle.
##
## f: The file handle to inquire the name for.
##
## Returns: The filename associated with *f*.
##
## .. bro:see:: open
global get_file_name: function(f: file ): string ;


## Rotates a file.
##
## f: An open file handle.
##
## Returns: Rotation statistics which include the original file name, the name
##          after the rotation, and the time when *f* was opened/closed.
##
## .. bro:see:: rotate_file_by_name calc_next_rotate
global rotate_file: function(f: file ): rotate_info ;


## Rotates a file identified by its name.
##
## f: The name of the file to rotate
##
## Returns: Rotation statistics which include the original file name, the name
##          after the rotation, and the time when *f* was opened/closed.
##
## .. bro:see:: rotate_file calc_next_rotate
global rotate_file_by_name: function(f: string ): rotate_info ;


## Calculates the duration until the next time a file is to be rotated, based
## on a given rotate interval.
##
## i: The rotate interval to base the calculation on.
##
## Returns: The duration until the next file rotation time.
##
## .. bro:see:: rotate_file rotate_file_by_name
global calc_next_rotate: function(i: interval ) : interval ;


## Returns the size of a given file.
##
## f: The name of the file whose size to lookup.
##
## Returns: The size of *f* in bytes.
global file_size: function(f: string ) : double ;


## Disables sending :bro:id:`print_hook` events to remote peers for a given
## file. In a
## distributed setup, communicating Bro instances generate the event
## :bro:id:`print_hook` for each print statement and send it to the remote
## side. When disabled for a particular file, these events will not be
## propagated to other peers.
##
## f: The file to disable :bro:id:`print_hook` events for.
##
## .. bro:see:: enable_raw_output
global disable_print_hook: function(f: file ): any ;


## Prevents escaping of non-ASCII characters when writing to a file.
## This function is equivalent to :bro:attr:`&raw_output`.
##
## f: The file to disable raw output for.
##
## .. bro:see:: disable_print_hook
global enable_raw_output: function(f: file ): any ;








## Installs a filter to drop packets from a given IP source address with
## a certain probability if none of a given set of TCP flags are set.
## Note that for IPv6 packets with a Destination options header that has
## the Home Address option, this filters out against that home address.
##
## ip: The IP address to drop.
##
## tcp_flags: If none of these TCP flags are set, drop packets from *ip* with
##            probability *prob*.
##
## prob: The probability [0.0, 1.0] used to drop packets from *ip*.
##
## Returns: True (unconditionally).
##
## .. bro:see:: Pcap::precompile_pcap_filter
##              Pcap::install_pcap_filter
##              install_src_net_filter
##              uninstall_src_addr_filter
##              uninstall_src_net_filter
##              install_dst_addr_filter
##              install_dst_net_filter
##              uninstall_dst_addr_filter
##              uninstall_dst_net_filter
##              Pcap::error
##
## .. todo:: The return value should be changed to any.
global install_src_addr_filter: function(ip: addr , tcp_flags: count , prob: double ) : bool ;


## Installs a filter to drop packets originating from a given subnet with
## a certain probability if none of a given set of TCP flags are set.
##
## snet: The subnet to drop packets from.
##
## tcp_flags: If none of these TCP flags are set, drop packets from *snet* with
##            probability *prob*.
##
## prob: The probability [0.0, 1.0] used to drop packets from *snet*.
##
## Returns: True (unconditionally).
##
## .. bro:see:: Pcap::precompile_pcap_filter
##              Pcap::install_pcap_filter
##              install_src_addr_filter
##              uninstall_src_addr_filter
##              uninstall_src_net_filter
##              install_dst_addr_filter
##              install_dst_net_filter
##              uninstall_dst_addr_filter
##              uninstall_dst_net_filter
##              Pcap::error
##
## .. todo:: The return value should be changed to any.
global install_src_net_filter: function(snet: subnet , tcp_flags: count , prob: double ) : bool ;


## Removes a source address filter.
##
## ip: The IP address for which a source filter was previously installed.
##
## Returns: True on success.
##
## .. bro:see:: Pcap::precompile_pcap_filter
##              Pcap::install_pcap_filter
##              install_src_addr_filter
##              install_src_net_filter
##              uninstall_src_net_filter
##              install_dst_addr_filter
##              install_dst_net_filter
##              uninstall_dst_addr_filter
##              uninstall_dst_net_filter
##              Pcap::error
global uninstall_src_addr_filter: function(ip: addr ) : bool ;


## Removes a source subnet filter.
##
## snet: The subnet for which a source filter was previously installed.
##
## Returns: True on success.
##
## .. bro:see:: Pcap::precompile_pcap_filter
##              Pcap::install_pcap_filter
##              install_src_addr_filter
##              install_src_net_filter
##              uninstall_src_addr_filter
##              install_dst_addr_filter
##              install_dst_net_filter
##              uninstall_dst_addr_filter
##              uninstall_dst_net_filter
##              Pcap::error
global uninstall_src_net_filter: function(snet: subnet ) : bool ;


## Installs a filter to drop packets destined to a given IP address with
## a certain probability if none of a given set of TCP flags are set.
## Note that for IPv6 packets with a routing type header and non-zero
## segments left, this filters out against the final destination of the
## packet according to the routing extension header.
##
## ip: Drop packets to this IP address.
##
## tcp_flags: If none of these TCP flags are set, drop packets to *ip* with
##            probability *prob*.
##
## prob: The probability [0.0, 1.0] used to drop packets to *ip*.
##
## Returns: True (unconditionally).
##
## .. bro:see:: Pcap::precompile_pcap_filter
##              Pcap::install_pcap_filter
##              install_src_addr_filter
##              install_src_net_filter
##              uninstall_src_addr_filter
##              uninstall_src_net_filter
##              install_dst_net_filter
##              uninstall_dst_addr_filter
##              uninstall_dst_net_filter
##              Pcap::error
##
## .. todo:: The return value should be changed to any.
global install_dst_addr_filter: function(ip: addr , tcp_flags: count , prob: double ) : bool ;


## Installs a filter to drop packets destined to a given subnet with
## a certain probability if none of a given set of TCP flags are set.
##
## snet: Drop packets to this subnet.
##
## tcp_flags: If none of these TCP flags are set, drop packets to *snet* with
##            probability *prob*.
##
## prob: The probability [0.0, 1.0] used to drop packets to *snet*.
##
## Returns: True (unconditionally).
##
## .. bro:see:: Pcap::precompile_pcap_filter
##              Pcap::install_pcap_filter
##              install_src_addr_filter
##              install_src_net_filter
##              uninstall_src_addr_filter
##              uninstall_src_net_filter
##              install_dst_addr_filter
##              uninstall_dst_addr_filter
##              uninstall_dst_net_filter
##              Pcap::error
##
## .. todo:: The return value should be changed to any.
global install_dst_net_filter: function(snet: subnet , tcp_flags: count , prob: double ) : bool ;


## Removes a destination address filter.
##
## ip: The IP address for which a destination filter was previously installed.
##
## Returns: True on success.
##
## .. bro:see:: Pcap::precompile_pcap_filter
##              Pcap::install_pcap_filter
##              install_src_addr_filter
##              install_src_net_filter
##              uninstall_src_addr_filter
##              uninstall_src_net_filter
##              install_dst_addr_filter
##              install_dst_net_filter
##              uninstall_dst_net_filter
##              Pcap::error
global uninstall_dst_addr_filter: function(ip: addr ) : bool ;


## Removes a destination subnet filter.
##
## snet: The subnet for which a destination filter was previously installed.
##
## Returns: True on success.
##
## .. bro:see:: Pcap::precompile_pcap_filter
##              Pcap::install_pcap_filter
##              install_src_addr_filter
##              install_src_net_filter
##              uninstall_src_addr_filter
##              uninstall_src_net_filter
##              install_dst_addr_filter
##              install_dst_net_filter
##              uninstall_dst_addr_filter
##              Pcap::error
global uninstall_dst_net_filter: function(snet: subnet ) : bool ;








## Enables the communication system. By default, the communication is off until
## explicitly enabled, and all other calls to communication-related functions
## will be ignored until done so.
global enable_communication: function(): any ;


## Flushes in-memory state tagged with the :bro:attr:`&persistent` attribute
## to disk. The function writes the state to the file ``.state/state.bst`` in
## the directory where Bro was started.
##
## Returns: True on success.
##
## .. bro:see:: rescan_state
global checkpoint_state: function() : bool ;


## Reads persistent state and populates the in-memory data structures
## accordingly. Persistent state is read from the ``.state`` directory.
## This function is the dual to :bro:id:`checkpoint_state`.
##
## Returns: True on success.
##
## .. bro:see:: checkpoint_state
global rescan_state: function() : bool ;


## Writes the binary event stream generated by the core to a given file.
## Use the ``-x <filename>`` command line switch to replay saved events.
##
## filename: The name of the file which stores the events.
##
## Returns: True if opening the target file succeeds.
##
## .. bro:see:: capture_state_updates
global capture_events: function(filename: string ) : bool ;


## Writes state updates generated by :bro:attr:`&synchronized` variables to a
## file.
##
## filename: The name of the file which stores the state updates.
##
## Returns: True if opening the target file succeeds.
##
## .. bro:see:: capture_events
global capture_state_updates: function(filename: string ) : bool ;


## Establishes a connection to a remote Bro or Broccoli instance.
##
## ip: The IP address of the remote peer.
##
## zone_id: If *ip* is a non-global IPv6 address, a particular :rfc:`4007`
##          ``zone_id`` can given here.  An empty string, ``""``, means
##          not to add any ``zone_id``.
##
## p: The port of the remote peer.
##
## our_class: If a non-empty string, then the remote (listening) peer checks it
##            against its class name in its peer table and terminates the
##            connection if they don't match.
##
## retry: If the connection fails, try to reconnect with the peer after this
##        time interval.
##
## ssl: If true, use SSL to encrypt the session.
##
## Returns: A locally unique ID of the new peer.
##
## .. bro:see:: disconnect
##              listen
##              request_remote_events
##              request_remote_sync
##              request_remote_logs
##              request_remote_events
##              set_accept_state
##              set_compression_level
##              send_state
##              send_id
global connect: function(ip: addr , zone_id: string , p: port , our_class: string , retry: interval , ssl: bool ) : count ;


## Terminate the connection with a peer.
##
## p: The peer ID returned from :bro:id:`connect`.
##
## Returns: True on success.
##
## .. bro:see:: connect listen
global disconnect: function(p: event_peer ) : bool ;


## Subscribes to all events from a remote peer whose names match a given
## pattern.
##
## p: The peer ID returned from :bro:id:`connect`.
##
## handlers: The pattern describing the events to request from peer *p*.
##
## Returns: True on success.
##
## .. bro:see:: request_remote_sync
##              request_remote_logs
##              set_accept_state
global request_remote_events: function(p: event_peer , handlers: pattern ) : bool ;


## Requests synchronization of IDs with a remote peer.
##
## p: The peer ID returned from :bro:id:`connect`.
##
## auth: If true, the local instance considers its current state authoritative
##       and sends it to *p* right after the handshake.
##
## Returns: True on success.
##
## .. bro:see:: request_remote_events
##              request_remote_logs
##              set_accept_state
global request_remote_sync: function(p: event_peer , auth: bool ) : bool ;


## Requests logs from a remote peer.
##
## p: The peer ID returned from :bro:id:`connect`.
##
## Returns: True on success.
##
## .. bro:see:: request_remote_events
##              request_remote_sync
global request_remote_logs: function(p: event_peer ) : bool ;


## Sets a boolean flag indicating whether Bro accepts state from a remote peer.
##
## p: The peer ID returned from :bro:id:`connect`.
##
## accept: True if Bro accepts state from peer *p*, or false otherwise.
##
## Returns: True on success.
##
## .. bro:see:: request_remote_events
##              request_remote_sync
##              set_compression_level
global set_accept_state: function(p: event_peer , accept: bool ) : bool ;


## Sets the compression level of the session with a remote peer.
##
## p: The peer ID returned from :bro:id:`connect`.
##
## level: Allowed values are in the range *[0, 9]*, where 0 is the default and
##        means no compression.
##
## Returns: True on success.
##
## .. bro:see:: set_accept_state
global set_compression_level: function(p: event_peer , level: count ) : bool ;


## Listens on a given IP address and port for remote connections.
##
## ip: The IP address to bind to.
##
## p: The TCP port to listen on.
##
## ssl: If true, Bro uses SSL to encrypt the session.
##
## ipv6: If true, enable listening on IPv6 addresses.
##
## zone_id: If *ip* is a non-global IPv6 address, a particular :rfc:`4007`
##          ``zone_id`` can given here.  An empty string, ``""``, means
##          not to add any ``zone_id``.
##
## retry_interval: If address *ip* is found to be already in use, this is
##                 the interval at which to automatically retry binding.
##
## Returns: True on success.
##
## .. bro:see:: connect disconnect
global listen: function(ip: addr , p: port , ssl: bool , ipv6: bool , zone_id: string , retry_interval: interval ) : bool ;


## Checks whether the last raised event came from a remote peer.
##
## Returns: True if the last raised event came from a remote peer.
global is_remote_event: function() : bool ;


## Sends all persistent state to a remote peer.
##
## p: The peer ID returned from :bro:id:`connect`.
##
## Returns: True on success.
##
## .. bro:see:: send_id send_ping send_current_packet send_capture_filter
global send_state: function(p: event_peer ) : bool ;


## Sends a global identifier to a remote peer, which then might install it
## locally.
##
## p: The peer ID returned from :bro:id:`connect`.
##
## id: The identifier to send.
##
## Returns: True on success.
##
## .. bro:see:: send_state send_ping send_current_packet send_capture_filter
global send_id: function(p: event_peer , id: string ) : bool ;


## Gracefully finishes communication by first making sure that all remaining
## data from parent and child has been sent out.
##
## Returns: True if the termination process has been started successfully.
global terminate_communication: function() : bool ;


## Signals a remote peer that the local Bro instance finished the initial
## handshake.
##
## p: The peer ID returned from :bro:id:`connect`.
##
## Returns: True on success.
global complete_handshake: function(p: event_peer ) : bool ;


## Sends a ping event to a remote peer. In combination with an event handler
## for :bro:id:`remote_pong`, this function can be used to measure latency
## between two peers.
##
## p: The peer ID returned from :bro:id:`connect`.
##
## seq: A sequence number (also included by :bro:id:`remote_pong`).
##
## Returns: True if sending the ping succeeds.
##
## .. bro:see:: send_state send_id send_current_packet send_capture_filter
global send_ping: function(p: event_peer , seq: count ) : bool ;


## Sends the currently processed packet to a remote peer.
##
## p: The peer ID returned from :bro:id:`connect`.
##
## Returns: True if sending the packet succeeds.
##
## .. bro:see:: send_id send_state send_ping send_capture_filter
##              dump_packet dump_current_packet get_current_packet
global send_current_packet: function(p: event_peer ) : bool ;


## Returns the peer who generated the last event.
##
## Returns: The ID of the peer who generated the last event.
##
## .. bro:see:: get_local_event_peer
global get_event_peer: function() : event_peer ;


## Returns the local peer ID.
##
## Returns: The peer ID of the local Bro instance.
##
## .. bro:see:: get_event_peer
global get_local_event_peer: function() : event_peer ;


## Sends a capture filter to a remote peer.
##
## p: The peer ID returned from :bro:id:`connect`.
##
## s: The capture filter.
##
## Returns: True if sending the packet succeeds.
##
## .. bro:see:: send_id send_state send_ping send_current_packet
global send_capture_filter: function(p: event_peer , s: string ) : bool ;


## Stops Bro's packet processing. This function is used to synchronize
## distributed trace processing with communication enabled
## (*pseudo-realtime* mode).
##
## .. bro:see:: continue_processing suspend_state_updates resume_state_updates
global suspend_processing: function() : any ;


## Resumes Bro's packet processing.
##
## .. bro:see:: suspend_processing suspend_state_updates resume_state_updates
global continue_processing: function() : any ;


## Stops propagating :bro:attr:`&synchronized` accesses.
##
## .. bro:see:: suspend_processing continue_processing resume_state_updates
global suspend_state_updates: function() : any ;


## Resumes propagating :bro:attr:`&synchronized` accesses.
##
## .. bro:see:: suspend_processing continue_processing suspend_state_updates
global resume_state_updates: function() : any ;








## Manually triggers the signature engine for a given connection.
## This is an internal function.
global match_signatures: function(c: connection , pattern_type: int , s: string , bol: bool , eol: bool , from_orig: bool , clear: bool ) : bool ;












## Preserves the prefix of an IP address in anonymization.
##
## a: The address to preserve.
##
## width: The number of bits from the top that should remain intact.
##
## .. bro:see:: preserve_subnet anonymize_addr
##
## .. todo:: Currently dysfunctional.
global preserve_prefix: function(a: addr , width: count ): any ;


## Preserves the prefix of a subnet in anonymization.
##
## a: The subnet to preserve.
##
## .. bro:see:: preserve_prefix anonymize_addr
##
## .. todo:: Currently dysfunctional.
global preserve_subnet: function(a: subnet ): any ;


## Anonymizes an IP address.
##
## a: The address to anonymize.
##
## cl: The anonymization class, which can take on three different values:
##
##     - ``ORIG_ADDR``: Tag *a* as an originator address.
##
##     - ``RESP_ADDR``: Tag *a* as an responder address.
##
##     - ``OTHER_ADDR``: Tag *a* as an arbitrary address.
##
## Returns: An anonymized version of *a*.
##
## .. bro:see:: preserve_prefix preserve_subnet
##
## .. todo:: Currently dysfunctional.
global anonymize_addr: function(a: addr , cl: IPAddrAnonymizationClass ): addr ;

} # end of export section
module GLOBAL;
