/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.sqleditor.internal.actions;

import java.sql.Connection;
import java.util.ArrayList;
import java.util.Iterator;
import java.util.List;
import java.util.ResourceBundle;

import org.eclipse.jface.text.Document;
import org.eclipse.jface.text.IDocument;
import org.eclipse.jface.text.ITextSelection;
import org.eclipse.jface.viewers.ISelection;
import org.eclipse.jface.viewers.ISelectionProvider;
import org.eclipse.ui.PartInitException;
import org.eclipse.ui.PlatformUI;
import org.eclipse.ui.texteditor.ResourceAction;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;
import org.eclipse.wst.rdb.server.internal.ui.query.execute.QueryOutputHelper;
import org.eclipse.wst.rdb.sqleditor.internal.SQLEditor;
import org.eclipse.wst.rdb.sqleditor.internal.SQLEditorDocumentSetupParticipant;
import org.eclipse.wst.rdb.sqleditor.internal.utils.SQLDBUtils;
import org.eclipse.wst.rdb.sqleditor.internal.utils.SQLStatementSupport;

/**
 * This class implements the "Run SQL" action for the SQL Editor.
 * 
 * @see org.eclipse.wst.rdb.sqleditor.internal.SQLEditorActionContributor
 * @see org.eclipse.ui.texteditor.ResourceAction
 */
public class SQLRunAction extends ResourceAction {
    private SQLEditor fSQLEditor;
    
    /**
     * Creates an instance of this class using the given resource bundle
     * and prefix of a set of resources within the bundle.
     * 
     * @param bundle the resource bundle to use
     * @param prefix the resource prefix to use
     */
    public SQLRunAction( ResourceBundle bundle, String prefix ) {
        super( bundle, prefix );
    }
    
    /**
     * Gets the SQLEditor instance associated with this action.
     * 
     * @return the SQLEditor instance for this action
     */
    public SQLEditor getSQLEditor() {
        return fSQLEditor;
    }
    
    /**
     * Runs this action.  Runs the SQL statements contained in the SQL editor
     * document with the connected database.
     * 
     * @see org.eclipse.jface.action.IAction#run()
     */
    public void run() {
        SQLEditor sqlEditor = getSQLEditor();
        if (sqlEditor != null) {
            /* Create an empty list of SQL statements to start. */
            List stmtList = new ArrayList();
            
            /* Get the current statement terminator.  The statement terminator to use
             * might be imbedded in the document itself as an XML-like tag. */
            SQLStatementSupport stmtSupport = sqlEditor.getSQLStatementSupport();
            String stmtTerminator = stmtSupport.getStatementTerminator();

            /* Get the selected text in the editor, if any.  If there is a selection,
             * create a temporary document from it so we can parse it into statements. */  
            ISelectionProvider selectionProvider = fSQLEditor.getSelectionProvider();
            ISelection selection = selectionProvider.getSelection();
            if (selection.isEmpty() == false) {
                if (selection instanceof ITextSelection) {
                    ITextSelection textSelection = (ITextSelection) selection;
                    String text = textSelection.getText();
                    if (text.length() > 0) {
                        IDocument doc = new Document( text );
                        SQLEditorDocumentSetupParticipant setupParticipant = new SQLEditorDocumentSetupParticipant();
                        setupParticipant.setup( doc );
                        SQLStatementSupport stmtSupport2 = new SQLStatementSupport( doc );
                        stmtSupport2.setStatementTerminator( stmtTerminator );
                        stmtList = stmtSupport2.getSQLStatementList();
                    }
                }
            }
            
            /* If we don't have any statements in our list, scan the entire document. */
            if (stmtList.size() == 0) {
                stmtList = stmtSupport.getSQLStatementList();
            }

            /* If we have at least one statement to run, get the connection
             * and try to run them. */
            if (stmtList.size() > 0) {
                Connection conn = null;
                ConnectionInfo connInfo = fSQLEditor.getConnectionInfo();
                
                /* If we don't have a connection, prompt the user for one. */
                if (connInfo == null) {
                    connInfo = sqlEditor.requestConnectionFromUser();
                    /* Remember the connection for next time. */
                    if (connInfo != null) {
                        sqlEditor.setConnectionInfo( connInfo );
                    }
                }
                
                /* If we have the connection info, make sure that the connection
                 * is established. */
                if (connInfo != null) {
                    boolean connected = SQLDBUtils.reestablishConnection( connInfo );
                    if (connected == true) {
                        conn = connInfo.getSharedConnection();
                    }
                }
                    
                /* If we were able to get a connection, run each SQL statement and
                 * display the results in the output view. */
                if (conn != null) {
                    Iterator stmtListIter = stmtList.iterator();
                    while (stmtListIter.hasNext()) {
                        /* Get and run the next statement. */
                        String sql = (String) stmtListIter.next();
                        QueryOutputHelper outputHelper = new QueryOutputHelper( sql, conn );
                        outputHelper.runSQLScrapbookQuery();
                        
                        /*  Make sure the output view is open. */
                        try {
                            PlatformUI.getWorkbench().getActiveWorkbenchWindow().getActivePage().showView("org.eclipse.wst.rdb.internal.outputview.OutputView");
                        } catch (PartInitException e) {
                        }
                    }
                }
            }
        }
    }

    /**
     * Sets the SQLEditor instance associated with this action to the given object.
     * 
     * @param sqlEditor the SQLEditor instance for this action
     */
    public void setSQLEditor( SQLEditor sqlEditor ) {
        fSQLEditor = sqlEditor;
    }
}
