/*******************************************************************************
 * Copyright (c) 2001, 2004 IBM Corporation and others. All rights reserved.
 * This program and the accompanying materials are made available under the
 * terms of the Eclipse Public License v1.0 which accompanies this distribution,
 * and is available at http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors: IBM Corporation - initial API and implementation
 ******************************************************************************/
package org.eclipse.wst.rdb.server.internal.ui.wizards;

import java.io.File;
import java.sql.Connection;
import java.text.MessageFormat;
import java.util.List;

import org.eclipse.jface.dialogs.IDialogSettings;
import org.eclipse.jface.dialogs.MessageDialog;
import org.eclipse.jface.viewers.IStructuredSelection;
import org.eclipse.jface.wizard.IWizardPage;
import org.eclipse.jface.wizard.Wizard;
import org.eclipse.swt.SWT;
import org.eclipse.swt.widgets.Composite;
import org.eclipse.swt.widgets.DirectoryDialog;
import org.eclipse.swt.widgets.Display;
import org.eclipse.swt.widgets.FileDialog;
import org.eclipse.ui.IWorkbench;
import org.eclipse.wst.rdb.connection.internal.ui.RDBConnectionUIPlugin;
import org.eclipse.wst.rdb.connection.internal.ui.factories.RSCCoreUIWidgetFactory;
import org.eclipse.wst.rdb.connection.internal.ui.icons.ImageDescription;
import org.eclipse.wst.rdb.internal.core.RDBCorePlugin;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;
import org.eclipse.wst.rdb.internal.core.definition.DatabaseDefinition;
import org.eclipse.wst.rdb.internal.core.rte.ICatalogObject;
import org.eclipse.wst.rdb.internal.core.util.DatabaseProviderHelper;
import org.eclipse.wst.rdb.server.internal.ui.util.resources.ResourceLoader;

public class NewConnectionWizard extends Wizard {
    private static final ResourceLoader resource = ResourceLoader.INSTANCE;

    public static final int ALIAS_CONNECTION_TYPE = 1;

    public static final int JDBC_CONNECTION_TYPE = 2;

    public static final int JNDI_CONNECTION_TYPE = 3;

    public final static int DEFAULT_OPTIONS = 0;

    public final static int SHOW_EXISTING_CONS = 1;

    public final static int DISABLE_DB = 2;

    public final static int DISABLE_CURRENT_USERID_PWD = 4;

    public final static int DISABLE_VENDOR_TYPE = 8;

    public final static int DISABLE_DRIVER = 16;

    public final static int SELECT_EXISTING = 32;

    public final static int SHOW_JNDI_DATASOURCES = 64;

    public final static int SHOW_ADD_ALIASES = 128;

    public final static int SHOW_COMBINED_FILTERS = 256;

    public final static int SHOW_CALLING_PAGE = 512;

    public static final String STORE_USE_OS_UID_PWD = "NewConnectionWizard.STORE_USE_OS_UID_PWD"; //$NON-NLS-1$

    public static final String STORE_USERIDS = "NewConnectionWizard.STORE_USERIDS"; //$NON-NLS-1$

    public static final String CONNECTION_NULL = resource
            .queryString("DATATOOLS.SERVER.UI.EXPLORER.NULL_CONNECTION"); //$NON-NLS-1$

    public static final String CONNECTION_EXCEPTION = resource
            .queryString("DATATOOLS.SERVER.UI.EXPLORER.CONNECTION_EXCEPTION"); //$NON-NLS-1$
    
    private static final String CONFIGURE_JDBC_WIZARD_PAGE_NAME = "org.eclipse.wst.rdb.server.internal.ui.wizards.JdbcPage"; //$NON-NLS-1$

    private static final  String  SCHEMA_FILTER_WIZARD_PAGE_NAME= "org.eclipse.wst.rdb.server.internal.ui.wizards.SchemaFilterPage"; //$NON-NLS-1$
    //    public static final String STORE_CONNECTION_TYPE =
    // "NewConnectionWizard.STORE_CONNECTION_TYPE";
    //    public static final String STORE_NEW_CONNECTION_NAMES =
    // "NewConnectionWizard.STORE_NEW_CONNECTION_NAMES";
    //    public static final String STORE_ALIAS_NAMES =
    // "NewConnectionWizard.STORE_ALIAS_NAMES";
    //    public static final String STORE_DATABASE_NAMES =
    // "NewConnectionWizard.STORE_DATABASE_NAMES";
    //    public static final String STORE_DATABASE_VENDOR_TYPE =
    // "NewConnectionWizard.STORE_DATABASE_VENDOR_TYPE";
    //    public static final String STORE_JDBC_DRIVER =
    // "NewConnectionWizard.STORE_JDBC_DRIVER";
    //    public static final String STORE_HOST_VALUES =
    // "NewConnectionWizard.STORE_HOST_VALUES";
    //    public static final String STORE_PORT_NUMBERS =
    // "NewConnectionWizard.STORE_PORT_NUMBERS";
    //    public static final String STORE_SERVER_NAMES =
    // "NewConnectionWizard.STORE_SERVER_NAMES";
    //    public static final String STORE_DATABASE_LOCATIONS =
    // "NewConnectionWizard.STORE_DBLOCATION_NAMES";
    //    public static final String STORE_JDBC_CLASSES =
    // "NewConnectionWizard.STORE_JDBC_CLASSES";
    //    public static final String STORE_CLASS_LOCATIONS =
    // "NewConnectionWizard.STORE_CLASS_LOCATIONS";
    //    public static final String STORE_CONNECTIONS_URL =
    // "NewConnectionWizard.STORE_CONNECTIONS_URL";
    //    public static final String STORE_SHOW_EXISTING_CONNECTIONS =
    // "NewConnectionWizard.STORE_SHOW_EXISTING_CONNECTIONS";
    //    public static final String STORE_SCHEMA_FILTERS =
    // "NewConnectionWizard.STORE_SCHEMA_FILTERS";
    //    public static final String STORE_TABLE_FILTERS =
    // "NewConnectionWizard.STORE_TABLE_FILTERS";
    //    public static final String STORE_ROUTINE_FILTERS =
    // "NewConnectionWizard.STORE_ROUTINE_FILTERS";
    //    public static final String STORE_JAVA_HOME_DIRS =
    // "NewConnectionWizard.STORE_JAVA_HOME_DIRS";
    //    public static final String STORE_SQLSCHEMA_IDS =
    // "NewConnectionWizard.STORE_SQLSCHEMA_IDS";
    //    public static final String STORE_PACKAGE_OWNERS =
    // "NewConnectionWizard.STORE_PACKAGE_OWNERS";
    //    public static final String STORE_BUILD_OWNERS =
    // "NewConnectionWizard.STORE_BUILD_OWNERS";
    //    public static final String STORE_INITIAL_CONTEXT_FACTORY =
    // "NewConnectionWizard.STORE_INITIAL_CONTEXT_FACTORY";
    //    public static final String STORE_URL_PROVIDER =
    // "NewConnectionWizard.STORE_URL_PROVIDER";
    //    public static final String STORE_JNDI_NAME =
    // "NewConnectionWizard.STORE_JNDI_NAME";

    private List existingConnectionNameList;

    private IWorkbench iWorkbench;

    private IStructuredSelection iSelection;

    private NewConnectionJDBCConfigurationWizardPage jdbcPage;

    private NewConnectionSchemaFilterWizardPage schemaFilterPage;

    private ConnectionInfo connection;

    private DatabaseDefinition definition;

    private boolean enableLogging;

    private static int OPTIONS = DEFAULT_OPTIONS;

    public static String[] fExtensions = new String[] { "*.jar;*.zip", "*.*" }; //$NON-NLS-1$ //$NON-NLS-2$

    private static String FILE_SEPARATOR = ";"; //$NON-NLS-1$

    public NewConnectionWizard() {
        super();
        setWindowTitle(resource.queryString("_UI_DIALOG_CONNECTION_TITLE")); //$NON-NLS-1$      
        setNeedsProgressMonitor(true);
        enableLogging = false;
        NewConnectionWizard.OPTIONS = NewConnectionWizard.DEFAULT_OPTIONS;
        wizardInit();
    }

    public NewConnectionWizard(int OPTIONS) {
        this();
        NewConnectionWizard.OPTIONS = OPTIONS;
    }

    public NewConnectionWizard(DatabaseDefinition object, boolean loadMetadata) {
        this(object, loadMetadata, NewConnectionWizard.DEFAULT_OPTIONS);
    }

    /**
     * @param object
     *            An RDBConnection or SQL vendor or null. If this is not null,
     *            the page restricts the vendor. This is for a top down
     *            approach, if you've created a DB2 7.2 database, and associated
     *            statements/SP/UDF's, if you should require a connection, you
     *            should be retricted to the type of database you reside under.
     *            This prevents connecting to an Oracle database when you're a
     *            member of the DB2 7.2 domain.
     */
    public NewConnectionWizard(DatabaseDefinition object, boolean loadMetaData,
            int OPTIONS) {
        this();
        this.definition = ((DatabaseDefinition) object);
        this.enableLogging = false;
        NewConnectionWizard.OPTIONS = OPTIONS;
    }

    /**
     * The <code>Wizard</code> implementation of this <code>IWizard</code>
     * method creates all the pages controls using
     * <code>IDialogPage.createControl</code>. Subclasses should reimplement
     * this method if they want to delay creating one or more of the pages
     * lazily. The framework ensures that the contents of a page will be created
     * before attempting to show it.
     */
    public void createPageControls(Composite pageContainer) {
        super.createPageControls(pageContainer);
    }

    /**
     * @param existingConnectionNamesList
     *            The existing set of Connection names registered in the Server
     *            Explorer
     */
    public void setExistingConnectionNames(List existingConnectionNamesList) {
        this.existingConnectionNameList = existingConnectionNamesList;
    }

    private void wizardInit() {
        IDialogSettings workbenchSettings = RDBConnectionUIPlugin.getDefault()
                .getDialogSettings();
        IDialogSettings section = workbenchSettings
                .getSection("NewConnectionWizard");//$NON-NLS-1$
        if (section == null) {
            section = workbenchSettings.addNewSection("NewConnectionWizard");//$NON-NLS-1$
        }
        setDialogSettings(section);
    }

    public static RSCCoreUIWidgetFactory getUIFactory() {
        return RSCCoreUIWidgetFactory.INSTANCE;
    }

    public void addPages() {
        super.addPages();

        // JDBC connection information
        jdbcPage = new NewConnectionJDBCConfigurationWizardPage(
                CONFIGURE_JDBC_WIZARD_PAGE_NAME, //$NON-NLS-1$
                NewConnectionWizard.OPTIONS, this.existingConnectionNameList);
        addPage(jdbcPage);
        jdbcPage.setDefaultDatabaseDefinition(this.definition);

        // Schema filter
        schemaFilterPage = new NewConnectionSchemaFilterWizardPage(
                SCHEMA_FILTER_WIZARD_PAGE_NAME     );
        addPage(schemaFilterPage);

    }

    /**
     * Overrides wizard to test only pages in the sequence. Default
     * implementation is to check if all pages are complete.
     */
    public boolean canFinish() {
        boolean canFinish = false;
        if (jdbcPage.isPageComplete()
                && schemaFilterPage.isPageComplete()) {
            canFinish = true;
        } 
        return canFinish;
    }

    public boolean performFinish() {
        boolean isPerformFinishSuccessful = false;
        if (getDBConnection(true) != null){
            schemaFilterPage.applyFilter();
            ((ICatalogObject)connection.getSharedDatabase()).refresh();
            isPerformFinishSuccessful = true;
        }
        return isPerformFinishSuccessful;
    }
    
    public boolean performCancel() {
        boolean isPerformCancelSuccessful = true;
        if (this.connection != null){
            RDBCorePlugin.getDefault().getConnectionManager()
            .removeConnectionInfo(connection.getName());
        }
        return isPerformCancelSuccessful;
    }
    
    public ConnectionInfo getDBConnection(boolean showServerMismatchWarning){
        if (this.connection == null) {
        internalSaveWidgetValues();
        jdbcPage.performTestConnection(false, showServerMismatchWarning);
        if (jdbcPage.isFinalConnection()) {
            this.connection = jdbcPage.getConnection();
        }
        if (jdbcPage.isFinalConnection()) {
            displayNewServer(getConnection());
        }   
        }
        return this.connection;
    }

    /**
     * Saves the dialog state of every relevant page.
     */
    protected void internalSaveWidgetValues() {
        jdbcPage.internalSaveWidgetValues();

        //       filterPage.internalSaveWidgetValues();
    }

    /**
     * Gets the connection we are creating.
     */
    public ConnectionInfo getConnection() {
        return connection;
    }

    public void init(IWorkbench aWorkbench, IStructuredSelection aSelection) {
        init(aWorkbench, aSelection, false);
    }

    public void init(IWorkbench aWorkbench, IStructuredSelection aSelection,
            boolean edit) {
        setDefaultPageImageDescriptor(ImageDescription
                .getDescriptor("NewConnectionWiz.gif")); //$NON-NLS-1$
    }

    /**
     * Helper to open the file chooser dialog to select multiple files.
     * 
     * @param wizard
     *            An instance of ourself, so we can get our shell.
     * @param fExtensions
     *            An array of file extensions
     */
    public static String getFile(Wizard wizard, String[] fExtensions) {
        StringBuffer result = new StringBuffer();
        FileDialog dialog = new FileDialog(wizard.getShell(), SWT.MULTI);
        if (fExtensions != null) {
            dialog.setFilterExtensions(fExtensions);
        }
        dialog.open();
        String path = dialog.getFilterPath();
        String[] files = dialog.getFileNames();
        if (files != null) {
            for (int i = 0; i < files.length; i++) {
                if (result.length() > 0) {
                    result.append(FILE_SEPARATOR);
                }
                result.append(path).append(File.separator).append(files[i]);
            }
        }
        return result.toString();
    }

    public static String getDirectory(Wizard wizard, String current,
            String message) {
        DirectoryDialog dialog = new DirectoryDialog(wizard.getShell(),
                SWT.SINGLE);
        dialog.setMessage(message);
        dialog.setFilterPath(current);
        return dialog.open();
    }

    /**
     * Gets the enableLogging
     * 
     * @return Returns a boolean
     */
    private boolean getLogging() {
        return enableLogging;
    }

    /**
     * Sets the enableLoggin
     * 
     * @param enableLogging
     *            The enableLogging to set
     */
    private void setLogging(boolean enableLogging) {
        this.enableLogging = enableLogging;
    }

    /**
     * Reports the connection type: alias, JDBC, or JNDI.
     * 
     * @return One of the CONNECTION_TYPE constants:
     *         <ul>
     *         <li>NewConnectionWizard.JNDI_CONNECTION_TYPE
     *         <li>NewConnectionWizard.JDBC_CONNECTION_TYPE
     *         <li>NewConnectionWizard.ALIAS_CONNECTION_TYPE
     *         </ul>
     */
    public int getConnectionType() {
        return NewConnectionWizard.JDBC_CONNECTION_TYPE;
    }

    /**
     * Sets the connection details. Used for populating new connections.
     * 
     * @param connection
     *            An RDBConnection object to reflect the choices in the wizard
     *            pages.
     */
    protected void setConnectionDetails(ConnectionInfo connection) {
        if (connection == null) {
            return;
        }

        jdbcPage.setConnectionDetails(connection);

        //       filterPage.setConnectionDetails(connection);

    }

    /**
     * Returns the accumulated messages of nested exceptions. Returns the class
     * name of the exception if no message is available.
     */
    public static String getMessages(Throwable e) {
        StringBuffer eb = new StringBuffer();
        Throwable t = (Throwable) e;
        if (t.getLocalizedMessage() != null)
            eb.append(t.getLocalizedMessage());
        else
            eb.append(t.getClass().getName());
        while (t.getCause() != null) {
            t = t.getCause();
            eb.append("; "); //$NON-NLS-1$
            if (t.getLocalizedMessage() != null)
                eb.append(t.getLocalizedMessage());
            else
                eb.append(t.getClass().getName());
        }
        return eb.toString();
    }

    private boolean displayNewServer(ConnectionInfo connectionInfo) {
        Connection connection = null;
        try {
            connection = connectionInfo.connect();
            connectionInfo.setSharedConnection(connection);
            connectionInfo.saveConnectionInfo();
            new DatabaseProviderHelper().setDatabase(connection,
                    connectionInfo, connectionInfo.getDatabaseName());
            return true;
        } catch (Exception e) {
            RDBCorePlugin.getDefault().getConnectionManager()
                    .removeConnectionInfo(connectionInfo.getName());
            displayError(e.getMessage());
            return false;
        }
    }

    private void displayError(String message) {
        MessageDialog.openError(Display.getCurrent().getActiveShell(),
                CONNECTION_NULL, MessageFormat.format(CONNECTION_EXCEPTION,
                        new String[] { message }));
    }
    
    public IWizardPage getNextPage(IWizardPage wizardPage) {
        IWizardPage nextPage = null;

        if (wizardPage.getName().equals(CONFIGURE_JDBC_WIZARD_PAGE_NAME)) {
            nextPage = schemaFilterPage;
        } else if (wizardPage.getName().equals(SCHEMA_FILTER_WIZARD_PAGE_NAME)) {
            nextPage = null;
        } else {
            nextPage = super.getNextPage(wizardPage);
        }
        return nextPage;
    }
}