/*
 * Copyright 2010-2015 JetBrains s.r.o.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package kotlin.reflect.jvm.internal.impl.util

import kotlin.reflect.jvm.internal.impl.builtins.KotlinBuiltIns
import kotlin.reflect.jvm.internal.impl.builtins.ReflectionTypes
import kotlin.reflect.jvm.internal.impl.descriptors.ClassDescriptor
import kotlin.reflect.jvm.internal.impl.descriptors.DeclarationDescriptor
import kotlin.reflect.jvm.internal.impl.descriptors.FunctionDescriptor
import kotlin.reflect.jvm.internal.impl.descriptors.ValueParameterDescriptor
import kotlin.reflect.jvm.internal.impl.resolve.descriptorUtil.builtIns
import kotlin.reflect.jvm.internal.impl.resolve.descriptorUtil.hasDefaultValue
import kotlin.reflect.jvm.internal.impl.resolve.descriptorUtil.isExtension
import kotlin.reflect.jvm.internal.impl.resolve.descriptorUtil.module
import kotlin.reflect.jvm.internal.impl.types.typeUtil.isSubtypeOf
import kotlin.reflect.jvm.internal.impl.types.typeUtil.makeNotNullable
import kotlin.reflect.jvm.internal.impl.util.OperatorNameConventions.ASSIGNMENT_OPERATIONS
import kotlin.reflect.jvm.internal.impl.util.OperatorNameConventions.BINARY_OPERATION_NAMES
import kotlin.reflect.jvm.internal.impl.util.OperatorNameConventions.COMPARE_TO
import kotlin.reflect.jvm.internal.impl.util.OperatorNameConventions.COMPONENT_REGEX
import kotlin.reflect.jvm.internal.impl.util.OperatorNameConventions.CONTAINS
import kotlin.reflect.jvm.internal.impl.util.OperatorNameConventions.DEC
import kotlin.reflect.jvm.internal.impl.util.OperatorNameConventions.EQUALS
import kotlin.reflect.jvm.internal.impl.util.OperatorNameConventions.GET
import kotlin.reflect.jvm.internal.impl.util.OperatorNameConventions.GET_VALUE
import kotlin.reflect.jvm.internal.impl.util.OperatorNameConventions.HAS_NEXT
import kotlin.reflect.jvm.internal.impl.util.OperatorNameConventions.INC
import kotlin.reflect.jvm.internal.impl.util.OperatorNameConventions.INVOKE
import kotlin.reflect.jvm.internal.impl.util.OperatorNameConventions.ITERATOR
import kotlin.reflect.jvm.internal.impl.util.OperatorNameConventions.NEXT
import kotlin.reflect.jvm.internal.impl.util.OperatorNameConventions.RANGE_TO
import kotlin.reflect.jvm.internal.impl.util.OperatorNameConventions.SET
import kotlin.reflect.jvm.internal.impl.util.OperatorNameConventions.SET_VALUE
import kotlin.reflect.jvm.internal.impl.util.OperatorNameConventions.SIMPLE_UNARY_OPERATION_NAMES

object OperatorChecks {
    fun canBeOperator(functionDescriptor: FunctionDescriptor): Boolean {
        val name = functionDescriptor.name

        return with (functionDescriptor) {
            if (!functionDescriptor.isMemberOrExtension) return false

            when {
                GET == name -> valueParameters.size >= 1
                SET == name -> {
                    val lastIsOk = valueParameters.lastOrNull()?.let { !it.hasDefaultValue() && it.varargElementType == null } ?: false
                    valueParameters.size >= 2 && lastIsOk
                }

                GET_VALUE == name -> noDefaultsAndVarargs && valueParameters.size >= 2 && valueParameters[1].isKProperty
                SET_VALUE == name -> noDefaultsAndVarargs && valueParameters.size >= 3 && valueParameters[1].isKProperty

                INVOKE == name -> isMemberOrExtension
                CONTAINS == name -> singleValueParameter && noDefaultsAndVarargs && returnsBoolean

                ITERATOR == name -> noValueParameters
                NEXT == name -> noValueParameters
                HAS_NEXT == name -> noValueParameters && returnsBoolean

                RANGE_TO == name -> singleValueParameter && noDefaultsAndVarargs

                EQUALS == name -> {
                    fun DeclarationDescriptor.isAny() = (this as? ClassDescriptor)?.let { KotlinBuiltIns.isAny(it) } ?: false
                    isMember && overriddenDescriptors.any { it.containingDeclaration.isAny() }
                }
                COMPARE_TO == name -> returnsInt && singleValueParameter && noDefaultsAndVarargs

                name in BINARY_OPERATION_NAMES -> singleValueParameter && noDefaultsAndVarargs

                name in SIMPLE_UNARY_OPERATION_NAMES -> noValueParameters

                INC == name || DEC == name -> {
                    val receiver = dispatchReceiverParameter ?: extensionReceiverParameter
                    isMemberOrExtension && (receiver != null) && (returnType?.let { it.isSubtypeOf(receiver.type) } ?: false)
                }

                name in ASSIGNMENT_OPERATIONS ->
                    returnsUnit && singleValueParameter && noDefaultsAndVarargs

                name.asString().matches(COMPONENT_REGEX) -> noValueParameters

                else -> false
            }
        }
    }

    private val ValueParameterDescriptor.isKProperty: Boolean
        get() = ReflectionTypes.createKPropertyStarType(module)?.isSubtypeOf(type.makeNotNullable()) ?: false

    private val FunctionDescriptor.isMember: Boolean
        get() = containingDeclaration is ClassDescriptor

    private val FunctionDescriptor.isMemberOrExtension: Boolean
        get() = isExtension || containingDeclaration is ClassDescriptor

    private val FunctionDescriptor.noValueParameters: Boolean
        get() = valueParameters.isEmpty()

    private val FunctionDescriptor.singleValueParameter: Boolean
        get() = valueParameters.size == 1

    private val FunctionDescriptor.returnsBoolean: Boolean
        get() = returnType?.let { KotlinBuiltIns.isBoolean(it) } ?: false

    private val FunctionDescriptor.returnsInt: Boolean
        get() = returnType?.let { builtIns.intType == it } ?: false //TODO

    private val FunctionDescriptor.returnsUnit: Boolean
        get() = returnType?.let { builtIns.unitType == it } ?: false //TODO

    private val FunctionDescriptor.noDefaultsAndVarargs: Boolean
        get() = valueParameters.all { !it.hasDefaultValue() && it.varargElementType == null }

}

fun FunctionDescriptor.isValidOperator() = isOperator && OperatorChecks.canBeOperator(this)