/*
 * Created on Jan 29, 2005
 */
package org.rubypeople.rdt.internal.core.util;

/**
 * @author Chris
 */
public class CharOperation {

	/**
	 * Answers the first index in the array for which the corresponding
	 * character is equal to toBeFound starting the search at index start.
	 * Answers -1 if no occurrence of this character is found. <br>
	 * <br>
	 * For example:
	 * <ol>
	 * <li>
	 * 
	 * <pre>
	 *  
	 *   
	 *    
	 *     
	 *      
	 *          toBeFound = 'c'
	 *          array = { ' a', 'b', 'c', 'd' }
	 *          start = 2
	 *          result =&gt; 2
	 *       
	 *      
	 *     
	 *    
	 *   
	 * </pre>
	 * 
	 * </li>
	 * <li>
	 * 
	 * <pre>
	 *  
	 *   
	 *    
	 *     
	 *      
	 *          toBeFound = 'c'
	 *          array = { ' a', 'b', 'c', 'd' }
	 *          start = 3
	 *          result =&gt; -1
	 *       
	 *      
	 *     
	 *    
	 *   
	 * </pre>
	 * 
	 * </li>
	 * <li>
	 * 
	 * <pre>
	 *  
	 *   
	 *    
	 *     
	 *      
	 *          toBeFound = 'e'
	 *          array = { ' a', 'b', 'c', 'd' }
	 *          start = 1
	 *          result =&gt; -1
	 *       
	 *      
	 *     
	 *    
	 *   
	 * </pre>
	 * 
	 * </li>
	 * </ol>
	 * 
	 * @param toBeFound
	 *            the character to search
	 * @param array
	 *            the array to be searched
	 * @param start
	 *            the starting index
	 * @return the first index in the array for which the corresponding
	 *         character is equal to toBeFound, -1 otherwise
	 * @throws NullPointerException
	 *             if array is null
	 * @throws ArrayIndexOutOfBoundsException
	 *             if start is lower than 0
	 */
	public static final int indexOf(char toBeFound, char[] array, int start) {
		for (int i = start; i < array.length; i++)
			if (toBeFound == array[i])
				return i;
		return -1;
	}

	/**
	 * Answers a new array of characters with substitutions. No side-effect is
	 * operated on the original array, in case no substitution happened, then
	 * the result is the same as the original one. <br>
	 * <br>
	 * For example:
	 * <ol>
	 * <li>
	 * 
	 * <pre>
	 *  
	 *   
	 *    
	 *        array = { 'a' , 'b', 'b', 'a', 'b', 'a' }
	 *        toBeReplaced = { 'b' }
	 *        replacementChar = { 'a', 'a' }
	 *        result =&gt; { 'a', 'a', 'a', 'a', 'a', 'a', 'a', 'a', 'a' }
	 *     
	 *    
	 *   
	 * </pre>
	 * 
	 * </li>
	 * <li>
	 * 
	 * <pre>
	 *  
	 *   
	 *    
	 *        array = { 'a' , 'b', 'b', 'a', 'b', 'a' }
	 *        toBeReplaced = { 'c' }
	 *        replacementChar = { 'a' }
	 *        result =&gt; { 'a' , 'b', 'b', 'a', 'b', 'a' }
	 *     
	 *    
	 *   
	 * </pre>
	 * 
	 * </li>
	 * </ol>
	 * 
	 * @param array
	 *            the given array
	 * @param toBeReplaced
	 *            characters to be replaced
	 * @param replacementChars
	 *            the replacement characters
	 * @return a new array of characters with substitutions or the given array
	 *         if none
	 * @throws NullPointerException
	 *             if the given array is null
	 */
	public static final char[] replace(char[] array, char[] toBeReplaced,
			char[] replacementChars) {

		int max = array.length;
		int replacedLength = toBeReplaced.length;
		int replacementLength = replacementChars.length;

		int[] starts = new int[5];
		int occurrenceCount = 0;

		if (!equals(toBeReplaced, replacementChars)) {

			next: for (int i = 0; i < max; i++) {
				int j = 0;
				while (j < replacedLength) {
					if (i + j == max)
						continue next;
					if (array[i + j] != toBeReplaced[j++])
						continue next;
				}
				if (occurrenceCount == starts.length) {
					System.arraycopy(starts, 0,
							starts = new int[occurrenceCount * 2], 0,
							occurrenceCount);
				}
				starts[occurrenceCount++] = i;
			}
		}
		if (occurrenceCount == 0)
			return array;
		char[] result = new char[max + occurrenceCount
				* (replacementLength - replacedLength)];
		int inStart = 0, outStart = 0;
		for (int i = 0; i < occurrenceCount; i++) {
			int offset = starts[i] - inStart;
			System.arraycopy(array, inStart, result, outStart, offset);
			inStart += offset;
			outStart += offset;
			System.arraycopy(replacementChars, 0, result, outStart,
					replacementLength);
			inStart += replacedLength;
			outStart += replacementLength;
		}
		System.arraycopy(array, inStart, result, outStart, max - inStart);
		return result;
	}

	/**
	 * Answers true if the two arrays are identical character by character,
	 * otherwise false. The equality is case sensitive. <br>
	 * <br>
	 * For example:
	 * <ol>
	 * <li>
	 * 
	 * <pre>
	 *  
	 *   
	 *       first = null
	 *       second = null
	 *       result =&gt; true
	 *    
	 *   
	 * </pre>
	 * 
	 * </li>
	 * <li>
	 * 
	 * <pre>
	 *  
	 *   
	 *       first = { }
	 *       second = null
	 *       result =&gt; false
	 *    
	 *   
	 * </pre>
	 * 
	 * </li>
	 * <li>
	 * 
	 * <pre>
	 *  
	 *   
	 *       first = { 'a' }
	 *       second = { 'a' }
	 *       result =&gt; true
	 *    
	 *   
	 * </pre>
	 * 
	 * </li>
	 * <li>
	 * 
	 * <pre>
	 *  
	 *   
	 *       first = { 'a' }
	 *       second = { 'A' }
	 *       result =&gt; false
	 *    
	 *   
	 * </pre>
	 * 
	 * </li>
	 * </ol>
	 * 
	 * @param first
	 *            the first array
	 * @param second
	 *            the second array
	 * @return true if the two arrays are identical character by character,
	 *         otherwise false
	 */
	public static final boolean equals(char[] first, char[] second) {
		if (first == second)
			return true;
		if (first == null || second == null)
			return false;
		if (first.length != second.length)
			return false;

		for (int i = first.length; --i >= 0;)
			if (first[i] != second[i])
				return false;
		return true;
	}

	/**
	 * Answers true if the pattern matches the filepath using the pathSepatator,
	 * false otherwise.
	 * 
	 * Path char[] pattern matching, accepting wild-cards '**', '*' and '?'
	 * (using Ant directory tasks conventions, also see
	 * "http://jakarta.apache.org/ant/manual/dirtasks.html#defaultexcludes").
	 * Path pattern matching is enhancing regular pattern matching in supporting
	 * extra rule where '**' represent any folder combination. Special rule: -
	 * foo\ is equivalent to foo\** When not case sensitive, the pattern is
	 * assumed to already be lowercased, the name will be lowercased character
	 * per character as comparing.
	 * 
	 * @param pattern
	 *            the given pattern
	 * @param filepath
	 *            the given path
	 * @param isCaseSensitive
	 *            to find out whether or not the matching should be case
	 *            sensitive
	 * @param pathSeparator
	 *            the given path separator
	 * @return true if the pattern matches the filepath using the pathSepatator,
	 *         false otherwise
	 */
	public static final boolean pathMatch(char[] pattern, char[] filepath,
			boolean isCaseSensitive, char pathSeparator) {

		if (filepath == null)
			return false; // null name cannot match
		if (pattern == null)
			return true; // null pattern is equivalent to '*'

		// offsets inside pattern
		int pSegmentStart = pattern[0] == pathSeparator ? 1 : 0;
		int pLength = pattern.length;
		int pSegmentEnd = CharOperation.indexOf(pathSeparator, pattern,
				pSegmentStart + 1);
		if (pSegmentEnd < 0)
			pSegmentEnd = pLength;

		// special case: pattern foo\ is equivalent to foo\**
		boolean freeTrailingDoubleStar = pattern[pLength - 1] == pathSeparator;

		// offsets inside filepath
		int fSegmentStart, fLength = filepath.length;
		if (filepath[0] != pathSeparator) {
			fSegmentStart = 0;
		} else {
			fSegmentStart = 1;
		}
		if (fSegmentStart != pSegmentStart) {
			return false; // both must start
			// with a separator
			// or none.
		}
		int fSegmentEnd = CharOperation.indexOf(pathSeparator, filepath,
				fSegmentStart + 1);
		if (fSegmentEnd < 0)
			fSegmentEnd = fLength;

		// first segments
		while (pSegmentStart < pLength
				&& !(pSegmentEnd == pLength && freeTrailingDoubleStar || (pSegmentEnd == pSegmentStart + 2
						&& pattern[pSegmentStart] == '*' && pattern[pSegmentStart + 1] == '*'))) {

			if (fSegmentStart >= fLength)
				return false;
			if (!CharOperation.match(pattern, pSegmentStart, pSegmentEnd,
					filepath, fSegmentStart, fSegmentEnd, isCaseSensitive)) {
				return false;
			}

			// jump to next segment
			pSegmentEnd = CharOperation.indexOf(pathSeparator, pattern,
					pSegmentStart = pSegmentEnd + 1);
			// skip separator
			if (pSegmentEnd < 0)
				pSegmentEnd = pLength;

			fSegmentEnd = CharOperation.indexOf(pathSeparator, filepath,
					fSegmentStart = fSegmentEnd + 1);
			// skip separator
			if (fSegmentEnd < 0)
				fSegmentEnd = fLength;
		}

		/* check sequence of doubleStar+segment */
		int pSegmentRestart;
		if ((pSegmentStart >= pLength && freeTrailingDoubleStar)
				|| (pSegmentEnd == pSegmentStart + 2
						&& pattern[pSegmentStart] == '*' && pattern[pSegmentStart + 1] == '*')) {
			pSegmentEnd = CharOperation.indexOf(pathSeparator, pattern,
					pSegmentStart = pSegmentEnd + 1);
			// skip separator
			if (pSegmentEnd < 0)
				pSegmentEnd = pLength;
			pSegmentRestart = pSegmentStart;
		} else {
			if (pSegmentStart >= pLength)
				return fSegmentStart >= fLength; // true
			// if
			// filepath
			// is
			// done
			// too.
			pSegmentRestart = 0; // force fSegmentStart check
		}
		int fSegmentRestart = fSegmentStart;
		checkSegment: while (fSegmentStart < fLength) {

			if (pSegmentStart >= pLength) {
				if (freeTrailingDoubleStar)
					return true;
				// mismatch - restart current path segment
				pSegmentEnd = CharOperation.indexOf(pathSeparator, pattern,
						pSegmentStart = pSegmentRestart);
				if (pSegmentEnd < 0)
					pSegmentEnd = pLength;

				fSegmentRestart = CharOperation.indexOf(pathSeparator,
						filepath, fSegmentRestart + 1);
				// skip separator
				if (fSegmentRestart < 0) {
					fSegmentRestart = fLength;
				} else {
					fSegmentRestart++;
				}
				fSegmentEnd = CharOperation.indexOf(pathSeparator, filepath,
						fSegmentStart = fSegmentRestart);
				if (fSegmentEnd < 0)
					fSegmentEnd = fLength;
				continue checkSegment;
			}

			/* path segment is ending */
			if (pSegmentEnd == pSegmentStart + 2
					&& pattern[pSegmentStart] == '*'
					&& pattern[pSegmentStart + 1] == '*') {
				pSegmentEnd = CharOperation.indexOf(pathSeparator, pattern,
						pSegmentStart = pSegmentEnd + 1);
				// skip separator
				if (pSegmentEnd < 0)
					pSegmentEnd = pLength;
				pSegmentRestart = pSegmentStart;
				fSegmentRestart = fSegmentStart;
				if (pSegmentStart >= pLength)
					return true;
				continue checkSegment;
			}
			/* chech current path segment */
			if (!CharOperation.match(pattern, pSegmentStart, pSegmentEnd,
					filepath, fSegmentStart, fSegmentEnd, isCaseSensitive)) {
				// mismatch - restart current path segment
				pSegmentEnd = CharOperation.indexOf(pathSeparator, pattern,
						pSegmentStart = pSegmentRestart);
				if (pSegmentEnd < 0)
					pSegmentEnd = pLength;

				fSegmentRestart = CharOperation.indexOf(pathSeparator,
						filepath, fSegmentRestart + 1);
				// skip separator
				if (fSegmentRestart < 0) {
					fSegmentRestart = fLength;
				} else {
					fSegmentRestart++;
				}
				fSegmentEnd = CharOperation.indexOf(pathSeparator, filepath,
						fSegmentStart = fSegmentRestart);
				if (fSegmentEnd < 0)
					fSegmentEnd = fLength;
				continue checkSegment;
			}
			// jump to next segment
			pSegmentEnd = CharOperation.indexOf(pathSeparator, pattern,
					pSegmentStart = pSegmentEnd + 1);
			// skip separator
			if (pSegmentEnd < 0)
				pSegmentEnd = pLength;

			fSegmentEnd = CharOperation.indexOf(pathSeparator, filepath,
					fSegmentStart = fSegmentEnd + 1);
			// skip separator
			if (fSegmentEnd < 0)
				fSegmentEnd = fLength;
		}

		return (pSegmentRestart >= pSegmentEnd)
				|| (fSegmentStart >= fLength && pSegmentStart >= pLength)
				|| (pSegmentStart == pLength - 2
						&& pattern[pSegmentStart] == '*' && pattern[pSegmentStart + 1] == '*')
				|| (pSegmentStart == pLength && freeTrailingDoubleStar);
	}

	/**
	 * Answers true if the a sub-pattern matches the subpart of the given name,
	 * false otherwise. char[] pattern matching, accepting wild-cards '*' and
	 * '?'. Can match only subset of name/pattern. end positions are
	 * non-inclusive. The subpattern is defined by the patternStart and
	 * pattternEnd positions. When not case sensitive, the pattern is assumed to
	 * already be lowercased, the name will be lowercased character per
	 * character as comparing. <br>
	 * <br>
	 * For example:
	 * <ol>
	 * <li>
	 * 
	 * <pre>
	 *  
	 *      pattern = { '?', 'b', '*' }
	 *      patternStart = 1
	 *      patternEnd = 3
	 *      name = { 'a', 'b', 'c' , 'd' }
	 *      nameStart = 1
	 *      nameEnd = 4
	 *      isCaseSensitive = true
	 *      result =&gt; true
	 *   
	 * </pre>
	 * 
	 * </li>
	 * <li>
	 * 
	 * <pre>
	 *  
	 *      pattern = { '?', 'b', '*' }
	 *      patternStart = 1
	 *      patternEnd = 2
	 *      name = { 'a', 'b', 'c' , 'd' }
	 *      nameStart = 1
	 *      nameEnd = 2
	 *      isCaseSensitive = true
	 *      result =&gt; false
	 *   
	 * </pre>
	 * 
	 * </li>
	 * </ol>
	 * 
	 * @param pattern
	 *            the given pattern
	 * @param patternStart
	 *            the given pattern start
	 * @param patternEnd
	 *            the given pattern end
	 * @param name
	 *            the given name
	 * @param nameStart
	 *            the given name start
	 * @param nameEnd
	 *            the given name end
	 * @param isCaseSensitive
	 *            flag to know if the matching should be case sensitive
	 * @return true if the a sub-pattern matches the subpart of the given name,
	 *         false otherwise
	 */
	public static final boolean match(char[] pattern, int patternStart,
			int patternEnd, char[] name, int nameStart, int nameEnd,
			boolean isCaseSensitive) {

		if (name == null)
			return false; // null name cannot match
		if (pattern == null)
			return true; // null pattern is equivalent to '*'
		int iPattern = patternStart;
		int iName = nameStart;

		if (patternEnd < 0)
			patternEnd = pattern.length;
		if (nameEnd < 0)
			nameEnd = name.length;

		/* check first segment */
		char patternChar = 0;
		while ((iPattern < patternEnd)
				&& (patternChar = pattern[iPattern]) != '*') {
			if (iName == nameEnd)
				return false;
			if (patternChar != (isCaseSensitive ? name[iName] : Character
					.toLowerCase(name[iName]))
					&& patternChar != '?') {
				return false;
			}
			iName++;
			iPattern++;
		}
		/* check sequence of star+segment */
		int segmentStart;
		if (patternChar == '*') {
			segmentStart = ++iPattern; // skip star
		} else {
			segmentStart = 0; // force iName check
		}
		int prefixStart = iName;
		checkSegment: while (iName < nameEnd) {
			if (iPattern == patternEnd) {
				iPattern = segmentStart; // mismatch - restart current
				// segment
				iName = ++prefixStart;
				continue checkSegment;
			}
			/* segment is ending */
			if ((patternChar = pattern[iPattern]) == '*') {
				segmentStart = ++iPattern; // skip start
				if (segmentStart == patternEnd) {
					return true;
				}
				prefixStart = iName;
				continue checkSegment;
			}
			/* check current name character */
			if ((isCaseSensitive ? name[iName] : Character
					.toLowerCase(name[iName])) != patternChar
					&& patternChar != '?') {
				iPattern = segmentStart; // mismatch - restart current
				// segment
				iName = ++prefixStart;
				continue checkSegment;
			}
			iName++;
			iPattern++;
		}

		return (segmentStart == patternEnd)
				|| (iName == nameEnd && iPattern == patternEnd)
				|| (iPattern == patternEnd - 1 && pattern[iPattern] == '*');
	}

	/**
	 * Answers a new array which is a copy of the given array starting at the
	 * given start and ending at the given end. The given start is inclusive and
	 * the given end is exclusive. Answers null if start is greater than end, if
	 * start is lower than 0 or if end is greater than the length of the given
	 * array. If end equals -1, it is converted to the array length. <br>
	 * <br>
	 * For example:
	 * <ol>
	 * <li>
	 * 
	 * <pre>
	 *     array = { 'a' , 'b' }
	 *     start = 0
	 *     end = 1
	 *     result =&gt; { 'a' }
	 * </pre>
	 * 
	 * </li>
	 * <li>
	 * 
	 * <pre>
	 *     array = { 'a', 'b' }
	 *     start = 0
	 *     end = -1
	 *     result =&gt; { 'a' , 'b' }
	 * </pre>
	 * 
	 * </li>
	 * </ol>
	 * 
	 * @param array
	 *            the given array
	 * @param start
	 *            the given starting index
	 * @param end
	 *            the given ending index
	 * @return a new array which is a copy of the given array starting at the
	 *         given start and ending at the given end
	 * @throws NullPointerException
	 *             if the given array is null
	 */
	public static final char[] subarray(char[] array, int start, int end) {
		if (end == -1)
			end = array.length;
		if (start > end)
			return null;
		if (start < 0)
			return null;
		if (end > array.length)
			return null;

		char[] result = new char[end - start];
		System.arraycopy(array, start, result, 0, end - start);
		return result;
	}

	/**
	 * Answers the concatenation of the two arrays inserting the separator
	 * character between the two arrays. It answers null if the two arrays are
	 * null. If the first array is null, then the second array is returned. If
	 * the second array is null, then the first array is returned. <br>
	 * <br>
	 * For example:
	 * <ol>
	 * <li>
	 * 
	 * <pre>
	 *     first = null
	 *     second = { 'a' }
	 *     separator = '/'
	 *     =&gt; result = { ' a' }
	 * </pre>
	 * 
	 * </li>
	 * <li>
	 * 
	 * <pre>
	 *     first = { ' a' }
	 *     second = null
	 *     separator = '/'
	 *     =&gt; result = { ' a' }
	 * </pre>
	 * 
	 * </li>
	 * <li>
	 * 
	 * <pre>
	 *     first = { ' a' }
	 *     second = { ' b' }
	 *     separator = '/'
	 *     =&gt; result = { ' a' , '/', 'b' }
	 * </pre>
	 * 
	 * </li>
	 * </ol>
	 * 
	 * @param first
	 *            the first array to concatenate
	 * @param second
	 *            the second array to concatenate
	 * @param separator
	 *            the character to insert
	 * @return the concatenation of the two arrays inserting the separator
	 *         character between the two arrays , or null if the two arrays are
	 *         null.
	 */
	public static final char[] concat(char[] first, char[] second,
			char separator) {
		if (first == null)
			return second;
		if (second == null)
			return first;

		int length1 = first.length;
		if (length1 == 0)
			return second;
		int length2 = second.length;
		if (length2 == 0)
			return first;

		char[] result = new char[length1 + length2 + 1];
		System.arraycopy(first, 0, result, 0, length1);
		result[length1] = separator;
		System.arraycopy(second, 0, result, length1 + 1, length2);
		return result;
	}

	/**
	 * Answers the last index in the array for which the corresponding character
	 * is equal to toBeFound starting from the end of the array. Answers -1 if
	 * no occurrence of this character is found. <br>
	 * <br>
	 * For example:
	 * <ol>
	 * <li>
	 * 
	 * <pre>
	 *     toBeFound = 'c'
	 *     array = { ' a', 'b', 'c', 'd' , 'c', 'e' }
	 *     result =&gt; 4
	 * </pre>
	 * 
	 * </li>
	 * <li>
	 * 
	 * <pre>
	 *     toBeFound = 'e'
	 *     array = { ' a', 'b', 'c', 'd' }
	 *     result =&gt; -1
	 * </pre>
	 * 
	 * </li>
	 * </ol>
	 * 
	 * @param toBeFound
	 *            the character to search
	 * @param array
	 *            the array to be searched
	 * @return the last index in the array for which the corresponding character
	 *         is equal to toBeFound starting from the end of the array, -1
	 *         otherwise
	 * @throws NullPointerException
	 *             if array is null
	 */
	public static final int lastIndexOf(char toBeFound, char[] array) {
		for (int i = array.length; --i >= 0;)
			if (toBeFound == array[i])
				return i;
		return -1;
	}

	/**
	 * Answers true if the two arrays are identical character by character,
	 * otherwise false. The equality is case sensitive. <br>
	 * <br>
	 * For example:
	 * <ol>
	 * <li>
	 * 
	 * <pre>
	 *     first = null
	 *     second = null
	 *     result =&gt; true
	 * </pre>
	 * 
	 * </li>
	 * <li>
	 * 
	 * <pre>
	 *     first = { { } }
	 *     second = null
	 *     result =&gt; false
	 * </pre>
	 * 
	 * </li>
	 * <li>
	 * 
	 * <pre>
	 *     first = { { 'a' } }
	 *     second = { { 'a' } }
	 *     result =&gt; true
	 * </pre>
	 * 
	 * </li>
	 * <li>
	 * 
	 * <pre>
	 *     first = { { 'A' } }
	 *     second = { { 'a' } }
	 *     result =&gt; false
	 * </pre>
	 * 
	 * </li>
	 * </ol>
	 * 
	 * @param first
	 *            the first array
	 * @param second
	 *            the second array
	 * @return true if the two arrays are identical character by character,
	 *         otherwise false
	 */
	public static final boolean equals(char[][] first, char[][] second) {
		if (first == second)
			return true;
		if (first == null || second == null)
			return false;
		if (first.length != second.length)
			return false;

		for (int i = first.length; --i >= 0;)
			if (!equals(first[i], second[i]))
				return false;
		return true;
	}

	public static final boolean equals(String[] first, String[] second) {
		if (first == second)
			return true;
		if (first == null || second == null)
			return false;
		if (first.length != second.length)
			return false;

		for (int i = first.length; --i >= 0;)
			if (!first[i].equals(second[i]))
				return false;
		return true;
	}

	/**
	 * Answers a hashcode for the array
	 * 
	 * @param array
	 *            the array for which a hashcode is required
	 * @return the hashcode
	 * @throws NullPointerException
	 *             if array is null
	 */
	public static final int hashCode(char[] array) {
		int length = array.length;
		int hash = length == 0 ? 31 : array[0];
		if (length < 8) {
			for (int i = length; --i > 0;)
				hash = (hash * 31) + array[i];
		} else {
			// 8 characters is enough to compute a decent hash code, don't waste
			// time examining every character
			for (int i = length - 1, last = i > 16 ? i - 16 : 0; i > last; i -= 2)
				hash = (hash * 31) + array[i];
		}
		return hash & 0x7FFFFFFF;
	}
}
