/**
 * <copyright>
 *
 * Copyright (c) 2003-2004 IBM Corporation and others.
 * All rights reserved.   This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *   IBM - Initial API and implementation
 *
 * </copyright>
 *
 * $Id: DataGraphResourceFactoryImpl.java,v 1.15 2006/07/09 13:39:32 emerks Exp $
 */
package org.eclipse.emf.ecore.sdo.util;


import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.eclipse.emf.common.util.EMap;
import org.eclipse.emf.common.util.URI;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EClassifier;
import org.eclipse.emf.ecore.EFactory;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EPackage;
import org.eclipse.emf.ecore.EReference;
import org.eclipse.emf.ecore.EStructuralFeature;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.impl.ResourceFactoryImpl;
import org.eclipse.emf.ecore.sdo.EChangeSummary;
import org.eclipse.emf.ecore.sdo.EDataGraph;
import org.eclipse.emf.ecore.sdo.SDOFactory;
import org.eclipse.emf.ecore.sdo.SDOPackage;
import org.eclipse.emf.ecore.sdo.impl.DynamicEDataObjectImpl;
import org.eclipse.emf.ecore.util.EcoreUtil;
import org.eclipse.emf.ecore.util.ExtendedMetaData;
import org.eclipse.emf.ecore.xmi.EcoreBuilder;
import org.eclipse.emf.ecore.xmi.NameInfo;
import org.eclipse.emf.ecore.xmi.XMLHelper;
import org.eclipse.emf.ecore.xmi.XMLLoad;
import org.eclipse.emf.ecore.xmi.XMLResource;
import org.eclipse.emf.ecore.xmi.XMLSave;
import org.eclipse.emf.ecore.xmi.impl.SAXXMLHandler;
import org.eclipse.emf.ecore.xmi.impl.XMLHelperImpl;
import org.eclipse.emf.ecore.xmi.impl.XMLLoadImpl;
import org.eclipse.emf.ecore.xmi.impl.XMLResourceImpl;
import org.eclipse.emf.ecore.xmi.impl.XMLSaveImpl;
import org.eclipse.emf.ecore.xmi.util.DefaultEcoreBuilder;
import org.w3c.dom.Element;
import org.xml.sax.helpers.DefaultHandler;


public class DataGraphResourceFactoryImpl extends ResourceFactoryImpl
{
  /**
   * Constructor for DataGraphResourceFactoryImpl.
   */
  public DataGraphResourceFactoryImpl()
  {
    super();
  }

  public Resource createResource(URI uri)
  {
    XMLResourceImpl result = new DataGraphResourceImpl(uri);

    result.setEncoding("UTF-8");

    result.getDefaultLoadOptions().put(XMLResource.OPTION_USE_LEXICAL_HANDLER, Boolean.TRUE);

    result.getDefaultLoadOptions().put(XMLResource.OPTION_EXTENDED_META_DATA, Boolean.TRUE);
    result.getDefaultSaveOptions().put(XMLResource.OPTION_EXTENDED_META_DATA, Boolean.TRUE);

    result.getDefaultSaveOptions().put(XMLResource.OPTION_USE_ENCODED_ATTRIBUTE_STYLE, Boolean.TRUE);
    result.getDefaultSaveOptions().put(XMLResource.OPTION_LINE_WIDTH, new Integer(80));

    result.getDefaultLoadOptions().put(XMLResource.OPTION_ANY_TYPE, SDOPackage.eINSTANCE.getEDataObjectAnyType());
    result.getDefaultSaveOptions().put(XMLResource.OPTION_ANY_TYPE, SDOPackage.eINSTANCE.getEDataObjectAnyType());

    result.getDefaultLoadOptions().put(XMLResource.OPTION_ANY_SIMPLE_TYPE, SDOPackage.eINSTANCE.getEDataObjectSimpleAnyType());
    result.getDefaultSaveOptions().put(XMLResource.OPTION_ANY_SIMPLE_TYPE, SDOPackage.eINSTANCE.getEDataObjectSimpleAnyType());

    return result;
  }

  public static class DataGraphResourceImpl extends XMLResourceImpl
  {
    public DataGraphResourceImpl(URI uri)
    {
      super(uri);
    }

    public static class HelperImpl extends XMLHelperImpl
    {
      protected EDataGraph eDataGraph;

      protected List resources;
      protected List uris;
      
      public HelperImpl(XMLResource xmlResource)
      {
        super(xmlResource);
      }
      
      public void setResource(XMLResource resource)
      {
        super.setResource(resource);
        if (!resource.getContents().isEmpty())
        {
          eDataGraph = (EDataGraph)resource.getContents().get(0);

          resources = new ArrayList();
          uris = new ArrayList();

          resources.add(eDataGraph.getRootResource());
          uris.add("#" + resource.getURIFragment(eDataGraph) + "/@eRootObject");

          if (eDataGraph.getEChangeSummary() != null)
          {
            // Ensure that resource exists.
            //
            resources.add(((EObject)eDataGraph.getChangeSummary()).eResource());
            uris.add("#" + resource.getURIFragment(eDataGraph) + "/@eChangeSummary");
          }

          if (eDataGraph.eResource() != null && eDataGraph.eResource().getResourceSet() != null)
          {
            int count = 0;
            for (Iterator i = eDataGraph.eResource().getResourceSet().getResources().iterator(); i.hasNext();)
            {
              Resource ePackageResource = (Resource)i.next();
              List resourceContents = ePackageResource.getContents();
              if (resourceContents.size() == 1 && resourceContents.get(0) instanceof EPackage)
              {
                resources.add(ePackageResource);
                uris.add("#" + resource.getURIFragment(eDataGraph) + "/@models." + count++);
              }
            }
          }
        }
      }

      public String getID(EObject eObject)
      {
        return super.getID(eObject);
      }

      public String getIDREF(EObject eObject)
      {
        String fragment = super.getIDREF(eObject);
        if (fragment.startsWith("/"))
        {
          int index = resources.indexOf(eObject.eResource());
          if (index != -1)
          {
            fragment = ((String)uris.get(index)).substring(1) + fragment.substring(1);
          }
        }
        return fragment;
      }

      public String getHREF(EObject eObject)
      {
        return super.getHREF(eObject);
      }

      protected URI getHREF(Resource otherResource, EObject obj)
      {
        int index = resources.indexOf(otherResource);
        if (index == -1)
        {
          return super.getHREF(otherResource, obj);
        }
        else
        {
          return createHREF((String)uris.get(index), otherResource.getURIFragment(obj));
        }
      }

      protected URI createHREF(String baseURI, String fragment)
      {
        if (fragment.startsWith("/"))
        {
          return URI.createURI(baseURI + fragment.substring(1));
        }
        else
        {
          return URI.createURI("#" + fragment);
        }
      }
           
      public EClassifier getType(EFactory eFactory, String typeName)
      {
        if (eFactory == SDOFactory.eINSTANCE && "datagraph".equals(typeName))              
        {
          return super.getType(eFactory, "EDataGraph");
        }
        return super.getType(eFactory, typeName);
      }
      
      public void populateNameInfo(NameInfo nameInfo, EClass c)
      {
        if (c == SDOPackage.eINSTANCE.getEDataGraph())
        {
          if (extendedMetaData != null)
          {
            extendedMetaData.demandPackage("commonj.sdo").setNsPrefix("sdo");
          }
          nameInfo.setQualifiedName(getQName("commonj.sdo", "datagraph"));
          nameInfo.setNamespaceURI("commonj.sdo");
          nameInfo.setLocalPart("datagraph");
        }
        else if (c == SDOPackage.eINSTANCE.getEChangeSummary())
        {
          if (extendedMetaData != null)
          {
            extendedMetaData.demandPackage("commonj.sdo").setNsPrefix("sdo");
          }
          nameInfo.setQualifiedName("changeSummary");
          nameInfo.setNamespaceURI(null);
          nameInfo.setLocalPart("changeSummary");
        }
        else
        {
          super.populateNameInfo(nameInfo, c);
        }
      }

      public String getQName(EClass c)
      {
        if (c == SDOPackage.eINSTANCE.getEDataGraph())
        {
          if (extendedMetaData != null)
          {
            extendedMetaData.demandPackage("commonj.sdo").setNsPrefix("sdo");
          }
          return getQName("commonj.sdo", "datagraph");
        }
        else if (c == SDOPackage.eINSTANCE.getEChangeSummary())
        {
          if (extendedMetaData != null)
          {
            extendedMetaData.demandPackage("commonj.sdo").setNsPrefix("sdo");
          }
          return getQName((String)null, "changeSummary");
        }
        else
        {
          return super.getQName(c);
        }
      }
    }

    protected XMLHelper createXMLHelper()
    {
      return new HelperImpl(this);
    }

    protected EObject getEObjectByID(String id)
    {
      List contents = getContents();
      if (contents.size() >= 1)
      {
        Object rootObject = contents.get(0);
        if (rootObject instanceof EDataGraph)
        {
          EDataGraph eDataGraph = (EDataGraph)rootObject;
          EObject result = eDataGraph.getRootResource().getEObject(id);
          if (result != null)
          {
            return result;
          }
          else
          {
            EChangeSummary eChangeSummary = eDataGraph.getEChangeSummary();
            if (eChangeSummary != null)
            {
              result = ((EObject)eDataGraph.getChangeSummary()).eResource().getEObject(id);
              if (result != null)
              {
                return result;
              }
            }
          }
        }
      }
      return super.getEObjectByID(id);
    }

    public static class SaveImpl extends XMLSaveImpl
    {
      protected EDataGraph eDataGraph;

      public SaveImpl(XMLHelper xmlHelper)
      {
        super(xmlHelper);
      }

      public void traverse(List contents)
      {
        if (contents.size() >= 1 && contents.get(0) instanceof EDataGraph)
        {
          eDataGraph = (EDataGraph)contents.get(0);
                   
          Object datagraphMark = null;
          if (!toDOM)
          {
            if (declareXML)
            {
              doc.add("<?xml version=\"" + xmlVersion + "\" encoding=\"" + encoding + "\"?>");
              doc.addLine();
            }
            String elementName = helper.getQName(eDataGraph.eClass());
            doc.startElement(elementName);
            datagraphMark = doc.mark();
          }
          else
          {
            helper.populateNameInfo(nameInfo, eDataGraph.eClass());
            currentNode = document.createElementNS(nameInfo.getNamespaceURI(), nameInfo.getQualifiedName());
            document.appendChild(currentNode);
            // not calling handler since there is no corresponding EObject
          }

          if (eDataGraph.eResource() != null && eDataGraph.eResource().getResourceSet() != null)
          {
            List ePackages = new ArrayList();
            for (Iterator i = eDataGraph.eResource().getResourceSet().getResources().iterator(); i.hasNext();)
            {
              List resourceContents = ((Resource)i.next()).getContents();
              if (resourceContents.size() == 1 && resourceContents.get(0) instanceof EPackage)
              {
                ePackages.add(resourceContents.get(0));
              }
            }
            if (!ePackages.isEmpty())
            {
              if (!toDOM)
              {
                doc.startElement("models");
                doc.addAttribute("xmlns", "");
              }
              else
              {
                currentNode = currentNode.appendChild(document.createElementNS(null, "models"));
                ((Element)currentNode).setAttributeNS(ExtendedMetaData.XMLNS_URI, ExtendedMetaData.XMLNS_PREFIX, "");
                //  not calling handler since there is no corresponding EObject
              }
              for (Iterator i = ePackages.iterator(); i.hasNext();)
              {
                writeTopObject((EPackage)i.next());
              }
              if (!toDOM)
              {
                doc.endElement();
              }
              else
              {
                currentNode = currentNode.getParentNode();
              }
            }
          }

          // use namespace declarations defined in the document (if any)
          EObject eRootObject = eDataGraph.getERootObject();
          EReference xmlnsPrefixMapFeature = extendedMetaData.getXMLNSPrefixMapFeature(eRootObject.eClass());
          if (xmlnsPrefixMapFeature != null)
          {
            EMap xmlnsPrefixMap = (EMap)eRootObject.eGet(xmlnsPrefixMapFeature);
            helper.setPrefixToNamespaceMap(xmlnsPrefixMap);
          }
          EChangeSummary changeSummary = eDataGraph.getEChangeSummary();

          if (changeSummary != null)
          {
            helper.setMustHavePrefix(true);
            if (changeSummary.isLogging())
            {
              changeSummary.summarize();
              writeTopObject(changeSummary);
            }
            else
            {
              writeTopObject(changeSummary);
            }
            helper.setMustHavePrefix(false);
          }

          if (eRootObject != null && writeTopObject(eRootObject) == null && !toDOM)
          {
            doc.addLine();
            doc.setMixed(false);
          }
          if (!toDOM)
          {
            doc.endElement();
            // reset to add namespace declarations
            //
            doc.resetToMark(datagraphMark);
          }
          else
          {
            currentNode = document.getFirstChild();
          }
          addNamespaceDeclarations();
        }
        else
        {
          super.traverse(contents);
        }
      }

      protected void writeTopAttributes(EObject top)
      {
        if (top == eDataGraph.getEChangeSummary())
        {
          if (!toDOM)
          {
            doc.addAttribute("xmlns", "");
          }
          else
          {
            ((Element)currentNode).setAttributeNS(ExtendedMetaData.XMLNS_URI, ExtendedMetaData.XMLNS_PREFIX, "");
          }
        }
      }

      protected EObject getSchemaLocationRoot(EObject eObject)
      {
        return eDataGraph.getERootObject();
      }
    }

    protected XMLSave createXMLSave()
    {
      return new SaveImpl(createXMLHelper());
    }

    public static class LoadImpl extends XMLLoadImpl
    {
      public LoadImpl(XMLHelper xmlHelper)
      {
        super(xmlHelper);
      }

      protected DefaultHandler makeDefaultHandler()
      {
        return new SAXXMLHandler(resource, helper, options)
          {
            protected EDataGraph eDataGraph;

            protected boolean isInModels;

            protected List ePackages = new ArrayList();

            protected EMap recordNamespacesSchemaLocations(EObject root)
            {
              EObject dgroot = eDataGraph.getERootObject();
              if (dgroot == null)
              {
                return null;
              }
              EMap prefixToNamespaceMap = super.recordNamespacesSchemaLocations(dgroot);
              if (prefixToNamespaceMap != null)
              {
                for (Iterator i = prefixToNamespaceMap.iterator(); i.hasNext();)
                {
                  Map.Entry entry = (Map.Entry)i.next();
                  String prefix = (String)entry.getKey();
                  String namespace = (String)entry.getValue();
                  if (namespace.equals("commonj.sdo"))
                  {
                    prefixToNamespaceMap.removeKey(prefix);
                    break;
                  }
                }
              }
              return prefixToNamespaceMap;
            }

            protected void handleFeature(String prefix, String name)
            {
              if (isInModels && objects.size() == 2)
              {
                EObject modelObject = createObjectByType(prefix, name, false);
                processObject(modelObject);
                ePackages.add(modelObject);
              }
              else if (objects.size() == 1)
              {
                eDataGraph = (EDataGraph)objects.peek();
                eDataGraph.getResourceSet();
                if ("".equals(prefix) && "changeSummary".equals(name))
                {
                  EChangeSummary eChangeSummary = (EChangeSummary)createObjectFromFactory(SDOFactory.eINSTANCE, "EChangeSummary");
                  eDataGraph.setEChangeSummary(eChangeSummary);
                  processObject(eChangeSummary);
                }
                else if ("".equals(prefix) && "models".equals(name))
                {
                  isInModels = true;
                  types.push(OBJECT_TYPE);
                  objects.push(eDataGraph);
                  mixedTargets.push(null);
                }
                else if (eDataGraph.getERootObject() == null)
                {
                  if (useNewMethods)
                  {
                    handleSchemaLocation();
                  }
                  processSchemaLocations(prefix, name);
                  if (processAnyXML)
                  {
                    // Ensure that anything can be handled, even if it's not recognized.
                    //
                    String namespaceURI = helper.getURI(prefix);
                    if (extendedMetaData.getPackage(namespaceURI) == null)
                    {
                      EStructuralFeature rootFeature = extendedMetaData.demandFeature(namespaceURI, name, true);
                      rootFeature.getEContainingClass().getEPackage().setEFactoryInstance(new DynamicEDataObjectImpl.FactoryImpl());
                    }
                  }

                  EObject rootObject = createObjectByType(prefix, name, false);
                  eDataGraph.setERootObject(rootObject);
                  processObject(rootObject);
                  if (rootObject != null
                    && rootObject.eClass() == ExtendedMetaData.INSTANCE.getDocumentRoot(rootObject.eClass().getEPackage()))
                  {
                    super.handleFeature(prefix, name);

                    // Remove the document root from the stack.
                    //
                    Object mixedTarget = mixedTargets.pop();
                    Object object = objects.pop();
                    mixedTargets.pop();
                    objects.pop();
                    mixedTargets.push(mixedTarget);
                    objects.push(object);
                  }
                }
              }
              else
              {
                super.handleFeature(prefix, name);
              }
            }

            public void endElement(String uri, String localName, String name)
            {
              if (isInModels && objects.size() == 2)
              {
                if (!ePackages.isEmpty())
                {
                  for (Iterator i = ePackages.iterator(); i.hasNext();)
                  {
                    EPackage ePackage = (EPackage)i.next();
                    ePackage.setEFactoryInstance(new DynamicEDataObjectImpl.FactoryImpl());
                    Resource resource = resourceSet.createResource(URI.createURI("*.ecore"));
                    resource.getContents().add(ePackage);
                    if (ePackage.getNsURI() != null)
                    {
                      resource.setURI(URI.createURI(ePackage.getNsURI()));
                    }

                    if (extendedMetaData != null)
                    {
                      extendedMetaData.putPackage(extendedMetaData.getNamespace(ePackage), ePackage);
                    }
                    else
                    {
                      packageRegistry.put(ePackage.getNsURI(), ePackage);
                    }
                  }
                  handleForwardReferences();
                }
                isInModels = false;
              }
              super.endElement(uri, localName, name);
            }

            protected EPackage getPackageForURI(String uriString)
            {
              if ("commonj.sdo".equals(uriString))
              {
                return SDOPackage.eINSTANCE;
              }
              else
              {
                return super.getPackageForURI(uriString);
              }
            }

            /**
             * @deprecated in 2.2.0
             */
            protected EObject createObjectFromFactory(EFactory factory, String typeName)
            {
              if (factory == SDOFactory.eINSTANCE)
              {
                if ("datagraph".equals(typeName))
                {
                  return super.createObjectFromFactory(factory, "EDataGraph");
                }
              }
              return super.createObjectFromFactory(factory, typeName);
            }
            
            protected void handleTopLocations(String prefix, String name)
            {
              // No need to process top locations since the top location should be the root element of datagraph
              // and it will be processed when we see root.
            }

            protected EcoreBuilder createEcoreBuilder(Map options, ExtendedMetaData extendedMetaData)
            {
              return new DefaultEcoreBuilder(extendedMetaData)
                {
                  public Collection generate(Map urisToLocations) throws Exception
                  {
                    Collection result = super.generate(urisToLocations);
                    return updateDynamicFactory(result);
                  }

                  public Collection generate(Collection urisToLocations) throws Exception
                  {
                    Collection result = super.generate(urisToLocations);
                    return updateDynamicFactory(result);
                  }

                  protected Collection updateDynamicFactory(Collection result)
                  {
                    for (Iterator i = result.iterator(); i.hasNext();)
                    {
                      Resource resource = (Resource)i.next();
                      for (Iterator j = EcoreUtil.getObjectsByType(resource.getContents(), EcorePackage.eINSTANCE.getEPackage()).iterator(); j.hasNext();)
                      {
                        EPackage ePackage = (EPackage)j.next();
                        ePackage.setEFactoryInstance(new DynamicEDataObjectImpl.FactoryImpl());
                      }
                    }
                    return result;
                  }

                };
            }

            protected EPackage handleMissingPackage(String uriString)
            {
              EPackage result = super.handleMissingPackage(uriString);
              if (processAnyXML && objects.size() == 1)
              {
                result = extendedMetaData.demandPackage(uriString);
              }
              return result;
            }
          };
      }
    }

    protected XMLLoad createXMLLoad()
    {
      return new LoadImpl(createXMLHelper());
    }
  }
}
