
/// @file Text.hpp Class to represent text. Class declarations.

//============================================================================
//
//  This file is part of GPSTk, the GPS Toolkit.
//
//  The GPSTk is free software; you can redistribute it and/or modify
//  it under the terms of the GNU Lesser General Public License as published
//  by the Free Software Foundation; either version 2.1 of the License, or
//  any later version.
//
//  The GPSTk is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Lesser General Public License for more details.
//
//  You should have received a copy of the GNU Lesser General Public
//  License along with GPSTk; if not, write to the Free Software Foundation,
//  Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//  
//  Copyright 2004, The University of Texas at Austin
//
//============================================================================


#ifndef VPLOT_TEXT_H
#define VPLOT_TEXT_H

#include <string>
#include <stdio.h>

#include "VPlotException.hpp"
#include "TextStyle.hpp"

namespace vplot
{
  /** \addtogroup BasicVectorGraphics */ 
  //@{


  /**
   * This class defines a text object that is to be displayed in an image.
   */
  class Text {

    public:

      /// X coordinate of the text.
      double x; 

      /// Y coordinate of the text.
      double y;

      /**
       * Enumeration of alignment constants
       */
      enum ALIGNMENT {LEFT, CENTER, RIGHT};

      /**
       * Constructs text at some coordinate.
       * @param str string to appear as text
       * @param ix x coordinate of the first point
       * @param iy y coordinate of the first point
       * @param align enumerated alignment constant, for which direction text 
       *   is aligned with respect to the (x,y) point
       * @param angle angle in degrees the text is to be pointing, starting at zero,
       *   counter-clockwise from ---> (3:00)
       */
      Text(const char* str, double ix, double iy, ALIGNMENT align=LEFT, int angle=0);

      /**
       * Constructs text at some coordinate.
       * @param str string to appear as text
       * @param ix x coordinate of the first point
       * @param iy y coordinate of the first point
       * @param its textStyle describing how this text is supposed to look
       * @param align enumerated alignment constant, for which direction text 
       *   is aligned with respect to the (x,y) point
       * @param angle angle in degrees the text is to be pointing, starting at zero,
       *   counter-clockwise from ---> (3:00)
       */
      Text(const char* str, double ix, double iy, TextStyle& its, ALIGNMENT align=LEFT, int angle=0);

      /**
       * Constructs text at some coordinate.
       * @param num number to be converted to text
       * @param ix x coordinate of the first point
       * @param iy y coordinate of the first point
       * @param align enumerated alignment constant, for which direction text 
       *   is aligned with respect to the (x,y) point
       * @param angle angle in degrees the text is to be pointing, starting at zero,
       *   counter-clockwise from ---> (3:00)
       */
      Text(int num, double ix, double iy, ALIGNMENT align=LEFT, int angle=0);

      /**
       * Constructs text at some coordinate.
       * @param num number to be converted to text
       * @param ix x coordinate of the first point
       * @param iy y coordinate of the first point
       * @param its textStyle describing how this text is supposed to look
       * @param align enumerated alignment constant, for which direction text 
       *   is aligned with respect to the (x,y) point
       * @param angle angle in degrees the text is to be pointing, starting at zero,
       *   counter-clockwise from ---> (3:00)
       */
      Text(int num, double ix, double iy, TextStyle& its, ALIGNMENT align=LEFT, int angle=0);


      /**
       * Modifier for the output string of the text.
       * @param str string to appear as text
       */
      void setText(const std::string& str);

      /**
       * Modifier for the style of the text.
       * @param its textStyle describing how this text is supposed to look
       */
      void setStyle(TextStyle& its);

      /**
       * Modifier for the position on the image of the text.
       * @param ix x coordinate of the first point
       * @param iy y coordinate of the first point
       */
      void setPosition(double ix, double iy);

      /**
       * Modifier for the angle of the text.
       * @param align new enumerated alignment constant, for which direction text 
       *   is aligned with respect to the (x,y) point
       */
      void setAlignment(ALIGNMENT align);

      /**
       * Modifier for the angle of the text.
       * @param angle angle in degrees the text is to be pointing, starting at zero,
       *   counter-clockwise from ---> (3:00)
       */
      void setAngle(int angle);

      /*
       * Accessors
       */

      /// Accessor to get the text string.
      std::string getString(void) const { return textString; }

      /// Accessor to see if text as its own style.
      bool hasOwnTextStyle(void) const { return hasOwnStyle; }

      /// Accessor to get the text style.
      TextStyle getStyle(void) const { return textStyle; }

      /// Accessor to see if text is centered.
      bool isCenter(void) const { return textAlign == CENTER; }

      /// Accessor to see if text is aligned anchored left.
      bool isLeft(void) const { return textAlign == LEFT; }

      /// Accessor to see if text is aligned anchored right.
      bool isRight(void) const { return textAlign == RIGHT; }

      /// Accessor to get the angle of the text.
      int getAngle(void) const { return textAngle; }

    protected:

    private:

      std::string textString;
      bool hasOwnStyle;
      TextStyle textStyle;
      ALIGNMENT textAlign;
      int textAngle;

  }; // class Text

  //@}

} // namespace vplot

#endif //VPLOT_TEXT_H

