# Copyright 2015 Google Inc. All Rights Reserved.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#    http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.

"""Fingerprinting code for the ASP.NET v5 runtime."""

import glob
import os
import textwrap

from googlecloudsdk.api_lib.app.ext_runtimes import fingerprinting
from googlecloudsdk.api_lib.app.images import config
from googlecloudsdk.core import log

# The name of the runtimes supported by this fingerprinter.
NAME = 'aspnet'
ALLOWED_RUNTIME_NAMES = ('custom', NAME)

KESTREL = 'Microsoft.AspNet.Server.Kestrel'
KESTREL_DLL = '%s.dll' % KESTREL
DEPENDENCIES = 'dependencies'
DNXCORE50 = 'dnxcore50'
DNX_RUNTIME_VERSION = '1.0.0-rc1-update1'
APP_ENTRY_POINT_NAME = 'app_engine_start'

# These paths are used to detect the runtime that was used when publishing the
# app, we do that by looking to see if the dll version for each runtime exists
# under the app's directory.
KESTREL_DNXCORE50_LIB = 'Microsoft.AspNet.Server.Kestrel/%s/lib/%s/%s' % (
    DNX_RUNTIME_VERSION, DNXCORE50, KESTREL_DLL)

# These versions of the dlls are the patched Kestrel versions, which are
# required because of a bad bug in RC1.
# TODO(user): Remove these once the patched version is no longer needed.
PATCHED_VERSION = '1.0.0-rc1-patched'
KESTREL_DNXCORE50_PATCHED_LIB = '%s/%s/lib/%s/%s' % (KESTREL, PATCHED_VERSION,
                                                     DNXCORE50, KESTREL_DLL)

# Specify the default app.yaml for the app, ignoring all of the local files as
# we're packaging everything into a Docker container anyway.
#   {runtime}, the runtime name to use.
ASPNET_APP_YAML = textwrap.dedent("""\
    runtime: {runtime}
    vm: true
    api_version: 1
    skip_files:
    - ^.*$
    """)
DOCKERIGNORE = textwrap.dedent("""\
    .dockerignore
    Dockerfile
    .git
    .hg
    .svn
    """)
DOCKERFILE = textwrap.dedent("""\
    # This Dockerfile describes the image used to run ASP.NET v5 apps that
    # target the dnxcore50 framework, which in Linux means using the .NET
    # Core runtime.
    FROM b.gcr.io/aspnet-docker/dotnet:1.0.0-rc1-update1

    # The app's published directory will be added to the app, the entry
    # point to the app is assumed to be the /app/app_engine_start script
    # which is typically generated by the tools, but can be a custom
    # script if required.
    ADD ./ /app

    # Ensure that he entry point is executable.
    RUN chmod +x /app/app_engine_start
    """)

# The entrypoint to generate for the app which looks for the package that
# represents the app, and starts the Kestrel server in that directory, listening
# to any IP address.
APP_ENTRY_POINT_CONTENT = textwrap.dedent("""\
    #! /usr/bin/env bash
    source /root/.dnx/dnvm/dnvm.sh
    dnvm use default
    cd /app/approot/packages/{name}
    cd $(ls -1 | head -n 1)/root
    dnx Microsoft.Dnx.ApplicationHost \
        Microsoft.AspNet.Hosting \
        --server Microsoft.AspNet.Server.Kestrel \
        --server.urls http://*:8080
    """)


class AspNETConfigurator(fingerprinting.Configurator):
  """Generates configuraton for an ASP.NET class."""

  def __init__(self, application, params):
    """Constructor.

    Args:
      application: (_Application) The application object.
      params: (fingerprinting.Params) Parameters passed through to the
        fingerprinters.
    """
    self.application = application
    self.params = params

  def GenerateConfigs(self):
    """Generate all config files for the module.

    Returns:
      (callable()) fingerprinting.Cleaner instance.
    """
    # Write the message to the user or to the log depending on whether
    # we're in "deploy" or not.
    if self.params.deploy:
      notify = log.info
    else:
      notify = log.status.Print

    cleaner = fingerprinting.Cleaner()
    generated_app_yaml = False

    # Generate app.yaml.
    if not self.params.appinfo:
      app_yaml = os.path.join(self.application.path, 'app.yaml')
      if not os.path.exists(app_yaml):
        notify('Saving [app.yaml] to [%s].' % self.application.path)
        runtime = 'custom' if self.params.custom else NAME
        with open(app_yaml, 'w') as f:
          f.write(ASPNET_APP_YAML.format(runtime=runtime))
        generated_app_yaml = True

    # Generate the necessary files to create the Docker image that will
    # ultimately run the app. If we're in "deploy" mode these files will be be
    # created and then deleted after the deployment is done. If created with the
    # --custom flag these files will be created and then left for the user to
    # customize the app's image.
    if self.params.custom or self.params.deploy:
      # Generate dockefile.
      dockerfile = os.path.join(self.application.path, config.DOCKERFILE)
      if not os.path.exists(dockerfile):
        notify('Saving [%s] to [%s].' %
               (config.DOCKERFILE, self.application.path))
        with open(dockerfile, 'w') as f:
          f.write(DOCKERFILE)
        cleaner.Add(dockerfile)

      # Generate .dockerignore.
      dockerignore = os.path.join(self.application.path, '.dockerignore')
      if not os.path.exists(dockerignore):
        notify('Saving [.dockerignore] to [%s].' % self.application.path)
        with open(dockerignore, 'w') as f:
          f.write(DOCKERIGNORE)
        cleaner.Add(dockerignore)

      # Generate the entry point if not already present in the target.
      app_entry_point_path = os.path.join(self.application.path,
                                          APP_ENTRY_POINT_NAME)
      if not os.path.exists(app_entry_point_path):
        notify('Creating [%s] to [%s].' %
               (APP_ENTRY_POINT_NAME, self.application.path))
        with open(app_entry_point_path, 'w') as f:
          f.write(self.application.GenerateEntryPoint())
        cleaner.Add(app_entry_point_path)

    if not generated_app_yaml and not cleaner.HasFiles():
      notify('All config files already exist, not generating anything.')

    return cleaner


class _Application(object):
  """Contains the information about an ASP.NET app."""

  def __init__(self, path, project_path):
    """Constructor.

    Args:
      path: (str) The root path of the application.
      project_path: (str) The path for the startup project.
    """
    self.path = path
    self.project_path = project_path

  def GetProjectName(self):
    """Returns the project name given the path to the project.json."""
    path = self.project_path
    # A project path for a package published app is of the form:
    #   approot/packages/<name>/<version>/root/project.json.
    parts = path.split(os.path.sep)
    if len(parts) < 4:
      return None
    return parts[-4]

  def GenerateEntryPoint(self):
    """Generates the entry point for the packaged published app."""
    name = self.GetProjectName()
    if not name:
      log.debug('ASP.NET checker: Invalid project path: %s' % self.project_path)
      return None
    return APP_ENTRY_POINT_CONTENT.format(name=name)


def _DetectAspNetApp(path):
  """Checks if the app was released as a packaged app.

  Args:
    path: (str) The path to the root of the app.

  Returns:
    (_Application or None) Returns the app data if found None otherwise.
  """
  app_packages = os.path.join(path, 'approot', 'packages')
  if not os.path.isdir(app_packages):
    log.debug('ASP.NET Checker: No app found.')
    return None

  projects = glob.glob(os.path.join(app_packages, '*', '*', 'root',
                                    'project.json'))
  if not projects:
    log.debug('ASP.NET checker: No projects found.')
    return None
  if len(projects) > 1:
    log.debug('ASP.NET checker: More than one startup project found.')
    return None

  project_path = projects[0]
  log.debug('ASP.NET checker: Found entry point project %s' % project_path)

  if not _IsCoreClr(path):
    log.debug('ASP.NET checker: Runtime not supported.')
    return None
  return _Application(path, project_path)


def Fingerprint(path, params):
  """Check for an ASP.NET app.

  Args:
    path: (str) Publihed application path.
    params: (finterprinting.Params) Parameters passed through to the
      fingerprinters.

  Returns:
    (AspNETConfigurator or None) Returns a module if the path contains
    an ASP.NET app.
  """
  log.info('Checking for ASP.NET')
  application = _DetectAspNetApp(path)
  if not application:
    log.debug('ASP.NET Checker: No application was found.')
    return None

  return AspNETConfigurator(application, params)


def _IsCoreClr(path):
  """Determines if the app was published for CoreCLR.

  Args:
    path: (str) Path to the root of the app.

  Returns:
    (bool) Whether the app was published for CoreCLR.
  """
  approot_packages = os.path.join(path, 'approot', 'packages')

  # TODO(user): Remove the _patched_ version once that is no longer necessary.
  kestrel_dnxcore50_path = os.path.join(approot_packages, KESTREL_DNXCORE50_LIB)
  kestrel_dnxcore50_patched_path = os.path.join(approot_packages,
                                                KESTREL_DNXCORE50_PATCHED_LIB)
  return os.path.exists(kestrel_dnxcore50_path) or os.path.exists(
      kestrel_dnxcore50_patched_path)
