#!/usr/bin/env ruby

$:.unshift(File.join(File.dirname(__FILE__), '..', '..', '..'))

require 'rex/test'
require 'rex/socket'
require 'rex/proto/dcerpc/client'
require 'rex/proto/dcerpc/handle'

begin
require 'flexmock'

class Rex::Proto::DCERPC::Client::UnitTest < Test::Unit::TestCase

	Klass = Rex::Proto::DCERPC::Client

	def setup
		@handle = Rex::Proto::DCERPC::Handle.new(['6bffd098-a112-3610-9833-46c3f87e345a', '1.0'], 'ncacn_ip_tcp', '1.2.3.4', [1026])
	end

	def setup_test(read, write, force_mock = false)
		srand(0)
		mock_unmake() # always do this, just to be sure.  yes, it slows it down.  but...

		if (force_mock or !$_REX_TEST_NO_MOCK)
			mock_make(read,write)
		end
	end

	def mock_make(read, write)
		func = '
			require "flexmock"
			def self.create_tcp(opts = {})
				read = ' + read.inspect + '
				write = ' + write.inspect + '
				write_size = ' + write.size.to_s + '
				mock = FlexMock.new
				mock.should_receive(:type?).returns("tcp")
				mock.should_receive(:peerport).returns(opts["PeerPort"])
				mock.mock_handle(:close) { nil }
				mock.mock_handle(:read) { read.shift }
				mock.mock_handle(:get_once) { read.shift }
				mock.mock_handle(:write) { |data|
					expected = write.shift
					if data != expected
						write_id = write_size - write.size
						require "pp"
						raise "write #{write_id} failed\n\n<#{expected.inspect}> was expected but was\n<#{data.inspect}>"
					end
					data.length
				}
				mock.mock_handle(:put) { |data|
					expected = write.shift
					if data != expected
						write_id = write_size - write.size
						require "pp"
						raise "write #{write_id} failed\n\n<#{expected.inspect}> was expected but was\n<#{data.inspect}>"
					end
					data.length
				}
				# mock.should_ignore_missing()
				return mock
			end
			'
		Rex::Socket.module_eval(func)
	end

	def mock_unmake
		if Rex::Socket.create_tcp.class != 'Socket'
			func = '
				def self.create_tcp(opts = {})
					return create_param(Rex::Socket::Parameters.from_hash(opts.merge("Proto" => "tcp")))
				end
			'
			Rex::Socket.module_eval(func)
		end
	end

	# undo the mock object stuff, just in case...
	def teardown
		mock_unmake()
	end

	def test_instance_no_setup
		setup_test([],[])
		handle = Rex::Proto::DCERPC::Handle.new(['6bffd098-a112-3610-9833-46c3f87e345a', '1.0'], 'ncacn_ip_tcp', $_REX_TEST_SMB_HOST, [1026])
		s = Rex::Socket.create_tcp( 'PeerHost' => $_REX_TEST_SMB_HOST, 'PeerPort' => 1026)
		c = Klass.new(@handle, s, 'no_autobind' => 1, 'no_socketsetup' => 1)
		assert_instance_of(Rex::Proto::DCERPC::Client, c, 'instance')
	end

	def test_bind
		write = ["\005\000\v\003\020\000\000\000H\000\000\000\000\000\000\000\320\026\320\026\000\000\000\000\001\000\000\000\000\000\001\000\010\203\257\341\037]\311\021\221\244\010\000+\024\240\372\003\000\000\000\004]\210\212\353\034\311\021\237\350\010\000+\020H`\002\000\000\000"]
		read = ["\x05\x00\x0C\x03\x10\x00\x00\x00\x44\x00\x00\x00\x00\x00\x00\x00\xB8\x10\xB8\x10\xB2\x73\x00\x00\x0C\x00\x5C\x50\x49\x50\x45\x5C\x6C\x73\x61\x73\x73\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x04\x5D\x88\x8A\xEB\x1C\xC9\x11\x9F\xE8\x08\x00\x2B\x10\x48\x60\x02\x00\x00\x00",]
		setup_test(read, write)
		socket = Rex::Socket.create_tcp( 'PeerHost' => $_REX_TEST_SMB_HOST, 'PeerPort' => 135)
		handle = Rex::Proto::DCERPC::Handle.new(['E1AF8308-5D1F-11C9-91A4-08002B14A0FA', '3.0'], 'ncacn_ip_tcp', $_REX_TEST_SMB_HOST, [135])
		c = Klass.new(handle, socket)
		assert_instance_of(Rex::Proto::DCERPC::Client, c, 'instance')
	end

	def test_bind_fail
		write = ["\x05\x00\x0b\x03\x10\x00\x00\x00\x48\x00\x00\x00\x00\x00\x00\x00\xd0\x16\xd0\x16\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x01\x00\xbb\xbb\xaa\xaa\x00\x00\x10\x36\x98\x33\x46\xc3\xf8\x7e\x34\x5a\x02\x00\x00\x00\x04\x5d\x88\x8a\xeb\x1c\xc9\x11\x9f\xe8\x08\x00\x2b\x10\x48\x60\x02\x00\x00\x00"]
		read = [ "\x05\x00\x0c\x03\x10\x00\x00\x00\x3c\x00\x00\x00\x00\x00\x00\x00\xd0\x16\xd0\x16\x6c\x66\x00\x00\x04\x00\x31\x33\x35\x00\x92\xc0\x01\x00\x00\x00\x02\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00"]
		setup_test(read, write)
		socket = Rex::Socket.create_tcp('PeerHost' => $_REX_TEST_SMB_HOST, 'PeerPort' => 135)
		handle = Rex::Proto::DCERPC::Handle.new(['AAAABBBB-0000-3610-9833-46c3f87e345a', '2.0'], 'ncacn_ip_tcp', $_REX_TEST_SMB_HOST, [135])

		assert_raise(RuntimeError) {
			c = Klass.new(handle, socket)
		}
	end

	def test_call
		write = [
			"\x05\x00\x0b\x03\x10\x00\x00\x00\x48\x00\x00\x00\x00\x00\x00\x00\xd0\x16\xd0\x16\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x01\x00\x98\xd0\xff\x6b\x12\xa1\x10\x36\x98\x33\x46\xc3\xf8\x7e\x34\x5a\x01\x00\x00\x00\x04\x5d\x88\x8a\xeb\x1c\xc9\x11\x9f\xe8\x08\x00\x2b\x10\x48\x60\x02\x00\x00\x00",
			"\x05\x00\x00\x03\x10\x00\x00\x00\x1c\x00\x00\x00\x00\x00\x00\x00\x04\x00\x00\x00\x00\x00\x0a\x00ABCD"
		]
		read = [
			"\x05\x00\x0C\x03\x10\x00\x00\x00\x44\x00\x00\x00\x00\x00\x00\x00\xB8\x10\xB8\x10\xB2\x73\x00\x00\x0C\x00\x5C\x50\x49\x50\x45\x5C\x6C\x73\x61\x73\x73\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x04\x5D\x88\x8A\xEB\x1C\xC9\x11\x9F\xE8\x08\x00\x2B\x10\x48\x60\x02\x00\x00\x00",
			"\x05\x00\x02\x23\x10\x00\x00\x00\x20\x00\x00\x00\x00\x00\x00\x00\x03\x00\x00\x00\x00\x00\x00\x00\x03\x00\x01\x1C\x00\x00\x00\x00foo"
		]
		setup_test(read, write, true)
		socket = Rex::Socket.create_tcp('PeerHost' => $_REX_TEST_SMB_HOST, 'PeerPort' => 135)

		c = Klass.new(@handle, socket)
		assert_instance_of(Rex::Proto::DCERPC::Client, c, 'instance')
		response = c.call(10, "ABCD")
		assert_equal(response, 'foo', 'call')
	end

	def test_fault
		write = [
			"\005\000\v\003\020\000\000\000H\000\000\000\000\000\000\000\320\026\320\026\000\000\000\000\001\000\000\000\000\000\001\000\010\203\257\341\037]\311\021\221\244\010\000+\024\240\372\003\000\000\000\004]\210\212\353\034\311\021\237\350\010\000+\020H`\002\000\000\000",
			"\x05\x00\x00\x03\x10\x00\x00\x00\x1c\x00\x00\x00\x00\x00\x00\x00\x04\x00\x00\x00\x00\x00\x0a\x00ABCD"
		]
		read = [
			"\x05\x00\x0C\x03\x10\x00\x00\x00\x44\x00\x00\x00\x00\x00\x00\x00\xB8\x10\xB8\x10\xB2\x73\x00\x00\x0C\x00\x5C\x50\x49\x50\x45\x5C\x6C\x73\x61\x73\x73\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x04\x5D\x88\x8A\xEB\x1C\xC9\x11\x9F\xE8\x08\x00\x2B\x10\x48\x60\x02\x00\x00\x00",
			"\x05\x00\x03\x23\x10\x00\x00\x00\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x00\x01\x1c\x00\x00\x00\x00"
		]
		setup_test(read, write)

		socket = Rex::Socket.create_tcp('PeerHost' => $_REX_TEST_SMB_HOST, 'PeerPort' => 135)
		handle = Rex::Proto::DCERPC::Handle.new(['E1AF8308-5D1F-11C9-91A4-08002B14A0FA', '3.0'], 'ncacn_ip_tcp', $_REX_TEST_SMB_HOST, [135])

		c = Klass.new(handle, socket)
		assert_instance_of(Rex::Proto::DCERPC::Client, c, 'instance')
		assert_raise(Rex::Proto::DCERPC::Exceptions::Fault) {
			response = c.call(10, "ABCD")
		}
		#                                    UGLY!
		assert_equal(c.last_response.status, 0x1C010002, 'call op 10 with stub response')
	end

	def test_segmented_read_writes
		bind_expected = "\005\000\v\003\020\000\000\000H\000\000\000\000\000\000\000\320\026\320\026\000\000\000\000\001\000\000\000\000\000\001\000\230\320\377k\022\241\0206\2303F\303\370~4Z\001\000\000\000\004]\210\212\353\034\311\021\237\350\010\000+\020H`\002\000\000\000"
		bind_response = "\x05\x00\x0C\x03\x10\x00\x00\x00\x44\x00\x00\x00\x00\x00\x00\x00\xB8\x10\xB8\x10\xB2\x73\x00\x00\x0C\x00\x5C\x50\x49\x50\x45\x5C\x6C\x73\x61\x73\x73\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x04\x5D\x88\x8A\xEB\x1C\xC9\x11\x9F\xE8\x08\x00\x2B\x10\x48\x60\x02\x00\x00\x00"

		socket = FlexMock.new
		#socket.should_ignore_missing()

		@write_data = ''

		socket.mock_handle(:write) {
			|data|
			@write_data += data
			data.length
		}

		read_data = bind_response.dup
		socket.mock_handle(:read) {
			|size|
			if size.nil?
				size = read_data.size
			end
			data = read_data.slice!(0, size)
			if data.length == 0
				data = nil
			end
			data
		}
		socket.mock_handle(:get_once) {
			|timeout, size|
			if size.nil?
				size = read_data.size
			end
			data = read_data.slice!(0, size)
			if data.length == 0
				data = nil
			end
			data
		}
		socket.should_receive(:type?).returns('tcp')

		c = Klass.new(@handle, socket, 'segment_write' => 1, 'segment_read' => 1)
		assert_equal(bind_expected, @write_data, 'bind')
		assert_instance_of(Rex::Proto::DCERPC::Client, c, 'instance')

		# XXX - would be nice to test a success, as well as a failure, but
		# hell... we don't care if the decoder is "right" here, just that it
		# returns data.  Test the decoder in the decoder I say!
		call_data = "\005\000\000\003\020\000\000\000\034\000\000\000\000\000\000\000\004\000\000\000\000\000\n\000ABCD"
		call_response = "\x05\x00\x02\x23\x10\x00\x00\x00\x20\x00\x00\x00\x00\x00\x00\x00\x03\x00\x00\x00\x00\x00\x00\x00\x03\x00\x01\x1C\x00\x00\x00\x00foo"
		read_data = call_response
		@write_data = ''
		socket.mock_handle(:write) {
			|data|
			@write_data += data
			data.length
		}
		read_data = call_response.dup
		socket.mock_handle(:read) {
			|size|
			data = read_data.slice!(0, size)
			if data.length == 0
				data = nil
			else
			end
			data
		}

		response = c.call(10, "ABCD")
		assert_equal(response, 'foo', 'response')
		assert_equal(@write_data, call_data, 'write')
	end

	def test_dcerpc_over_smb
		write = [
		"\x00\x00\x00\x54\xff\x53\x4d\x42\x72\x00\x00\x00\x00\x18\x01\x28\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x00\x00\xac\x0a\x00\x31\x00\x02\x4c\x41\x4e\x4d\x41\x4e\x31\x2e\x30\x00\x02\x4c\x4d\x31\x2e\x32\x58\x30\x30\x32\x00\x02\x4e\x54\x20\x4c\x41\x4e\x4d\x41\x4e\x20\x31\x2e\x30\x00\x02\x4e\x54\x20\x4c\x4d\x20\x30\x2e\x31\x32\x00",
		"\x00\x00\x00\xb0\xff\x53\x4d\x42\x73\x00\x00\x00\x00\x18\x01\x28\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x00\x00\xac\x0a\x0c\xff\x00\x00\x00\xdf\xff\x02\x00\x01\x00\x00\x00\x00\x00\x52\x00\x00\x00\x00\x00\x5c\xd0\x00\x80\x75\x00\x60\x50\x06\x06\x2b\x06\x01\x05\x05\x02\xa0\x46\x30\x44\xa0\x0e\x30\x0c\x06\x0a\x2b\x06\x01\x04\x01\x82\x37\x02\x02\x0a\xa2\x32\x04\x30\x4e\x54\x4c\x4d\x53\x53\x50\x00\x01\x00\x00\x00\x01\x02\x08\x00\x00\x00\x00\x00\x20\x00\x00\x00\x10\x00\x10\x00\x20\x00\x00\x00\x31\x41\x44\x37\x44\x6e\x4a\x54\x56\x79\x6b\x58\x47\x59\x59\x4d\x57\x69\x6e\x64\x6f\x77\x73\x20\x32\x30\x30\x30\x20\x32\x31\x39\x35\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x32\x30\x30\x30\x20\x35\x2e\x30\x00",
		"\x00\x00\x00\xfa\xff\x53\x4d\x42\x73\x00\x00\x00\x00\x18\x01\x28\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x00\x08\xac\x0a\x0c\xff\x00\x00\x00\xdf\xff\x02\x00\x01\x00\x00\x00\x00\x00\x9c\x00\x00\x00\x00\x00\x5c\xd0\x00\x80\xbf\x00\xa1\x81\x99\x30\x81\x96\xa2\x81\x93\x04\x81\x90\x4e\x54\x4c\x4d\x53\x53\x50\x00\x03\x00\x00\x00\x18\x00\x18\x00\x40\x00\x00\x00\x18\x00\x18\x00\x58\x00\x00\x00\x00\x00\x00\x00\x70\x00\x00\x00\x00\x00\x00\x00\x70\x00\x00\x00\x20\x00\x20\x00\x70\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x02\x08\x00\x3a\xc1\xe6\x27\x57\xae\x58\x51\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x11\x48\x4d\x21\x82\x94\x0e\xb8\xf1\xe3\x71\xbd\x12\x19\x97\x88\x90\x8b\xa2\x23\xad\x71\x89\x4a\x31\x00\x41\x00\x44\x00\x37\x00\x44\x00\x6e\x00\x4a\x00\x54\x00\x56\x00\x79\x00\x6b\x00\x58\x00\x47\x00\x59\x00\x59\x00\x4d\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x32\x30\x30\x30\x20\x32\x31\x39\x35\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x32\x30\x30\x30\x20\x35\x2e\x30\x00",
		"\x00\x00\x00\x62\xff\x53\x4d\x42\x73\x00\x00\x00\x00\x18\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x00\x00\xac\x0a\x0d\xff\x00\x00\x00\xdf\xff\x02\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x40\x00\x00\x00\x25\x00\x00\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x32\x30\x30\x30\x20\x32\x31\x39\x35\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x32\x30\x30\x30\x20\x35\x2e\x30\x00",
		"\x00\x00\x00\x37\xff\x53\x4d\x42\x75\x00\x00\x00\x00\x18\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x01\x08\xac\x0a\x04\xff\x00\x00\x00\x00\x00\x01\x00\x0c\x00\x00\x49\x50\x43\x24\x00\x3f\x3f\x3f\x3f\x3f\x00",
		"\x00\x00\x00\x5c\xff\x53\x4d\x42\xa2\x00\x00\x00\x00\x18\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x18\xff\x00\x00\x00\x00\x08\x00\x16\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x07\x00\x00\x00\x03\x00\x00\x00\x00\x00\x00\x00\x02\x00\x00\x00\x00\x09\x00\x5c\x42\x52\x4f\x57\x53\x45\x52\x00",
		"\x00\x00\x00\x87\xff\x53\x4d\x42\x2f\x00\x00\x00\x00\x18\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x0e\xff\x00\x00\x00\x00\x40\x00\x00\x00\x00\xff\xff\xff\xff\x08\x00\x48\x00\x00\x00\x48\x00\x3f\x00\x00\x00\x00\x00\x48\x00\x05\x00\x0b\x03\x10\x00\x00\x00\x48\x00\x00\x00\x00\x00\x00\x00\xd0\x16\xd0\x16\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x01\x00\xc8\x4f\x32\x4b\x70\x16\xd3\x01\x12\x78\x5a\x47\xbf\x6e\xe1\x88\x03\x00\x00\x00\x04\x5d\x88\x8a\xeb\x1c\xc9\x11\x9f\xe8\x08\x00\x2b\x10\x48\x60\x02\x00\x00\x00",
		"\x00\x00\x00\x37\xff\x53\x4d\x42\x2e\x00\x00\x00\x00\x18\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x0a\xff\x00\x00\x00\x00\x40\x00\x00\x00\x00\x80\xbb\x80\xbb\x00\x00\x00\x00\x00\x00\x00\x00"
		]

		read = [
		"\x00\x00\x00\x55\xff\x53\x4d\x42\x72\x00\x00\x00\x00\x98\x01\x28\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x00\x00\xac\x0a\x11\x03\x00\x03\x0a\x00\x01\x00\x04\x11\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\xfd\xe3\x00\x80\x16\x12\x0a\x27\x2e\xfc\xc5\x01\xe0\x01\x00\x10\x00\x8b\x3c\x73\xd2\x53\x9e\x28\x48\xa2\xac\x68\xae\xd1\x2e\x41\x5a",
		"\x00\x00\x01\xb3\xff\x53\x4d\x42\x73\x16\x00\x00\xc0\x98\x01\x68\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x00\x08\xac\x0a\x04\xff\x00\xb3\x01\x00\x00\x63\x01\x88\x01\xa1\x82\x01\x5f\x30\x82\x01\x5b\xa0\x03\x0a\x01\x01\xa1\x0c\x06\x0a\x2b\x06\x01\x04\x01\x82\x37\x02\x02\x0a\xa2\x81\xa1\x04\x81\x9e\x4e\x54\x4c\x4d\x53\x53\x50\x00\x02\x00\x00\x00\x12\x00\x12\x00\x30\x00\x00\x00\x05\x02\x8a\x00\x08\x78\x80\x62\x32\xc8\xeb\xe7\x00\x00\x00\x00\x00\x00\x00\x00\x5c\x00\x5c\x00\x42\x00\x00\x00\x57\x00\x49\x00\x4e\x00\x32\x00\x4b\x00\x42\x00\x41\x00\x53\x00\x45\x00\x02\x00\x12\x00\x57\x00\x49\x00\x4e\x00\x32\x00\x4b\x00\x42\x00\x41\x00\x53\x00\x45\x00\x01\x00\x12\x00\x57\x00\x49\x00\x4e\x00\x32\x00\x4b\x00\x42\x00\x41\x00\x53\x00\x45\x00\x04\x00\x12\x00\x77\x00\x69\x00\x6e\x00\x32\x00\x6b\x00\x62\x00\x61\x00\x73\x00\x65\x00\x03\x00\x12\x00\x77\x00\x69\x00\x6e\x00\x32\x00\x6b\x00\x62\x00\x61\x00\x73\x00\x65\x00\x00\x00\x00\x00\xa3\x81\xa1\x04\x81\x9e\x4e\x54\x4c\x4d\x53\x53\x50\x00\x02\x00\x00\x00\x12\x00\x12\x00\x30\x00\x00\x00\x05\x02\x8a\x00\x08\x78\x80\x62\x32\xc8\xeb\xe7\x00\x00\x00\x00\x00\x00\x00\x00\x5c\x00\x5c\x00\x42\x00\x00\x00\x57\x00\x49\x00\x4e\x00\x32\x00\x4b\x00\x42\x00\x41\x00\x53\x00\x45\x00\x02\x00\x12\x00\x57\x00\x49\x00\x4e\x00\x32\x00\x4b\x00\x42\x00\x41\x00\x53\x00\x45\x00\x01\x00\x12\x00\x57\x00\x49\x00\x4e\x00\x32\x00\x4b\x00\x42\x00\x41\x00\x53\x00\x45\x00\x04\x00\x12\x00\x77\x00\x69\x00\x6e\x00\x32\x00\x6b\x00\x62\x00\x61\x00\x73\x00\x65\x00\x03\x00\x12\x00\x77\x00\x69\x00\x6e\x00\x32\x00\x6b\x00\x62\x00\x61\x00\x73\x00\x65\x00\x00\x00\x00\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x35\x2e\x30\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x32\x30\x30\x30\x20\x4c\x41\x4e\x20\x4d\x61\x6e\x61\x67\x65\x72\x00",
		"\x00\x00\x00\x23\xff\x53\x4d\x42\x73\x6d\x00\x00\xc0\x98\x01\x68\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x00\x08\xac\x0a\x00\x00\x00",
		"\x00\x00\x00\x58\xff\x53\x4d\x42\x73\x00\x00\x00\x00\x98\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x01\x08\xac\x0a\x03\xff\x00\x58\x00\x00\x00\x2f\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x35\x2e\x30\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x32\x30\x30\x30\x20\x4c\x41\x4e\x20\x4d\x61\x6e\x61\x67\x65\x72\x00\x57\x4f\x52\x4b\x47\x52\x4f\x55\x50\x00",
		"\x00\x00\x00\x2e\xff\x53\x4d\x42\x75\x00\x00\x00\x00\x98\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x03\xff\x00\x2e\x00\x01\x00\x05\x00\x49\x50\x43\x00\x00",
		"\x00\x00\x00\x87\xff\x53\x4d\x42\xa2\x00\x00\x00\x00\x98\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x2a\xff\x00\x87\x00\x00\x00\x40\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x00\x00\x00\x00\x10\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x00\xff\x05\x00\x00\x00\x00\x00\x00\x00\x00\x04\x5d\x88\x8a\xeb\x1c\xc9\x11\x9f\xe8\x08\x00\x2b\x10\x48\x60\x02\x9b\x01\x12\x00\x9b\x01\x12\x00\x00\x00",
		"\x00\x00\x00\x2f\xff\x53\x4d\x42\x2f\x00\x00\x00\x00\x98\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x06\xff\x00\x2f\x00\x48\x00\xff\xff\x00\x00\x00\x00\x00\x00",
		"\x00\x00\x00\x80\xff\x53\x4d\x42\x2e\x00\x00\x00\x00\x98\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x0c\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x00\x3c\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x45\x00\x00\x05\x00\x0c\x03\x10\x00\x00\x00\x44\x00\x00\x00\x00\x00\x00\x00\xb8\x10\xb8\x10\xdd\xe2\x00\x00\x0d\x00\x5c\x50\x49\x50\x45\x5c\x6e\x74\x73\x76\x63\x73\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x04\x5d\x88\x8a\xeb\x1c\xc9\x11\x9f\xe8\x08\x00\x2b\x10\x48\x60\x02\x00\x00\x00",
		]
		setup_test(read, write)
		require 'rex/proto/smb/simpleclient'
		begin
			Timeout.timeout($_REX_TEST_TIMEOUT) {
				s = Rex::Socket.create_tcp( 'PeerHost' => $_REX_TEST_SMB_HOST, 'PeerPort' => 445)

				smb = Rex::Proto::SMB::SimpleClient.new(s, true)
				assert_instance_of(Rex::Proto::SMB::SimpleClient, smb, 'instance of smb')

				user = ''
				pass = ''

				smb.login('*SMBSERVER', user, pass)
				smb.connect('IPC$')
				f = smb.create_pipe('\BROWSER')
				assert_instance_of(Rex::Proto::SMB::SimpleClient::OpenPipe, f, 'pipe')

				handle = Rex::Proto::DCERPC::Handle.new(['4b324fc8-1670-01d3-1278-5a47bf6ee188', '3.0'], 'ncacn_np', $_REX_TEST_SMB_HOST, ['\BROWSER'])
				assert_instance_of(Rex::Proto::DCERPC::Handle, handle, 'handle')
				dcerpc = Rex::Proto::DCERPC::Client.new(handle, f) # , 'segment_read' => 0)
				assert_instance_of(Rex::Proto::DCERPC::Client, dcerpc, 'bind')
				s.close
			}
		rescue Timeout::Error
			flunk('timeout')
		end
	end

	def test_ncacn_ip_tcp
		write = [
		"\x05\x00\x0b\x03\x10\x00\x00\x00\x48\x00\x00\x00\x00\x00\x00\x00\xd0\x16\xd0\x16\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x01\x00\x08\x83\xaf\xe1\x1f\x5d\xc9\x11\x91\xa4\x08\x00\x2b\x14\xa0\xfa\x03\x00\x00\x00\x04\x5d\x88\x8a\xeb\x1c\xc9\x11\x9f\xe8\x08\x00\x2b\x10\x48\x60\x02\x00\x00\x00",
		"\x05\x00\x00\x03\x10\x00\x00\x00\x40\x00\x00\x00\x00\x00\x00\x00\x28\x00\x00\x00\x00\x00\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00"
		]
		read = [
		"\x05\x00\x0c\x03\x10\x00\x00\x00\x3c\x00\x00\x00\x00\x00\x00\x00\xd0\x16\xd0\x16\x12\x66\x00\x00\x04\x00\x31\x33\x35\x00\xec\xd7\x01\x00\x00\x00\x00\x00\x00\x00\x04\x5d\x88\x8a\xeb\x1c\xc9\x11\x9f\xe8\x08\x00\x2b\x10\x48\x60\x02\x00\x00\x00",
		"\x05\x00\x02\x03\x10\x00\x00\x00\xb4\x00\x00\x00\x00\x00\x00\x00\x9c\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xc4\x49\xc2\xb8\x92\x78\x4e\x4f\xb3\xe7\xbf\x7e\x00\x2a\xd3\x39\x01\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x4b\x00\x00\x00\x4b\x00\x00\x00\x05\x00\x13\x00\x0d\x80\xa9\x88\x10\xe5\xea\xd0\x11\x8d\x9b\x00\xa0\x24\x53\xc3\x37\x01\x00\x02\x00\x00\x00\x13\x00\x0d\x04\x5d\x88\x8a\xeb\x1c\xc9\x11\x9f\xe8\x08\x00\x2b\x10\x48\x60\x02\x00\x02\x00\x00\x00\x01\x00\x0b\x02\x00\x00\x00\x01\x00\x07\x02\x00\x08\x39\x01\x00\x09\x04\x00\x0a\x04\x0a\x3a\x00\x00\x00\x00\x00"
		]
		setup_test(read, write)
		begin
			Timeout.timeout($_REX_TEST_TIMEOUT) {
				s = Rex::Socket.create_tcp('PeerHost' => $_REX_TEST_SMB_HOST, 'PeerPort' => 135)
				handle = Rex::Proto::DCERPC::Handle.new(['E1AF8308-5D1F-11C9-91A4-08002B14A0FA', '3.0'], 'ncacn_ip_tcp', $_REX_TEST_SMB_HOST, [135])
				assert_instance_of(Rex::Proto::DCERPC::Handle, handle, 'handle')
				dcerpc = Rex::Proto::DCERPC::Client.new(handle, s)
				assert_instance_of(Rex::Proto::DCERPC::Client, dcerpc, 'bind')

				stub = "\x00" * (4 * 9) + "\x01\x00\x00\x00"
				got = dcerpc.call(0x02, stub)
				s.close
			}
		rescue Timeout::Error
			flunk('timeout')
		end
	end

	def test_ncacn_ip_tcp_objectid
		write = [
		"\x05\x00\x0b\x03\x10\x00\x00\x00\x48\x00\x00\x00\x00\x00\x00\x00\xd0\x16\xd0\x16\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x01\x00\x08\x83\xaf\xe1\x1f\x5d\xc9\x11\x91\xa4\x08\x00\x2b\x14\xa0\xfa\x03\x00\x00\x00\x04\x5d\x88\x8a\xeb\x1c\xc9\x11\x9f\xe8\x08\x00\x2b\x10\x48\x60\x02\x00\x00\x00",
		"\x05\x00\x00\x83\x10\x00\x00\x00\x50\x00\x00\x00\x00\x00\x00\x00\x28\x00\x00\x00\x00\x00\x02\x00\010\203\257\341\037]\311\021\221\244\010\000+\024\240\372\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00"
		]
		read = [
		"\x05\x00\x0c\x03\x10\x00\x00\x00\x3c\x00\x00\x00\x00\x00\x00\x00\xd0\x16\xd0\x16\x12\x66\x00\x00\x04\x00\x31\x33\x35\x00\xec\xd7\x01\x00\x00\x00\x00\x00\x00\x00\x04\x5d\x88\x8a\xeb\x1c\xc9\x11\x9f\xe8\x08\x00\x2b\x10\x48\x60\x02\x00\x00\x00",
		"\x05\x00\x02\x03\x10\x00\x00\x00\xb4\x00\x00\x00\x00\x00\x00\x00\x9c\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xc4\x49\xc2\xb8\x92\x78\x4e\x4f\xb3\xe7\xbf\x7e\x00\x2a\xd3\x39\x01\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x4b\x00\x00\x00\x4b\x00\x00\x00\x05\x00\x13\x00\x0d\x80\xa9\x88\x10\xe5\xea\xd0\x11\x8d\x9b\x00\xa0\x24\x53\xc3\x37\x01\x00\x02\x00\x00\x00\x13\x00\x0d\x04\x5d\x88\x8a\xeb\x1c\xc9\x11\x9f\xe8\x08\x00\x2b\x10\x48\x60\x02\x00\x02\x00\x00\x00\x01\x00\x0b\x02\x00\x00\x00\x01\x00\x07\x02\x00\x08\x39\x01\x00\x09\x04\x00\x0a\x04\x0a\x3a\x00\x00\x00\x00\x00"
		]
		setup_test(read, write)
		begin
			Timeout.timeout($_REX_TEST_TIMEOUT) {
				s = Rex::Socket.create_tcp('PeerHost' => $_REX_TEST_SMB_HOST, 'PeerPort' => 135)
				handle = Rex::Proto::DCERPC::Handle.new(['E1AF8308-5D1F-11C9-91A4-08002B14A0FA', '3.0'], 'ncacn_ip_tcp', $_REX_TEST_SMB_HOST, [135])
				assert_instance_of(Rex::Proto::DCERPC::Handle, handle, 'handle')
				dcerpc = Rex::Proto::DCERPC::Client.new(handle, s, {'object_call' => 1})
				assert_instance_of(Rex::Proto::DCERPC::Client, dcerpc, 'bind')

				stub = "\x00" * (4 * 9) + "\x01\x00\x00\x00"
				got = dcerpc.call(0x02, stub)
				s.close
			}
		rescue Timeout::Error
			flunk('timeout')
		end
	end
	def test_ncacn_ip_tcp_objectid_random
		write = [
		"\x05\x00\x0b\x03\x10\x00\x00\x00\x48\x00\x00\x00\x00\x00\x00\x00\xd0\x16\xd0\x16\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x01\x00\x08\x83\xaf\xe1\x1f\x5d\xc9\x11\x91\xa4\x08\x00\x2b\x14\xa0\xfa\x03\x00\x00\x00\x04\x5d\x88\x8a\xeb\x1c\xc9\x11\x9f\xe8\x08\x00\x2b\x10\x48\x60\x02\x00\x00\x00",
		"\005\000\000\203\020\000\000\000P\000\000\000\000\000\000\000(\000\000\000\000\000\002\000\254/u\300C\373\303g\t\323\025\362$WF\330\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\001\000\000\000"
		]
		read = [
		"\x05\x00\x0c\x03\x10\x00\x00\x00\x3c\x00\x00\x00\x00\x00\x00\x00\xd0\x16\xd0\x16\x12\x66\x00\x00\x04\x00\x31\x33\x35\x00\xec\xd7\x01\x00\x00\x00\x00\x00\x00\x00\x04\x5d\x88\x8a\xeb\x1c\xc9\x11\x9f\xe8\x08\x00\x2b\x10\x48\x60\x02\x00\x00\x00",
		"\x05\x00\x02\x03\x10\x00\x00\x00\xb4\x00\x00\x00\x00\x00\x00\x00\x9c\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\xc4\x49\xc2\xb8\x92\x78\x4e\x4f\xb3\xe7\xbf\x7e\x00\x2a\xd3\x39\x01\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x4b\x00\x00\x00\x4b\x00\x00\x00\x05\x00\x13\x00\x0d\x80\xa9\x88\x10\xe5\xea\xd0\x11\x8d\x9b\x00\xa0\x24\x53\xc3\x37\x01\x00\x02\x00\x00\x00\x13\x00\x0d\x04\x5d\x88\x8a\xeb\x1c\xc9\x11\x9f\xe8\x08\x00\x2b\x10\x48\x60\x02\x00\x02\x00\x00\x00\x01\x00\x0b\x02\x00\x00\x00\x01\x00\x07\x02\x00\x08\x39\x01\x00\x09\x04\x00\x0a\x04\x0a\x3a\x00\x00\x00\x00\x00"
		]
		setup_test(read, write)
		begin
			Timeout.timeout($_REX_TEST_TIMEOUT) {
				s = Rex::Socket.create_tcp('PeerHost' => $_REX_TEST_SMB_HOST, 'PeerPort' => 135)
				handle = Rex::Proto::DCERPC::Handle.new(['E1AF8308-5D1F-11C9-91A4-08002B14A0FA', '3.0'], 'ncacn_ip_tcp', $_REX_TEST_SMB_HOST, [135])
				assert_instance_of(Rex::Proto::DCERPC::Handle, handle, 'handle')
				dcerpc = Rex::Proto::DCERPC::Client.new(handle, s, {'random_object_id' => 1})
				assert_instance_of(Rex::Proto::DCERPC::Client, dcerpc, 'bind')

				stub = "\x00" * (4 * 9) + "\x01\x00\x00\x00"
				got = dcerpc.call(0x02, stub)
				s.close
			}
		rescue Timeout::Error
			flunk('timeout')
		end
	end

	# this test is the same as test_ncanc_ip_tcp
	def test_setup_socket_ncacn_ip_tcp
	end

	def test_setup_socket_ncacn_ip_tcp_nil_socket
		read = ["\x05\x00\x0c\x03\x10\x00\x00\x00\x3c\x00\x00\x00\x00\x00\x00\x00\xd0\x16\xd0\x16\x56\x66\x00\x00\x04\x00\x31\x33\x35\x00\xcd\x71\x01\x00\x00\x00\x00\x00\x00\x00\x04\x5d\x88\x8a\xeb\x1c\xc9\x11\x9f\xe8\x08\x00\x2b\x10\x48\x60\x02\x00\x00\x00"]
		write = ["\005\000\v\003\020\000\000\000H\000\000\000\000\000\000\000\320\026\320\026\000\000\000\000\001\000\000\000\000\000\001\000\010\203\257\341\037]\311\021\221\244\010\000+\024\240\372\003\000\000\000\004]\210\212\353\034\311\021\237\350\010\000+\020H`\002\000\000\000"]
		setup_test(read, write)
		begin
			Timeout.timeout($_REX_TEST_TIMEOUT) {
				handle = Rex::Proto::DCERPC::Handle.new(['E1AF8308-5D1F-11C9-91A4-08002B14A0FA', '3.0'], 'ncacn_ip_tcp', $_REX_TEST_SMB_HOST, [135])
				assert_instance_of(Rex::Proto::DCERPC::Handle, handle, 'handle')
				dcerpc = Rex::Proto::DCERPC::Client.new(handle, nil)
				assert_instance_of(Rex::Proto::DCERPC::Client, dcerpc, 'bind')
			}
		rescue Timeout::Error
			flunk('timeout')
		end
	end

	def test_setup_socket_ncacn_np_socket
		read = ["\x82\x00\x00\x00",
		"\x00\x00\x00\x55\xff\x53\x4d\x42\x72\x00\x00\x00\x00\x98\x01\x28\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x00\x00\xac\x0a\x11\x03\x00\x03\x0a\x00\x01\x00\x04\x11\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\xfd\xe3\x00\x80\xca\x3d\x91\x3c\x0b\xfd\xc5\x01\xe0\x01\x00\x10\x00\x8b\x3c\x73\xd2\x53\x9e\x28\x48\xa2\xac\x68\xae\xd1\x2e\x41\x5a",
		"\x00\x00\x01\xb3\xff\x53\x4d\x42\x73\x16\x00\x00\xc0\x98\x01\x68\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x00\x08\xac\x0a\x04\xff\x00\xb3\x01\x00\x00\x63\x01\x88\x01\xa1\x82\x01\x5f\x30\x82\x01\x5b\xa0\x03\x0a\x01\x01\xa1\x0c\x06\x0a\x2b\x06\x01\x04\x01\x82\x37\x02\x02\x0a\xa2\x81\xa1\x04\x81\x9e\x4e\x54\x4c\x4d\x53\x53\x50\x00\x02\x00\x00\x00\x12\x00\x12\x00\x30\x00\x00\x00\x05\x02\x8a\x00\x4a\xda\x31\xb4\x0f\x33\x09\x9f\x00\x00\x00\x00\x00\x00\x00\x00\x5c\x00\x5c\x00\x42\x00\x00\x00\x57\x00\x49\x00\x4e\x00\x32\x00\x4b\x00\x42\x00\x41\x00\x53\x00\x45\x00\x02\x00\x12\x00\x57\x00\x49\x00\x4e\x00\x32\x00\x4b\x00\x42\x00\x41\x00\x53\x00\x45\x00\x01\x00\x12\x00\x57\x00\x49\x00\x4e\x00\x32\x00\x4b\x00\x42\x00\x41\x00\x53\x00\x45\x00\x04\x00\x12\x00\x77\x00\x69\x00\x6e\x00\x32\x00\x6b\x00\x62\x00\x61\x00\x73\x00\x65\x00\x03\x00\x12\x00\x77\x00\x69\x00\x6e\x00\x32\x00\x6b\x00\x62\x00\x61\x00\x73\x00\x65\x00\x00\x00\x00\x00\xa3\x81\xa1\x04\x81\x9e\x4e\x54\x4c\x4d\x53\x53\x50\x00\x02\x00\x00\x00\x12\x00\x12\x00\x30\x00\x00\x00\x05\x02\x8a\x00\x4a\xda\x31\xb4\x0f\x33\x09\x9f\x00\x00\x00\x00\x00\x00\x00\x00\x5c\x00\x5c\x00\x42\x00\x00\x00\x57\x00\x49\x00\x4e\x00\x32\x00\x4b\x00\x42\x00\x41\x00\x53\x00\x45\x00\x02\x00\x12\x00\x57\x00\x49\x00\x4e\x00\x32\x00\x4b\x00\x42\x00\x41\x00\x53\x00\x45\x00\x01\x00\x12\x00\x57\x00\x49\x00\x4e\x00\x32\x00\x4b\x00\x42\x00\x41\x00\x53\x00\x45\x00\x04\x00\x12\x00\x77\x00\x69\x00\x6e\x00\x32\x00\x6b\x00\x62\x00\x61\x00\x73\x00\x65\x00\x03\x00\x12\x00\x77\x00\x69\x00\x6e\x00\x32\x00\x6b\x00\x62\x00\x61\x00\x73\x00\x65\x00\x00\x00\x00\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x35\x2e\x30\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x32\x30\x30\x30\x20\x4c\x41\x4e\x20\x4d\x61\x6e\x61\x67\x65\x72\x00",
		"\x00\x00\x00\x23\xff\x53\x4d\x42\x73\x6d\x00\x00\xc0\x98\x01\x68\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x00\x08\xac\x0a\x00\x00\x00",
		"\x00\x00\x00\x58\xff\x53\x4d\x42\x73\x00\x00\x00\x00\x98\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x01\x08\xac\x0a\x03\xff\x00\x58\x00\x00\x00\x2f\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x35\x2e\x30\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x32\x30\x30\x30\x20\x4c\x41\x4e\x20\x4d\x61\x6e\x61\x67\x65\x72\x00\x57\x4f\x52\x4b\x47\x52\x4f\x55\x50\x00",
		"\x00\x00\x00\x2e\xff\x53\x4d\x42\x75\x00\x00\x00\x00\x98\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x03\xff\x00\x2e\x00\x01\x00\x05\x00\x49\x50\x43\x00\x00",
		"\x00\x00\x00\x87\xff\x53\x4d\x42\xa2\x00\x00\x00\x00\x98\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x2a\xff\x00\x87\x00\x00\x00\x40\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x00\x00\x00\x00\x10\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x00\xff\x05\x00\x00\x00\x70\x00\x00\x00\x00\x00\x00\x00\x70\x00\x00\x00\x20\x00\x20\x00\x70\x00\x00\x00\x00\x00\x9b\x01\x12\x00\x9b\x01\x12\x00\x08\x00",
		"\x00\x00\x00\x2f\xff\x53\x4d\x42\x2f\x00\x00\x00\x00\x98\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x06\xff\x00\x2f\x00\x48\x00\xff\xff\x00\x00\x00\x00\x00\x00",
		"\x00\x00\x00\x80\xff\x53\x4d\x42\x2e\x00\x00\x00\x00\x98\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x0c\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x00\x3c\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x45\x00\x00\x05\x00\x0c\x03\x10\x00\x00\x00\x44\x00\x00\x00\x00\x00\x00\x00\xb8\x10\xb8\x10\x98\x2d\x01\x00\x0d\x00\x5c\x50\x49\x50\x45\x5c\x6e\x74\x73\x76\x63\x73\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x04\x5d\x88\x8a\xeb\x1c\xc9\x11\x9f\xe8\x08\x00\x2b\x10\x48\x60\x02\x00\x00\x00",
		"\x00\x00\x00\x2f\xff\x53\x4d\x42\x2f\x00\x00\x00\x00\x98\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x06\xff\x00\x2f\x00\x64\x00\xff\xff\x00\x00\x00\x00\x00\x00",
		"\x00\x00\x00\x70\xff\x53\x4d\x42\x2e\x00\x00\x00\x00\x98\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x0c\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x34\x00\x3c\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x35\x00\x00\x05\x00\x02\x03\x10\x00\x00\x00\x34\x00\x00\x00\x00\x00\x00\x00\x1c\x00\x00\x00\x00\x00\x00\x00\x0a\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x34\x00\x00\xc0"
		]


		write = [
		"\x81\x00\x00\x44\x20\x43\x4b\x46\x44\x45\x4e\x45\x43\x46\x44\x45\x46\x46\x43\x46\x47\x45\x46\x46\x43\x43\x41\x43\x41\x43\x41\x43\x41\x43\x41\x43\x41\x00\x20\x43\x41\x43\x41\x43\x41\x43\x41\x43\x41\x43\x41\x43\x41\x43\x41\x43\x41\x43\x41\x43\x41\x43\x41\x43\x41\x43\x41\x43\x41\x41\x41\x00",
		"\x00\x00\x00\x54\xff\x53\x4d\x42\x72\x00\x00\x00\x00\x18\x01\x28\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x00\x00\xac\x0a\x00\x31\x00\x02\x4c\x41\x4e\x4d\x41\x4e\x31\x2e\x30\x00\x02\x4c\x4d\x31\x2e\x32\x58\x30\x30\x32\x00\x02\x4e\x54\x20\x4c\x41\x4e\x4d\x41\x4e\x20\x31\x2e\x30\x00\x02\x4e\x54\x20\x4c\x4d\x20\x30\x2e\x31\x32\x00",
		"\x00\x00\x00\xb0\xff\x53\x4d\x42\x73\x00\x00\x00\x00\x18\x01\x28\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x00\x00\xac\x0a\x0c\xff\x00\x00\x00\xdf\xff\x02\x00\x01\x00\x00\x00\x00\x00\x52\x00\x00\x00\x00\x00\x5c\xd0\x00\x80\x75\x00\x60\x50\x06\x06\x2b\x06\x01\x05\x05\x02\xa0\x46\x30\x44\xa0\x0e\x30\x0c\x06\x0a\x2b\x06\x01\x04\x01\x82\x37\x02\x02\x0a\xa2\x32\x04\x30\x4e\x54\x4c\x4d\x53\x53\x50\x00\x01\x00\x00\x00\x01\x02\x08\x00\x00\x00\x00\x00\x20\x00\x00\x00\x10\x00\x10\x00\x20\x00\x00\x00\x31\x41\x44\x37\x44\x6e\x4a\x54\x56\x79\x6b\x58\x47\x59\x59\x4d\x57\x69\x6e\x64\x6f\x77\x73\x20\x32\x30\x30\x30\x20\x32\x31\x39\x35\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x32\x30\x30\x30\x20\x35\x2e\x30\x00",
		"\x00\x00\x00\xfa\xff\x53\x4d\x42\x73\x00\x00\x00\x00\x18\x01\x28\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x00\x08\xac\x0a\x0c\xff\x00\x00\x00\xdf\xff\x02\x00\x01\x00\x00\x00\x00\x00\x9c\x00\x00\x00\x00\x00\x5c\xd0\x00\x80\xbf\x00\xa1\x81\x99\x30\x81\x96\xa2\x81\x93\x04\x81\x90\x4e\x54\x4c\x4d\x53\x53\x50\x00\x03\x00\x00\x00\x18\x00\x18\x00\x40\x00\x00\x00\x18\x00\x18\x00\x58\x00\x00\x00\x00\x00\x00\x00\x70\x00\x00\x00\x00\x00\x00\x00\x70\x00\x00\x00\x20\x00\x20\x00\x70\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x01\x02\x08\x00\x3a\xc1\xe6\x27\x57\xae\x58\x51\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x8c\x4d\x20\xdf\x63\xbd\xda\xdc\x62\x0d\x49\xe7\xf2\x7a\x86\x88\xe9\x5f\x03\x95\xbb\x9b\xc6\x85\x31\x00\x41\x00\x44\x00\x37\x00\x44\x00\x6e\x00\x4a\x00\x54\x00\x56\x00\x79\x00\x6b\x00\x58\x00\x47\x00\x59\x00\x59\x00\x4d\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x32\x30\x30\x30\x20\x32\x31\x39\x35\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x32\x30\x30\x30\x20\x35\x2e\x30\x00",
		"\x00\x00\x00\x62\xff\x53\x4d\x42\x73\x00\x00\x00\x00\x18\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x00\x00\xac\x0a\x0d\xff\x00\x00\x00\xdf\xff\x02\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x40\x00\x00\x00\x25\x00\x00\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x32\x30\x30\x30\x20\x32\x31\x39\x35\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x32\x30\x30\x30\x20\x35\x2e\x30\x00",
		"\x00\x00\x00\x37\xff\x53\x4d\x42\x75\x00\x00\x00\x00\x18\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x01\x08\xac\x0a\x04\xff\x00\x00\x00\x00\x00\x01\x00\x0c\x00\x00\x49\x50\x43\x24\x00\x3f\x3f\x3f\x3f\x3f\x00",
		"\x00\x00\x00\x5c\xff\x53\x4d\x42\xa2\x00\x00\x00\x00\x18\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x18\xff\x00\x00\x00\x00\x08\x00\x16\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x07\x00\x00\x00\x03\x00\x00\x00\x00\x00\x00\x00\x02\x00\x00\x00\x00\x09\x00\x5c\x42\x52\x4f\x57\x53\x45\x52\x00",
		"\x00\x00\x00\x87\xff\x53\x4d\x42\x2f\x00\x00\x00\x00\x18\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x0e\xff\x00\x00\x00\x00\x40\x00\x00\x00\x00\xff\xff\xff\xff\x08\x00\x48\x00\x00\x00\x48\x00\x3f\x00\x00\x00\x00\x00\x48\x00\x05\x00\x0b\x03\x10\x00\x00\x00\x48\x00\x00\x00\x00\x00\x00\x00\xd0\x16\xd0\x16\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x01\x00\xc8\x4f\x32\x4b\x70\x16\xd3\x01\x12\x78\x5a\x47\xbf\x6e\xe1\x88\x03\x00\x00\x00\x04\x5d\x88\x8a\xeb\x1c\xc9\x11\x9f\xe8\x08\x00\x2b\x10\x48\x60\x02\x00\x00\x00",
		"\000\000\0007\377SMB.\000\000\000\000\030\001 \000\000\000\000\000\000\000\000\000\000\000\000\000\010/\252\001\010\254\n\n\377\000\000\000\000@\000\000\000\000\200\273\200\273\000\000\000\000\000\000\000\000"
		]
		setup_test(read, write)
		begin
			Timeout.timeout($_REX_TEST_TIMEOUT) {
				s = Rex::Socket.create_tcp('PeerHost' => $_REX_TEST_SMB_HOST, 'PeerPort' => 139)

				handle = Rex::Proto::DCERPC::Handle.new(['4b324fc8-1670-01d3-1278-5a47bf6ee188', '3.0'], 'ncacn_np', $_REX_TEST_SMB_HOST, ['\BROWSER'])
				assert_instance_of(Rex::Proto::DCERPC::Handle, handle, 'handle')
				dcerpc = Rex::Proto::DCERPC::Client.new(handle, s)
				assert_instance_of(Rex::Proto::DCERPC::Client, dcerpc, 'bind')
			}
		rescue Timeout::Error
			flunk('timeout')
		end
	end

	# this test is the same as test_dcerpc_over_smb
	def test_setup_socket_ncacn_np_filehandle
	end

	# test connect to a dcerpc service via ncacn_np, without passing a socket.
	# this should test the autosocket & autosmb foo
	def test_setup_socket_ncacn_np_nil_socket

write = [
"\x00\x00\x00\x54\xff\x53\x4d\x42\x72\x00\x00\x00\x00\x18\x01\x28\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x00\x00\xac\x0a\x00\x31\x00\x02\x4c\x41\x4e\x4d\x41\x4e\x31\x2e\x30\x00\x02\x4c\x4d\x31\x2e\x32\x58\x30\x30\x32\x00\x02\x4e\x54\x20\x4c\x41\x4e\x4d\x41\x4e\x20\x31\x2e\x30\x00\x02\x4e\x54\x20\x4c\x4d\x20\x30\x2e\x31\x32\x00",
"\x00\x00\x00\xb0\xff\x53\x4d\x42\x73\x00\x00\x00\x00\x18\x01\x28\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x00\x00\xac\x0a\x0c\xff\x00\x00\x00\xdf\xff\x02\x00\x01\x00\x00\x00\x00\x00\x52\x00\x00\x00\x00\x00\x5c\xd0\x00\x80\x75\x00\x60\x50\x06\x06\x2b\x06\x01\x05\x05\x02\xa0\x46\x30\x44\xa0\x0e\x30\x0c\x06\x0a\x2b\x06\x01\x04\x01\x82\x37\x02\x02\x0a\xa2\x32\x04\x30\x4e\x54\x4c\x4d\x53\x53\x50\x00\x01\x00\x00\x00\x01\x02\x08\x00\x00\x00\x00\x00\x20\x00\x00\x00\x10\x00\x10\x00\x20\x00\x00\x00\x31\x41\x44\x37\x44\x6e\x4a\x54\x56\x79\x6b\x58\x47\x59\x59\x4d\x57\x69\x6e\x64\x6f\x77\x73\x20\x32\x30\x30\x30\x20\x32\x31\x39\x35\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x32\x30\x30\x30\x20\x35\x2e\x30\x00",
"\000\000\000\372\377SMBs\000\000\000\000\030\001(\000\000\000\000\000\000\000\000\000\000\000\000\000\000/\252\000\010\254\n\f\377\000\000\000\337\377\002\000\001\000\000\000\000\000\234\000\000\000\000\000\\\320\000\200\277\000\241\201\2310\201\226\242\201\223\004\201\220NTLMSSP\000\003\000\000\000\030\000\030\000@\000\000\000\030\000\030\000X\000\000\000\000\000\000\000p\000\000\000\000\000\000\000p\000\000\000 \000 \000p\000\000\000\000\000\000\000\000\000\000\000\001\002\010\000:\301\346'W\256XQ\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\000\314\256\352Sm%S:f/\335\230\200\327K\023\241\366\353vX\016\256\3571\000A\000D\0007\000D\000n\000J\000T\000V\000y\000k\000X\000G\000Y\000Y\000M\000Windows 2000 2195\000Windows 2000 5.0\000",
"\x00\x00\x00\x62\xff\x53\x4d\x42\x73\x00\x00\x00\x00\x18\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x00\x00\xac\x0a\x0d\xff\x00\x00\x00\xdf\xff\x02\x00\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x40\x00\x00\x00\x25\x00\x00\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x32\x30\x30\x30\x20\x32\x31\x39\x35\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x32\x30\x30\x30\x20\x35\x2e\x30\x00",
"\x00\x00\x00\x37\xff\x53\x4d\x42\x75\x00\x00\x00\x00\x18\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x01\x08\xac\x0a\x04\xff\x00\x00\x00\x00\x00\x01\x00\x0c\x00\x00\x49\x50\x43\x24\x00\x3f\x3f\x3f\x3f\x3f\x00",
"\x00\x00\x00\x5c\xff\x53\x4d\x42\xa2\x00\x00\x00\x00\x18\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x18\xff\x00\x00\x00\x00\x08\x00\x16\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x07\x00\x00\x00\x03\x00\x00\x00\x00\x00\x00\x00\x02\x00\x00\x00\x00\x09\x00\x5c\x42\x52\x4f\x57\x53\x45\x52\x00",
"\x00\x00\x00\x87\xff\x53\x4d\x42\x2f\x00\x00\x00\x00\x18\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x0e\xff\x00\x00\x00\x00\x40\x00\x00\x00\x00\xff\xff\xff\xff\x08\x00\x48\x00\x00\x00\x48\x00\x3f\x00\x00\x00\x00\x00\x48\x00\x05\x00\x0b\x03\x10\x00\x00\x00\x48\x00\x00\x00\x00\x00\x00\x00\xd0\x16\xd0\x16\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\x01\x00\xc8\x4f\x32\x4b\x70\x16\xd3\x01\x12\x78\x5a\x47\xbf\x6e\xe1\x88\x03\x00\x00\x00\x04\x5d\x88\x8a\xeb\x1c\xc9\x11\x9f\xe8\x08\x00\x2b\x10\x48\x60\x02\x00\x00\x00",
"\000\000\0007\377SMB.\000\000\000\000\030\001 \000\000\000\000\000\000\000\000\000\000\000\000\000\010/\252\001\010\254\n\n\377\000\000\000\000@\000\000\000\000\200\273\200\273\000\000\000\000\000\000\000\000",
"\x00\x00\x00\xa3\xff\x53\x4d\x42\x2f\x00\x00\x00\x00\x18\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x0e\xff\x00\x00\x00\x00\x40\x00\x00\x00\x00\xff\xff\xff\xff\x08\x00\x64\x00\x00\x00\x64\x00\x3f\x00\x00\x00\x00\x00\x64\x00\x05\x00\x00\x03\x10\x00\x00\x00\x64\x00\x00\x00\x00\x00\x00\x00\x4c\x00\x00\x00\x00\x00\x30\x00\x01\x00\x00\x00\x05\x00\x00\x00\x00\x00\x00\x00\x05\x00\x00\x00\x41\x00\x41\x00\x41\x00\x41\x00\x00\x00\x00\x00\x01\x00\x00\x00\x02\x00\x03\x00\x04\x00\x00\x00\x41\x41\x41\x41\x05\x00\x00\x00\x00\x00\x00\x00\x05\x00\x00\x00\x42\x00\x42\x00\x42\x00\x42\x00\x00\x00\x00\x00\x04\x00\x00\x00\x0a\x00\x00\x00",
"\000\000\0007\377SMB.\000\000\000\000\030\001 \000\000\000\000\000\000\000\000\000\000\000\000\000\010/\252\001\010\254\n\n\377\000\000\000\000@\000\000\000\000\200\273\200\273\000\000\000\000\000\000\000\000"
]
read = [
"\x00\x00\x00\x55\xff\x53\x4d\x42\x72\x00\x00\x00\x00\x98\x01\x28\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x00\x00\xac\x0a\x11\x03\x00\x03\x0a\x00\x01\x00\x04\x11\x00\x00\x00\x00\x01\x00\x00\x00\x00\x00\xfd\xe3\x00\x80\x20\x5f\xf5\x32\x11\xfd\xc5\x01\xe0\x01\x00\x10\x00\x8b\x3c\x73\xd2\x53\x9e\x28\x48\xa2\xac\x68\xae\xd1\x2e\x41\x5a",
"\x00\x00\x01\xb3\xff\x53\x4d\x42\x73\x16\x00\x00\xc0\x98\x01\x68\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x00\x08\xac\x0a\x04\xff\x00\xb3\x01\x00\x00\x63\x01\x88\x01\xa1\x82\x01\x5f\x30\x82\x01\x5b\xa0\x03\x0a\x01\x01\xa1\x0c\x06\x0a\x2b\x06\x01\x04\x01\x82\x37\x02\x02\x0a\xa2\x81\xa1\x04\x81\x9e\x4e\x54\x4c\x4d\x53\x53\x50\x00\x02\x00\x00\x00\x12\x00\x12\x00\x30\x00\x00\x00\x05\x02\x8a\x00\xf5\xe4\x8c\x90\x50\xe3\xcd\x63\x00\x00\x00\x00\x00\x00\x00\x00\x5c\x00\x5c\x00\x42\x00\x00\x00\x57\x00\x49\x00\x4e\x00\x32\x00\x4b\x00\x42\x00\x41\x00\x53\x00\x45\x00\x02\x00\x12\x00\x57\x00\x49\x00\x4e\x00\x32\x00\x4b\x00\x42\x00\x41\x00\x53\x00\x45\x00\x01\x00\x12\x00\x57\x00\x49\x00\x4e\x00\x32\x00\x4b\x00\x42\x00\x41\x00\x53\x00\x45\x00\x04\x00\x12\x00\x77\x00\x69\x00\x6e\x00\x32\x00\x6b\x00\x62\x00\x61\x00\x73\x00\x65\x00\x03\x00\x12\x00\x77\x00\x69\x00\x6e\x00\x32\x00\x6b\x00\x62\x00\x61\x00\x73\x00\x65\x00\x00\x00\x00\x00\xa3\x81\xa1\x04\x81\x9e\x4e\x54\x4c\x4d\x53\x53\x50\x00\x02\x00\x00\x00\x12\x00\x12\x00\x30\x00\x00\x00\x05\x02\x8a\x00\xf5\xe4\x8c\x90\x50\xe3\xcd\x63\x00\x00\x00\x00\x00\x00\x00\x00\x5c\x00\x5c\x00\x42\x00\x00\x00\x57\x00\x49\x00\x4e\x00\x32\x00\x4b\x00\x42\x00\x41\x00\x53\x00\x45\x00\x02\x00\x12\x00\x57\x00\x49\x00\x4e\x00\x32\x00\x4b\x00\x42\x00\x41\x00\x53\x00\x45\x00\x01\x00\x12\x00\x57\x00\x49\x00\x4e\x00\x32\x00\x4b\x00\x42\x00\x41\x00\x53\x00\x45\x00\x04\x00\x12\x00\x77\x00\x69\x00\x6e\x00\x32\x00\x6b\x00\x62\x00\x61\x00\x73\x00\x65\x00\x03\x00\x12\x00\x77\x00\x69\x00\x6e\x00\x32\x00\x6b\x00\x62\x00\x61\x00\x73\x00\x65\x00\x00\x00\x00\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x35\x2e\x30\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x32\x30\x30\x30\x20\x4c\x41\x4e\x20\x4d\x61\x6e\x61\x67\x65\x72\x00",
"\x00\x00\x00\x23\xff\x53\x4d\x42\x73\x6d\x00\x00\xc0\x98\x01\x68\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x00\x08\xac\x0a\x00\x00\x00",
"\x00\x00\x00\x58\xff\x53\x4d\x42\x73\x00\x00\x00\x00\x98\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x2f\xaa\x01\x08\xac\x0a\x03\xff\x00\x58\x00\x00\x00\x2f\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x35\x2e\x30\x00\x57\x69\x6e\x64\x6f\x77\x73\x20\x32\x30\x30\x30\x20\x4c\x41\x4e\x20\x4d\x61\x6e\x61\x67\x65\x72\x00\x57\x4f\x52\x4b\x47\x52\x4f\x55\x50\x00",
"\x00\x00\x00\x2e\xff\x53\x4d\x42\x75\x00\x00\x00\x00\x98\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x03\xff\x00\x2e\x00\x01\x00\x05\x00\x49\x50\x43\x00\x00",
"\x00\x00\x00\x87\xff\x53\x4d\x42\xa2\x00\x00\x00\x00\x98\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x2a\xff\x00\x87\x00\x00\x00\x40\x01\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x80\x00\x00\x00\x00\x10\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x02\x00\xff\x05\x00\x00\x00\x00\x00\x00\x00\x00\xa0\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x10\x00\x00\xff\x9b\x01\x12\x00\x9b\x01\x12\x00\x12\x00",
"\x00\x00\x00\x2f\xff\x53\x4d\x42\x2f\x00\x00\x00\x00\x98\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x06\xff\x00\x2f\x00\x48\x00\xff\xff\x00\x00\x00\x00\x00\x00",
"\x00\x00\x00\x80\xff\x53\x4d\x42\x2e\x00\x00\x00\x00\x98\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x0c\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x44\x00\x3c\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x45\x00\x00\x05\x00\x0c\x03\x10\x00\x00\x00\x44\x00\x00\x00\x00\x00\x00\x00\xb8\x10\xb8\x10\xbe\x2f\x01\x00\x0d\x00\x5c\x50\x49\x50\x45\x5c\x6e\x74\x73\x76\x63\x73\x00\x00\x01\x00\x00\x00\x00\x00\x00\x00\x04\x5d\x88\x8a\xeb\x1c\xc9\x11\x9f\xe8\x08\x00\x2b\x10\x48\x60\x02\x00\x00\x00",
"\x00\x00\x00\x2f\xff\x53\x4d\x42\x2f\x00\x00\x00\x00\x98\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x06\xff\x00\x2f\x00\x64\x00\xff\xff\x00\x00\x00\x00\x00\x00",
"\x00\x00\x00\x70\xff\x53\x4d\x42\x2e\x00\x00\x00\x00\x98\x01\x20\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x08\x2f\xaa\x01\x08\xac\x0a\x0c\xff\x00\x00\x00\x00\x00\x00\x00\x00\x00\x34\x00\x3c\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x35\x00\x25\x05\x00\x02\x03\x10\x00\x00\x00\x34\x00\x00\x00\x00\x00\x00\x00\x1c\x00\x00\x00\x00\x00\x00\x00\x0a\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x00\x34\x00\x00\xc0",
]
		setup_test(read, write)

		Timeout.timeout($_REX_TEST_TIMEOUT) {
			handle = Rex::Proto::DCERPC::Handle.new(['4b324fc8-1670-01d3-1278-5a47bf6ee188', '3.0'], 'ncacn_np', $_REX_TEST_SMB_HOST, ['\BROWSER'])
			dcerpc = Rex::Proto::DCERPC::Client.new(handle, nil)
			assert_instance_of(Rex::Proto::DCERPC::Client, dcerpc, 'bind')

			require 'rex/proto/dcerpc/ndr'
			stub =
				Rex::Proto::DCERPC::NDR.long(1) +
				Rex::Proto::DCERPC::NDR.UnicodeConformantVaryingString('AAAA') +

				# Type 33
				Rex::Proto::DCERPC::NDR.long(1) +
				Rex::Proto::DCERPC::NDR.short(2) +
				Rex::Proto::DCERPC::NDR.short(3) +
				Rex::Proto::DCERPC::NDR.UniConformantArray('AAAA') +

				Rex::Proto::DCERPC::NDR.UnicodeConformantVaryingString('BBBB') +
				Rex::Proto::DCERPC::NDR.long(4) +
				Rex::Proto::DCERPC::NDR.long(10);

			response = dcerpc.call(48, stub)
			assert_equal(response.length, 28, 'response')
		}
	end

	# makes two calls to create_tcp, first should raise Rex::ConnectionRefused, second should connect
	def test_setup_socket_ncacn_np_nil_socket_first_connect_refused

	end

	def test_read
	end

	def test_write
	end
end

rescue LoadError
end

