module Examples.Simple.Fixedpoint where

import Prelude ()
import qualified Prelude as P
import Feldspar
import Feldspar.FixedPoint
import Feldspar.Compiler
import Feldspar.Vector

-- | Generic (not compilable) adder
fpex1 :: (Num a) => a -> a -> a
fpex1 x y = x + y

-- | Wrapper for 'fpex1' using fixed-point numbers
-- (with type Int32, input exponents are (-2) and (-3), 
--                     while the output exponent is (-4))
fpex1Fix32 :: Data Int32 -> Data Int32 -> Data Int32
fpex1Fix32 x y = freezeFix' (-2) $ fpex1 ((unfreezeFix' (-3) x)) 
                                        ((unfreezeFix' (-4) y))

-- | Wrapper for 'fpex1' using floating-point numbers
fpex1Float :: Data Float -> Data Float -> Data Float
fpex1Float x y = fpex1 x y

-- | Generic (not compilable) division
fpex2 :: (Fractional a, Fixable a) => a -> a -> a
fpex2 x y = x / y

-- | Wrapper for 'fpex2' using fixed-point numbers
fpex2Fix32 :: Data Int32 -> Data Int32 -> Data Int32
fpex2Fix32 x y = freezeFix' (-4) $ fpex2 ((unfreezeFix' (-8) x)) 
                                        ((unfreezeFix' (-4) y))

-- | Wrapper for 'fpex3' using floating-point numbers
fpex2Float :: Data Float -> Data Float -> Data Float
fpex2Float x y = fpex2 x y

-- | Generic (not compilable) function adding 0.25 to the input
fpex3 :: (Num a,Fractional a,Fixable a) => a -> a
fpex3 x = x + (fix (-2) 0.25)

-- | Wrapper for 'fpex3' using fixed-point numbers (with type Int32)
fpex3Fix32 :: Data Int32 -> Data Int32
fpex3Fix32 x = freezeFix' (-4) $ fpex3 $ unfreezeFix' (-2) x

-- | Wrapper for 'fpex3' using floating-point numbers
fpex3Float :: Data Float -> Data Float
fpex3Float x = fpex3 x

-- | Generic (not compilable) function increasing each element of the input vector
fpex4 :: (Num a) => 
         a -> Vector a -> Vector a
fpex4 x = map (x+)

-- | Wrapper for 'fpex4' using fixed-point numbers (with type Int32)
fpex4Fix32 :: Data Int32 -> DVector Int32 -> DVector Int32
fpex4Fix32 x xs = map (freezeFix' (-4)) $ fpex4 x' xs'
   where
      xs' :: Vector (Fix Int32)
      xs' = map (unfreezeFix' (-6)) xs
      x'  :: Fix Int32
      x'  = unfreezeFix' (-8) x

-- | Wrapper for 'fpex4' using floating-point numbers
fpex4Float :: Data Float -> Data [Float] -> Data [Float]
fpex4Float x xs = freezeVector $ fpex4 x xs'
   where
      xs' = unfreezeVector' 256 xs
      

-- | Generic (not compilable) average function
fpex5 :: (Num a,Fractional a) => a -> a -> a
fpex5 x y = (x + y) / 2 

-- | Wrapper for 'fpex5' using fixed-point numbers (with type Int32)
fpex5Fix32 :: Data Int32 -> Data Int32 -> Data Int32
fpex5Fix32 x y = freezeFix' (-5)  $ fpex5 x' y'
  where
    x' = unfreezeFix' (-4) x
    y' = unfreezeFix' (-6) y

-- | Wrapper for 'fpex5' using floating-point numbers
fpex5Float :: Data Float -> Data Float -> Data Float
fpex5Float x y = fpex5 x y

-- | Generic (not compilable) function with condition
fpex6 :: (Fractional a, Syntactic a, Fixable a) => Data Bool -> a -> a
fpex6 cond x = cond ?! (x, x+100.256)

-- | Wrapper for 'fpex6' using fixed-point numbers (with type Int32)
fpex6Fix32 :: Data Bool -> Data Int32 -> Data Int32
fpex6Fix32 cond = freezeFix' (-2) . fpex6 cond . unfreezeFix' (-3)

-- | Wrapper for 'fpex6' using floating-point numbers
fpex6Float :: Data Bool -> Data Float -> Data Float
fpex6Float = fpex6

-- | Generic (not compilable) scalar product function
fpScalarProd :: (Num a, Syntactic a, Fixable a) =>
                Vector a -> Vector a -> a
fpScalarProd x y = fixFold (+) (fix (-8) 0) $ zipWith (*) x y

-- | Wrapper for 'fpScalarProd' using fixed-point numbers (with type Int32)
fpScalarProdFix32 :: DVector Int32 -> DVector Int32 -> Data Int32
fpScalarProdFix32 x y = freezeFix' (-8) $ fpScalarProd x' y'
  where
    x' = map (unfreezeFix' (-6)) x
    y' = map (unfreezeFix' (-4)) y

-- | Wrapper for 'fpScalarProd' using floating-point numbers
fpScalarProdFloat :: DVector Float -> DVector Float -> Data Float
fpScalarProdFloat = fpScalarProd
