/*******************************************************************************
 * Copyright (c) 2000, 2004 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.internal.outputview;

import java.net.URL;
import java.text.MessageFormat;
import java.util.ResourceBundle;
import org.eclipse.core.resources.IWorkspace;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IStatus;
import org.eclipse.core.runtime.Status;
import org.eclipse.jface.resource.ImageDescriptor;
import org.eclipse.jface.resource.ImageRegistry;
import org.eclipse.swt.graphics.Image;
import org.eclipse.ui.plugin.AbstractUIPlugin;

/**
 * The plugin class for the output view
 * declared in the plugin.xml.
 * @see AbstractUIPlugin for additional information on UI plugins
 */
public class OutputViewPlugin extends AbstractUIPlugin
{
   /** The name of our properties that are not needed during startup. */
   protected static final String PROPERTIES = "org/eclipse/wst/rdb/internal/outputview/nl1/OutputView";
   /** A static reference to our resource bundle. */
   protected static ResourceBundle properties = null;

   /**
    * Default instance of the plugin class.
    */
   private static OutputViewPlugin inst;

   /**
    * Creates the plugin and cache its default instance.
    */
   public OutputViewPlugin() {
      super();
      if (inst==null) inst = this;
   }

   /**
    * Gets the plugin singleton.
    */
   public static OutputViewPlugin getPlugin() {
      return inst;
   }

   /**
    * Gets the resource bundle for non-startup properties.
    * @return The bundle named by #PROPERTIES.
    */
   public static ResourceBundle getResourceBundle()
   {
      if (properties == null)
      {
         try
         {
            properties = ResourceBundle.getBundle(PROPERTIES);
         }
         catch (java.util.MissingResourceException exception)
         {
            OutputViewPlugin.getPlugin().writeLog(IStatus.ERROR, 0, "OutputViewPlugin:getResourceBundle.",exception);
         }
      }
      return properties;
   }

   /**
    * This gets a .gif from the image registry, which caches the icons folder.
    * @param key Identifies an Image in the registry.
    * @return The Image corresponding to the key.
    */
   public Image getImage(String key)
   {
      ImageRegistry imageRegistry = getImageRegistry();
      Image image = imageRegistry.get(key);
      if (image == null)
      {
         imageRegistry.put(key, getImageDescriptor(key));
         image = imageRegistry.get(key);
      }
      return image;
   }

   /**
    * Creates an ImageDescriptor for an Image in the registry.
    * @param key Identifies an Image in the registry.
    * @return An ImageDescriptor corresponding to the key.
    */
   public ImageDescriptor getImageDescriptor(String key) {
      try {
         return ImageDescriptor.createFromURL(new URL(getBundle().getEntry("/"), "icons" + java.io.File.separator + key + ".gif"));
      } catch (Exception e) {
         //   System.err.println("Invalid URL." + e);
         OutputViewPlugin.getPlugin().writeLog(IStatus.ERROR, 0, "OutputViewPlugin - Invalid URL.",e);
      }
      return null;
   }

   /**
    * Returns the workspace.
    * @return The workspace.
    */
   static public IWorkspace getWorkspace() {
      return ResourcesPlugin.getWorkspace();
   }

   /**
    * Gets translated text from the resource bundle.
    * @param key The key specified in plugin.properties for the string.
    * @return The string value for the given key.
    */
   public static String getString(String key)
   {
      return OutputViewPlugin.getString(key, null);
   }

   /**
    * Gets translated text with substitutions from the resource bundle.
    * @param key The key specified in plugin.properties for the string.
    * @param substitutions An Object array containing substitions for {0}, {1}, etc.
    * @return The string value for the given key, with substitutions.
    */
   public static String getString(String key, Object[] substitutions)
   {
      String str = OutputViewPlugin.getResourceBundle().getString(key);
      if (substitutions != null)
         str = MessageFormat.format(str, substitutions);
      return str;
   }

   /**
    * Writes a message to the log.
    * @param severity - the severity; one of IStatus.OK, IStatus.ERROR, IStatus.INFO, or IStatus.WARNING
    * @param code - the plug-in-specific status code, or OK
    * @param message - a human-readable message, localized to the current locale
    * @param exception- a low-level exception, or null if not applicable
    */
   public void writeLog(int severity, int code, String message, Throwable exception)
   {
      getLog().log(new Status(severity, getBundle().getSymbolicName(),
               code, message, exception));
   }
}
