/*******************************************************************************
 * Copyright (c) 2005 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License v1.0
 * which is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *******************************************************************************/
package org.eclipse.wst.rdb.sqleditor.internal;

import java.io.File;
import java.io.FileWriter;
import java.io.StringReader;

import org.eclipse.core.resources.IFile;
import org.eclipse.core.resources.ResourcesPlugin;
import org.eclipse.core.runtime.IPath;
import org.eclipse.core.runtime.Path;
import org.eclipse.ui.IMemento;
import org.eclipse.ui.part.FileEditorInput;
import org.eclipse.wst.rdb.internal.core.connection.ConnectionInfo;
import org.eclipse.wst.rdb.internal.models.sql.schema.Database;

/**
 * This class implements the <code>ISQLEditorInput</code> interface using a
 * <code>FileEditorInput</code> as the base.  It is provided as a convenience
 * for callers of the SQL Editor who want to open the SQL Editor on a file.  
 */
public class SQLEditorFileEditorInput extends FileEditorInput implements ISQLEditorInput {
	
    /** Contains connection information associated with this object. */
    private ConnectionInfo fConnInfo;
    /** The database catalog associated with this object. */
    private Database fDatabase;
    /** The name of the default schema to use with the associated database.  */
    private String fDefaultSchemaName;

    /**
     * Creates and returns a file resource in a temporary location (determined by the
     * operating system) with the given file content.  This can be used as the file
     * resource input for constructing an instance of this class.  The name of the
     * returned file resource will be "SQLStatement" + nnnnn + ".sql", where nnnnnn is 
     * a number chosen by the OS to ensure the file name is unique.
     *  
     * @param initialFileContent the initial content of the file
     * @return the new file temporary file, or <code>null</code> if the file was not
     * created successfully
     */
    public static IFile createTempFileResource( String initialFileContent ) {
        String fileNamePrefix = "SQLStatement"; //$NON-NLS-1$
        String fileExtension = ".sql"; //$NON-NLS-1$
        
        return createTempFileResource( initialFileContent, fileNamePrefix, fileExtension );
    }

    /**
     * Creates and returns a file resource in a temporary location (determined by the
     * operating system) with the given file content, file name prefix and file extension.  
     * This can be used as the file resource input for constructing an instance of this 
     * class.  The file name of the new file will be <fileNamePrefix> + nnnnn + <fileExtension>.
     * For example, if the file name prefix is "SQLStatement" and the extension is ".sql",
     * the returned file name would look something like "SQLStatement45934.sql".
     *  
     * @param initialFileContent the initial content of the file
     * @param fileNamePrefix the name (prefix) of the new file
     * @param fileExtension the extension of the file.  It should begin with "."
     * @return the new file temporary file, or <code>null</code> if the file was not
     * created successfully
     */
    public static IFile createTempFileResource( String initialFileContent, String fileNamePrefix, String fileExtension ) {
        IFile fileResource = null;
        
        // Clean up our input, if necessary.
        if (initialFileContent == null) {
            initialFileContent = ""; //$NON-NLS-1$
        }
        if (fileNamePrefix == null) {
            fileNamePrefix = ""; //$NON-NLS-1$
        }
        if (fileExtension == null) {
            fileExtension = ".sql"; //$NON-NLS-1$
        }
        else if (!fileExtension.startsWith( "." )) { //$NON-NLS-1$
            fileExtension = "." + fileExtension; //$NON-NLS-1$
        }
        
        try {
            // Create the temporary file.
            File tempFile = File.createTempFile( fileNamePrefix, fileExtension );
            
            // Load the initial file content into the file.
            StringReader reader = new StringReader( initialFileContent );
            FileWriter writer = new FileWriter( tempFile );
            int c;
            while ((c = reader.read()) != -1) {
                writer.write( c );
            }
            reader.close();
            writer.close();
            
            // Create a file resource object for the file.
            String pathStr = tempFile.getAbsolutePath();
            IPath path = new Path( pathStr );
            fileResource = ResourcesPlugin.getWorkspace().getRoot().getFile( path );
        }
        catch( Exception e ) {
            // do nothing, let it return null
        }

        return fileResource;
    }
    
	/**
	 * Creates an instance of this class with the given file as the input.
	 * 
	 * @param aFile the file to associate with this input
	 */
	public SQLEditorFileEditorInput( IFile aFile ) {
		super( aFile );
	}
	
	/**
	 * Gets the <code>ConnectionInfo</code> associated with this input.
	 * 
	 * @return the current <code>ConnectionInfo</code> object
	 */
	public ConnectionInfo getConnectionInfo() {
		return fConnInfo;
	}
	
	/**
	 * Gets the <code>Database</code> object associated with this input.
	 * 
	 * @return the <code>Database</code> object associated with this input
	 */
	public Database getDatabase() {
	    return fDatabase;
	}
	
	/**
     * Gets the default schema name to use with the connection or database
     * associated with this input.
     * 
     * @return the default schema name to use with this input, or null if none
     */
    public String getDefaultSchemaName() {
        return fDefaultSchemaName;
    }

    /**
     * Returns the id of the element factory which should be used to re-create this
     * object.
     * 
     * @see org.eclipse.ui.IPersistableElement#getFactoryId()
     */
    public String getFactoryId() {
        return SQLEditorInputFactory.ID_FACTORY;
    }

    /**
     * Saves the state of the object in the given memento.
     * 
     * @param memento the storage area for object's state
     * @see org.eclipse.ui.IPersistableElement#saveState(org.eclipse.ui.IMemento)
     */
    public void saveState(IMemento memento) {
        SQLEditorInputFactory.saveState( memento, this );
    }

	/**
	 * Sets the <code>ConnectionInfo</code> associated with this input to the given 
	 * object.
	 * 
	 * @param connInfo the <code>ConnectionInfo</code> object to set
	 */
	public void setConnectionInfo( ConnectionInfo connInfo ) {
		fConnInfo = connInfo;
	}

	/**
	 * Sets the <code>Database</code> object associated with this input to the
	 * given object.
	 * 
	 * @param database the <code>Database</code> object to set
	 */
	public void setDatabase( Database database ) {
	    fDatabase = database;
	}

	/**
     * Sets the default schema name to use with the connection or database
     * associated with this input.
     * 
     * @param schemaname the default schema name to set
     */
    public void setDefaultSchemaName( String schemaName ) {
        fDefaultSchemaName = schemaName;
    }

}