#! /bin/sh
# Create wrappers for include files instead of replacing them.
#
# This script is designed for systems whose include files can be fixed
# by creating small wrappers around them.
# An advantage of this method is that if the system include files are changed
# (e.g. by OS upgrade), you need not re-run fixincludes.
#
# See README-fixinc for more information.

# Directory in which to store the results.
LIB=${1?"fixincludes: output directory not specified"}

# Make sure it exists.
if [ ! -d $LIB ]; then
  mkdir $LIB || exit 1
fi

ORIG_DIR=`${PWDCMD-pwd}`

# Make LIB absolute if it is relative.
# Don't do this if not necessary, since may screw up automounters.
case $LIB in
/*)
	;;
*)
	cd $LIB; LIB=`${PWDCMD-pwd}`
	;;
esac

echo Building fixed headers in ${LIB}
# Directory containing the original header files.
shift
if [ $# -eq 0 ] ; then
  set /usr/include
fi

INLIST="$@"

for INPUT in ${INLIST} ; do
cd ${ORIG_DIR}
cd ${INPUT}

# Some math.h files define struct exception, which conflicts with
# the class exception defined in the C++ file std/stdexcept.h.  We
# redefine it to __math_exception.  This is not a great fix, but I
# haven't been able to think of anything better.
file=math.h
if [ -r $INPUT/$file ]; then
  echo Checking $INPUT/$file
  if grep 'struct exception' $INPUT/$file >/dev/null
  then
    echo Fixed $file
    rm -f $LIB/$file
    cat <<'__EOF__' >$LIB/$file
#ifndef _MATH_H_WRAPPER
#ifdef __cplusplus
# define exception __math_exception
#endif
 #include_next <math.h>
#ifdef __cplusplus
# undef exception
#endif
#define _MATH_H_WRAPPER
#endif /* _MATH_H_WRAPPER */
__EOF__
    # Define _MATH_H_WRAPPER at the end of the wrapper, not the start,
    # so that if #include_next gets another instance of the wrapper,
    # this will follow the #include_next chain until we arrive at
    # the real <math.h>.
    chmod a+r $LIB/$file
  fi
fi

# Similarly for struct queue in sys/stream.h.
file=sys/stream.h
if [ -r $INPUT/$file ]; then
  echo Checking $INPUT/$file
  if grep 'struct[ 	]*queue' $INPUT/$file >/dev/null
  then
    echo Fixed $file
    mkdir -p $LIB/`dirname $file`
    rm -f $LIB/$file
    cat <<'__EOF__' >$LIB/$file
#ifndef _SYS_STREAM_H_WRAPPER
#ifdef __cplusplus
# define queue __stream_queue
#endif
 #include_next <sys/stream.h>
#ifdef __cplusplus
# undef queue
#endif
#define _SYS_STREAM_H_WRAPPER
#endif /* _SYS_STREAM_H_WRAPPER */
__EOF__
    # Define _SYS_STREAM_H_WRAPPER at the end of the wrapper, not the start,
    # so that if #include_next gets another instance of the wrapper,
    # this will follow the #include_next chain until we arrive at
    # the real <sys/stream.h>.
    chmod a+r $LIB/$file
  fi
fi

# Avoid the definition of the bool type in the Solaris 2.x curses.h when using
# g++, since it's now an official type in the C++ language.
file=curses.h
if [ -r $INPUT/$file ]; then
  echo Checking $INPUT/$file
  w='[	 ]'
  if grep "typedef$w$w*char$w$w*bool$w*;" $INPUT/$file >/dev/null
  then
    echo Fixed $file
    rm -f $LIB/$file
    cat <<'__EOF__' >$LIB/$file
#ifndef _CURSES_H_WRAPPER
#ifdef __cplusplus
# define bool __curses_bool_t
#endif
 #include_next <curses.h>
#ifdef __cplusplus
# undef bool
#endif
#define _CURSES_H_WRAPPER
#endif /* _CURSES_H_WRAPPER */
__EOF__
    # Define _CURSES_H_WRAPPER at the end of the wrapper, not the start,
    # so that if #include_next gets another instance of the wrapper,
    # this will follow the #include_next chain until we arrive at
    # the real <curses.h>.
    chmod a+r $LIB/$file
  fi
fi

done

exit 0
