# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2006-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.
#
# Author: Florian Boucault <florian@fluendo.com>
#

import unittest

from pkg_resources import resource_filename
import pgm

from elisa.core.input_event import EventValue, EventSource, EventType, InputEvent

from elisa.plugins.pigment.widgets.button import AbstractButton, Button
from elisa.plugins.pigment.widgets.const import STATE_NORMAL, \
                                                STATE_SELECTED, \
                                                STATE_PRESSED, \
                                                STATE_DISABLED
from elisa.plugins.pigment.tests.widgets.test_widget import TestWidget

class TestAbstractButton(unittest.TestCase):

    def test_creation(self):
        button = Button()

    def test_default_state(self):
        button = Button()
        self.assertEqual(button.state, STATE_NORMAL)

    def test_click(self):
        button = Button()

        button.emit('clicked', 0, 0, 0, pgm.BUTTON_LEFT, 0, 0)
        self.assertEqual(button.state, STATE_NORMAL)

    def test_press_release(self):
        button = Button()

        button.emit('pressed', 0, 0, 0, pgm.BUTTON_LEFT, 0, 0)
        self.assertEqual(button.state, STATE_PRESSED)

        button.emit('released', 0, 0, 0, pgm.BUTTON_LEFT, 0)
        self.assertEqual(button.state, STATE_NORMAL)

    def test_enter_leave(self):
        button = Button()

        previous_state = button.state
        button.emit('entered', 0, 0, 0, 0)
        self.assertEqual(button.state, STATE_SELECTED)

        button.emit('left', 0, 0, 0, 0)
        self.assertEqual(button.state, previous_state)

    def test_enter_press_leave_release(self):
        button = Button()

        button.emit('entered', 0, 0, 0, 0)
        self.assertEqual(button.state, STATE_SELECTED)

        button.emit('pressed', 0, 0, 0, pgm.BUTTON_LEFT, 0, 0)
        self.assertEqual(button.state, STATE_PRESSED)

        button.emit('left', 0, 0, 0, 0)
        self.assertEqual(button.state, STATE_NORMAL)

        button.emit('released', 0, 0, 0, pgm.BUTTON_LEFT, 0)
        self.assertEqual(button.state, STATE_NORMAL)

    def test_enter_press_leave_enter(self):
        button = Button()

        button.emit('entered', 0, 0, 0, 0)
        self.assertEqual(button.state, STATE_SELECTED)

        button.emit('pressed', 0, 0, 0, pgm.BUTTON_LEFT, 0, 0)
        self.assertEqual(button.state, STATE_PRESSED)

        button.emit('left', 0, 0, 0, 0)
        self.assertEqual(button.state, STATE_NORMAL)

        button.emit('entered', 0, 0, 0, 0)
        self.assertEqual(button.state, STATE_PRESSED)


class TestDisabledAbstractButton(unittest.TestCase):
    """
    Test case for disabled state.
    """

    def test_press_release(self):
        button = Button()
        button.state = STATE_DISABLED

        button.emit('pressed', 0, 0, 0, pgm.BUTTON_LEFT, 0, 0)
        self.assertEquals(button.state, STATE_DISABLED)

        button.emit('released', 0, 0, 0, pgm.BUTTON_LEFT, 0)
        self.assertEquals(button.state, STATE_DISABLED)

    def test_enter_leave(self):
        button = Button()
        button.state = STATE_DISABLED

        previous_state = button.state
        button.emit('entered', 0, 0, 0, 0)
        self.assertEquals(button.state, STATE_DISABLED)

        button.emit('left', 0, 0, 0, 0)
        self.assertEquals(button.state, STATE_DISABLED)

    def test_enter_press_leave_release(self):
        button = Button()
        button.state = STATE_DISABLED

        button.emit('entered', 0, 0, 0, 0)
        self.assertEquals(button.state, STATE_DISABLED)

        button.emit('pressed', 0, 0, 0, pgm.BUTTON_LEFT, 0, 0)
        self.assertEquals(button.state, STATE_DISABLED)

        button.emit('left', 0, 0, 0, 0)
        self.assertEquals(button.state, STATE_DISABLED)

        button.emit('released', 0, 0, 0, pgm.BUTTON_LEFT, 0)
        self.assertEquals(button.state, STATE_DISABLED)

    def test_enter_press_leave_enter(self):
        button = Button()
        button.state = STATE_DISABLED

        button.emit('entered', 0, 0, 0, 0)
        self.assertEquals(button.state, STATE_DISABLED)

        button.emit('pressed', 0, 0, 0, pgm.BUTTON_LEFT, 0, 0)
        self.assertEquals(button.state, STATE_DISABLED)

        button.emit('left', 0, 0, 0, 0)
        self.assertEquals(button.state, STATE_DISABLED)

        button.emit('entered', 0, 0, 0, 0)
        self.assertEquals(button.state, STATE_DISABLED)


class TestAbstractButtonActivation(TestWidget):
    """
    Test case for button 'activated' signal emission.

    Inherits from TestWidget in order to have access to 'signal_connect_after'
    facility.
    """

    def test_click(self):
        button = Button()
        button.test_activated = False

        def on_activated(widget):
            widget.test_activated = True

        dfr = self.signal_connect_after(button, 'activated', on_activated)
        button.emit('clicked', 0, 0, 0, pgm.BUTTON_LEFT, 0, 0)
        self.assertTrue(button.test_activated)
        return dfr

    def test_input_ok(self):
        button = Button()
        button.test_activated = False

        def on_activated(widget):
            widget.test_activated = True

        dfr = self.signal_connect_after(button, 'activated', on_activated)
        event = InputEvent(EventSource.OTHER, EventType.OTHER, EventValue.KEY_OK)
        handled = button.handle_input(None, event)
        self.assertTrue(handled)
        self.assertTrue(button.test_activated)
        return dfr

class TestButton(TestAbstractButton):

    def test_icon(self):
        """
        Test loading of an image in the 'icon' of a Button
        """
        button = Button()
        valid_image = resource_filename("elisa.plugins.pigment.widgets",
                                        "data/star.png")
        button.icon.set_from_file(valid_image)

    def test_text(self):
        """
        Test setting the label of the button's text
        """
        button = Button()
        self.assertEqual(button.text.label, "")

        button.text.label = "ciao"
        self.assertEqual(button.text.label, "ciao")

